/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.container.ByteArrayFileHolder;
import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.crypto.CryptographicServiceAuthenticationFactory;
import com.openexchange.annotation.NonNull;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographicAwareIDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographyMode;
import com.openexchange.log.LogFactory;
import com.openexchange.office.document.DocumentMetaData;
import com.openexchange.office.tools.common.StatusProperties;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.rt.EncodeUtility;
import com.openexchange.office.tools.session.SessionUtils;
import com.openexchange.realtime.dispatch.MessageDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceExceptionCode;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link DocumentFilterAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
/**
 * {@link DocumentRESTAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:malte.timmermann@open-xchange.com">Malte Timmermann</a>
 */
public abstract class DocumentRESTAction implements AJAXActionService {

	public static final String USE_DEFAULT_TARGET_FOLDER_ID = null;

    private static final Log LOG = LogFactory.getLog(DocumentRESTAction.class);

    static final String m_operationsSyntheticUser = "operations";

    static final String m_errorKey = "hasErrors";

    static final String m_eTagBase = "http://www.open-xchange.com/filestore/";

    /**
     * Initializes a new {@link DocumentRESTAction}.
     *
     * @param services
     * @param oqm
     */
    public DocumentRESTAction(final ServiceLookup servicesDEPRECATED) {
    }

    /**
     * @param request
     * @param data
     * @param format
     * @param mimeType
     * @param fileName (may be null, if fileName should be retrieved from requestt)
     * @return AJAXRequestResult
     * @throws OXException
     */
    static protected AJAXRequestResult createFileRequestResult(@NonNull AJAXRequestData request, @NonNull byte[] data,
            @NonNull String format, @NonNull String mimeType, String fileName) {
        AJAXRequestResult requestResult = null;
        final String fileId = request.getParameter("id");
        final String fileVersion = request.getParameter("version");
        String resultFileName = (null != fileName) ? fileName : request.getParameter("filename");

        if (null == resultFileName) {
            resultFileName = "file";
        }

        if (format.equals("json")) {

            JSONObject jsonObject = new JSONObject();
            HashMap<String, String> map = new HashMap<>();
            StringBuilder dataUrlBuilder = new StringBuilder("data:");

            dataUrlBuilder.append(mimeType).append(";base64,").append(Base64.encodeBase64String(data));

            map.put("mimeType", mimeType);
            map.put("fileName", resultFileName);
            map.put("dataUrl", dataUrlBuilder.toString());
            dataUrlBuilder = null;

            try {
                jsonObject.put("data", new JSONObject(map));
            } catch (JSONException e) {
                LOG.error("Exception detected while trying to create json result for request", e);
            }

            request.setFormat("json");
            requestResult = new AJAXRequestResult(jsonObject, "json");
        } else {
            final ByteArrayFileHolder fileHolder = new ByteArrayFileHolder(data);

            fileHolder.close();
            fileHolder.setName(resultFileName);
            fileHolder.setContentType(mimeType);

            requestResult = new AJAXRequestResult(fileHolder, "file");

            setETag(getETag(fileId, fileVersion), 0, requestResult);
        }

        return requestResult;
    }

    protected File getNewFile(AJAXRequestData request, String newTargetFolderId, String mimeType, String extensionType, File templateFile, boolean binaryConverted) {

        String targetFolderId = StringUtils.isEmpty(newTargetFolderId) ? request.getParameter("target_folder_id") : newTargetFolderId;
        String initialFileName = request.getParameter("target_filename");

        boolean preserveFileName = Boolean.parseBoolean(request.getParameter("preserve_filename"));

        File file = new DefaultFile();

        // set the required file properties
        String createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
        file.setId(FileStorageFileAccess.NEW);
        file.setFolderId(targetFolderId);

        if (!preserveFileName || templateFile == null) {
            createFileName = ((null == initialFileName) || (initialFileName.length() < 1)) ? "unnamed" : initialFileName;
        } else {
            createFileName = FileHelper.getBaseName(templateFile.getFileName());
        }

        setFileName(file, createFileName, mimeType, extensionType, request);

        if (binaryConverted) {
            Map<String, Object> meta = file.getMeta();

            if (null == meta) {
                file.setMeta(meta = new HashMap<>());
            }

            meta.put(DocumentMetaData.META_BINARY_CONVERTED, Boolean.TRUE);
        }

        return file;
    }

    protected String getCryptoAction(AJAXRequestData request) {
        String cryptoAction = request != null ? request.getParameter("cryptoAction") : "";
        return cryptoAction == null ? "" : cryptoAction;
    }

    protected IDBasedFileAccess getFileAccess(ServerSession session, AJAXRequestData request, String cryptoAction) throws OXException {

        if(!cryptoAction.isEmpty()) {

            //Parsing authentication from the request. Might be null since not all crypto-actions require authentication.
            String authentication = null;
            CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = ServiceLookupRegistry.get().getService(CryptographicServiceAuthenticationFactory.class);
            if(encryptionAuthenticationFactory != null && request != null && request.optHttpServletRequest() != null) {
                authentication = encryptionAuthenticationFactory.createAuthenticationFrom(request.optHttpServletRequest());
            }

            //Creating file access with crypto functionalities
            CryptographicAwareIDBasedFileAccessFactory encryptionAwareFileAccessFactory = ServiceLookupRegistry.get().getService(CryptographicAwareIDBasedFileAccessFactory.class);
            if (encryptionAwareFileAccessFactory != null) {
                EnumSet<CryptographyMode> cryptMode = CryptographyMode.createSet(cryptoAction);
                if (cryptMode.size() > 0) {
                    return encryptionAwareFileAccessFactory.createAccess(ServiceLookupRegistry.get().getService(IDBasedFileAccessFactory.class).createAccess(session), cryptMode, session, authentication);
                }

                throw AjaxExceptionCodes.INVALID_PARAMETER_VALUE.create("cryptoAction",cryptoAction);
            }
            throw ServiceExceptionCode.SERVICE_UNAVAILABLE.create(CryptographicAwareIDBasedFileAccessFactory.class.getSimpleName());
        }

        return ServiceLookupRegistry.get().getService(IDBasedFileAccessFactory.class).createAccess(session);
    }

    protected IDBasedFileAccess getFileAccess(ServerSession session, AJAXRequestData request) throws OXException {

        String cryptoAction = getCryptoAction(request);
        return getFileAccess(session, request, cryptoAction);
    }

    protected void setFileName(File file, String fileName, String mimeType, String extensionType, AJAXRequestData request) {
        if ("Encrypt".equals(request.getParameter("cryptoAction"))) {
            file.setFileName(fileName + "." + extensionType + ".pgp");
            file.setFileMIMEType("application/pgp-encrypted");

        } else {
            file.setFileName(fileName + "." + extensionType);
            file.setFileMIMEType(mimeType);
        }
    }


    /** return JSON which contain information about current user session.
     *
     *  @note    Retrieving session information is implemented very gracefully.
     *          It's because the session API seems not well designed and show
     *          different behavior on different installations.
     *          Especially on custom SSO (single sign on) installations the
     *          source of context information is not (real) clear.
     *
     *     @param    serverSession [IN]
     *             the server session object where we should retrieve all informations from.
     *
     *  @return a JSON object which contain all session data we need for RT.
     */
    protected JSONObject getJSONObjectFromServerSession(final ServerSession session)
    {
        JSONObject jsonSession = new JSONObject();

        try
        {
            final String sessionId = SessionUtils.getSessionId (session);
            final JSONObject jsonSessionDetails
                = new JSONObject()
                       .put("protocol" , "ox"           )
                       .put("component", JSONObject.NULL)
                       .put("resource" , sessionId      );

            jsonSession = new JSONObject().put("session", jsonSessionDetails);
        }
        catch (final Throwable ex)
        {
            LOG.error(ex.getMessage(), ex);
            LOG.error(SessionUtils.dumpSessionToString(session));
            throw new RuntimeException ("Not able to retrieve needed information from session.", ex);
        }

        return jsonSession;
    }

    /**
     * @param request
     * @param session
     * @param methodName
     * @param elementName
     * @param jsonMessageData
     * @return
     */
    protected JSONObject sendConnectionMessageSynchronously(AJAXRequestData request, ServerSession session, String methodName, String elementName, JSONObject messageData) {
        final String fileId = request.getParameter("id");
        final String sharedResId = request.getParameter("com.openexchange.realtime.resourceID");
        final String appType = request.getParameter("app");
        final String rtData = request.getParameter(RTActionParameters.KEY_RTDATA);
        final String rtID = request.getParameter(RTActionParameters.KEY_RTID);
        final JSONObject requestData = getJSONObjectFromServerSession(session);
        JSONObject resultData = null;
        Stanza resultStanza = null;

        // Check valid must-have parameters before creating a rt-message to be sent via real-time
        if ((StringUtils.isNotEmpty(fileId) || (StringUtils.isNotEmpty(sharedResId))) &&
            (StringUtils.isNotEmpty(rtData) || (StringUtils.isNotEmpty(rtID)))) {
            try {
                final Message message = new Message();
                final String toEncode = StringUtils.isNotEmpty(sharedResId) ? sharedResId : fileId;
                final String resource = EncodeUtility.encodeResourcePart(toEncode);

                JSONObject rtDataJSON = null;
                if (null != rtData)
                    rtDataJSON = new JSONObject(rtData);
                else {
                    rtDataJSON = new JSONObject();
                    rtDataJSON.put(RTActionParameters.KEY_RTID, rtID);
                    rtDataJSON.put(RTActionParameters.KEY_SESSION, session.getSessionID());
                }

                message.setTo(new ID(
                    "synthetic",
                    appType,
                    m_operationsSyntheticUser,
                    SessionUtils.getContextIdString(session),
                    resource));
                message.addPayload(ActionHandler.getMethodCall(methodName));
                message.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    rtDataJSON, "json", RTActionParameters.KEY_OFFICE, RTActionParameters.KEY_RTDATA).build()));

                if (null != messageData) {
                    for (final Iterator<String> keys = messageData.keys(); keys.hasNext();) {
                        final String curKey = keys.next();
                        requestData.put(curKey, messageData.get(curKey));
                    }
                }

                message.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                    new PayloadElement(requestData, "json", appType, elementName)).build()));

                resultStanza = ServiceLookupRegistry.get().getService(MessageDispatcher.class).sendSynchronously(message, 60, TimeUnit.SECONDS);
            } catch (Exception e) {
                LOG.error("Exception detected while trying to send message to rt-environment", e);
            }

            // check result of the RT connection call and return appropriately
            if (null != resultStanza) {
                Collection<PayloadTree> payloads = resultStanza.getPayloadTrees(new ElementPath(appType, elementName));

                if (!payloads.isEmpty()) {
                    resultData = (JSONObject) payloads.iterator().next().getRoot().getData();
                }
            }
        } else {
            resultData = new JSONObject();
            try {
                resultData.put(StatusProperties.KEY_ERRORCODE, ErrorCode.GENERAL_ARGUMENTS_ERROR);
                resultData.put(m_errorKey, true);
            } catch (JSONException e) {
                LOG.error("Exception detected while trying to create error result for request", e);
            }
        }

        return resultData;
    }

    /**
     * @param resultData
     * @return
     */
    static final AJAXRequestResult getAjaxRequestResult(JSONObject resultData) {
        JSONObject ret = resultData;

        if ((null == ret) || ret.isEmpty()) {
            ret = new JSONObject();

            try {
                ret.put(m_errorKey, true);
            } catch (JSONException e) {
                //
            }
        }

        return new AJAXRequestResult(ret);
    }

    /**
     * @param fileId
     * @param fileVersion
     * @return
     */
    static protected String getETag(String fileId, String fileVersion) {
        return (new StringBuilder(m_eTagBase)).append(fileId).append('/').append(fileVersion).append(".pdf").toString();
    }

    /**
     * @param eTag
     * @param expires
     * @param result
     * @throws OXException
     */
    static protected void setETag(String eTag, long expires, AJAXRequestResult result) {
        result.setExpires(expires);

        if (eTag != null) {
            result.setHeader("ETag", eTag);
        }
    }

    /**
     * @param request
     * @param testETag
     * @return
     * @throws OXException
     */
    static protected boolean checkETag(String fileId, String fileVersion, String testETag) {
        return getETag(fileId, fileVersion).equals(testETag);
    }

    /**
     *
     * @param rtId
     * @param session
     * @return
     */
    static protected JSONObject createRTData(final String rtId, final Session session) {
        final JSONObject rtData = new JSONObject();

        try {
            rtData.put(RTActionParameters.KEY_RTID, rtId);
            rtData.put(RTActionParameters.KEY_SESSION, SessionUtils.getSessionId(session));
        } catch (Exception e) {
            // TODO: return null or empty object??
        }

        return rtData;
    }
}
