/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odt.dom.components;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.DLNode;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.IParagraph;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.ComponentContext;
import com.openexchange.office.filter.odf.components.TextComponent;
import com.openexchange.office.filter.odf.components.TextSpan_Base;
import com.openexchange.office.filter.odf.draw.CustomShape;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawImage;
import com.openexchange.office.filter.odf.draw.DrawTextBox;
import com.openexchange.office.filter.odf.draw.GroupShape;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleFamily;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StyleParagraph;
import com.openexchange.office.filter.odt.dom.Annotation;
import com.openexchange.office.filter.odt.dom.AnnotationEnd;
import com.openexchange.office.filter.odt.dom.Paragraph;
import com.openexchange.office.filter.odt.dom.Text;
import com.openexchange.office.filter.odt.dom.TextField;
import com.openexchange.office.filter.odt.dom.TextLineBreak;
import com.openexchange.office.filter.odt.dom.TextList;
import com.openexchange.office.filter.odt.dom.TextListItem;
import com.openexchange.office.filter.odt.dom.TextSpan;
import com.openexchange.office.filter.odt.dom.TextTab;

public class ParagraphComponent extends Component implements IParagraph {

	final Paragraph paragraph;

	public ParagraphComponent(ComponentContext parentContext, DLNode<Object> paragraphNode, int componentNumber) {
		super(parentContext, paragraphNode, componentNumber);
		this.paragraph = (Paragraph)getObject();
	}

	@Override
	public String simpleName() {
	    return "Para";
	}

	public Paragraph getParagraph() {
	    return paragraph;
	}

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {

        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().next : ((Paragraph)getObject()).getContent().getFirstNode();
        while(nextNode!=null) {
			final Object child = nextNode.getObject();
			if(child instanceof TextSpan) {
				final TextSpanContext textSpanContext =  new TextSpanContext(this, nextNode);
				final Component childComponent = textSpanContext.getNextChildComponent(null, previousChildComponent);
				if(childComponent!=null) {
					return childComponent;
				}
			}
			nextNode = nextNode.next;
		}
		return null;
	}

	@Override
	public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs)
		throws JSONException {

		if(attrs!=null) {
			final JSONObject paragraphAttrs = attrs.optJSONObject(OCKey.PARAGRAPH.value());
			final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
			if(paragraphAttrs!=null) {

			    int level = -1;
                final Object listLevel = paragraphAttrs.opt(OCKey.LIST_LEVEL.value());
                if(listLevel!=null) {
                    if(listLevel instanceof Integer) {
                        level = ((Integer)listLevel).intValue();
                    }
                }
                else {
                    final TextListItem listItem = paragraph.getTextListItem();
                    if(listItem!=null) {
                        level = listItem.getListLevel();
                    }
                }

                boolean labelHidden = false;
                final Object listLabelHidden = paragraphAttrs.opt(OCKey.LIST_LABEL_HIDDEN.value());
                if(listLabelHidden!=null) {
                    if(listLabelHidden instanceof Boolean) {
                        labelHidden = ((Boolean)listLabelHidden).booleanValue();
                    }
                    else {
                        labelHidden = false;
                    }
                }
                else {
                    final TextListItem listItem = paragraph.getTextListItem();
                    if(listItem!=null) {
                        labelHidden = listItem.isHeader();
                    }
                }

			    final Object listStyleId = paragraphAttrs.opt(OCKey.LIST_STYLE_ID.value());
                if(listStyleId!=null) {
                    if(listStyleId instanceof String) {
                        final TextListItem textListItem;
                        if(level>0) {
                            paragraph.setListLevel(level-1);
                            textListItem = new TextListItem(new TextList(paragraph.getTextListItem()));
                        }
                        else {
                            textListItem = new TextListItem(new TextList(null));
                        }
                        paragraph.setTextListItem(textListItem);
                        textListItem.getParentTextList().setStyleName((String)listStyleId);
                        textListItem.getParentTextList().setContinueNumbering(true);
                        textListItem.setIsHeader(labelHidden);
                        paragraph.setListLevel(level);
                        // Insert listStyle operations are inserting the listStyle into content.xml...
                        // If such a style is used within style.xml (such as header/footer are doing), then
                        // we need to clone the listStyle and also insert it into the style.xml...
                        // (great idea to put header footers into style.xml)
                        if(!isContentAutoStyle()) {
                            // check if this style is available in style.xml
                            StyleBase listStyle = styleManager.getAutoStyle((String)listStyleId, StyleFamily.LIST_STYLE, false);
                            if(listStyle==null) {
                                // clone from content.xml ...
                                listStyle = styleManager.getAutoStyle((String)listStyleId, StyleFamily.LIST_STYLE, true);
                                if(listStyle!=null) {
                                    final StyleBase listStyleClone = listStyle.clone();
                                    listStyleClone.setIsContentStyle(false);
                                    styleManager.addStyle(listStyleClone);
                                }
                            }
                        }
                    }
                    else if(listStyleId==JSONObject.NULL) {
                        paragraph.setListLevel(-1);
                    }
                }
                else {
                    if(listLevel!=null) {
                        paragraph.setListLevel(level);
                    }
                    if(listLabelHidden!=null) {
                        final TextListItem textListItem = paragraph.getTextListItem();
                        if(textListItem!=null) {
                            textListItem.setIsHeader(labelHidden);
                        }
                    }
                }

                final Object oListStartValue = paragraphAttrs.opt(OCKey.LIST_START_VALUE.value());
                if(oListStartValue!=null) {
                    final Integer listStartValue = oListStartValue instanceof Integer ? (Integer)oListStartValue : null;
                    paragraph.setStartValue(listStartValue);
                    final TextListItem textListItem = paragraph.getTextListItem();
                    if(textListItem!=null) {
                        textListItem.setStartValue(listStartValue);
                    }
                }
			}
			paragraph.setStyleName(styleManager.createStyle(StyleFamily.PARAGRAPH, paragraph.getStyleName(), isContentAutoStyle(), attrs));
		}
	}

	@Override
	public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs) {

	    final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
	    final String styleName = paragraph.getStyleName();
	    final StyleParagraph paragraphStyle = styleName!=null&&!styleName.isEmpty() ? (StyleParagraph)styleManager.getStyle(styleName, StyleFamily.PARAGRAPH, isContentAutoStyle()) : null;
		if(paragraphStyle!=null) {
		    if(paragraphStyle.isAutoStyle()) {
		        styleManager.createAutoStyleAttributes(attrs, styleName, StyleFamily.PARAGRAPH, isContentAutoStyle());
		    }
		    else {
		        attrs.put(OCKey.STYLE_ID.value(), styleName);
		    }
		}
        final Map<String, Object> paragraphAttrs  = attrs.getMap(OCKey.PARAGRAPH.value(), true);
        final Integer outlineLevel = paragraph.getOutlineLevel();
        if(outlineLevel!=null) {
            paragraphAttrs.put(OCKey.OUTLINE_LEVEL.value(), outlineLevel-1);
        }
        final TextListItem textListItem = paragraph.getTextListItem();
		if(textListItem!=null) {
			TextList textList = textListItem.getParentTextList();
			final String styleId = textList.getId();
			if(styleId!=null&&!styleId.isEmpty()) {
				paragraphAttrs.put(OCKey.LIST_ID.value(), styleId);
			}
			if(textListItem.isHeader()) {
				paragraphAttrs.put(OCKey.LIST_LABEL_HIDDEN.value(), true);
			}
			paragraphAttrs.put(OCKey.LIST_LEVEL.value(), textListItem.getListLevel());
			final Integer startValue = textListItem.getStartValue();
			if(startValue!=null) {
				paragraphAttrs.put(OCKey.LIST_START_VALUE.value(), startValue);
			}
			final String listStyleName = textList.getStyleName(true);
			if(listStyleName!=null&&!listStyleName.isEmpty()) {
				paragraphAttrs.put(OCKey.LIST_STYLE_ID.value(), listStyleName);
			}
		}
		else if(paragraph.isHeader()&&!paragraph.isListHeader()) {
		    // check the outline level
		    final int listLevel;
		    if(outlineLevel!=null) {
		        listLevel = outlineLevel.intValue();
		    }
		    else if(paragraphStyle!=null) {
		        listLevel = paragraphStyle.getDefaultOutlineLevel(styleManager);
		    }
		    else {
		        listLevel = 0;
		    }
		    if(listLevel>0) {
		        final HashMap<String, DLNode<Object>> outlineStyles = styleManager.getFamilyStyles(StyleFamily.OUTLINE_STYLE);
		        if(outlineStyles!=null) {
		            final Iterator<Entry<String, DLNode<Object>>> outlineIter = outlineStyles.entrySet().iterator();
		            if(outlineIter.hasNext()) {
		                final Entry<String, DLNode<Object>> outlineEntry = outlineIter.next();
		                paragraphAttrs.put(OCKey.LIST_STYLE_ID.value(), outlineEntry.getKey());
		                paragraphAttrs.put(OCKey.LIST_LEVEL.value(), listLevel-1);
		                final Integer startValue = paragraph.getStartValue();
		                if(startValue!=null) {
		                    paragraphAttrs.put(OCKey.LIST_START_VALUE.value(), startValue);
		                }
		                if(paragraph.isRestartNumbering()) {
		                    paragraphAttrs.put(OCKey.LIST_START_VALUE.value(), 1);
		                }
		            }
		        }
		    }
		}
        if(paragraphAttrs.isEmpty()) {
            attrs.remove(OCKey.PARAGRAPH.value());
        }
	}

	@Override
	public void insertText(OdfOperationDoc operationDocument, int textPosition, String text, JSONObject attrs)
		throws JSONException, SAXException {

    	if(text.length()>0) {
            Text t = null;
            Component childComponent = getNextChildComponent(null, null);
            Component cRet = null;

            if(childComponent!=null) {
            	if(textPosition>0) {
            		childComponent = childComponent.getComponent(textPosition-1);
            	}
                // check if the character could be inserted into an existing text:
                if(childComponent instanceof TextComponent) {
                    t = (Text)childComponent.getObject();
                    final StringBuffer s = new StringBuffer(t.getText());
                    s.insert(textPosition-((TextComponent)childComponent).getComponentNumber(), text);
                    t.setText(s.toString());
                    cRet = childComponent;
                }
                else {
                	t = new Text(text, true);
                	final TextSpanContext spanContext = (TextSpanContext)childComponent.getParentContext();
                    ((TextSpan)spanContext.getObject()).getContent().addNode(childComponent.getNode(), new DLNode<Object>(t), textPosition==0);
                    cRet = childComponent;

                    if(textPosition>0) {
                        cRet = childComponent.getNextComponent();
                    }
                    else {
                        cRet = getNextChildComponent(null, null);
                    }
                }
            }
            else {

            	// the paragraph is empty, we have to create R and its text
                final TextSpan newRun = new TextSpan();
                paragraph.getContent().add(newRun);
/*
                if(paragraph.getPPr()!=null) {
                    final RPr rPr = TextUtils.cloneParaRPrToRPr(paragraph.getPPr().getRPr());
                    if(rPr!=null) {
                        rPr.setParent(newRun);
                        newRun.setRPr(rPr);
                    }
                }
*/
                t = new Text(text, true);
                newRun.getContent().add(t);
                cRet = getNextChildComponent(null, null);
            }
            if(attrs!=null) {
            	cRet.splitStart(textPosition);
            	cRet.splitEnd(textPosition+text.length()-1);
            	cRet.applyAttrsFromJSON(operationDocument, attrs);
            }
        }
	}

    @Override
    public Component insertChildComponent(OdfOperationDoc operationDocument, int textPosition, JSONObject attrs, Type childType)
        throws JSONException, SAXException {

        Object newChild = null;
        switch(childType) {
            case TAB : {
                newChild = new TextTab(null);
                break;
            }
            case HARDBREAK : {
            	newChild = new TextLineBreak();
            	break;
            }
            case FIELD : {
            	newChild = new TextField();
            	break;
            }
            case COMMENT_REFERENCE : {
            	newChild = new Annotation(null, false);
            	break;
            }
            case COMMENT_RANGE_END : {
            	newChild = new AnnotationEnd(null);
            	break;
            }
            case AC_IMAGE : {
            	final DrawFrame drawFrame = new DrawFrame(operationDocument, null, true, isContentAutoStyle());
            	drawFrame.getContent().add(new DrawImage(drawFrame));
            	newChild = drawFrame;
            	break;
            }
            case AC_SHAPE : {
            	newChild = Shape.createShape(operationDocument, attrs, null, true, isContentAutoStyle());
            	break;
            }
            case AC_CONNECTOR : {
                newChild = new CustomShape(operationDocument, null, true, isContentAutoStyle());
                break;
            }
            case AC_GROUP : {
            	newChild = new GroupShape(operationDocument, null, true, isContentAutoStyle());
            	break;
            }
            case AC_FRAME : {
            	final DrawFrame drawFrame = new DrawFrame(operationDocument, null, true, isContentAutoStyle());
            	drawFrame.getContent().add(new DrawTextBox(drawFrame));
            	newChild = drawFrame;
            	break;
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
        Component childComponent = getNextChildComponent(null, null);
        if(childComponent!=null) {
            if(textPosition>0) {
                childComponent = childComponent.getComponent(textPosition-1);
                childComponent.splitEnd(textPosition-1);
            }
    		// the new child can be added into an existing textRun
            final TextSpan textSpan = ((TextSpan_Base)childComponent).getTextSpan();
            textSpan.getContent().addNode(childComponent.getNode(), new DLNode<Object>(newChild), textPosition==0);
        }
        else {
            final TextSpan newTextSpan = new TextSpan();
            paragraph.getContent().add(newTextSpan);
            newTextSpan.getContent().add(newChild);
        }
        if(textPosition>0) {
            childComponent = childComponent.getNextComponent();
        }
        else {
            childComponent = getNextChildComponent(null, null);
        }
        if(attrs!=null) {
            childComponent.splitStart(textPosition);
            childComponent.splitEnd(textPosition);
            childComponent.applyAttrsFromJSON(operationDocument, attrs);
        }
        return childComponent;
    }

	@Override
    public void splitParagraph(OdfOperationDoc operationDocument, int textPosition) {
        final Paragraph destParagraph = new Paragraph(
        	paragraph.getTextListItem()!=null
        	? new TextListItem(paragraph.getTextListItem().getParentTextList())
        	: null);

        final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
        // taking care of paragraphs sharing the same textListItem
        if(destParagraph.getTextListItem()!=null) {
        	DLNode<Object> nextParaNode = getNode().next;
        	while(nextParaNode!=null&&nextParaNode.getObject() instanceof Paragraph) {
	        	final Paragraph nextPara = (Paragraph)getNode().next.getObject();
	        	if(nextPara.getTextListItem()!=paragraph.getTextListItem()) {
	        		break;
	        	}
	        	nextPara.setTextListItem(destParagraph.getTextListItem());
    	        nextParaNode = nextParaNode.next;
        	}
        }

        final DLNode<Object> destParagraphNode = new DLNode<Object>(destParagraph);
        ((INodeAccessor)getParentContext().getObject()).getContent().addNode(getNode(), destParagraphNode, false);

        String paragraphStyleName = paragraph.getStyleName();
        final StyleBase paragraphStyle = styleManager.getStyle(paragraphStyleName, StyleFamily.PARAGRAPH, isContentAutoStyle());
        if(paragraphStyle!=null) {
            final AttributesImpl styleAttributes = paragraphStyle.getAttributes();
            if(styleAttributes.containsKey("style:master-page-name")||styleAttributes.containsKey("fo:break-before")||styleAttributes.containsKey("fo:break-after")) {
                final StyleBase paragraphClone = paragraphStyle.clone();
                paragraphClone.getAttributes().remove("style:master-page-name");
                paragraphClone.getAttributes().remove("fo:break-before");
                paragraphClone.getAttributes().remove("fo:break-after");
                paragraphStyleName = styleManager.getExistingStyleIdForStyleBase(paragraphClone);
                if(paragraphStyleName==null||paragraphStyleName.isEmpty()) {
                    paragraphStyleName = styleManager.getUniqueStyleName(StyleFamily.PARAGRAPH, isContentAutoStyle());
                    paragraphClone.setName(paragraphStyleName);
                    styleManager.addStyle(paragraphClone);
                }
            }
        }
        destParagraph.setStyleName(paragraphStyleName);
        Component component = getNextChildComponent(null, null);
    	while(component!=null&&component.getNextComponentNumber()<=textPosition) {
    		component = component.getNextComponent();
    	}
        if(component!=null) {
        	component.splitStart(textPosition);

        	// moving text spans into the new paragraph
        	paragraph.getContent().moveNodes(component.getParentContext().getNode(), -1, destParagraph.getContent(), null, true);
        }
    }

	@Override
	public void mergeParagraph() {
    	final Component nextParagraphComponent = getNextComponent();
    	if(nextParagraphComponent instanceof ParagraphComponent) {
        	final DLList<Object> sourceContent = ((Paragraph)nextParagraphComponent.getObject()).getContent();
            sourceContent.moveNodes(paragraph.getContent());
            ((INodeAccessor)getParentContext().getObject()).getContent().removeNode(nextParagraphComponent.getNode());
    	}
	}

	public static Paragraph getFirstParagraph(Component rootComponent, boolean forceCreate) {
	    final ParagraphComponent paragraphComponent = _getFirstParagraphComponent(rootComponent.getChildComponent(0));
	    if(paragraphComponent!=null) {
	        return paragraphComponent.getParagraph();
	    }
	    if(forceCreate) {
	        final Paragraph paragraph = new Paragraph(null);
	        ((INodeAccessor)rootComponent.getObject()).getContent().add(paragraph);
	        return paragraph;
	    }
	    return null;
	}

	private static ParagraphComponent _getFirstParagraphComponent(Component component) {
        while(component!=null) {
            if(component instanceof ParagraphComponent) {
                return (ParagraphComponent)component;
            }
            final Component childParagraphComponent = component.getChildComponent(0);
            if(childParagraphComponent!=null) {
                return _getFirstParagraphComponent(childParagraphComponent);
            }
            component = component.getNextComponent();
        }
        return null;
    }
}
