/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odt.dom;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfTextDocument;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.api.OCValue;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.Settings;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.TextComponent;
import com.openexchange.office.filter.odf.components.TextFieldComponent;
import com.openexchange.office.filter.odf.components.TextLineBreakComponent;
import com.openexchange.office.filter.odf.components.TextTabComponent;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.IDrawingType;
import com.openexchange.office.filter.odf.styles.MasterStyles;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleFamily;
import com.openexchange.office.filter.odf.styles.StyleHeaderFooter;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StyleMasterPage;
import com.openexchange.office.filter.odf.styles.StylePageLayout;
import com.openexchange.office.filter.odf.table.Table;
import com.openexchange.office.filter.odt.dom.components.AnnotationComponent;
import com.openexchange.office.filter.odt.dom.components.AnnotationEndComponent;
import com.openexchange.office.filter.odt.dom.components.CellComponent;
import com.openexchange.office.filter.odt.dom.components.DrawFrameComponent;
import com.openexchange.office.filter.odt.dom.components.FrameRootComponent;
import com.openexchange.office.filter.odt.dom.components.ParagraphComponent;
import com.openexchange.office.filter.odt.dom.components.RootComponent;
import com.openexchange.office.filter.odt.dom.components.RowComponent;
import com.openexchange.office.filter.odt.dom.components.ShapeComponent;
import com.openexchange.office.filter.odt.dom.components.ShapeRootComponent;
import com.openexchange.office.filter.odt.dom.components.TableComponent;

public class JsonOperationProducer {

    final JSONArray operationQueue;

    private final OdfOperationDoc opsDoc;
    private final OdfTextDocument doc;
    private final TextStyles styles;
    private final TextContent content;
    private final StyleManager styleManager;
    private final Settings settings;

    public JsonOperationProducer(OdfOperationDoc opsDoc)
    	throws SAXException {

    	operationQueue = new JSONArray();
    	this.opsDoc = opsDoc;
        doc = (OdfTextDocument)opsDoc.getDocument();
        styles = (TextStyles)doc.getStylesDom();
        content = (TextContent)doc.getContentDom();
        settings = doc.getSettingsDom();
        styleManager = doc.getStyleManager();
        styleManager.setTabsRelativeToIndent(settings.hasTabsRelativeToIndent());
    }

    public JSONObject getDocumentOperations()
    	throws JSONException, SAXException {

    	createDocumentAttributes();

    	styleManager.createInsertStyleOperations(styleManager.getStyles(), "text", operationQueue);
    	styleManager.createInsertAutoListStylesOperations(false, operationQueue);
    	styleManager.createInsertAutoListStylesOperations(true, operationQueue);

    	createHeaderFooterOperations(content.getBody().getMasterPageName(true));
    	createTextOperations(new RootComponent(content.getBody(), true), new ArrayList<Integer>(), null);

    	final JSONObject operations = new JSONObject(1);
        operations.put("operations", operationQueue);
        return operations;
    }

    public void createDocumentAttributes()
    	throws JSONException {

    	final OpAttrs attrs = new OpAttrs();
    	final Map<String, Object> documentAttributes = attrs.getMap(OCKey.DOCUMENT.value(), true);
    	documentAttributes.put(OCKey.FILE_FORMAT.value(), "odf");
        final String masterPageName = content.getBody().getMasterPageName(true);
    	final MasterStyles masterStyles = styleManager.getMasterStyles();
    	final Iterator<Object> styleMasterPageIter = masterStyles.getContent().iterator();
    	while(styleMasterPageIter.hasNext()) {
    		final Object o = styleMasterPageIter.next();
    		if(o instanceof StyleMasterPage) {
				if(((StyleMasterPage)o).getName().equals(masterPageName)) {
					final String pageLayoutName = ((StyleMasterPage)o).getPageLayoutName();
					if(pageLayoutName!=null&&!pageLayoutName.isEmpty()) {
						final StyleBase baseStyle = styleManager.getStyle(pageLayoutName, StyleFamily.PAGE_LAYOUT, false);
						if(baseStyle instanceof StylePageLayout) {
							((StylePageLayout)baseStyle).getPageLayoutProperties().createAttrs(styleManager, false, attrs);
						}
					}
				}
    		}
    	}
        final JSONObject setDocumentAttributesOperation = new JSONObject(5);
        setDocumentAttributesOperation.put(OCKey.NAME.value(), OCValue.SET_DOCUMENT_ATTRIBUTES.value());
        setDocumentAttributesOperation.put(OCKey.ATTRS.value(), attrs);
        operationQueue.put(setDocumentAttributesOperation);
    }

    public void createHeaderFooterOperations(String name)
    	throws JSONException, SAXException {

    	final MasterStyles masterStyles = styleManager.getMasterStyles();
    	final Iterator<Object> styleMasterPageIter = masterStyles.getContent().iterator();
    	while(styleMasterPageIter.hasNext()) {
    		final Object o = styleMasterPageIter.next();
    		if(o instanceof StyleMasterPage) {
				if(((StyleMasterPage)o).getName().equals(name)) {
	    			for(StyleHeaderFooter styleHeaderFooter:((StyleMasterPage)o).getStyleHeaderFooters()) {
	    				addInsertHeaderFooterOperation(styleHeaderFooter.getId(), styleHeaderFooter.getType());
	    				createTextOperations(new RootComponent(styleHeaderFooter, false), new ArrayList<Integer>(), styleHeaderFooter.getId());
	    			}
	    			return;
				}
    		}
		}
    }

    public void createTextOperations(Component parentComponent, List<Integer> parentPosition, String target)
    	throws JSONException, SAXException {

        Component component = parentComponent.getNextChildComponent(null, null);
        while (component!=null) {
        	if(component instanceof ParagraphComponent) {
        		createParagraphOperations((ParagraphComponent)component, parentPosition, target);
        	}
        	else if(component instanceof TableComponent) {
        		createTableOperations((TableComponent)component, parentPosition, target);
        	}
            component = component.getNextComponent();
        }
    }

    public void createParagraphOperations(ParagraphComponent paragraphComponent, List<Integer> parentPosition, String target)
    	throws JSONException, SAXException {

    	// insert paragraph and apply paragraph attributes
        final List<Integer> paragraphPosition = new ArrayList<Integer>(parentPosition);
        paragraphPosition.add(paragraphComponent.getComponentNumber());
        OpAttrs attrs = new OpAttrs();
        paragraphComponent.createAttrs(opsDoc, attrs);
        addInsertParagraphOperation(paragraphPosition, target, attrs);

        // insert components (text and or other drawing objects)
        Component component = paragraphComponent.getNextChildComponent(null, null);
        while(component!=null) {
        	final List<Integer> textPosition = new ArrayList<Integer>(paragraphPosition);
        	textPosition.add(component.getComponentNumber());
        	if(component instanceof TextComponent) {
        		addInsertTextOperation(textPosition, ((Text)(component.getObject())).getText(), target);
        	}
        	else if(component instanceof TextTabComponent) {
        		addInsertTabOperation(textPosition, target);
        	}
        	else if(component instanceof TextLineBreakComponent) {
        		addInsertHardBreakOperation(textPosition, target);
        	}
        	else if(component instanceof TextFieldComponent) {
        		addInsertFieldOperation(textPosition, ((TextField)component.getObject()), target);
        	}
        	else if(component instanceof FrameRootComponent) {
        		createFrameOperations((FrameRootComponent)component, textPosition, target);
        	}
        	else if(component instanceof ShapeRootComponent) {
        		createShapeOperations(component, textPosition, target);
        	}
        	else if(component instanceof AnnotationComponent) {
        		final Annotation annotation = (Annotation)component.getObject();
        		addInsertCommentOperation(textPosition, target, annotation);
        		createTextOperations(new RootComponent(annotation, paragraphComponent.isContentAutoStyle()), new ArrayList<Integer>(), annotation.getId());
        	}
        	else if(component instanceof AnnotationEndComponent) {
        		addInsertRangeOperation(textPosition, target, (AnnotationEnd)component.getObject());
        	}
        	component = component.getNextComponent();
        }

        component = paragraphComponent.getNextChildComponent(null, null);
        while(component!=null) {
        	attrs = new OpAttrs();
        	component.createAttrs(opsDoc, attrs);
            if (!attrs.isEmpty()) {
	            int startComponent = component.getComponentNumber();
	            int endComponent   = component.getNextComponentNumber()-1;
	            final List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
	            startPosition.add(startComponent);
	            if(startComponent==endComponent) {
	                addSetAttributesOperation(attrs, startPosition, null, target);
	            }
	            else {
	            	final List<Integer> endPosition = new ArrayList<Integer>(paragraphPosition);
	                endPosition.add(endComponent);
	                addSetAttributesOperation(attrs, startPosition, endPosition, target);
	            }
            }
        	component = component.getNextComponent();
        }
    }

    public void createTableOperations(TableComponent tableComponent, List<Integer> parentPosition, String target)
        	throws JSONException, SAXException {

        final List<Integer> tablePosition = new ArrayList<Integer>(parentPosition);
        tablePosition.add(tableComponent.getComponentNumber());
        final OpAttrs attrs = new OpAttrs();
        tableComponent.createAttrs(opsDoc, attrs);
        if(addInsertTableOperation(tablePosition, target, (Table)tableComponent.getObject(), attrs)) {
	    	Component component= tableComponent.getNextChildComponent(null, null);
	    	while(component!=null) {
	    		if(component instanceof RowComponent) {
	    	        final List<Integer> rowPosition = new ArrayList<Integer>(tablePosition);
	    	        rowPosition.add(component.getComponentNumber());
	    			createRowOperations((RowComponent)component, rowPosition, target);
	    		}
	    		component = component.getNextComponent();
	    	}
        }
    }

    public void createRowOperations(RowComponent rowComponent, List<Integer> rowPosition, String target)
        	throws JSONException, SAXException {

        OpAttrs attrs = new OpAttrs();
        rowComponent.createAttrs(opsDoc, attrs);
    	addInsertRowsOperation(rowPosition, 1, target, attrs);
    	Component component= rowComponent.getNextChildComponent(null, null);
    	while(component!=null) {
    		if(component instanceof CellComponent) {
    			final List<Integer> cellPosition = new ArrayList<Integer>(rowPosition);
    	        cellPosition.add(component.getComponentNumber());
    	        attrs = new OpAttrs();
    	        component.createAttrs(opsDoc, attrs);
    	        addInsertCellsOperation(cellPosition, target, attrs);
    	   		createTextOperations(component, cellPosition, target);
    		}
    		component = component.getNextComponent();
    	}
    }

    public void createFrameOperations(FrameRootComponent frameComponent, List<Integer> position, String target)
    	throws JSONException, SAXException {

    	final OpAttrs attrs = new OpAttrs();
    	frameComponent.createAttrs(opsDoc, attrs);
    	addInsertDrawingOperation(position, frameComponent.getType(), attrs, target);
    	// a frame component (textBox, image may have childs)...
        Component component = frameComponent.getNextChildComponent(null, null);
        while(component!=null) {
        	if(component instanceof ParagraphComponent) {
        		createParagraphOperations((ParagraphComponent)component, position, target);
        	}
        	else if(component instanceof TableComponent) {
        		createTableOperations((TableComponent)component, position, target);
        	}
        	component = component.getNextComponent();
        }
    }

    public void createShapeOperations(Component shapeComponent, List<Integer> position, String target)
    	throws JSONException, SAXException {

    	final OpAttrs attrs = new OpAttrs();
    	shapeComponent.createAttrs(opsDoc, attrs);
    	addInsertDrawingOperation(position, ((IDrawingType)shapeComponent).getType(), attrs, target);
    	Component component = shapeComponent.getNextChildComponent(null, null);
    	while(component!=null) {
    		if(component instanceof ParagraphComponent) {
    			createParagraphOperations((ParagraphComponent)component, position, target);
    		}
    		else if(component instanceof DrawFrameComponent) {
                final List<Integer> childPosition = new ArrayList<Integer>(position);
                childPosition.add(component.getComponentNumber());
    		    createShapeOperations(component, childPosition, target);
    		}
    		else if(component instanceof ShapeComponent) {
    	        final List<Integer> childPosition = new ArrayList<Integer>(position);
    	        childPosition.add(component.getComponentNumber());
    			createShapeOperations(component, childPosition, target);
    		}
    		component = component.getNextComponent();
    	}
    }

    public void addInsertParagraphOperation(final List<Integer> start, String target, final OpAttrs attrs)
        throws JSONException {

        final JSONObject insertParagraphObject = new JSONObject(4);
        insertParagraphObject.put(OCKey.NAME.value(), OCValue.INSERT_PARAGRAPH.value());
        insertParagraphObject.put(OCKey.START.value(), start);
        if(target!=null) {
        	insertParagraphObject.put(OCKey.TARGET.value(), target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertParagraphObject.put(OCKey.ATTRS.value(), attrs);
        }
        operationQueue.put(insertParagraphObject);
    }

    public boolean addInsertTableOperation(final List<Integer> start, String target, Table table, final OpAttrs attrs)
   		throws JSONException {

    	final boolean tableSizeExceeded = table.getTableSizeExceeded();
        final JSONObject insertTableObject = new JSONObject(4);
        insertTableObject.put(OCKey.NAME.value(), OCValue.INSERT_TABLE.value());
        insertTableObject.put(OCKey.START.value(), start);
        if(target!=null) {
        	insertTableObject.put(OCKey.TARGET.value(), target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertTableObject.put(OCKey.ATTRS.value(), attrs);
        }
        if(tableSizeExceeded) {
        	final JSONObject sizeExceeded = new JSONObject(2);
        	sizeExceeded.put(OCKey.ROWS.value(), table.getRows().getContent().size());
        	sizeExceeded.put(OCKey.COLUMNS.value(), table.getColumns().size());
        	insertTableObject.put(OCKey.SIZE_EXCEEDED.value(), sizeExceeded);
        }
        operationQueue.put(insertTableObject);
        return !tableSizeExceeded;
    }

    public void addInsertRowsOperation(final List<Integer> start, int count, String target, final OpAttrs attrs)
        throws JSONException {

        final JSONObject insertRowObject = new JSONObject(4);
        insertRowObject.put(OCKey.NAME.value(), OCValue.INSERT_ROWS.value());
        insertRowObject.put(OCKey.START.value(), start);
        if(count!=1) {
        	insertRowObject.put(OCKey.COUNT.value(), count);
        }
        if(target!=null) {
        	insertRowObject.put(OCKey.TARGET.value(), target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertRowObject.put(OCKey.ATTRS.value(), attrs);
        }
        operationQueue.put(insertRowObject);
    }

    public void addInsertCellsOperation(final List<Integer> start, String target, final OpAttrs attrs)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject(3);
        insertCellsObject.put(OCKey.NAME.value(), OCValue.INSERT_CELLS.value());
        insertCellsObject.put(OCKey.START.value(), start);
        if(target!=null) {
        	insertCellsObject.put(OCKey.TARGET.value(), target);
        }
        if(attrs!=null&&!attrs.isEmpty()) {
        	insertCellsObject.put(OCKey.ATTRS.value(), attrs);
        }
        operationQueue.put(insertCellsObject);
    }
    
    public void addInsertTextOperation(final List<Integer> start, String text, String target)
        throws JSONException {

        final JSONObject insertTextObject = new JSONObject(4);
        insertTextObject.put(OCKey.NAME.value(), OCValue.INSERT_TEXT.value());
        insertTextObject.put(OCKey.START.value(), start);
        insertTextObject.put(OCKey.TEXT.value(), text);
        if(target!=null) {
        	insertTextObject.put(OCKey.TARGET.value(), target);
        }
        operationQueue.put(insertTextObject);
    }

    public void addInsertTabOperation(final List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertTabObject = new JSONObject(3);
        insertTabObject.put(OCKey.NAME.value(), OCValue.INSERT_TAB.value());
        insertTabObject.put(OCKey.START.value(), start);
        if(target!=null) {
        	insertTabObject.put(OCKey.TARGET.value(), target);
        }
        operationQueue.put(insertTabObject);
    }

    public void addInsertHardBreakOperation(List<Integer> start, String target)
        throws JSONException {

        final JSONObject insertHardBreakObject = new JSONObject(3);
        insertHardBreakObject.put(OCKey.NAME.value(), OCValue.INSERT_HARD_BREAK.value());
        insertHardBreakObject.put(OCKey.START.value(), start);
        if(target!=null) {
        	insertHardBreakObject.put(OCKey.TARGET.value(), target);
        }
        operationQueue.put(insertHardBreakObject);
    }

    public void addInsertFieldOperation(List<Integer> start, TextField field, String target)
        throws JSONException {
        final JSONObject insertFieldObject = new JSONObject(5);
        insertFieldObject.put(OCKey.NAME.value(), OCValue.INSERT_FIELD.value());
        insertFieldObject.put(OCKey.START.value(), start);
        insertFieldObject.put(OCKey.TYPE.value(), field.getType());
        insertFieldObject.put(OCKey.REPRESENTATION.value(), field.getRepresentation());
        if(target!=null) {
        	insertFieldObject.put(OCKey.TARGET.value(), target);
        }
        operationQueue.put(insertFieldObject);
    }

    public void addInsertDrawingOperation(List<Integer> start, DrawingType type, OpAttrs attrs, String target)
        throws JSONException {

        final JSONObject insertDrawingObject = new JSONObject(5);
        insertDrawingObject.put(OCKey.NAME.value(), OCValue.INSERT_DRAWING.value());
        insertDrawingObject.put(OCKey.START.value(), start);
        insertDrawingObject.put(OCKey.TYPE.value(), type.toString());
        if(!attrs.isEmpty()) {
        	insertDrawingObject.put(OCKey.ATTRS.value(), attrs);
        }
        if(target!=null) {
        	insertDrawingObject.put(OCKey.TARGET.value(), target);
        }
        operationQueue.put(insertDrawingObject);
    }

    public void addSetAttributesOperation(final OpAttrs attrs, final List<Integer> startPosition, final List<Integer> endPosition, String target)
        throws JSONException {

    	if(!attrs.isEmpty()) {
	        JSONObject setAttributeObject = new JSONObject(5);
	        setAttributeObject.put(OCKey.NAME.value(), OCValue.SET_ATTRIBUTES.value());
	        setAttributeObject.put(OCKey.ATTRS.value(), attrs );
	        setAttributeObject.put(OCKey.START.value(), startPosition);
	        if(target!=null) {
	        	setAttributeObject.put(OCKey.TARGET.value(), target);
	        }
	        if(endPosition!=null) {
	            setAttributeObject.put(OCKey.END.value(), endPosition);
	        }
	        operationQueue.put(setAttributeObject);
    	}
    }

    public void addInsertCommentOperation(final List<Integer> start,  String target, Annotation annotation)
        throws JSONException {

    	final JSONObject setCommentObject = new JSONObject(6);
    	setCommentObject.put(OCKey.NAME.value(), OCValue.INSERT_COMMENT.value());
    	setCommentObject.put(OCKey.START.value(), start);
        if(target!=null) {
        	setCommentObject.put(OCKey.TARGET.value(), target);
        }
        setCommentObject.put(OCKey.ID.value(), annotation.getId());
        if(annotation.getCreator()!=null) {
        	setCommentObject.put(OCKey.AUTHOR.value(), annotation.getCreator());
        }
        if(annotation.getDate()!=null) {
        	setCommentObject.put(OCKey.DATE.value(), annotation.getDate());
        }
        operationQueue.put(setCommentObject);
    }

    public void addInsertRangeOperation(final List<Integer> start,  String target, AnnotationEnd annotationEnd)
        throws JSONException {

    	final JSONObject insertRangeObject = new JSONObject(6);
    	insertRangeObject.put(OCKey.NAME.value(), OCValue.INSERT_RANGE.value());
    	insertRangeObject.put(OCKey.START.value(), start);
    	insertRangeObject.put(OCKey.TYPE.value(), "comment");
        if(target!=null) {
        	insertRangeObject.put(OCKey.TARGET.value(), target);
        }
    	insertRangeObject.put(OCKey.POSITION.value(), "end");
    	insertRangeObject.put(OCKey.ID.value(), annotationEnd.getId());
        operationQueue.put(insertRangeObject);
    }

    public void addInsertHeaderFooterOperation(String id, String type)
        throws JSONException {

    	final JSONObject insertHeaderFooterObject = new JSONObject(3);
    	insertHeaderFooterObject.put(OCKey.NAME.value(), OCValue.INSERT_HEADER_FOOTER.value());
    	insertHeaderFooterObject.put(OCKey.ID.value(), id);
    	insertHeaderFooterObject.put(OCKey.TYPE.value(), type);
        operationQueue.put(insertHeaderFooterObject);
    }
}
