/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/render/rectangle'
], function (Rectangle) {

    'use strict';

    // private global functions ===============================================

    function rad(a) { return a * Math.PI / 180; }

    function expectPoint(point, expX, expY) {
        expect(point).to.have.a.property('x').that.is.closeTo(expX, 1e-12);
        expect(point).to.have.a.property('y').that.is.closeTo(expY, 1e-12);
    }

    function expectPolar(polar, expR, expA) {
        expect(polar).to.have.a.property('r').that.is.closeTo(expR, 1e-12);
        expect(polar).to.have.a.property('a').that.is.closeTo(rad(expA), 1e-12);
    }

    // class Rectangle ========================================================

    describe('Toolkit class Rectangle', function () {

        it('should exist', function () {
            expect(Rectangle).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        function r(l, t, w, h) {
            return new Rectangle(l, t, w, h);
        }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should take all parameters', function () {
                var rect = new Rectangle(-1, 1, 10, 11.5);
                expect(rect).to.have.a.property('left', -1);
                expect(rect).to.have.a.property('top', 1);
                expect(rect).to.have.a.property('width', 10);
                expect(rect).to.have.a.property('height', 11.5);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "from"', function () {
            it('should exist', function () {
                expect(Rectangle).itself.to.respondTo('from');
            });
            it('should create a rectangle', function () {
                var rect = Rectangle.from({ left: -1, top: 1, width: 10, height: 11.5 });
                expect(rect).to.be.an.instanceof(Rectangle);
                expect(rect).to.have.a.property('left', -1);
                expect(rect).to.have.a.property('top', 1);
                expect(rect).to.have.a.property('width', 10);
                expect(rect).to.have.a.property('height', 11.5);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var rect1 = r(0, 1, 2, 3);
                var rect2 = rect1.clone();
                expect(rect2).to.not.equal(rect1);
                expect(rect2).to.deep.equal(rect1);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('equals');
            });
            var rect = r(0, 1, 2, 3);
            it('should return whether the rectangles are equal', function () {
                expect(rect.equals(rect)).to.equal(true);
                expect(rect.equals(r(0, 1, 2, 3))).to.equal(true);
                expect(rect.equals(r(9, 1, 2, 3))).to.equal(false);
                expect(rect.equals(r(0, 9, 2, 3))).to.equal(false);
                expect(rect.equals(r(0, 1, 9, 3))).to.equal(false);
                expect(rect.equals(r(0, 1, 2, 9))).to.equal(false);
            });
            it('should accept an object', function () {
                expect(rect.equals({ left: 0, top: 1, width: 2, height: 3 })).to.equal(true);
            });
        });

        describe('method "differs"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('differs');
            });
            var rect = r(0, 1, 2, 3);
            it('should return whether the rectangles are different', function () {
                expect(rect.differs(rect)).to.equal(false);
                expect(rect.differs(r(0, 1, 2, 3))).to.equal(false);
                expect(rect.differs(r(9, 1, 2, 3))).to.equal(true);
                expect(rect.differs(r(0, 9, 2, 3))).to.equal(true);
                expect(rect.differs(r(0, 1, 9, 3))).to.equal(true);
                expect(rect.differs(r(0, 1, 2, 9))).to.equal(true);
            });
            it('should accept an object', function () {
                expect(rect.differs({ left: 0, top: 1, width: 2, height: 3 })).to.equal(false);
            });
        });

        describe('method "right"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('right');
            });
            it('should return the position of the right border', function () {
                expect(r(1, 2, 5, 5).right()).to.equal(6);
            });
        });

        describe('method "bottom"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('bottom');
            });
            it('should return the position of the lower border', function () {
                expect(r(1, 2, 5, 5).bottom()).to.equal(7);
            });
        });

        describe('method "centerX"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('centerX');
            });
            it('should return the X coordinate of the center', function () {
                expect(r(1, 2, 5, 5).centerX()).to.equal(3.5);
            });
        });

        describe('method "centerY"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('centerY');
            });
            it('should return the Y coordinate of the center', function () {
                expect(r(1, 2, 5, 5).centerY()).to.equal(4.5);
            });
        });

        describe('method "diag"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('diag');
            });
            it('should return the length of the diagonal', function () {
                expect(r(1, 2, 3, 4).diag()).to.equal(5);
            });
        });

        describe('method "area"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('area');
            });
            it('should return the area size', function () {
                expect(r(1, 2, 3, 4).area()).to.equal(12);
            });
        });

        describe('method "square"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('square');
            });
            it('should return whether a rectangle is a square', function () {
                expect(r(1, 2, 3, 4).square()).to.equal(false);
                expect(r(1, 2, 3, 3).square()).to.equal(true);
                expect(r(1, 2, 4, 3).square()).to.equal(false);
            });
        });

        describe('method "topLeft"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('topLeft');
            });
            it('should return the top-left point', function () {
                expect(r(1, 2, 3, 4).topLeft()).to.deep.equal({ x: 1, y: 2 });
            });
        });

        describe('method "topRight"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('topRight');
            });
            it('should return the top-right point', function () {
                expect(r(1, 2, 3, 4).topRight()).to.deep.equal({ x: 4, y: 2 });
            });
        });

        describe('method "bottomLeft"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('bottomLeft');
            });
            it('should return the bottom-left point', function () {
                expect(r(1, 2, 3, 4).bottomLeft()).to.deep.equal({ x: 1, y: 6 });
            });
        });

        describe('method "bottomRight"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('bottomRight');
            });
            it('should return the bottom-right point', function () {
                expect(r(1, 2, 3, 4).bottomRight()).to.deep.equal({ x: 4, y: 6 });
            });
        });

        describe('method "leftCenter"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('leftCenter');
            });
            it('should return the left center point', function () {
                expect(r(1, 2, 3, 4).leftCenter()).to.deep.equal({ x: 1, y: 4 });
            });
        });

        describe('method "rightCenter"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('rightCenter');
            });
            it('should return the right center point', function () {
                expect(r(1, 2, 3, 4).rightCenter()).to.deep.equal({ x: 4, y: 4 });
            });
        });

        describe('method "topCenter"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('topCenter');
            });
            it('should return the top center point', function () {
                expect(r(1, 2, 3, 4).topCenter()).to.deep.equal({ x: 2.5, y: 2 });
            });
        });

        describe('method "bottomCenter"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('bottomCenter');
            });
            it('should return the bottom center point', function () {
                expect(r(1, 2, 3, 4).bottomCenter()).to.deep.equal({ x: 2.5, y: 6 });
            });
        });

        describe('method "center"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('center');
            });
            it('should return the center point', function () {
                expect(r(1, 2, 3, 4).center()).to.deep.equal({ x: 2.5, y: 4 });
            });
        });

        describe('method "containsX"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('containsX');
            });
            it('should return whether the X coordinate is inside', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.containsX(0.5)).to.equal(false);
                expect(rect.containsX(1)).to.equal(true);
                expect(rect.containsX(3)).to.equal(true);
                expect(rect.containsX(6)).to.equal(true);
                expect(rect.containsX(6.5)).to.equal(false);
            });
        });

        describe('method "containsY"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('containsY');
            });
            it('should return whether the Y coordinate is inside', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.containsY(1.5)).to.equal(false);
                expect(rect.containsY(2)).to.equal(true);
                expect(rect.containsY(4)).to.equal(true);
                expect(rect.containsY(7)).to.equal(true);
                expect(rect.containsY(7.5)).to.equal(false);
            });
        });

        describe('method "containsPoint"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('containsPoint');
            });
            var rect = r(1, 2, 5, 5);
            it('should return whether the point is inside', function () {
                expect(rect.containsPoint(0.5, 2)).to.equal(false);
                expect(rect.containsPoint(1, 1.5)).to.equal(false);
                expect(rect.containsPoint(1, 2)).to.equal(true);
                expect(rect.containsPoint(3, 4)).to.equal(true);
                expect(rect.containsPoint(6, 7)).to.equal(true);
                expect(rect.containsPoint(6.5, 7)).to.equal(false);
                expect(rect.containsPoint(6, 7.5)).to.equal(false);
            });
            it('should accept an object', function () {
                expect(rect.containsPoint({ x: 3, y: 4 })).to.equal(true);
            });
        });

        describe('method "containsPixel"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('containsPixel');
            });
            var rect = r(1, 2, 5, 5);
            it('should return whether the pixel is inside', function () {
                expect(rect.containsPixel(0.5, 2)).to.equal(false);
                expect(rect.containsPixel(1, 1.5)).to.equal(false);
                expect(rect.containsPixel(1, 2)).to.equal(true);
                expect(rect.containsPixel(3, 4)).to.equal(true);
                expect(rect.containsPixel(5, 6)).to.equal(true);
                expect(rect.containsPixel(5.5, 6)).to.equal(false);
                expect(rect.containsPixel(5, 6.5)).to.equal(false);
            });
            it('should accept an object', function () {
                expect(rect.containsPixel({ x: 3, y: 4 })).to.equal(true);
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('contains');
            });
            var rect = r(1, 2, 5, 5);
            it('should return whether the rectangle is inside', function () {
                expect(rect.contains(r(0.5, 2, 3, 3))).to.equal(false);
                expect(rect.contains(r(1, 1.5, 3, 3))).to.equal(false);
                expect(rect.contains(r(1, 2, 3, 3))).to.equal(true);
                expect(rect.contains(r(3, 4, 3, 3))).to.equal(true);
                expect(rect.contains(r(3.5, 4, 3, 3))).to.equal(false);
                expect(rect.contains(r(3, 4.5, 3, 3))).to.equal(false);
            });
            it('should accept a plain JS object', function () {
                expect(rect.contains({ left: 1, top: 2, width: 3, height: 3 })).to.equal(true);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('overlaps');
            });
            var rect = r(1, 2, 5, 5);
            it('should return whether the rectangles overlap', function () {
                expect(rect.overlaps(r(0, 3, 1, 1))).to.equal(false);
                expect(rect.overlaps(r(0.5, 3, 1, 1))).to.equal(true);
                expect(rect.overlaps(r(5.5, 3, 1, 1))).to.equal(true);
                expect(rect.overlaps(r(6, 3, 1, 1))).to.equal(false);
                expect(rect.overlaps(r(2, 1, 1, 1))).to.equal(false);
                expect(rect.overlaps(r(2, 1.5, 1, 1))).to.equal(true);
                expect(rect.overlaps(r(2, 6.5, 1, 1))).to.equal(true);
                expect(rect.overlaps(r(2, 7, 1, 1))).to.equal(false);
                expect(rect.overlaps(r(3, 3, 1, 1))).to.equal(true);
                expect(rect.overlaps(r(0, 3, 10, 1))).to.equal(true);
                expect(rect.overlaps(r(3, 0, 1, 10))).to.equal(true);
                expect(rect.overlaps(r(0, 0, 10, 10))).to.equal(true);
            });
            it('should accept a plain JS object', function () {
                expect(rect.overlaps({ left: 3, top: 3, width: 1, height: 1 })).to.equal(true);
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('boundary');
            });
            var rect = r(1, 2, 5, 5);
            it('should return the bounding rectangle', function () {
                expect(rect.boundary(r(0, 0, 1, 1))).to.deep.equal(r(0, 0, 6, 7));
                expect(rect.boundary(r(4, 4, 1, 1))).to.deep.equal(r(1, 2, 5, 5));
                expect(rect.boundary(r(8, 8, 1, 1))).to.deep.equal(r(1, 2, 8, 7));
            });
            it('should accept a plain JS object', function () {
                expect(rect.boundary({ left: 0, top: 0, width: 1, height: 1 })).to.deep.equal(r(0, 0, 6, 7));
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('intersect');
            });
            var rect = r(1, 2, 5, 5);
            it('should return the common area', function () {
                expect(rect.intersect(r(0, 3, 1, 1))).to.equal(null);
                expect(rect.intersect(r(0.5, 3, 1, 1))).to.deep.equal(r(1, 3, 0.5, 1));
                expect(rect.intersect(r(5.5, 3, 1, 1))).to.deep.equal(r(5.5, 3, 0.5, 1));
                expect(rect.intersect(r(6, 3, 1, 1))).to.equal(null);
                expect(rect.intersect(r(2, 1, 1, 1))).to.equal(null);
                expect(rect.intersect(r(2, 1.5, 1, 1))).to.deep.equal(r(2, 2, 1, 0.5));
                expect(rect.intersect(r(2, 6.5, 1, 1))).to.deep.equal(r(2, 6.5, 1, 0.5));
                expect(rect.intersect(r(2, 7, 1, 1))).to.equal(null);
                expect(rect.intersect(r(3, 3, 1, 1))).to.deep.equal(r(3, 3, 1, 1));
                expect(rect.intersect(r(0, 3, 10, 1))).to.deep.equal(r(1, 3, 5, 1));
                expect(rect.intersect(r(3, 0, 1, 10))).to.deep.equal(r(3, 2, 1, 5));
                expect(rect.intersect(r(0, 0, 10, 10))).to.deep.equal(r(1, 2, 5, 5));
            });
            it('should accept a plain JS object', function () {
                expect(rect.intersect({ left: 3, top: 3, width: 1, height: 1 })).to.deep.equal(r(3, 3, 1, 1));
            });
        });

        describe('method "remaining"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('remaining');
            });
            it('should return the remaining areas', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.remaining(r(0, 0, 1, 1))).to.deep.equal([r(1, 2, 5, 5)]);
                expect(rect.remaining(r(0, 0, 3, 3))).to.deep.equal([r(3, 2, 3, 1), r(1, 3, 5, 4)]);
                expect(rect.remaining(r(2, 0, 3, 3))).to.deep.equal([r(1, 2, 1, 1), r(5, 2, 1, 1), r(1, 3, 5, 4)]);
                expect(rect.remaining(r(4, 0, 3, 3))).to.deep.equal([r(1, 2, 3, 1), r(1, 3, 5, 4)]);
                expect(rect.remaining(r(0, 3, 3, 3))).to.deep.equal([r(1, 2, 5, 1), r(3, 3, 3, 3), r(1, 6, 5, 1)]);
                expect(rect.remaining(r(2, 3, 3, 3))).to.deep.equal([r(1, 2, 5, 1), r(1, 3, 1, 3), r(5, 3, 1, 3), r(1, 6, 5, 1)]);
                expect(rect.remaining(r(4, 3, 3, 3))).to.deep.equal([r(1, 2, 5, 1), r(1, 3, 3, 3), r(1, 6, 5, 1)]);
                expect(rect.remaining(r(0, 6, 3, 3))).to.deep.equal([r(1, 2, 5, 4), r(3, 6, 3, 1)]);
                expect(rect.remaining(r(2, 6, 3, 3))).to.deep.equal([r(1, 2, 5, 4), r(1, 6, 1, 1), r(5, 6, 1, 1)]);
                expect(rect.remaining(r(4, 6, 3, 3))).to.deep.equal([r(1, 2, 5, 4), r(1, 6, 3, 1)]);
                expect(rect.remaining(r(0, 0, 9, 3))).to.deep.equal([r(1, 3, 5, 4)]);
                expect(rect.remaining(r(0, 3, 9, 3))).to.deep.equal([r(1, 2, 5, 1), r(1, 6, 5, 1)]);
                expect(rect.remaining(r(0, 6, 9, 3))).to.deep.equal([r(1, 2, 5, 4)]);
                expect(rect.remaining(r(0, 0, 3, 9))).to.deep.equal([r(3, 2, 3, 5)]);
                expect(rect.remaining(r(2, 0, 3, 9))).to.deep.equal([r(1, 2, 1, 5), r(5, 2, 1, 5)]);
                expect(rect.remaining(r(4, 0, 3, 9))).to.deep.equal([r(1, 2, 3, 5)]);
                expect(rect.remaining(r(0, 0, 9, 9))).to.deep.equal([]);
            });
        });

        describe('method "pointToPolar"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('pointToPolar');
            });
            var rect = r(1, 2, 4, 6);
            it('should return the polar coordinates of a point', function () {
                var d = Math.sqrt(12.5);
                expect(rect.pointToPolar(3, 5)).to.deep.equal({ r: 0, a: 0 });
                expectPolar(rect.pointToPolar(8, 5), 5, 0);
                expectPolar(rect.pointToPolar(3 + d, 5 + d), 5, 45);
                expectPolar(rect.pointToPolar(3, 10), 5, 90);
                expectPolar(rect.pointToPolar(3 - d, 5 + d), 5, 135);
                expectPolar(rect.pointToPolar(-2, 5), 5, 180);
                expectPolar(rect.pointToPolar(3 + d, 5 - d), 5, -45);
                expectPolar(rect.pointToPolar(3, 0), 5, -90);
                expectPolar(rect.pointToPolar(3 - d, 5 - d), 5, -135);
            });
            it('should accept an object', function () {
                expect(rect.pointToPolar({ x: 3, y: 5 })).to.deep.equal({ r: 0, a: 0 });
            });
        });

        describe('method "polarToPoint"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('polarToPoint');
            });
            var rect = r(1, 2, 4, 6);
            it('should return the coordinates of a point', function () {
                var d = Math.sqrt(12.5);
                expect(rect.polarToPoint(0, 0)).to.deep.equal({ x: 3, y: 5 });
                expectPoint(rect.polarToPoint(5, rad(0)), 8, 5);
                expectPoint(rect.polarToPoint(5, rad(45)), 3 + d, 5 + d);
                expectPoint(rect.polarToPoint(5, rad(90)), 3, 10);
                expectPoint(rect.polarToPoint(5, rad(135)), 3 - d, 5 + d);
                expectPoint(rect.polarToPoint(5, rad(180)), -2, 5);
                expectPoint(rect.polarToPoint(5, rad(225)), 3 - d, 5 - d);
                expectPoint(rect.polarToPoint(5, rad(270)), 3, 0);
                expectPoint(rect.polarToPoint(5, rad(315)), 3 + d, 5 - d);
            });
            it('should accept an object', function () {
                expect(rect.polarToPoint({ r: 0, a: 0 })).to.deep.equal({ x: 3, y: 5 });
            });
        });

        describe('method "rotatePoint"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('rotatePoint');
            });
            var rect = r(1, 2, 4, 6);
            it('should rotate the point', function () {
                var d = Math.sqrt(12.5);
                expect(rect.rotatePoint(0, 3, 5)).to.deep.equal({ x: 3, y: 5 });
                expectPoint(rect.rotatePoint(rad(0), 3 + d, 5 + d), 3 + d, 5 + d);
                expectPoint(rect.rotatePoint(rad(45), 3 + d, 5 + d), 3, 10);
                expectPoint(rect.rotatePoint(rad(90), 3 + d, 5 + d), 3 - d, 5 + d);
                expectPoint(rect.rotatePoint(rad(135), 3 + d, 5 + d), -2, 5);
                expectPoint(rect.rotatePoint(rad(180), 3 + d, 5 + d), 3 - d, 5 - d);
                expectPoint(rect.rotatePoint(rad(225), 3 + d, 5 + d), 3, 0);
                expectPoint(rect.rotatePoint(rad(270), 3 + d, 5 + d), 3 + d, 5 - d);
                expectPoint(rect.rotatePoint(rad(315), 3 + d, 5 + d), 8, 5);
            });
            it('should accept an object', function () {
                expect(rect.rotatePoint(0, { x: 3, y: 5 })).to.deep.equal({ x: 3, y: 5 });
            });
        });

        describe('method "rotatedBoundingBox"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('rotatedBoundingBox');
            });
            it('should return the rotated bounding box', function () {
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(0)).roundSelf()).to.deep.equal(r(100, 50, 1000, 500));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(30)).roundSelf()).to.deep.equal(r(42, -167, 1116, 933));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(60)).roundSelf()).to.deep.equal(r(133, -258, 933, 1116));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(90)).roundSelf()).to.deep.equal(r(350, -200, 500, 1000));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(110)).roundSelf()).to.deep.equal(r(194, -255, 812, 1111));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(155)).roundSelf()).to.deep.equal(r(41, -138, 1118, 876));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(180)).roundSelf()).to.deep.equal(r(100, 50, 1000, 500));
                expect(r(100, 50, 1000, 500).rotatedBoundingBox(rad(225)).roundSelf()).to.deep.equal(r(70, -230, 1061, 1061));
            });
        });

        describe('method "set"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('set');
            });
            it('should set new coordinates', function () {
                var rect = r(0, 0, 0, 0);
                expect(rect.set(1, 2, 3, 4)).to.equal(rect);
                expect(rect).to.deep.equal(r(1, 2, 3, 4));
            });
        });

        describe('method "assign"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('assign');
            });
            it('should set new coordinates', function () {
                var rect1 = r(0, 0, 0, 0), rect2 = r(1, 2, 3, 4);
                expect(rect1.assign(rect2)).to.equal(rect1);
                expect(rect1).to.deep.equal(r(1, 2, 3, 4));
            });
        });

        describe('method "expandSelf"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('expandSelf');
            });
            it('should expand the rectangle in-place', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.expandSelf(1, 2, 3, 4)).to.equal(rect);
                expect(rect.left).to.equal(0);
                expect(rect.top).to.equal(0);
                expect(rect.width).to.equal(9);
                expect(rect.height).to.equal(11);
            });
            it('should shrink the rectangle in-place', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.expandSelf(-1, -2, -3, -4)).to.equal(rect);
                expect(rect.left).to.equal(2);
                expect(rect.top).to.equal(4);
                expect(rect.width).to.equal(1);
                expect(rect.height).to.equal(0);
            });
            it('should support omitted parameters', function () {
                var rect = r(1, 2, 5, 5);
                rect.expandSelf(1);
                expect(rect.left).to.equal(0);
                expect(rect.top).to.equal(1);
                expect(rect.width).to.equal(7);
                expect(rect.height).to.equal(7);
                rect.expandSelf(-1, -2);
                expect(rect.left).to.equal(1);
                expect(rect.top).to.equal(3);
                expect(rect.width).to.equal(5);
                expect(rect.height).to.equal(3);
                rect.expandSelf(1, 2, 3);
                expect(rect.left).to.equal(0);
                expect(rect.top).to.equal(1);
                expect(rect.width).to.equal(9);
                expect(rect.height).to.equal(7);
            });
        });

        describe('method "translateSelf"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('translateSelf');
            });
            it('should translate the rectangle in-place', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.translateSelf(2, -0.5)).to.equal(rect);
                expect(rect.left).to.equal(3);
                expect(rect.top).to.equal(1.5);
                expect(rect.width).to.equal(5);
                expect(rect.height).to.equal(5);
            });
        });

        describe('method "scaleSelf"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('scaleSelf');
            });
            it('should scale the rectangle in-place', function () {
                var rect = r(1, 2, 5, 5);
                expect(rect.scaleSelf(2, 0.5)).to.equal(rect);
                expect(rect.left).to.equal(2);
                expect(rect.top).to.equal(1);
                expect(rect.width).to.equal(10);
                expect(rect.height).to.equal(2.5);
            });
        });

        describe('method "rotateSelf"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('rotateSelf');
            });
            it('should rotate the rectangle in-place', function () {
                var rect = r(1, 2, 5, 6);
                expect(rect.rotateSelf()).to.equal(rect);
                expect(rect.left).to.equal(0.5);
                expect(rect.top).to.equal(2.5);
                expect(rect.width).to.equal(6);
                expect(rect.height).to.equal(5);
            });
        });

        describe('method "roundSelf"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('roundSelf');
            });
            it('should round the coordinates in-place', function () {
                var rect = r(1.2, 2.5, 5.8, 5);
                expect(rect.roundSelf()).to.equal(rect);
                expect(rect.left).to.equal(1);
                expect(rect.top).to.equal(3);
                expect(rect.width).to.equal(6);
                expect(rect.height).to.equal(5);
            });
        });

        describe('method "toCSS"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('toCSS');
            });
            var rect = r(1, 2, 3, 4);
            it('should return the CSS properties', function () {
                var css = rect.toCSS();
                expect(css).to.be.an('object');
                expect(css).to.not.be.an.instanceof(Rectangle);
                expect(css).to.deep.equal({ left: '1px', top: '2px', width: '3px', height: '4px' });
            });
            it('should take an arbitrary unit', function () {
                expect(rect.toCSS('em')).to.deep.equal({ left: '1em', top: '2em', width: '3em', height: '4em' });
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('toString');
            });
            it('should stringify the rectangle', function () {
                expect(r(1, 2, 3, 4).toString()).to.be.a('string');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Rectangle).to.respondTo('toJSON');
            });
            var rect = r(1, 2, 3, 4);
            var exp = { left: 1, top: 2, width: 3, height: 4 };
            it('should return the JSON representation', function () {
                var json = rect.toJSON();
                expect(json).to.be.an('object');
                expect(json).to.not.be.an.instanceof(Rectangle);
                expect(json).to.deep.equal(exp);
            });
            it('should stringify implicitly', function () {
                expect(JSON.parse(JSON.stringify(rect))).to.deep.equal(exp);
            });
        });
    });

    // ========================================================================
});
