/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/intervalarray'
], function (SheetHelper, IntervalArray) {

    'use strict';

    // class IntervalArray ====================================================

    describe('Spreadsheet class IntervalArray', function () {

        it('should exist', function () {
            expect(IntervalArray).to.be.a('function');
        });
        it('should be a subclass of Array', function () {
            expect(new IntervalArray()).to.be.an.instanceof(Array);
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;
        var ia = SheetHelper.ia;
        var orderedMatcher = SheetHelper.orderedIntervalsMatcher;
        var unorderedMatcher = SheetHelper.unorderedIntervalsMatcher;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create an interval array', function () {
                var ia1 = new IntervalArray();
                expect(ia1).to.be.an.instanceof(IntervalArray);
                expect(ia1).to.be.empty;
            });
            var i1 = i('1:2'), i2 = i('3:4');
            it('should insert single intervals', function () {
                var ia1 = new IntervalArray(i1);
                expect(ia1).to.have.length(1);
                expect(ia1[0]).to.equal(i1);
                var ia2 = new IntervalArray(i1, i2, i1);
                expect(ia2).to.have.length(3);
                expect(ia2[0]).to.equal(i1);
                expect(ia2[1]).to.equal(i2);
                expect(ia2[2]).to.equal(i1);
            });
            it('should insert plain arrays of intervals', function () {
                var ia1 = new IntervalArray([i1]);
                expect(ia1).to.have.length(1);
                expect(ia1[0]).to.equal(i1);
                var ia2 = new IntervalArray([i1, i2], [], [i1]);
                expect(ia2).to.have.length(3);
                expect(ia2[0]).to.equal(i1);
                expect(ia2[1]).to.equal(i2);
                expect(ia2[2]).to.equal(i1);
            });
            it('should copy-construct interval arrays', function () {
                var ia0 = new IntervalArray(i1, i2),
                    ia1 = new IntervalArray(ia0);
                expect(ia1).to.have.length(2);
                expect(ia1[0]).to.equal(i1);
                expect(ia1[1]).to.equal(i2);
                var ia2 = new IntervalArray(ia0, new IntervalArray(), ia1);
                expect(ia2).to.have.length(4);
                expect(ia2[0]).to.equal(i1);
                expect(ia2[1]).to.equal(i2);
                expect(ia2[2]).to.equal(i1);
                expect(ia2[3]).to.equal(i2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "mergeIndexes"', function () {
            it('should exist', function () {
                expect(IntervalArray).itself.to.respondTo('mergeIndexes');
            });
            it('should join indexes to intervals', function () {
                expect(IntervalArray.mergeIndexes([1, 3, 4, 2, 1])).to.deep.equal(ia('2:5'));
                expect(IntervalArray.mergeIndexes([5, 3, 5, 1, 2])).to.deep.equal(ia('2:4 6:6'));
            });
            it('should accept an empty array', function () {
                expect(IntervalArray.mergeIndexes([])).to.be.empty;
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('key');
            });
            it('should return unique key for the array', function () {
                expect(ia('2:4 3:5').key()).to.equal('1:3 2:4');
                expect(ia('').key()).to.equal('');
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('clone');
            });
            it('should return a shallow clone', function () {
                var ia1 = ia('2:4 3:5'), ia2 = ia1.clone();
                expect(ia2).to.be.an.instanceof(IntervalArray);
                expect(ia2).to.not.equal(ia1);
                expect(ia2[0]).to.equal(ia1[0]);
                expect(ia2[1]).to.equal(ia1[1]);
                expect(ia2).to.deep.equal(ia1);
            });
            it('should return a deep clone', function () {
                var ia1 = ia('2:4 3:5'), ia2 = ia1.clone(true);
                expect(ia2).to.be.an.instanceof(IntervalArray);
                expect(ia2).to.not.equal(ia1);
                expect(ia2[0]).to.not.equal(ia1[0]);
                expect(ia2[1]).to.not.equal(ia1[1]);
                expect(ia2).to.deep.equal(ia1);
            });
        });

        describe('method "size"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('size');
            });
            it('should count indexes in intervals', function () {
                expect(ia('1:3 2:5 7:7').size()).to.equal(8);
                expect(new IntervalArray().size()).to.equal(0);
            });
        });

        describe('method "containsIndex"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('containsIndex');
            });
            var ia1 = ia('2:3 5:7');
            it('should return false for indexes outside the intervals', function () {
                expect(ia1.containsIndex(0)).to.equal(false);
                expect(ia1.containsIndex(3)).to.equal(false);
                expect(ia1.containsIndex(7)).to.equal(false);
                expect(ia1.containsIndex(8)).to.equal(false);
                expect(ia1.containsIndex(9)).to.equal(false);
            });
            it('should return true for indexes inside the intervals', function () {
                expect(ia1.containsIndex(1)).to.equal(true);
                expect(ia1.containsIndex(2)).to.equal(true);
                expect(ia1.containsIndex(4)).to.equal(true);
                expect(ia1.containsIndex(5)).to.equal(true);
                expect(ia1.containsIndex(6)).to.equal(true);
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('contains');
            });
            var ia1 = ia('2:3 5:7');
            it('should return false for intervals outside the intervals', function () {
                expect(ia1.contains(ia('1:1 2:3 5:7'))).to.equal(false);
                expect(ia1.contains(ia('2:3 5:7 4:4'))).to.equal(false);
                expect(ia1.contains(ia('8:9 2:3 5:7'))).to.equal(false);
            });
            it('should return false for overlapping intervals', function () {
                expect(ia1.contains(ia('1:3 5:7'))).to.equal(false);
                expect(ia1.contains(ia('2:4 5:7'))).to.equal(false);
                expect(ia1.contains(ia('2:3 4:7'))).to.equal(false);
                expect(ia1.contains(ia('2:3 5:8'))).to.equal(false);
            });
            it('should return true for contained intervals', function () {
                expect(ia1.contains(ia('5:6 6:7 2:3'))).to.equal(true);
                expect(ia1.contains(ia1)).to.equal(true);
            });
            it('should accept empty array', function () {
                var ia0 = new IntervalArray();
                expect(ia0.contains(ia1)).to.equal(false);
                expect(ia1.contains(ia0)).to.equal(true);
                expect(ia0.contains(ia0)).to.equal(true);
            });
            it('should accept single interval', function () {
                expect(ia1.contains(i('3:5'))).to.equal(false);
                expect(ia1.contains(i('2:3'))).to.equal(true);
                expect(ia1.contains(i('6:7'))).to.equal(true);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('overlaps');
            });
            var ia1 = ia('2:3 5:7');
            it('should return false for distinct intervals', function () {
                expect(ia1.overlaps(ia('1:1 4:4 8:9'))).to.equal(false);
            });
            it('should return true for overlapping intervals', function () {
                expect(ia1.overlaps(ia('1:1 4:5 8:9'))).to.equal(true);
                expect(ia1.overlaps(ia('4:4 1:2 8:9'))).to.equal(true);
                expect(ia1.overlaps(ia('1:1 4:4 7:9'))).to.equal(true);
            });
            it('should accept empty array', function () {
                var ia0 = new IntervalArray();
                expect(ia0.overlaps(ia)).to.equal(false);
                expect(ia1.overlaps(ia0)).to.equal(false);
                expect(ia0.overlaps(ia0)).to.equal(false);
            });
            it('should accept single interval', function () {
                expect(ia1.overlaps(i('4:4'))).to.equal(false);
                expect(ia1.overlaps(i('7:9'))).to.equal(true);
            });
        });

        describe('method "overlapsSelf"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('overlapsSelf');
            });
            it('should return false for distinct intervals', function () {
                expect(ia('5:7 2:3 4:4 1:1').overlapsSelf()).to.equal(false);
            });
            it('should return true for overlapping intervals', function () {
                expect(ia('5:7 2:3 4:4 1:2').overlapsSelf()).to.equal(true);
                expect(ia('5:7 2:3 4:5 1:1').overlapsSelf()).to.equal(true);
                expect(ia('5:7 1:3 4:4 1:1').overlapsSelf()).to.equal(true);
                expect(ia('4:7 2:3 4:4 1:1').overlapsSelf()).to.equal(true);
            });
            it('should accept empty array', function () {
                expect(new IntervalArray().overlapsSelf()).to.equal(false);
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('boundary');
            });
            it('should return the bounding interval', function () {
                expect(ia('1:3 5:7').boundary()).to.deep.equal(i('1:7'));
                expect(ia('3:7 1:5').boundary()).to.deep.equal(i('1:7'));
            });
            it('should return a clone of a single interval', function () {
                var ia1 = ia('1:3'), i1 = ia1.boundary();
                expect(i1).to.deep.equal(ia1[0]);
                expect(i1).to.not.equal(ia1[0]);
            });
            it('should return null for an empty array', function () {
                expect(new IntervalArray().boundary()).to.equal(null);
            });
        });

        describe('method "unify"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('unify');
            });
            it('should return a shallow copy', function () {
                var ia1 = ia('1:2 4:5'), ia2 = ia1.unify();
                expect(ia2).to.not.equal(ia1);
                expect(ia2[0]).to.equal(ia1[0]);
                expect(ia2[1]).to.equal(ia1[1]);
            });
            it('should remove all duplicates', function () {
                var ia1 = ia('2:3 3:4 2:3 2:4 3:4 2:4'), ia2 = ia1.unify();
                expect(ia2).to.have.length(3);
                expect(ia2[0]).to.equal(ia1[0]);
                expect(ia2[1]).to.equal(ia1[1]);
                expect(ia2[2]).to.equal(ia1[3]);
            });
            it('should accept empty array', function () {
                expect(new IntervalArray().unify()).to.be.empty;
            });
        });

        describe('method "merge"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('merge');
            });
            it('should return a new array', function () {
                var ia1 = ia('1:2 4:5'), ia2 = ia1.merge();
                expect(ia2).to.deep.equal(ia1);
                expect(ia2).to.not.equal(ia1);
            });
            it('should sort but not merge distinct intervals', function () {
                expect(ia('5:6 2:3 8:9').merge()).to.deep.equal(ia('2:3 5:6 8:9'));
            });
            it('should merge adjacent intervals', function () {
                expect(ia('4:5 2:3 6:7 8:9').merge()).to.deep.equal(ia('2:9'));
            });
            it('should merge overlapping intervals', function () {
                expect(ia('4:7 2:4 4:7 7:9').merge()).to.deep.equal(ia('2:9'));
            });
            it('should accept empty array', function () {
                expect(new IntervalArray().merge()).to.be.empty;
            });
        });

        describe('method "partition"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('partition');
            });
            it('should not partition distinct intervals', function () {
                expect(ia('7:8 1:2 4:5').partition()).to.satisfy(orderedMatcher('1:2 4:5 7:8'));
            });
            it('should not partition adjacent ranges', function () {
                expect(ia('1:2 3:4').partition()).to.satisfy(orderedMatcher('1:2 3:4'));
                expect(ia('3:4 1:2').partition()).to.satisfy(orderedMatcher('1:2 3:4'));
            });
            it('should partition overlapping intervals', function () {
                expect(ia('1:4 3:6').partition()).to.satisfy(orderedMatcher('1:2 3:4 5:6'));
                expect(ia('1:3 5:7 3:5').partition()).to.satisfy(orderedMatcher('1:2 3:3 4:4 5:5 6:7'));
                expect(ia('3:4 1:6 3:4').partition()).to.satisfy(orderedMatcher('1:2 3:4 5:6'));
            });
            it('should contain the original intervals', function () {
                var intervals = ia('1:4 3:6');
                var partition = intervals.partition();
                expect(partition[0].coveredBy).to.satisfy(unorderedMatcher('1:4'));
                expect(partition[1].coveredBy).to.satisfy(unorderedMatcher('1:4 3:6'));
                expect(partition[2].coveredBy).to.satisfy(unorderedMatcher('3:6'));
            });
            it('should clone all intervals', function () {
                var i1 = i('3:5'), intervals = new IntervalArray(i1).partition();
                expect(intervals).to.satisfy(orderedMatcher('3:5'));
                expect(intervals[0]).to.not.equal(i1);
            });
            it('should accept empty array', function () {
                var ia1 = new IntervalArray(), ia2 = ia1.partition();
                expect(ia2).to.be.empty;
                expect(ia2).to.not.equal(ia1);
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('intersect');
            });
            it('should return empty array for distinct intervals', function () {
                expect(ia('1:2 5:6 9:10').intersect(ia('3:4 7:8'))).to.be.empty;
                expect(ia('1:2 5:6 9:10').intersect(i('3:4'))).to.be.empty;
            });
            it('should accept empty arrays', function () {
                expect(new IntervalArray().intersect(ia('3:4'))).to.be.empty;
                expect(ia('3:4').intersect(new IntervalArray())).to.be.empty;
            });
            it('should return intersection intervals', function () {
                var ia1 = ia('1:3 2:4 3:5 4:6 5:7');
                expect(ia1.intersect(ia('2:4 4:6'))).to.deep.equal(ia('2:3 2:4 4:4 3:4 4:5 4:4 4:6 5:6'));
                expect(ia1.intersect(i('2:4'))).to.deep.equal(ia('2:3 2:4 3:4 4:4'));
            });
        });

        describe('method "difference"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('difference');
            });
            it('should return remaining intervals', function () {
                expect(ia('2:4 6:8 10:12 14:16').difference(ia('4:6 11:11 14:16'))).to.deep.equal(ia('2:3 7:8 10:10 12:12'));
                expect(ia('2:6 4:10').difference(ia('5:5'))).to.deep.equal(ia('2:4 6:6 4:4 6:10'));
                expect(ia('2:6 4:10').difference(ia('9:9 5:6'))).to.deep.equal(ia('2:4 4:4 7:8 10:10'));
            });
            it('should accept empty array', function () {
                var ia0 = new IntervalArray(), ia1 = ia('4:6'), ia2 = ia1.difference(ia0);
                expect(ia2).to.deep.equal(ia1);
                expect(ia2).to.not.equal(ia1);
                expect(ia0.difference(ia1)).to.be.empty;
                expect(ia0.difference(ia0)).to.be.empty;
                expect(ia0.difference(ia0)).to.not.equal(ia0);
            });
            it('should accept single interval object instead of array', function () {
                expect(ia('2:6 4:10').difference(i('5:5'))).to.deep.equal(ia('2:4 6:6 4:4 6:10'));
            });
        });

        describe('method "move"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('move');
            });
            it('should move the indexes in the intervals', function () {
                expect(ia('5:6 7:8').move(-3)).to.deep.equal(ia('2:3 4:5'));
                expect(ia('5:6 7:8').move(-2)).to.deep.equal(ia('3:4 5:6'));
                expect(ia('5:6 7:8').move(-1)).to.deep.equal(ia('4:5 6:7'));
                expect(ia('5:6 7:8').move(0)).to.deep.equal(ia('5:6 7:8'));
                expect(ia('5:6 7:8').move(1)).to.deep.equal(ia('6:7 8:9'));
                expect(ia('5:6 7:8').move(2)).to.deep.equal(ia('7:8 9:10'));
                expect(ia('5:6 7:8').move(3)).to.deep.equal(ia('8:9 10:11'));
            });
            it('should return itself', function () {
                var ia1 = ia('5:6 7:8');
                expect(ia1.move(1)).to.equal(ia1);
            });
        });

        describe('method "indexIterator"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('indexIterator');
            });
            it('should visit all indexes in an interval array', function () {
                var ia1 = ia('5:7 7:9'), it = ia1.indexIterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.contain({ done: false, interval: ia1[0], value: 4 });
                expect(it.next()).to.contain({ done: false, interval: ia1[0], value: 5 });
                expect(it.next()).to.contain({ done: false, interval: ia1[0], value: 6 });
                expect(it.next()).to.contain({ done: false, interval: ia1[1], value: 6 });
                expect(it.next()).to.contain({ done: false, interval: ia1[1], value: 7 });
                expect(it.next()).to.contain({ done: false, interval: ia1[1], value: 8 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should visit all indexes in an interval array in reversed order', function () {
                var ia1 = ia('5:7 7:9 9:11'), it = ia1.indexIterator({ reverse: true, begin: 1 });
                expect(it).to.respondTo('next');
                expect(it.next()).to.contain({ done: false, interval: ia1[1], value: 8 });
                expect(it.next()).to.contain({ done: false, interval: ia1[1], value: 7 });
                expect(it.next()).to.contain({ done: false, interval: ia1[1], value: 6 });
                expect(it.next()).to.contain({ done: false, interval: ia1[0], value: 6 });
                expect(it.next()).to.contain({ done: false, interval: ia1[0], value: 5 });
                expect(it.next()).to.contain({ done: false, interval: ia1[0], value: 4 });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "stringifyAsCols"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('stringifyAsCols');
            });
            var ia0 = new IntervalArray(), ia1 = ia('A:B C:D');
            it('should stringify to column intervals', function () {
                expect(ia1.stringifyAsCols()).to.equal('A:B,C:D');
            });
            it('should use the separator string', function () {
                expect(ia1.stringifyAsCols(' + ')).to.equal('A:B + C:D');
            });
            it('should return empty string for an empty array', function () {
                expect(ia0.stringifyAsCols()).to.equal('');
            });
        });

        describe('method "stringifyAsRows"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('stringifyAsRows');
            });
            var ia0 = new IntervalArray(), ia1 = ia('1:2 3:4');
            it('should stringify to row intervals', function () {
                expect(ia1.stringifyAsRows()).to.equal('1:2,3:4');
            });
            it('should use the separator string', function () {
                expect(ia1.stringifyAsRows(' + ')).to.equal('1:2 + 3:4');
            });
            it('should return empty string for an empty array', function () {
                expect(ia0.stringifyAsRows()).to.equal('');
            });
        });

        describe('method "stringifyAs"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('stringifyAs');
            });
            var ia0 = new IntervalArray(), ia1 = ia('A:B C:D');
            it('should stringify to column intervals', function () {
                expect(ia1.stringifyAs(true)).to.equal('A:B,C:D');
                expect(ia1.stringifyAs(true, ' + ')).to.equal('A:B + C:D');
                expect(ia0.stringifyAs(true)).to.equal('');
            });
            it('should stringify to row intervals', function () {
                expect(ia1.stringifyAs(false)).to.equal('1:2,3:4');
                expect(ia1.stringifyAs(false, ' + ')).to.equal('1:2 + 3:4');
                expect(ia0.stringifyAs(false)).to.equal('');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('toString');
            });
            var ia1 = ia('1:2 3:4');
            it('should stringify to row interval', function () {
                expect(ia1.toString()).to.equal('1:2,3:4');
            });
            it('should use the separator string', function () {
                expect(ia1.toString(' + ')).to.equal('1:2 + 3:4');
            });
            it('should stringify implicitly', function () {
                expect('<' + ia1 + '>').to.equal('<1:2,3:4>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(IntervalArray).to.respondTo('toJSON');
            });
            var ia1 = ia('1:2 3:4'), ia2 = [{ first: 0, last: 1 }, { first: 2, last: 3 }];
            it('should convert to JSON data', function () {
                expect(ia1.toJSON()).to.deep.equal(ia2);
            });
            it('should stringify to JSON', function () {
                expect(JSON.parse(JSON.stringify(ia1))).to.deep.equal(ia2);
            });
        });
    });

    // ========================================================================
});
