/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/address'
], function (SheetHelper, Address) {

    'use strict';

    // class Address ==========================================================

    describe('Spreadsheet class Address', function () {

        it('should exist', function () {
            expect(Address).to.be.a('function');
        });
        it('should be extendable', function () {
            expect(Address).itself.to.respondTo('extend');
        });

        // private helpers ----------------------------------------------------

        var a = SheetHelper.a;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create an address', function () {
                var a1 = new Address(1, 3);
                expect(a1).to.have.property('0', 1);
                expect(a1).to.have.property('1', 3);
            });
        });

        // constants ----------------------------------------------------------

        describe('constant "COL_PATTERN"', function () {
            it('should exist', function () {
                expect(Address.COL_PATTERN).to.be.a('string');
            });
        });

        describe('constant "ROW_PATTERN"', function () {
            it('should exist', function () {
                expect(Address.ROW_PATTERN).to.be.a('string');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('key');
            });
            it('should return unique key for cell address', function () {
                expect(Address.key(0, 1)).to.equal('0,1');
                expect(Address.key(1, 2)).to.equal('1,2');
            });
        });

        describe('method "stringifyCol"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('stringifyCol');
            });
            it('should return the correct column name', function () {
                expect(Address.stringifyCol(0)).to.equal('A');
                expect(Address.stringifyCol(1)).to.equal('B');
                expect(Address.stringifyCol(2)).to.equal('C');
                expect(Address.stringifyCol(25)).to.equal('Z');
                expect(Address.stringifyCol(26)).to.equal('AA');
                expect(Address.stringifyCol(27)).to.equal('AB');
                expect(Address.stringifyCol(28)).to.equal('AC');
                expect(Address.stringifyCol(51)).to.equal('AZ');
                expect(Address.stringifyCol(52)).to.equal('BA');
                expect(Address.stringifyCol(78)).to.equal('CA');
                expect(Address.stringifyCol(701)).to.equal('ZZ');
                expect(Address.stringifyCol(702)).to.equal('AAA');
                expect(Address.stringifyCol(18277)).to.equal('ZZZ');
                expect(Address.stringifyCol(18278)).to.equal('AAAA');
                expect(Address.stringifyCol(475253)).to.equal('ZZZZ');
            });
        });

        describe('method "stringifyRow"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('stringifyRow');
            });
            it('should return the correct row name', function () {
                expect(Address.stringifyRow(0)).to.equal('1');
                expect(Address.stringifyRow(1)).to.equal('2');
                expect(Address.stringifyRow(2)).to.equal('3');
                expect(Address.stringifyRow(8)).to.equal('9');
                expect(Address.stringifyRow(9)).to.equal('10');
                expect(Address.stringifyRow(10)).to.equal('11');
                expect(Address.stringifyRow(99999998)).to.equal('99999999');
            });
        });

        describe('method "stringifyIndex"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('stringifyIndex');
            });
            it('should return the column name', function () {
                expect(Address.stringifyIndex(0, true)).to.equal('A');
            });
            it('should return the row name', function () {
                expect(Address.stringifyIndex(0, false)).to.equal('1');
            });
        });

        describe('method "parseCol"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('parseCol');
            });
            it('should return the correct column index', function () {
                expect(Address.parseCol('A')).to.equal(0);
                expect(Address.parseCol('B')).to.equal(1);
                expect(Address.parseCol('C')).to.equal(2);
                expect(Address.parseCol('Z')).to.equal(25);
                expect(Address.parseCol('AA')).to.equal(26);
                expect(Address.parseCol('AB')).to.equal(27);
                expect(Address.parseCol('AC')).to.equal(28);
                expect(Address.parseCol('AZ')).to.equal(51);
                expect(Address.parseCol('BA')).to.equal(52);
                expect(Address.parseCol('CA')).to.equal(78);
                expect(Address.parseCol('ZZ')).to.equal(701);
                expect(Address.parseCol('AAA')).to.equal(702);
                expect(Address.parseCol('ZZZ')).to.equal(18277);
                expect(Address.parseCol('AAAA')).to.equal(18278);
                expect(Address.parseCol('ZZZZ')).to.equal(475253);
            });
            it('should accept lower-case and mixed-case column names', function () {
                expect(Address.parseCol('a')).to.equal(0);
                expect(Address.parseCol('z')).to.equal(25);
                expect(Address.parseCol('aa')).to.equal(26);
                expect(Address.parseCol('aA')).to.equal(26);
                expect(Address.parseCol('Aa')).to.equal(26);
            });
            it('should return -1 for invalid column names', function () {
                expect(Address.parseCol('')).to.equal(-1);
                expect(Address.parseCol('0')).to.equal(-1);
                expect(Address.parseCol('$')).to.equal(-1);
                expect(Address.parseCol(' A ')).to.equal(-1);
                expect(Address.parseCol('AAAAA')).to.equal(-1);
            });
            it('should return -1 if custom maximum does not fit', function () {
                expect(Address.parseCol('C', 2)).to.equal(2);
                expect(Address.parseCol('C', 1)).to.equal(-1);
            });
        });

        describe('method "parseRow"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('parseRow');
            });
            it('should return the correct row index', function () {
                expect(Address.parseRow('1')).to.equal(0);
                expect(Address.parseRow('2')).to.equal(1);
                expect(Address.parseRow('3')).to.equal(2);
                expect(Address.parseRow('9')).to.equal(8);
                expect(Address.parseRow('10')).to.equal(9);
                expect(Address.parseRow('11')).to.equal(10);
                expect(Address.parseRow('99999999')).to.equal(99999998);
            });
            it('should accept leading zero characters', function () {
                expect(Address.parseRow('01')).to.equal(0);
                expect(Address.parseRow('0002')).to.equal(1);
                expect(Address.parseRow('099999999')).to.equal(99999998);
            });
            it('should return -1 for invalid row names', function () {
                expect(Address.parseRow('')).to.equal(-1);
                expect(Address.parseRow('0')).to.equal(-1);
                expect(Address.parseRow('-1')).to.equal(-1);
                expect(Address.parseRow('A')).to.equal(-1);
                expect(Address.parseRow('$')).to.equal(-1);
                expect(Address.parseRow(' 1 ')).to.equal(-1);
                expect(Address.parseRow('100000000')).to.equal(-1);
            });
            it('should return -1 if custom maximum does not fit', function () {
                expect(Address.parseRow('3', 2)).to.equal(2);
                expect(Address.parseRow('3', 1)).to.equal(-1);
            });
        });

        describe('method "parse"', function () {
            it('should exist', function () {
                expect(Address).itself.to.respondTo('parse');
            });
            it('should return the correct cell address', function () {
                expect(Address.parse('A1')).to.deep.equal(new Address(0, 0));
                expect(Address.parse('B1')).to.deep.equal(new Address(1, 0));
                expect(Address.parse('C1')).to.deep.equal(new Address(2, 0));
                expect(Address.parse('A2')).to.deep.equal(new Address(0, 1));
                expect(Address.parse('A3')).to.deep.equal(new Address(0, 2));
                expect(Address.parse('ZZZZ99999999')).to.deep.equal(new Address(475253, 99999998));
            });
            it('should accept lower-case columns and leading zero characters', function () {
                expect(Address.parse('aa010')).to.deep.equal(new Address(26, 9));
                expect(Address.parse('aA0010')).to.deep.equal(new Address(26, 9));
                expect(Address.parse('Aa00010')).to.deep.equal(new Address(26, 9));
            });
            it('should return null for invalid cell addresses', function () {
                expect(Address.parse('')).to.equal(null);
                expect(Address.parse('A')).to.equal(null);
                expect(Address.parse('1')).to.equal(null);
                expect(Address.parse('$')).to.equal(null);
                expect(Address.parse('$A$1')).to.equal(null);
                expect(Address.parse('A0')).to.equal(null);
                expect(Address.parse('AAAAA1')).to.equal(null);
                expect(Address.parse('A100000000')).to.equal(null);
            });
            it('should return null if custom maximum does not fit', function () {
                expect(Address.parse('C3', new Address(2, 2))).to.deep.equal(new Address(2, 2));
                expect(Address.parse('C3', new Address(1, 2))).to.equal(null);
                expect(Address.parse('C3', new Address(2, 1))).to.equal(null);
            });
        });

        describe('method "compare"', function () {
            var a1 = a('C3');
            it('should exist', function () {
                expect(Address).itself.to.respondTo('compare');
            });
            it('should return 0 for equal addresses', function () {
                expect(Address.compare(a1, a1)).to.equal(0);
            });
            it('should return negative value for addresses in order', function () {
                expect(Address.compare(a1, a('D3'))).to.be.below(0);
                expect(Address.compare(a1, a('C4'))).to.be.below(0);
                expect(Address.compare(a1, a('A4'))).to.be.below(0);
            });
            it('should return positive value for addresses in reversed order', function () {
                expect(Address.compare(a1, a('B3'))).to.be.above(0);
                expect(Address.compare(a1, a('C2'))).to.be.above(0);
                expect(Address.compare(a1, a('E2'))).to.be.above(0);
            });
        });

        describe('method "compareVert"', function () {
            var a1 = a('C3');
            it('should exist', function () {
                expect(Address).itself.to.respondTo('compareVert');
            });
            it('should return 0 for equal addresses', function () {
                expect(Address.compareVert(a1, a1)).to.equal(0);
            });
            it('should return negative value for addresses in order', function () {
                expect(Address.compareVert(a1, a('C4'))).to.be.below(0);
                expect(Address.compareVert(a1, a('D3'))).to.be.below(0);
                expect(Address.compareVert(a1, a('D1'))).to.be.below(0);
            });
            it('should return positive value for addresses in reversed order', function () {
                expect(Address.compareVert(a1, a('C2'))).to.be.above(0);
                expect(Address.compareVert(a1, a('B3'))).to.be.above(0);
                expect(Address.compareVert(a1, a('B5'))).to.be.above(0);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('key');
            });
            it('should return unique key for cell address', function () {
                expect(a('A2').key()).to.equal('0,1');
                expect(a('B3').key()).to.equal('1,2');
                expect(a('ZZ999').key()).to.equal('701,998');
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var a1 = a('B4'), a2 = a1.clone();
                expect(a2).to.be.an.instanceof(Address);
                expect(a2).to.not.equal(a1);
                expect(a2).to.deep.equal(a1);
            });
        });

        describe('method "equals"', function () {
            var a1 = a('B4');
            it('should exist', function () {
                expect(Address).to.respondTo('equals');
            });
            it('should return true for equal addresses', function () {
                expect(a1.equals(a1)).to.equal(true);
                expect(a1.equals(a1.clone())).to.equal(true);
            });
            it('should return false for different addresses', function () {
                expect(a1.equals(a('C4'))).to.equal(false);
                expect(a1.equals(a('B3'))).to.equal(false);
                expect(a1.equals(a('C3'))).to.equal(false);
            });
        });

        describe('method "differs"', function () {
            var a1 = a('B4');
            it('should exist', function () {
                expect(Address).to.respondTo('differs');
            });
            it('should return false for equal addresses', function () {
                expect(a1.differs(a1)).to.equal(false);
                expect(a1.differs(a1.clone())).to.equal(false);
            });
            it('should return true for different addresses', function () {
                expect(a1.differs(a('C4'))).to.equal(true);
                expect(a1.differs(a('B3'))).to.equal(true);
                expect(a1.differs(a('C3'))).to.equal(true);
            });
        });

        describe('method "compareTo"', function () {
            var a1 = a('C3');
            it('should exist', function () {
                expect(Address).to.respondTo('compareTo');
            });
            it('should return 0 for equal addresses', function () {
                expect(a1.compareTo(a1)).to.equal(0);
            });
            it('should return negative value for addresses in order', function () {
                expect(a1.compareTo(a('D3'))).to.be.below(0);
                expect(a1.compareTo(a('C4'))).to.be.below(0);
                expect(a1.compareTo(a('A4'))).to.be.below(0);
            });
            it('should return positive value for addresses in reversed order', function () {
                expect(a1.compareTo(a('B3'))).to.be.above(0);
                expect(a1.compareTo(a('C2'))).to.be.above(0);
                expect(a1.compareTo(a('E2'))).to.be.above(0);
            });
        });

        describe('method "compareVertTo"', function () {
            var a1 = a('C3');
            it('should exist', function () {
                expect(Address).to.respondTo('compareVertTo');
            });
            it('should return 0 for equal addresses', function () {
                expect(a1.compareVertTo(a1)).to.equal(0);
            });
            it('should return negative value for addresses in order', function () {
                expect(a1.compareVertTo(a('C4'))).to.be.below(0);
                expect(a1.compareVertTo(a('D3'))).to.be.below(0);
                expect(a1.compareVertTo(a('D1'))).to.be.below(0);
            });
            it('should return positive value for addresses in reversed order', function () {
                expect(a1.compareVertTo(a('C2'))).to.be.above(0);
                expect(a1.compareVertTo(a('B3'))).to.be.above(0);
                expect(a1.compareVertTo(a('B5'))).to.be.above(0);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('get');
            });
            it('should return the correct index', function () {
                var a1 = a('B3');
                expect(a1.get(true)).to.equal(1);
                expect(a1.get(false)).to.equal(2);
            });
        });

        describe('method "mirror"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('mirror');
            });
            it('should return the mirrored address', function () {
                expect(a('A2').mirror()).to.deep.equal(a('B1'));
                expect(a('B1').mirror()).to.deep.equal(a('A2'));
            });
        });

        describe('method "set"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('set');
            });
            it('should change the correct index', function () {
                var a1 = a('A1');
                expect(a1).to.deep.equal(a('A1'));
                a1.set(1, true);
                expect(a1).to.deep.equal(a('B1'));
                a1.set(2, false);
                expect(a1).to.deep.equal(a('B3'));
            });
            it('should return itself', function () {
                var a1 = a('A1');
                expect(a1.set(1, true)).to.equal(a1);
            });
        });

        describe('method "assign"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('assign');
            });
            it('should change the indexes', function () {
                var a1 = a('C4');
                expect(a1).to.deep.equal(a('C4'));
                a1.assign(a('B3'));
                expect(a1).to.deep.equal(a('B3'));
            });
            it('should return itself', function () {
                var a1 = a('A1');
                expect(a1.assign(a('B3'))).to.equal(a1);
            });
        });

        describe('method "move"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('move');
            });
            it('should change the correct index', function () {
                var a1 = a('C4');
                expect(a1).to.deep.equal(a('C4'));
                a1.move(2, true);
                expect(a1).to.deep.equal(a('E4'));
                a1.move(-2, true);
                expect(a1).to.deep.equal(a('C4'));
                a1.move(2, false);
                expect(a1).to.deep.equal(a('C6'));
                a1.move(-2, false);
                expect(a1).to.deep.equal(a('C4'));
            });
            it('should return itself', function () {
                var a1 = a('A1');
                expect(a1.move(1, true)).to.equal(a1);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('toString');
            });
            it('should stringify the address', function () {
                expect(a('A1').toString()).to.equal('A1');
                expect(a('B1').toString()).to.equal('B1');
                expect(a('C1').toString()).to.equal('C1');
                expect(a('A2').toString()).to.equal('A2');
                expect(a('A3').toString()).to.equal('A3');
                expect(a('ZZZZ99999999').toString()).to.equal('ZZZZ99999999');
            });
            it('should stringify implicitly', function () {
                expect('<' + a('A3') + '>').to.equal('<A3>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Address).to.respondTo('toJSON');
            });
            var a1 = a('A3'), a2 = [0, 2];
            it('should convert to JSON data', function () {
                expect(a1.toJSON()).to.deep.equal(a2);
            });
            it('should stringify implicitly', function () {
                expect(JSON.parse(JSON.stringify(a1))).to.deep.equal(a2);
            });
        });
    });

    // ========================================================================
});
