/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/spreadsheet/model/drawing/drawingcollection'
], function (AppHelper, DrawingFrame, DrawingCollection) {

    'use strict';

    // class DrawingCollection ================================================

    describe('Spreadsheet class DrawingCollection', function () {

        it('should exist', function () {
            expect(DrawingCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertDrawing', start: [0, 0], type: 'image', attrs: { drawing: { startCol: 1, startRow: 1, endCol: 2, endRow: 2 } } },
            { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { drawing: { startCol: 1, startRow: 1, endCol: 2, endRow: 2 }, geometry: { presetShape: 'rect' } } },
            { name: 'insertParagraph', start: [0, 1, 0], attrs: { paragraph: { alignment: 'left' } } },
            { name: 'insertText', start: [0, 1, 0, 0], text: 'First line.' },
            { name: 'insertParagraph', start: [0, 1, 1], attrs: { paragraph: { alignment: 'right' } } },
            { name: 'insertText', start: [0, 1, 1, 0], text: 'Second line.' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' }
        ];

        // initialize test document
        var docModel = null;
        var sheetModel1 = null, drawingColl1 = null;
        var sheetModel2 = null, drawingColl2 = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel1 = docModel.getSheetModel(0);
            drawingColl1 = sheetModel1.getDrawingCollection();
            sheetModel2 = docModel.getSheetModel(1);
            drawingColl2 = sheetModel2.getDrawingCollection();
        });

        function expectChildNode(parentNode, testNode) {
            expect(testNode).to.be.an.instanceof(HTMLElement);
            expect($.contains(parentNode, testNode)).to.equal(true);
        }

        // public methods -----------------------------------------------------

        describe('method "getContainerNode"', function () {
            it('should exist', function () {
                expect(drawingColl1).to.respondTo('getContainerNode');
            });
            it('should return the DOM container node', function () {
                var pageNode = docModel.getNode()[0];
                var node1 = drawingColl1.getContainerNode();
                expectChildNode(pageNode, node1);
                var node2 = drawingColl2.getContainerNode();
                expectChildNode(pageNode, node2);
                expect(node1).to.not.equal(node2);
            });
        });

        describe('method "getDrawingFrameForModel"', function () {
            it('should exist', function () {
                expect(drawingColl1).to.respondTo('getDrawingFrameForModel');
            });
            it('should return the frame node for images', function () {
                var imageModel = drawingColl1.getModel([0]);
                var imageFrame = drawingColl1.getDrawingFrameForModel(imageModel);
                expectChildNode(drawingColl1.getContainerNode(), imageFrame[0]);
                expect(DrawingFrame.getModel(imageFrame)).to.equal(imageModel);
            });
            it('should return the frame node for shapes', function () {
                var shapeModel = drawingColl1.getModel([1]);
                var shapeFrame = drawingColl1.getDrawingFrameForModel(shapeModel);
                expectChildNode(drawingColl1.getContainerNode(), shapeFrame[0]);
                expect(DrawingFrame.getModel(shapeFrame)).to.equal(shapeModel);
                var textFrame = shapeFrame.find('>*>.textframe');
                expectChildNode(shapeFrame[0], textFrame[0]);
                var paraNodes = textFrame.children();
                expect(paraNodes).to.have.length(2);
                expect(paraNodes.eq(0).text()).to.equal('First line.');
                expect(paraNodes.eq(1).text()).to.equal('Second line.');
            });
        });

        describe('method "getInactiveAnchorAttributes"', function () {
            it('should exist', function () {
                expect(drawingColl1).to.respondTo('getInactiveAnchorAttributes');
            });
        });

        describe('method "getRectangleForModel"', function () {
            it('should exist', function () {
                expect(drawingColl1).to.respondTo('getRectangleForModel');
            });
        });

        describe('method "getAnchorAttributesForRect"', function () {
            it('should exist', function () {
                expect(drawingColl1).to.respondTo('getAnchorAttributesForRect');
            });
        });
    });

    // ========================================================================
});
