/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

(function () {

    const round = Math.round;
    const floor = Math.floor;
    const proc = typeof global === 'object' ? global.process : null;
    const perf = typeof window === 'object' ? window.performance : null;

    // static functions =======================================================

    exports.pause = async t => new Promise(resolve => setTimeout(resolve, t));

    exports.getTime = proc ? proc.hrtime : perf ? perf.now.bind(perf) : Date.now;

    exports.getDuration =
        proc ? t0 => { const [s, n] = proc.hrtime(t0); return s * 1e9 + n; } :
        perf ? t0 => round((perf.now() - t0) * 1e6) :
        t0 => (Date.now() - t0) * 1e6;

    // class LogTable =========================================================

    const SP = ' '.repeat(60);
    const DASH = '-'.repeat(60);

    exports.LogTable = class {

        constructor() {
            this._cols = [];
        }

        // public methods -----------------------------------------------------

        col(width, sep) {
            this._cols.push({ w: width, sep });
            return this;
        }

        sep() {
            const msg = this._cols.reduce((txt, col) => txt + DASH.substr(-col.w) + (col.sep ? '+' : '-'), '');
            console.log(msg);
            return this;
        }

        head(...args) {
            let msg = '', ci = 0;
            args.forEach(v => {
                let span = 0;
                if (typeof v === 'object') {
                    span = v.span || 1;
                    v = String(v.msg);
                } else {
                    v = String(v);
                    span = 1;
                }
                let w = span - 1, sep = false;
                for (; span > 0; span -= 1) {
                    const col = this._cols[ci];
                    w += col.w;
                    sep = col.sep;
                    ci += 1;
                }
                msg += (SP + v + SP).substr(SP.length - floor((w - v.length) / 2), w);
                msg += (sep ? '|' : ' ');
            });
            console.log(msg);
            return this;
        }

        row(...args) {
            const msg = this._cols.reduce((txt, col, i) => {
                const v = args[i];
                if (typeof v === 'number') {
                    txt += (SP + v).substr(-col.w);
                } else {
                    txt += (v + SP).substr(0, col.w);
                }
                return txt + (col.sep ? '|' : ' ');
            }, '');
            console.log(msg);
            return this;
        }
    };
}());
