/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

const fs = require('fs');
const path = require('path');

let root = require.main.filename;
while (path.basename(root) !== 'perf') {
    root = path.dirname(root);
}

function noop() { return this; }

const document = global.document = {};
document.body = {};
document.addEventListener = noop;

const window = global.window = {};
window.document = document;
window.console = global.console;

const _ = global._ = require('underscore');
_.device = noop;
_.browser = {};

_.makeExtendable = function (BaseClass) {
    function HelperClass() {}
    BaseClass.extend = function (protoProps) {
        const SubClass = protoProps.constructor;
        _.extend(SubClass, BaseClass);
        HelperClass.prototype = BaseClass.prototype;
        SubClass.prototype = new HelperClass();
        SubClass.prototype.constructor = SubClass;
        SubClass.extend = BaseClass.extend;
        return SubClass;
    };
    return BaseClass;
};

_.escapeRegExp = function (s) {
    return (s || '').replace(/([$^*+?!:=.|(){}[\]\\])/g, (a, b) => '\\' + b);
};

const jq = [{}];
jq.prependTo = jq.append = jq.remove = jq.on = jq.off = jq.addClass = jq.css = noop;
jq.width = jq.height = _.constant(1);
const $ = global.$ = _.constant(jq);
$.noop = noop;

global.ox = { language: 'en_US' };

// promises for all loaded modules
const modules = {};

// mocked core modules
modules['io.ox/core/event'] = {
    extend(obj) {
        obj.on = obj.off = obj.one = obj.trigger = noop;
        return obj;
    }
};
modules['io.ox/core/http'] = {};
modules['io.ox/core/api/user'] = {};
modules['io.ox/office/tk/io'] = {
    loadJSON(fileName) {
        return new Promise((resolve, reject) =>
            fs.readFile(path.resolve(__dirname, '../../apps', fileName + '.json'), (err, str) =>
                err ? reject(err) : resolve(JSON.parse(str))
            )
        );
    }
};
modules['io.ox/office/tk/render/font'] = {};
modules['static/3rd.party/purify.min.js'] = {};

const logDeps = process.argv.slice(2).includes('--log-deps');
let indent = '';

function define(name, deps, def) {

    if (!def) { def = deps; deps = []; }

    deps = deps.map(dep => {
        if (!(dep in modules)) {
            if (logDeps) { console.log(indent + dep); }
            indent += '  ';
            if (/^less!/.test(dep)) {
                modules[dep] = {};
            } else if (/^gettext!/.test(dep)) {
                modules[dep] = _.identity;
            } else if (/^settings!/.test(dep)) {
                modules[dep] = { get: noop };
            } else if (/^perf!/.test(dep)) {
                dep = dep.substr(dep.indexOf('!') + 1);
                require(path.join(root, dep));
            } else {
                require('../../apps/' + dep);
            }
            indent = indent.slice(2);
        }
        return modules[dep];
    });

    modules[name] = Promise.all(deps).then(resolved => def(...resolved));
}

module.exports = global.define = define.async = define;
