/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('cellmatrix/cellmatrix32', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Iterator, ValueMap, BaseObject, SheetUtils) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var floor = Math.floor;
    var ceil = Math.ceil;
    var SingleIterator = Iterator.SingleIterator;
    var ArrayIterator = Iterator.ArrayIterator;
    var FilterIterator = Iterator.FilterIterator;
    var NestedIterator = Iterator.NestedIterator;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var IntervalArray = SheetUtils.IntervalArray;

    // private functions ======================================================

    /**
     * Creates an interval iterator for the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be visited in reversed order.
     *
     * @returns {Iterator<Interval>}
     *  The new iterator.
     */
    function createIntervalIterator(intervals, reverse) {
        return (intervals instanceof Array) ? intervals.iterator({ reverse: reverse }) : new SingleIterator({ value: intervals });
    }

    /**
     * Creates an interval iterator for the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be visited in reversed order.
     *
     * @returns {Iterator<Interval>}
     *  The new iterator.
     */
    function forEachInterval(intervals, reverse, callback, context) {
        var arrayForEach = reverse ? IntervalArray.forEachReverse : IntervalArray.forEach;
        arrayForEach(intervals, callback, context);
    }

    // class IndexVector ======================================================

    /**
     * An instance of this class represents a sorted sparse array of column/row
     * indexes. It wraps an instance of Int32Array for maximum performance of
     * mutating array operations.
     *
     * @constructor
     */
    function IndexVector() {

        // the sorted index array (oversized to minimize memory reallocations)
        this._array = new Int32Array(20);
        // the actual number of indexes in the array
        this._length = 0;
    }

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this index vector is empty.
     *
     * @returns {boolean}
     *  Whether this index vector is empty.
     */
    IndexVector.prototype.empty = function () {
        return this._length === 0;
    };

    /**
     * Returns all column/row indexes contained in this vector.
     *
     * @returns {Int32Array}
     *  All column/row indexes contained in this vector, as sorted typed array.
     */
    IndexVector.prototype.values = function () {
        return this._array.subarray(0, this._length);
    };

    /**
     * Returns all column/row indexes of this vector contained in the passed
     * index interval.
     *
     * @param {Interval} interval
     *  The index interval.
     *
     * @returns {Int32Array}
     *  All column/row indexes of this vector contained in the passed index
     *  interval, as sorted typed array.
     */
    IndexVector.prototype.slice = function (interval) {

        // fast binary search for the first element at or following the interval start
        var ai1 = this._findArrayIndex(interval.first);
        // fast binary search for the first element following the interval
        var ai2 = this._findArrayIndex(interval.last + 1, ai1);

        // return the sub array (a slim view on the same buffer data)
        return this._array.subarray(ai1, ai2);
    };

    /**
     * Creates an iterator that visits all column/row indexes of this vector
     * contained in the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {boolean} [reverse=false]
     *  If set to true, the passed intervals will be processed in reversed
     *  order, as well as the column/row indexes of this vector in each
     *  interval.
     *
     * @returns {Iterator<number>}
     *  The new iterator. The result values will be the column/row indexes.
     */
    IndexVector.prototype.iterator = function (intervals, reverse) {

        // create an array iterator that visits all index intervals
        var iterator = createIntervalIterator(intervals, reverse);

        // create an iterator that visits the indexes in the passed intervals
        var options = reverse ? { reverse: true } : null;
        return new NestedIterator(iterator, function (interval) {
            return new ArrayIterator(this.slice(interval), options);
        }, null, this);
    };

    /**
     * Inserts the passed column/row index into this vector.
     *
     * @param {number} index
     *  The column/row index to be inserted into this vector. If the index
     *  exists already in this vector, nothing will happen.
     */
    IndexVector.prototype.insert = function (index) {

        // fast binary search for the first element equal to or greater than "index"
        var ai = this._findArrayIndex(index);

        // nothing more to do, if the element exists already
        var array = this._array, length = this._length;
        if ((ai < length) && (array[ai] === index)) { return; }

        // expand storage array on demand (add half of the size)
        if (array.length === length) {
            this._array = new Int32Array(ceil(length * 1.5));
            this._array.set(array);
            array = this._array;
        }

        // shift existing array elements away, insert the new array element
        array.copyWithin(ai + 1, ai, length);
        array[ai] = index;
        this._length += 1;
    };

    /**
     * Removes an existing column/row index from this vector.
     *
     * @param {number} index
     *  The column/row index to be removed from this vector. If the index does
     *  not exist in this vector, nothing will happen.
     */
    IndexVector.prototype.remove = function (index) {

        // fast binary search for the first element equal to or greater than "index"
        var ai = this._findArrayIndex(index);

        // nothing more to do, if the element does not exist
        var array = this._array, length = this._length;
        if ((length < ai) || (array[ai] !== index)) { return; }

        // shift existing array elements to front
        array.copyWithin(ai, ai + 1, length);
        length = (this._length -= 1);

        // shrink storage array on demand (remove half of the size)
        if ((array.length >= 30) && (length * 3 < array.length)) {
            this._array = new Int32Array(ceil(length * 1.5));
            this._array.set(array.subarray(0, length));
        }
    };

    // private methods --------------------------------------------------------

    /**
     * Fast binary search for the first element equal to or greater than the
     * passed column/row index.
     *
     * @param {number} index
     *  The column/row index to be found.
     *
     * @param {number} [start=0]
     *  The array index of the element where the binary search should start. By
     *  default, the entire array will be searched.
     *
     * @returns {number}
     *  The array index of the element equal to or greater than the passed
     *  column/row index.
     */
    IndexVector.prototype._findArrayIndex = function (index, start) {
        var array = this._array, ai0 = start || 0, ai1 = this._length;
        if ((ai0 < ai1) && (index <= array[ai0])) { return ai0; }
        while (ai0 < ai1) {
            var ai = floor((ai0 + ai1) / 2);
            var cmp = array[ai] - index;
            if (cmp > 0) { ai1 = ai; } else if (cmp < 0) { ai0 = ai + 1; } else { return ai; }
        }
        return ai0;
    };

    // class VectorEntry ======================================================

    /**
     * Extendable storage wrapper holding a single index vector. Instances of
     * this class are contained in the internal array of CellMatrix instances.
     *
     * @param {ColRowCollection} collection
     *  The column/row collection needed to resolve the initial visibility of
     *  the cells covered by the index vector.
     *
     * @param {number} index
     *  The index of the column/row represented by the index vector hold by
     *  this instance.
     */
    function VectorEntry(collection, index) {
        this.vector = new IndexVector();
        this.index = index;
        this.visible = collection.isEntryVisible(index);
    }

    // class CellMatrix =======================================================

    /**
     * An instance of this class represents a sparse two-dimensional matrix of
     * cell addresses. The double-sorted data structure allows fast binary
     * lookup, and fast iterator implementations for existing addresses in
     * large cell ranges.
     *
     * Example: An instance of this class is used as row matrix. In this case,
     * it will contain several row vectors that contain the column indexes of
     * all cell addresses in the respective rows.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SheetModel} sheetModel
     *  The parent sheet model containing the cell collection with this matrix.
     *
     * @param {ValueMap<CellModel>} modelMap
     *  A referenec to the map of all cell models whose addresses are contained
     *  in this matrix. This map will be maintained and kept in sync by the
     *  parent CellTable instance owning this matrix.
     *
     * @param {boolean} vertical
     *  Whether this matrix contains column vectors (true), or row vectors
     *  (false).
     */
    var CellMatrix = BaseObject.extend({ constructor: function (sheetModel, modelMap, vertical) {

        // base constructor
        BaseObject.call(this, sheetModel);

        // the sorted array of vector entries (instances of VectorEntry)
        this._array = [];
        // the column/row collection associated to the index vectors
        this._collection = vertical ? sheetModel.getColCollection() : sheetModel.getRowCollection();
        // the map of all existing cell models, mapped by cell address keys
        this._models = modelMap;
        // the orientation of the index vectors
        this._vertical = vertical;

        // update cached visibility flags if columns/rows will be hidden or shown
        this.listenTo(this._collection, 'change:entries', function (_event, interval, changeInfo) {
            if (changeInfo.visibilityChanged) {
                var ai1 = this._findArrayIndex(interval.first);
                var ai2 = this._findArrayIndex(interval.last + 1, ai1);
                if (ai1 < ai2) {
                    var visible = this._collection.isEntryVisible(interval.first);
                    for (var array = this._array; ai1 < ai2; ai1 += 1) {
                        array[ai1].visible = visible;
                    }
                }
            }
        }.bind(this));
    } });

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this matrix is empty.
     *
     * @returns {boolean}
     *  Whether this matrix is empty.
     */
    CellMatrix.prototype.empty = function () {
        return this._array.length === 0;
    };

    /**
     * Returns the smallest index interval containing all index vectors in this
     * matrix.
     *
     * @returns {Interval|null}
     *  The smallest index interval containing all index vectors in this
     *  matrix; or null, if this matrix is empty.
     */
    CellMatrix.prototype.getUsedInterval = function () {
        var array = this._array, length = array.length;
        return (length === 0) ? null : new Interval(array[0], array[length - 1]);
    };

    /**
     * Creates an iterator that visits all index vectors of this matrix
     * contained in the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the passed index intervals will be visited in
     *      reversed order, as well as the index vectors covered by each of the
     *      intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible index vectors (associated to visible
     *      columns/rows) will be visited. If set to false, only hidden index
     *      vectors (associated to hidden columns/rows) will be visited. By
     *      default, all index vectors will be visited.
     *
     * @returns {Iterator<VectorEntry>}
     *  The new iterator.
     */
    CellMatrix.prototype.createVectorIterator = function (intervals, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // create an array iterator that visits all index intervals
        var iterator = createIntervalIterator(intervals, reverse);

        // create an iterator that visits the single vectors in the passed intervals
        iterator = new NestedIterator(iterator, function (interval) {

            // fast binary search for the first and last array elements
            var ai1 = this._findArrayIndex(interval.first);
            var ai2 = this._findArrayIndex(interval.last + 1, ai1);
            if (ai1 === ai2) { return null; }

            // adjust array indexes for reverse mode
            var begin = reverse ? (ai2 - 1) : ai1;
            var end = reverse ? (ai1 - 1) : ai2;
            return new ArrayIterator(this._array, { begin: begin, end: end, reverse: reverse });

        }, null, this);

        // filter for visible or hidden index vectors
        var visible = options && options.visible;
        if (typeof visible === 'boolean') {
            iterator = new FilterIterator(iterator, function (entry) { return entry.visible === visible; });
        }

        return iterator;
    };

    /**
     * Invokes the specified callback function for all index vectors of this
     * matrix contained in the passed index intervals.
     *
     * @param {IntervalArray|Interval} intervals
     *  An array of index intervals, or a single index interval.
     *
     * @param {function} callback
     *  The callback function to be invoked for each matching index vector.
     *  Receives the vector entry (instance of VectorEntry) as first parameter.
     *  The return value will be ignored.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the passed index intervals will be visited in
     *      reversed order, as well as the index vectors covered by each of the
     *      intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible index vectors (associated to visible
     *      columns/rows) will be visited. If set to false, only hidden index
     *      vectors (associated to hidden columns/rows) will be visited. By
     *      default, all index vectors will be visited.
     */
    CellMatrix.prototype.forEachVector = function (intervals, callback, options) {

        // the iteration order
        var reverse = options && options.reverse;

        // invokes the callback function according to visibility of the vector
        var invoke = (function () {
            switch (options && options.visible) {
                case true:  return function (entry) { if (entry.visible) { callback(entry); } };
                case false: return function (entry) { if (!entry.visible) { callback(entry); } };
                default:    return callback;
            }
        }());

        // visit the passed intervals in the specified order
        forEachInterval(intervals, reverse, function (interval) {

            // fast binary search for the first and last array elements
            var ai1 = this._findArrayIndex(interval.first);
            var ai2 = this._findArrayIndex(interval.last + 1, ai1);
            if (ai1 === ai2) { return; }

            // adjust array indexes for reverse mode
            var begin = reverse ? (ai2 - 1) : ai1;
            var end = reverse ? (ai1 - 1) : ai2;
            var step = reverse ? -1 : 1;
            for (var ai = begin; ai !== end; ai += step) {
                invoke(this._array[ai]);
            }
        }, this);
    };

    /**
     * Creates an iterator that visits all existing cell models in this matrix
     * covered by the ranges formed by the passed index intervals. The cell
     * models will be visited vector-by-vector.
     *
     * @param {IntervalArray|Interval} outerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all index vectors of this matrix to be visited.
     *
     * @param {IntervalArray|Interval} innerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all column/row indexes to be visited in each index vector.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the cell models will be visited starting from the
     *      bottom-right edge of the last inner/outer intervals, and ending in
     *      the top-left edge of the first inner/outer intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cells (cells in visible columns AND
     *      visible rows) will be visited. If set to false, only hidden cells
     *      (cells in hidden columns OR hidden rows) will be visited. By
     *      default, all cells will be visited.
     *
     * @returns {Iterator<CellModel>}
     *  The new iterator.
     */
    CellMatrix.prototype.createModelIterator = function (outerIntervals, innerIntervals, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // the cell model map
        var models = this._models;
        // create an iterator that visits the index vectors in the passed outer intervals
        var iterator = this.createVectorIterator(outerIntervals, options);

        // callback for the NestedIterator to create inner iterators for an index vector
        var createIndexIterator = function (entry) {
            return entry.vector.iterator(innerIntervals, reverse);
        };

        // filter callback whether to visit visible/hidden cells only
        var visibilityFilter = (function (prop2) {
            switch (options && options.visible) {
                case true:  return function (cellModel) { return cellModel[prop2].visible; };
                case false: return function (cellModel) { return !cellModel[prop2].visible; };
                default:    return null;
            }
        }(this._prop2));

        // combines the results of the outer and inner iterators to cell address keys
        var getModel = this._vertical ?
            function (outerResult, innerResult) { return models.get(Address.key(outerResult.value.index, innerResult.value)); } :
            function (outerResult, innerResult) { return models.get(Address.key(innerResult.value, outerResult.value.index)); };

        // callback for the NestedIterator combining outer and inner iterator results
        var combineResults = visibilityFilter ?
            function (outerResult, innerResult) {
                var cellModel = getModel(outerResult, innerResult);
                return visibilityFilter(cellModel) ? { value: cellModel } : null;
            } : function (outerResult, innerResult) {
                return { value: getModel(outerResult, innerResult) };
            };

        // create an iterator that visits the inner intervals for every single vector, and returns cell models
        return new NestedIterator(iterator, createIndexIterator, combineResults);
    };

    /**
     * Invokes the specified callback function for all existing cell models in
     * this matrix covered by the ranges formed by the passed index intervals.
     * The cell models will be visited vector-by-vector.
     *
     * @param {IntervalArray|Interval} outerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all index vectors of this matrix to be visited.
     *
     * @param {IntervalArray|Interval} innerIntervals
     *  An array of index intervals, or a single index interval, specifying
     *  all column/row indexes to be visited in each index vector.
     *
     * @param {function} callback
     *  The callback function to be invoked for each cell model. Receives the
     *  cell model (instance of CellModel) as first parameter. The return value
     *  will be ignored.
     *
     * @param {object} [options]
     *  Optional parameters:
     *  - {boolean} [options.reverse=false]
     *      If set to true, the cell models will be visited starting from the
     *      bottom-right edge of the last inner/outer intervals, and ending in
     *      the top-left edge of the first inner/outer intervals.
     *  - {boolean} [options.visible]
     *      If set to true, only visible cells (cells in visible columns AND
     *      visible rows) will be visited. If set to false, only hidden cells
     *      (cells in hidden columns OR hidden rows) will be visited. By
     *      default, all cells will be visited.
     */
    CellMatrix.prototype.forEachModel = function (outerIntervals, innerIntervals, callback, options) {

        // the iteration order
        var reverse = options && options.reverse;
        // the cell model map
        var models = this._models;

        // combines the results of the outer and inner iterators to cell address keys
        var getModel = this._vertical ?
            function (entry, index) { return models.get(Address.key(entry.index, index)); } :
            function (entry, index) { return models.get(Address.key(index, entry.index)); };

        // invokes the callback function according to visibility of the vector
        var invoke = (function (prop2) {
            switch (options && options.visible) {
                case true:  return function (cellModel) { if (cellModel[prop2].visible) { callback(cellModel); } };
                case false: return function (cellModel) { if (!cellModel[prop2].visible) { callback(cellModel); } };
                default:    return callback;
            }
        }(this._prop2));

        // visit the index vectors in the passed outer intervals
        this.forEachVector(outerIntervals, function (entry) {
            forEachInterval(innerIntervals, reverse, function (interval) {
                var indexes = entry.vector.slice(interval);
                var begin = reverse ? (indexes.length - 1) : 0;
                var end = reverse ? -1 : indexes.length;
                var step = reverse ? -1 : 1;
                for (var ai = begin; ai !== end; ai += step) {
                    invoke(getModel(entry, indexes[ai]));
                }
            });
        }, options);
    };

    /**
     * Inserts the passed cell address into this matrix.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted into this matrix. This matrix DOES NOT
     *  take ownership of the cell models; and it DOES NOT check whether it
     *  already contains a cell model with the same address as the passed cell
     *  model (it is up to the caller to ensure uniqueness of the cell models).
     *
     * @returns {VectorEntry}
     *  The internal array element containing the index vector.
     */
    CellMatrix.prototype.insertModel = function (cellModel) {

        // fast binary search for the first vector equal to or greater than "index"
        var index = cellModel.a.get(this._vertical);
        var ai = this._findArrayIndex(index);

        // create a new index vector on demand
        var array = this._array, entry = array[ai];
        if (!entry || (entry.index !== index)) {
            entry = new VectorEntry(this._collection, index);
            array.splice(ai, 0, entry);
        }

        // insert the respective address index into the index vector
        entry.vector.insert(cellModel.a.get(!this._vertical));

        // return the vector entry for further processing
        return entry;
    };

    /**
     * Removes an existing cell address from this matrix.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed from this matrix.
     *
     * @returns {VectorEntry|null}
     *  The internal array element containing the index vector; or null, if no
     *  such index vector has been found, or if the existing index vector
     *  became empty and has been deleted.
     */
    CellMatrix.prototype.removeModel = function (cellModel) {

        // fast binary search for the first vector equal to or greater than "index"
        var index = cellModel.a.get(this._vertical);
        var ai = this._findArrayIndex(index);

        // nothing more to do, if the vector does not exist
        var array = this._array, entry = array[ai];
        if (!entry || (entry.index !== index)) { return null; }

        // remove the respective address index from the index vector
        entry.vector.remove(cellModel.a.get(!this._vertical));

        // remove the entire vector, if it became empty
        if (entry.vector.empty()) {
            array.splice(ai, 1);
            return null;
        }

        // return the vector entry for further processing
        return entry;
    };

    // private methods --------------------------------------------------------

    /**
     * Fast binary search for the first index vector whose column/row index is
     * equal to or greater than the passed column/row index.
     *
     * @param {number} index
     *  The column/row index of the index vector to be found.
     *
     * @param {number} [start=0]
     *  The array index of the vector element where the binary search should
     *  start. By default, the entire array of vectors will be searched.
     *
     * @returns {number}
     *  The array index of the index vector whose column/row index is equal to
     *  or greater than the passed column/row index.
     */
    CellMatrix.prototype._findArrayIndex = function (index, start) {
        var array = this._array, ai0 = start || 0, ai1 = array.length;
        if ((ai0 < ai1) && (index <= array[ai0].index)) { return ai0; }
        while (ai0 < ai1) {
            var ai = floor((ai0 + ai1) / 2);
            var cmp = array[ai].index - index;
            if (cmp > 0) { ai1 = ai; } else if (cmp < 0) { ai0 = ai + 1; } else { return ai; }
        }
        return ai0;
    };

    // exports ================================================================

    return CellMatrix;

});
