/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Mario Schroeder <mario.schroeder@open-xchange.com>
 */

define('io.ox/office/tk/utils/presenterutils', [
    'io.ox/files/api',
    'io.ox/office/tk/utils/driveutils'
], function (FilesAPI, DriveUtils) {

    'use strict';

    // static class PresenterUtils ============================================

    var PresenterUtils = {};

    var PRESENTER_APP  = 'io.ox/office/presenter/main';

    // static private functions ===============================================

    // launch OX Presenter
    function launch(model) {
        return ox.launch(PRESENTER_APP, { model: model });
    }

    // authorize with Guard
    function authorize(model) {
        // set authorization data to the model
        var file = {
            data: model.toJSON()
        };
        return DriveUtils.authorizeGuard(file).then(function (auth_code) {
            var params = {
                cryptoAction: 'Decrypt',
                cryptoAuth: auth_code
            };
            model.set('file_options', { params: params });
            return model;
        });
    }

    // public methods ---------------------------------------------------------

    /**
     * Launches a new OX Presenter application.
     *
     * @param {Object} model
     *  The file model object.
     *
     * @param {String} auth_code
     *  The auth_code for a encrypted presentation, if the presentation is encrypted and the
     *  auth_code is null/undefined the Authorizer dialog will be shown.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved when the application is launched
     *  or rejected in case of an error.
     */
    PresenterUtils.launchPresenter = function (model, auth_code) {
        if (PresenterUtils.isEncrypted(model)) {
            if (_.isString(auth_code)) {
                var params = {
                    cryptoAction: 'Decrypt',
                    cryptoAuth: auth_code
                };
                model.set('file_options', { params: params });
                return launch(model);
            }
            // make a copy since the Viewer clears the file authentication
            // from file models when closing
            model = new FilesAPI.Model(model.toJSON());

            return authorize(model).then(launch);
        } else {
            return launch(model);
        }
    };

    /**
     * Returns true if the file data is OX Guard encrypted.
     *
     * @param {FilesAPI.Model} model
     *  The file model.
     *
     * @returns {Boolean}
     *  Whether the file data is encrypted.
     */
    PresenterUtils.isEncrypted = function (model) {
        if (!model) { return false; }

        var result = false;

        if (model.isFile()) {
            result = model.isEncrypted();

        } else if (model.isMailAttachment()) {
            result = /application\/pgp/.test(model.getMimeType());
        }

        return result;
    };

    // exports ================================================================

    return PresenterUtils;

});
