/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils/enum', [
    'io.ox/office/tk/utils/class'
], function (Class) {

    'use strict';

    // class Enum =============================================================

    /**
     * Base class for all enumeration values. An instance of this class
     * represents a single value of an enumeration.
     *
     * @constructor
     *
     * @property {String} name
     *  The upper-case name of the enumeration value.
     *
     * @property {String} json
     *  The JSON string representation of the enumeration value.
     *
     * @property {Number} index
     *  The sort index of the enumeration value (non-negative integer).
     */
    var Enum = Class.extendable(function (json, index) {

        if (!/^[a-z]([a-z0-9_]*)$/i.test(json)) { throw new Error('invalid enumeration value'); }

        this.name = json.replace(/([a-z0-9])([A-Z])/g, '$1_$2').replace(/([a-z])([0-9])/g, '$1_$2').toUpperCase();
        this.json = json;
        this.index = index;

    }); // class Enum

    // public methods ---------------------------------------------------------

    Enum.prototype.toString = function () {
        return this.name;
    };

    Enum.prototype.toJSON = function () {
        return this.json;
    };

    Enum.prototype.valueOf = function () {
        return this.index;
    };

    // static methods ---------------------------------------------------------

    /**
     * Creates a new enumeration class from the passed definition of all values
     * for the enumeration.
     *
     * @param {String} jsonNames
     *  The JSON names of all values for the new enumeration class, as string
     *  with the space-separated JSON string representations of all values. The
     *  enumeration keys will be created by converting the JSON camel-case
     *  string representations to upper-case strings with underscore separators
     *  (e.g. the JSON string 'someValue' will cause to create the enumeration
     *  value 'SOME_VALUE'). Digits will be separated from preceding lower-case
     *  characters, and from following upper-case characters with underscore
     *  separators too (e.g. the JSON string 'my2ndValue' will cause to create
     *  the enumeration value 'MY_2ND_VALUE').
     *
     * @returns {Function}
     *  The new enumeration class, containing all values specified in the value
     *  passed to this method as static class properties. All value properties
     *  will be instances of the new enumeration class. The static class method
     *  'parse' will parse a JSON string to an enumeration value, if possible,
     *  otherwise the provided default enumeration value or null will be
     *  returned.
     */
    Enum.create = function (jsonNames) {

        var map = {};

        var EnumClass = Enum.extend(function (json, index) { Enum.call(this, json, index); });
        EnumClass.parse = function (json, def) { return map[json] || def || null; };

        jsonNames.split(/\s+/).forEach(function (json, index) {
            var value = new EnumClass(json, index);
            if (value.name in EnumClass) { throw new Error('duplicate enumeration value'); }
            EnumClass[value.name] = map[json] = value;
        });

        return EnumClass;
    };

    // exports ================================================================

    return Enum;

});
