/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialog/inputdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialog/basedialog'
], function (Utils, BaseDialog) {

    'use strict';

    // class InputDialog ======================================================

    /**
     * Creates a simple text input dialog. Dialog validation will always be
     * triggered when typing into the text input field.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application that is
     *  creating the dialog, or an object with a method getWindowId() that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  BaseDialog. Additionally, the following options are supported:
     *  - {String} [initOptions.inputLabel]
     *      A leading label for the input field.
     *  - {String} [initOptions.value='']
     *      The initial value of the text field.
     *  - {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  - {Boolean} [initOptions.allowEmpty=false]
     *      If set to true, empty text will be considered valid. Otherwise, the
     *      primary button will be disabled as long as the text input field is
     *      empty.
     *  - {Number} [initOptions.maxLength]
     *      If specified, the maximum number of characters that can be entered.
     */
    var InputDialog = BaseDialog.extend(function (windowId, initOptions) {

        // maximum length allowed in the text field
        var maxLength = Utils.getIntegerOption(initOptions, 'maxLength', null, 1);

        // whether to allow empty text
        var allowEmpty = Utils.getBooleanOption(initOptions, 'allowEmpty', false);

        // the label text
        var inputLabel = Utils.getStringOption(initOptions, 'inputLabel', '');

        // the text label element
        var labelNode = inputLabel ? $('<label>').text(inputLabel) : $();

        // the text field control
        var inputNode = $('<input type="text" class="form-control" tabindex="0">').attr({
            placeholder: Utils.getStringOption(initOptions, 'placeholder', ''),
            value: Utils.getStringOption(initOptions, 'value', '')
        });

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, windowId, initOptions);

        // methods ------------------------------------------------------------

        /**
         * Returns the text input element, as jQuery object.
         *
         * @returns {jQuery}
         *  The text input element, as jQuery object.
         */
        this.getInputNode = function () {
            return inputNode;
        };

        /**
         * Returns the text that is contained in the text input element.
         *
         * @returns {String}
         *  The text that is contained in the text input element.
         */
        this.getText = function () {
            var text = inputNode.val();
            return (typeof maxLength === 'number') ? text.substr(0, maxLength) : text;
        };

        /**
         * Changes the text contained in the text input element.
         *
         * @param {String} text
         *  The text to be inserted into the text input element.
         *
         * @returns {InputDialog}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            inputNode.val(text);
            return this;
        };

        // initialization -----------------------------------------------------

        // add the text field
        this.append(labelNode, inputNode);

        // initialize maximum length
        if (typeof maxLength === 'number') {
            inputNode.val(inputNode.val().substr(0, maxLength));
            inputNode.attr('maxlength', maxLength);
        }

        // focus and select the text in the input field
        this.setFocusNode(inputNode);

        // trigger validation when typing in the input field
        this.validateOn(inputNode, 'input');

        // create a non-empty validator for the text input field
        if (!allowEmpty) {
            this.addOkValidator(function () { return inputNode.val().length; });
        }

    }); // class InputDialog

    // exports ================================================================

    return InputDialog;

});
