/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/typedarray', [
    'io.ox/office/tk/container/extarray'
], function (ExtArray) {

    'use strict';

    // convenience shortcuts
    var ArrayProto = Array.prototype;

    // class template TypedArray ==============================================

    var TypedArray = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates a sub class of the class ExtArray, intended to store array
     * elements of a specific type.
     *
     * @param {Function} ElementType
     *  The type (the constructor function) of the array elements. Must provide
     *  the instance method clone(). May or may not provide the static method
     *  ElementType.compare() which will be used as natural element order for
     *  sorting the array.
     *
     * @returns {Function}
     *  The constructor function of the typed array class. This class will be
     *  referred to as TypedArray<ElementType> in the following documentation.
     */
    TypedArray.create = function (ElementType) {

        /**
         * An array class with elements of a specific type that behave like
         * regular arrays (its super class is ExtArray), with a few additions.
         *
         * @constructor
         *
         * @extends ExtArray
         *
         * @param {Array<ElementType>|ElementType} ...
         *  An arbitrary number of parameters that will be inserted into the
         *  new instance. Each constructor parameter can be another instance of
         *  a plain or extended array class, or a single instance of the class
         *  ElementType. Array parameters will be concatenated (no nested
         *  arrays as elements).
         */
        var ArrayClass = ExtArray.extend(function () {
            ExtArray.call(this);
            this.append.apply(this, arguments);
        });

        // constants ----------------------------------------------------------

        /**
         * The type class of the elements contained in instances of this array.
         *
         * @type Function
         * @constant
         */
        ArrayClass.ElementType = ElementType;

        // static methods -----------------------------------------------------

        /**
         * If the passed argument is an instance of ArrayClass, it will be
         * returned directly. Otherwise, a new array will be created by passing
         * the value to the ArrayClass constructor.
         *
         * @param {Any} arg
         *  The value to be converted to a typed array.
         *
         * @returns {TypedArray<ElementType>}
         *  A typed array containing the passed value.
         */
        ArrayClass.from = ArrayClass.get = function (arg) {
            return (arg instanceof ArrayClass) ? arg : new ArrayClass(arg);
        };

        /**
         * Returns a new typed array with only the elements of the passed array
         * that pass a truth test. Works like the instance method
         * Array.filter().
         *
         * @param {TypedArray<ElementType>|Array<ElementType>|ElementType} arg
         *  An instance of TypedArray, a plain JS array, or a single instance
         *  of ElementType. In the latter case, the callback function will be
         *  invoked once for that value (as if it was a one-element array).
         *
         * @param {Function} predicate
         *  The predicate callback function invoked for each array element.
         *  Receives the following parameters:
         *  (1) {ElementType} value
         *      The value of the array element.
         *  (2) {Number} index
         *      The array index of the element.
         *  (3) {TypedArray<ElementType>|Array<ElementType>|ElementType} array
         *      The array parameter passed to the method.
         *  MUST return a truthy value if the element passes the test and has
         *  to be inserted into the resulting array.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {TypedArray<ElementType>}
         *  The new typed array with all elements that pass the truth test.
         */
        ArrayClass.filter = function (arg, predicate, context) {
            var array = new ArrayClass();
            ArrayClass.forEach(arg, function (element) {
                if (predicate.apply(context, arguments)) { array.append(element); }
            });
            return array;
        };

        /**
         * Returns a new typed array with only the elements of the passed array
         * that do not pass a truth test. Works like the instance method
         * Array.filter() but reverses the predicate.
         *
         * @param {TypedArray<ElementType>|Array<ElementType>|ElementType} arg
         *  An instance of TypedArray, a plain JS array, or a single instance
         *  of ElementType. In the latter case, the callback function will be
         *  invoked once for that value (as if it was a one-element array).
         *
         * @param {Function} predicate
         *  The predicate callback function invoked for each array element.
         *  Receives the following parameters:
         *  (1) {ElementType} value
         *      The value of the array element.
         *  (2) {Number} index
         *      The array index of the element.
         *  (3) {TypedArray<ElementType>|Array<ElementType>|ElementType} array
         *      The array parameter passed to the method.
         *  MUST return a truthy value if the element does not pass the test
         *  and will not be inserted into the resulting array.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {TypedArray<ElementType>}
         *  The new typed array with all elements that do not pass the truth
         *  test.
         */
        ArrayClass.reject = function (arg, predicate, context) {
            var array = new ArrayClass();
            ArrayClass.forEach(arg, function (element) {
                if (!predicate.apply(context, arguments)) { array.append(element); }
            });
            return array;
        };

        /**
         * Returns a new typed array with the return values of a callback
         * function transforming arbitrary values to instances of ElementType.
         * Works like the instance method Array.map().
         *
         * @param {TypedArray<Any>|Array<Any>|Any} arg
         *  An instance of TypedArray, a plain JS array, or any other value. In
         *  the latter case, the callback function will be invoked once for
         *  that value (as if it was a one-element array).
         *
         * @param {Function} callback
         *  The callback function invoked for each array element. Receives the
         *  following parameters:
         *  (1) {Any} value
         *      The value of the array element.
         *  (2) {Number} index
         *      The array index of the element.
         *  (3) {Any} array
         *      The array parameter passed to the method.
         *  MUST return an instance of ElementType, an array of such elements
         *  (either a plain JS array, or an instance of TypedArray), or a falsy
         *  value that will be filtered from the resulting array. If the return
         *  value is an array, its elements will be appended flatly (no nested
         *  array elements will be created).
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {TypedArray<ElementType>}
         *  The new typed array with the return values of the passed callback
         *  function.
         */
        ArrayClass.map = function (arg, callback, context) {
            var array = new ArrayClass();
            ArrayClass.forEach(arg, function () {
                array.append(callback.apply(context, arguments));
            });
            return array;
        };

        /**
         * Returns a new typed array with the return values of an instance
         * method of the array elements.
         *
         * @param {TypedArray<Object>|Array<Object>|Object} arg
         *  An instance of TypedArray containing objects, a plain JS array with
         *  objects, or any other object. In the latter case, the instance
         *  method will be invoked once for that object (as if it was a
         *  one-element array).
         *
         * @param {String} method
         *  The name of the instance method that will be invoked on the array
         *  elements. The method MUST return an instance of ElementType, an
         *  array of such elements (either a plain JS array, or an instance of
         *  TypedArray), or a falsy value that will be skipped. If the return
         *  value is an array, its elements will be appended flatly (no nested
         *  array elements will be created).
         *
         * @param {Any} ...
         *  All following parameters will be passed to the element method.
         *
         * @returns {TypedArray<ElementType>}
         *  The new typed array with the return values of the instance method.
         */
        ArrayClass.invoke = function (arg, method) {
            var methodArgs = ArrayProto.slice.call(arguments, 2);
            return ArrayClass.map(arg, function (element) {
                return element[method].apply(element, methodArgs);
            });
        };

        /**
         * Distributes the elements of the source array into a map (a plain JS
         * object) of typed arrays that are keyed by the return value of the
         * callback function.
         *
         * @param {TypedArray<ElementType>|Array<ElementType>|ElementType} arg
         *  An instance of TypedArray, a plain JS array, or a single instance
         *  of ElementType. In the latter case, the callback function will be
         *  invoked once for that value (as if it was a one-element array).
         *
         * @param {Function} callback
         *  The callback function invoked for each array element. Receives the
         *  following parameters:
         *  (1) {ElementType} value
         *      The value of the array element.
         *  (2) {Number} index
         *      The array index of the element.
         *  (3) {TypedArray<ElementType>|Array<ElementType>|ElementType} array
         *      The array parameter passed to the method.
         *  MUST return a string (or any other value that can be converted to a
         *  string) that will be used as map key for the result.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {Object<String,TypedArray<ElementType>>}
         *  A map with multiple typed arrays containing the elements grouped by
         *  the results of the callback function.
         */
        ArrayClass.group = function (arg, callback, context) {
            var map = {};
            ArrayClass.forEach(arg, function (element) {
                var key = callback.apply(context, arguments);
                (map[key] || (map[key] = new ArrayClass())).push(element);
            });
            return map;
        };

        // public methods -----------------------------------------------------

        /**
         * Appends all passed parameters to this array in-place, similar to the
         * methods push() and unshift().
         *
         * @param {TypedArray<ElementType>|Array<ElementType>|ElementType} ...
         *  All values to be appended to this array. Each parameter can be an
         *  instance of TypedArray, a plain JS array, or a single instance of
         *  ElementType. All elements from array parameters will be appended
         *  directly to this array (no nested array elements). Invalid values
         *  (for example null or undefined) will be skipped silently.
         *
         * @returns {TypedArray<ElementType>}
         *  A reference to this array.
         */
        ArrayClass.prototype.append = function () {
            var push = function (element) { this.push(element); }.bind(this);
            for (var i = 0; i < arguments.length; i += 1) {
                // work-around for a problem in PhantomJS (and probably other JS engines) where
                // using Array.push.apply() with an instance of ArrayClass as parameter fails
                var arg = arguments[i];
                if (arg instanceof ArrayClass) {
                    arg.forEach(push);
                } else if (arg instanceof Array) {
                    this.push.apply(this, arg);
                } else if (arg instanceof ElementType) {
                    this.push(arg);
                }
            }
            return this;
        };

        /**
         * Creates a shallow clone of this array, and appends the passed
         * parameters to it. Overwrites the instance method Array.concat() to
         * be able to return a TypedArray instance.
         *
         * @param {TypedArray<ElementType>|Array<ElementType>|ElementType} ...
         *  All values to be appended to the clone of this array. Each
         *  parameter can be an instance of TypedArray, a plain JS array, or a
         *  single instance of ElementType. All elements from array parameters
         *  will be appended directly to the cloned array (no nested array
         *  elements). Invalid values (for example null or undefined) will be
         *  skipped silently.
         *
         * @returns {TypedArray<ElementType>}
         *  The new array containing the elements of this array, and the passed
         *  parameters.
         */
        ArrayClass.prototype.concat = function () {
            var array = this.clone();
            array.append.apply(array, arguments);
            return array;
        };

        /**
         * Sorts the elements of this array in-place. Overwrites the instance
         * method Array.sort() to be able to change the default behaviour
         * without a sort order callback function.
         *
         * @param {Function} [comparator=ElementType.compare]
         *  If specified, a callback function that receives two array elements,
         *  and returns whether the first element is less than, equal to, or
         *  greater then the second element. If omitted, and the ElementType
         *  contains a static method compare(), it will be used automatically
         *  for element comparison.
         *
         * @returns {TypedArary<ElementType>}
         *  A reference to this array.
         */
        ArrayClass.prototype.sort = (typeof ElementType.compare === 'function') ? function (comparator) {
            return ArrayProto.sort.call(this, comparator || ElementType.compare);
        } : ArrayProto.sort;

        /**
         * Sorts the elements of this array in-place. The elements will be
         * sorted according to the return values of the callback function, or
         * to the property value of the elements specified as string parameter.
         *
         * @param {Function|String} callback
         *  A callback function that receives an array element, and returns a
         *  sort index associated to that array element. The sort indexes MUST
         *  be comparable with the built-in JS compare operator "less than".
         *  Alternatively, this parameter can be a string specifying the name
         *  of an object property to be fetched from the object elements for
         *  comparison.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {TypedArary<ElementType>}
         *  A reference to this array.
         */
        ArrayClass.prototype.sortBy = function (callback, context) {
            var key = (typeof callback === 'string') ? _.property(callback) : callback.bind(context);
            return this.sort(function (elem1, elem2) {
                var key1 = key(elem1), key2 = key(elem2);
                return (key1 < key2) ? -1 : (key2 < key1) ? 1 : 0;
            });
        };

        /**
         * Converts this array to a plain JSON array with JSON elements.
         *
         * @returns {Array<Any>}
         *  A plain JSON array with JSON elements.
         */
        ArrayClass.prototype.toJSON = (typeof ElementType.prototype.toJSON === 'function') ?
            function () { return _.invoke(this, 'toJSON'); } :
            function () { return ArrayProto.map.call(this, _.identity); };

        // --------------------------------------------------------------------

        return ArrayClass;
    };

    // exports ================================================================

    return TypedArray;

});
