/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/extarray', [
    'io.ox/office/tk/utils/class',
    'io.ox/office/tk/utils/iterator'
], function (Class, Iterator) {

    'use strict';

    // convenience shortcuts
    var ArrayProto = Array.prototype;
    var ArrayIterator = Iterator.ArrayIterator;

    // class ExtArray =========================================================

    /**
     * A sub class of Array with additional instance methods.
     *
     * @constructor
     *
     * @extends Array
     */
    var ExtArray = Class.extend(Array, function () {
        Array.call(this);
    });

    // static methods ---------------------------------------------------------

    // convenience shortcuts
    var ExtArrayProto = ExtArray.prototype;

    /**
     * Creates an array iterator that can be used to visit the elements of the
     * passed value (see class Iterator.ArrayIterator for details).
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, an
     *  iterator for that value will be returned (as if it was a one-element
     *  array).
     */
    ExtArray.iterator = function (arg, options) {
        return new ArrayIterator((arg instanceof Array) ? arg : [arg], options);
    };

    /**
     * Visits all elements in the passed array. Works like the instance method
     * Array.forEach().
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, the
     *  callback function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * @param {Function} callback
     *  The callback function invoked for each array element. Receives the
     *  following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {Array|Any} array
     *      The parameter passed to the method.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    ExtArray.forEach = function (arg, callback, context) {
        if (arg instanceof Array) { arg.forEach(callback, context); } else { callback.call(context, arg, 0, arg); }
    };

    /**
     * Visits all elements in the passed array in reversed order. Works like
     * the instance method Array.forEach().
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, the
     *  callback function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * @param {Function} callback
     *  The callback function invoked for each array element. Receives the
     *  following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {Array|Any} array
     *      The parameter passed to the method.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    ExtArray.forEachReverse = function (arg, callback, context) {
        if (arg instanceof Array) {
            ExtArrayProto.forEachReverse.call(arg, callback, context);
        } else {
            callback.call(context, arg, 0, arg);
        }
    };

    /**
     * Returns whether at least one element in the passed array passes a truth
     * test. Works like the instance method Array.some().
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, the
     *  callback function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {Array|Any} array
     *      The parameter passed to the method.
     *  MUST return a truthy value if the element passes the test.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Boolean}
     *  Whether at least one element in the passed array passes the truth test.
     */
    ExtArray.some = function (arg, predicate, context) {
        return (arg instanceof Array) ? arg.some(predicate, context) : !!predicate.call(context, arg, 0, arg);
    };

    /**
     * Returns whether at least one element in the passed array passes a truth
     * test. Works like the instance method Array.some(), but visits the array
     * elements in reversed order.
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, the
     *  callback function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {Array|Any} array
     *      The parameter passed to the method.
     *  MUST return a truthy value if the element passes the test.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Boolean}
     *  Whether at least one element in the passed array passes the truth test.
     */
    ExtArray.someReverse = function (arg, predicate, context) {
        return (arg instanceof Array) ? ExtArrayProto.someReverse.call(arg, predicate, context) : !!predicate.call(context, arg, 0, arg);
    };

    /**
     * Returns whether all elements in the passed array pass a truth test.
     * Works like the instance method Array.every().
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, the
     *  callback function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {Array|Any} array
     *      The parameter passed to the method.
     *  MUST return a truthy value if the element passes the test.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Boolean}
     *  Whether all elements in the passed array pass the truth test.
     */
    ExtArray.every = function (arg, predicate, context) {
        return (arg instanceof Array) ? arg.every(predicate, context) : !!predicate.call(context, arg, 0, arg);
    };

    /**
     * Returns whether all elements in the passed array pass a truth test.
     * Works like the instance method Array.every(), but visits the array
     * elements in reversed order.
     *
     * @param {Array|Any} arg
     *  A plain or extended array, or any other value. In the latter case, the
     *  callback function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {Array|Any} array
     *      The parameter passed to the method.
     *  MUST return a truthy value if the element passes the test.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Boolean}
     *  Whether all elements in the passed array pass the truth test.
     */
    ExtArray.everyReverse = function (arg, predicate, context) {
        return (arg instanceof Array) ? ExtArrayProto.everyReverse.call(arg, predicate, context) : !!predicate.call(context, arg, 0, arg);
    };

    // protected methods ------------------------------------------------------

    /**
     * Creates a new empty instance of this class. Intended to be overwritten
     * by sub classes if construction of new instances needs additional work,
     * e.g. if the constructor expects some parameters.
     *
     * @returns {ExtArray}
     *  A new empty array of the same type as this instance.
     */
    ExtArray.prototype._new = function () {
        return new this.constructor();
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether the array is empty.
     *
     * @returns {Boolean}
     *  Whether the array is empty.
     */
    ExtArray.prototype.empty = function () {
        return this.length === 0;
    };

    /**
     * Returns the first element of this array.
     *
     * @returns {Any|Null}
     *  The first element of this array; or null, if the array is empty.
     */
    ExtArray.prototype.first = function () {
        return this.empty() ? null : this[0];
    };

    /**
     * Returns the last element of this array.
     *
     * @returns {Any|Null}
     *  The last element of this array; or null, if the array is empty.
     */
    ExtArray.prototype.last = function () {
        return this.empty() ? null : this[this.length - 1];
    };

    /**
     * Removes all elements from this array.
     *
     * @returns {ExtArray}
     *  A reference to this array.
     */
    ExtArray.prototype.clear = function () {
        ArrayProto.splice.call(this, 0);
        return this;
    };

    /**
     * Returns a shallow or deep clone of this array.
     *
     * @param {Boolean} [deep=false]
     *  Whether to return a shallow copy (false, all array elements copied by
     *  reference), or a deep copy (true, array elements will be cloned too).
     *  For a deep clone, the array elements MUST be objects, and they MUST
     *  provide their own clone() instance method. The parameter "deep" will be
     *  passed to the element method in order to recursively clone the elements
     *  deeply.
     *
     * @returns {ExtArray}
     *  The clone of this array. If this method will be used by a sub class of
     *  ExtArray, another instance of that sub class will be created
     *  automatically.
     */
    ExtArray.prototype.clone = function (deep) {
        var array = this._new();
        this.forEach(function (element) {
            ArrayProto.push.call(array, deep ? element.clone(deep) : element);
        });
        return array;
    };

    /**
     * Returns whether this array is equal to the specified array.
     *
     * @param {Array<Any>} array
     *  The array to be compared with this array.
     *
     * @param {Boolean} [deep=false]
     *  Whether to compare the array elements by reference (false, uses the
     *  "===" operator for the array elements), or to compare them deeply
     *  (true). To compare array elements deeply, the elements MUST provide an
     *  equals() instance method. The parameter "deep" will be passed after the
     *  array element as second parameter to the method equals() of the array
     *  element in order recursively compare the elements deeply.
     *
     * @returns {Boolean}
     *  Whether this array is equal to the specified array.
     */
    ExtArray.prototype.equals = function (array, deep) {
        return (this.length === array.length) && this.every(function (element, index) {
            return deep ? element.equals(array[index], true) : (element === array[index]);
        });
    };

    /**
     * Creates an array iterator that can be used to visit the elements of this
     * array.
     *
     * @param {Object} [options]
     *  Optional parameters passed to the constructor of the array iterator.
     *  See class Iterator.ArrayIterator for details.
     *
     * @returns {ArrayIterator}
     *  The array iterator that will visit the elements of this array.
     */
    ExtArray.prototype.iterator = function (options) {
        return new ArrayIterator(this, options);
    };

    /**
     * Returns a subset of the elements of this array. Overwrites the instance
     * method Array.slice() to be able to return an instance of this class.
     *
     * @param {Number} [begin]
     *  The array index of the first element to be returned. Behaviour is
     *  equal to the standard method Array.slice().
     *
     * @param {Number} [end]
     *  The array index of the first element not to be returned anymore.
     *  Behaviour is equal to the standard method Array.slice().
     *
     * @returns {ExtArray}
     *  The new array containing the specified elements of this array. If this
     *  method will be used by a sub class of ExtArray, another instance of
     *  that sub class will be created automatically.
     */
    ExtArray.prototype.slice = function () {
        // resulting array may be a sub class (order and type of elements do not change)
        var array = this._new();
        ArrayProto.slice.apply(this, arguments).forEach(function (element) {
            ArrayProto.push.call(array, element);
        });
        return array;
    };

    /**
     * Visits all elements in this array in reversed order. Works like the
     * instance method Array.forEach().
     *
     * @param {Function} callback
     *  The callback function invoked for each array element. Receives the
     *  following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    ExtArray.prototype.forEachReverse = function (callback, context) {
        var iterator = new ArrayIterator(this, { reverse: true });
        Iterator.forEach(iterator, function (element, result) {
            callback.call(context, element, result.index, this);
        }, this);
    };

    /**
     * Returns whether at least one element in this array passes a truth test.
     * Works like the instance method Array.some(), but visits the array
     * elements in reversed order.
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a truthy value if the element passes the test.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Boolean}
     *  Whether at least one element in this array passes the truth test.
     */
    ExtArray.prototype.someReverse = function (predicate, context) {
        var iterator = new ArrayIterator(this, { reverse: true });
        return Iterator.some(iterator, function (element, result) {
            return predicate.call(context, element, result.index, this);
        }, this);
    };

    /**
     * Returns whether all elements in this array pass a truth test. Works like
     * the instance method Array.every(), but visits the array elements in
     * reversed order.
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a truthy value if the element passes the test.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Boolean}
     *  Whether all elements in this array pass the truth test.
     */
    ExtArray.prototype.everyReverse = function (predicate, context) {
        var iterator = new ArrayIterator(this, { reverse: true });
        return Iterator.every(iterator, function (element, result) {
            return predicate.call(context, element, result.index, this);
        }, this);
    };

    /**
     * Returns a new array with only the elements of this array that pass a
     * truth test. Overwrites the instance method Array.filter() to be able to
     * return an instance of this class.
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a truthy value if the element passes the test and has to be
     *  inserted into the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {ExtArray}
     *  The new array with all elements that pass the truth test. If this
     *  method will be used by a sub class of ExtArray, another instance of
     *  that sub class will be created automatically.
     */
    ExtArray.prototype.filter = function (predicate, context) {
        // resulting array may be a sub class (order and type of elements do not change)
        var array = this._new();
        this.forEach(function (element, index) {
            if (predicate.call(context, element, index, this)) {
                ArrayProto.push.call(array, element);
            }
        }, this);
        return array;
    };

    /**
     * Returns a new array with only the elements of this array that do not
     * pass a truth test. Works like the instance method ExtArray.filter() but
     * reverses the predicate.
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a truthy value if the element does not pass the test and
     *  will not be inserted into the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {ExtArray}
     *  The new array with all elements that do not pass the truth test. If
     *  this method will be used by a sub class of ExtArray, another instance
     *  of that sub class will be created automatically.
     */
    ExtArray.prototype.reject = function (predicate, context) {
        return this.filter(function (element, index) {
            return !predicate.call(context, element, index, this);
        }, this);
    };

    /**
     * Returns a new array with the results of a callback function invoked for
     * every element in this array. Overwrites the instance method
     * Array.filter() to be able to return an instance of this class.
     *
     * @param {Function} callback
     *  The callback function invoked for each array element. Receives the
     *  following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  The return value will be inserted into the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {ExtArray}
     *  The new array with all return values of the callback function.
     */
    ExtArray.prototype.map = function (callback, context) {
        // do not create a sub class (order and type of elements may change)
        var array = new ExtArray();
        this.forEach(function (element, index) {
            ArrayProto.push.call(array, callback.call(context, element, index, this));
        }, this);
        return array;
    };

    /**
     * Returns a new array with the values of a property of the array elements.
     *
     * @param {String} prop
     *  The name of the property that will be returned from the elements.
     *
     * @returns {ExtArray}
     *  The new array with the values of the element properties.
     */
    ExtArray.prototype.pluck = function (prop) {
        return this.map(function (element) { return element[prop]; });
    };

    /**
     * Returns a new array with the return values of an instance method of the
     * array elements.
     *
     * @param {String} method
     *  The name of the instance method that will be invoked on the array
     *  elements.
     *
     * @param {Any} ...
     *  All following parameters will be passed to the element method.
     *
     * @returns {ExtArray}
     *  The new array with the return values of the instance method.
     */
    ExtArray.prototype.invoke = function (method) {
        var args = ArrayProto.slice.call(arguments, 1);
        return this.map(function (element) {
            return element[method].apply(element, args);
        });
    };

    /**
     * Distributes the elements of this array into a map (a plain JS object) of
     * arrays that are keyed by the return value of the callback function.
     *
     * @param {Function} callback
     *  The callback function invoked for each array element. Receives the
     *  following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a string (or any other value that can be converted to a
     *  string) that will be used as map key for the result.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object<String,ExtArray>}
     *  A map with multiple arrays containing the elements grouped by the
     *  results of the callback function. If this method will be used by a sub
     *  class of ExtArray, other instances of that sub class will be created as
     *  map elements automatically.
     */
    ExtArray.prototype.group = function (callback, context) {
        var map = {};
        this.forEach(function (element, index) {
            var key = callback.call(context, element, index, this);
            // resulting arrays may be a sub class (order and type of elements do not change)
            var array = (map[key] || (map[key] = this._new()));
            ArrayProto.push.call(array, element);
        }, this);
        return map;
    };

    /**
     * Returns the first element of this array that passes a truth test.
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a truthy value if the element passes the test and has to be
     *  returned.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The first element from this array that passes the truth test; or
     *  undefined, if no such element exists.
     */
    ExtArray.prototype.find = ArrayProto.find || function (predicate, context) {
        var found;
        this.some(function (element, index) {
            if (predicate.call(context, element, index, this)) {
                found = element;
                return true;
            }
        }, this);
        return found;
    };

    /**
     * Returns the last element of this array that passes a truth test.
     *
     * @param {Function} predicate
     *  The predicate callback function invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the array element.
     *  (2) {Number} index
     *      The array index of the element.
     *  (3) {ExtArray} array
     *      A reference to this array.
     *  MUST return a truthy value if the element passes the test and has to be
     *  returned.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The last element from this array that passes the truth test; or
     *  undefined, if no such element exists.
     */
    ExtArray.prototype.findLast = function (predicate, context) {
        var found;
        this.someReverse(function (element, index) {
            if (predicate.call(context, element, index, this)) {
                found = element;
                return true;
            }
        }, this);
        return found;
    };

    /**
     * Convenience method to destroy all elements of this array, and clear the
     * array afterwards. All elements MUST be objects with a method destroy().
     *
     * @returns {ExtArray}
     *  A reference to this (now empty) array.
     */
    ExtArray.prototype.destroyElements = function () {
        this.forEach(function (element) { element.destroy(); });
        return this.clear();
    };

    /**
     * Stringifies all array elements, and joins them with the passed separator
     * text.
     *
     * @param {String} [separator=',']
     *  The separator text to be inserted between the string representation of
     *  the array elements.
     *
     * @returns {String}
     *  The string representation of this array.
     */
    ExtArray.prototype.toString = function (separator) {
        return this.join(separator);
    };

    /**
     * Converts this array to a JSON array with JSON elements.
     *
     * @returns {Array}
     *  A JSON array with JSON elements.
     */
    ExtArray.prototype.toJSON = function () {
        return ArrayProto.map.call(this, function (element) {
            return (element.toJSON instanceof Function) ? element.toJSON() : element;
        });
    };

    // exports ================================================================

    return ExtArray;

});
