/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/storageoperationmixin', [
    'io.ox/office/textframework/utils/operations'
], function (Operations) {

    'use strict';

    // mix-in class StorageOperationMixin =====================================

    /**
     * A mix-in class for saving the operations that are needed to load a
     * document from local storage.
     *
     * @param {EditApplication} app
     *  The application containing this model instance.
     *
     * @constructor
     */
    function StorageOperationMixin(app) {

        var // self reference for local functions
            self = this,
            // a collector for all those operations that are needed to load a document from the local storage
            storageOperationCollector = [],
            // Load performance: List if operation names for operations that need to be stored in the local storage, so that the
            // document can be loaded without operations and fast load string.
            // These operations do not modify the DOM, but register data in the global models:
            // document attributes, themes, font descriptions, style sheets and list styles.
            STORAGE_OPERATIONS = {},
            // whether the collector already contains the unique 'setDocumentAttributes' operation
            sdaIncluded = false,
            // whether the local storage is supported
            isLocalStorageSupported = true;

        // list of operations that need to be handled for local storage support
        (function () {
            STORAGE_OPERATIONS[Operations.SET_DOCUMENT_ATTRIBUTES] = { isSda: true };
            STORAGE_OPERATIONS[Operations.INSERT_THEME] = { insert: true };
            STORAGE_OPERATIONS[Operations.INSERT_FONT_DESCRIPTION] = { insert: true };
            STORAGE_OPERATIONS[Operations.INSERT_STYLESHEET] = { insert: true };
            STORAGE_OPERATIONS[Operations.DELETE_STYLESHEET] = { delete: Operations.INSERT_STYLESHEET, properties: ['styleId', 'type'] };
            STORAGE_OPERATIONS[Operations.CHANGE_STYLESHEET] = { isCss: true, change: Operations.INSERT_STYLESHEET, properties: ['styleId', 'type'] };
            STORAGE_OPERATIONS[Operations.INSERT_LIST] = { insert: true };
            STORAGE_OPERATIONS[Operations.DELETE_LIST] = { delete: Operations.INSERT_LIST, properties: ['listStyleId'] };
        }());

        // private methods ----------------------------------------------------

        /**
         * Helper function for the operations deleteList and deleteStyleSheet.
         * These functions need to remove the corresponding insertList or
         * insertStyleSheet operations from the operations collector.
         *
         * @param {Object} operation
         *  An operation object.
         */
        function deleteOperationFromCollector(operation) {

            // insertStyleSheet and deleteStyleSheet have properties 'styleId' and 'type'
            // insertListStyle and deleteListStyle have property 'listStyleId' (only via undo)

            var compareObject = { name: STORAGE_OPERATIONS[operation.name].delete };
            _.each(STORAGE_OPERATIONS[operation.name].properties, function (prop) {
                compareObject[prop] = operation[prop];
            });

            // searching the corresponding operation to remove it from collector
            var op = _.findWhere(storageOperationCollector, compareObject);

            // removing the one found operation from the collector
            if (op) { storageOperationCollector = _.without(storageOperationCollector, op); }
        }

        /**
         * Helper function to merge 'setDocumentAttributes' operations.
         *
         * @param {Object} operation
         *  An operation object.
         */
        function mergeSetDocumentAttributesOperation(operation) {

            // searching the corresponding operation in the collector
            var op = _.findWhere(storageOperationCollector, operation.name);

            // merging content of new operation into existing operation
            if (op && op.attrs && operation.attrs) {
                self.getAttributePool().extendAttributeSet(op.attrs, operation.attrs);
                // special handling for document attributes (they are not handled by attributePool) // 56033
                if (operation.attrs.document) {
                    _.each(operation.attrs.document, function (val, key) {
                        op.attrs.document = op.attrs.document || {};
                        op.attrs.document[key] = _.copy(val, true);
                    });
                }
            }
        }

        /**
         * Helper function to merge 'changeStyleSheet' operations.
         *
         * @param {Object} operation
         *  An operation object.
         */
        function mergeChangeStyleSheetOperation(operation) {

            var compareObject = { name: STORAGE_OPERATIONS[operation.name].change };
            _.each(STORAGE_OPERATIONS[operation.name].properties, function (prop) {
                compareObject[prop] = operation[prop];
            });

            // searching the corresponding operation in the collector
            var op = _.findWhere(storageOperationCollector, compareObject);

            // merging content of new operation into existing operation
            if (op && op.attrs && operation.attrs) { self.getAttributePool().extendAttributeSet(op.attrs, operation.attrs); }
        }

        // public methods -----------------------------------------------------

        /**
         * Load performance: Getting the selector for selected operations that need to be stored in the local storage.
         *
         * @returns {Object[]}
         *  A container with all operations that need to be stored in the local storage.
         */
        this.getStorageOperationCollector = function () {
            return storageOperationCollector;
        };

        /**
         * Load performance: Collecting selected operations in collector for saving them in local storage. This
         * makes it possible to load documents without any operations or fast load string, because all required
         * data is saved in the local storage.
         * The operations need to be saved for every supported loading process. Also if 'Use local storage' is
         * disabled, because it might be enabled by the user later.
         * The operations should not be collected, if the user does not has the ability to use the local storage.
         *
         * @param {Object} operation
         *  An operation object.
         */
        this.saveStorageOperation = function (operation) {

            // Leave immediately, if local immediately storage is not supported.
            // Checking:
            // localStorageApp: whether the app supports local storage -> only those apps contain this function.
            // isLocalStorageSupported: whether the system supports local storage (this is checked below)
            // saveFileInLocalStorage: whether the user can use local storage -> might be changed by the user
            //                         during editing the document.

            if (!isLocalStorageSupported || !operation || !operation.name || !STORAGE_OPERATIONS[operation.name]) { return; }

            // insert operations can be added simply to the collector
            if (STORAGE_OPERATIONS[operation.name].insert) {
                storageOperationCollector.push(operation);
                return;
            }

            // delete operations must remove an already existing operation
            if (STORAGE_OPERATIONS[operation.name].delete) {
                deleteOperationFromCollector(operation);
                return;
            }

            // handling for setDocumentAttributes operation, that must stay unique
            if (STORAGE_OPERATIONS[operation.name].isSda) {
                if (sdaIncluded) {
                    mergeSetDocumentAttributesOperation(operation);
                } else {
                    storageOperationCollector.push(operation);
                    sdaIncluded = true;
                }
                return;
            }

            // handling for changeStyleSheet operation
            if (STORAGE_OPERATIONS[operation.name].isCss) {
                mergeChangeStyleSheetOperation(operation);
                return;
            }

        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = storageOperationCollector = null;
        });

        // getting info about support of local storage
        isLocalStorageSupported = app.isLocalStorageSupported();

    } // class StorageOperationMixin

    // exports ================================================================

    return StorageOperationMixin;

});
