/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define.async('io.ox/office/textframework/model/numberformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter'
], function (Utils, LocaleData, Parser, Formatter) {

    'use strict';

    // all predefined number format codes, mapped by category
    var categoryCodesDatabase = {};

    // load the localized number format definitions
    var numberFormatsPromise = LocaleData.loadResource('io.ox/office/textframework/resource/numberformats').done(function (data) {
        categoryCodesDatabase = data.categories;
    });

    // class NumberFormatter ==================================================

    /**
     * A number formatter for a text document.
     *
     * @constructor
     *
     * @extends Formatter
     *
     * @param {TextModel} docModel
     *  The document model containing this instance.
     */
    var NumberFormatter = Formatter.extend({ constructor: function (docModel) {

        // the file format may influence number formats
        var fileFormat = docModel.getApp().getFileFormat();

        // base constructor ---------------------------------------------------

        Formatter.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Returns the parsed number format descriptor for the specified number
         * format.
         *
         * @param {String} formatCode
         *  The format code to be parsed.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.grammarId='op']
         *      The identifier of a format grammar. See description of the
         *      static method Parser.parseFormatCode() for details.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor of a number format code.
         */
        this.getParsedFormat = function (formatCode, options) {
            var grammarId = Utils.getStringOption(options, 'grammarId', 'op');
            return Parser.parseFormatCode(fileFormat, grammarId, formatCode);
        };

        /**
         * Returns all available number format codes for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {Array|Null}
         *  The format codes for the specified category, or null if no format
         *  codes have been defined for the category. Each element in the
         *  returned array is an object with the following properties:
         *  - {String} value
         *      The actual format code.
         *  - {String} label
         *      A caption label with a formatted example number, intended to be
         *      used in GUI form controls.
         *  - {Boolean} [red=false]
         *      If set to true, the format code contains a [RED] color modifier
         *      for negative numbers.
         */
        this.getCategoryCodes = function (category) {
            var categoryFormats = categoryCodesDatabase[category.toLocaleLowerCase()];
            return _.isArray(categoryFormats) ? _.copy(categoryFormats, true) : null;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class NumberFormatter

    // exports ================================================================

    return numberFormatsPromise.then(_.constant(NumberFormatter));

});
