/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/field/fieldmanager', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/forms',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/field/simplefield',
    'io.ox/office/textframework/components/field/complexfield',
    'gettext!io.ox/office/textframework/main'
], function (DateUtils, LocaleData, TriggerObject, Forms, AppObjectMixin, Utils, Operations, DOM, Position, SimpleField, ComplexField, gt) {

    'use strict';

    // class FieldManager =====================================================

    /**
     * An instance of this class represents the dispatcher class
     * for all fields (simple and complex) in the edited document.
     *
     * @constructor
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function FieldManager(app) {

        var // self reference
            self = this,
            // simple field instance
            simpleField = new SimpleField(app),
            // complex field instance
            complexField = new ComplexField(app),
            // number formatter instance
            numberFormatter = null,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // range marker object
            rangeMarker = null,
            // flag to determine if cursor position is inside complex field
            isCursorHighlighted = false,
            // saved id of currently highlighted complex field during cursor traversal or click
            currentlyHighlightedId,
            // for simple fields we store node instance of highlighted field
            currentlyHighlightedNode,
            // field format popup promise
            fieldFormatPopupTimeout = null,
            // holds value of type of highlighted field
            highlightedFieldType = null,
            // holds value of format instruction of highlighted field
            highlightedFieldInstruction = null,
            // structure that contains field formats for given field type
            authorFileNameArray = [
                { option: 'default', value: /*#. dropdown list option: original string value, no formatting applied */ gt('(no format)') },
                { option: 'Lower', value: /*#. dropdown list option: all letters in string lowercase */ gt('lowercase') },
                { option: 'Upper', value: /*#. dropdown list option: all letters in string uppercase */ gt('UPPERCASE') },
                { option: 'FirstCap', value: /*#. dropdown list option: first letter uppercase only */ gt('First capital') },
                { option: 'Caps', value: /*#. dropdown list option: all begining letters in words uppercase */ gt('Title case') }
            ],
            pageNumberArray = [
                { option: 'default', value: /*#. dropdown list option: original string value */ gt('(no format)') },
                { option: 'roman', value: 'i, ii, iii, ...' },
                { option: 'ROMAN', value: 'I, II, III, ...' },
                { option: 'alphabetic', value: 'a, b, c, ...' },
                { option: 'ALPHABETIC', value: 'A, B, C, ...' },
                { option: 'ArabicDash', value: '- 1 -, - 2 -, - 3 -, ...' }
            ],
            localFormatList = (app.isODF()) ? {
                date: [],
                time: [],
                creator: [],
                'file-name': [],
                'page-number': [
                    { option: 'default', value: /*#. dropdown list option: original string value, no formatting applied */ gt('(no format)') },
                    { option: 'i', value: 'i, ii, iii, ...' },
                    { option: 'I', value: 'I, II, III, ...' },
                    { option: 'a', value: 'a, b, c, ...' },
                    { option: 'A', value: 'A, B, C, ...' }
                ],
                'page-count': [
                    { option: 'default', value: /*#. dropdown list option: original string value, no formatting applied */ gt('(no format)') },
                    { option: 'i', value: 'i, ii, iii, ...' },
                    { option: 'I', value: 'I, II, III, ...' },
                    { option: 'a', value: 'a, b, c, ...' },
                    { option: 'A', value: 'A, B, C, ...' }
                ]
            } : {
                date: [],
                time: [],
                author: authorFileNameArray,
                filename: authorFileNameArray,
                pagenumber: pageNumberArray,
                page: pageNumberArray,
                numpages: [
                    { option: 'default', value: /*#. dropdown list option: original string value, no formatting applied */ gt('(no format)') },
                    { option: 'roman', value: 'i, ii, iii, ...' },
                    { option: 'ROMAN', value: 'I, II, III, ...' },
                    { option: 'alphabetic', value: 'a, b, c, ...' },
                    { option: 'ALPHABETIC', value: 'A, B, C, ...' },
                    { option: 'Hex', value: /*#. dropdown list option: hexadecimal number formatting */ gt('hexadecimal') },
                    { option: 'Ordinal', value: '1., 2., 3., ...' }
                ]
            },
            // locale date formats
            categoryCodesDate = [],
            // locale time formats
            categoryCodesTime = [],
            // locale mix date-time formats
            categoryCodesDateTime = [],
            // temporary collection of special complex field nodes, used during changetracking
            tempSpecFieldCollection = [],
            // complex field stack for id
            complexFieldStackId = [];

        // base constructors --------------------------------------------------

        TriggerObject.call(this, app);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Private function called after importing document.
         * It loops all fields in collection and:
         *  - marks them for highlighting,
         *  - updates date and time fields
         *  - converts page fields in header&footer to special fields
         */
        function updateDateTimeAndSpecialFields() {
            complexField.updateDateTimeAndSpecialFields();
            simpleField.updateDateTimeFieldOnLoad();
        }

        /**
         * Receiving the start and end range markers for complex fields. Using the
         * current selection, this needs to be expanded, so that a complex field
         * is always inserted completely.
         * If the current selection is a range, it might be necessary to modify only
         * the start position or only the end position or both. If the range is inside
         * one complex field, it is not expanded. The aim of this expansion is, that,
         * if the selection includes a start range marker, the end range marker must
         * also be part of the selection. And vice versa.
         * There is another type of expansion for complex fields of type placeholder.
         * In this case, the selection shall always contain the complete placeholder.
         * Therefore the expansion also happens, if the selection has no range. This
         * check needs to be enabled because of performance reasons by the option
         * 'handlePlaceHolderComplexField'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.handlePlaceHolderComplexField=false]
         *      If set to true, it is not necessary, that the current selection
         *      has a range, that will be expanded. For complex fields of type
         *      'placeholder' also a cursor selection is automatically expanded.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'start' and 'end', that contain
         *  as values the range marker start node and the range marker end node.
         *  If they can be determined, they are already jQuerified. If not, this
         *  values can be null. If range markers cannot be determined, null is
         *  returned instead of an object.
         */
        function getRangeMarkersFromSelection(options) {

            var // the start node of the current selection
                startNode = null,
                // the end node of the current selection
                endNode = null,
                // whether the start of the selection is inside a complex field range
                startIsComplexField = false,
                // whether the end of the selection is inside a complex field range
                endIsComplexField = false,
                // the id of the complex field at the start position
                startId = null,
                // the id of the complex field at the end position
                endId = null,
                // the range marker start node for the complex field
                startRangeMarker = null,
                // the range marker end node for the complex field
                endRangeMarker = null,
                // whether the range of a complex field place holder node need to be set completely
                handlePlaceHolderComplexField = Utils.getBooleanOption(options, 'handlePlaceHolderComplexField', false);

            // only expand ranges or complex fields of type 'placeholder', not text cursors inside complex fields
            if (!((selection.hasRange() || handlePlaceHolderComplexField) && selection.getSelectionType() === 'text')) { return null; }

            startNode = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
            endNode = Position.getDOMPosition(selection.getRootNode(), selection.getEndPosition());

            // TODO: Ignoring nested complex fields yet

            if (startNode && startNode.node && endNode && endNode.node) {

                startNode = startNode.node;
                endNode = endNode.node;

                if (startNode.nodeType === 3) { startNode = startNode.parentNode; }
                if (endNode.nodeType === 3) { endNode = endNode.parentNode; }

                startIsComplexField = DOM.isInsideComplexFieldRange(startNode);
                endIsComplexField = DOM.isInsideComplexFieldRange(endNode);

                // no complex field involved in selection
                if (!startIsComplexField && !endIsComplexField) { return null; }

                // at least one of the two nodes is inside a complex field
                if (startIsComplexField && !endIsComplexField) {

                    // start of selection needs to be shifted to the left
                    startId = DOM.getComplexFieldMemberId(startNode);
                    startRangeMarker = rangeMarker.getStartMarker(startId);

                } else if (endIsComplexField && !startIsComplexField) {

                    // end of selection needs to be shifted to the right
                    endId = DOM.getComplexFieldMemberId(endNode);
                    endRangeMarker = rangeMarker.getEndMarker(endId);

                } else if (startIsComplexField && endIsComplexField) {

                    startId = DOM.getComplexFieldMemberId(startNode);
                    endId = DOM.getComplexFieldMemberId(endNode);

                    // both nodes are inside the same complex field -> nothing to do
                    if (startId === endId && !handlePlaceHolderComplexField) { return null; }

                    // start and end node are inside different complex fields
                    // -> the selection needs to be expanded in both directions
                    startRangeMarker = rangeMarker.getStartMarker(startId);
                    endRangeMarker = rangeMarker.getEndMarker(endId);
                }
            }

            return { start: startRangeMarker, end: endRangeMarker };
        }

        /**
         * Check, if an existing selection range needs to be expanded, so that a complex field
         * is completely part of the selection range. So it is not possible to remove only a part
         * of a selected range.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object
         *
         * @param {Function} selectionHandler
         *  The private selection handler function, that allows modification of start position or
         *  end position.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.handlePlaceHolderComplexField=false]
         *      If set to true, it is not necessary, that the current selection
         *      has a range, that will be expanded. For complex fields of type
         *      'placeholder' also a cursor selection is automatically expanded.
         *      This value is used inside the function 'getRangeMarkersFromSelection'.
         */
        function checkRangeSelection(event, selectionHandler, options) {

            var // an object with the neighboring range marker nodes
                rangeMarkers = getRangeMarkersFromSelection(options);

            if (rangeMarkers !== null) {
                if (rangeMarkers.start) {
                    selectionHandler(Position.getOxoPosition(selection.getRootNode(), rangeMarkers.start), { start: true, selectionExternallyModified: true });
                }

                if (rangeMarkers.end) { // end position need to be increased by 1 for selection range
                    selectionHandler(Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), rangeMarkers.end)), { start: false, selectionExternallyModified: true });
                }
            }
        }

        /**
         * Handler for highlighting of complex fields, after selection has changed.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} [options]
         *  Passed selection options.
         *  @param {Boolean} [options.simpleTextSelection=false]
         */
        function checkHighlighting(event, options) {
            var isSimpleTextSelection = Utils.getBooleanOption(options, 'simpleTextSelection', false);
            var domPos = null;
            var node = null;
            var atEndOfSpan = null;
            var relevantEvents = ['mousedown', 'mouseup', 'touchstart', 'touchend'];
            var browserEvent = (options && options.event) ? options.event : null;
            var mouseTouchEvent = _.isObject(browserEvent) && _.contains(relevantEvents, browserEvent.type);
            var rightClick = _.isObject(browserEvent) && (browserEvent.type === 'mousedown') && (browserEvent.button === 2);
            var isFieldCollectionEmpty = complexField.isEmpty() && simpleField.isEmpty();
            var skipHighlighting = (isFieldCollectionEmpty || isSimpleTextSelection || selection.hasRange());

            function isBeforeCxFldStart(element) {
                return DOM.isRangeMarkerStartNode(element) && DOM.getRangeMarkerType(element) === 'field';
            }

            if (!self.isHighlightState() && skipHighlighting) {
                return;
            } else {
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }

                    if ((DOM.isComplexFieldNode(node) || DOM.isInsideComplexFieldRange(node))) {
                        self.createEnterHighlight(node, { mouseTouchEvent: mouseTouchEvent, rightClick: rightClick });
                    } else if (atEndOfSpan && isBeforeCxFldStart(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling.nextSibling, { mouseTouchEvent: mouseTouchEvent, rightClick: rightClick }); // spec. handling for cursor positioned before cx field
                    } else if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true, mouseTouchEvent: mouseTouchEvent, rightClick: rightClick });
                    } else if (self.isHighlightState()) {
                        self.removeEnterHighlight();
                    }
                }
            }
        }

        /**
         * Callback function for event listener after initial page breaks
         * rendering is finished and after document is loaded.
         *
         * Stops operation distribution to server, and updates possible date fields,
         * and convert and update special page fields in headers.
         */
        function updateFieldsCallback() {
            app.stopOperationDistribution(function () {
                updateDateTimeAndSpecialFields();
                model.getUndoManager().clearUndoActions();
            });
            if (!simpleField.isEmpty()) {
                model.getPageLayout().updatePageSimpleFields();
            }
        }

        /**
         * Create current date(time) value string with passed format type.
         *
         * @param {String} formatCode
         * @return {String}
         */
        function getDateTimeRepresentation(formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        }

        /**
         * Callback function for triggering event fielddatepopup:change, when value of datepicker of input field is changed.
         * @param {Event} event
         * @param {Object} options
         *   @param {String} options.value
         *       Changed value for the field.
         *   @param {String} options.format
         *       Format for the field.
         *   @param {String} options.standard
         *       Odf needs this standard date for fixed fields
         */
        function updateDateFromPopup(event, options) {

            var node,
                domPos,
                atEndOfSpan,
                fieldValue = (options && options.value) || null,
                fieldFormat = (options && options.format) || null,
                standardizedDate = (options && options.standard) || null;

            if (!fieldValue) {
                if (!fieldFormat) { return; } // no data to update, exit
                if (fieldFormat === 'default') { fieldFormat = LocaleData.SHORT_DATE; } // #54713
                fieldValue = getDateTimeRepresentation(fieldFormat);
            }

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                simpleField.updateDateFromPopupValue(node, fieldValue, standardizedDate);

                // remove highlight from deleted node
                currentlyHighlightedNode = null;
                self.setCursorHighlightState(false);

                // add highlight to newly created node
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }
                    if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true });
                    }
                }
            } else if (currentlyHighlightedId) { // complex field
                node = self.getComplexField(currentlyHighlightedId);
                complexField.updateDateFromPopupValue(node, fieldValue);
            }
        }

        /**
         * Sets state of date field to be updated automatically or to be fixed value.
         * It is callback from field popup checkbox toggle.
         *
         * @param {Event} event
         *  jQuery Event that is triggered
         * @param {Boolean} state
         *  State of the field, fixed or updated automatically
         */
        function setDateFieldToAutoUpdate(event, state) {
            var node,
                pos,
                generator = model.createOperationGenerator(),
                operation = {},
                // target for operation - if exists, it's for ex. header or footer
                target = model.getActiveTarget(),
                isComplex = true,
                representation,
                attrs = {},
                type,
                instruction;

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                isComplex = false;
                representation = $(node).text();
            } else if (currentlyHighlightedId) { // complex field
                node = self.getComplexField(currentlyHighlightedId);
            }
            if (!node) {
                Utils.warn('fieldmanager.setDateFieldToAutoUpdate(): node not fetched!');
                return;
            }
            pos = Position.getOxoPosition(selection.getRootNode(), node);

            if (pos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    if (isComplex) {
                        operation = { start: pos, attrs: { character: { autoDateField: state } } };
                        model.extendPropertiesWithTarget(operation, target);
                        generator.generateOperation(Operations.COMPLEXFIELD_UPDATE, operation);
                        model.applyOperations(generator.getOperations());

                        instruction = DOM.getComplexFieldInstruction(node);
                    } else {
                        if (app.isODF()) {
                            type = 'date';
                            attrs = { character: { field: { 'text:fixed': JSON.stringify(!state), dateFormat: DOM.getFieldDateTimeFormat(node) } } };
                        } else {
                            type = DOM.getFieldInstruction(node);
                            attrs = { character: { autoDateField: JSON.stringify(state) } };
                        }
                        operation = { start: pos, type: type, representation: representation, attrs: attrs };
                        model.extendPropertiesWithTarget(operation, target);
                        generator.generateOperation(Operations.FIELD_UPDATE, operation);
                        model.applyOperations(generator.getOperations());

                        instruction = DOM.getFieldInstruction(node);
                    }
                    // on activating auto update, refresh the date
                    if (state) { self.updateByInstruction(node, instruction, { complex: isComplex }); }
                });
            } else {
                Utils.warn('fieldmanager.setDateFieldToAutoUpdate(): invalid position for field node!');
                return;
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Updating field with given instruction.
         * Depending of passed instruction, different type of field is updated.
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} instruction
         *  Complex field instruction, containing type,
         *  formating and optionally style, separated by \ .
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complex=false]
         *      If field is simple or complex type.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved immediately in case of simple field,
         *  or when the promise from complex field update is resolved.
         */
        this.updateByInstruction = function (node, instruction, options) {
            var isComplex = Utils.getBooleanOption(options, 'complex', false);
            var promise = $.when();

            if (instruction) {
                if (isComplex) {
                    promise = complexField.updateByInstruction(node, instruction); // updateToc returns promise
                } else {
                    simpleField.updateByInstruction(node, instruction);
                }
            } else {
                Utils.warn('FieldManager.updateByInstruction(): missing instruction!');
            }
            return promise;
        };

        /**
         * Set highlighting state of complex field during cursor traversal.
         *
         * @param {Boolean} state
         *  If cursor is inside complex field or not.
         */
        this.setCursorHighlightState = function (state) {
            isCursorHighlighted = state === true;
        };

        /**
         * If complex field highlight state is active, or not.
         *
         * @returns {Boolean}
         *  Whether the complex field highlight state is active, or not.
         */
        this.isHighlightState = function () {
            return isCursorHighlighted;
        };

        /**
         * Provides a jQuerified range start marker node.
         *
         * @returns {jQuery|null}
         *  The start marker with the specified id, or null, if no such start marker exists.
         */
        this.getComplexField = function (id) {
            return complexField.getComplexField(id);
        };

        /**
         * Getter for target container id of marginal complex field node.
         *
         * @param{String} id
         * Id of the field in collection model.
         *
         * @returns {jQuery|null}
         *  Target container id of marginal complex field node.
         */
        this.getMarginalComplexFieldsTarget = function (id) {
            return complexField.getMarginalComplexFieldsTarget(id);
        };

        /**
         * Check if there are complex fields in document.
         *
         * @returns {Boolean}
         */
        this.isComplexFieldEmpty = function () {
            return complexField.isEmpty();
        };

        /**
         * Gets simple field from collection by passed id.
         *
         * @param {String} id
         *  Id of queried field.
         *
         * @returns {jQuery|null}
         *  The simple field with the specified id, or null, if no such start marker exists.
         *
         */
        this.getSimpleField = function (id) {
            return simpleField.getSimpleField(id);
        };

        /**
         * Check if there are simple fields in document.
         *
         * @returns {Boolean}
         */
        this.isSimpleFieldEmpty = function () {
            return simpleField.isEmpty();
        };

        /**
         * Check if there are any type of fields in document.
         *
         * @returns {Boolean}
         */
        this.fieldsAreInDocument = function () {
            return !simpleField.isEmpty() || !complexField.isEmpty();
        };

        /**
         * Check, whether the specified target string is a valid target for a complex field.
         *
         * @param {String} target
         *  The target string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a complex field node.
         */
        this.isComplexFieldId = function (id) {
            return self.getComplexField(id) !== null;
        };

        /**
         * Warning: This accessor is used only for unit testing! Please use DOM's public method!
         *
         * @param {Node|jQuery|Null} node
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing or null, returns false.
         *
         * @param {Boolean} isODF
         *  If document format is odf, or ooxml.
         *
         * @returns {Boolean}
         *  If the field has fixed date property or not
         */
        this.isFixedSimpleField = function (field, isODF) {
            return DOM.isFixedSimpleField(field, isODF);
        };

        /**
         * When pasting content, that contains complex fields, it is necessary, that the
         * complex fields get a new unique id.
         *
         * @param {Object[]} operations
         *  The collection with all paste operations.
         */
        this.handlePasteOperationTarget = function (operations) {
            complexField.handlePasteOperationTarget(operations);
        };

        /**
         * Callback function for mouseenter event trigger on complex field.
         *
         * @param {Object} event
         *  jQuery object event triggered on hover in.
         */
        this.createHoverHighlight = function (event) {
            var id = DOM.getComplexFieldMemberId(event.currentTarget); // id of the hovered field
            var node = self.getComplexField(id);
            var instruction = DOM.getComplexFieldInstruction(node);
            var isTOCfield = instruction && (/TOC/i).test(instruction);
            var tooltipText = gt('Table of Contents. Right click...');
            var isRotatedShape = (node && node.length) ? node.parentsUntil('.page', '.rotated-drawing').length > 0 : false;

            if (!isRotatedShape && !isTOCfield && currentlyHighlightedId !== id) { // no highlighting of toc fields, #50504
                rangeMarker.highLightRange(id, 'cx-field-highlight');
            }
            // set tooltip to table of contents field element, if hyperlink switch is set in instruction
            if (isTOCfield && (/\\h/i).test(instruction)) {
                Forms.setToolTip(event.currentTarget, tooltipText);
            }

            // fallback with jquery/css highlighting
            //model.getCurrentRootNode().find('.complex' + id).addClass('cx-field-highlight');
        };

        /**
         * Callback function for mouseleave event trigger on complex field.
         *
         * @param {Object} event
         *  jQuery object event triggered on hover out.
         */
        this.removeHoverHighlight = function (event) {
            var // id of the hovered field
                id = DOM.getComplexFieldMemberId(event.currentTarget);

            if (currentlyHighlightedId !== id) {
                rangeMarker.removeHighLightRange(id);
            }

            // fallback with jquery/css highlighting
            //model.getCurrentRootNode().find('.complex' + id).removeClass('cx-field-highlight');
        };

        /**
         * Creates highlight range on enter cursor in complex field,
         * with keyboard key pressed or mouse click.
         *
         * @param {jQuery} field
         *  Field to be highlighted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.simpleField=false]
         *      If it's simple field or not.
         *  @param {Boolean} [options.mouseTouchEvent=false]
         *      If event is triggered by mouse or touch event.
         *  @param {Boolean} [options.rightClick=false]
         *      If its right click or not.
         */
        this.createEnterHighlight = function (field, options) {
            var id = DOM.getComplexFieldMemberId(field); // id of the highlighted field
            var isSimpleField = Utils.getBooleanOption(options, 'simpleField', false);
            var isMouseTouchEvent = Utils.getBooleanOption(options, 'mouseTouchEvent', false);
            var isRightClick = Utils.getBooleanOption(options, 'rightClick', false);
            var fieldinstruction = null;
            var node = null;
            var complexFieldNode = null;
            var fieldFormats, isDateTimeField, isSupportedFormat, isNotChangeTracked;
            var localOptions = {};

            if (isSimpleField) {
                if (currentlyHighlightedId || currentlyHighlightedNode) {
                    self.removeEnterHighlight();
                }
                currentlyHighlightedNode = $(field);
                currentlyHighlightedNode.addClass('sf-highlight');

                localOptions.autoDate = !DOM.isFixedSimpleField(field, app.isODF());
                fieldinstruction = { type: DOM.getFieldInstruction(field) };

                if (fieldinstruction.type && app.isODF() && (simpleField.isCurrentDate(fieldinstruction.type) || simpleField.isCurrentTime(fieldinstruction.type))) {
                    fieldinstruction.instruction = DOM.getFieldDateTimeFormat(field);
                } else if (fieldinstruction.type && app.isODF() && simpleField.isPageNumber(fieldinstruction.type)) {
                    fieldinstruction.instruction = DOM.getFieldPageFormat(field);
                } else {
                    fieldinstruction = simpleField.cleanUpAndExtractType(fieldinstruction.type);
                }
                // fallback to default
                fieldinstruction.instruction = fieldinstruction.instruction || 'default';
            } else {
                complexFieldNode = self.getComplexField(id);
                if (currentlyHighlightedId !== id) {
                    if (currentlyHighlightedId || currentlyHighlightedNode) {
                        self.removeEnterHighlight();
                    }
                    currentlyHighlightedId = id;

                    // make no highlighting inside rotated shapes and rotated group shapes
                    if (!$(complexFieldNode).parentsUntil('.page', '.rotated-drawing').length) {
                        rangeMarker.highLightRange(id, 'cx-field-highlight');
                    }
                }
                if (!complexFieldNode) {
                    Utils.warn('fieldmanager.createEnterHighlight(): failed to fetch complex field node!');
                    self.removeEnterHighlight();
                    return;
                }
                localOptions.autoDate = DOM.isAutomaticDateField(complexFieldNode);
                fieldinstruction = DOM.getComplexFieldInstruction(complexFieldNode);
                fieldinstruction = complexField.cleanUpAndExtractType(fieldinstruction);
            }

            if (fieldinstruction && fieldinstruction.type) {
                fieldinstruction.formats = localFormatList[fieldinstruction.type.toLowerCase()];
            }
            highlightedFieldType = fieldinstruction.type || null;
            highlightedFieldInstruction = fieldinstruction.instruction || null;
            fieldFormats = fieldinstruction.formats || null;

            self.setCursorHighlightState(true);
            node = complexFieldNode || field;

            self.destroyFieldPopup();
            isDateTimeField = (highlightedFieldType && (/^DATE|^TIME/i).test(highlightedFieldType)) || false;
            isSupportedFormat = (fieldFormats && fieldFormats.length) || isDateTimeField;
            isNotChangeTracked = !DOM.isChangeTrackNode(field) && !DOM.isChangeTrackNode($(field).parent());

            if (highlightedFieldType && isSupportedFormat && isMouseTouchEvent && !isRightClick && isNotChangeTracked) {
                fieldFormatPopupTimeout = self.executeDelayed(function () {
                    app.getView().showFieldFormatPopup(fieldinstruction, node, localOptions);
                }, 'FieldManager.createEnterHighlight', 500);
            }

            // for debugging ->
            // if (!highlightedFieldType) { Utils.error('no highlightedFieldType'); }
        };

        /**
         * Removes highlight range on enter cursor in complex field, or in simple field,
         * with keyboard key pressed or mouse click.
         */
        this.removeEnterHighlight = function () {
            if (currentlyHighlightedNode) {
                // simple field
                currentlyHighlightedNode.removeClass('sf-highlight');
                currentlyHighlightedNode = null;
            } else {
                rangeMarker.removeHighLightRange(currentlyHighlightedId);
                currentlyHighlightedId = null;
            }
            self.setCursorHighlightState(false);
            highlightedFieldType = null;
            highlightedFieldInstruction = null;

            self.destroyFieldPopup();
        };

        /**
         * Forces update of highlighted simple/complex field content.
         */
        this.updateHighlightedField = function () {
            var node,
                instruction,
                atEndOfSpan,
                domPos,
                promise = null;

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                instruction = DOM.getFieldInstruction(node);
                self.updateByInstruction(node, instruction);

                // remove highlight from deleted node
                currentlyHighlightedNode = null;
                self.setCursorHighlightState(false);

                // add highlight to newly created node
                domPos = Position.getDOMPosition(selection.getRootNode(), selection.getStartPosition());
                if (domPos && domPos.node) {
                    node = domPos.node;
                    if (node.nodeType === 3) {
                        atEndOfSpan = domPos.offset === node.textContent.length;
                        node = node.parentNode;
                    }
                    if (atEndOfSpan && DOM.isFieldNode(node.nextSibling)) {
                        self.createEnterHighlight(node.nextSibling, { simpleField: true });
                    }
                }
            } else if (currentlyHighlightedId) { // complex field
                node = self.getComplexField(currentlyHighlightedId);
                instruction = DOM.getComplexFieldInstruction(node);
                promise = self.updateByInstruction(node, instruction, { complex: true });
            }

            return promise ? promise : $.when();
        };

        /**
         * Forces update of complex fields inside selection range.
         */
        this.updateHighlightedFieldSelection = function () {
            var instruction = null;
            if (simpleField.isEmpty() && complexField.isEmpty()) { return; }

            selection.iterateNodes(function (node) {
                if (DOM.isComplexFieldNode(node)) {
                    instruction = DOM.getComplexFieldInstruction(node);
                    self.updateByInstruction(node, instruction, { complex: true });
                } else if (DOM.isFieldNode(node)) {
                    instruction = DOM.getFieldInstruction(node);
                    self.updateByInstruction(node, instruction);
                }
            });
        };

        /**
         * Update all fields in document.
         *
         */
        this.updateAllFields = function () {
            var // the generate operations deferred
                generateDef = null,
                mergedFields;

            // helper iterator function for handling field update
            function handleFieldUpdate(entry, fieldId) {
                var field,
                    instruction,
                    options,
                    isSimpleField = fieldId && fieldId.indexOf('s') > -1;

                field = isSimpleField ? self.getSimpleField(fieldId) : self.getComplexField(fieldId);
                if (!field) {
                    // this usually happens with nested fields, that are beeing deleted durring update of base level field,
                    // where update was triggered from this method. Iterator still has references to this dynamically deleted fields,
                    // but they are removed from document and model.
                    // Not expected to break anything, therefore log level. For debugging, change to Utils.error
                    Utils.log('fieldmanager.updateAllFields(): field with id: ' + fieldId + ' is not in the document anymore ');
                    return;
                }
                if (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field)) {
                    instruction = isSimpleField ? DOM.getFieldInstruction(field) : DOM.getComplexFieldInstruction(field);
                    if (!DOM.isSpecialField(field)) {
                        // #42671 and #45752
                        selection.swichToValidRootNode(field);
                        options = isSimpleField ? null : { complex: true };
                        return self.updateByInstruction(field, instruction, options);
                    }
                }
                return $.when();
            }

            if (self.fieldsAreInDocument()) {
                mergedFields = _.extend({}, simpleField.getAllFields(), complexField.getAllFields());

                // show a message with cancel button
                app.getView().enterBusy({
                    cancelHandler: function () {
                        if (generateDef && generateDef.abort) {
                            generateDef.abort();
                        }
                    },
                    warningLabel: gt('Updating all fields in document will take some time.')
                });

                return model.getUndoManager().enterUndoGroup(function () {
                    // iterate objects
                    generateDef = self.iterateObjectSliced(mergedFields, handleFieldUpdate, 'FieldManager.updateAllFields')
                        .progress(function (partialProgress) {
                            // add progress handling
                            var progress = 0.2 + (partialProgress * 0.8);
                            app.getView().updateBusyProgress(progress);
                        }).always(function () {
                            // leave busy state
                            app.getView().leaveBusy().grabFocus();
                        });

                    return generateDef;
                });
            }
        };

        /**
         * This method dispatch parent node to simple and complex field, for removing it.
         *
         * @param {jQuery|Node} parentNode
         *  Element that's being searched for simple and complex fields.
         */
        this.removeAllFieldsInNode = function (parentNode) {
            // checking, if the paragraph contains additional complex field nodes, that need to be removed, too.
            complexField.removeAllInsertedComplexFields(parentNode);
            // checking, if the paragraph contains additional simple field nodes, that need to be removed, too.
            simpleField.removeSimpleFieldsInNode(parentNode);
        };

        /**
         * Dispatch function to complexField class to check if search node in given start and end range
         * contains special fields that needs to be restored, like before delete operation.
         *
         * @param {Node} searchNode
         *  Node which we search for special fields.
         *
         * @param {Number} startOffset
         *  Start offset of given node.
         *
         * @param {Number} endOffset
         *  End offset of given node.
         */
        this.checkRestoringSpecialFields = function (searchNode, startOffset, endOffset) {
            if (_.isNumber(startOffset) && _.isNumber(endOffset)) {
                complexField.checkRestoringSpecialFields(searchNode, startOffset, endOffset);
            } else {
                complexField.checkRestoringSpecialFieldsInContainer(searchNode); // if startOffset or endOffset are undefined, we search whole node. See #42235
            }
        };

        /**
         * Dispatch function to complexField class to restore special field to normal before delete operation.
         *
         * @param {Node} node
         *  Node which we convert.
         */
        this.restoreSpecialField = function (node) {
            complexField.restoreSpecialField(node);
        };

        /**
         * Dispatch call to complex field to handle restoring of special fields in container node.
         *
         * @param {jQuery|Node} containerNode
         *  Node which we search for special fields.
         */
        this.checkRestoringSpecialFieldsInContainer = function (containerNode) {
            complexField.checkRestoringSpecialFieldsInContainer(containerNode);
        };

        /**
         * Dispatch function to simpleField class's method removeSimpleFieldFromCollection.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         */
        this.removeSimpleFieldFromCollection = function (node) {
            simpleField.removeSimpleFieldFromCollection(node);
        };

        /**
         * Dispatch function to complexField class's method removeFromComplexFieldCollection.
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         */
        this.removeFromComplexFieldCollection = function (node) {
            // if there is open field popup, close it and its promise before removing field, #42090
            self.destroyFieldPopup();
            complexField.removeFromComplexFieldCollection(node);
        };

        /**
         * Dispatch to simpleField for adding node to collection.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} [target]
         *  The target, where the simple field is located.
         */
        this.addSimpleFieldToCollection = function (fieldNode, target) {
            simpleField.addSimpleFieldToCollection(fieldNode, target);
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         * This method dispatches to complex field class.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateComplexFieldCollector = function (paragraph) {
            complexField.updateComplexFieldCollector(paragraph);
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         * This method dispatches to complex field class.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateSimpleFieldCollector = function (paragraph) {
            simpleField.updateSimpleFieldCollector(paragraph);
        };

        /**
         * Dispatch to complex field's method updateCxPageFieldsInMarginals.
         */
        this.updatePageFieldsInMarginals = function () {
            complexField.updateCxPageFieldsInMarginals();
        };

        /**
         * Inserts a simple text field component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text field.
         *
         * @param {String} type
         *  A property describing the field type.
         *
         * @param {String} representation
         *  A fallback value, if the placeholder cannot be substituted with a
         *  reasonable value.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text field, as map of attribute
         *  maps (name/value pairs), keyed by attribute family.
         *
         * @param {Boolean} [external]
         *  Will be set to true, if the invocation of this method originates
         *  from an external operation.
         *
         * @returns {Boolean}
         *  Whether the text field has been inserted successfully.
         */
        this.implInsertField = function (start, type, representation, attrs, target) {
            var // text span that will precede the field
                span = model.prepareTextSpanForInsertion(start, {}, target),
                // new text span for the field node
                fieldSpan = null,
                // the field node
                fieldNode = null,
                // format of the field
                format;

            if (!span) { return false; }

            // expanding attrs to disable change tracking, if not explicitely set
            attrs = model.checkChangesMode(attrs);

            // split the text span again to get initial character formatting
            // for the field, and insert the field representation text

            // Fix for 29265: Removing empty text node in span with '.contents().remove().end()'.
            // Otherwise there are two text nodes in span after '.text(representation)' in IE.
            fieldSpan = DOM.splitTextSpan(span, 0).contents().remove().end().text(representation);
            if (!representation) { DOM.ensureExistingTextNode(fieldSpan); }

            // insert a new text field before the addressed text node, move
            // the field span element into the field node
            fieldNode = DOM.createFieldNode();
            fieldNode.append(fieldSpan).insertAfter(span);
            fieldNode.data('type', type);

            // odf - field types needed to be updated in frontend
            if (type === 'page-number' || type === 'page-count') {
                fieldNode.addClass('field-' + type);
                if (attrs && attrs.character && attrs.character.field && attrs.character.field.pageNumFormat) {
                    format = attrs.character.field.pageNumFormat;
                }
                if (type === 'page-count') {
                    model.getPageLayout().updatePageCountField(fieldNode, format);
                } else {
                    model.getPageLayout().updatePageNumberField(fieldNode, format);
                }
            }

            // microsoft - field types needed to be updated in frontend
            if ((/NUMPAGES/i).test(type)) {
                fieldNode.addClass('field-NUMPAGES');
                model.getPageLayout().updatePageCountField(fieldNode);
            } else if ((/PAGENUM/i).test(type)) {
                fieldNode.addClass('field-page-number');
                model.getPageLayout().updatePageNumberField(fieldNode);
            }

            if (!representation) { fieldNode.addClass('empty-field'); }

            // apply the passed field attributes
            if (_.isObject(attrs)) {
                if (_.isObject(attrs.character) && attrs.character.field) {
                    _.each(attrs.character.field, function (element, name) {
                        fieldNode.data(name, element);
                        if (name === 'name') {
                            fieldNode.addClass('user-field');
                        }
                    });
                }
                model.getCharacterStyles().setElementAttributes(fieldSpan, attrs);
            }

            self.addSimpleFieldToCollection(fieldNode, target);
            if (model.getPageLayout().isIdOfMarginalNode(target)) {
                fieldNode.addClass(DOM.MARGINAL_NODE_CLASSNAME); // #54118
            }

            // validate paragraph, store new cursor position
            model.implParagraphChanged(span.parentNode);
            model.setLastOperationEnd(Position.increaseLastIndex(start));
            return true;
        };

        /**
         * Dispatches to handler for insertComplexField operations.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        this.insertComplexFieldHandler = function (start, instruction, attrs, target) {
            return complexField.insertComplexFieldHandler(start, instruction, attrs, target);
        };

        /**
         * Dispatches to handler for updateComplexField operations.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.

         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        this.updateComplexFieldHandler = function (start, instruction, attrs, target) {
            return complexField.updateComplexFieldHandler(start, instruction, attrs, target);
        };

        /**
         * Dispatches to handler for updateField operations.
         *
         * @param {Number[]} start
         *  The logical start position for the new simple field.
         *
         * @param {String} type
         *  Type of the simple field.
         *
         * @param {String} representation
         *  Content of the field that's updated.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the simple field has been inserted successfully.
         */
        this.updateSimpleFieldHandler = function (start, type, representation, attrs, target) {
            return simpleField.updateSimpleFieldHandler(start, type, representation, attrs, target);
        };

        /**
         * Dispatch to class complexField, to convert field to special field.
         *
         * @param {String} fieldId
         *  Id of the field
         *
         * @param {String} marginalTarget
         *  Id of the marginal target node
         *
         * @param {String} type
         *  Type of field: num pages or page number
         */
        this.convertToSpecialField = function (fieldId, marginalTarget, type) {
            complexField.convertToSpecialField(fieldId, marginalTarget, type);
        };

        /**
         * Dispatch update of date time fields on download to field classes.
         */
        this.prepareDateTimeFieldsDownload = function () {
            var mergedFields = null;

            // helper iterator function for handling field update
            function handleFieldUpdate(entry, fieldId) {
                var field,
                    instruction;

                self.destroyFieldPopup();

                if (fieldId && fieldId.indexOf('s') > -1) { // simple field
                    field = self.getSimpleField(fieldId);
                    if (field && !DOM.isFixedSimpleField(field, app.isODF())) {
                        instruction = DOM.getFieldInstruction(field);
                        selection.swichToValidRootNode(field);
                        simpleField.updateDateTimeField(field, instruction);
                    }
                } else { // complex field
                    field = self.getComplexField(fieldId);
                    if (field && DOM.isAutomaticDateField(field)) {
                        instruction = DOM.getComplexFieldInstruction(field);
                        selection.swichToValidRootNode(field);
                        complexField.updateDateTimeFieldOnDownload(field, instruction);
                    }
                }
            }
            // do nothing in read-only mode
            if (!app.isEditable()) { return; }

            mergedFields = _.extend({}, simpleField.getAllFields(), complexField.getAllFields());
            _.each(mergedFields, handleFieldUpdate);

        };

        /**
         * Calls method in simpleField class, for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNode = function ($node, pageCount) {
            simpleField.updatePageCountInCurrentNode($node, pageCount);
        };

        /**
         * Calls method in simpleField class, for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNode = function ($node, isHeader, pageCount) {
            simpleField.updatePageNumInCurrentNode($node, isHeader, pageCount);
        };

        /**
         * Calls method in complexField class, for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNodeCx = function ($node, pageCount) {
            complexField.updatePageCountInCurrentNodeCx($node, pageCount);
        };

        /**
         * Calls method in complexField class, for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNodeCx = function ($node, isHeader, pageCount) {
            complexField.updatePageNumInCurrentNodeCx($node, isHeader, pageCount);
        };

        this.updatePageNumInCurrentMarginalCx = function ($node, isHeader, pageNum) {
            complexField.updatePageNumInCurrentMarginalCx($node, isHeader, pageNum);
        };

        /**
         * Checks if selection contains special complex fields of type page inside header/footer.
         *
         * @returns {Boolean}
         */
        this.checkIfSpecialFieldsSelected = function () {
            var flag = false;
            if (model.isHeaderFooterEditState() && !self.isComplexFieldEmpty()) {
                selection.iterateNodes(function (node) {
                    if (DOM.isSpecialField(node)) {
                        flag = true;
                        return Utils.BREAK;
                    }
                });
            }
            return flag;
        };

        /**
         * If document is odf format, insert simple fields, otherwise complex fields.
         *
         * @param {String} fieldType
         *
         * @param {String} fieldFormat
         */
        this.dispatchInsertField = function (fieldType, fieldFormat) {
            if (app.isODF()) {
                simpleField.insertField(fieldType, fieldFormat);
            } else {
                complexField.insertComplexField(fieldType, fieldFormat);
            }
        };

        /**
         * Triggered by user from dropdown, inserts table of contents as complex field.
         *
         * @param {Number} value
         *  Value of the type of toc picked by user from dropdown menu.
         * @param {Object} [options]
         *  @param {Boolean} [options.testMode=false]
         *      When set to true, pagination is ignored.
         *      Warning: used for speeding up unit tests only. Do not use this flag for other purposes!
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of table of contents
         *  ended. It is rejected, if the dialog has been canceled.
         */
        this.dispatchInsertToc = function (value, options) {
            var tabStyle = value === 2 ? 'none' : 'dot';
            var fieldFormat = value === 3 ? 'TOC \\o "1-3" \\n \\h \\u' : 'TOC \\o "1-3" \\h \\u';
            var testMode = Utils.getBooleanOption(options, 'testMode', false);

            return complexField.insertTOCField(fieldFormat, { tabStyle: tabStyle, testMode: testMode });
        };

        /**
         * Public accessor method to dispatch update of passed complex field node and instruction.
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String|Array} format
         *  Format of the field with switches. Can be one string, or array of strings.
         *
         * @param {Object} options
         *  @param {String} options.tabStyle
         *      Style of the tabstops in table of contents
         *  @param {Boolean} [options.testMode=false]
         *      When set to true, pagination is ignored.
         *      Warning: used for speeding up unit tests only. Do not use this flag for other purposes!
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the insertion of table of contents
         *  ended. It is rejected, if the dialog has been canceled.
         */
        this.dispatchUpdateTOC = function (node, instruction, options) {
            return complexField.updateTOCField(node, instruction, options);
        };

        /**
         * Clicking on remove button in popup, removes highlighted simple or complex field.
         */
        this.removeField = function () {
            var
                node,
                rootNode = selection.getRootNode(),
                startPos,
                endPos; // used for complex field

            // hide popup
            app.getView().hideFieldFormatPopup();
            if (fieldFormatPopupTimeout) {
                fieldFormatPopupTimeout.abort();
            }

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
                startPos = Position.getOxoPosition(rootNode, node);

                if (startPos) {
                    return model.getUndoManager().enterUndoGroup(function () {
                        model.deleteRange(startPos);
                    });
                }

            } else if (currentlyHighlightedId) { // complex field
                startPos = Position.getOxoPosition(rootNode, rangeMarker.getStartMarker(currentlyHighlightedId));
                endPos = Position.getOxoPosition(rootNode, rangeMarker.getEndMarker(currentlyHighlightedId));

                if (startPos && endPos) {
                    return model.getUndoManager().enterUndoGroup(function () {
                        selection.setTextSelection(startPos, endPos);

                        if (model.isHeaderFooterEditState()) {
                            node = complexField.getComplexField(currentlyHighlightedId);
                            if (DOM.isSpecialField(node)) {
                                complexField.restoreSpecialField(node);
                                selection.setTextSelection(startPos, endPos);
                            }
                        }

                        model.deleteSelected();
                    });
                }
            }
        };

        /**
         * Clicking on field format from popup will update field with new formatting.
         *
         * @param {String} format
         */
        this.updateFieldFormatting = function (format) {
            var
                node,
                rootNode = selection.getRootNode(),
                startPos,
                endPos, // used for complex field
                fieldNodePos,
                fieldType = highlightedFieldType, // store values before closing popup and deleting field
                highlightedNode = currentlyHighlightedNode,
                highlightedId = currentlyHighlightedId,
                isSpecialField = false,
                isFixedDate,
                complexFieldNode,
                rangeEndMarkerNode;

            // hide popup
            self.destroyFieldPopup();

            if (!fieldType) {
                Utils.warn('fieldmanager.updateFieldFormatting(): missing field type!');
                return;
            }

            if (highlightedNode) { // simple field
                node = highlightedNode;
                startPos = Position.getOxoPosition(rootNode, node);
                isFixedDate = DOM.isFixedSimpleField(node, app.isODF());

                if (startPos) {
                    return model.getUndoManager().enterUndoGroup(function () {
                        model.deleteRange(startPos);
                        simpleField.insertField(fieldType, format, { isFixed: isFixedDate });
                    });
                }

            } else if (highlightedId) { // complex field
                complexFieldNode = complexField.getComplexField(highlightedId);
                rangeEndMarkerNode = rangeMarker.getEndMarker(highlightedId);
                if (!complexFieldNode) {
                    Utils.error('fieldManager.updateFieldFormatting(): missing field node for the id: ', highlightedId);
                    return;
                }
                if (!rangeEndMarkerNode) {
                    Utils.error('fieldManager.updateFieldFormatting(): missing range end node for field id: ', highlightedId);
                    return;
                }
                fieldNodePos = Position.getOxoPosition(rootNode, complexFieldNode);
                startPos = Position.increaseLastIndex(fieldNodePos);
                endPos = Position.getOxoPosition(rootNode, rangeEndMarkerNode);
                node = complexField.getComplexField(highlightedId);
                if (node.length && DOM.isSpecialField(node)) {
                    isSpecialField = true;
                }

                if (startPos && endPos) {
                    return model.getUndoManager().enterUndoGroup(function () {
                        selection.setTextSelection(startPos, endPos);

                        if (model.isHeaderFooterEditState()) {
                            node = complexField.getComplexField(highlightedId);
                            if (isSpecialField) {
                                complexField.restoreSpecialField(node);
                                selection.setTextSelection(startPos, endPos);
                            }
                        }

                        if (model.getChangeTrack().isActiveChangeTracking()) {
                            complexField.updateComplexFieldFormat(fieldType, format, fieldNodePos, highlightedId);
                            if (isSpecialField) {
                                complexField.convertToSpecialField(highlightedId, selection.getRootNodeTarget(), fieldType);
                            }
                            return $.when();
                        } else {
                            return model.deleteSelected().done(function () {
                                complexField.updateComplexFieldFormat(fieldType, format, fieldNodePos, highlightedId);
                                if (isSpecialField) {
                                    complexField.convertToSpecialField(highlightedId, selection.getRootNodeTarget(), fieldType);
                                }
                            });
                        }
                    });
                }
            }
        };

        /**
         * Returns highlighted field type if is explicitly setr.
         *
         * @returns {String}
         *  Highlighted field type.
         */
        this.getSelectedFieldType = function () {
            return highlightedFieldType || '';
        };

        /**
         * Returns highlighted field format, or default one.
         *
         * @returns {String}
         *  Highlighted field format.
         */
        this.getSelectedFieldFormat = function () {
            return highlightedFieldInstruction || 'default';
        };

        /**
         * Gets the anchor (start) node of the given field.
         *
         * @returns {jQuery | null}
         *  The anchor node of the given field, otherwise null if its not found.
         */
        this.getSelectedFieldNode = function () {
            var node = null;

            if (currentlyHighlightedNode) { // simple field
                node = currentlyHighlightedNode;
            } else if (currentlyHighlightedId) { // complex field
                node = complexField.getComplexField(currentlyHighlightedId);
            }
            return node;
        };

        /**
         * Method to open field format popup from context menu.
         *
         */
        this.editFieldFromContextMenu = function () {
            var node = self.getSelectedFieldNode();

            this.createEnterHighlight(node, { mouseTouchEvent: true, simpleField: !DOM.isComplexFieldNode(node) });
        };

        /**
         * During delete of complex field, take care
         * that following span (or other) elements don't keep the coresponding class name.
         *
         * @param {Node|jQuery} node
         *  Node that is beeing deleted, and from which we fetch field id.
         */
        this.cleanUpClassWhenDeleteCx = function (node) {
            var cxFieldContentSelector = 'complex' + DOM.getComplexFieldId(node);

            selection.getRootNode().find('.' + cxFieldContentSelector).removeClass(cxFieldContentSelector + ' ' + DOM.COMPLEXFIELDMEMBERNODE_CLASS).removeData('fieldId');
        };

        /**
         * On loosing edit right, close possible open field popup, and abort promise.
         *
         */
        this.destroyFieldPopup = function () {
            app.getView().hideFieldFormatPopup();
            if (fieldFormatPopupTimeout) {
                fieldFormatPopupTimeout.abort();
            }
        };

        /**
         * Add special field that is changetracked to temporary collection.
         * @param {jQuery} trackingNode
         */
        this.addToTempSpecCollection = function (trackingNode) {
            tempSpecFieldCollection.push(trackingNode);
        };

        /**
         * Return array of collected special field nodes.
         */
        this.getSpecFieldsFromTempCollection = function () {
            return tempSpecFieldCollection;
        };

        /**
         * Empty temporary collection after change track resolve is finished.
         */
        this.emptyTempSpecCollection = function () {
            tempSpecFieldCollection = [];
        };

        /**
         * Converts number for page and numPages fields into proper format code.
         *
         * @param {Number} number
         *  Number to be inserted into field.
         *
         * @param {String} format
         *  Format of the number inserted into field.
         *
         * @param {Boolean} [numPages]
         *  NumPages fields have some formats different from PageNum.
         *
         * @returns {String}
         *  Formatted number.
         */
        this.formatFieldNumber = function (pageCount, format, isNumPages) {
            return simpleField.formatPageFieldInstruction(pageCount, format, isNumPages);
        };

        /**
         * Public function to mark complex field for highlighting, after inserting into dom,
         * especially after inserting range end marker.
         *
         * @param {String} id - Id of the field to be highlighted.
         * @param {Node|jQuery} [fieldNode] - optional parameter for field node, if we already have it
         */
        this.markFieldForHighlighting = function (id, fieldNode) {
            var node = fieldNode || complexField.getComplexField(id);

            complexField.markComplexFieldForHighlighting(node, id);
        };

        /**
         * Warning: Public accessor used only for unit tests! Not meant to be called directly.
         *
         */
        this.setDateFieldToAutoUpdate = function (event, state) {
            setDateFieldToAutoUpdate(event, state);
        };

        /**
         * Public method to refresh collection of simple fields.
         */
        this.refreshSimpleFieldsCollection = function () {
            simpleField.refreshSimpleFields();
        };

        /**
         * Checks if highlighted field is supported type by OX Text.
         *
         * @returns {Boolean}
         */
        this.isSupportedFieldType = function () {
            var supportedTypes = [/NUMPAGES/i, /PAGE/i, /^DATE/i, /^TIME/i, /TOC/, /FILE-?NAME/i, /^AUTHOR$/i, /^author-name$/, /^creator$/];
            var result = false;

            if (highlightedFieldType && !(/PAGEREF/i).test(highlightedFieldType)) {
                _.each(supportedTypes, function (type) {
                    if (type.test(highlightedFieldType)) {
                        result = true;
                    }
                });
            }

            return result;
        };

        /**
         * Checks if highlighted field is Table of contents (TOC) field type.
         *
         * @returns {Boolean}
         */
        this.isTableOfContentsHighlighted = function () {
            return highlightedFieldType && (/TOC/i).test(highlightedFieldType);
        };

        /**
         * Check if currently highlighted field doesn't support format change.
         * Usually ODF autor and document name fields don't support this feature.
         * Info: when we extend list of supported fields, this regex needs to be expanded!
         *
         * @return {Boolean}
         */
        this.supportsFieldFormatting = function () {
            var unsupportedRegex = /(^author-name$)|(^creator$)|(^file-name$)|(^TOC$)/;

            return highlightedFieldType && !unsupportedRegex.test(highlightedFieldType);
        };

        /**
         * Used to stack locally id for field during creation of complex field and preceding range start and following range end.
         *
         * @param {Boolean} rangeStart
         *  If is requested from insert range start.
         * @param {Boolean} rangeEnd
         *  If is requested from insert range end.
         *
         * @returns {String} result
         *  Id of the field from the stack.
         */
        this.getComplexFieldIdFromStack = function (rangeStart, rangeEnd) {
            var result = '';
            var localId;

            if (rangeStart) {
                localId = complexField.getNextComplexFieldID();
                complexFieldStackId.push(localId);
                result = localId;
            } else {
                if (rangeEnd) {
                    result = complexFieldStackId.pop();
                } else {
                    result = _.last(complexFieldStackId);
                }
            }

            return result;
        };

        /**
         * Convinience method to get number of all fields present in the document.
         * If document is ODF, it will get count of simple fields,
         * otherwise count of complex fields from coresponding models.
         *
         * @return {Number}
         */
        this.getCountOfAllFields = function () {
            return simpleField.getAllFieldsCount() + complexField.getAllFieldsCount();
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {

            var deferred1 = self.createDeferred('FieldManager.onInit#1');
            var deferred2 = self.createDeferred('FieldManager.onInit#2');

            model = app.getModel();
            selection = model.getSelection();
            rangeMarker = model.getRangeMarker();
            numberFormatter = model.getNumberFormatter();

            self.listenTo(app, 'docs:editmode:leave', function () {
                self.destroyFieldPopup();
            });

            self.waitForImportSuccess(function () {
                complexField.refreshComplexFields(model.isLocalStorageImport(), model.isFastLoadImport());
                simpleField.refreshSimpleFields();
                // refreshing also the range marker model, if it was not done before
                rangeMarker.refreshModel({ onlyOnce: true });
                // resolve import success for update fields callback
                deferred2.resolve();

                // category codes loaded from local resource
                categoryCodesDate = numberFormatter.getCategoryCodes('date');
                categoryCodesTime = numberFormatter.getCategoryCodes('time');
                categoryCodesDateTime = numberFormatter.getCategoryCodes('datetime');
                // fill in date codes
                if (categoryCodesDate.length) {
                    _.each(categoryCodesDate, function (formatCode) {
                        localFormatList.date.push({ option: formatCode.value, value: getDateTimeRepresentation(formatCode.value) });
                    });
                    if (categoryCodesDateTime.length) {
                        _.each(categoryCodesDateTime, function (formatCode) {
                            localFormatList.date.push({ option: formatCode.value, value: getDateTimeRepresentation(formatCode.value) });
                        });
                    }
                } else {
                    localFormatList.date.push({ option: LocaleData.SHORT_DATE, value: getDateTimeRepresentation(LocaleData.SHORT_DATE) },
                        { option: LocaleData.LONG_DATE, value: getDateTimeRepresentation(LocaleData.LONG_DATE) });
                }
                // append two most common time formats in date menu
                localFormatList.date.push({ option: LocaleData.SHORT_TIME, value: getDateTimeRepresentation(LocaleData.SHORT_TIME) },
                    { option: LocaleData.LONG_TIME, value: getDateTimeRepresentation(LocaleData.LONG_TIME) });

                // fill in time codes
                if (categoryCodesTime.length) {
                    _.each(categoryCodesTime, function (formatCode) {
                        localFormatList.time.push({ option: formatCode.value, value: getDateTimeRepresentation(formatCode.value) });
                    });
                } else {
                    localFormatList.time.push({ option: LocaleData.SHORT_TIME, value: getDateTimeRepresentation(LocaleData.SHORT_TIME) },
                        { option: LocaleData.LONG_TIME, value: getDateTimeRepresentation(LocaleData.LONG_TIME) });
                }
            });

            self.waitForImportFailure(function () {
                // reject promise on import failed, so that update fields doesn't hang
                deferred1.reject();
                deferred2.reject();
            });

            self.listenTo(app, 'docs:beforequit', function () {
                deferred1.reject();
                deferred2.reject();
            });

            // mark all complex fields contents after undo/redo
            self.listenTo(model.getUndoManager(), 'undo:after redo:after', function () {
                complexField.markAllFieldsAfterUndo();
            });

            model.one('pageBreak:after initialPageBreaks:done', function () { return deferred1.resolve(); });
            // only when import is finished and page breaks are rendered call update fields
            $.when(deferred1, deferred2).then(updateFieldsCallback);

            selection.on('position:calculated', checkRangeSelection);
            selection.on('update', checkHighlighting);

            model.on('document:reloaded', function () {
                complexField.refreshComplexFields(true, false);
                simpleField.refreshSimpleFields();
            });

            app.getView().on('fielddatepopup:change', updateDateFromPopup);
            app.getView().on('fielddatepopup:autoupdate', setDateFieldToAutoUpdate);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            simpleField.destroy();
            complexField.destroy();
            model = selection = rangeMarker = numberFormatter = null;
            currentlyHighlightedNode = authorFileNameArray = pageNumberArray = localFormatList = null;
            categoryCodesDate = categoryCodesTime = categoryCodesDateTime = tempSpecFieldCollection = complexFieldStackId = null;
        });

    } // class FieldManager

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: FieldManager });
});
