/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/renderframemanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/spreadsheet/model/drawing/framenodemanager'
], function (Utils, Forms, DrawingFrame, DOMUtils, FrameNodeManager) {

    'use strict';

    // class RenderFrameManager ===============================================

    /**
     * Extends the class FrameNodeManager by functionality needed to actually
     * render a specific drawing layer (either regular drawing objects, or cell
     * comments).
     *
     * @constructor
     *
     * @extends FrameNodeManager
     */
    var RenderFrameManager = FrameNodeManager.extend({ constructor: function (docView, layerNode) {

        // the spreadsheet model
        var docModel = docView.getDocModel();

        // the page node serving as root node for the text framework
        var pageRootNode = DOMUtils.createPageNode().appendTo(layerNode)[0];

        // the content node of the page node (child of page node)
        var pageContentNode = DOMUtils.getPageContentNode(pageRootNode)[0];

        // the container node for all top-level drawing frames
        var frameRootNode = null;

        // base constructor ---------------------------------------------------

        FrameNodeManager.call(this, docModel);

        // public methods -----------------------------------------------------

        /**
         * Initializes the page node containing the drawing frames. Prepares
         * the leading empty container nodes for the preceding sheets (needed
         * by the text framework to be able to generate correct text positions
         * in document operations).
         *
         * @returns {RenderFrameManager}
         *  A reference to this instance.
         */
        this.initializePageNode = function () {

            pageContentNode.removeChild(frameRootNode);

            var activeSheet = docView.getActiveSheet();
            var childNodes = pageContentNode.childNodes;

            for (var missingNodes = activeSheet - childNodes.length; missingNodes > 0; missingNodes -= 1) {
                pageContentNode.appendChild(FrameNodeManager.createRootNode());
            }

            pageContentNode.insertBefore(frameRootNode, childNodes.item(activeSheet));

            return this;
        };

        /**
         * Returns all selected DOM drawing frame nodes.
         *
         * @returns {jQuery}
         *  The selected drawing frames, as jQuery object.
         */
        this.getSelectedDrawingFrames = function () {
            return $(frameRootNode).find(DrawingFrame.NODE_SELECTOR + Forms.SELECTED_SELECTOR);
        };

        // initialization -----------------------------------------------------

        // get the root node of the drawing frames
        frameRootNode = this.getRootNode();

        // insert the frame root node into the page
        pageContentNode.appendChild(frameRootNode);

        // bug 31479: suppress double-clicks for drawings
        layerNode.on('dblclick', false);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = docView = layerNode = null;
            pageRootNode = pageContentNode = frameRootNode = null;
        });

    } }); // class RenderFrameManager

    // exports ================================================================

    return RenderFrameManager;

});
