/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/definednamedialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialog/inputdialog',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, InputDialog, SheetUtils, gt) {

    'use strict';

    // class DefinedNameDialog ================================================

    /**
     * Shows a dialog that allows to create or modify a defined name, and will
     * invoke an arbitrary callback function for the defined name. The dialog
     * will be kept open until the settings for the defined name are valid.
     *
     * @constructor
     *
     * @extends InputDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view instance that has created this dialog.
     *
     * @param {Function} actionHandler
     *  The callback function invoked when the OK button of the dialog has
     *  been clicked. Receives the current label and formula entered in the
     *  dialog, and MUST return a boolean value or a promise that states
     *  whether the operation using that label and formula expression has been
     *  finished successfully. In case of an error, the dialog will be kept
     *  open with its current settings.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  - {String} [initOptions.label='']
     *      The initial label for the defined name to be shown when the dialog
     *      is opened.
     *  - {String} [initOptions.formula='']
     *      The initial formula expression to be shown when the dialog is
     *      opened.
     *  - {Number|Null} [initOptions.sheet=null]
     *      The initial value for the sheet selector (null for globally defined
     *      names, or the index of a sheet for sheet-local names) to be shown
     *      when the dialog is opened.
     *  - {Boolean} [initOptions.change=false]
     *      False for a dialog that is used to create a new defined name, or
     *      true for a dialog that is used to change an existing defined name.
     */
    var DefinedNameDialog = InputDialog.extend(function (docView, actionHandler, initOptions) {

        // self reference
        var self = this;

        // whether to change an existing defined name
        var changeName = Utils.getBooleanOption(initOptions, 'change', false);

        // text input field for the formula expression
        var formulaField = null;

        // base constructor ---------------------------------------------------

        InputDialog.call(this, docView, {
            title: changeName ? gt('Change name') : gt('Insert name'),
            okLabel: changeName ? gt('Change') : gt('Insert'),
            inputLabel: gt('Name'),
            value: Utils.getStringOption(initOptions, 'label', ''),
            maxLength: SheetUtils.MAX_NAME_LENGTH,
            placeholder: gt('Enter name')
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the current label for the defined name shown in the dialog.
         *
         * @returns {String}
         *  The current label for the defined name shown in the dialog.
         */
        function getNameLabel() {
            return self.getText();
        }

        /**
         * Returns the current formula expression for the defined name shown in
         * the dialog.
         *
         * @returns {String}
         *  The current formula expression for the defined name shown in the
         *  dialog.
         */
        function getNameFormula() {
            return formulaField.val();
        }

        /**
         * Returns the sheet index currently selected in the dialog.
         *
         * @returns {Number|Null}
         *  The sheet index currently selected in the dialog, or null for names
         *  to be inserted into the document.
         */
        function getNameSheet() {
            return Utils.getIntegerOption(initOptions, 'sheet', null); // TODO: from drop-down control
        }

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // create and insert the formula text field into the dialog body
        formulaField = $('<input tabindex="0" type="text" class="form-control">', { placeholder: gt('Enter formula') });
        formulaField.val(Utils.getStringOption(initOptions, 'formula', ''));
        this.getContentNode().append($('<label>').text(gt('Formula')), formulaField);

        // register the action handler for the OK button
        this.setOkHandler(function () {
            return actionHandler.call(this, getNameLabel(), getNameFormula(), getNameSheet());
        }, { keepOpen: 'fail' });

    }); // class DefinedNameDialog

    // exports ================================================================

    return DefinedNameDialog;

});
