/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/modelattributesmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/textframework/format/characterstyles',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/pagestylecollection',
    'io.ox/office/spreadsheet/model/cellstylecollection',
    'io.ox/office/spreadsheet/model/tablestylecollection',
    'io.ox/office/spreadsheet/model/paragraphstylecollection',
    'io.ox/office/spreadsheet/model/cellautostylecollection',
    'io.ox/office/spreadsheet/model/drawing/text/fieldmanager'
], function (Utils, ValueMap, Color, Border, AttributeUtils, DrawingStyleCollection, CharacterStyleCollection, SheetUtils, PageStyleCollection, CellStyleCollection, TableStyleCollection, ParagraphStyleCollection, CellAutoStyleCollection, FieldManager) {

    'use strict';

    // definitions for global document attributes -----------------------------

    var DOCUMENT_ATTRIBUTE_DEFINITIONS = {

        /**
         * Number of columns available in each sheet of the document.
         */
        cols: { def: 1024 },

        /**
         * Number of rows available in each sheet of the document.
         */
        rows: { def: 1048576 },

        /**
         * Specifies whether all formulas in the document shall be relaculated
         * once after import.
         */
        calcOnLoad: { def: false },

        /**
         * The null date (the date represented by the serial value 0) of the
         * workbook. Must be a date in ISO format YYYY-MM-DD.
         */
        nullDate: { def: '1899-12-30' },

        // view attributes

        /**
         * Index of the active (displayed) sheet.
         */
        activeSheet: { def: 0 }
    };

    // definitions for character attributes in cells --------------------------

    var CHARACTER_ATTRIBUTE_DEFINITIONS = {

        /**
         * Name of the font family used to render the cell contents. Bound to
         * the apply flag 'font'.
         */
        fontName: { def: AttributeUtils.MINOR_FONT_KEY, apply: 'font' },

        /**
         * Size of the font used to render the cell contents, in points. Bound
         * to the apply flag 'font'.
         */
        fontSize: { def: 11, apply: 'font' },

        /**
         * Specifies whether to render the cell contents with bold characters.
         * Bound to the apply flag 'font'.
         */
        bold: { def: false, apply: 'font' },

        /**
         * Specifies whether to render the cell contents with italic
         * characters. Bound to the apply flag 'font'.
         */
        italic: { def: false, apply: 'font' },

        /**
         * Specifies whether to underline the cell contents. Bound to the apply
         * flag 'font'.
         */
        underline: { def: false, apply: 'font' },

        /**
         * Specifies whether to strike the cell contents. Supported values are
         * 'none', 'single', and 'double'. Bound to the apply flag 'font'.
         */
        strike: { def: 'none', apply: 'font' },

        /**
         * Text color used to render the cell contents, underlines, and strike
         * lines. Bound to the apply flag 'font'.
         */
        color: { def: Color.AUTO, apply: 'font' }
    };

    // definitions for cell attributes ----------------------------------------

    var CELL_ATTRIBUTE_DEFINITIONS = {

        /**
         * Horizontal alignment of the text contents in the cell. Bound to the
         * apply flag 'align'.
         */
        alignHor: { def: 'auto', apply: 'align' },

        /**
         * Vertical alignment of the text contents in the cell. Bound to the
         * apply flag 'align'.
         */
        alignVert: { def: 'bottom', apply: 'align' },

        /**
         * Specifies whether to break long text automatically into multiple
         * text lines. Has no effect on non-text cell results. Bound to the
         * apply flag 'align'.
         */
        wrapText: { def: false, apply: 'align' },

        /**
         * The identifier of a number format code (used in OOXML together with
         * the operation 'insertNumberFormat'). Bound to the apply flag
         * 'number'.
         */
        formatId: { def: 0, apply: 'number' },

        /**
         * An explicit number format code, used in ODF, and in contexts where
         * the number format has not been created with the operation
         * 'insertNumberFormat' (e.g. in conditional formatting rules). Bound
         * to the apply flag 'number'.
         */
        formatCode: { def: 'General', apply: 'number' },

        /**
         * The fill type ('solid', 'pattern', 'gradient', or 'inherit'). Bound
         * to the apply flag 'fill'.
         */
        fillType: { def: 'solid', apply: 'fill', merge: function (type1, type2) {
            // fill type 'inherit' is used in conditional formatting to switch from 'gradient' to 'pattern', but inherit all other fill types
            return (type2 !== 'inherit') ? type2 : (type1 === 'gradient') ? 'pattern' : type1;
        } },

        /**
         * The fill color for the cell area, the background color of a pattern,
         * or the start color of a gradient. Bound to the apply flag 'fill'.
         */
        fillColor: { def: Color.AUTO, apply: 'fill' },

        /**
         * The foreground color of a pattern, or the end color of a gradient.
         * Bound to the apply flag 'fill'.
         */
        foreColor: { def: Color.AUTO, apply: 'fill' },

        /**
         * The identifier of a fill pattern. Bound to the apply flag 'fill'.
         */
        pattern: { def: 'solid', apply: 'fill' },

        /**
         * The style of the left border of the cell. Bound to the apply flag
         * 'border'.
         */
        borderLeft: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the right border of the cell. Bound to the apply flag
         * 'border'.
         */
        borderRight: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the top border of the cell. Bound to the apply flag
         * 'border'.
         */
        borderTop: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the bottom border of the cell. Bound to the apply flag
         * 'border'.
         */
        borderBottom: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the diagonal border of the cell from top-left corner to
         * bottom-right corner. Bound to the apply flag 'border'.
         */
        borderDown: { def: Border.NONE, apply: 'border' },

        /**
         * The style of the diagonal border of the cell from bottom-left corner
         * to top-right corner. Bound to the apply flag 'border'.
         */
        borderUp: { def: Border.NONE, apply: 'border' },

        /**
         * Specifies whether the cell is unlocked. Locked cells in locked
         * sheets cannot be modified. Has no effect in unlocked sheets. Bound
         * to the apply flag 'protect'.
         */
        unlocked: { def: false, apply: 'protect' },

        /**
         * Specifies whether the formula expression of a formula cell will be
         * hidden in the user interface of a locked sheet. Has no effect in
         * unlocked sheets. Bound to the apply flag 'protect'.
         */
        hidden: { def: false, apply: 'protect' }
    };

    // definitions for apply group flags --------------------------------------

    // used in OOXML only!
    var APPLY_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the font attributes of the style sheet will reset
         * explicit attributes of the target cell; or whether font attributes
         * of a cell have been changed manually.
         */
        font: { def: true },

        /**
         * Specifies whether the fill attributes of the style sheet will reset
         * explicit attributes of the target cell; or whether fill attributes
         * of a cell have been changed manually.
         */
        fill: { def: true },

        /**
         * Specifies whether the border attributes of the style sheet will
         * reset explicit attributes of the target cell; or whether border
         * attributes of a cell have been changed manually.
         */
        border: { def: true },

        /**
         * Specifies whether the alignment attributes of the style sheet will
         * reset explicit attributes of the target cell; or whether alignment
         * attributes of a cell have been changed manually.
         */
        align: { def: true },

        /**
         * Specifies whether the number format of the style sheet will reset
         * explicit attributes of the target cell; or whether the number format
         * of a cell has been changed manually.
         */
        number: { def: true },

        /**
         * Specifies whether the protection flags of the style sheet will reset
         * explicit attributes of the target cell; or whether protection flags
         * of a cell have been changed manually.
         */
        protect: { def: true }
    };

    // definitions for column attributes --------------------------------------

    var COLUMN_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the column is visible in the GUI.
         */
        visible: { def: true, resize: true },

        /**
         * The width of the column, in number of digits of default document
         * font (OOXML), or in 1/100 mm (ODF). The actual default column width
         * will be set dynamically per sheet.
         */
        width: { def: 0, resize: true },

        /**
         * Specifies whether the width of the column has been set or changed in
         * the GUI and is fixed.
         */
        customWidth: { def: false }
    };

    // definitions for row attributes -----------------------------------------

    var ROW_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the row is visible in the GUI.
         */
        visible: { def: true, resize: true },

        /**
         * The height of the row, in points (OOXML), or in 1/100 mm (ODF). The
         * actual default row height will be set dynamically per sheet.
         */
        height: { def: 0, resize: true },

        /**
         * Specifies whether the height of the row has been set or changed in
         * the GUI and is fixed. If set to false, the row height will be
         * adjusted automatically due to cell contents and font settings.
         */
        customHeight: { def: false },

        /**
         * Specifies whether the row contains default formatting attributes for
         * all undefined cells. If set to false, undefined cells will be
         * formatted according to the default column formatting.
         */
        customFormat: { def: false },

        /**
         * Specifies whether the row has been hidden implicitly by a filter
         * (auto-filter, or filtered table range). The row will be hidden
         * although the attribute 'visible' is true. Used by ODF only!
         */
        filtered: { def: false, resize: true }
    };

    // definitions for column/row grouping attributes -------------------------

    var GROUP_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies the grouping level of the column or row.
         */
        level: { def: 0 },

        /**
         * Specifies whether the affected column/row groups are collapsed. Will
         * be a boolean value in OOXML documents, but an integral bit set in
         * ODF documents!
         */
        collapse: { def: false }
    };

    // definitions for sheet attributes ---------------------------------------

    var SHEET_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies whether the sheet is visible in the GUI.
         */
        visible: { def: true },

        /**
         * Specifies whether the sheet is locked. Contents in locked sheets
         * cannot be changed (except unlocked cells).
         */
        locked: { def: false },

        /**
         * Base column width for undefined columns, in number of digits of the
         * default document font. Used by OOXML only.
         */
        baseColWidth: { def: 8 },

        /**
         * Whether the total columns of all column groups in the sheet are
         * located before the groups.
         */
        colTotalBefore: { def: false },

        /**
         * Whether the total rows of all row groups in the sheet are located
         * before the groups.
         */
        rowTotalBefore: { def: false },

        // view attributes

        /**
         * The current zoom factor. The value 1 represents the standard zoom of
         * 100%.
         */
        zoom: { def: 1 },

        /**
         * The selected cell ranges, as array of JSON cell range addresses.
         */
        selectedRanges: { def: [{ start: [0, 0], end: [0, 0] }] },

        /**
         * The array index of the active cell range in the 'selectedRanges'
         * attribute value (the index of the cell range containing the active
         * cell).
         */
        activeIndex: { def: 0 },

        /**
         * The address of the active cell in the selected cell ranges, as JSON
         * cell position (array with column and row index). Must be located
         * inside the active cell range (as specified by the attributes
         * 'selectedRanges' and 'activeIndex').
         */
        activeCell: { def: [0, 0] },

        /**
         * The positions of all selected drawing objects. Each element of this
         * array is a sheet-local drawing position (an array of integers
         * without leading sheet index).
         */
        selectedDrawings: { def: [] },

        /**
         * Whether the grid lines between the cells will be rendered.
         */
        showGrid: { def: true },

        /**
         * The color of the grid lines between the cells.
         */
        gridColor: { def: Color.AUTO },

        /**
         * The split mode used when split view is active. Must be one of the
         * following values:
         * - 'split': Dynamic split (grid panes are resizable).
         * - 'frozen': Frozen split (grid panes are not resizable).
         * - 'frozenSplit': Like 'frozen', but thawing will change split mode
         *      to 'split' instead of removing the split.
         */
        splitMode: { def: 'split' },

        /**
         * Width of the left grid panes, as 1/100 mm for dynamic split mode, or
         * number of columns in frozen split mode. The value zero hides the
         * right grid panes.
         */
        splitWidth: { def: 0 },

        /**
         * Height of the upper grid panes, as 1/100 mm for dynamic split mode,
         * or number of rows in frozen split mode. The value zero hides the
         * lower grid panes.
         */
        splitHeight: { def: 0 },

        /**
         * Identifier of the active (focused) grid pane. Must be one of
         * 'topLeft', 'topRight', 'bottomLeft', or 'bottomRight'. Must be a
         * visible grid pane, according to the current split settings.
         */
        activePane: { def: 'topLeft' },

        /**
         * Horizontal scroll position in the left grid panes, as zero-based
         * column index.
         */
        scrollLeft: { def: 0 },

        /**
         * Horizontal scroll position in the right grid panes, as zero-based
         * column index.
         */
        scrollRight: { def: 0 },

        /**
         * Vertical scroll position in the upper grid panes, as zero-based row
         * index.
         */
        scrollTop: { def: 0 },

        /**
         * Vertical scroll position in the lower grid panes, as zero-based row
         * index.
         */
        scrollBottom: { def: 0 }
    };

    // definitions for table attributes ---------------------------------------

    var TABLE_ATTRIBUTE_DEFINITIONS = {

        /**
         * Whether the drop-down buttons of all columns will be hidden.
         */
        hideButtons: { def: false },

        /**
         * Whether the first row of the table is a header row. Header rows will
         * not be filtered or sorted.
         */
        headerRow: { def: false, scope: 'element' },

        /**
         * Whether the last row of the table is a footer row. Footer rows will
         * not be filtered or sorted.
         */
        footerRow: { def: false, scope: 'element' },

        /**
         * Whether to show special formatting for the cells in the first table
         * column, according to the current table style sheet.
         */
        firstCol: { def: false, scope: 'element' },

        /**
         * Whether to show special formatting for the cells in the last table
         * column, according to the current table style sheet.
         */
        lastCol: { def: false, scope: 'element' },

        /**
         * Whether to show alternating row formatting for the data cells,
         * according to the current table style sheet.
         */
        bandsHor: { def: false, scope: 'element' },

        /**
         * Whether to show alternating column formatting for the data cells,
         * according to the current table style sheet.
         */
        bandsVert: { def: false, scope: 'element' },

        /**
         * Whether the string values in the table range will be sorted
         * case-sensitively.
         */
        caseSensitive: { def: false, scope: 'element' }
    };

    // definitions for filter attributes in table columns ---------------------

    var TABLE_FILTER_ATTRIBUTE_DEFINITIONS = {

        /**
         * The type of the filter rule in this column. The following values are
         * supported:
         * - 'none': No filter rule is active.
         * - 'discrete': Discrete filter. The attribute 'entries' contains
         *      all cell entries to be shown for the table column.
         *
         * The property 'undo' contains the names of all attributes that need
         * to be inserted into the undo operation that will be generated when
         * changing this attribute.
         */
        type: { def: 'none', undo: 'entries' },

        /**
         * All entries (cell display strings) that will be shown for this table
         * column. Only used, if the filter type is 'discrete'.
         */
        entries: { def: [] },

        /**
         * Whether the drop-down button of the filter column will be hidden.
         */
        hideButton: { def: false }
    };

    // definitions for sorting attributes in table columns --------------------

    var TABLE_SORT_ATTRIBUTE_DEFINITIONS = {

        /**
         * Specifies how to sort the values in the column.
         * - 'none': The column will not be sorted.
         * - 'value': The column will be sorted by the values in the cells.
         * - 'font': The column will be sorted by the text color of the cells.
         * - 'fill': The column will be sorted by the fill color of the cells.
         * - 'icon': The column will be sorted by icons in the cells.
         *
         * The property 'undo' contains the names of all attributes that need
         * to be inserted into the undo operation that will be generated when
         * changing this attribute.
         */
        type: { def: 'none', undo: '*' },

        /**
         * If set to true, the values in the column will be sorted in
         * descending order, and the selected cell color or font color will be
         * pushed to the bottom of the table.
         */
        descending: { def: false },

        /**
         * The internal index of the format structure containing the font color
         * or fill color. Only used for sort types 'font' or 'fill'.
         */
        dxf: { def: 0 },

        /**
         * The name of the icon set used for sorting the column. Only used for
         * sort type 'icon'.
         */
        is: { def: '' },

        /**
         * The identifier of the icon in the icon set as specified by the
         * attribute 'is'. Only used for sort type 'icon'.
         */
        id: { def: 0 }
    };

    // definitions for data validation attributes -----------------------------

    var VALIDATION_ATTRIBUTE_DEFINITIONS = {

        /**
         * The type of the data validation. Supports all types used in the
         * document operation 'insertValidation'.
         *
         * The property 'undo' contains the names of all attributes that need
         * to be inserted into the undo operation that will be generated when
         * changing this attribute.
         */
        type: { def: 'all', undo: 'compare value1 value2' },

        /**
         * The comparison operator for most validation types, used together
         * with the attributes 'value1' and 'value2'.
         */
        compare: { def: '' },

        /**
         * The first value used together with the comparison operator. Must be
         * a formula expression as string, without the leading equality sign.
         */
        value1: { def: '' },

        /**
         * The second value used together with the comparison operators
         * 'between' and 'notBetween'. Must be a formula expression as string,
         * without the leading equality sign.
         */
        value2: { def: '' },

        /**
         * Specifies whether to show an informational tooltip when a cell with
         * this data validation is selected.
         */
        showInfo: { def: true },

        /**
         * The headline for the informational tooltip.
         */
        infoTitle: { def: '' },

        /**
         * The message text for the informational tooltip.
         */
        infoText: { def: '' },

        /**
         * Specifies whether to show an alert box when an invalid value has
         * been entered into a cell with this data validation.
         */
        showError: { def: true },

        /**
         * The headline for the alert box.
         */
        errorTitle: { def: '' },

        /**
         * The message text for the alert box.
         */
        errorText: { def: '' },

        /**
         * The type of the alert box. Must be one of 'info', 'warning', or
         * 'error'.
         */
        errorType: { def: 'error' },

        /**
         * Specifies whether to show a drop-down button at the selected cell,
         * if the data validation type is 'list' or 'source'.
         */
        showDropDown: { def: true },

        /**
         * Specifies whether to allow blank cells regardless of the actual
         * validation type and conditions.
         */
        ignoreEmpty: { def: true }
    };

    // definitions for formatting rule attributes -----------------------------

    var FORMAT_RULE_ATTRIBUTE_DEFINITIONS = {

        /**
         * The type of the formatting rule. Supports all types used in the
         * document operation 'insertCondFormatRule'.
         *
         * The property 'undo' contains the names of all attributes that need
         * to be inserted into the undo operation that will be generated when
         * changing this attribute.
         */
        type: { def: 'formula', undo: '*' },

        /**
         * The first value, used by most conditional rules. Depending on the
         * rule type, must either be a formula expression as string (without
         * the leading equality sign), or a constant number (for 'average'
         * rules only).
         */
        value1: { def: '' },

        /**
         * The second value, used by interval rules of type 'between' and
         * 'notBetween' only. Must be a formula expression as string, without
         * the leading equality sign.
         */
        value2: { def: '' },

        /**
         * A data structure that describes the start point, the end point, and
         * an arbitrary number of additional points inbetween a color scale
         * rule.
         */
        colorScale: { def: [] },

        /**
         * A data structure that describes a data bar rule.
         */
        dataBar: { def: {} },

        /**
         * A data structure that describes an icon set rule.
         */
        iconSet: { def: {} },

        /**
         * The sheet-global priority of this rule among all formatting rules in
         * the sheet. The lower this value, the higher the priority. Rules with
         * equal priority will be applied in insertion order.
         */
        priority: { def: 0 },

        /**
         * Specifies whether to stop evaluating other formatting rules matching
         * a specific cell, if the conditions of this rule has matched.
         */
        stop: { def: false },

        /**
         * An incomplete formatting attribute set containing cell attributes,
         * character attributes, and the identifier of a style sheet. If the
         * conditions of the rule are matching, these attributes will be used
         * to render in the respective cell, above its own attributes.
         */
        attrs: { def: {} }
    };

    // additional definitions for drawing attributes --------------------------

    var DRAWING_ATTRIBUTE_DEFINITIONS = {

        /**
         * Type of the drawing anchor (specifies which anchor attributes have
         * to be used), and runtime behavior of the drawing while inserting,
         * removing, or changing columns and rows. Supported values are
         * 'twoCell', 'oneCell', 'absolute', 'twoCellAsOneCell', and
         * 'twoCellAsAbsolute'.
         */
        anchorType: { def: 'twoCell', scope: 'element' },

        /**
         * Zero-based index of the column containing the leading border of the
         * drawing object.
         */
        startCol: { def: 0, scope: 'element' },

        /**
         * The exact position of the leading border in the column, as ratio of
         * the column width (in the range [0,1]).
         */
        startColOffset: { def: 0, scope: 'element' },

        /**
         * Zero-based index of the row containing the top border of the drawing
         * object.
         */
        startRow: { def: 0, scope: 'element' },

        /**
         * The exact position of the top border in the row, as ratio of the row
         * height (in the range [0,1]).
         */
        startRowOffset: { def: 0, scope: 'element' },

        /**
         * Zero-based index of the column containing the trailing border of the
         * drawing object.
         */
        endCol: { def: 0, scope: 'element' },

        /**
         * The exact position of the trailing border in the column, as ratio of
         * the column width (in the range [0,1]).
         */
        endColOffset: { def: 0, scope: 'element' },

        /**
         * Zero-based index of the row containing the bottom border of the
         * drawing object.
         */
        endRow: { def: 0, scope: 'element' },

        /**
         * The exact position of the bottom border in the row, as ratio of the
         * row height (in the range [0,1]).
         */
        endRowOffset: { def: 0, scope: 'element' }
    };

    // precalculated standard row heights -------------------------------------

    // effective pixel row height for different fonts and font sizes used in OOXML
    var PIXEL_ROW_HEIGHT_MAP = _.mapObject({
        arial:             '701120131121221222031131131131131121222122121131131131121421222122121131131131211222122131121131131221232122131112131121221231121121222221122121131242121121141131121132121131131131121222121211133131131122121221221131131131122121221131131131322122121221131131131122121221221131131122122141221131131131122121221221131131122122121221',
        calibri:           '70112013113122122122212212212213113113113113113113122122122122212212213113113113113113113113122122122212212212213113113113113113113122122122122212212213113113113113113113113122122122212212212213113113113113113113122122122122212212213113113113113113113122122122122212212212213113113113113113113122122122122212212213113113111',
        cambria:           '70112012212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112212212122122113113113113112211',
        consolas:          '701120131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131131221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221221',
        'courier new':     '7011201312212212221221221311311311311312212212221221221311311311311311312212212221221311311311311311312212212221221221311311311311311312212221221221311311311311311312212212221221221311311311A113113122122213113113122122213113113122122213113113122122213113113122212213113113122212213113113122212213113113122212213113',
        'times new roman': '70112013113113113113112212212212212212212214122122122122122122122113113113113115113113113113113113113113113112210214212212212211122122122122122122122133113113113113113113113113112113113313110122113113113112212212122122115113113112212122122113113113112212214122113113113113112212122122113113133112212212122113113113112212212122'
    }, function (diffMap) {
        var fontSizes = [null];
        var fontSize = 0;
        _.each(diffMap, function (diff) {
            fontSize += parseInt(diff, 16);
            fontSizes.push(fontSize);
        });
        return fontSizes;
    });

    // class ModelAttributesMixin =============================================

    /**
     * A mix-in class for the document model class SpreadsheetModel providing
     * the style sheet containers for all attribute families used in a
     * spreadsheet document.
     *
     * @constructor
     */
    function ModelAttributesMixin() {

        // self reference
        var self = this;

        // special behavior for OOXML files
        var ooxml = this.getApp().isOOXML();

        // shortcut to the cell attribute pool
        var cellAttrPool = null;

        // the style sheet collection for cells, columns, and rows
        var cellStyles = null;

        // the auto-style collection for cells, columns, and rows
        var cellAutoStyles = null;

        // the style sheet collection for table ranges
        var tableStyles = null;

        // shortcut to the drawing attribute pool
        var drawingAttrPool = null;

        // the style sheet collection for drawing objects
        var drawingStyles = null;

        // cache for default digit width, mapped by different zoom factors
        var digitWidthCache = new ValueMap();

        // the conversion factor for column widths (1 operation unit to 1/100 mm, initialized lazily)
        var colWidthFactor = null;

        // the conversion factor for row heights (1 operation unit to 1/100 mm, initialized lazily)
        var rowHeightFactor = null;

        // the maximum column width in 1/100 mm (initialized lazily)
        var maxColWidthHmm = null;

        // the maximum wow height in 1/100 mm (initialized lazily)
        var maxRowHeightHmm = null;

        // private methods ----------------------------------------------------

        function getColWidthFactor() {
            return colWidthFactor || (colWidthFactor = ooxml ? (100 * Utils.convertLength(self.getDefaultDigitWidth(1), 'px', 'mm')) : 1);
        }

        function getRowHeightFactor() {
            return rowHeightFactor || (rowHeightFactor = ooxml ? (100 * Utils.convertLength(1, 'pt', 'mm')) : 1);
        }

        /**
         * Returns the precalculated row height for the passed font settings.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the row height, as used
         *  in document operations.
         *
         * @returns {Number|Null}
         *  The precalculated row height for the passed font settings, in
         *  pixels; or null, if no row height is available for the passed font
         *  settings.
         */
        function getPresetRowHeight(charAttributes) {
            var fontName = self.resolveFontName(charAttributes.fontName);
            var presetRowHeights = PIXEL_ROW_HEIGHT_MAP[fontName.toLowerCase()];
            if (!presetRowHeights) { return null; }
            var fontSize = Math.round(charAttributes.fontSize);
            return presetRowHeights[fontSize] || _.last(presetRowHeights);
        }

        /**
         * Calculates the row height in 1/100 of millimeters to be used for the
         * passed character attributes.
         */
        function calculateRowHeightHmm(charAttributes) {

            // the normal line height according to the font settings
            var lineHeight = self.getRenderFont(charAttributes).getNormalLineHeight();

            // use at least 125% of the font size, to compensate for browsers
            // reporting very small line heights (especially Chrome)
            // Bug 40718: increased 120% to 125% to cover the standard case 'Arial 11pt'.
            lineHeight = Math.max(lineHeight, Utils.convertLength(charAttributes.fontSize, 'pt', 'px') * 1.25);

            // enlarge the resulting 'normal' line height by another 10% to add top/bottom cell padding
            return Utils.convertLengthToHmm(Math.floor(lineHeight * 1.1), 'px');
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the special attribute pool for cell formatting attributes
         * used by the spreadsheet model. Used to be able to distinguish cell
         * character attributes from drawing character attributes.
         *
         * @returns {AttributePool}
         *  The attribute pool for cell formatting attributes used by the
         *  spreadsheet model.
         */
        this.getCellAttributePool = function () {
            return cellAttrPool;
        };

        /**
         * Returns the cell style collection of this document.
         *
         * @returns {CellStyleCollection}
         *  The cell style collection of this document.
         */
        this.getCellStyles = function () {
            return cellStyles;
        };

        /**
         * Returns the cell auto-style collection of this document.
         *
         * @returns {AutoStyleCollection}
         *  The cell auto-style collection of this document.
         */
        this.getCellAutoStyles = function () {
            return cellAutoStyles;
        };

        /**
         * Returns the style collection for table ranges of this document.
         *
         * @returns {TableStyleCollection}
         *  The style collection for table ranges of this document.
         */
        this.getTableStyles = function () {
            return tableStyles;
        };

        /**
         * Returns the attribute pool for drawing formatting attributes used by
         * the spreadsheet model. Used to be able to distinguish cell character
         * attributes from drawing character attributes.
         *
         * @returns {AttributePool}
         *  The attribute pool for drawing formatting attributes used by the
         *  spreadsheet model.
         */
        this.getDrawingAttributePool = function () {
            return drawingAttrPool;
        };

        /**
         * Returns the maximum width of the digits in the specified font,
         * according to the passed zoom factor.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes, as used in document operations.
         *
         * @param {Number} zoom
         *  The zoom factor to be applied to the font size.
         *
         * @returns {Number}
         *  The maximum width of the digits in the specified font, in pixels.
         */
        this.getDigitWidth = function (charAttributes, zoom) {
            return this.getRenderFont(charAttributes, null, zoom).getDigitWidths().maxWidth;
        };

        /**
         * Calculates the effective horizontal padding between cell grid lines
         * and the text contents of the cell in pixels to be used for the
         * passed character attributes and zoom factor.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the padding, as used in
         *  document operations.
         *
         * @param {Number} zoom
         *  The zoom factor used to scale the font size in the passed character
         *  attributes.
         *
         * @returns {Number}
         *  The effective horizontal text padding for the passed character
         *  attributes, in pixels.
         */
        this.getTextPadding = function (charAttributes, zoom) {
            return SheetUtils.getTextPadding(this.getDigitWidth(charAttributes, zoom));
        };

        /**
         * Returns the total size of all horizontal padding occupied in a cell
         * that cannot be used for the cell contents for the passed character
         * attributes and zoom factor. This value includes the text padding
         * (twice, for left and right border), and additional space needed for
         * the grid lines.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the padding, as used in
         *  document operations.
         *
         * @returns {Number}
         *  The total size of the horizontal cell content padding, in pixels.
         */
        this.getTotalCellPadding = function (charAttributes, zoom) {
            return SheetUtils.getTotalCellPadding(this.getDigitWidth(charAttributes, zoom));
        };

        /**
         * Returns the maximum width of the digits in the font of the current
         * default cell style, according to the passed zoom factor.
         *
         * @param {Number} zoom
         *  The zoom factor to be applied to the default font.
         *
         * @returns {Number}
         *  The maximum width of the digits in the font of the current default
         *  cell style, in pixels.
         */
        this.getDefaultDigitWidth = function (zoom) {
            return digitWidthCache.getOrCreate(Math.round(zoom * 100), function () {

                // attributes of the default cell style (NOT the default auto-style!)
                var charAttributes = cellStyles.getDefaultStyleAttributeSet().character;

                // the maximum pixel width of any of the digits of the default font
                return self.getDigitWidth(charAttributes, zoom);
            });
        };

        /**
         * Converts the passed column width from 1/100 of millimeters to
         * operation units according to the file format of the edited document.
         *
         * @param {Number} colWidthHmm
         *  A column width in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The column width in operation units.
         */
        this.convertColWidthToUnit = function (colWidthHmm) {
            var colWidth = colWidthHmm / getColWidthFactor();
            return ooxml ? (Math.floor(colWidth * 256) / 256) : colWidth;
        };

        /**
         * Converts the passed column width from operation units according to
         * the file format of the edited document to 1/100 of millimeters.
         *
         * @param {Number} colWidth
         *  A column width in operation units.
         *
         * @returns {Number}
         *  The column width in 1/100 of millimeters.
         */
        this.convertColWidthFromUnit = function (colWidth) {
            return Math.ceil(colWidth * getColWidthFactor());
        };

        /**
         * Converts the passed row height from 1/100 of millimeters to
         * operation units according to the file format of the edited document.
         *
         * @param {Number} rowHeightHmm
         *  A row height in 1/100 of millimeters.
         *
         * @returns {Number}
         *  The row height in operation units.
         */
        this.convertRowHeightToUnit = function (rowHeightHmm) {
            var rowHeight = rowHeightHmm / getRowHeightFactor();
            return ooxml ? Utils.round(rowHeight, 0.25) : rowHeight;
        };

        /**
         * Converts the passed row height from operation units according to the
         * file format of the edited document to 1/100 of millimeters.
         *
         * @param {Number} rowHeight
         *  A row height in operation units.
         *
         * @returns {Number}
         *  The row height in 1/100 of millimeters.
         */
        this.convertRowHeightFromUnit = function (rowHeight) {
            return Math.round(rowHeight * getRowHeightFactor());
        };

        /**
         * Returns the maximum width for columns, as supported by the file
         * format of the edited document.
         *
         * @returns {Number}
         *  The maximum width for columns.
         */
        this.getMaxColWidthHmm = function () {
            return maxColWidthHmm || (maxColWidthHmm = ooxml ? this.convertColWidthFromUnit(255) : 100000);
        };

        /**
         * Returns the maximum height for rows, as supported by the file format
         * of the edited document.
         *
         * @returns {Number}
         *  The maximum height for rows.
         */
        this.getMaxRowHeightHmm = function () {
            return maxRowHeightHmm || (maxRowHeightHmm = ooxml ? this.convertRowHeightFromUnit(409.5) : 100000);
        };

        /**
         * Returns the maximum width for columns or the maximum height for
         * rows, as supported by the file format of the edited document.
         *
         * @param {Boolean} columns
         *  Whether to the maximum column width (true), or the maximum row
         *  height (false).
         *
         * @returns {Number}
         *  The maximum width for columns, or the maximum height for rows.
         */
        this.getMaxColRowSizeHmm = function (columns) {
            return columns ? this.getMaxColWidthHmm() : this.getMaxRowHeightHmm();
        };

        /**
         * Calculates the row height in 1/100 of millimeters to be used for the
         * passed character attributes. The row height is about 10% larger than
         * the normal line height for the passed font settings, as returned
         * e.g. by the method Font.getNormalLineHeight().
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height, as
         *  used in document operations.
         *
         * @returns {Number}
         *  The row height for the passed character attributes, in 1/100 mm.
         */
        this.getRowHeightHmm = function (charAttributes) {

            // bug 50743: use precalculated row heights if available, to get rid of browser-dependent 'normal line height'
            var presetHeight = ooxml ? getPresetRowHeight(charAttributes) : null;
            if (presetHeight) { return Utils.convertLengthToHmm(presetHeight, 'px'); }

            // calculate row height in 1/100mm
            return calculateRowHeightHmm(charAttributes);
        };

        /**
         * Calculates the effective scaled row height in pixels to be used for
         * the passed character attributes and zoom factor.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height, as
         *  used in document operations.
         *
         * @param {Number} zoom
         *  The zoom factor used to scale the font size in the passed character
         *  attributes.
         *
         * @returns {Number}
         *  The row height for the passed character attributes, in pixels.
         */
        this.getRowHeight = function (charAttributes, zoom) {

            // bug 50743: use precalculated row heights if available, to get rid of browser-dependent 'normal line height'
            var presetHeight = ooxml ? getPresetRowHeight(charAttributes) : null;
            if (presetHeight) { return Math.round(presetHeight * zoom); }

            // calculate row height, and convert it from 1/100mm to pixels
            return Utils.convertHmmToLength(calculateRowHeightHmm(charAttributes) * zoom, 'px', 1);
        };

        /**
         * Extends the passed attribute set in-place with formatting attributes
         * resolved from a table style sheet. These attributes will be merged
         * "under" the explicit attributes.
         *
         * @param {Object} attributeSet
         *  The (incomplete) attribute set to be extended in-place with the
         *  passed table style attributes.
         *
         * @param {Object} mergedAttributeSet
         *  The (complete) merged attribute set used to decide whether the
         *  passed table attributes will become part of the extended attribute
         *  set. Only the attributes where this merged attribute set contains
         *  the same values as the default auto-style can be extended with the
         *  table style attributes.
         *
         * @param {Object} tableAttributeSet
         *  The attribute set resolved from a table style sheet to be merged
         *  into the passed (incomplete) attribute set.
         *
         * @returns {Object}
         *  The attribute set passed to this method, that has been extended
         *  in-place.
         */
        this.extendWithTableAttributeSet = function (attributeSet, mergedAttributeSet, tableAttributeSet) {
            var defaultAttrSet = cellAutoStyles.getDefaultAttributeSet();
            _.forEach(tableAttributeSet, function (tableAttrs, family) {
                if (family in mergedAttributeSet) {
                    var targetAttrs = attributeSet[family] || (attributeSet[family] = {});
                    var mergedAttrs = mergedAttributeSet[family];
                    var defaultAttrs = defaultAttrSet[family];
                    _.forEach(tableAttrs, function (value, name) {
                        if ((name in mergedAttrs) && _.isEqual(mergedAttrs[name], defaultAttrs[name])) {
                            targetAttrs[name] = value;
                        }
                    });
                }
            });
            return attributeSet;
        };

        // initialization -----------------------------------------------------

        // global document attributes registration
        this.registerDocumentAttributes(DOCUMENT_ATTRIBUTE_DEFINITIONS);

        // formatting attributes registration with a dedicated spreadsheet attribute pool
        this.registerAttributes('character', CHARACTER_ATTRIBUTE_DEFINITIONS, { poolId: SheetUtils.CELL_POOL_ID });
        this.registerAttributes('cell', CELL_ATTRIBUTE_DEFINITIONS, { poolId: SheetUtils.CELL_POOL_ID });
        this.registerAttributes('apply', APPLY_ATTRIBUTE_DEFINITIONS, { poolId: SheetUtils.CELL_POOL_ID });
        this.registerAttributes('table', TABLE_ATTRIBUTE_DEFINITIONS, { poolId: SheetUtils.CELL_POOL_ID });
        this.registerAttributes('filter', TABLE_FILTER_ATTRIBUTE_DEFINITIONS/*, { poolId: SheetUtils.CELL_POOL_ID }*/);
        this.registerAttributes('sort', TABLE_SORT_ATTRIBUTE_DEFINITIONS/*, { poolId: SheetUtils.CELL_POOL_ID }*/);

        // formatting attributes registration used by text framework (default attribute ppol)
        this.registerAttributes('column', COLUMN_ATTRIBUTE_DEFINITIONS);
        this.registerAttributes('row', ROW_ATTRIBUTE_DEFINITIONS);
        this.registerAttributes('group', GROUP_ATTRIBUTE_DEFINITIONS);
        this.registerAttributes('sheet', SHEET_ATTRIBUTE_DEFINITIONS);
        this.registerAttributes('validation', VALIDATION_ATTRIBUTE_DEFINITIONS);
        this.registerAttributes('rule', FORMAT_RULE_ATTRIBUTE_DEFINITIONS);
        this.registerAttributes('drawing', DRAWING_ATTRIBUTE_DEFINITIONS);

        // style sheet collections
        this.addStyleCollection(cellStyles = new CellStyleCollection(this));
        this.addStyleCollection(drawingStyles = new DrawingStyleCollection(this));
        this.addStyleCollection(new PageStyleCollection(this));
        this.addStyleCollection(tableStyles = new TableStyleCollection(this));

        // style collections needed for text in shapes
        this.addStyleCollection(new CharacterStyleCollection(this));
        this.addStyleCollection(new ParagraphStyleCollection(this));

        // auto-style collections
        this.addAutoStyleCollection(cellAutoStyles = new CellAutoStyleCollection(this));

        // shortcuts to attribute pools
        cellAttrPool = cellStyles.getAttributePool();
        drawingAttrPool = drawingStyles.getAttributePool();

        // further helper classes expected by the text framework used for text in shapes
        this.setFieldManager(new FieldManager(this));

        // setting the handler function for updating lists
        this.setUpdateLists(this.getUpdateListsHandler());

        // OOXML: listen to changes of the default style sheet, invalidate cached font measures for column width
        if (ooxml) {
            this.listenTo(cellStyles, 'triggered', function (event, type, styleId) {
                if (styleId === cellStyles.getDefaultStyleId()) {
                    digitWidthCache.clear();
                    colWidthFactor = maxColWidthHmm = null;
                }
            });
        }

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = cellAttrPool = cellStyles = cellAutoStyles = tableStyles = drawingStyles = drawingAttrPool = null;
        });

    } // class ModelAttributesMixin

    // exports ================================================================

    return ModelAttributesMixin;

});
