/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/matrix', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/model/formula/utils/scalar',
    'io.ox/office/spreadsheet/model/formula/utils/dimension'
], function (Iterator, ErrorCode, Address, Scalar, Dimension) {

    'use strict';

    // convenience shortcuts
    var NestedIterator = Iterator.NestedIterator;

    // class Matrix ===========================================================

    /**
     * Representation of a matrix literal: a two-dimensional array of constant
     * scalar values (numbers, strings, booleans, and error codes).
     *
     * @constructor
     *
     * @param {Array<Array<Any>>} array
     *  The two-dimensional array of scalars which will become the contents of
     *  this matrix instance. The array MUST contain at least one inner array.
     *  All inner arrays MUST contain at least one value, and all MUST contain
     *  the same number of values.
     */
    function Matrix(array) {

        this._array = array;

    } // class Matrix

    // constants --------------------------------------------------------------

    /**
     * Maximum number of elements supported in matrixes.
     *
     * @constant
     * @type Number
     */
    Matrix.MAX_SIZE = 10000;

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed matrix dimension is inside the supported
     * limits.
     *
     * @param {Dimension|Number} dim
     *  The dimension of a matrix, or the number of rows in a matrix.
     *
     * @param {Number} [cols]
     *  The number of columns in a matrix. MUST be specified, if the first
     *  parameter is a number (row count). Ignored, if the first parameter is
     *  an instance of Dimension.
     *
     * @returns {Boolean}
     *  Whether the passed matrix dimension is inside the limits supported by
     *  the formula interpreter.
     */
    Matrix.isValidDim = function (dim, cols) {
        var size = (typeof dim === 'number') ? (dim * cols) : dim.size();
        return (size > 0) && (size <= Matrix.MAX_SIZE);
    };

    /**
     * Creates a new matrix with the specified size, and fills it with the
     * passed constant values.
     *
     * @param {Number} rows
     *  The number of rows contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Number} cols
     *  The number of columns contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The value to be inserted into the matrix elements.
     *
     * @returns {Matrix}
     *  The new matrix instance.
     */
    Matrix.create = function (rows, cols, value) {
        return Matrix.generate(rows, cols, _.constant(value));
    };

    /**
     * Creates a new matrix with the specified size, and fills it with the
     * results of the specified generator callback function.
     *
     * @param {Number} rows
     *  The number of rows contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Number} cols
     *  The number of columns contained in the new matrix. Must be a positive
     *  integer.
     *
     * @param {Function} generator
     *  The generator callback function that will be invoked for every matrix
     *  element. Receives the following parameters:
     *  (1) {Number} row
     *      The zero-based row index of the matrix element.
     *  (2) {Number} col
     *      The zero-based column index of the matrix element.
     *  Note the parameter order (row index before column index), which is the
     *  usual order for mathematical matrixes, but differs from the parameter
     *  order of other methods taking column/row indexes of cell addresses
     *  (column index before row index). The return value of the callback
     *  function MUST be a scalar value used in spreadsheet formulas (numbers,
     *  strings, booleans, error codes, or null), and becomes the value of the
     *  respective matrix element.

     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {Matrix}
     *  The new matrix instance.
     */
    Matrix.generate = function (rows, cols, generator, context) {

        // create the two-dimensional array
        var array = _.times(rows, function (row) {
            return _.times(cols, function (col) {
                return generator.call(context, row, col);
            });
        });

        // create the matrix
        return new Matrix(array);
    };

    /**
     * Creates an identity matrix of the specified size.
     *
     * @param {Number} size
     *  The number of rows and columns of the identity matrix.
     *
     * @returns {Matrix}
     *  The new matrix instance.
     */
    Matrix.identity = function (size) {
        return Matrix.generate(size, size, function (row, col) {
            return (row === col) ? 1 : 0;
        });
    };

    /**
     * Creates a new matrix consisting of a single row, and fills it with the
     * passed scalar values.
     *
     * @param {Array<Any>} values
     *  The scalar values to be inserted into the row vector. The array MUST
     *  NOT be empty. For performance, the new matrix takes ownership of this
     *  array.
     *
     * @returns {Matrix}
     *  A new matrix, consisting of one row, containing the elements of the
     *  passed array.
     */
    Matrix.createRowVector = function (values) {
        return new Matrix([values]);
    };

    /**
     * Creates a new matrix consisting of a single column, and fills it with
     * the passed scalar values.
     *
     * @param {Array<Any>} values
     *  The scalar values to be inserted into the column vector.
     *
     * @returns {Matrix}
     *  A new matrix, consisting of one column, containing the elements of the
     *  passed array.
     */
    Matrix.createColVector = function (values) {
        return Matrix.generate(values.length, 1, function (row) { return values[row]; });
    };

    /**
     * Returns the address of the cell in the passed cell range located at the
     * specified relative row and column indexes.
     *
     * @param {Range} range
     *  The address of a cell range.
     *
     * @param {Number} row
     *  The relative row index of a matrix element. This value will be ignored,
     *  if the passed range consists of a single row (auto-repetition in matrix
     *  formulas).
     *
     * @param {Number} col
     *  The relative column index of a matrix element. This value will be
     *  ignored, if the passed range consists of a single row (auto-repetition
     *  in matrix formulas).
     *
     * @returns {Address}
     *  The address of the cell in the passed range specified by the passed row
     *  and column indexes.
     *
     * @throws {ErrorCode}
     *  The #N/A error code, if the passed row index or column index exceeds
     *  the size of the passed cell range address.
     */
    Matrix.getAddress = function (range, row, col) {
        if (range.singleCol()) { col = 0; }
        if (range.singleRow()) { row = 0; }
        if ((col >= range.cols()) || (row >= range.rows())) { throw ErrorCode.NA; }
        return new Address(range.start[0] + col, range.start[1] + row);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a deep clone of this matrix.
     *
     * @returns {Matrix}
     *  A deep clone of this matrix.
     */
    Matrix.prototype.clone = function () {
        return new Matrix(_.invoke(this._array, 'slice'));
    };

    /**
     * Returns the elements of this matrix as plain JavaScript array of arrays.
     *
     * @returns {Array<Array<Any>>}
     *  The elements of this matrix, as array of row vectors. These arrays MUST
     *  NOT be changed.
     */
    Matrix.prototype.values = function () {
        return this._array;
    };

    /**
     * Returns the number of rows contained in this matrix.
     *
     * @returns {Number}
     *  The number of rows contained in this matrix.
     */
    Matrix.prototype.rows = function () {
        return this._array.length;
    };

    /**
     * Returns the number of columns contained in this matrix.
     *
     * @returns {Number}
     *  The number of columns contained in this matrix.
     */
    Matrix.prototype.cols = function () {
        return this._array[0].length;
    };

    /**
     * Returns the dimension of this matrix, i.e. the number of its rows and
     * columns in a single object.
     *
     * @returns {Dimension}
     *  The number of rows and columns contained in this matrix.
     */
    Matrix.prototype.dim = function () {
        return new Dimension(this.rows(), this.cols());
    };

    /**
     * Returns the number of elements contained in this matrix.
     *
     * @returns {Number}
     *  The number of elements contained in this matrix.
     */
    Matrix.prototype.size = function () {
        return this.rows() * this.cols();
    };

    /**
     * Returns the value of the specified element in this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the element. If the matrix consists of a
     *  single row only, this index will be ignored, and the element from the
     *  existing row will be used instead (matrix auto-expansion as used in
     *  spreadsheet formulas).
     *
     * @param {Number} col
     *  The zero-based column index of the element. If the matrix consists of a
     *  single column only, this index will be ignored, and the element from
     *  the existing column will be used instead (matrix auto-expansion as used
     *  in spreadsheet formulas).
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The value of the matrix element; or the #N/A error code, if the passed
     *  indexes are located outside the matrix (except for auto-expansion, see
     *  above).
     */
    Matrix.prototype.get = function (row, col) {
        var rows = this.rows(), cols = this.cols();
        // auto-expansion for single row or column vector
        if (rows === 1) { row = 0; }
        if (cols === 1) { col = 0; }
        return ((row < rows) && (col < cols)) ? this._array[row][col] : ErrorCode.NA;
    };

    /**
     * Returns the value of the specified element in this matrix.
     *
     * @param {Number} index
     *  The zero-based row-oriented index of a matrix element. MUST be a valid
     *  index (not negative, less than the number of elements in the matrix).
     *
     * @returns {Number|String|Boolean|ErrorCode|Null}
     *  The value of the matrix element.
     */
    Matrix.prototype.getByIndex = function (index) {
        var cols = this.cols();
        return this._array[Math.floor(index / cols)][index % cols];
    };

    /**
     * Returns the specified row vector of this matrix as plain JS array.
     *
     * @param {Number} row
     *  The zero-based row index of the vector. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @returns {Array<Number|String|Boolean|ErrorCode|Null>}
     *  The specified row vector of this matrix.
     */
    Matrix.prototype.getRowAsArray = function (row) {
        return this._array[row].slice();
    };

    /**
     * Returns the specified row vector of this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the vector. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @returns {Matrix}
     *  The specified row vector of this matrix.
     */
    Matrix.prototype.getRowVector = function (row) {
        return Matrix.createRowVector(this.getRowAsArray(row));
    };

    /**
     * Returns the specified column vector of this matrix as plain JS array.
     *
     * @param {Number} col
     *  The zero-based column index of the vector. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @returns {Array<Number|String|Boolean|ErrorCode|Null>}
     *  The specified column vector of this matrix.
     */
    Matrix.prototype.getColAsArray = function (col) {
        return this._array.map(function (row) { return row[col]; });
    };

    /**
     * Returns the specified column vector of this matrix as plain JS array.
     *
     * @param {Number} col
     *  The zero-based column index of the vector. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @returns {Array<Number|String|Boolean|ErrorCode|Null>}
     *  The specified column vector of this matrix.
     */
    Matrix.prototype.getColVector = function (col) {
        return Matrix.createColVector(this.getColAsArray(col));
    };

    /**
     * Changes the value of the specified element in this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the element. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @param {Number} col
     *  The zero-based column index of the element. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The new value for the matrix element.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.set = function (row, col, value) {
        this._array[row][col] = value;
        return this;
    };

    /**
     * Changes the value of the specified element in this matrix.
     *
     * @param {Number} index
     *  The zero-based row-oriented index of a matrix element. MUST be a valid
     *  index (not negative, less than the number of elements in the matrix).
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The new value for the matrix element.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.setByIndex = function (index, value) {
        var cols = this.cols();
        this._array[Math.floor(index / cols)][index % cols] = value;
        return this;
    };

    /**
     * Updates the value of the specified element in this matrix to the return
     * value of the specified callback function.
     *
     * @param {Number} row
     *  The zero-based row index of the element. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @param {Number} col
     *  The zero-based column index of the element. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @param {Function} callback
     *  The callback function. Receives the current value of the matrix element
     *  as first parameter. The return value of this function will become the
     *  new value of the matrix element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.update = function (row, col, callback, context) {
        this._array[row][col] = callback.call(context, this._array[row][col]);
        return this;
    };

    /**
     * Updates the value of the specified element in this matrix to the return
     * value of the specified callback function.
     *
     * @param {Number} index
     *  The zero-based row-oriented index of a matrix element. MUST be a valid
     *  index (not negative, less than the number of elements in the matrix).
     *
     * @param {Function} callback
     *  The callback function. Receives the current value of the matrix element
     *  as first parameter. The return value of this function will become the
     *  new value of the matrix element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.updateByIndex = function (index, callback, context) {
        var cols = this.cols();
        return this.update(Math.floor(index / cols), index % cols, callback, context);
    };

    /**
     * Adds the passed number to the specified element in this matrix.
     *
     * @param {Number} row
     *  The zero-based row index of the element. MUST be a valid index (not
     *  negative, less than the number of rows in the matrix).
     *
     * @param {Number} col
     *  The zero-based column index of the element. MUST be a valid index (not
     *  negative, less than the number of columns in the matrix).
     *
     * @param {Number} number
     *  The number to be added to the matrix element.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.add = function (row, col, number) {
        this._array[row][col] += number;
        return this;
    };

    /**
     * Adds the passed number to the specified element in this matrix.
     *
     * @param {Number} index
     *  The zero-based row-oriented index of a matrix element. MUST be a valid
     *  index (not negative, less than the number of elements in the matrix).
     *
     * @param {Number} number
     *  The number to be added to the matrix element.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.addByIndex = function (index, number) {
        var cols = this.cols();
        this._array[Math.floor(index / cols)][index % cols] += number;
        return this;
    };

    /**
     * Changes the value of the specified rectangular subrange in this matrix.
     *
     * @param {Number} row1
     *  The zero-based row index of the first element. MUST be a valid index
     *  (not negative, less than the number of rows in the matrix).
     *
     * @param {Number} col1
     *  The zero-based column index of the first element. MUST be a valid index
     *  (not negative, less than the number of columns in the matrix).
     *
     * @param {Number} row2
     *  The zero-based row index of the second element. MUST be a valid index
     *  (not negative, less than the number of rows in the matrix, not less
     *  than 'row1').
     *
     * @param {Number} col2
     *  The zero-based column index of the second element. MUST be a valid
     *  index (not negative, less than the number of columns in the matrix, not
     *  less than 'col1').
     *
     * @param {Number|String|Boolean|ErrorCode|Null} value
     *  The new value for the matrix elements.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.fill = function (row1, col1, row2, col2, value) {
        for (var row = row1; row <= row2; row += 1) {
            for (var col = col1; col <= col2; col += 1) {
                this.set(row, col, value);
            }
        }
        return this;
    };

    /**
     * Creates an iterator that visits all elements contained in this matrix.
     * The elements will be visited row-by-row.
     *
     * @returns {Iterator}
     *  The new iterator. The result objects will contain the following value
     *  properties:
     *  - {Number|String|Boolean|ErrorCode|Null} value
     *      The value of the current matrix element.
     *  - {Number} row
     *      The zero-based row index of the matrix element.
     *  - {Number} col
     *      The zero-based column index of the matrix element.
     */
    Matrix.prototype.iterator = function () {
        // create a nested iterator that visits the matrix rows (outer), and the elements in a row (inner)
        return new NestedIterator(this._array, _.identity, function (rowResult, colResult) {
            return { value: colResult.value, row: rowResult.index, col: colResult.index };
        });
    };

    /**
     * Invokes the passed callback function for all elements contained in this
     * matrix.
     *
     * @param {Function} callback
     *  The callback function invoked for all elements in this matrix. Receives
     *  the following parameters:
     *  (1) {Number|String|Boolean|ErrorCode|Null} value
     *      The current matrix element.
     *  (2) {Number} row
     *      The zero-based row index of the matrix element.
     *  (3) {Number} col
     *      The zero-based column index of the matrix element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.forEach = function (callback, context) {
        this._array.forEach(function (vector, row) {
            vector.forEach(function (value, col) {
                callback.call(context, value, row, col);
            });
        });
        return this;
    };

    /**
     * Creates a new matrix with the same size of this matrix, and the elements
     * set to the return values of the specified callback function.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element in this
     *  matrix. Receives the following parameters:
     *  (1) {Number|String|Boolean|ErrorCode|Null} value
     *      The value of the current matrix element.
     *  (2) {Number} row
     *      The zero-based row index of the matrix element.
     *  (3) {Number} col
     *      The zero-based column index of the matrix element.
     *  The return value of the callback function MUST be a scalar value that
     *  will be inserted into the new matrix.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  The new matrix with the return values of the callback function.
     */
    Matrix.prototype.map = function (callback, context) {
        return Matrix.generate(this.rows(), this.cols(), function (row, col) {
            return callback.call(context, this[row][col], row, col);
        }, this._array);
    };

    /**
     * Invokes the passed callback function for all elements in this matrix,
     * and reduces the values to a single result value.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element in this
     *  matrix. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The return value of the previous invocation of this callback
     *      function (or the initial value on first invocation).
     *  (2) {Number|String|Boolean|ErrorCode|Null} value
     *      The value of the current matrix element.
     *  (3) {Number} row
     *      The zero-based row index of the matrix element.
     *  (4) {Number} col
     *      The zero-based column index of the matrix element.
     *  The return value of the callback function will become the new result
     *  value that will be passed as 'prevValue' with the next matrix element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The final result (the last return value of the callback function).
     */
    Matrix.prototype.reduce = function (initial, callback, context) {
        this.forEach(function (value, row, col) {
            initial = callback.call(context, initial, value, row, col);
        });
        return initial;
    };

    /**
     * Invokes the passed callback function for all elements contained in this
     * matrix, and sets the return values as new element values in-place.
     *
     * @param {Function} callback
     *  The callback function invoked for all elements in this matrix. Receives
     *  the following parameters:
     *  (1) {Number|String|Boolean|ErrorCode|Null} element
     *      The current matrix element.
     *  (2) {Number} row
     *      The zero-based row index of the matrix element.
     *  (3) {Number} col
     *      The zero-based column index of the matrix element.
     *  Must return the new value for the element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.transform = function (callback, context) {
        this.forEach(function (value, row, col) {
            this.set(row, col, callback.call(context, value, row, col));
        }, this);
        return this;
    };

    /**
     * Swaps the specified rows in this matrix in-place.
     *
     * @param {Number} row1
     *  The index of the first row to be swapped with the other row. MUST be a
     *  valid row index.
     *
     * @param {Number} row1
     *  The index of the second row to be swapped with the other row. MUST be a
     *  valid row index.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.swapRows = function (row1, row2) {
        var vector = this._array[row1];
        this._array[row1] = this._array[row2];
        this._array[row2] = vector;
        return this;
    };

    /**
     * Multiplies all elements of the specified row in-place with the passed
     * factor.
     *
     * @param {Number} row
     *  The index of the row to be changed. MUST be a valid row index. All
     *  elements of the specified row MUST be numbers.
     *
     * @param {Number} factor
     *  The factor to be used to multiply all row elements.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.scaleRow = function (row, factor) {
        this._array[row].forEach(function (value, col, vector) {
            vector[col] *= factor;
        });
        return this;
    };

    /**
     * Adds the scaled elements of a row in this matrix to the elements of
     * another row in-place.
     *
     * @param {Number} dstRow
     *  The index of the target row to be changed. MUST be a valid row index.
     *  All elements of the specified row MUST be numbers.
     *
     * @param {Number} srcRow
     *  The index of the source row whose elements will be scaled and added to
     *  the elements in the target row. MUST be a valid row index. All elements
     *  of the specified row MUST be numbers.
     *
     * @param {Number} factor
     *  The factor to be used to multiply all elements of the target row.
     *
     * @returns {Matrix}
     *  A reference to this instance.
     */
    Matrix.prototype.addScaledRow = function (dstRow, srcRow, factor) {
        var dstVector = this._array[dstRow];
        this._array[srcRow].forEach(function (value, col) {
            dstVector[col] += factor * value;
        });
        return this;
    };

    /**
     * Returns the transposed matrix of this matrix.
     *
     * @returns {Matrix}
     *  The transposed matrix of this matrix.
     */
    Matrix.prototype.transpose = function () {
        return Matrix.generate(this.cols(), this.rows(), function (row, col) {
            return this[col][row];
        }, this._array);
    };

    /**
     * Returns the string representation of this matrix for debug logging.
     *
     * @attention
     *  DO NOT USE this method for GUI related code where correct locale
     *  dependent representation of the matrix elements is required.
     *
     * @returns {String}
     *  The string representation of this matrix for debug logging.
     */
    Matrix.prototype.toString = function () {
        return '{' + this._array.map(function (vector) {
            return vector.map(Scalar.toString).join(';');
        }).join('|') + '}';
    };

    // exports ================================================================

    return Matrix;

});
