/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/complex', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/utils/mathutils'
], function (Utils, ErrorCode, MathUtils) {

    'use strict';

    // shortcuts to mathematical functions
    var exp = Math.exp;
    var log = Math.log;
    var pow = Math.pow;
    var sin = Math.sin;
    var cos = Math.cos;
    var atan2 = MathUtils.atan2;
    var sinh = MathUtils.sinh;
    var cosh = MathUtils.cosh;
    var isZero = MathUtils.isZero;

    // private global functions ===============================================

    /**
     * Throws the #NUM! error code, if the passed number is equal to, or too
     * close to zero.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Number}
     *  The passed number, if it is not zero.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number is zero.
     */
    function checkNotZero(number) {
        if (isZero(number)) { throw ErrorCode.NUM; }
        return number;
    }

    /**
     * Checks the passed complex numbers for their imaginary unit. Throws the
     * #VALUE! error code, if the imaginary units of the numbers are different.
     *
     * @param {Complex} c1
     *  The first complex number to be checked.
     *
     * @param {Complex} c2
     *  The second complex number to be checked.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the imaginary units of the passed complex
     *  numbers are different.
     */
    function checkUnits(c1, c2) {
        if (c1.unit && c2.unit && (c1.unit !== c2.unit)) { throw ErrorCode.VALUE; }
    }

    // class Complex ==========================================================

    /**
     * Representation of a complex number literal: a data structure with a
     * floating-point property for the real coefficient, and a floating-point
     * property for the imaginary coefficient.
     *
     * @constructor
     *
     * @property {Number} real
     *  The real coefficient of the complex number.
     *
     * @property {Number} imag
     *  The imaginary coefficient of the complex number.
     *
     * @property {String|Null} unit
     *  The imaginary unit (either 'i' or 'j'). If set to null, the imaginary
     *  unit to be used is undetermined yet (by default, 'i' will be used).
     */
    function Complex(real, imag, unit) {

        this.real = real;
        this.imag = imag;
        this.unit = (typeof unit === 'string') ? unit : null;

    } // class Complex

    // static methods ---------------------------------------------------------

    /**
     * Returns the sum of the passed complex numbers.
     *
     * @param {Complex} c1
     *  The first complex summand.
     *
     * @param {Complex} c2
     *  The second complex summand.
     *
     * @returns {Complex}
     *  The sum of the passed complex number.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the imaginary units of the passed complex
     *  numbers are different.
     */
    Complex.add = function (c1, c2) {
        checkUnits(c1, c2);
        return new Complex(c1.real + c2.real, c1.imag + c2.imag, c1.unit || c2.unit);
    };

    /**
     * Returns the difference between the passed complex numbers.
     *
     * @param {Complex} c1
     *  The complex minuend.
     *
     * @param {Complex} c2
     *  The complex subtrahend.
     *
     * @returns {Complex}
     *  The difference between the passed complex number.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the imaginary units of the passed complex
     *  numbers are different.
     */
    Complex.sub = function (c1, c2) {
        checkUnits(c1, c2);
        return new Complex(c1.real - c2.real, c1.imag - c2.imag, c1.unit || c2.unit);
    };

    /**
     * Returns the product of the passed complex numbers.
     *
     * @param {Complex} c1
     *  The first complex factor.
     *
     * @param {Complex} c2
     *  The second complex factor.
     *
     * @returns {Complex}
     *  The product of the passed complex number.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the imaginary units of the passed complex
     *  numbers are different.
     */
    Complex.mul = function (c1, c2) {
        checkUnits(c1, c2);
        return new Complex(c1.real * c2.real - c1.imag * c2.imag, c1.real * c2.imag + c1.imag * c2.real, c1.unit || c2.unit);
    };

    /**
     * Returns the quotient of the passed complex numbers.
     *
     * @param {Complex} c1
     *  The complex dividend.
     *
     * @param {Complex} c2
     *  The complex divisor.
     *
     * @returns {Complex}
     *  The quotient of the passed complex number.
     *
     * @throws {ErrorCode}
     *  The #VALUE! error code, if the imaginary units of the passed complex
     *  numbers are different. The #NUM! error code after a division by zero.
     */
    Complex.div = function (c1, c2) {
        checkUnits(c1, c2);
        // division of complex number by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(c2.real * c2.real + c2.imag * c2.imag);
        var real = c1.real * c2.real + c1.imag * c2.imag;
        var imag = c1.imag * c2.real - c1.real * c2.imag;
        return new Complex(real / quot, imag / quot, c1.unit || c2.unit);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this complex number.
     *
     * @returns {Complex}
     *  A clone of this complex number.
     */
    Complex.prototype.clone = function () {
        return new Complex(this.real, this.imag, this.unit);
    };

    /**
     * Returns the absolute value of this complex number.
     *
     * @returns {Number}
     *  The absolute value of this complex number.
     */
    Complex.prototype.abs = function () {
        return Utils.radius(this.real, this.imag);
    };

    /**
     * Returns the argument of this complex number, as radiant.
     *
     * @returns {Number}
     *  The argument of this complex number, as radiant.
     *
     * @throws {ErrorCode}
     *  Throws a #DIV/0! error, if this complex number is 0.
     */
    Complex.prototype.arg = function () {
        return atan2(this.real, this.imag);
    };

    /**
     * Returns the conjugate of this complex number.
     *
     * @returns {Complex}
     *  The conjugate (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     */
    Complex.prototype.conj = function () {
        return new Complex(this.real, -this.imag, this.unit);
    };

    /**
     * Returns the exponential of this complex number.
     *
     * @returns {Complex}
     *  The complex exponential (as new instance of the class Complex), using
     *  the imaginary unit of this complex number.
     */
    Complex.prototype.exp = function () {
        var cabs = exp(this.real);
        return new Complex(cabs * cos(this.imag), cabs * sin(this.imag), this.unit);
    };

    /**
     * Returns the power of this complex number to the passed real exponent.
     *
     * @param {Number} exp
     *  The exponent, as a real number.
     *
     * @returns {Complex}
     *  The complex power (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if both numbers are zero.
     */
    Complex.prototype.pow = function (exp) {

        // 0^exp is undefined for non-positive exponents, and 0 otherwise
        if (isZero(this.real) && isZero(this.imag)) {
            if (exp <= 0) { throw ErrorCode.NUM; }
            return new Complex(0, 0, this.unit);
        }

        // calculate complex power via polar representation
        var cabs = pow(this.abs(), exp), carg = exp * this.arg();
        return new Complex(cabs * cos(carg), cabs * sin(carg), this.unit);
    };

    /**
     * Returns the logarithm of this complex number to the passed base.
     *
     * @param {Number} base
     *  The natural logarithm of the base, as a real number (e.g., for the
     *  value 1, this method will return the natural logarithm of this complex
     *  number; for the value Math.LN10, this method will return the decimal
     *  logarithm of this complex number).
     *
     * @returns {Complex}
     *  The complex logarithm (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  Throws a #DIV/0! error, if this complex number is 0.
     */
    Complex.prototype.log = function (base) {
        return new Complex(log(this.abs()) / base, this.arg() / base, this.unit);
    };

    /**
     * Returns the sine of this complex number.
     *
     * @returns {Complex}
     *  The complex sine (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     */
    Complex.prototype.sin = function () {
        return new Complex(sin(this.real) * cosh(this.imag), cos(this.real) * sinh(this.imag), this.unit);
    };

    /**
     * Returns the secant of this complex number.
     *
     * @returns {Complex}
     *  The complex secant (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code after a division by zero.
     */
    Complex.prototype.sec = function () {
        // division by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(cos(2 * this.real) + cosh(2 * this.imag));
        return new Complex(2 * cos(this.real) * cosh(this.imag) / quot, 2 * sin(this.real) * sinh(this.imag) / quot, this.unit);
    };

    /**
     * Returns the cosine of this complex number.
     *
     * @returns {Complex}
     *  The complex cosine (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     */
    Complex.prototype.cos = function () {
        return new Complex(cos(this.real) * cosh(this.imag), -sin(this.real) * sinh(this.imag), this.unit);
    };

    /**
     * Returns the cosecant of this complex number.
     *
     * @returns {Complex}
     *  The complex cosecant (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code after a division by zero.
     */
    Complex.prototype.csc = function () {
        // division by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(cos(2 * this.real) - cosh(2 * this.imag));
        return new Complex(-2 * sin(this.real) * cosh(this.imag) / quot, 2 * cos(this.real) * sinh(this.imag) / quot, this.unit);
    };

    /**
     * Returns the tangent of this complex number.
     *
     * @returns {Complex}
     *  The complex tangent (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code after a division by zero.
     */
    Complex.prototype.tan = function () {
        // division by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(cos(2 * this.real) + cosh(2 * this.imag));
        return new Complex(sin(2 * this.real) / quot, sinh(2 * this.imag) / quot, this.unit);
    };

    /**
     * Returns the cotangent of this complex number.
     *
     * @returns {Complex}
     *  The complex cotangent (as new instance of the class Complex), using the
     *  imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code after a division by zero.
     */
    Complex.prototype.cot = function () {
        // division by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(cos(2 * this.real) - cosh(2 * this.imag));
        return new Complex(-sin(2 * this.real) / quot, sinh(2 * this.imag) / quot, this.unit);
    };

    /**
     * Returns the hyperbolic sine of this complex number.
     *
     * @returns {Complex}
     *  The complex hyperbolic sine (as new instance of the class Complex),
     *  using the imaginary unit of this complex number.
     */
    Complex.prototype.sinh = function () {
        return new Complex(sinh(this.real) * cos(this.imag), cosh(this.real) * sin(this.imag), this.unit);
    };

    /**
     * Returns the hyperbolic secant of this complex number.
     *
     * @returns {Complex}
     *  The complex hyperbolic secant (as new instance of the class Complex),
     *  using the imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code after a division by zero.
     */
    Complex.prototype.sech = function () {
        // division by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(cosh(2 * this.real) + cos(2 * this.imag));
        return new Complex(2 * cosh(this.real) * cos(this.imag) / quot, -2 * sinh(this.real) * sin(this.imag) / quot, this.unit);
    };

    /**
     * Returns the hyperbolic cosine of this complex number.
     *
     * @returns {Complex}
     *  The complex hyperbolic cosine (as new instance of the class Complex),
     *  using the imaginary unit of this complex number.
     */
    Complex.prototype.cosh = function () {
        return new Complex(cosh(this.real) * cos(this.imag), sinh(this.real) * sin(this.imag), this.unit);
    };

    /**
     * Returns the hyperbolic cosecant of this complex number.
     *
     * @returns {Complex}
     *  The complex hyperbolic cosecant (as new instance of the class Complex),
     *  using the imaginary unit of this complex number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code after a division by zero.
     */
    Complex.prototype.csch = function () {
        // division by zero results in #NUM! instead of #DIV/0! as in regular formulas
        var quot = checkNotZero(cosh(2 * this.real) - cos(2 * this.imag));
        return new Complex(2 * sinh(this.real) * cos(this.imag) / quot, -2 * cosh(this.real) * sin(this.imag) / quot, this.unit);
    };

    /**
     * Returns the string representation of this complex number for debug
     * logging.
     *
     * @attention
     *  DO NOT USE this method for GUI related code where correct locale
     *  dependent representation of the complex number is required.
     *
     * @returns {String}
     *  The string representation of this complex number for debug logging.
     */
    Complex.prototype.toString = function () {
        return this.real + ((this.imag >= 0) ? '+' : '') + this.imag + (this.unit || 'i');
    };

    // exports ================================================================

    return Complex;

});
