/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/parser/expressionparser', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/class',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/tokens'
], function (Utils, Class, Parser, SheetUtils, FormulaUtils, Tokens) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Matrix = FormulaUtils.Matrix;
    var SheetRef = FormulaUtils.SheetRef;

    // class TokenDescriptor ==================================================

    /**
     * A descriptor for a single formula token parsed from a formula expression
     * string, together with additional information such as the original
     * display text of the token.
     *
     * @property {BaseToken} token
     *  The formula token.
     *
     * @property {Number} index
     *  The array index of the formula token.
     *
     * @property {String} text
     *  The display text of the token from the original formula expression.
     *
     * @property {Number} start
     *  The start position of the text in the original formula expression.
     *
     * @property {Number} end
     *  The end position of the text in the original formula expression.
     */
    function TokenDescriptor(token, text) {

        this.token = token;
        this.index = -1;
        this.text = text;
        this.start = 0;
        this.end = 0;

    } // class TokenDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns a text description of the token for debugging (!) purposes.
     */
    TokenDescriptor.prototype.toString = function () {
        return this.token.toString();
    };

    // class ExpressionParserBase =============================================

    /**
     * An instance of this class parses formula expressions according to a
     * specific formula grammar. This is the base class of several class
     * specializations for the different grammars and reference syntaxes.
     *
     * @constructor
     */
    var ExpressionParserBase = Class.extendable(function (docModel, formulaGrammar) {

        // the document model
        this.docModel = docModel;

        // formula grammar settings
        this.grammar = formulaGrammar;
        this.RE = formulaGrammar.RE;

        // reference grammar configuration
        this.refGrammar = formulaGrammar.refGrammar;

    }); // class ExpressionParserBase

    // private methods --------------------------------------------------------

    /**
     * Initializes the members of this parser instance according to the passed
     * options.
     *
     * @param {Object} [options]
     *  Optional parameters.
     */
    ExpressionParserBase.prototype._initOptions = function (options) {

        // position of the reference cell
        this.refAddress = Utils.getOption(options, 'refAddress', Address.A1);
        this.refSheet = Utils.getIntegerOption(options, 'refSheet', -1);
        this.refSheetModel = this.docModel.getSheetModel(this.refSheet);

        // whether to wrap relocated range addresses at the sheet borders
        this.wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false);
        // whether to automatically correct missing parentheses etc.
        this.autoCorrect = Utils.getBooleanOption(options, 'autoCorrect', false);
        // whether to extend local references with the reference sheet
        this.extendSheet = Utils.getBooleanOption(options, 'extendSheet', false);
        // whether to resolve unqualified table reference according to the reference address
        this.unqualifiedTables = Utils.getBooleanOption(options, 'unqualifiedTables', false);
    };

    /**
     * Extracts the next token from the passed formula subexpression.
     *
     * @param {String} expr
     *  The formula subexpression to be parsed.
     *
     * @returns {TokenDescriptor|Null}
     *  A descriptor for the parsed formula token, or null on error.
     */
    ExpressionParserBase.prototype._parseNextToken = function (expr) {

        // the collection of regular expressions
        var RE = this.RE;
        // the matches of the regular expressions
        var matches = null;
        // a token descriptor built by another helper method of this class
        var tokenDesc = null;

        // string literal
        if ((matches = RE.STRING_LIT.exec(expr))) {
            return this.createLiteralToken(matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"'), matches[0]);
        }

        // boolean literal (before defined names)
        if ((matches = RE.BOOLEAN_LIT.exec(expr))) {
            return this.createLiteralToken(this.grammar.getBooleanValue(matches[0]), matches[0]);
        }

        // error code literal
        if ((matches = RE.ERROR_LIT.exec(expr))) {
            return this.createLiteralToken(this.grammar.getErrorCode(matches[0]), matches[0]);
        }

        // fraction literal, e.g. '1 2/3'
        if ((matches = RE.FRACTION_LIT.exec(expr))) {
            return this.createLiteralToken(parseInt(matches[1], 10) + parseInt(matches[2], 10) / parseInt(matches[3], 10), matches[0]);
        }

        // whitespace (MUST be parsed before operators)
        if ((matches = RE.WHITESPACE.exec(expr))) {
            return this.createFixedToken('ws', matches[0]);
        }

        // list/parameter separator (MUST be parsed before operators)
        if ((matches = RE.SEPARATOR.exec(expr))) {
            return this.createToken(new Tokens.SeparatorToken(), matches[0]);
        }

        // unary/binary operator
        if ((matches = RE.OPERATOR.exec(expr))) {
            return this.createToken(new Tokens.OperatorToken(this.grammar.getOperatorKey(matches[0])), matches[0]);
        }

        // opening parenthesis
        if ((matches = RE.OPEN.exec(expr))) {
            return this.createToken(new Tokens.ParenthesisToken(true), matches[0]);
        }

        // closing parenthesis
        if ((matches = RE.CLOSE.exec(expr))) {
            return this.createToken(new Tokens.ParenthesisToken(false), matches[0]);
        }

        // opening parenthesis of a matrix literal
        if ((matches = RE.MATRIX_OPEN.exec(expr))) {
            return this._parseMatrixLit(expr);
        }

        // any type of cell reference without and with sheets
        if ((tokenDesc = this.parseNextRefToken(expr))) {
            return tokenDesc;
        }

        // number literal (must be parsed after row ranges, e.g. 1:3 is a row range instead of two numbers)
        if ((matches = Parser.parseLeadingNumber(expr, { dec: this.grammar.DEC }))) {
            return this.createLiteralToken(matches.number, matches.text);
        }

        // defined names, table ranges, and function calls
        if ((tokenDesc = this.parseNextNameToken(expr))) {
            return tokenDesc;
        }
    };

    /**
     * Parses a complete matrix literal from the passed formula subexpression.
     *
     * @param {String} expr
     *  The formula subexpression to be parsed.
     *
     * @returns {TokenDescriptor|Null}
     *  A descriptor for the parsed matrix token, or null on error.
     */
    ExpressionParserBase.prototype._parseMatrixLit = function (expr) {

        // copy of the passed original subexpression
        var origExpr = expr;
        // current state of the table reference processor
        var state = 'start';
        // the collection of regular expressions
        var RE = this.RE;
        // the matches of the regular expressions
        var matches = null;
        // the two-dimensional array with the matrix elements
        var matrixValues = [];
        // the last row in the matrix to be filled by the parser
        var lastRow = null;

        // returns whether all rows in 'matrixValues' have the same length
        function checkMatrixSize() {
            return matrixValues.every(function (row, index) {
                return (index === 0) || (row.length === matrixValues[index - 1].length);
            });
        }

        // parse until the closing brace has been found, or an error occurs
        while (state !== 'end') {

            // try to parse the next particle of the matrix literal
            switch (state) {

                case 'start':
                    if ((matches = RE.MATRIX_OPEN.exec(expr))) {
                        matrixValues.push(lastRow = []);
                        state = 'open';
                    }
                    break;

                case 'open':
                case 'sep':
                    if ((matches = RE.WHITESPACE.exec(expr))) {
                        // whitespace: keep current state
                    } else if ((matches = RE.FRACTION_LIT.exec(expr))) {
                        // fraction literal, e.g. '1 2/3'
                        lastRow.push(parseInt(matches[1], 10) + parseInt(matches[2], 10) / parseInt(matches[3], 10));
                        state = 'lit';
                    } else if ((matches = Parser.parseLeadingNumber(expr, { dec: this.grammar.DEC, sign: true }))) {
                        // decimal/scientific number literal
                        lastRow.push(matches.number);
                        matches = [matches.text];
                        state = 'lit';
                    } else if ((matches = RE.STRING_LIT.exec(expr))) {
                        // string literal
                        lastRow.push(matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"'));
                        state = 'lit';
                    } else if ((matches = RE.BOOLEAN_LIT.exec(expr))) {
                        // boolean literal
                        lastRow.push(this.grammar.getBooleanValue(matches[0]));
                        state = 'lit';
                    } else if ((matches = RE.ERROR_LIT.exec(expr))) {
                        // error code literal
                        lastRow.push(this.grammar.getErrorCode(matches[0]));
                        state = 'lit';
                    }
                    break;

                case 'lit':
                    // literal value: a separator or closing brace must follow
                    if (expr.length === 0) {
                        // end of expression reached: accept missing closing brace in auto-correct mode
                        matches = (this.autoCorrect && checkMatrixSize()) ? [''] : null;
                        state = 'end';
                    } else if ((matches = RE.WHITESPACE.exec(expr))) {
                        // whitespace: keep current state
                    } else if ((matches = RE.MATRIX_ROW_SEPARATOR.exec(expr))) {
                        matrixValues.push(lastRow = []);
                        state = 'sep';
                    } else if ((matches = RE.MATRIX_COL_SEPARATOR.exec(expr))) {
                        state = 'sep';
                    } else if ((matches = RE.MATRIX_CLOSE.exec(expr))) {
                        if (!checkMatrixSize()) { matches = null; }
                        state = 'end';
                    }
                    break;
            }

            // immediately return on error; otherwise reduce the formula subexpression
            if (!matches) { return null; }
            expr = expr.substr(matches[0].length);
        }

        // create a new matrix token, and insert the entire parsed text of the formula expression into the token descriptor
        var text = origExpr.substr(0, origExpr.length - expr.length);
        return this.createToken(new Tokens.MatrixToken(new Matrix(matrixValues)), text);
    };

    // protected methods ------------------------------------------------------

    /**
     * Creates a token descriptor from the passed token and original string
     * representation.
     *
     * @param {BaseToken} token
     *  The new token instance to be inserted into the token descriptor
     *  returned from this method.
     *
     * @param {String} text
     *  The original text representation of the passed token in the parsed
     *  formula expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing the passed token and text.
     */
    ExpressionParserBase.prototype.createToken = function (token, text) {
        return new TokenDescriptor(token, text);
    };

    /**
     * Creates a token descriptor containing a new fixed text token.
     *
     * @param {String} type
     *  The type of the fixed token to be inserted into the new token
     *  descriptor returned from this method.
     *
     * @param {String} text
     *  The text representation of the fixed token in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new fixed token, and the passed text.
     */
    ExpressionParserBase.prototype.createFixedToken = function (type, text) {
        return this.createToken(new Tokens.FixedToken(type, text), text);
    };

    /**
     * Creates a token descriptor containing a new literal token.
     *
     * @param {Any} value
     *  The value for the literal token to be inserted into the new token
     *  descriptor returned from this method.
     *
     * @param {String} text
     *  The original text representation of the value in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new literal token, and the passed
     *  text.
     */
    ExpressionParserBase.prototype.createLiteralToken = function (value, text) {
        return this.createToken(new Tokens.LiteralToken(value), text);
    };

    /**
     * Creates a token descriptor containing a new reference token.
     *
     * @param {Object|Null} sheetRefs
     *  An object containing zero, one, or two sheet references (instances of
     *  the class SheetRef), in the properties 'r1' and 'r2'. If null is passed
     *  as parameter value, or if property 'r1' of the object is null, a local
     *  reference token without sheets will be created.
     *
     * @param {Object|Null} cellRefs
     *  An object containing zero, one, or two cell references (instances of
     *  the class CellRef), in the properties 'r1' and 'r2'. If null is passed
     *  as parameter value, or if property 'r1' of the object is null, a
     *  reference token representing a reference error will be created.
     *
     * @param {String} text
     *  The original text representation of the reference in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new reference token, and the passed
     *  text.
     */
    ExpressionParserBase.prototype.createReferenceToken = function (sheetRefs, cellRefs, text) {

        // bug 40293: add sheet reference to sheet-local cell references if specified
        if (!sheetRefs && this.extendSheet && this.refSheetModel) {
            sheetRefs = { r1: new SheetRef(this.refSheet, true), r2: null };
        }

        return this.createToken(new Tokens.ReferenceToken(
            this.docModel,
            cellRefs ? cellRefs.r1 : null,
            cellRefs ? cellRefs.r2 : null,
            sheetRefs ? sheetRefs.r1 : null,
            sheetRefs ? sheetRefs.r2 : null
        ), text);
    };

    /**
     * Creates a token descriptor containing a new name token.
     *
     * @param {SheetRef|Null} sheetRef
     *  A sheet reference structure for sheet-local names; or null for a global
     *  name token without sheet.
     *
     * @param {String} label
     *  The label of the defined name.
     *
     * @param {String} text
     *  The original text representation of the defined name in the parsed
     *  formula expression.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.resolveTable=false]
     *      If set to true, and the passed label refers to a table range, a
     *      table reference token will be created.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new name token, and the passed text.
     */
    ExpressionParserBase.prototype.createNameToken = function (sheetRef, label, text, options) {

        // bug 40293: add sheet reference to existing sheet-local names if specified
        if (!sheetRef && this.extendSheet && this.refSheetModel && this.refSheetModel.getNameCollection().hasName(label)) {
            sheetRef = new SheetRef(this.refSheet, true);
        }

        // workaround for bugs 51203 and 52728, TODO: support for arbitrary external references
        var globalRef = !sheetRef && /^\[0\]!/.test(text);
        // create the name token
        var nameToken = new Tokens.NameToken(this.docModel, label, sheetRef, globalRef ? 0 : null);

        // try to find a table range model referred by the passed label
        var tableModel = Utils.getBooleanOption(options, 'resolveTable', false) ? nameToken.resolveTableModel() : null;
        if (tableModel) { nameToken = new Tokens.TableToken(this.docModel, label); }

        return this.createToken(nameToken, text);
    };

    /**
     * Creates a token descriptor containing a new function/macro token.
     *
     * @param {SheetRef|Null} sheetRef
     *  A sheet reference structure for macro calls with sheet/module name; or
     *  null for a built-in function, or a global macro call without sheet.
     *
     * @param {String} name
     *  The name of the function/macro.
     *
     * @param {String} text
     *  The original text representation of the function in the parsed formula
     *  expression.
     *
     * @returns {TokenDescriptor}
     *  The token descriptor containing a new function token, and the passed
     *  text.
     */
    ExpressionParserBase.prototype.createFunctionToken = function (sheetRef, name, text) {

        // the resource key of a built-in sheet function (method getFunctionKey() returns null for unknown functions)
        var funcKey = sheetRef ? null : this.grammar.getFunctionKey(name);

        return this.createToken(funcKey ? new Tokens.FunctionToken(funcKey) : new Tokens.MacroToken(this.docModel, name, sheetRef), text);
    };

    /**
     * Returns a sheet reference structure with either a fixed (implicit), or a
     * variable absolute flag, for the passed matches of a regular expression.
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index
     *  The start array index of the sheet reference in the passed matches.
     *
     * @returns {SheetRef}
     *  A sheet reference structure for the passed matches.
     */
    ExpressionParserBase.prototype.createSheetRef = function (matches, index) {
        return this.refGrammar.createSheetRef(this.docModel, matches, index);
    };

    /**
     * Returns an object with sheet reference structures for a single sheet
     * name, or for a range of sheet names from the passed matches of a regular
     * expression, as expected by ExpressionParserBase.createReferenceToken().
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first sheet reference in the passed
     *  matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second sheet reference in the passed
     *  matches. If omitted, the matches for a single sheet name will be
     *  converted to the result.
     *
     * @returns {Object}
     *  An object with properties 'r1' and 'r2' set to the sheet reference
     *  structures for the passed matches (property 'r2' will be null for a
     *  single sheet name).
     */
    ExpressionParserBase.prototype.createSheetRefs = function (matches, index1, index2) {
        return this.refGrammar.createSheetRefs(this.docModel, matches, index1, index2);
    };

    /**
     * Returns an object with two cell reference structures for a cell range
     * address from the passed matches of a regular expression, as expected by
     * the public method ExpressionParserBase.createReferenceToken().
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first cell address in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second cell address in the passed matches.
     *  If omitted, the matches for a single cell address will be converted to
     *  the result.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address (or null for single cell addresses), if the
     *  column and row indexes are all valid; otherwise null.
     */
    ExpressionParserBase.prototype.createCellRangeRefs = function (matches, index1, index2) {
        return this.refGrammar.createCellRangeRefs(this.docModel, this.refAddress, this.wrapReferences, matches, index1, index2);
    };

    /**
     * Returns an object with two cell reference structures for a column range
     * address from the passed matches of a regular expression, as expected by
     * the public method ExpressionParserBase.createReferenceToken().
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first column index in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second column index in the passed matches.
     *  If omitted, the matches for a single column reference will be converted
     *  to the result.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the column indexes are all valid;
     *  otherwise null.
     */
    ExpressionParserBase.prototype.createColRangeRefs = function (matches, index1, index2) {
        return this.refGrammar.createColRangeRefs(this.docModel, this.refAddress, this.wrapReferences, matches, index1, index2);
    };

    /**
     * Returns an object with two cell reference structures for a row range
     * address from the passed matches of a regular expression, as expected by
     * the public method ExpressionParserBase.createReferenceToken().
     *
     * @param {Array<String>} matches
     *  The matches of a regular expression.
     *
     * @param {Number} index1
     *  The start array index of the first row index in the passed matches.
     *
     * @param {Number} [index2]
     *  The start array index of the second row index in the passed matches. If
     *  omitted, the matches for a single column reference will be converted to
     *  the result.
     *
     * @returns {Object|Null}
     *  An object with 'r1' property set to a cell reference structure for the
     *  first cell address, and 'r2' property set to a cell reference structure
     *  for the second cell address, if the row indexes are all valid;
     *  otherwise null.
     */
    ExpressionParserBase.prototype.createRowRangeRefs = function (matches, index1, index2) {
        return this.refGrammar.createRowRangeRefs(this.docModel, this.refAddress, this.wrapReferences, matches, index1, index2);
    };

    // virtual methods --------------------------------------------------------

    /**
     * Sub classes MUST provide an implementation to parse a reference token
     * from the passed formula subexpression.
     */
    ExpressionParserBase.prototype.parseNextRefToken = function (/*expr*/) {
        Utils.error('ExpressionParserBase.parseNextRefToken(): missing implementation');
        return null;
    };

    /**
     * Sub classes MUST provide an implementation to parse a name or function
     * token from the passed formula subexpression.
     */
    ExpressionParserBase.prototype.parseNextNameToken = function (/*expr*/) {
        Utils.error('ExpressionParserBase.parseNextNameToken(): missing implementation');
        return null;
    };

    // public methods ---------------------------------------------------------

    /**
     * Parses and tokenizes the passed complete formula expression.
     *
     * @param {String} formula
     *  The formula expression to be parsed and tokenized.
     *
     * @param {Object} [options]
     *  Optional parameters. See method FormulaParser.parseFormula() for more
     *  details.
     *
     * @returns {Array<TokenDescriptor>}
     *  An array of token descriptors containing the parsed formula tokens, and
     *  the original text representations of the tokens.
     */
    ExpressionParserBase.prototype.parseFormula = function (formula, options) {

        // initialize all instance properties
        this._initOptions(options);

        // the descriptors of the resulting formula tokens
        var tokenDescs = [];

        // extract all supported tokens from start of formula string
        while (formula.length > 0) {

            // try to parse the next formula token from the beginning of the (remaining) formula expression;
            // push entire remaining formula text as 'bad' token on parser error
            var tokenDesc = this._parseNextToken(formula) || this.createFixedToken('bad', formula);
            tokenDescs.push(tokenDesc);

            // remove the text from the formula expression
            formula = formula.substr(tokenDesc.text.length);
        }
        FormulaUtils.logTokens('tokenize', tokenDescs);

        // whether the intersection operator is a space and needs post-processing
        var spaceAsIsect = this.grammar.isSpaceIntersection();

        // post-process the token array
        Utils.iterateArray(tokenDescs, function (tokenDesc, index) {

            // the current token
            var token = tokenDesc.token;
            // the token preceding and following the current token
            var prevTokenDesc = tokenDescs[index - 1];
            var nextTokenDesc = tokenDescs[index + 1];

            // replace combination of 'op[sub] lit[number]' with a negative number, if the
            // minus is the leading token or preceded by an operator or opening parenthesis
            if (token.isType('op') && (token.getValue() === 'sub') && nextTokenDesc && nextTokenDesc.token.isType('lit')) {

                // the scalar value of the next token
                var nextValue = nextTokenDesc.token.getValue();
                if ((typeof nextValue === 'number') && (nextValue > 0)) {

                    // the next non-whitespace token preceding the current token
                    var prevNonWsTokenDesc = (prevTokenDesc && prevTokenDesc.token.isType('ws')) ? tokenDescs[index - 2] : prevTokenDesc;
                    if (!prevNonWsTokenDesc || prevNonWsTokenDesc.token.matchesType(/^(op|sep|open)$/)) {
                        tokenDesc.token = new Tokens.LiteralToken(-nextValue);
                        tokenDesc.text += nextTokenDesc.text;
                        tokenDescs.splice(index + 1, 1);
                    }
                }
            }

            // if the range intersection operator is a space character, insert an operator
            // token for a white-space token containing a SPACE character and surrounded
            // by reference/name tokens or other subexpressions in parentheses
            if (spaceAsIsect && token.isType('ws')) {

                var spaceIndex = spaceIndex = tokenDesc.text.indexOf(' ');
                if ((spaceIndex >= 0) &&
                    prevTokenDesc && prevTokenDesc.token.matchesType(/^(ref|name|close)$/) &&
                    nextTokenDesc && nextTokenDesc.token.matchesType(/^(ref|name|open|func)$/)
                ) {

                    // insert the remaining white-space following the first SPACE character
                    if (spaceIndex + 1 < tokenDesc.text.length) {
                        var trailingSpace = tokenDesc.text.slice(spaceIndex + 1);
                        tokenDescs.splice(index + 1, 0, this.createFixedToken('ws', trailingSpace));
                    }

                    // insert an intersection operator for the SPACE character after the current token
                    tokenDescs.splice(index + 1, 0, this.createToken(new Tokens.OperatorToken('isect'), ' '));

                    // shorten the current white-space token to the text preceding the first SPACE character, or remove it
                    if (spaceIndex > 0) {
                        token.setValue(tokenDesc.text.slice(0, spaceIndex));
                        tokenDesc.text = token.getValue();
                    } else {
                        tokenDescs.splice(index, 1);
                    }
                }
            }

        }, { reverse: true, context: this });
        FormulaUtils.logTokens('postprocess', tokenDescs);

        // add text positions to the token descriptors
        tokenDescs.forEach(function (tokenDesc, index) {
            tokenDesc.index = index;
            tokenDesc.start = (index === 0) ? 0 : tokenDescs[index - 1].end;
            tokenDesc.end = tokenDesc.start + tokenDesc.text.length;
        });

        return tokenDescs;
    };

    /**
     * Releases the references to other document objects.
     */
    ExpressionParserBase.prototype.destroy = function () {
        this.docModel = this.grammar = this.RE = this.refSheetModel = null;
    };

    // class ExpressionParserOOX ==============================================

    /**
     * A formula expression parser for the OOXML reference syntax.
     *
     * @constructor
     *
     * @extends ExpressionParserBase
     */
    var ExpressionParserOOX = ExpressionParserBase.extend(function (docModel, formulaGrammar) {

        // base constructor
        ExpressionParserBase.call(this, docModel, formulaGrammar);

    }); // class ExpressionParserOOX

    // private methods --------------------------------------------------------

    /**
     * Extracts a complete structured table reference from the passed formula
     * subexpression.
     *
     * @param {String} expr
     *  The formula subexpression to be parsed.
     *
     * @param {Object|Null} nameTokenDesc
     *  Token descriptor of a defined name preceding the structured table
     *  reference. If set to null, an unqualified table reference has been
     *  found in the formula expression.
     *
     * @returns {TokenDescriptor|Null}
     *  A descriptor for the parsed formula token, or null on error.
     */
    ExpressionParserOOX.prototype._parseTableRef = function (expr, nameTokenDesc) {

        // the collection of regular expressions
        var RE = this.RE;
        // copy of the passed original subexpression
        var origExpr = expr;
        // remove the parsed name token from the expression
        if (nameTokenDesc) { expr = expr.substr(nameTokenDesc.text.length); }

        // whether a structured table reference starts in the formula expression
        var hasTableRef = RE.TABLE_OPEN.test(expr);
        // try to find a table range model referred by the name token
        var tableModel = nameTokenDesc ? nameTokenDesc.token.resolveTableModel() : null;

        // a qualified table reference must be preceded by a name token referring to an existing table range
        if (nameTokenDesc && hasTableRef && !tableModel) { return null; }

        // create a table token for an existing table range without brackets
        if (nameTokenDesc && !hasTableRef && tableModel) {
            return this.createToken(new Tokens.TableToken(this.docModel, nameTokenDesc.token.getValue()), nameTokenDesc.text);
        }

        // return the original name token, if no opening bracket is available
        if (!hasTableRef) { return nameTokenDesc; }

        // UI grammars: use the table at the reference address for unqualified table references
        if (!nameTokenDesc && this.grammar.UI && this.unqualifiedTables && this.refSheetModel) {
            tableModel = this.refSheetModel.getTableCollection().findTable(this.refAddress);
        }

        // table model must be available at the end
        if (!tableModel) { return null; }

        // reference to the grammar configuration for local functions
        var grammar = this.grammar;
        // auto-correction mode, for usage in inner functions
        var autoCorrect = this.autoCorrect;
        // current state of the table reference processor
        var state = 'start';
        // the matches of the regular expressions
        var matches = null;
        // the options needed to create a table token
        var tableOptions = {};
        // all collected region keys, as flag set
        var regionKeys = {};

        // adds a table region key to the table options object
        function handleRegionName() {
            regionKeys[grammar.getTableRegionKey(matches[1])] = true;
            state = 'ref';
        }

        // adds one or two column names to the table options object
        function setColNames(match1, match2) {
            if ('col1Name' in tableOptions) {
                matches = null;
                state = 'end';
            } else {
                tableOptions.col1Name = grammar.decodeTableColumn(match1);
                tableOptions.col2Name = match2 ? grammar.decodeTableColumn(match2) : null;
                state = 'ref';
            }
        }

        function handleColRef() {
            setColNames(matches[1], matches[2]);
        }

        function handleMixedColRef(leading) {
            if (leading && matches[1]) { tableOptions.openWs = true; }
            setColNames(matches[2] || matches[3], matches[4] || matches[5]);
        }

        // adds whitespace characters for the closing bracket to the table options object
        function handleTableClose() {
            if (matches[1]) { tableOptions.closeWs = true; }
            // accept missing closing bracket in auto-correct mode only
            if (!autoCorrect && (matches[2].length === 0)) { matches = null; }
            state = 'end';
        }

        // parse until the closing bracket has been found, or an error occurs
        while (state !== 'end') {

            // try to parse the next particle of the table reference
            switch (state) {

                case 'start':
                    if ((matches = RE.TABLE_OPEN.exec(expr))) {
                        if (matches[1]) { tableOptions.openWs = true; }
                        state = 'open';
                    }
                    break;

                case 'open':
                    if ((matches = RE.TABLE_CLOSE.exec(expr))) {
                        // empty table reference, e.g. Table1[]
                        handleTableClose();
                    } else if ((matches = RE.TABLE_SINGLE_REGION_NAME.exec(expr))) {
                        // single table region without brackets, e.g. #All in Table1[#All]
                        handleRegionName();
                    } else if ((matches = RE.TABLE_SINGLE_COLUMN_NAME.exec(expr))) {
                        // single table column without brackets, e.g. Column1 in Table1[Column1]
                        handleColRef();
                    } else if ((matches = RE.TABLE_REGION_REF.exec(expr))) {
                        // table region, e.g. [#All] in Table1[[#All],[Column1]]
                        handleRegionName();
                    } else if (!this.grammar.UI && (matches = RE.TABLE_COLUMN_RANGE_REF.exec(expr))) {
                        // table column range, e.g. [Column1]:[Column2] in Table1[[#All],[Column1]:[Column2]]
                        handleColRef();
                    } else if (!this.grammar.UI && (matches = RE.TABLE_COLUMN_REF.exec(expr))) {
                        // table column name in brackets, e.g. [Column1] in Table1[[#All],[Column1]]
                        handleColRef();
                    } else if (this.grammar.UI && (matches = RE.TABLE_MIXED_COLUMN_RANGE_REF.exec(expr))) {
                        // table column range with or without brackets, e.g. Column1:[Column 2] in Table1[[#All],Column1:[Column 2]]
                        handleMixedColRef(true);
                    } else if (this.grammar.UI && (matches = RE.TABLE_MIXED_COLUMN_REF.exec(expr))) {
                        // table column name with or without brackets, e.g. [Column1] in Table1[[#All],Column1]
                        handleMixedColRef(true);
                    } else if (this.grammar.UI && (matches = RE.TABLE_THISROW_REF.exec(expr))) {
                        // at-sign as shortcut for [#This Row] in UI grammars
                        regionKeys.ROW = true;
                        state = 'row';
                    }
                    break;

                case 'ref':
                    if ((matches = RE.TABLE_CLOSE.exec(expr))) {
                        // closing bracket (add preceding whitespace to the options)
                        handleTableClose();
                    } else if ((matches = RE.TABLE_SEPARATOR.exec(expr))) {
                        // separator in table references, e.g. the comma in Table1[[#All],[Column1]]
                        if (matches[2]) { tableOptions.sepWs = true; }
                        state = 'sep';
                    }
                    break;

                case 'sep':
                    if ((matches = RE.TABLE_REGION_REF.exec(expr))) {
                        // table region (always in brackets), e.g. [#All] in Table1[[Column1],[#All]]
                        handleRegionName();
                    } else if (!this.grammar.UI && (matches = RE.TABLE_COLUMN_RANGE_REF.exec(expr))) {
                        // table column range, e.g. [Column1]:[Column2] in Table1[[#All],[Column1]:[Column2]]
                        handleColRef();
                    } else if (!this.grammar.UI && (matches = RE.TABLE_COLUMN_REF.exec(expr))) {
                        // table column name in brackets, e.g. [Column1] in Table1[[#All],[Column1]]
                        handleColRef();
                    } else if (this.grammar.UI && (matches = RE.TABLE_MIXED_COLUMN_RANGE_REF.exec(expr))) {
                        // table column range with or without brackets, e.g. Column1:[Column 2] in Table1[[#All],Column1:[Column 2]]
                        handleMixedColRef();
                    } else if (this.grammar.UI && (matches = RE.TABLE_MIXED_COLUMN_REF.exec(expr))) {
                        // table column name with or without brackets, e.g. [Column1] in Table1[[#All],Column1]
                        handleMixedColRef();
                    }
                    break;

                case 'row':
                    if ((matches = RE.TABLE_CLOSE.exec(expr))) {
                        // simple row reference, e.g. Table1[@]
                        handleTableClose();
                        // compatibility: do not accept whitespace between at-sign and closing bracket
                        if (tableOptions.closeWs) { matches = null; }
                    } else if ((matches = RE.WHITESPACE.exec(expr))) {
                        // ignore any whitespace following the at-sign
                    } else if ((matches = RE.TABLE_MIXED_COLUMN_RANGE_REF.exec(expr))) {
                        // table column range, e.g. Column1:[Column 2] in Table1[@Column1:[Column 2]]
                        handleMixedColRef();
                    } else if ((matches = RE.TABLE_MIXED_COLUMN_REF.exec(expr))) {
                        // table column name, e.g. Column1 in Table1[@Column1]
                        handleMixedColRef();
                    }
                    break;
            }

            // immediately return on error; otherwise reduce the formula subexpression
            if (!matches) { return null; }
            expr = expr.substr(matches[0].length);
        }

        // build a valid region key
        regionKeys = _.keys(regionKeys);
        switch (regionKeys.length) {
            case 0:
                break;
            case 1:
                tableOptions.regionKey = regionKeys[0];
                break;
            case 2:
                regionKeys = regionKeys.join(',');
                if ((regionKeys === 'HEADERS,DATA') || (regionKeys === 'DATA,HEADERS')) {
                    tableOptions.regionKey = 'HEADERS,DATA';
                } else if ((regionKeys === 'DATA,TOTALS') || (regionKeys === 'TOTALS,DATA')) {
                    tableOptions.regionKey = 'DATA,TOTALS';
                } else {
                    return null;
                }
                break;
            default:
                return null;
        }

        // create a new table token, and insert the entire parsed text of the formula expression into the token descriptor
        var text = origExpr.substr(0, origExpr.length - expr.length);
        return this.createToken(new Tokens.TableToken(this.docModel, tableModel.getName(), tableOptions), text);
    };

    // virtual methods --------------------------------------------------------

    /**
     * Extracts a reference token from the passed formula subexpression.
     */
    ExpressionParserOOX.prototype.parseNextRefToken = function (expr) {

        // the collection of regular expressions
        var RE = this.refGrammar.RE;
        // the matches of the regular expressions
        var matches = null;
        // cell references built from the matches of a regular expression
        var cellRefs = null;

        // local cell range reference (before cell references!), e.g. $A$1:$C$3
        if ((matches = RE.RANGE_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 1, 5))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // cell range reference with sheet (before cell references!), e.g. Sheet1!$A$1:$C$3
        if ((matches = RE.RANGE_3D_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 4, 8))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // cell range reference in multiple sheets (before cell references!), e.g. Sheet1:Sheet3!$A$1:$C$3
        if ((matches = RE.RANGE_CUBE_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 5, 9))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), cellRefs, matches[0]);
        }

        // local cell reference, e.g. $A$1
        if ((matches = RE.CELL_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // cell reference with sheet, e.g. Sheet1!$A$1
        if ((matches = RE.CELL_3D_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 4))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // cell reference in multiple sheets, e.g. Sheet1:Sheet3!$A$1
        if ((matches = RE.CELL_CUBE_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 5))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), cellRefs, matches[0]);
        }

        // local column range reference, e.g. $A:$C
        if ((matches = RE.COLS_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // column range reference with sheet, e.g. Sheet1!$A:$C
        if ((matches = RE.COLS_3D_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 4, 6))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // column range reference in multiple sheets, e.g. Sheet1:Sheet3!$A:$C
        if ((matches = RE.COLS_CUBE_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 5, 7))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), cellRefs, matches[0]);
        }

        // local column reference, e.g. C[1]
        if ((matches = RE.COL_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // column reference with sheet, e.g. Sheet1!C[1]
        if ((matches = RE.COL_3D_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 4))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // column reference in multiple sheets, e.g. Sheet1:Sheet3!C[1]
        if ((matches = RE.COL_CUBE_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 5))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), cellRefs, matches[0]);
        }

        // local row range reference, e.g. $1:$3
        if ((matches = RE.ROWS_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // row range reference with sheet, e.g. Sheet1!$1:$3
        if ((matches = RE.ROWS_3D_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 4, 6))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // row range reference in multiple sheets, e.g. Sheet1:Sheet3!$1:$3
        if ((matches = RE.ROWS_CUBE_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 5, 7))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), cellRefs, matches[0]);
        }

        // local row reference, e.g. R[1]
        if ((matches = RE.ROW_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // row reference with sheet, e.g. Sheet1!R[1]
        if ((matches = RE.ROW_3D_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 4))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // row reference in multiple sheets, e.g. Sheet1:Sheet3!R[1]
        if ((matches = RE.ROW_CUBE_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 5))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), cellRefs, matches[0]);
        }

        // reference error with sheet, e.g. Sheet1!#REF!
        if ((matches = RE.ERROR_3D_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), null, matches[0]);
        }

        // reference error in multiple sheets, e.g. Sheet1:Sheet3!#REF!
        if ((matches = RE.ERROR_CUBE_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, true), null, matches[0]);
        }
    };

    /**
     * Extracts a name or function token from the passed formula subexpression.
     */
    ExpressionParserOOX.prototype.parseNextNameToken = function (expr) {

        // the collection of regular expressions
        var RE = this.RE;
        var RE2 = this.refGrammar.RE;
        // the matches of the regular expressions
        var matches = null;

        // local function call, or global macro call, e.g. SUM() or my_macro()
        if ((matches = RE2.FUNCTION_REF.exec(expr))) {
            return this.createFunctionToken(null, matches[1], matches[0]);
        }

        // macro call with sheet, e.g. Module1!my_macro()
        if ((matches = RE2.FUNCTION_3D_REF.exec(expr))) {
            return this.createFunctionToken(this.createSheetRef(matches, 1), matches[4], matches[0]);
        }

        // defined name with sheet, e.g. Sheet1!my_name
        if ((matches = RE2.NAME_3D_REF.exec(expr))) {
            return this._parseTableRef(expr, this.createNameToken(this.createSheetRef(matches, 1), matches[4], matches[0]));
        }

        // defined name without sheet reference
        if ((matches = RE2.NAME_REF.exec(expr))) {
            return this._parseTableRef(expr, this.createNameToken(null, matches[1], matches[0]));
        }

        // opening bracket of an unqualified structured table reference (UI grammars only)
        if (this.grammar.UI && (matches = RE.TABLE_OPEN.exec(expr))) {
            return this._parseTableRef(expr, null);
        }
    };

    // class ExpressionParserODFUI ============================================

    /**
     * A formula expression parser for the ODF UI reference syntax.
     *
     * @constructor
     *
     * @extends ExpressionParserBase
     */
    var ExpressionParserODFUI = ExpressionParserBase.extend(function (docModel, formulaGrammar) {

        // base constructor
        ExpressionParserBase.call(this, docModel, formulaGrammar);

    }); // class ExpressionParserODFUI

    // protected methods ------------------------------------------------------

    /**
     * Returns an object with two SheetRef instances for the passed RE matches
     * of a sheet range.
     */
    ExpressionParserODFUI.prototype.createSheetRangeData = function (matches, index) {
        var sheet1Ref = SheetRef.create(this.docModel, matches[index + 1], matches[index + 2], null, !!matches[index]);
        var sheet2Ref = SheetRef.create(this.docModel, matches[index + 4], matches[index + 5], null, !!matches[index + 3]);
        return { r1: sheet1Ref, r2: sheet2Ref };
    };

    // virtual methods --------------------------------------------------------

    /**
     * Extracts a reference token from the passed formula subexpression.
     */
    ExpressionParserODFUI.prototype.parseNextRefToken = function (expr) {

        // the collection of regular expressions
        var RE = this.refGrammar.RE;
        // the matches of the regular expressions
        var matches = null;
        // cell references built from the matches of a regular expression
        var cellRefs = null;

        // local cell range reference (before cell references!), e.g. $A$1:$C$3
        if ((matches = RE.RANGE_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 1, 5))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // cell range reference with sheet (before cell references!), e.g. Sheet1!$A$1:$C$3
        if ((matches = RE.RANGE_3D_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 5, 9))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // cell range reference in multiple sheets (before cell references!), e.g. Sheet1:Sheet3!$A$1:$C$3
        if ((matches = RE.RANGE_CUBE_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 7, 11))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), cellRefs, matches[0]);
        }

        // local cell reference, e.g. $A$1
        if ((matches = RE.CELL_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // cell reference with sheet, e.g. Sheet1!$A$1
        if ((matches = RE.CELL_3D_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 5))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // cell reference in multiple sheets, e.g. Sheet1:Sheet3!$A$1
        if ((matches = RE.CELL_CUBE_REF.exec(expr)) && (cellRefs = this.createCellRangeRefs(matches, 7))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), cellRefs, matches[0]);
        }

        // local column range reference, e.g. $A:$C
        if ((matches = RE.COLS_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // column range reference with sheet, e.g. Sheet1!$A:$C
        if ((matches = RE.COLS_3D_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 5, 7))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // column range reference in multiple sheets, e.g. Sheet1:Sheet3!$A:$C
        if ((matches = RE.COLS_CUBE_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 7, 9))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), cellRefs, matches[0]);
        }

        // local column reference, e.g. C[1]
        if ((matches = RE.COL_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // column reference with sheet, e.g. Sheet1!C[1]
        if ((matches = RE.COL_3D_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 5))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // column reference in multiple sheets, e.g. Sheet1:Sheet3!C[1]
        if ((matches = RE.COL_CUBE_REF.exec(expr)) && (cellRefs = this.createColRangeRefs(matches, 7))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), cellRefs, matches[0]);
        }

        // local row range reference, e.g. $1:$3
        if ((matches = RE.ROWS_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 1, 3))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // row range reference with sheet, e.g. Sheet1!$1:$3
        if ((matches = RE.ROWS_3D_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 5, 7))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // row range reference in multiple sheets, e.g. Sheet1:Sheet3!$1:$3
        if ((matches = RE.ROWS_CUBE_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 7, 9))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), cellRefs, matches[0]);
        }

        // local row reference, e.g. R[1]
        if ((matches = RE.ROW_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 1))) {
            return this.createReferenceToken(null, cellRefs, matches[0]);
        }

        // row reference with sheet, e.g. Sheet1!R[1]
        if ((matches = RE.ROW_3D_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 5))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), cellRefs, matches[0]);
        }

        // row reference in multiple sheets, e.g. Sheet1:Sheet3!R[1]
        if ((matches = RE.ROW_CUBE_REF.exec(expr)) && (cellRefs = this.createRowRangeRefs(matches, 7))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), cellRefs, matches[0]);
        }

        // reference error with sheet, e.g. Sheet1!#REF!
        if ((matches = RE.ERROR_3D_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), null, matches[0]);
        }

        // reference error in multiple sheets, e.g. Sheet1:Sheet3!#REF!
        if ((matches = RE.ERROR_CUBE_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRangeData(matches, 1), null, matches[0]);
        }
    };

    /**
     * Extracts a name or function token from the passed formula subexpression.
     */
    ExpressionParserODFUI.prototype.parseNextNameToken = function (expr) {

        // the collection of regular expressions
        var RE = this.refGrammar.RE;
        // the matches of the regular expressions
        var matches = null;

        // local function call, or global macro call, e.g. SUM() or my_macro()
        if ((matches = RE.FUNCTION_REF.exec(expr))) {
            return this.createFunctionToken(null, matches[1], matches[0]);
        }

        // defined name without sheet reference, or table reference
        if ((matches = RE.NAME_REF.exec(expr))) {
            return this.createNameToken(null, matches[1], matches[0], { resolveTable: true });
        }
    };

    // class ExpressionParserOF ===============================================

    /**
     * A formula expression parser for the OpenFormula reference syntax.
     *
     * @constructor
     *
     * @extends ExpressionParserBase
     */
    var ExpressionParserOF = ExpressionParserBase.extend(function (docModel, formulaGrammar) {

        // base constructor
        ExpressionParserBase.call(this, docModel, formulaGrammar);

    }); // class ExpressionParserOF

    // virtual methods --------------------------------------------------------

    /**
     * Extracts a reference token from the passed formula subexpression.
     */
    ExpressionParserOF.prototype.parseNextRefToken = function (expr) {

        // the collection of regular expressions
        var RE = this.refGrammar.RE;
        // the matches of the regular expressions
        var matches = null;

        // local cell reference, e.g. [.$A$1]
        if ((matches = RE.CELL_REF.exec(expr))) {
            return this.createReferenceToken(null, this.createCellRangeRefs(matches, 1), matches[0]);
        }

        // cell reference with sheet, e.g. [$Sheet1.$A$1]
        if ((matches = RE.CELL_3D_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), this.createCellRangeRefs(matches, 5), matches[0]);
        }

        // local cell range reference, e.g. [.$A$1:.$C$3]
        if ((matches = RE.RANGE_REF.exec(expr))) {
            return this.createReferenceToken(null, this.createCellRangeRefs(matches, 1, 5), matches[0]);
        }

        // cell range reference with sheet, e.g. [$Sheet1.$A$1:.$C$3]
        if ((matches = RE.RANGE_3D_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1), this.createCellRangeRefs(matches, 5, 9), matches[0]);
        }

        // cell range reference in multiple sheets, e.g. [$Sheet1.$A$1:$Sheet3.$C$3]
        if ((matches = RE.RANGE_CUBE_REF.exec(expr))) {
            return this.createReferenceToken(this.createSheetRefs(matches, 1, 9), this.createCellRangeRefs(matches, 5, 13), matches[0]);
        }
    };

    /**
     * Extracts a name or function token from the passed formula subexpression.
     */
    ExpressionParserOF.prototype.parseNextNameToken = function (expr) {

        // the collection of regular expressions
        var RE = this.refGrammar.RE;
        // the matches of the regular expressions
        var matches = null;

        // local function call, or global macro call, with optional whitespace, e.g. SUM () or my_macro()
        if ((matches = RE.FUNCTION_REF.exec(expr))) {
            return this.createFunctionToken(null, matches[1], matches[0]);
        }

        // defined name without sheet reference
        if ((matches = RE.NAME_REF.exec(expr))) {
            return this.createNameToken(null, matches[1], matches[0], { resolveTable: true });
        }
    };

    // static class ExpressionParser ==========================================

    var ExpressionParser = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates a new expression parser instance for the specified spreadsheet
     * document and formula grammar.
     *
     * @param {SpreadsheetModel} docModel
     *  The document model to be associated with the new expression parser.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar to be used to parse the formula
     *  expressions. See SpreadsheetDocument.getFormulaGrammar() for more
     *  details.
     *
     * @returns {ExpressionParserBase}
     *  A new expression parser for the specified document and formula grammar.
     */
    ExpressionParser.create = function (docModel, grammarId) {
        var formulaGrammar = docModel.getFormulaGrammar(grammarId);
        switch (formulaGrammar.REF_SYNTAX) {
            case 'ooxml':
                return new ExpressionParserOOX(docModel, formulaGrammar);
            case 'odfui':
                return new ExpressionParserODFUI(docModel, formulaGrammar);
            case 'of':
                return new ExpressionParserOF(docModel, formulaGrammar);
        }
        Utils.error('ExpressionParser.create(): invalid reference syntax: "' + formulaGrammar.REF_SYNTAX + '"');
    };

    // exports ================================================================

    return ExpressionParser;

});
