/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/interpret/filteraggregator', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Iterator, SheetUtils, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;

    // class FilterAggregator =================================================

    /**
     * An instance of this class implements numeric aggregation of function
     * parameters based on some filter criteria.
     *
     * @constructor
     *
     * @param {FormulaContext} formulaContext
     *  The formula context providing the parameters of a function used for
     *  filtering and aggregation.
     *
     * @param {Number} filterIndex
     *  The zero-based index of the first function operand contained in the
     *  passed formula context representing a filter criterion.
     */
    function FilterAggregator(formulaContext, filterIndex) {

        // reference to the current formula context
        this._context = formulaContext;
        // whether to return a matrix result
        this._matMode = formulaContext.isMatrixMode();
        // dimension of the matrixes used by this instance
        this._matDim = formulaContext.getMatrixDim() || new Dimension(1, 1);
        // the number of filter ranges
        this._rangeCount = Math.floor((formulaContext.getOperandCount() - filterIndex) / 2);
        // the filter ranges
        this._filterRanges = [];
        // filter matcher matrixes (one matrix per filter range)
        this._filterMatchers = [];
        // fall-back matcher for missing elements in filter matcher matrixes (matches #N/A error code only)
        this._fallbackMatcher = formulaContext.createFilterMatcher(ErrorCode.NA);

        for (var i = 0; i < this._rangeCount; i += 1, filterIndex += 2) {

            // extract the filter range
            var filterRange = formulaContext.getOperand(filterIndex, 'ref:val');
            this._filterRanges.push(filterRange);
            // check that all filter ranges have equal size (throw a #VALUE! error if not)
            formulaContext.checkRangeDims(this._filterRanges[0], filterRange);

            // extract the filter criterion (or a matrix with filter criteria in matrix context)
            if (this._matDim) {
                var filterMatchers = formulaContext.getOperand(filterIndex + 1, 'mat:any').map(formulaContext.createFilterMatcher, formulaContext);
                this._filterMatchers.push(filterMatchers);
            } else {
                var filterMatcher = formulaContext.createFilterMatcher(formulaContext.getOperand(filterIndex + 1, 'val:any'));
                // use a criteria matrix in value context for simpler implementation
                this._filterMatchers.push(Matrix.create(1, 1, filterMatcher));
            }
        }

    } // class FilterAggregator

    // private methods --------------------------------------------------------

    /**
     * Creates a new matrix with the matrix result dimension, filled with the
     * specified scalar value.
     *
     * @param {Any} value
     *  The scalar value to be filled into all elements of matrix.
     *
     * @returns {Matrix}
     *  The new matrix instance.
     */
    FilterAggregator.prototype._createMatrix = function (value) {
        return Matrix.create(this._matDim.rows, this._matDim.cols, value);
    };

    /**
     * Returns whether the passed value matches the criterion at the specified
     * matrix position.
     *
     * @param {Matrix<Function>} matchers
     *  A matrix with filter matcher callback functions as elements.
     *
     * @param {Number} row
     *  The row index in the filter matcher matrix.
     *
     * @param {Number} col
     *  The column index in the filter matcher matrix.
     *
     * @param {Any} value
     *  The scalar value to be matched against the filter criterion stored at
     *  the specified matrix position.
     *
     * @returns {Boolean}
     *  Whether the passed value matches the filter criterion.
     */
    FilterAggregator.prototype._matchValue = function (matchers, row, col, value) {
        var matcher = matchers.get(row, col);
        return (typeof matcher === 'function') ? matcher(value) : this._fallbackMatcher(value);
    };

    /**
     * Returns whether the passed value matches all criteria at the specified
     * matrix position.
     *
     * @param {Number} row
     *  The row index in the criteria matrixes.
     *
     * @param {Number} col
     *  The column index in the criteria matrixes.
     *
     * @param {Any} value
     *  The scalar value to be matched against all filter criteria stored at
     *  the specified matrix position.
     *
     * @returns {Boolean}
     *  Whether the passed value matches all filter criteria.
     */
    FilterAggregator.prototype._testValue = function (row, col, value) {
        return this._filterMatchers.every(function (matchers) {
            return this._matchValue(matchers, row, col, value);
        }, this);
    };

    /**
     * Returns whether the passed values match the respective criteria at the
     * specified matrix position.
     *
     * @param {Number} row
     *  The row index in the criteria matrixes.
     *
     * @param {Number} col
     *  The column index in the criteria matrixes.
     *
     * @param {Array<Any>} values
     *  The scalar values to be matched against the respective filter criteria
     *  stored at the specified matrix position. MUST have the same number of
     *  elements as filter ranges exist.
     *
     * @returns {Boolean}
     *  Whether the passed values match all filter criteria.
     */
    FilterAggregator.prototype._testValues = function (row, col, values) {
        return this._filterMatchers.every(function (matchers, index) {
            return this._matchValue(matchers, row, col, values[index]);
        }, this);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the number of filter ranges extracted from the operands passed
     * to the constructor.
     *
     * @returns {Number}
     *  The number of filter ranges.
     */
    FilterAggregator.prototype.getRangeCount = function () {
        return this._rangeCount;
    };

    /**
     * Returns the specified filter range extracted from the operands passed to
     * the constructor.
     *
     * @param {Number} index
     *  The array index of the filter range. MUST be a valid array index.
     *
     * @returns {Range3D}
     *  The specified filter range.
     */
    FilterAggregator.prototype.getFilterRange = function (index) {
        return this._filterRanges[index];
    };

    /**
     * Returns the number of matching cells in the filter ranges according to
     * the filter criteria.
     *
     * @returns {Number|Matrix}
     *  The number of matching cells in the filter ranges according to the
     *  filter criteria; or a matrix with the counts of all matching cells, if
     *  the formula context is in matrix formula mode.
     */
    FilterAggregator.prototype.countMatches = function () {

        // create a scalar iterator for each filter range
        var iterators = this._filterRanges.map(function (filterRange) {
            return this.createScalarIterator(filterRange, {
                acceptErrors: true, // error codes can be matched by the filter criterion
                complexRef: false // do not accept multi-range and multi-sheet references
            });
        }, this._context);

        // create a parallel iterator that visits all entries with at least one non-blank value
        var iterator = this._context.createParallelIteratorSome(iterators);

        // the number of iterator stops containing at least one non-blank cell
        var valueCount = 0;
        // the number of iterator stops matching the filter criteria
        var matchCounts = this._createMatrix(0);

        // visit all entries in the filter ranges with at least one non-blank cell
        Iterator.forEach(iterator, function (values) {

            // count all iterator stops, needed to calculate the completely blank entries
            valueCount += 1;

            // count the iterator stops that match the filter criteria
            matchCounts.transform(function (matchCount, row, col) {
                return matchCount + (this._testValues(row, col, values) ? 1 : 0);
            }, this);
        }, this);

        // Check whether completely blank entries will be counted too. This is the case if all
        // filter criteria will accept blank cells. For performance, the scalar iterators created
        // above will skip blank cells in its own filter range. Therefore, if the cells at a
        // specific offset are blank in all filter ranges, that offset will be skipped by the
        // parallel iterator. The number of completely blank entries that need to be counted is
        // the difference between number of cells in a filter range (all filter ranges have equal
        // size) and the total number of entries visited by the parallel iterator.
        var blankCount = this.getFilterRange(0).cells() - valueCount;
        matchCounts.transform(function (matchCount, row, col) {
            return matchCount + (this._testValue(row, col, null) ? blankCount : 0);
        }, this);

        // return the matched counts (convert 1x1 matrix back to scalar if not in matrix context)
        return this._matMode ? matchCounts : matchCounts.get(0, 0);
    };

    /**
     * Aggregates the numbers in the passed data range according to the filter
     * ranges and filter criteria of this instance.
     *
     * @param {Range3D} dataRange
     *  The address of the cell range whose numbers will be aggregated.
     *
     * @param {Any} initial
     *  The initial intermediate result, passed to the first invocation of the
     *  aggregation callback function.
     *
     * @param {Function} aggregate
     *  The aggregation callback function. Receives the following parameters:
     *  (1) {Any} result
     *      The current intermediate result. On first invocation, this is the
     *      value passed in the 'initial' parameter.
     *  (2) {Number} number
     *      The new number to be combined with the intermediate result.
     *  (3) {Number} index
     *      The zero-based sequence index of the number (a counter for the
     *      numbers that ignores all other skipped values in the operands).
     *  (4) {Number} offset
     *      The relative position of the number in all operands (including all
     *      skipped empty parameters, and blank cells).
     *  Must return the new intermediate result which will be passed to the
     *  next invocation of this callback function, or may throw an error code.
     *  Will be called with the formula context as calling context.
     *
     * @param {Function} finalize
     *  A callback function invoked at the end to convert the intermediate
     *  result (returned by the last invocation of the aggregation callback
     *  function) to the actual function result. Receives the following
     *  parameters:
     *  (1) {Any} result
     *      The last intermediate result to be converted to the result of the
     *      implemented function.
     *  (2) {Number} count
     *      The total count of all visited numbers.
     *  (3) {Number} size
     *      The total size of all operands, including all skipped values.
     *  Must return the final result of the aggregation. Alternatively, the
     *  finalizer may throw an error code. Will be called with the formula
     *  context as calling context.
     *
     * @returns {Number|Matrix}
     *  The aggregated function result; or a matrix with aggregated function
     *  results, if the formula context is in matrix formula mode.
     */
    FilterAggregator.prototype.aggregateNumbers = function (dataRange, initial, aggregate, finalize) {

        // the data range must have the same size as the filter ranges
        this._context.checkRangeDims(dataRange, this.getFilterRange(0));

        // shortcut to this instance to be used in the callback functions
        var aggregator = this;
        // the initial value for aggregation (number or matrix)
        var initialValues = this._createMatrix(initial);
        // the number of iterator stops matching the filter criteria
        var matchCounts = this._createMatrix(0);

        // aggregator that processes only the numbers that match the filter criteria
        function aggregateImpl(resultValues, number, index, offset) {
            // inside this callback, 'this' is the FormulaContext instance

            // pluck the cell values from all filter ranges
            var filterValues = aggregator._filterRanges.map(function (filterRange) {
                return this.getCellValue(filterRange.sheet1, filterRange.addressAt(offset));
            }, this);

            // call the aggregate() callback function for all matching result values in the matrix
            return resultValues.transform(function (resultValue, row, col) {
                if (aggregator._testValues(row, col, filterValues)) {
                    matchCounts.add(row, col, 1);
                    return aggregate.call(this, resultValue, number);
                }
                return resultValue;
            }, this);
        }

        // finalizer that passed the actual match count to the external finalizer
        function finalizeImpl(resultValues) {
            // inside this callback, 'this' is the FormulaContext instance

            // call the finalize() callback function for all matching result values in the matrix
            if (finalize) {
                resultValues.transform(function (resultValue, row, col) {
                    var matchCount = matchCounts.get(row, col);
                    return finalize.call(this, resultValue, matchCount);
                }, this);
            }

            // return the result value (convert 1x1 matrix back to scalar if not in matrix context)
            return this._matMode ? resultValues : resultValues.get(0, 0);
        }

        // aggregate all numbers of the data range that match the criteria
        return this._context.aggregateNumbers([dataRange], initialValues, aggregateImpl, finalizeImpl, {
            refMode: 'skip', // skip all strings and boolean values
            complexRef: false // do not accept multi-range and multi-sheet references
        });
    };

    // exports ================================================================

    return FilterAggregator;

});
