/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/dependencymanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/celloperationsbuilder',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils',
    'io.ox/office/spreadsheet/model/formula/deps/timerhelper',
    'io.ox/office/spreadsheet/model/formula/deps/formuladescriptor',
    'io.ox/office/spreadsheet/model/formula/deps/sheetdescriptor',
    'io.ox/office/spreadsheet/model/formula/deps/formuladictionary',
    'io.ox/office/spreadsheet/model/formula/deps/pendingcollection'
], function (Utils, Iterator, Scheduler, ValueSet, ValueMap, TriggerObject, Config, SheetUtils, CellOperationsBuilder, FormulaUtils, DependencyUtils, TimerHelper, FormulaDescriptor, SheetDescriptor, FormulaDictionary, PendingCollection) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;
    var Scalar = FormulaUtils.Scalar;
    var AddressSet = DependencyUtils.AddressSet;
    var FormulaSet = DependencyUtils.FormulaSet;
    var getCellKey = DependencyUtils.getCellKey;

    // class ResultSet ========================================================

    /**
     * A set of cached formula results, mapped by cell address key.
     *
     * @constructor
     *
     * @extends ValueSet
     */
    var ResultSet = ValueSet.extend(function () {
        ValueSet.call(this, 'key');
    }); // class ResultSet

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new formula result into this set.
     *
     * @param {Address} address
     *  The cell address of the formula result.
     *
     * @param {Any} value
     *  The scalar formula result.
     *
     * @param {Number} time
     *  The timestamp when the formula has been calculated.
     *
     * @returns {Object}
     *  The result descriptor object inserted into the set.
     */
    ResultSet.prototype.insert = function (address, value, time) {
        return ValueSet.prototype.insert.call(this, {
            address: address,
            key: address.key(),
            value: Scalar.getCellValue(value),
            time: time
        });
    };

    // class ResultCollection =================================================

    /**
     * Encapsulates all data that will be collected during a single update
     * cycle of a dependency manager.
     *
     * @constructor
     *
     * @property {FormulaSet} formulaSet
     *  All dirty formula descriptors collected by an update cycle of a
     *  dependency manager. This set contains all formula descriptors contained
     *  in the properties 'cellFormulaSet' and 'modelFormulasMap'.
     *
     * @property {FormulaSet} cellFormulaSet
     *  The descriptors of all dirty cell formulas collected by an update cycle
     *  of a dependency manager. All formulas in this map will be calculated
     *  while the calculation of another formula looks up the cell contents of
     *  that dirty formulas. After calculation of a formula has finished, the
     *  formula result will be stored in the property 'resultsMap', and will be
     *  returned when looking up the cell value the next time.
     *
     * @property {ValueMap<FormulaSet>} modelFormulasMap
     *  The descriptors of all dirty formulas contained in other model objects
     *  collected by an update cycle of a dependency manager, mapped as submaps
     *  by sheet UID.
     *
     * @property {ValueMap<ResultSet>} resultsMap
     *  The results of all dirty formulas (normal formulas, shared formulas,
     *  and matrix formulas) that have been calculated, as sets of objects
     *  mapped by sheet UID. Each descriptor object in the inner set contains
     *  the cell address (property 'address'), the scalar result value
     *  (property 'value'), and a timestamp (property 'time') used to detect
     *  timeouts during formula calculation.
     *
     * @property {Array<AddressArray>} referenceCycles
     *  All formula cycles found during calculation of formulas. Each element
     *  in the array is a cell address array representing the reference cycle.
     *
     * @property {Boolean} hasErrors
     *  Whether any of the formulas could not be calculated correctly, e.g. due
     *  to unimplemented functions.
     */
    function ResultCollection() {

        this.formulaSet = new FormulaSet();
        this.cellFormulaSet = new FormulaSet();
        this.modelFormulasMap = new ValueMap();
        this.resultsMap = new ValueMap();
        this.referenceCycles = [];
        this.hasErrors = false;

    } // class ResultCollection

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new formula descriptor into the internal collections of this
     * instance.
     *
     * @param {FormulaDescriptor} formulaDesc
     *  The descriptor for the new formula to be registered.
     *
     * @returns {ResultCollection}
     *  A reference to this instance.
     */
    ResultCollection.prototype.insertFormula = function (formulaDesc) {
        this.formulaSet.insert(formulaDesc);
        if (formulaDesc.tokenModel) {
            this.modelFormulasMap.getOrConstruct(formulaDesc.sheetUid, FormulaSet).insert(formulaDesc);
        } else if (formulaDesc.cellAddress) {
            this.cellFormulaSet.insert(formulaDesc);
        } else {
            Utils.error('ResultCollection.insertFormula(): unknown formula type');
        }
        return this;
    };

    /**
     * Returns an existing result set, or creates a new result set on demand.
     *
     * @param {String} sheetUid
     *  The UID of the sheet containing the formula results.
     *
     * @returns {ResultSet}
     *  The result set for the specified sheet.
     */
    ResultCollection.prototype.createResultSet = function (sheetUid) {
        return this.resultsMap.getOrConstruct(sheetUid, ResultSet);
    };

    /**
     * Returns an existing result for the specified cell.
     *
     * @param {String} sheetUid
     *  The UID of the sheet containing the formula result.
     *
     * @param {Address} address
     *  The address of the cell.
     *
     * @returns {Object|Null}
     *  An existing result descriptor, otherwise null.
     */
    ResultCollection.prototype.getResult = function (sheetUid, address) {
        return this.resultsMap.with(sheetUid, function (resultSet) {
            return resultSet.get(address.key(), null);
        }) || null;
    };

    // class DependencyManager ================================================

    /**
     * Collects all change events in the document, and recalculates all formula
     * expressions depending on these changes.
     *
     * Triggers the following events:
     * - 'recalc:start'
     *      When the background task for calculating the dependent formulas has
     *      started. Will be followed by either one 'recalc:end' event, or by
     *      one 'recalc:cancel' event.
     * - 'recalc:end'
     *      When the background task for calculating the dependent formulas has
     *      been finished successfully. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event}
     *          The jQuery event object.
     *      (2) {Array<AddressArray>} referenceCycles
     *          An array of arrays of cell addresses. Each address contains the
     *          sheet index in the property 'sheet', and the UID of the sheet
     *          model in the property 'sheetUid'. Each of the inner arrays
     *          represents a reference cycle found during calculation of dirty
     *          formulas.
     * - 'recalc:cancel'
     *      When the background task for calculating the dependent formulas has
     *      been aborted due to new document changes.
     * - 'recalc:overflow'
     *      Will be triggered once, if the total number of formulas in the
     *      document exceeds a specific limit in the server configuration.
     *      After that, the dependency manager will not process any formulas in
     *      the document anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetModel} docModel
     *  The spreadsheet document model containing this instance.
     */
    var DependencyManager = TriggerObject.extend({ constructor: function (docModel) {

        // private properties -------------------------------------------------

        // self reference
        var self = this;

        // the application containing this dependency manager
        var app = docModel.getApp();

        // A helper instance to execute background timer loops with different
        // priorities (interval durations).
        //
        var timerHelper = new TimerHelper(docModel);

        // Stores the descriptors of all registered formulas in the document.
        //
        // The formula descriptors (instances of the class FormulaDescriptor)
        // will be mapped by their formula keys.
        //
        var formulaSet = new FormulaSet();

        // The total number of all registered formulas currently contained in
        // the spreadsheet document.
        //
        var formulaCount = 0;

        // Stores a sheet descriptor object per existing sheet in the document.
        //
        // The sheet descriptors (instances of the class SheetDescriptor) will
        // be mapped by the UID of the sheet model.
        //
        var sheetDescMap = new ValueMap();

        // A dictionary for all formula descriptors of this dependency manager.
        //
        // This is the central storage of this dependency manager that will be
        // used to find formulas depending on specific dirty cells, or other
        // changed document contents, in a very efficient way.
        //
        var formulaDictionary = new FormulaDictionary(docModel);

        // Collected data for all document change events, and other information
        // needed to decide which formulas need to be recalculated in the next
        // update cycle.
        //
        var pendingCollection = new PendingCollection();

        // All dirty formula descriptors collected by an update cycle of this
        // dependency manager, and the formula results already calculated.
        //
        var resultCollection = null;

        // Whether this dependency manager did not finish its initial update
        // cycle after importing the document yet.
        //
        // Will be set to false, after the first update cycle has finished
        // successfully.
        //
        var initialUpdateCycle = true;

        // Specifies whether the initial update cycle needs to recalculate all
        // formulas in the document.
        //
        // This flag will be initialized after importing the document, and will
        // be reset after the first successful update cycle.
        //
        var calcOnLoad = false;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Creates a new operations generator, and invokes the passed callback
         * function. If the spreadsheet document is in edit mode, the generated
         * operations will be sent to the server.
         *
         * @param {Function} callback
         *  The callback function. Receives the operations generator as first
         *  parameter.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         */
        function generateAndSendOperations(callback, context) {

            // create a new generator that applies the generated operations immediately
            var generator = docModel.createSheetOperationGenerator({ applyImmediately: true });

            // invoke the callback function with the new generator
            callback.call(context, generator);

            // send the operations to the server, but only if the document is in edit mode
            if (app.isEditable()) {
                docModel.sendOperations(generator, { defer: !app.isLocallyModified() });
            }
        }

        /**
         * Sends the 'calcOnLoad' document flag to the server, if it has not
         * been set yet.
         */
        function setCalcOnLoadFlag() {

            // nothing to do, if 'calculate on load' mode is already activated
            // -> not generating operation in long running processes (recursive call), 53641
            if (calcOnLoad || docModel.isProcessingActions()) { return; }

            // set the 'calcOnLoad' flag via document operation
            generateAndSendOperations(function (generator) {
                generator.generateDocAttrsOperation({ calcOnLoad: true });
            });
            calcOnLoad = true;
        }

        /**
         * Checks the current number of formulas. If the maximum allowed number
         * of formulas (according to server configuration) has been exceeded,
         * this dependency manager will be detached from all document model
         * events (this prevents processing any further edit actions from the
         * document), a "recalc:overflow" event will be triggered (this will
         * abort the update cycle currently running), and  the "calcOnLoad"
         * document flag will be set.
         */
        var checkFormulaCount = (Config.MAX_FORMULA_COUNT > 0) ? function () {

            // formula count still valid: return true
            if (formulaCount <= Config.MAX_FORMULA_COUNT) { return true; }

            // detach and deactivate dependency manager
            self.disconnect();
            self.trigger('recalc:overflow');
            setCalcOnLoadFlag();
            return false;

        } : _.constant(true);

        /**
         * Inserts a new formula descriptor into all collections of this
         * dependency manager.
         *
         * @param {FormulaDescriptor} formulaDesc
         *  The descriptor for the new formula to be registered.
         *
         * @returns {Boolean}
         *  Whether the formula descriptor has been inserted into the internal
         *  formula collections. If the return value is false, this method has
         *  triggered a 'recalc:overflow' event, and has stopped the entire
         *  recalculation cycle.
         */
        function insertFormula(formulaDesc) {

            // update the total number of formulas
            formulaCount += 1;
            if (!checkFormulaCount()) { return false; }

            // register the formula descriptor in the global map
            formulaSet.insert(formulaDesc);

            // insert the formula descriptor into the sheet descriptor maps
            sheetDescMap.get(formulaDesc.sheetUid).insertFormula(formulaDesc);

            // insert the formula descriptor into the dictionary
            formulaDictionary.insertFormula(formulaDesc);

            // formulas has been registered successfully
            return true;
        }

        /**
         * Removes a formula descriptor from all collections of this dependency
         * manager.
         *
         * @param {FormulaDesc} formulaDesc
         *  The descriptor of the formula to be removed.
         *
         * @returns {Boolean}
         *  Whether the formula descriptor has been removed from the internal
         *  formula collections.
         */
        function removeFormula(formulaDesc) {

            // remove the formula descriptor from the global map
            if (!formulaSet.remove(formulaDesc)) { return false; }

            // update the total number of formulas in this dependency manager, if the formula exists
            formulaCount -= 1;

            // remove the formula descriptor from the sheet descriptor (check existence, sheet
            // descriptor has been removed already while deleting a sheet in the document)
            sheetDescMap.with(formulaDesc.sheetUid, function (sheetDesc) {
                sheetDesc.removeFormula(formulaDesc);
            });

            // remove the formula descriptor from the dictionary
            formulaDictionary.removeFormula(formulaDesc);

            // formulas has been unregistered successfully
            return true;
        }

        /**
         * Refreshes the dependency settings of a formula descriptor that is
         * registered for this dependency manager.
         *
         * @param {FormulaDescriptor} formulaDesc
         *  The descriptor of the formula to be refreshed.
         *
         * @returns {Boolean}
         *  Whether the formula descriptor has been refreshed in the internal
         *  formula collections.
         */
        function refreshFormula(formulaDesc) {

            // find and remove an existing formula descriptor
            if (!removeFormula(formulaDesc)) { return false; }

            // refresh the source dependencies, and reinsert the descriptor
            formulaDesc.refreshReferences();
            return insertFormula(formulaDesc);
        }

        /**
         * Registers a new or changed model object with embedded token arrays.
         *
         * @param {RuleModel|SourceLinkMixin} tokenModel
         *  The new or changed model instance with embedded token arrays.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.update=false]
         *      If set to true, this method will update an existing formula
         *      descriptor, otherwise it will insert a new formula descriptor.
         */
        var registerTokenModel = DependencyUtils.profileMethod('registerTokenModel()', function (tokenModel, options) {

            // remove old settings when updating an existing formula descriptor
            if (Utils.getBooleanOption(options, 'update', false)) {
                unregisterTokenModel(tokenModel);
            }

            // register the token model for the next update cycle
            pendingCollection.registerTokenModel(tokenModel);
        });

        /**
         * Unregisters a deleted model object with embedded token arrays.
         *
         * @param {RuleModel|SourceLinkMixin} tokenModel
         *  The model instance with embedded token arrays to be deleted.
         */
        var unregisterTokenModel = DependencyUtils.profileMethod('unregisterTokenModel()', function (tokenModel) {

            // remove the token model from the pending settings
            pendingCollection.unregisterTokenModel(tokenModel);

            // remove all formula descriptors registered for the token model
            sheetDescMap.with(tokenModel.getSheetModel().getUid(), function (sheetDesc) {
                sheetDesc.modelFormulasMap.with(tokenModel.getUid(), function (formulaSet) {
                    formulaSet.forEach(removeFormula);
                });
            });
        });

        /**
         * Updates the settings of this dependency manager for a new sheet in
         * the spreadsheet document.
         *
         * @param {SheetModel} sheetModel
         *  The model instance of the new sheet.
         */
        var registerSheetModel = DependencyUtils.profileMethod('registerSheetModel()', function (sheetModel) {
            DependencyUtils.withLogging(function () {
                DependencyUtils.log('sheet="' + sheetModel.getName() + '" key=' + sheetModel.getUid());
            });

            // create the sheet descriptor object for the new sheet
            sheetDescMap.insert(sheetModel.getUid(), new SheetDescriptor(sheetModel));

            // initialize the pending settings of the sheet for the next update cycle
            pendingCollection.registerSheetModel(sheetModel);
        });

        /**
         * Updates the settings of this dependency manager before a sheet in
         * the spreadsheet document will be deleted.
         *
         * @param {SheetModel} sheetModel
         *  The model instance of the deleted sheet.
         */
        var unregisterSheetModel = DependencyUtils.profileMethod('unregisterSheetModel()', function (sheetModel) {
            DependencyUtils.withLogging(function () {
                DependencyUtils.log('sheet="' + sheetModel.getName() + '" key=' + sheetModel.getUid());
            });

            // unregister the pending settings of the sheet
            pendingCollection.unregisterSheetModel(sheetModel);

            // remove the sheet descriptor object for the deleted sheet (sheet descriptor may not
            // exist yet, e.g. when inserting/deleting a sheet very quickly in unit tests)
            var sheetDesc = sheetDescMap.remove(sheetModel.getUid());
            if (!sheetDesc) { return; }

            // remove all formulas of the deleted sheet from the internal structures
            sheetDesc.formulaSet.forEach(removeFormula);
        });

        /**
         * Processes all document change events that have been collected in the
         * pending collection.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise that will be resolved after all collected
         *  document changes have been processed.
         */
        var processPendingCollection = DependencyUtils.profileAsyncMethod('processPendingCollection()', function () {

            // process the dirty addresses and moved ranges of all sheets
            return timerHelper.iterate(pendingCollection.changedSheetMap, function (pendingSheetData) {

                // the model, UID, and cell collection of the affected sheet
                var sheetModel = pendingSheetData.sheetModel;
                var sheetUid = sheetModel.getUid();
                var cellCollection = sheetModel.getCellCollection();
                // the addresses of the dirty (moved) cell ranges
                var dirtyRanges = pendingSheetData.dirtyRanges = pendingSheetData.dirtyRanges.merge();
                // the addresses of the changed formula cells
                var dirtyCells = pendingSheetData.dirtyFormulaCells;
                // used to prevent double registration from 'dirtyRanges' and 'dirtyCells'
                var registeredCellSet = {};

                // returns whether the cell value type matches the root operator of the formula (bug 49463)
                function isMatchingCellResult(tokenArray, address) {

                    // missing cell value (no further checks needed)
                    var cellValue = cellCollection.getValue(address);
                    if (cellValue === null) { return false; }

                    // formulas may always result in an error
                    if (cellValue instanceof ErrorCode) { return true; }

                    // get the expected scalar result type from the root operator
                    var rootDesc = tokenArray.compileFormula().root.descriptor;
                    var valType = rootDesc ? rootDesc.valType : 'any';

                    // compare the expected result type with the cell value type
                    switch (valType) {
                        case 'num':
                        case 'date':    return typeof cellValue === 'number';
                        case 'str':
                        case 'comp':    return typeof cellValue === 'string';
                        case 'bool':    return typeof cellValue === 'boolean';
                        case 'err':     return cellValue instanceof ErrorCode;
                    }

                    return true;
                }

                // removes the specified cell formulas from the internal collections
                function unregisterFormulaCells(addresses) {
                    return timerHelper.iterate(addresses, function (address) {
                        var formulaKey = getCellKey(sheetUid, address);
                        formulaSet.with(formulaKey, removeFormula);
                    });
                }

                // inserts the specified cell formulas into the internal collections
                function registerFormulaCells(iterator, recalcAll) {
                    return timerHelper.iterate(iterator, function (address) {

                        // the cell key will be used as formula descriptor key (prevent double registration)
                        var formulaKey = getCellKey(sheetUid, address);
                        if (formulaKey in registeredCellSet) { return; }
                        registeredCellSet[formulaKey] = true;

                        // create a new descriptor for the token array (nothing more to do, if there is no
                        // token array available for the key, e.g. deleted cell formulas)
                        var tokenDesc = cellCollection.getTokenArray(address);
                        if (!tokenDesc) { return; }
                        var formulaDesc = new FormulaDescriptor(formulaKey, tokenDesc.tokenArray, tokenDesc.refAddress, address, tokenDesc.matrixRange);

                        // recalculate formula cell, if it has been registered in preceding update cycle (bug 56767: even if it is fixed now)
                        var recalcFormula = recalcAll || pendingSheetData.recalcAddressSet.has(address);

                        // recalculate ALL formula cells, if there is a cell without a valid result type
                        if (!recalcFormula && initialUpdateCycle && !tokenDesc.matrixRange && !isMatchingCellResult(tokenDesc.tokenArray, address)) {
                            DependencyUtils.warn('recalculating all formulas caused by wrong result type in ' + formulaDesc);
                            recalcFormula = recalcAll = pendingCollection.recalcAll = true;
                        }

                        // ignore the formula if it does not contain any dependencies
                        if (recalcFormula || !formulaDesc.isFixed()) {
                            // immediately escape the loop, if the number of formulas exceeds the configuration limit
                            if (!insertFormula(formulaDesc)) { return Utils.BREAK; }
                        } else {
                            DependencyUtils.log('skipped fixed formula ' + formulaDesc);
                        }

                        // register the new formula for recalculation
                        if (recalcFormula) { pendingSheetData.recalcAddressSet.insert(address); }
                    });
                }

                // register all relevant formulas of a new sheet (e.g. a copied sheet, bug 51453)
                if (pendingSheetData.initialUpdate) {
                    DependencyUtils.withLogging(function () {
                        DependencyUtils.log('collecting all formulas of new sheet "' + sheetModel.getName() + '"...');
                    });

                    // collect the formulas of the entire sheet range
                    dirtyRanges = pendingSheetData.dirtyRanges = dirtyRanges.append(cellCollection.getUsedRange()).merge();

                    // collect the formatting rules in the new sheet (copied sheets may contain conditional formattings)
                    var ruleIterator = sheetModel.getCondFormatCollection().createModelIterator();
                    Iterator.forEach(ruleIterator, pendingSheetData.registerTokenModel, pendingSheetData);

                    pendingSheetData.initialUpdate = false;
                }

                // remove formula descriptors of all dirty formula cells
                var promise = unregisterFormulaCells(dirtyCells);

                // remove all formula descriptors inside the dirty ranges
                promise = promise.then(function () {
                    var sheetDesc = sheetDescMap.get(sheetUid, null);
                    return timerHelper.iterate(dirtyRanges, function (dirtyRange) {
                        return unregisterFormulaCells(sheetDesc.cellFormulaSet.findAddresses(dirtyRange));
                    });
                });

                // register the changed formula cells (before dirty ranges; always recalculate all changed formula cells)
                promise = promise.then(function () {
                    return registerFormulaCells(dirtyCells.iterator(), true).done(function () { dirtyCells.clear(); });
                });

                // collect the formula cells in the moved ranges at their current positions
                promise = promise.then(function () {
                    var iterator = cellCollection.createAddressIterator(dirtyRanges, { type: 'formula', covered: true });
                    var recalc = initialUpdateCycle && calcOnLoad;
                    return registerFormulaCells(iterator, recalc).done(function () { dirtyRanges.clear(); });
                });

                // register the formulas of all formatting rules and source links
                var dirtyModelSet = pendingSheetData.dirtyModelSet;
                if (!dirtyModelSet.empty()) {
                    promise = promise.then(function () {
                        return timerHelper.iterate(dirtyModelSet, function (tokenModel) {
                            var modelUid = tokenModel.getUid();
                            var refAddress = tokenModel.getRefAddress();
                            Iterator.forEach(tokenModel.createTokenArrayIterator(), function (tokenArray, iterResult) {
                                var formulaKey = modelUid + '!' + iterResult.key;
                                var formulaDesc = new FormulaDescriptor(formulaKey, tokenArray, refAddress, null, null, tokenModel);
                                if (!formulaDesc.isFixed()) { insertFormula(formulaDesc); }
                            });
                        });
                    });
                }

                return promise;
            });
        });

        /**
         * Collects the descriptors of all dirty formulas depending directly on
         * changed cells, or indirectly on other dirty formulas, in the class
         * member variable 'resultCollection', recalculates all collected dirty
         * formulas, and writes the calculated results into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all dirty formulas have been
         *  collected and recalculated, and the cells in the document have been
         *  updated.
         */
        var recalcDirtyFormulas = DependencyUtils.profileAsyncMethod('recalcDirtyFormulas()', function () {

            // realculate-all mode: calculate and collect the results of all cell formulas in the document
            if (pendingCollection.recalcAll) {
                return timerHelper.iterate(docModel.createSheetIterator(), function (sheetModel) {
                    var iterator = sheetModel.getCellCollection().createAddressIterator(docModel.getSheetRange(), { type: 'formula', covered: true });
                    return timerHelper.iterate(iterator, function (address) {
                        self.getCellValue(sheetModel, address);
                    });
                });
            }

            // copy of the formula dictionary that will be reduced during lookup to improve performance
            var lookupDictionary = formulaDictionary.clone();
            // starting point of the abortable promise chain
            var promise = self.createResolvedPromise(null);

            // inserts a new formula descriptor into the dirty formula map, and removes it from the lookup dictionary
            function registerDirtyFormula(formulaDesc) {
                lookupDictionary.removeFormula(formulaDesc);
                resultCollection.insertFormula(formulaDesc);
            }

            // inserts a map of formula descriptors into the dirty formula map, and removes them from the lookup dictionary
            function registerDirtyFormulas(formulaSet) {
                if (formulaSet) { formulaSet.forEach(registerDirtyFormula); }
            }

            // Collect all formulas depending on defined names with changed formula expression.
            if (!pendingCollection.changedNamesSet.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('collecting formulas for dirty names...', function () {
                    return timerHelper.iterate(pendingCollection.changedNamesSet.keyIterator(), function (nameKey) {
                        registerDirtyFormulas(lookupDictionary.findFormulasForName(nameKey));
                    });
                }));
            }

            // Collect all formulas depending on table ranges with changed target range.
            if (!pendingCollection.changedTablesSet.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('collecting formulas for dirty table ranges...', function () {
                    return timerHelper.iterate(pendingCollection.changedTablesSet.keyIterator(), function (tableKey) {
                        registerDirtyFormulas(lookupDictionary.findFormulasForTable(tableKey));
                    });
                }));
            }

            // Refresh the dependencies of all found formulas for dirty names and tables. The dependencies will include the new
            // source references of the changed defined names, and the target ranges of the changed tables.
            promise = promise.then(function () {
                if (!resultCollection.formulaSet.empty()) { // bug 48014: check MUST be inside promise.then()!
                    return DependencyUtils.takeAsyncTime('refreshing formulas for dirty names and tables...', function () {
                        return timerHelper.iterate(resultCollection.formulaSet, refreshFormula);
                    });
                }
            });

            // Collect all formulas with a #NAME? error, after a defined name changed its label, and refresh
            // their dependencies too, in order to resolve #NAME? errors to valid name references.
            if (!pendingCollection.changedLabelsSet.empty()) {
                promise = promise.then(DependencyUtils.profileAsyncMethod('refreshing formulas for dirty name labels...', function () {
                    // visit all sheets (each sheet descriptor contains its own map of formulas with missing names)...
                    return timerHelper.iterate(sheetDescMap, function (sheetDesc) {
                        // process all defined names with changed label for the current sheet...
                        return timerHelper.iterate(pendingCollection.changedLabelsSet, function (nameModel) {
                            // if the sheet contains formulas with #NAME? errors using the current label...
                            return sheetDesc.missingNamesMap.with(nameModel.getKey(), function (destFormulaSet) {
                                // visit all these formulas with #NAME? errors...
                                return timerHelper.iterate(destFormulaSet, function (formulaDesc) {
                                    // the formula may have been refreshed already (multiple unresolved names)
                                    if (!resultCollection.formulaSet.has(formulaDesc)) {
                                        refreshFormula(formulaDesc);
                                        if (formulaDesc.references.containsName(nameModel)) {
                                            registerDirtyFormula(formulaDesc);
                                        }
                                    }
                                });
                            });
                        });
                    });
                }));
            }

            // Collect all initially dirty formulas from all sheets in the document.
            promise = promise.then(DependencyUtils.profileAsyncMethod('collecting initially dirty formulas...', function () {
                return timerHelper.iterate(sheetDescMap, function (sheetDesc) {

                    // Add all formulas that are permanently dirty (recalculation mode 'always'). These formulas will
                    // always be recalculated, regardless whether they depend on any dirty cells.
                    if (initialUpdateCycle || pendingCollection.recalcVolatile) {
                        registerDirtyFormulas(sheetDesc.recalcAlwaysSet);
                    }

                    // Add all formulas that are initially dirty (recalculation mode 'once'). These formulas will be
                    // recalculated once after importing the document.
                    if (initialUpdateCycle) {
                        registerDirtyFormulas(sheetDesc.recalcOnceSet);
                    }

                    // the model of the affected sheet
                    var sheetModel = sheetDesc.sheetModel;
                    // the sheet UID, used as map key
                    var sheetUid = sheetModel.getUid();
                    // collected cell addresses in the sheet from document change actions
                    var pendingSheetData = pendingCollection.getPendingSheetData(sheetModel);

                    // the addresses of all formula cells that will be recalculated
                    var recalcAddressSet = pendingSheetData ? pendingSheetData.recalcAddressSet.clone() : new AddressSet();

                    // Collect the dirty value cells. Dirty value cells that are formulas by themselves need
                    // to be recalculated (the correct formula result may have been destroyed).
                    var dirtyValueCells = pendingSheetData ? pendingSheetData.dirtyValueCells.reject(function (address) {
                        if (formulaSet.hasKey(getCellKey(sheetUid, address))) {
                            recalcAddressSet.insert(address);
                            return true;
                        }
                    }) : new AddressArray();

                    // nothing more to do without changed formula nor value cells
                    if (recalcAddressSet.empty() && dirtyValueCells.empty()) { return; }

                    // join the value cells to ranges for faster lookup of dependent formulas
                    var dirtyValueRanges = RangeArray.mergeAddresses(dirtyValueCells);

                    DependencyUtils.withLogging(function () {
                        var sheetName = sheetModel.getName();
                        if (!recalcAddressSet.empty()) {
                            var addresses = new AddressArray(recalcAddressSet.values());
                            DependencyUtils.log('outdated formula results in ' + sheetName + '!' + RangeArray.mergeAddresses(addresses));
                        }
                        if (!dirtyValueRanges.empty()) {
                            DependencyUtils.log('dirty cells in ' + sheetName + '!' + dirtyValueRanges);
                        }
                    });

                    // staring point of the abortable promise chain
                    var promise2 = self.createResolvedPromise(null);

                    // Collect all dirty cell formulas. Cells with changed formula expressions must be recalculated in
                    // order to get a valid initial result.
                    if (!recalcAddressSet.empty()) {
                        promise2 = promise2.then(function () {
                            return timerHelper.iterate(recalcAddressSet, function (address) {
                                var formulaKey = getCellKey(sheetUid, address);
                                sheetDesc.formulaSet.with(formulaKey, registerDirtyFormula);
                            });
                        });
                    }

                    // Collect all formulas depending directly on dirty value cells.
                    if (!dirtyValueRanges.empty()) {
                        promise2 = promise2.then(function () {
                            return timerHelper.iterate(dirtyValueRanges, function (range) {
                                registerDirtyFormulas(lookupDictionary.findFormulasForRange(sheetUid, range));
                            });
                        });
                    }

                    return promise2;
                }).done(function () {
                    DependencyUtils.logFormulaSet(resultCollection.formulaSet);
                });
            }));

            // Extend the map of dirty formulas with other formulas depending on them.
            promise = promise.then(function () {

                // descriptors of all cell formulas to collect dependent formulas for
                var pendingFormulaSet = resultCollection.cellFormulaSet.clone();

                // repeat as long as new formulas can be found depending on the formulas in 'pendingFormulaSet'
                return timerHelper.loop(function (index) {

                    // exit the (endless) loop, if the last iteration step did not find a new formula
                    if (pendingFormulaSet.empty()) { return Utils.BREAK; }

                    return DependencyUtils.takeAsyncTime('collecting dependent formulas for chain depth: ' + (index + 1) + '...', function () {

                        // all new formula descriptors not yet contained in the dirty formula map
                        var newFormulaSet = new FormulaSet();

                        // process all formulas whose dependencies have not been calculated yet
                        var promise2 = timerHelper.iterate(pendingFormulaSet, function (sourceFormulaDesc) {

                            // find all formulas that depend directly on the current formula (ignore formulas from token models)
                            var destFormulaSet = lookupDictionary.findFormulasForFormula(sourceFormulaDesc);
                            if (!destFormulaSet) { return; }

                            // process all found formulas depending on the current formula
                            return timerHelper.iterate(destFormulaSet, function (destFormulaDesc) {

                                // nothing more to do, if the formula has already been collected
                                if (resultCollection.formulaSet.has(destFormulaDesc)) { return; }

                                // insert the formula descriptor into the maps
                                registerDirtyFormula(destFormulaDesc);
                                newFormulaSet.insert(destFormulaDesc);
                            });
                        });

                        // initialize the next iteration cycle
                        return promise2.done(function () {
                            pendingFormulaSet = newFormulaSet;
                            DependencyUtils.logFormulaSet(newFormulaSet);
                        });
                    });
                });
            });

            // calculate and collect all formula results, as long as dirty formulas are contained in the map
            promise = promise.then(DependencyUtils.profileAsyncMethod('recalculating all dirty formulas...', function () {
                return timerHelper.loop(function () {

                    // get a formula descriptor from the dirty formula map
                    var formulaDesc = resultCollection.cellFormulaSet.getAny();
                    if (!formulaDesc) { return Utils.BREAK; }

                    // calculate and store the result of the formula (this may recursively calculate other dirty
                    // formulas as well, which will be removed from the dirty formula map)
                    var sheetModel = sheetDescMap.get(formulaDesc.sheetUid).sheetModel;
                    self.getCellValue(sheetModel, formulaDesc.cellAddress);
                });
            }));

            return promise;
        });

        /**
         * Writes the collected formula results into the document.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the cells in the document have
         *  been updated.
         */
        var sendFormulaResults = DependencyUtils.profileAsyncMethod('sendFormulaResults()', function () {

            // bug 55078: bind the promise chain to own lifetime
            var promise = self.createResolvedPromise();

            // wait for another operations generator process currently running (needed to prevent internal
            // application error due to nested operation generators), this may even abort the entire update
            // cycle, e.g. after deleting a sheet which makes the calculated formula results useless
            promise = promise.then(function () { return docModel.waitForActionsProcessed(); });

            // write the collected results (synchronously) into the document
            promise.done(DependencyUtils.profileMethod('updating document contents...', function () {

                // generate the cell operations, and send them to the server
                generateAndSendOperations(function (generator) {

                    // update the 'calcOnLoad' flag in the original spreadsheet document
                    var newCalcOnLoad = resultCollection.hasErrors ? true : initialUpdateCycle ? false : calcOnLoad;
                    if (newCalcOnLoad !== calcOnLoad) {
                        generator.generateDocAttrsOperation({ calcOnLoad: newCalcOnLoad });
                        calcOnLoad = newCalcOnLoad;
                    }

                    // create a 'changeCell' operation per sheet
                    resultCollection.resultsMap.forEach(function (resultSet, sheetUid) {

                        // generate cell operations for the remaining results
                        var sheetModel = sheetDescMap.get(sheetUid).sheetModel;
                        generator.setSheetIndex(sheetModel.getIndex());

                        // the cell operation builder
                        var operationsBuilder = new CellOperationsBuilder(sheetModel, generator, { skipShared: true });

                        // sort the formula results by address (needed by cell operation builder)
                        var resultDescs = resultSet.values().sort(function (desc1, desc2) {
                            return Address.compare(desc1.address, desc2.address);
                        });

                        // add the new result value to the 'changeCells' operation
                        resultDescs.forEach(function (resultDesc) {
                            operationsBuilder.createCells(resultDesc.address, { v: resultDesc.value }, 1);
                        });

                        // generate the cell operations
                        operationsBuilder.finalizeOperations({ createUndo: false, filter: true });
                    });
                });
            }));

            // refresh all model objects containing dirty formulas locally (without operations)
            promise.done(DependencyUtils.profileMethod('updating models with token arrays...', function () {

                // the dirty models, as sets mapped by sheet UID
                var dirtyModelMap = new ValueMap();

                // collects the token models that contain the passed dirty formulas
                function registerTokenModels(sheetUid, formulaSet) {
                    formulaSet.forEach(function (formulaDesc) {
                        var tokenModel = formulaDesc.tokenModel;
                        dirtyModelMap.getOrConstruct(sheetUid, ValueSet, 'getUid()').insert(tokenModel);
                        formulaDesc.tokenArray.clearResultCache();
                    });
                }

                // in 'recalculate all' mode, collect all models from the entire document
                if (pendingCollection.recalcAll) {
                    sheetDescMap.forEach(function (sheetDesc, sheetUid) {
                        sheetDesc.modelFormulasMap.forEach(function (formulaSet) {
                            registerTokenModels(sheetUid, formulaSet);
                        });
                    });
                } else {
                    resultCollection.modelFormulasMap.forEachKey(registerTokenModels);
                }

                // refresh the dirty models
                dirtyModelMap.forEach('forEach', 'refreshFormulas');
            }));

            return promise;
        });

        /**
         * Starts the background task that processes all collected and pending
         * document events, refreshes the dependency chains, recalculates all
         * dirty formulas, and updates the model contents (cells, conditional
         * formattings, etc.).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.continue=false]
         *      If set to true, an update cycle currently running will be
         *      continued. By default, a running update cycle will be aborted,
         *      and a fresh update cycle will be started.
         *  - {Boolean} [options.priority=false]
         *      If set to true, the update cycle will be started with high
         *      timer priority. If used together with the option 'continue',
         *      the continued update cycle will be swiched from low to high
         *      priority.
         *  - {Boolean} [options.volatile=true]
         *      If set to false (default is true!), volatile formulas will NOT
         *      be recalculated (unless they depend on dirty cells). This
         *      option will affect new update cycles only, but not a running
         *      continued update cycle.
         */
        var startUpdateCycle = (function () {

            // the timer promise representing the update cycle currently running
            var updateTimer = null;
            // first update after import will be delayed further
            var firstUpdate = true;
            // deferred object for a full cycle (remains pending until any cycle has been completed)
            var fullCycleDef = null;
            // the abortable promise of the deferred to be returned to external callers
            var fullCyclePromise = null;

            // aborts the update cycle currently running
            function abortUpdateCycle() {
                if (updateTimer) {
                    DependencyUtils.log('cancelling update cycle');
                    updateTimer.abort(); // additionally, this sets 'updateTimer' to null
                }
            }

            // immediately abort an update cycle, if a formula count overflow has been detected
            self.on('recalc:overflow', abortUpdateCycle);

            // starts a new background task (debounced)
            var runUpdateCycle = self.createDebouncedMethod('DependencyManager.runUpdateCycle', null, function () {

                // collect all dirty formulas and the results, used later e.g. in the public method getCellValue()
                resultCollection = new ResultCollection();

                // trigger the start event (additional delay for initial update after import)
                var delay = firstUpdate ? DependencyUtils.IMPORT_UPDATE_DELAY : 0;
                updateTimer = self.executeDelayed(function () { self.trigger('recalc:start'); }, 'DependencyManager.startUpdateCycle', delay);
                firstUpdate = false;

                // wait for running document operation generators before starting update cycle
                updateTimer = updateTimer.then(function () { return docModel.waitForActionsProcessed(); });

                // log the asynchronous runtime of the update cycle only, without the preparation work
                updateTimer = updateTimer.then(Utils.profileAsyncMethod('dependency update cycle', function () {

                    // Process all pending document changes (changed value cells, changed formula
                    // expressions, moved cells, etc.), before starting to update the token arrays.
                    var promise = processPendingCollection();

                    // Collect and recalculate all dirty token arrays (all token arrays referring to
                    // dirty cells, unless all formulas have to be recalculated).
                    promise = promise.then(recalcDirtyFormulas);

                    // Update the values of the formula cells in the document model.
                    promise = promise.then(sendFormulaResults);

                    return promise;
                }));

                // Notify listeners, and clean up the caches after the update cycle has finished successfully.
                updateTimer.done(function () {

                    // clean up all internal data
                    var refCycles = resultCollection.referenceCycles;
                    pendingCollection = new PendingCollection();
                    resultCollection = null;
                    initialUpdateCycle = false;
                    updateTimer = null;

                    // notify the listeners
                    self.trigger('recalc:end', refCycles);

                    // resolve the deferred object that represents a full calculation cycle (must be
                    // the last action, application quit waits for the promise and kills the application)
                    if (fullCycleDef) { fullCycleDef.resolve(); }
                });

                // notify listeners if the update cycle has been aborted; 'fullCycleDef' remains pending
                updateTimer.fail(function () {
                    updateTimer = null;
                    self.trigger('recalc:cancel');
                });

                return updateTimer;
            }, { delay: DependencyUtils.UPDATE_DEBOUNCE_DELAY });

            // the implementation of the method startUpdateCycle() returned from local scope
            return function (options) {

                // nothing to do in disconnected state
                if (!pendingCollection) {
                    return self.createResolvedPromise();
                }

                // abort a running background task (unless the task shall be continued)
                if (!Utils.getBooleanOption(options, 'continue', false)) {
                    abortUpdateCycle();
                }

                // start (or continue) timers with high priority if specified
                if (Utils.getBooleanOption(options, 'priority', false)) {
                    timerHelper.setPriority('high');
                }

                // create the promise that will be returned to external callers from public methods
                // (remains in pending state until an update cycle can be finished successfully)
                if (!fullCycleDef) {
                    fullCycleDef = Scheduler.createDeferred(docModel, 'DependencyManager.startUpdateCycle');
                    fullCyclePromise = self.createAbortablePromise(fullCycleDef);
                    fullCyclePromise.done(function () { timerHelper.setPriority('low'); });
                    fullCyclePromise.always(function () { fullCycleDef = null; });
                }

                // whether to recalculate all volatile formulas that do NOT depend on dirty cells
                pendingCollection.recalcVolatile = Utils.getBooleanOption(options, 'volatile', true);

                // start the new background task (unless an update cycle exists already and will be reused)
                if (!updateTimer) { runUpdateCycle();  }

                return fullCyclePromise;
            };
        }());

        // public methods -----------------------------------------------------

        /**
         * Switches this dependency manager to permenent disconnected state. A
         * disconnected dependency manager will not collect or process any
         * formulas inserted into the document anymore. Once the dependency
         * manager is in disconnected state, it cannot return to active state.
         *
         * @returns {DependencyManager}
         *  A reference to this instance.
         */
        this.disconnect = function () {
            this.stopListeningTo(docModel);
            formulaSet.clear();
            sheetDescMap.clear();
            formulaDictionary.clear();
            pendingCollection = null;
            resultCollection = null;
            return this;
        };

        /**
         * Returns whether this dependency manager is in active state, i.e. it
         * listens to changes in the document, and recalculates all formulas on
         * demand in a background task.
         *
         * @see DependencyManager.disconnect()
         *
         * @returns {Boolean}
         *  Whether this dependency manager is in active state.
         */
        this.isConnected = function () {
            return pendingCollection !== null;
        };

        /**
         * Registers a new source link model (e.g. the data series model of a
         * chart object) for automatic updates after the referred cells have
         * been changed.
         *
         * @param {SourceLinkMixin} linkModel
         *  The new source link model.
         *
         * @returns {DependencyManager}
         *  A reference to this instance.
         */
        this.registerLinkModel = function (linkModel) {

            // nothing to do in disconnected state
            if (!this.isConnected()) { return this; }

            DependencyUtils.withLogging(function () {
                DependencyUtils.log('inserted source link ' + DependencyUtils.getLinkLabel(linkModel));
            });
            registerTokenModel(linkModel);
            this.listenTo(linkModel, 'change:sourcelinks', function () {
                registerTokenModel(linkModel, { update: true });
            });

            startUpdateCycle();
            return this;
        };

        /**
         * Unregisters a source link model (e.g. the data series model of a
         * chart object) that will be deleted from the spreadsheet document.
         *
         * @param {SourceLinkMixin} linkModel
         *  The source link model to be deleted.
         *
         * @returns {DependencyManager}
         *  A reference to this instance.
         */
        this.unregisterLinkModel = function (linkModel) {

            // nothing to do in disconnected state
            if (!this.isConnected()) { return this; }

            DependencyUtils.withLogging(function () {
                DependencyUtils.log('deleted source link ' + DependencyUtils.getLinkLabel(linkModel));
            });
            this.stopListeningTo(linkModel);
            unregisterTokenModel(linkModel);

            startUpdateCycle();
            return this;
        };

        /**
         * Recalculates all dirty formulas in the document, and refreshes all
         * cells with outdated formula results, and all other document contents
         * that depend on formula results (e.g. charts, or conditional
         * formatting). The background task that calculates the formula results
         * will run with high priority (in difference to background tasks
         * started automatically due to change events of the document processed
         * by this dependency manager).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.all=false]
         *      If set to true, all availavle formulas in the document will be
         *      recalculated, regardless of their diryt state.
         *  - {Boolean} [options.volatile=true]
         *      If set to false (default is true!), volatile formulas will NOT
         *      be recalculated (unless they depend on dirty cells). This
         *      option will be ignored, if the option 'all' has been set.
         *  - {Number} [options.timeout]
         *      The maximum time allowed for the recalculation cycle, in
         *      milliseconds. If the time elapses before all formulas have been
         *      calculated, the recalculation cycle will be aborted, and the
         *      'calcOnLoad' document flag will be set.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all dirty formulas have been
         *  recalculated successfully.
         */
        this.recalcFormulas = function (options) {

            // nothing to do in disconnected state
            if (!this.isConnected()) { return this.createResolvedPromise(); }

            // initialize the 'recalcAll' flag in the internal data
            if (Utils.getBooleanOption(options, 'all', false)) {
                pendingCollection.recalcAll = true;
            }

            // start the update cycle, or continue a running update cycle
            var promise = startUpdateCycle({
                continue: !pendingCollection.recalcAll,
                priority: true,
                volatile: Utils.getBooleanOption(options, 'volatile', true)
            });

            // initialize a timer that aborts the returned promise after the specified delay
            var timeout = Utils.getIntegerOption(options, 'timeout', 0);
            if (timeout > 0) {

                // create a new deferred object that represents this method invocation
                // (this allows to call the method multiple times with individual timeouts)
                var deferred = Scheduler.createDeferred(docModel, 'DependencyManager.recalcFormulas');
                promise.done(deferred.resolve.bind(deferred));
                promise.fail(deferred.reject.bind(deferred));
                promise.progress(deferred.notify.bind(deferred));

                // create a promise with timeout from the new deferred object,
                // set the 'calcOnLoad' document flag if the timeout is reached
                promise = this.createAbortablePromise(deferred, setCalcOnLoadFlag, timeout);
            }

            return promise;
        };

        /**
         * Returns an up-to-date cell value for the specified cell in the
         * document. If the cell is a formula cell that this dependency manager
         * has marked as dirty, the formula will be recalculated, and all cells
         * referred by the formula that are dirty too will be recalculated
         * recursively. Otherwise, the current value of the cell as stored in
         * the cell model will be returned.
         *
         * @param {SheetModel} sheetModel
         *  The model of the sheet containing the cell to be resolved.
         *
         * @param {Address} address
         *  The address of the cell whose value will be resolved.
         *
         * @returns {Object}
         *  A descriptor object with information about the resolved cell value,
         *  with the following properties:
         *  - {Any} value
         *      The resolved scalar value of the specified cell.
         *  - {String} type
         *      How the cell value has been resolved. If set to 'calculated',
         *      the value has just been calculated by interpreting the dirty
         *      cell formula. If set to 'cached', the result has already been
         *      calculated by a preceding call of this method during the update
         *      cycle  currently running. If set to 'fixed', the value has been
         *      returned directly from the cell model (either because the cell
         *      is not a formula cell, or the formula was not dirty).
         *  - {Number} time
         *      The time in milliseconds it took to calculate the result of a
         *      dirty formula cell (type 'calculated'). Will be zero for all
         *      other value types ('cached' and 'fixed').
         */
        this.getCellValue = (function () {

            // stack with the addresses of all formulas currently calculated recursively
            var formulaAddressStack = new AddressArray();

            // calculates the result of a cell formula
            function calculateCellFormula(sheetModel, tokenArray, refAddress, targetAddress, matrixRange) {

                // take the time it needs to calculate the formula (and all formulas it depends on)
                var startTime = _.now();

                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('calculating formula in cell ' + sheetModel.getName() + '!' + targetAddress);
                });

                // put the target address with sheet UID onto the stack (used to detect circular references)
                var formulaAddress = targetAddress.clone();
                formulaAddress.sheetUid = sheetModel.getUid();
                formulaAddress.sheet = sheetModel.getIndex();
                formulaAddressStack.push(formulaAddress);

                // first, calculate the formula result
                var formulaResult = tokenArray.interpretFormula(matrixRange ? 'mat' : 'val', {
                    refAddress: refAddress,
                    targetAddress: matrixRange ? refAddress : targetAddress,
                    matrixRange: matrixRange,
                    recalcDirty: true
                });

                // detect circular reference errors, and other warnings and errors
                if (formulaResult.code === 'circular') {
                    DependencyUtils.withLogging(function () {
                        var addressNames = formulaAddressStack.map(function (address) {
                            return docModel.getSheetName(address.sheet) + '!' + address;
                        });
                        DependencyUtils.warn('reference cycle found: ' + addressNames.join());
                    });
                    resultCollection.referenceCycles.push(formulaAddressStack.clone());
                } else if (formulaResult.type !== 'valid') {
                    resultCollection.hasErrors = true;
                }
                formulaAddressStack.pop();

                // keep old result in the map, if the formula has been recalculated already due to a circular reference
                var resultSet = resultCollection.createResultSet(sheetModel.getUid());
                var addressKey = targetAddress.key();
                if (resultSet.hasKey(addressKey)) { return null; }

                // immediately update the URL of the HYPERLINK function
                var cellCollection = sheetModel.getCellCollection();
                cellCollection.setFormulaURL(targetAddress, formulaResult.url || null);

                // add the timestamp to the result map, needed to detect timeouts during calculation
                var time = _.now() - startTime;

                // distribute the elements of a matrix result into the result map
                if (matrixRange) {
                    var matrix = formulaResult.value;
                    var col1 = matrixRange.start[0];
                    var row1 = matrixRange.start[1];
                    Iterator.forEach(matrixRange, function (address) {
                        var value = matrix.get(address[1] - row1, address[0] - col1);
                        resultSet.insert(address, value, time);
                    });
                    // return the cell result of the correct matrix element
                    return resultSet.get(addressKey, null);
                }

                // insert the new result into the result map
                return resultSet.insert(targetAddress, formulaResult.value, time);
            }

            // the implementation of the method getCellValue()
            function getCellValue(sheetModel, address) {

                // the UID of the sheet, used as map key
                var sheetUid = sheetModel.getUid();
                // the result descriptor
                var valueDesc = { value: null, type: null, time: 0 };

                // first, try to find a result already calculated by a preceding call of this method
                var resultDesc = resultCollection.getResult(sheetUid, address);
                if (resultDesc) {
                    valueDesc.value = resultDesc.value;
                    valueDesc.type = 'cached';
                    return valueDesc;
                }

                // the cell collection of the current sheet
                var cellCollection = sheetModel.getCellCollection();

                // in 'recalculate all' mode, resolve the token array directly from the cell collection
                if (pendingCollection.recalcAll) {

                    var tokenDesc = cellCollection.getTokenArray(address, { fullMatrix: true });
                    resultDesc = tokenDesc ? calculateCellFormula(sheetModel, tokenDesc.tokenArray, tokenDesc.refAddress, address, tokenDesc.matrixRange) : null;

                } else {

                    // try to find a dirty formula, and calculate its result
                    var formulaKey = getCellKey(sheetUid, address);
                    resultCollection.cellFormulaSet.with(formulaKey, function (formulaDesc) {
                        // calculate the result of the cell formula
                        resultDesc = calculateCellFormula(sheetModel, formulaDesc.tokenArray, formulaDesc.refAddress, address, formulaDesc.matrixRange);
                        // remove the formula descriptor (MUST be done after calculation, needed to detect circular references!)
                        // if the formula has been removed already due to a circular reference, keep that result in the map
                        resultCollection.cellFormulaSet.remove(formulaDesc);
                    });
                }

                // fill and return the descriptor if a formula has been calculated
                if (resultDesc) {
                    valueDesc.value = resultDesc.value;
                    valueDesc.type = 'calculated';
                    valueDesc.time = resultDesc.time;
                    return valueDesc;
                }

                // return the current cell value for all other formulas, and simple cells
                valueDesc.value = cellCollection.getValue(address);
                valueDesc.type = 'fixed';
                return valueDesc;
            }

            return getCellValue;
        }());

        /**
         * Returns the resource keys of all unimplemented functions that occur
         * in any formula known by this dependency manager.
         *
         * @returns {Array<String>}
         *  The resource keys of all unimplemented functions that occur in any
         *  formula known by this dependency manager.
         */
        this.getMissingFunctionKeys = function () {

            // collect all function keys from all sheets in a map to prevent duplicates
            var missingFuncMap = new ValueMap();
            sheetDescMap.forEach(function (sheetDesc) {
                missingFuncMap.merge(sheetDesc.missingFuncMap);
            });

            // return the function resource keys as array
            return missingFuncMap.keys();
        };

        // initialization -----------------------------------------------------

        // initialization after document import
        docModel.waitForImportSuccess(function () {

            // recalculate all formulas loaded from the document if specified
            calcOnLoad = docModel.getDocumentAttribute('calcOnLoad');

            // initialize dependency manager for all existing sheets after import
            Iterator.forEach(docModel.createSheetIterator(), registerSheetModel);

            // inserted sheet: register all existing formulas (e.g. in a copied sheet)
            this.listenTo(docModel, 'insert:sheet:after', function (event, sheet, sheetModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: inserted sheet "' + sheetModel.getName() + '"');
                });
                // register the new sheet model for lazy update
                registerSheetModel(sheetModel);
                startUpdateCycle();
            });

            // deleted sheet: remove all registered formulas of the sheet
            this.listenTo(docModel, 'delete:sheet:before', function (event, sheet, sheetModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: deleted sheet "' + sheetModel.getName() + '"');
                });
                // delete the sheet model and related data from all collections
                unregisterSheetModel(sheetModel);
                startUpdateCycle();
            });

            // recalculate all volatile formulas after sheet has been moved (e.g. SHEET function)
            this.listenTo(docModel, 'move:sheet:before', function (event, from, to, sheetModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: moved sheet "' + sheetModel.getName() + '"');
                });
                // simply recalculate all volatile formulas, nothing more to do
                startUpdateCycle();
            });

            // recalculate all volatile formulas after sheet has been renamed (e.g. INDIRECT function)
            this.listenTo(docModel, 'rename:sheet', function (event, sheet, sheetName) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed sheet name to "' + sheetName + '"');
                });
                // simply recalculate all volatile formulas, nothing more to do
                startUpdateCycle();
            });

            // new defined name: update formulas with #NAME! error referring to the new name
            this.listenTo(docModel, 'insert:name', function (event, sheet, nameModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: inserted ' + DependencyUtils.getNameLabel(nameModel));
                });
                // register the model of the new defined name for lazy update
                pendingCollection.registerNameModel(nameModel, true, true);
                startUpdateCycle();
            });

            // deleted defined name: unregister the name model, and update all affected formulas
            this.listenTo(docModel, 'delete:name', function (event, sheet, nameModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: deleted ' + DependencyUtils.getNameLabel(nameModel));
                });
                // delete the name model from the collection of pending document actions
                pendingCollection.unregisterNameModel(nameModel);
                startUpdateCycle();
            });

            // changed defined name: update all formulas referring to the name
            this.listenTo(docModel, 'change:name', function (event, sheet, nameModel, changeFlags) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed ' + DependencyUtils.getNameLabel(nameModel));
                });
                // register the model of the changed defined name for lazy update
                pendingCollection.registerNameModel(nameModel, changeFlags.formula, changeFlags.label);
                startUpdateCycle();
            });

            // deleted table model: unregister the table model (do not process pending change events)
            this.listenTo(docModel, 'delete:table', function (event, sheet, tableModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: deleted ' + DependencyUtils.getTableLabel(tableModel));
                });
                // delete the table model from the collection of pending document actions
                pendingCollection.unregisterTableModel(tableModel);
                startUpdateCycle();
            });

            // changed table model: update formulas if the target range of the table has changed
            this.listenTo(docModel, 'change:table', function (event, sheet, tableModel, type) {
                if (type === 'change:range') {
                    DependencyUtils.withLogging(function () {
                        DependencyUtils.log('Event: changed target range of ' + DependencyUtils.getTableLabel(tableModel));
                    });

                    // register the model of the changed table range for lazy update
                    pendingCollection.registerTableModel(tableModel);
                    startUpdateCycle();
                }
            });

            // register changed cell values and cell formulas
            this.listenTo(docModel, 'change:cells', function (event, sheet, changeDesc, external, results) {
                // ignore the change events that have been generated due to the own last update cycle
                if (results) { return; }
                DependencyUtils.withLogging(function () {
                    var sheetName = docModel.getSheetName(sheet);
                    if (!changeDesc.valueCells.empty()) {
                        DependencyUtils.log('Event: changed values in ' + sheetName + '!' + RangeArray.mergeAddresses(changeDesc.valueCells));
                    }
                    if (!changeDesc.formulaCells.empty()) {
                        DependencyUtils.log('Event: changed formulas in ' + sheetName + '!' + RangeArray.mergeAddresses(changeDesc.formulaCells));
                    }
                });
                // register the addresses of all dirty value and formula cells
                if (pendingCollection.registerDirtyCells(docModel.getSheetModel(sheet), changeDesc)) {
                    startUpdateCycle();
                }
            });

            // moved cells, including inserted or deleted columns/rows
            this.listenTo(docModel, 'move:cells', function (event, sheet, moveDesc) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: moved cells in ' + docModel.getSheetName(sheet) + '!' + moveDesc.dirtyRange);
                });
                // move the addresses of all dirty value and formula cells
                pendingCollection.moveDirtyCells(docModel.getSheetModel(sheet), moveDesc);
                startUpdateCycle();
            });

            // recalculate all volatile formulas after rows have been shown or hidden (e.g. SUBTOTAL function)
            this.listenTo(docModel, 'change:rows', function (event, sheet, interval, changeInfo) {
                // restrict to changed visibility of the rows
                if (!changeInfo.visibilityChanged) { return; }
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed row visibility in ' + docModel.getSheetName(sheet) + '!' + interval.stringifyAsRows());
                });
                // simply recalculate all volatile formulas, nothing more to do
                startUpdateCycle();
            });

            // new formatting rule: register all formula expressions of the conditions
            this.listenTo(docModel, 'insert:rule', function (event, sheet, ruleModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: inserted ' + DependencyUtils.getRuleLabel(ruleModel));
                });
                registerTokenModel(ruleModel);
                startUpdateCycle();
            });

            // deleted formatting rule: unregister all formula expressions of the conditions
            this.listenTo(docModel, 'delete:rule', function (event, sheet, ruleModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: deleted ' + DependencyUtils.getRuleLabel(ruleModel));
                });
                unregisterTokenModel(ruleModel);
                startUpdateCycle();
            });

            // changed formatting rule (target range, or formula expressions of conditions)
            this.listenTo(docModel, 'change:rule', function (event, sheet, ruleModel) {
                DependencyUtils.withLogging(function () {
                    DependencyUtils.log('Event: changed ' + DependencyUtils.getRuleLabel(ruleModel));
                });
                registerTokenModel(ruleModel, { update: true });
                startUpdateCycle();
            });

            // initial update cycle for the imported sheets
            startUpdateCycle();
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            timerHelper.destroy();
            formulaSet.clear();
            sheetDescMap.clear();
            formulaDictionary.clear();
            self = app = docModel = timerHelper = null;
            formulaSet = sheetDescMap = formulaDictionary = null;
            pendingCollection = resultCollection = null;
        });

    } }); // class DependencyManager

    // exports ================================================================

    return DependencyManager;

});
