/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/framenodemanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils'
], function (Utils, ValueMap, BaseObject, DrawingFrame, DOMUtils, TextFrameUtils) {

    'use strict';

    // private global functions ===============================================

    /**
     * Creates a new text span element with an empty text node.
     *
     * @returns {HTMLSpanElement}
     *  A new text span element with an empty text node.
     */
    function createEmptySpan() {
        var spanNode = document.createElement('span');
        spanNode.appendChild(document.createTextNode(''));
        return spanNode;
    }

    // class FrameNodeManager =================================================

    /**
     * Provides a DOM container node for drawing frames used as model instances
     * for text contents as expected by the text framework.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetModel} docModel
     *  The model of the spreadsheet document.
     */
    var FrameNodeManager = BaseObject.extend({ constructor: function (docModel) {

        // the application instance
        var app = docModel.getApp();

        // the paragraph style collection used to update the text formatting of shape objects
        var paraStyles = null;

        // the root container node for all top-level drawing frames
        var rootNode = FrameNodeManager.createRootNode();

        // all existing drawing frames, mapped by model UID (faster access than DOM lookups)
        var drawingFrameMap = new ValueMap();

        // base constructor ---------------------------------------------------

        BaseObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Clears all contents from the root layer node, and prepares it for
         * insertion of drawing frames.
         */
        function clearRootNode() {
            rootNode.innerHTML = '';
            rootNode.appendChild(createEmptySpan());
        }

        /**
         * Returns the DOM container node that becomes the parent node for the
         * drawing frame associated to the passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model to find the DOM conatiner node for.
         *
         * @returns {HTMLElement|Null}
         *  The DOM container node for the passed drawing model; or null on any
         *  error.
         */
        function getContainerNode(drawingModel) {

            // resolve parent drawing model (passed drawing object may be part of a group object)
            var parentModel = drawingModel.getParentModel();

            // no parent model: the root layer node becomes the container node
            if (!parentModel) { return rootNode; }

            // the content node of the parent drawing frame is the container for embedded drawing frames
            var contentNode = drawingFrameMap.with(parentModel.getUid(), DrawingFrame.getContentNode);
            if (contentNode && (contentNode.length === 1)) { return contentNode[0]; }

            // failure: cannot resolve parent content node
            Utils.error('FrameNodeManager.getContainerNode(): missing container node for parent model');
            return null;
        }

        /**
         * Inserts the passed DOM drawing frame into its container node.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing object.
         *
         * @param {HTMLElement} containerNode
         *  The container node that will receive the passed drawing frame.
         *
         * @param {jQuery} drawingFrame
         *  The drawing frame to be inserted into the passed container node.
         */
        function insertDrawingFrame(drawingModel, containerNode, drawingFrame) {

            // the insertion position in the node list
            var zIndex = drawingModel.getIndex();
            // the child node list of the container element
            var childNodes = containerNode.childNodes;

            // text framework expects empty span elements between top-level drawings
            if (containerNode === rootNode) {
                zIndex *= 2;
                containerNode.insertBefore(createEmptySpan(), childNodes.item(zIndex));
                zIndex += 1;
            }

            // insert the drawing frame at the correct position into the DOM collection
            containerNode.insertBefore(drawingFrame[0], childNodes.item(zIndex));
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the root container node for all top-level drawing frames.
         *
         * @returns {HTMLElement}
         *  The root container node for all top-level drawing frames.
         */
        this.getRootNode = function () {
            return rootNode;
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to return the DOM drawing frame for.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.update=false]
         *      If set to true, the settings of the drawing frame associated
         *      with the passed drawing model, and its embedded text frame,
         *      will be updated. In detail, the explicit attributes of the
         *      drawing model will be copied to the DOM drawing frame (as
         *      expected by the text framework, e.g. to resolve default
         *      character formatting attributes), and the original unscaled CSS
         *      location of the drawing frame and its embedded text frame will
         *      be updated.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.getDrawingFrame = function (drawingModel) {
            return drawingFrameMap.get(drawingModel.getUid(), null);
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object to return the DOM drawing frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.withDrawingFrame = function (drawingModel, callback, context) {
            return drawingFrameMap.with(drawingModel.getUid(), callback, context);
        };

        /**
         * Creates a new DOM drawing frame that represents the passed drawing
         * model, and inserts it into the DOM tree represented by this
         * instance.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object to create a DOM drawing frame for.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.internal=false]
         *      If set to true, the drawing frame will be marked with a special
         *      CSS class that makes it an internal model container with
         *      reduced rendering (improved performance and reduced usage of
         *      memory).
         *
         * @returns {jQuery|Null}
         *  The new DOM drawing frame for the passed drawing model; or null, if
         *  the drawing frame could not be created correctly.
         */
        this.createDrawingFrame = function (drawingModel, options) {

            // the container node for the new drawing frame
            var containerNode = getContainerNode(drawingModel);
            if (!containerNode) { return null; }

            // the new drawing frame, as jQuery object
            var drawingFrame = DrawingFrame.createDrawingFrame(drawingModel);

            // store the drawing frame in a local map (faster access than DOM lookup)
            drawingFrameMap.insert(drawingModel.getUid(), drawingFrame);

            // insert the drawing frame at the correct position into the DOM collection
            insertDrawingFrame(drawingModel, containerNode, drawingFrame);

            // additional updates for internal drawing objects with text frames
            if (Utils.getBooleanOption(options, 'internal', false)) {
                drawingFrame.addClass(DrawingFrame.INTERNAL_MODEL_CLASS);
                DrawingFrame.checkEmptyTextShape(docModel, drawingFrame);
            }

            // create the drawing frames of all existing embedded drawing objects (e.g. when cloning)
            drawingModel.forEachChildModel(function (childModel) {
                this.createDrawingFrame(childModel, options);
            }, this);

            return drawingFrame;
        };

        /**
         * Moves the DOM drawing frame associated to the passed drawing model
         * to a new position in the DOM tree represented by this instance.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object whose DOM drawing frame will be
         *  moved.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model that has been
         *  moved to its current position; or null, if the drawing frame could
         *  not be moved correctly.
         */
        this.moveDrawingFrame = function (drawingModel) {

            // get the drawing frame from the map
            var drawingFrame = drawingFrameMap.get(drawingModel.getUid(), null);
            if (!drawingFrame) { return null; }

            // the new container node for the drawing frame
            var containerNode = getContainerNode(drawingModel);
            if (!containerNode) { return null; }

            // detach the drawing frame (text framework expects empty span elements
            // between top-level drawings)
            drawingFrame.prev('span').remove();
            drawingFrame.detach();

            // insert the drawing frame at the correct position into the DOM collection
            insertDrawingFrame(drawingModel, containerNode, drawingFrame);
            return drawingFrame;
        };

        /**
         * Removes the DOM drawing frame associated to the passed drawing model
         * from the DOM tree represented by this instance.
         *
         * @param {DrawingModel} drawingModel
         *  The model of a drawing object whose DOM drawing frame will be
         *  removed.
         *
         * @returns {FrameNodeManager}
         *  A reference to this instance.
         */
        this.removeDrawingFrame = function (drawingModel) {

            // remove the drawing frame from the DOM container
            this.withDrawingFrame(drawingModel, function (drawingFrame) {

                // text framework expects empty span elements between top-level drawings
                drawingFrame.prev('span').remove();

                // remove the drawing frame (this removes all embedded drawing frames too)
                app.destroyImageNodes(drawingFrame);
                drawingFrame.remove();
            });

            // remove the drawing frames of all embedded child objects from the map
            (function removeDrawingFrameFromMap(parentModel) {
                // remove the drawing frame of the current drawing model
                drawingFrameMap.remove(parentModel.getUid());
                // remove the drawing frames of all embedded drawing models
                parentModel.forEachChildModel(removeDrawingFrameFromMap);
            }(drawingModel));

            return this;
        };

        /**
         * Removes all DOM drawing frames contained in this instance.
         *
         * @returns {FrameNodeManager}
         *  A reference to this instance.
         */
        this.removeAllDrawingFrames = function () {
            app.destroyImageNodes(rootNode);
            drawingFrameMap.clear();
            clearRootNode();
            return this;
        };

        /**
         * Updates the CSS text formatting of the text contents in the drawing
         * frame associated to the passed drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  The model of the drawing objects whose text formatting will be
         *  updated.
         *
         * @returns {FrameNodeManager}
         *  A reference to this instance.
         */
        this.updateTextFormatting = function (drawingModel) {

            // nothing to do, if the drawing frame or text frame does not exist (yet)
            this.withDrawingFrame(drawingModel, function (drawingFrame) {
                TextFrameUtils.withTextFrame(drawingFrame, function (textFrame) {

                    // drawing object contains default text formatting for all paragraphs
                    var updateOptions = { baseAttributes: drawingModel.getMergedAttributeSet(true) };
                    // a collection of all paragraph children of the drawing frame
                    var allParagraphs = textFrame.children(DOMUtils.PARAGRAPH_NODE_SELECTOR);

                    // update paragraph/character formatting of the embedded paragraph nodes
                    allParagraphs.each(function () {
                        paraStyles.updateElementFormatting(this, updateOptions);
                    });

                    // update the lists
                    docModel.updateLists(updateOptions, allParagraphs);
                });
            });

            return this;
        };

        // initialization -----------------------------------------------------

        // text framework expects empty span elements between top-level drawings
        clearRootNode();

        // deferred initialization
        app.onInit(function () {
            paraStyles = docModel.getParagraphStyles();
        }, this);

        // destroy all class members
        this.registerDestructor(function () {
            this.removeAllDrawingFrames();
            app = docModel = paraStyles = rootNode = drawingFrameMap = null;
        });

    } }); // class FrameNodeManager

    // constants --------------------------------------------------------------

    /**
     * Returns the root container node for all top-level drawing frames.
     *
     * @returns {HTMLElement}
     *  The root container node for all top-level drawing frames.
     */
    FrameNodeManager.createRootNode = function () {
        var rootNode = document.createElement('div');
        rootNode.className = 'p sheet';
        rootNode.appendChild(createEmptySpan());
        return rootNode;
    };

    // exports ================================================================

    return FrameNodeManager;

});
