/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/commentmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/drawinglayer/model/drawingmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin'
], function (Utils, DateUtils, DrawingModel, Operations, SheetUtils, DrawingModelMixin) {

    'use strict';

    // convenience shortcuts
    var MoveDescriptor = SheetUtils.MoveDescriptor;

    // class SheetCommentModel ================================================

    /**
     * The model of a generic drawing object contained in a sheet.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this drawing object.
     *
     * @param {DrawingCollection} parentCollection
     *  The parent drawing collection that will contain this drawing object.
     *
     * @param {String} drawingType
     *  The type of this drawing object.
     *
     * @param {Object|Null} initAttributes
     *  An attribute set with initial formatting attributes for the object.
     *
     * @param {Object} initSettings
     *  Additional comment settings:
     *  - {Address} initOptions.anchor
     *      The anchor address of the comment (the address of the cell the
     *      comment is associated with). This is a required property!
     *  - {String} [initOptions.author]
     *      The name of the author who created the comment.
     *  - {Date} [initOptions.date]
     *      The creation date of the comment, as UTC date object.
     *  - {String} [initOptions.text]
     *      The text contents of the comment box.
     */
    var SheetCommentModel = DrawingModel.extend({ constructor: function (sheetModel, parentCollection, initAttributes, initSettings) {

        // the anchor address of the comment
        var anchor = initSettings.anchor.clone();

        var author = Utils.getStringOption(initSettings, 'author', '');
        var date = Utils.getStringOption(initSettings, 'date', '');
        var text = Utils.getStringOption(initSettings, 'text', '');

        // base constructors --------------------------------------------------

        DrawingModel.call(this, parentCollection, 'comment', initAttributes, { families: 'shape fill line character paragraph' });
        DrawingModelMixin.call(this, sheetModel, cloneConstructor);

        // private methods ----------------------------------------------------

        function cloneConstructor(targetModel, targetCollection) {
            var settings = { anchor: anchor, author: author, date: date, text: text };
            return new SheetCommentModel(targetModel, targetCollection, this.getExplicitAttributeSet(true), settings);
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the address of the anchor cell of this comment model.
         *
         * @returns {Address}
         *  The address of the anchor cell of this comment model.
         */
        this.getAnchor = function () {
            return anchor;
        };

        /**
         * Returns the location of the anchor cell of this comment model in the
         * sheet, either in 1/100 of millimeters, or in pixels according to the
         * current sheet zoom factor.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  SheetModel.getCellRectangle().
         *
         * @returns {Rectangle}
         *  The location of the anchor cell in the sheet.
         */
        this.getAnchorRectangle = function (options) {
            return sheetModel.getCellRectangle(anchor, options);
        };

        /**
         * Returns the name of the author of this comment model.
         *
         * @returns {String}
         *  The name of the author of this comment model. May be an empty
         *  string.
         */
        this.getAuthor = function () {
            return author;
        };

        /**
         * Returns the creation date of this comment model.
         *
         * @returns {Date}
         *  The creation of this comment model.
         */
        this.getDate = function (options) {
            var formatted = Utils.getBooleanOption(options, 'formatted', true);
            return (formatted) ? DateUtils.format(new Date(date), 'DD.MM.Y HH:mm') : date;
        };

        /**
         * Returns the text contents of this comment model.
         *
         * @returns {String}
         *  The text contents of this comment model.
         */
        this.getText = function () {
            return text;
        };

        /**
         * Moves this comment model to a new anchor cell.
         *
         * @param {Address} newAnchor
         *  The address of the new anchor cell for this comment model.
         *
         * @returns {Boolean}
         *  Whether the anchor address has been changed.
         */
        this.setAnchor = function (newAnchor) {
            if (anchor.equals(newAnchor)) { return false; }
            anchor = newAnchor;
            return true;
        };

        /**
         * Changes the text contents of this comment model.
         *
         * @param {String} newText
         *  The new text contents for this comment model.
         *
         * @returns {Boolean}
         *  Whether the text contents have been changed.
         */
        this.setText = function (newText) {
            if (text === newText) { return false; }
            text = newText;
            return true;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations, and the undo operations, to change the
         * attributes of this cell comment.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} attributeSet
         *  The incomplete attribute set with all formatting attributes to be
         *  changed.
         *
         * @param {Array<MoveDescriptor>} [moveDescs]
         *  If specified, the move descriptors that describe how cells in the
         *  own spreadsheet have been moved. Needed to calculate the original
         *  anchor position of this comment for the undo operation.
         *
         * @returns {SheetCommentModel}
         *  A reference to this instance.
         */
        this.generateChangeOperations = function (generator, attributeSet, moveDescs) {

            // create the undo attributes; nothing to do if no attribute actually changes
            var undoAttrSet = this.getUndoAttributeSet(attributeSet);
            if (_.isEmpty(undoAttrSet)) { return this; }

            // bug 58272: change operation is generated after moving cells (for new anchor position);
            // but the undo operation needs to be generated for the original anchor position
            var undoAnchor = moveDescs ? MoveDescriptor.transformAddress(anchor, moveDescs, { reverse: true }) : anchor;
            if (!undoAnchor) { return this; }

            generator.generateCellOperation(Operations.CHANGE_COMMENT, anchor, { attrs: attributeSet });
            generator.generateCellOperation(Operations.CHANGE_COMMENT, undoAnchor, { attrs: undoAttrSet }, { undo: true });
            return this;
        };

        /**
         * Generates the undo operations needed to restore this cell comment.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @returns {SheetCommentModel}
         *  A reference to this instance.
         */
        this.generateRestoreOperations = function (generator) {
            var operProps = { attrs: this.getExplicitAttributeSet(true) };
            if (author) { operProps.author = author; }
            if (date) { operProps.date = date; }
            if (text) { operProps.text = text; }
            generator.generateCellOperation(Operations.INSERT_COMMENT, anchor, operProps, { undo: true });
            return this;
        };

        /**
         * Generates the operations to delete, and the undo operations to
         * restore this cell comment.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {SheetCommentModel}
         *  A reference to this instance.
         */
        this.generateDeleteOperations = function (generator) {
            this.generateRestoreOperations(generator);
            generator.generateCellOperation(Operations.DELETE_COMMENT, anchor);
            return this;
        };

        /**
         * Generates the undo operations to restore this comment if it will be
         * deleted implicitly by moving cells (e.g. deleting columns or rows)
         * in the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDescs
         *  An array of move descriptors that specify how the cells in the
         *  sheet will be moved.
         *
         * @returns {SheetCommentModel}
         *  A reference to this instance.
         */
        this.generateBeforeMoveCellsOperations = _.wrap(this.generateBeforeMoveCellsOperations, function (baseMethod, generator, moveDescs) {

            // deleting anchor cells will delete comments implicitly, just create the restore operations
            var newAnchor = MoveDescriptor.transformAddress(anchor, moveDescs);
            if (!newAnchor) { return this.generateRestoreOperations(generator); }

            // let the base class check if the entire comment frame will be deleted
            return baseMethod.call(this, generator, moveDescs);
        });

        /**
         * Generates the operations, and the undo operations, to update the
         * position of the drawing frame of this cell comment, while moving it
         * to a new anchor cell.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Address} newAnchor
         *  The new anchor position of this cell comment.
         *
         * @returns {SheetCommentModel}
         *  A reference to this instance.
         */
        this.generateMoveAnchorOperations = function (generator, newAnchor) {

            // resolve the current anchor rectangle (expand to merged ranges)
            var oldAnchorRect = this.getAnchorRectangle({ pixel: true, expandMerged: true });
            // resolve the new anchor rectangle (expand to merged ranges)
            var newAnchorRect = sheetModel.getCellRectangle(newAnchor, { pixel: true, expandMerged: true });
            // the difference of the anchor positions (top-right edges of the anchor cells)
            var moveX = newAnchorRect.right() - oldAnchorRect.right();
            var moveY = newAnchorRect.top - oldAnchorRect.top;

            // calculate the new location of the drawing frame
            var oldFrameRect = this.getRectangle();
            var newFrameRect = oldFrameRect.clone().translateSelf(moveX, moveY);

            // keep new position of the drawing frame inside sheet area
            var sheetRect = sheetModel.getSheetRectangle({ pixel: true });
            newFrameRect.left = Utils.minMax(newFrameRect.left, 0, sheetRect.right() - newFrameRect.width);
            newFrameRect.top = Utils.minMax(newFrameRect.top, 0, sheetRect.bottom() - newFrameRect.height);

            // generate the operations to update the location of the drawing frame
            var anchorAttrs = parentCollection.generateAnchorAttributes(this, { rectangle: newFrameRect });
            if (anchorAttrs) { this.generateChangeOperations(generator, { drawing: anchorAttrs }); }

            return this;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sheetModel = parentCollection = null;
        });

    } }); // class SheetCommentModel

    // exports ================================================================

    return SheetCommentModel;

});
