/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/commentcollection', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/sortedarray',
    'io.ox/office/drawinglayer/model/drawingcollection',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/drawingcollectionmixin',
    'io.ox/office/spreadsheet/model/drawing/commentmodel'
], function (Iterator, SortedArray, DrawingCollection, Operations, SheetUtils, SheetDrawingCollectionMixin, SheetCommentModel) {

    'use strict';

    // convenience shortcuts
    var TransformIterator = Iterator.TransformIterator;
    var FilterIterator = Iterator.FilterIterator;
    var NestedIterator = Iterator.NestedIterator;
    var MergeMode = SheetUtils.MergeMode;
    var Address = SheetUtils.Address;
    var AddressSet = SheetUtils.AddressSet;
    var RangeArray = SheetUtils.RangeArray;

    // private global functions ===============================================

    /**
     * Converts the passed anchor address of a cell comment to its drawing
     * position as used in document operations.
     *
     * @param {Address} address
     *  The anchor address of a cell comment.
     *
     * @returns {Array<Number>}
     *  The drawing position of the cell comment used in document operations.
     */
    function addressToPosition(address) {
        return address.toJSON();
    }

    /**
     * Converts the passed drawing position as used in document operations to
     * the anchor address of a cell comment.
     *
     * @param {Array<Number>} position
     *  The drawing position of a cell comment used in document operations.
     *
     * @returns {Address|Null}
     *  The anchor address of the cell comment, if the passed drawing position
     *  is valid; otherwise null.
     */
    function positionToAddress(position) {
        return (position.length === 2) ? new Address(position[0], position[1]) : null;
    }

    /**
     * Sorter callback function for SortedArray.
     */
    function modelSorter(commentModel) {
        var anchor = commentModel.getAnchor();
        return anchor[1] * 1e6 + anchor[0];
    }

    // class CommentCollection ================================================

    /**
     * Represents the collection of cell comments of a single sheet of a
     * spreadsheet document.
     *
     * @constructor
     *
     * @extends DrawingCollection
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var CommentCollection = DrawingCollection.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // document model
        var docModel = sheetModel.getDocModel();

        // all comment models, mapped by anchor cell address
        var addressSet = new AddressSet();

        // all comment models, sorted by their anchor address (row by row)
        var commentModels = new SortedArray(modelSorter);

        // base constructors --------------------------------------------------

        DrawingCollection.call(this, docModel, modelFactory, {
            clientResolveModelHandler: resolveCommentModel,
            clientResolvePositionHandler: resolveCommentPosition,
            clientResolveIndexHandler: resolveCommentIndex,
            clientRegisterModelHandler: registerCommentModel,
            clientUnregisterModelHandler: unregisterCommentModel
        });
        SheetDrawingCollectionMixin.call(this, sheetModel);

        // private methods ----------------------------------------------------

        /**
         * Constructs a comment model instance.
         *
         * @param {DrawingCollection} parentCollection
         *  The parent drawing collection that will contain the new comment.
         *
         * @param {String} drawingType
         *  The type of the drawing model to be created. The value 'comment' is
         *  expected by this implementation.
         *
         * @param {Object|Null} attributeSet
         *  Initial formatting attributes for the comment model.
         *
         * @param {Object} settings
         *  Additional comment settings.
         *
         * @returns {SheetCommentModel|Null}
         *  The new comment model instance, if the passed type is valid.
         */
        function modelFactory(parentCollection, drawingType, attributeSet, settings) {
            return (drawingType === 'comment') ? new SheetCommentModel(sheetModel, parentCollection, attributeSet, settings) : null;
        }

        /**
         * Returns the drawing object position of the passed comment model (the
         * serialized anchor cell address) as used in document operations.
         *
         * @param {SheetCommentModel} commentModel
         *  The comment model whose position will be returned.
         *
         * @returns {Array<Number>}
         *  The drawing object position of the passed comment model.
         */
        function resolveCommentPosition(commentModel) {
            // use the JSON representation of an address (array with two integer elements)
            return addressToPosition(commentModel.getAnchor());
        }

        /**
         * Returns the Z order index of the specified comment model.
         *
         * @param {SheetCommentModel} commentModel
         *  The comment model whose Z index will be returned.
         *
         * @param {Number|Null}
         *  The Z order index of the passed comment model, if it is part of
         *  this collection; otherwise null.
         */
        function resolveCommentIndex(commentModel) {
            var desc = commentModels.find(commentModels.index(commentModel), 'exact');
            return desc ? desc._ai : null;
        }

        /**
         * Returns the comment model addressed by the passed drawing object
         * position (the serialized anchor cell address) as used in document
         * operations.
         *
         * @param {Array<Number>} position
         *  The drawing object position to be resolved.
         *
         * @returns {SheetCommentModel|Null}
         *  The comment model if available, otherwise null.
         */
        function resolveCommentModel(position) {
            var address = positionToAddress(position);
            var element = address ? addressSet.get(address) : null;
            return element ? element.model : null;
        }

        /**
         * Registers the passed comment model after it has been inserted into
         * this drawing collection.
         *
         * @param {SheetCommentModel} commentModel
         *  The new comment model to be registered.
         *
         * @returns {Boolean}
         *  Whether the comment model has been registered successfully.
         */
        function registerCommentModel(commentModel) {

            // comments must have unique anchor addresses
            var element = commentModel.getAnchor().clone();
            if (addressSet.has(element)) { return false; }

            // add model to the address, insert it into the address set
            element.model = commentModel;
            addressSet.insert(element);

            // insert the comment model into the sorted array
            commentModels.insert(commentModel);

            return true;
        }

        /**
         * Unregisters the passed comment model after it has been removed from
         * this drawing collection.
         *
         * @param {SheetCommentModel} commentModel
         *  The comment model to be unregistered.
         *
         * @returns {Boolean}
         *  Whether the comment model has been unregistered successfully.
         */
        function unregisterCommentModel(commentModel) {

            // remove the comment model from the sorted array
            commentModels.remove(commentModel);

            // remove the comment model from the address set
            return addressSet.remove(commentModel.getAnchor());
        }

        /**
         * Recalculates the position of all cell comments, after cells have
         * been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {

            // no preparation work needed without comment models
            if (addressSet.empty()) { return; }

            // collect comments to be moved or deleted (do not alter the container while iterating)
            var movedModels = [];
            var deletedModels = [];

            // when inserting, visit the comments in reversed order to prevent overlapping anchor addresses
            var iterator = addressSet.rangeIterator(addressSet.boundary(), { reverse: moveDesc.insert });
            Iterator.forEach(iterator, function (element) {
                var newAnchor = moveDesc.transformAddress(element);
                if (!newAnchor) {
                    deletedModels.push(element.model);
                } else if (newAnchor.differs(element)) {
                    movedModels.push({ model: element.model, target: newAnchor });
                }
            });

            // first, delete all models (do not move comments over comments to be deleted)
            deletedModels.forEach(function (commentModel) {
                self.deleteModel(resolveCommentPosition(commentModel));
            });

            // move all comment models to their new position
            movedModels.forEach(function (entry) {

                var commentModel = entry.model;
                var oldAnchor = commentModel.getAnchor();
                var newAnchor = entry.target;

                // update the anchor address, and the comment model containers
                unregisterCommentModel(commentModel);
                commentModel.setAnchor(newAnchor);
                registerCommentModel(commentModel);

                // moved anchor address will be notified with a 'move:drawing' event
                self.trigger('move:drawing', commentModel, oldAnchor);
            });
        }

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * cell comments from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {CommentCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // clone the contents of the source collection
            var iterator = collection.createModelIterator();
            Iterator.forEach(iterator, function (commentModel) {
                var cloneModel = commentModel.clone(sheetModel, this);
                this.implInsertModel(resolveCommentPosition(cloneModel), cloneModel);
            }, this);
        };

        /**
         * Callback handler for the document operation 'insertComment'. Creates
         * and stores a new comment, and triggers an 'insert:drawing' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertComment' document operation.
         */
        this.applyInsertCommentOperation = function (context) {

            // multiple comments must not be located in the same cell
            var anchor = context.getJSONAddress();
            context.ensure(!this.getByAddress(anchor), 'duplicate anchor address %s', anchor);

            // the formatting attributes of the comment drawing frame
            var attrSet = context.getOptObj('attrs');

            // additional settings for the comment model
            var settings = {
                anchor: anchor,
                author: context.getOptStr('author'),
                date: context.getOptStr('date'),
                text: context.getOptStr('text', '', true)
            };

            // create the comment model (via collection base class)
            var commentModel = this.insertModel(addressToPosition(anchor), 'comment', attrSet, settings);
            context.ensure(commentModel, 'cannot insert comment into cell %s', anchor);
        };

        /**
         * Callback handler for the document operation 'deleteComment'. Deletes
         * an existing comment, and triggers a 'delete:drawing' event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteComment' document operation.
         */
        this.applyDeleteCommentOperation = function (context) {
            var anchor = context.getJSONAddress();
            var result = this.deleteModel(addressToPosition(anchor));
            context.ensure(result, 'cannot delete comment from cell %s', anchor);
        };

        /**
         * Callback handler for the document operation 'changeComment'. Changes
         * the settings of an existing comment, and triggers a 'change:drawing'
         * event.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeComment' document operation.
         */
        this.applyChangeCommentOperation = function (context) {

            // operation must refer to an existing cell comment
            var anchor = context.getJSONAddress();
            var commentModel = self.getByAddress(anchor);
            context.ensure(commentModel, 'no comment found in cell %s', anchor);

            // set the new formatting attributes
            if (context.has('attrs')) {
                commentModel.setAttributes(context.getObj('attrs'));
            }

            // set the new text contents of the comment
            if (context.has('text') && commentModel.setText(context.getStr('text', true))) {
                this.trigger('change:drawing:text', commentModel);
            }
        };

        /**
         * Callback handler for the document operation "moveComments". Moves
         * the anchors of multiple comments to new cell addresses, and triggers
         * 'change:drawing' events.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeComment' document operation.
         */
        this.applyMoveCommentsOperation = function (context) {

            // get the source and target anchor addresses
            var fromAnchors = context.getAddressList('from');
            var toAnchors = context.getAddressList('to');
            context.ensure(fromAnchors.length === toAnchors.length, 'anchor from/to count mismatch');

            // collect all comment models before inserting them to prevent address collisions
            var movedModels = [];

            // move all comments to their destination anchors
            fromAnchors.forEach(function (fromAnchor, index) {

                // look up the comment model at the old cell anchor
                var commentModel = this.getByAddress(fromAnchor);
                context.ensure(commentModel, 'no comment found in cell %s', fromAnchor);

                // get the target anchor position (nothing to do if comment does not move)
                var toAnchor = toAnchors[index];
                if (fromAnchor.equals(toAnchor)) { return; }

                // remove comment model from internal collections, collect all models
                unregisterCommentModel(commentModel);
                movedModels.push({ model: commentModel, target: toAnchor });
            }, this);

            // insert all comment models at their new anchor positions
            movedModels.forEach(function (entry) {

                var commentModel = entry.model;
                var oldAnchor = commentModel.getAnchor();
                var newAnchor = entry.target;

                // update the anchor address, and the comment model containers
                commentModel.setAnchor(newAnchor);
                registerCommentModel(commentModel);

                // moved anchor address will be notified with a 'move:drawing' event
                this.trigger('move:drawing', commentModel, oldAnchor);
            }, this);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns an iterator for the comment models in this collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {RangeArray|Range} [options.ranges]
         *      An array with addresses of cell ranges, or the address of a
         *      single cell range to search for comments. If omitted, visits
         *      all comments in the sheet.
         *  - {String} [options.direction='auto']
         *      Can be set to one of the values 'horizontal' or 'vertical' to
         *      force to create an iterator into the specified direction. The
         *      value 'horizontal' will cause to visit the comments from first
         *      to last row, and in each row from left to right. The value
         *      'vertical' will cause to visit the comments from first to last
         *      column, and in each row from top to bottom. If omitted or set
         *      to another value, the direction will be picked automatically
         *      according to the contents of this collection for optimal
         *      performance.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the order of the visited comments will be
         *      reversed (e.g. from bottom row to top row, and in each row from
         *      right to left).
         *  - {Boolean} [options.visible]
         *      If set to a boolean value, the iterator will skip comments with
         *      a visibility state (formatting attribute) different from the
         *      option value, regardless of their position in hidden or visible
         *      columns/rows. If omitted, the visibility state of the comments
         *      will be ignored.
         *  - {Boolean} [options.visibleCells=false]
         *      If set to true, the iterator will skip comments contained in
         *      hidden columns or hidden rows.
         *
         * @returns {Iterator}
         *  The model iterator providing the models of the cell comments as
         *  result values.
         */
        this.createModelIterator = function (options) {

            // create an array iterator for the cell range addresses to iterate in
            var ranges = (options && options.ranges) || docModel.getSheetRange();
            var iterator = RangeArray.iterator(ranges, options);

            // create a nested iterator for the comment models
            iterator = new NestedIterator(iterator, function (range) {
                return addressSet.rangeIterator(range, options);
            });

            // convert the address set entries to comment models
            iterator = new TransformIterator(iterator, 'model');

            // create an iterator that filters for the "visible" option
            var visible = options ? options.visible : null;
            if (typeof visible === 'boolean') {
                iterator = new FilterIterator(iterator, function (commentModel) {
                    return commentModel.isVisible() === visible;
                });
            }

            // create an iterator that filters for the "visibleCells" option
            if (options && options.visibleCells) {
                iterator = new FilterIterator(iterator, function (commentModel) {
                    return commentModel.getAnchorRectangle({ expandMerged: true }).area() > 0;
                });
            }

            return iterator;
        };

        /**
         * Returns whether this collection contains any cell comments.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.visible]
         *      If set to a boolean value, only comments with a visibility
         *      state (formatting attribute) equal to the option value will be
         *      considered, regardless of their position in hidden or visible
         *      columns/rows. If omitted, the visibility state of the comments
         *      will be ignored.
         *  - {Boolean} [options.visibleCells=false]
         *      If set to true, only comments contained in visible columns and
         *      rows will be considered.
         *
         * @returns {Boolean}
         *  Whether this collection contains any cell comments.
         */
        this.hasComments = function (options) {
            return !this.createModelIterator(options).next().done;
        };

        /**
         * Returns the comment model anchored at the specified cell address.
         *
         * @param {Address} address
         *  The address of the comment's anchor cell.
         *
         * @returns {SheetCommentModel|Null}
         *  The model of the cell comment anchored at the specified cell, if
         *  existing; otherwise null.
         */
        this.getByAddress = function (address) {
            var element = addressSet.get(address);
            return element ? element.model : null;
        };

        // generator methods --------------------------------------------------

        /**
         * Generates the operations, and the undo operations, to delete all
         * cell comments from this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateDeleteAllCommentsOperations = function (generator) {
            return this.iterateSliced(addressSet, function (element) {
                element.model.generateDeleteOperations(generator);
            }, 'CommentCollection.generateDeleteAllCommentsOperations');
        };

        /**
         * Generates the operations, and the undo operations, to show or hide
         * all cell comments from this collection. If the collection contains a
         * visible comment, all comments will be hidden; otherwise all comments
         * will be shown.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the operations have been
         *  generated successfully, or that will be rejected on any error.
         */
        this.generateToggleAllCommentsOperations = function (generator) {

            // if there is a visible comment in this collection, all comments will be hidden
            var hideComments = this.hasComments({ visible: true });
            // the properties for the change operations
            var attrSet = { drawing: { hidden: hideComments } };

            return this.iterateSliced(addressSet, function (element) {
                if (element.model.isVisible() === hideComments) {
                    element.model.generateChangeOperations(generator, attrSet);
                }
            }, 'CommentCollection.generateToggleAllCommentsOperations');
        };

        /**
         * Generates the operations, and the undo operations, to update the
         * cell comments after merging the specified cell ranges. All cell
         * comments covered by a merged range will be deleted.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} mergedRanges
         *  An array of cell range addresses, or a single cell range address
         *  that will be merged.
         *
         * @param {MergeMode} mergeMode
         *  The merge mode for the specified ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMergeCellsOperations = function (generator, mergedRanges, mergeMode) {

            // nothing to do without comments, or when unmerging a cell range
            if (addressSet.empty() || (mergeMode === MergeMode.UNMERGE)) {
                return this.createResolvedPromise();
            }

            // collect comments to be deleted (do not alter the container while iterating)
            var deletedModels = [];

            // process all passed merged ranges
            var promise = this.iterateArraySliced(RangeArray.get(mergedRanges), function (mergedRange) {
                var iterator = addressSet.rangeIterator(mergedRange);
                Iterator.forEach(iterator, function (element) {
                    if (!SheetUtils.isOriginCell(element, mergedRange, mergeMode)) {
                        deletedModels.push(element.model);
                    }
                });
            }, 'CommentCollection.generateMergeCellsOperations');

            // generate all operations to delete the comments
            promise = promise.then(function () {
                return self.iterateArraySliced(deletedModels, function (commentModel) {
                    return commentModel.generateDeleteOperations(generator);
                }, 'CommentCollection.generateMergeCellsOperations');
            });

            return promise;
        };

        /**
         * Generates the operations, and the undo operations, to move multiple
         * cell comments to new anchor cells. The positions of the drawing
         * frames will be updated relative to the new anchor cells.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {AddressArray} fromAddresses
         *  An array of cell addresses specifying the current anchor position
         *  of the cell comments.
         *
         * @param {AddressArray} toAddresses
         *  An array of cell addresses specifying the new anchor position of
         *  the cell comments. This array MUST contain exactly the same number
         *  of cell addresses as parameter "fromAddresses".
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveAnchorOperations = SheetUtils.profileAsyncMethod('CommentCollection.generateMoveAnchorOperations()', function (generator, fromAddresses, toAddresses) {

            // the address properties for the "moveComments" operation
            var fromStr = fromAddresses.toString(' ');
            var toStr = toAddresses.toString(' ');

            // generate the "moveComments" undo operation before updating the drawing frames
            generator.generateSheetOperation(Operations.MOVE_COMMENTS, { from: toStr, to: fromStr }, { undo: true });

            // update the position of all comment drawing frames
            var promise = this.iterateArraySliced(fromAddresses, function (fromAddress, index) {
                var commentModel = self.getByAddress(fromAddress);
                if (!commentModel) { return SheetUtils.makeRejected('operation'); }
                commentModel.generateMoveAnchorOperations(generator, toAddresses[index]);
            }, 'CommentCollection.generateMoveAnchorOperations');

            // generate the "moveComments" operation after updating the drawing frames
            promise = promise.then(function () {
                generator.generateSheetOperation(Operations.MOVE_COMMENTS, { from: fromStr, to: toStr });
            });

            return promise;
        });

        // event listeners ----------------------------------------------------

        // update position of cell anchors when moving cells in the sheet
        this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        // destroy all class members
        this.registerDestructor(function () {
            // base class DrawingCollection owns the comment models
            addressSet.clear();
            commentModels.clear();
            self = docModel = sheetModel = addressSet = commentModels = null;
        });

    } }); // class CommentCollection

    // exports ================================================================

    return CommentCollection;

});
