/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellstylecollection', [
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Border, StyleCollection, SheetUtils) {

    'use strict';

    // class CellStyleCollection ==============================================

    /**
     * Contains the cell style sheets of a spreadsheet document.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    var CellStyleCollection = StyleCollection.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // special behavior for OOXML files
        var ooxml = docModel.getApp().isOOXML();

        // the identifier of the built-in 'Hyperlink' style
        var hyperlinkStyleId = null;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'cell', { poolId: SheetUtils.CELL_POOL_ID, families: 'character apply' });

        // private methods ----------------------------------------------------

        /**
         * Creates a new style sheet, if not existing yet. Updates the category
         * and priority of existing and created style sheets.
         */
        function createStyleSheet(styleId, category, priority, attributeSet, applyFlags) {

            // update category and priority in existing style sheets
            if (self.containsStyleSheet(styleId)) {
                self.setStyleOptions(styleId, { category: category, priority: priority, builtIn: true });
                return;
            }

            // add the apply flags for OOXML files only
            if (ooxml && applyFlags) {
                attributeSet = _.clone(attributeSet);
                attributeSet.apply = applyFlags;
            }

            // create a new (dirty) style sheet, use name as identifier
            self.createDirtyStyleSheet(styleId, styleId, attributeSet, {
                category: category,
                priority: priority,
                hidden: category === 'hidden',
                builtIn: true
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Creates all missing built-in cell styles.
         *
         * @internal
         *  Not intended to be called from any other code than the post-process
         *  handler of the application.
         *
         * @returns {CellStyleCollection}
         *  A reference to this instance.
         */
        this.createMissingStyles = function () {

            // the merged attribute set of the default style
            var attributes = this.getDefaultStyleAttributeSet();
            // the default font name of the document
            var defFontName = attributes.character.fontName;

            // JSON color helper functions
            function rgbColor(value) { return { type: 'rgb', value: value }; }
            var schemeColor = docModel.createSchemeColor.bind(docModel);

            // predefined colors
            var DARK1 = schemeColor('dark1', 0);
            var DARK2 = schemeColor('dark2', 0);
            var LIGHT1 = schemeColor('light1', 0);
            var ACCENT1 = schemeColor('accent1', 0);
            var ACCENT1_40 = schemeColor('accent1', 40);

            // JSON border helper functions
            function singleBorder(width, color) { return { style: 'single', width: width, color: color }; }
            function thinBorder(color) { return singleBorder(Border.THIN_WIDTH_HMM, color); }
            function mediumBorder(color) { return singleBorder(Border.MEDIUM_WIDTH_HMM, color); }
            function thickBorder(color) { return singleBorder(Border.THICK_WIDTH_HMM, color); }
            function doubleBorder(color) { return { style: 'double', width: Border.THICK_WIDTH_HMM, color: color }; }

            // predefined border values
            var LIGHT_GRAY_BORDER  = thinBorder(rgbColor('B2B2B2'));
            var MEDIUM_GRAY_BORDER = thinBorder(rgbColor('7F7F7F'));
            var DARK_GRAY_BORDER   = thinBorder(rgbColor('3F3F3F'));
            var DOUBLE_GRAY_BORDER = doubleBorder(rgbColor('3F3F3F'));

            // JSON attribute helper functions
            function charAttrs(color, attrs) { return _.extend({ fontName: defFontName, color: color }, attrs); }
            function fillAttrs(color, border) { return _.extend({ fillType: 'solid', fillColor: color }, border ? { borderTop: border, borderBottom: border, borderLeft: border, borderRight: border } : null); }

            // predefined combinations for apply flags (default value for apply flags is true)
            var APPLY_FONT             = { fill: false, border: false, align: false, number: false, protect: false };
            var APPLY_FONT_FILL        = { border: false, align: false, number: false, protect: false };
            var APPLY_FONT_BORDER      = { fill: false, align: false, number: false, protect: false };
            var APPLY_FONT_FILL_BORDER = { align: false, number: false, protect: false };
            var APPLY_FILL_BORDER      = { font: false, align: false, number: false, protect: false };

            // standard styles
            createStyleSheet('Standard', 'markup', 1, {});
            createStyleSheet('Positive', 'markup', 2, { character: charAttrs(rgbColor('006100')), cell: fillAttrs(rgbColor('C6EFCE')) }, APPLY_FONT_FILL);
            createStyleSheet('Neutral',  'markup', 3, { character: charAttrs(rgbColor('9C6500')), cell: fillAttrs(rgbColor('FFEB9C')) }, APPLY_FONT_FILL);
            createStyleSheet('Negative', 'markup', 4, { character: charAttrs(rgbColor('9C0006')), cell: fillAttrs(rgbColor('FFC7CE')) }, APPLY_FONT_FILL);
            createStyleSheet('Note',     'markup', 5, { character: charAttrs(DARK1),              cell: fillAttrs(rgbColor('FFFFCC'), LIGHT_GRAY_BORDER) }, APPLY_FILL_BORDER);

            // heading styles
            createStyleSheet('Title',     'headings', 1, { character: charAttrs(DARK2, { bold: true, fontSize: 18 }) }, APPLY_FONT);
            createStyleSheet('Heading 1', 'headings', 2, { character: charAttrs(DARK2, { bold: true, fontSize: 15 }), cell: { borderBottom: thickBorder(ACCENT1) } },     APPLY_FONT_BORDER);
            createStyleSheet('Heading 2', 'headings', 3, { character: charAttrs(DARK2, { bold: true, fontSize: 13 }), cell: { borderBottom: thickBorder(ACCENT1_40) } },  APPLY_FONT_BORDER);
            createStyleSheet('Heading 3', 'headings', 4, { character: charAttrs(DARK2, { bold: true, fontSize: 11 }), cell: { borderBottom: mediumBorder(ACCENT1_40) } }, APPLY_FONT_BORDER);
            createStyleSheet('Heading 4', 'headings', 5, { character: charAttrs(DARK2, { bold: true, fontSize: 11 }) }, APPLY_FONT);
            createStyleSheet('Total',     'headings', 9, { character: charAttrs(DARK1, { bold: true, fontSize: 11 }), cell: { borderTop: thinBorder(ACCENT1), borderBottom: doubleBorder(ACCENT1) } }, APPLY_FONT_BORDER);

            // other content styles
            createStyleSheet('Output',           'hidden', 1, { character: charAttrs(rgbColor('3F3F3F'), { bold: true }), cell: fillAttrs(rgbColor('F2F2F2'), DARK_GRAY_BORDER) },   APPLY_FONT_FILL_BORDER);
            createStyleSheet('Calculation',      'hidden', 2, { character: charAttrs(rgbColor('FA7D00'), { bold: true }), cell: fillAttrs(rgbColor('F2F2F2'), MEDIUM_GRAY_BORDER) }, APPLY_FONT_FILL_BORDER);
            createStyleSheet('Input',            'hidden', 3, { character: charAttrs(rgbColor('3F3F76')),                 cell: fillAttrs(rgbColor('FFCC99'), MEDIUM_GRAY_BORDER) }, APPLY_FONT_FILL_BORDER);
            createStyleSheet('Explanatory Text', 'hidden', 4, { character: charAttrs(rgbColor('7F7F7F'), { italic: true }) }, APPLY_FONT);
            createStyleSheet('Linked Cell',      'hidden', 5, { character: charAttrs(rgbColor('FA7D00')), cell: { borderBottom: doubleBorder(rgbColor('FA7D00')) } }, APPLY_FONT_BORDER);
            createStyleSheet('Warning Text',     'hidden', 6, { character: charAttrs(rgbColor('FF0000')) }, APPLY_FONT);
            createStyleSheet('Check Cell',       'hidden', 7, { character: charAttrs(LIGHT1, { bold: true }), cell: fillAttrs(rgbColor('A5A5A5'), DOUBLE_GRAY_BORDER) }, APPLY_FONT_FILL_BORDER);

            // accent styles
            for (var index = 0; index < 6; index += 1) {
                var styleName = 'Accent ' + (index + 1);
                var colorName = 'accent' + (index + 1);
                createStyleSheet('20% - ' + styleName, 'themes', index,      { character: charAttrs(DARK1),  cell: fillAttrs(schemeColor(colorName, 80)) }, APPLY_FONT_FILL);
                createStyleSheet('40% - ' + styleName, 'themes', index + 6,  { character: charAttrs(DARK1),  cell: fillAttrs(schemeColor(colorName, 60)) }, APPLY_FONT_FILL);
                createStyleSheet('60% - ' + styleName, 'themes', index + 12, { character: charAttrs(LIGHT1), cell: fillAttrs(schemeColor(colorName, 40)) }, APPLY_FONT_FILL);
                createStyleSheet(styleName,            'themes', index + 18, { character: charAttrs(LIGHT1), cell: fillAttrs(schemeColor(colorName, 0)) },  APPLY_FONT_FILL);
            }

            // hyperlink style
            createStyleSheet('Hyperlink', 'hidden', 99, { character: charAttrs(docModel.createHlinkColor(), { underline: true }) }, APPLY_FONT);
            hyperlinkStyleId = this.getStyleIdByName('Hyperlink');

            return this;
        };

        /**
         * Returns the identifier of the built-in 'Hyperlink' cell style sheet.
         *
         * @returns {String}
         *  The identifier of the built-in 'Hyperlink' cell style sheet.
         */
        this.getHyperlinkStyleId = function () {
            return hyperlinkStyleId;
        };

        // initialization -----------------------------------------------------

        // destroy class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } }); // class CellStyleCollection

    // exports ================================================================

    return CellStyleCollection;

});
