/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/indexset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/subtotalresult',
    'io.ox/office/spreadsheet/model/usedrangecollection',
    'io.ox/office/spreadsheet/model/cellmodel',
    'io.ox/office/spreadsheet/model/cellmodelmatrix',
    'io.ox/office/spreadsheet/model/celloperationsbuilder',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, DateUtils, Iterator, IndexSet, ValueMap, Border, AttributeUtils, HyperlinkUtils, Operations, SheetUtils, SubtotalResult, UsedRangeCollection, CellModel, CellModelMatrix, CellOperationsBuilder, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var SingleIterator = Iterator.SingleIterator;
    var ArrayIterator = Iterator.ArrayIterator;
    var ObjectIterator = Iterator.ObjectIterator;
    var GeneratorIterator = Iterator.GeneratorIterator;
    var TransformIterator = Iterator.TransformIterator;
    var FilterIterator = Iterator.FilterIterator;
    var ReduceIterator = Iterator.ReduceIterator;
    var NestedIterator = Iterator.NestedIterator;
    var SerialIterator = Iterator.SerialIterator;
    var OrderedIterator = Iterator.OrderedIterator;
    var ParallelIterator = Iterator.ParallelIterator;
    var Direction = SheetUtils.Direction;
    var MergeMode = SheetUtils.MergeMode;
    var UpdateMode = SheetUtils.UpdateMode;
    var SortDirection = SheetUtils.SortDirection;
    var ErrorCode = SheetUtils.ErrorCode;
    var Interval = SheetUtils.Interval;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var IntervalArray = SheetUtils.IntervalArray;
    var AddressArray = SheetUtils.AddressArray;
    var RangeArray = SheetUtils.RangeArray;
    var AddressSet = SheetUtils.AddressSet;
    var RangeSet = SheetUtils.RangeSet;
    var ChangeDescriptor = SheetUtils.ChangeDescriptor;
    var MoveDescriptor = SheetUtils.MoveDescriptor;
    var MathUtils = FormulaUtils.Math;
    var Scalar = FormulaUtils.Scalar;

    // regular expression to match a leading integer in a string
    var LEADING_INT_RE = /^(\d+)(.*)$/i;

    // regular expression to match a trailing integer in a string
    var TRAILING_INT_RE = /^(.*?)(\d+)$/i;

    // private global functions ===============================================

    /**
     * Returns the visibility flags for columns and rows according to the
     * option 'visible' supported by various methods of a cell collection.
     *
     * @param {Object} options
     *  Optional parameters passed to a method of class CellCollection with a
     *  'visible' option that can be a boolean, or one of the strings 'columns'
     *  or 'rows'.
     *
     * @returns {Object}
     *  An object with the boolean flags 'cols' and 'rows' specifying whether
     *  to visit visible columns and/or visible rows only.
     */
    function getVisibleMode(options) {
        var mode = Utils.getOption(options, 'visible');
        return { cols: (mode === true) || (mode === 'columns'), rows: (mode === true) || (mode === 'rows') };
    }

    // class SharedModel ======================================================

    /**
     * A structure that is used to collect data about a shared formula in the
     * cell collection.
     *
     * @constructor
     *
     * @property {SheetModel} sheetModel
     *  The model of the sheet containing this shared formula.
     *
     * @property {Number} index
     *  The index of the shared formula, as used for the 'si' property of cell
     *  models.
     *
     * @property {Address|Null} anchorAddress
     *  The address of the anchor cell; or null, if the anchor cell has been
     *  detached from the shared formula (used e.g. as intermediate state while
     *  changing the cells in the cell collection, and updating the shared
     *  formula afterwards).
     *
     * @property {AddressSet} addressSet
     *  The addresses of all cells that are part of the shared formula.
     *
     * @property {Address|Null} refAddress
     *  The reference address used to parse the formula expression of the
     *  shared formula. This address remains intact when deleting the anchor
     *  address, in order to be able to generate a relocated formula expression
     *  for a new anchor address afterwards. Will be null for uninitialized
     *  model instances (no anchor cell with formula available).
     *
     * @property {TokenArray|Null} tokenArray
     *  The parsed formula expression of the shared formula. Will be null for
     *  uninitialized model instances (no anchor cell with formula available).
     */
    function SharedModel(sheetModel, index) {

        // the sheet model owning the shared formula
        this.sheetModel = sheetModel;

        // the shared index (will never change)
        this.index = index;

        // the address of the current anchor cell in the cell collection
        this.anchorAddress = null;

        // the addresses of all cells that are currently part of the shared formula
        this.addressSet = new AddressSet();

        // the reference address for the formula definition (may differ from the anchor address)
        this.refAddress = null;

        // the parsed formula expression for the reference address
        this.tokenArray = null;

    } // class SharedModel

    // public methods ---------------------------------------------------------

    /**
     * Inserts a new cell into this shared formula.
     *
     * @param {CellModel} cellModel
     *  The cell model to be inserted.
     */
    SharedModel.prototype.insertAddress = function (cellModel) {
        if (cellModel.isSharedAnchor()) {
            this.anchorAddress = this.refAddress = cellModel.a.clone();
            this.tokenArray = this.sheetModel.createCellTokenArray();
            this.tokenArray.parseFormula('op', cellModel.f || '', { refAddress: this.refAddress });
        }
        this.addressSet.insert(cellModel.a.clone());
    };

    /**
     * Removes a cell from this shared formula.
     *
     * @param {CellModel} cellModel
     *  The cell model to be removed.
     */
    SharedModel.prototype.removeAddress = function (cellModel) {
        if (this.anchorAddress && this.anchorAddress.equals(cellModel.a)) {
            this.anchorAddress = null;
        }
        this.addressSet.remove(cellModel.a);
    };

    /**
     * Returns the formula expression of this shared formula, relocated to the
     * specified target address.
     *
     * @param {String} grammarId
     *  The identifier of the formula grammar for the formula expression. See
     *  SpreadsheetDocument.getFormulaGrammar() for more details.
     *
     * @param {Address} targetAddress
     *  The target address to generate the formula expression for.
     *
     * @returns {String|Null}
     *  The formula expression for the specified target address; or null, is
     *  this instance is invalid (no reference address, no formula available).
     */
    SharedModel.prototype.getFormula = function (grammarId, targetAddress) {
        return this.refAddress ? this.tokenArray.getFormula(grammarId, { refAddress: this.refAddress, targetAddress: targetAddress }) : null;
    };

    // class CellCollection ===================================================

    /**
     * Collects cell contents and formatting attributes for a single sheet.
     * To save memory and to improve performance, instances of this class store
     * specific parts of the sheet only. More cell data will be fetched from
     * the server on demand.
     *
     * Triggers the following events:
     *  - 'change:cells'
     *      After the value, formula expression, or auto-style of one or more
     *      cells in this collection have been changed due to a 'changeCells'
     *      document operation. Event handler callbacks receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {ChangeDescriptor} changeDesc
     *          A descriptor object for the changed cells.
     *      (3) {Boolean} external
     *          Whether the changed cells originate from an external document
     *          operation received from the server.
     *      (4) {Boolean} results
     *          Whether this change event has been caused by the dependency
     *          manager while updating the results of formula cells.
     *  - 'move:cells'
     *      After some cells in this collection have been moved to a new
     *      position due to a 'moveCells', 'insertColumns', 'deleteColumns',
     *      'insertRows', or 'deleteRows' document operation. Event handler
     *      callbacks receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {MoveDescriptor} moveDesc
     *          A descriptor object for the moved cells.
     *      (3) {Boolean} external
     *          Whether the moved cells originate from an external document
     *          operation received from the server.
     *
     * @constructor
     *
     * @extends UsedRangeCollection
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    var CellCollection = UsedRangeCollection.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // the document model, and model containers
        var docModel = sheetModel.getDocModel();
        var numberFormatter = docModel.getNumberFormatter();
        var formulaGrammar = docModel.getFormulaGrammar('op');
        var formulaParser = docModel.getFormulaParser();
        var autoStyles = docModel.getCellAutoStyles();
        var listCollection = docModel.getListCollection();

        // the column/row collections of the active sheet
        var colCollection = sheetModel.getColCollection();
        var rowCollection = sheetModel.getRowCollection();

        // all cell models mapped by address key
        var modelMap = new ValueMap();

        // all cells, as sorted array of sorted columns
        var colMatrix = new CellModelMatrix(sheetModel, true);

        // all cells, as sorted array of sorted columns
        var rowMatrix = new CellModelMatrix(sheetModel, false);

        // the addresses of shared formulas, mapped by shared index
        var sharedModelMap = new ValueMap();

        // the range addresses of all matrix formulas
        var matrixRangeSet = new RangeSet();

        // special handling for ODF
        var odf = docModel.getApp().isODF();

        // the preferred date format for cell edit mode
        var DATE_FORMAT = numberFormatter.getPresetCode(14);
        // the preferred time format for cell edit mode
        var TIME_FORMAT = numberFormatter.getTimeCode({ seconds: true });

        // base constructor ---------------------------------------------------

        UsedRangeCollection.call(this, sheetModel, usedRangeResolver, { trigger: 'always' });

        // private methods ----------------------------------------------------

        /**
         * Calculates the used range of this collection (will be called from
         * base class).
         */
        function usedRangeResolver() {
            var usedCols = colMatrix.getUsedInterval();
            var usedRows = rowMatrix.getUsedInterval();
            return (usedCols && usedRows) ? Range.createFromIntervals(usedCols, usedRows) : null;
        }

        /**
         * Inserts the address of the passed cell model into the model of a
         * shared formula, if the cell refers to a shared formula; or registers
         * the bounding range of a matrix formula.
         */
        function implRegisterFormulaCell(cellModel) {

            // shared formulas: 'si' property contains shared index, 'sr' property denotes anchor cell
            if (cellModel.isSharedFormula()) {
                var sharedModel = sharedModelMap.getOrConstruct(cellModel.si, SharedModel, sheetModel, cellModel.si);
                sharedModel.insertAddress(cellModel);
            }

            // matrix formulas: 'mr' property denotes top-left cell
            if (cellModel.isMatrixAnchor()) {
                matrixRangeSet.insert(cellModel.mr.clone());
            }
        }

        /**
         * Removes the address of the passed cell model from the model of a
         * shared formula, if the cell refers to a shared formula; or removes
         * the bounding range of a matrix formula.
         */
        function implUnregisterFormulaCell(cellModel) {

            // shared formulas: 'si' property contains shared index, 'sr' property denotes anchor cell
            if (cellModel.isSharedFormula()) {
                sharedModelMap.with(cellModel.si, function (sharedModel) {
                    sharedModel.removeAddress(cellModel);
                });
            }

            // matrix formulas: 'mr' property denotes top-left cell
            if (cellModel.isMatrixAnchor()) {
                matrixRangeSet.remove(cellModel.mr);
            }
        }

        /**
         * Inserts the passed new cell model into the internal containers.
         */
        function implInsertCellModel(cellModel, skipColMatrix, skipRowMatrix) {
            modelMap.insert(cellModel.key(), cellModel);
            if (!skipColMatrix) { colMatrix.insertModel(cellModel); }
            if (!skipRowMatrix) { rowMatrix.insertModel(cellModel); }
            implRegisterFormulaCell(cellModel);
        }

        /**
         * Removes the passed new cell model from the internal containers.
         */
        function implRemoveCellModel(cellModel, skipColMatrix, skipRowMatrix) {
            implUnregisterFormulaCell(cellModel);
            modelMap.remove(cellModel.key());
            if (!skipColMatrix) { colMatrix.removeModel(cellModel); }
            if (!skipRowMatrix) { rowMatrix.removeModel(cellModel); }
        }

        /**
         * Creates a new blank cell model for this collection. The collection
         * MUST NOT contain an existing cell model at the specified address.
         *
         * @param {Address} address
         *  The address of the cell. The cell MUST be undefined.
         *
         * @returns {CellModel}
         *  A new blank cell model.
         */
        function createNewCellModel(address) {
            var cellModel = new CellModel(address);
            implInsertCellModel(cellModel);
            return cellModel;
        }

        /**
         * Removes a cell model from this collection.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Object}
         *  A flag set containing change flags for modified properties of the
         *  cell. The set will contain the following properties:
         *  - {Boolean} value
         *      Whether the deleted cell was not blank.
         *  - {Boolean} formula
         *      Whether the deleted cell contained a formula expression.
         *  - {Boolean} style
         *      Whether the auto-style of the deleted cell was different than
         *      the default column/row auto-style.
         */
        function deleteCellModel(address) {

            var cellModel = self.getCellModel(address);
            var changeFlags = { value: false, formula: false, style: false };

            if (cellModel) {
                changeFlags.value = cellModel.v !== null;
                changeFlags.formula = cellModel.isAnyFormula();
                changeFlags.style = !autoStyles.areEqualStyleIds(cellModel.s, self.getDefaultStyleId(address));
                implRemoveCellModel(cellModel);
            }
            return changeFlags;
        }

        /**
         * Updates the value, formula, and auto-style of a cell model.
         *
         * @param {Address} address
         *  The address of the cell to be updated.
         *
         * @param {Object} contents
         *  The properties of the cell to be changed, with the following
         *  optional properties:
         *  - {Number|String|Boolean|ErrorCode|Null} [contents.v]
         *      The new value (or formula result) of the cell.
         *  - {String|Null} [contents.f]
         *      The new formula expression for the cell, or null to remove an
         *      existing formula.
         *  - {Number|Null} [contents.si]
         *      The new index of a shared formula for the cell, or null to
         *      remove an existing index.
         *  - {Range|Null} [contents.sr]
         *      The new bounding range of a shared formula for the cell, or
         *      null to remove an existing bounding range from the anchor cell.
         *  - {Range|Null} [contents.mr]
         *      The new bounding range of a matrix formula starting at the
         *      cell, or null to remove an existing bounding range from the
         *      cell.
         *  - {String} [contents.s]
         *      The identifier of an auto-style. The empty string can be used
         *      for the default auto-style of the document.
         *
         * @param {Boolean} [importing=false]
         *  If set to true, special behavior for the document import process
         *  will be actiavted (for formula cells, the token array of the cell
         *  model will NOT be created).
         *
         * @returns {Object}
         *  A flag set containing change flags for modified properties of the
         *  cell. The set will contain the following properties:
         *  - {Boolean} value
         *      Whether the cell value (or formula result) has changed.
         *  - {Boolean} formula
         *      Whether the formula expression has been changed.
         *  - {Boolean} style
         *      Whether the auto-style identifier has been changed.
         */
        function updateCellModel(address, contents, importing) {

            // the cell model to be updated
            var cellModel = self.getCellModel(address);
            // the old auto-style (default to column/row auto style)
            var oldStyleId = cellModel ? cellModel.s : self.getDefaultStyleId(address);
            // the changed flags
            var changeFlags = { value: false, formula: false, style: false };

            // update cell value
            if ('v' in contents) {
                cellModel = cellModel || createNewCellModel(address);
                changeFlags.value = cellModel.setValue(contents.v);
            }

            // update cell formula
            if ('f' in contents) {
                cellModel = cellModel || createNewCellModel(address);
                if (cellModel.setFormula(sheetModel, contents.f, importing)) {
                    changeFlags.formula = true;
                }
            }

            // remove the address from the shared formula maps here, and add it
            // below after updating all related properties for shared formulas
            if (cellModel) { implUnregisterFormulaCell(cellModel); }

            // update the index of a shared formula
            if ('si' in contents) {
                cellModel = cellModel || createNewCellModel(address);
                if (cellModel.setSharedIndex(contents.si)) {
                    changeFlags.formula = true;
                }
            }

            // update the bounding range of a shared formula
            if ('sr' in contents) {
                cellModel = cellModel || createNewCellModel(address);
                cellModel.setSharedRange(contents.sr);
                // do not set the 'changeFlags.formula' flag (changing the bounding
                // range does not have any effect on the formula result)
            }

            // update the bounding range of a matrix formula
            if ('mr' in contents) {
                cellModel = cellModel || createNewCellModel(address);
                if (cellModel.setMatrixRange(contents.mr)) {
                    changeFlags.formula = true;
                }
            }

            // if the cell is part of a shared formula, add the address to the internal maps
            if (cellModel) { implRegisterFormulaCell(cellModel); }

            // update cell auto-style
            if ('s' in contents) {
                cellModel = cellModel || createNewCellModel(address);
                cellModel.s = contents.s;
                changeFlags.style = !autoStyles.areEqualStyleIds(oldStyleId, contents.s);
            }

            // update the parsed number format (also for new cell models without style change)
            if (cellModel && (changeFlags.style || !cellModel.pf)) {
                cellModel.pf = autoStyles.getParsedFormat(cellModel.s);
            }

            // update the display string, if value or attributes have changed
            // (performance: nothing to do if formula has changed, but the result has not)
            if (changeFlags.value || changeFlags.style) {
                cellModel.d = numberFormatter.formatValue(cellModel.pf, cellModel.v);
            }

            // delete the cached subtotal settings in the column/row matrixes, if the value has changed
            if (changeFlags.value) {
                cellModel.cv.clearSubtotals();
                cellModel.rv.clearSubtotals();
            }

            return changeFlags;
        }

        /**
         * Returns the model of the shared formula referred by the specified
         * cell model.
         *
         * @param {CellModel} cellModel
         *  The model of a cell that may be part of a shared formula.
         *
         * @returns {SharedModel|Null}
         *  The model of the shared formula referred by the specified cell; or
         *  null, if the specified cell is not part of a shared formula.
         */
        function getSharedModel(cellModel) {
            return cellModel.isSharedFormula() ? sharedModelMap.get(cellModel.si, null) : null;
        }

        /**
         * Returns a predicate function that maches cell models for specific
         * type specifiers.
         *
         * @param {String} type
         *  The cell type specifier. See option 'type' of the public method
         *  CellCollection.createAddressIterator() for details.
         *
         * @returns {Function}
         *  A predicate function that receives either null or an instance of
         *  CellModel, and return whether that input value matches the type
         *  specifier.
         */
        var getCellTypePredicate = (function () {

            var PREDICATE_FUNCS = {
                any:     _.constant(true),
                defined: _.identity,
                value:   function (cellModel) { return cellModel && !cellModel.isBlank(); },
                formula: function (cellModel) { return cellModel && cellModel.isAnyFormula(); },
                anchor:  function (cellModel) { return cellModel && cellModel.isAnyAnchor(); }
            };

            return function (type) {
                return PREDICATE_FUNCS[type] || PREDICATE_FUNCS.any;
            };
        }());

        /**
         * Creates an iterator that generates the cell addresses for all, or
         * specific, cells contained in the passed column and row intervals.
         * The addresses will be generated row-by-row across all passed column
         * intervals.
         *
         * @param {IntervalArray|Interval} colIntervals
         *  An array of column intervals, or a single column interval.
         *
         * @param {IntervalArray|Interval} rowIntervals
         *  An array of row intervals, or a single row interval.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.type='all']
         *      Specifies which type of cells will be covered by the iterator.
         *      See method CellCollection.createAddressIterator() for details.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the row intervals AND the column intervals in
         *      each row will be visited in reversed order.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {Address} value
         *      The cell address currently visited.
         *  - {CellModel|Null} model
         *      The cell model at the current address if existing; otherwise
         *      null.
         */
        function createAddresIteratorForIntervals(colIntervals, rowIntervals, options) {

            // which cell entries (by type) will be visited
            var type = Utils.getStringOption(options, 'type', 'all');
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            // the matcher predicate function for the cell type
            var matcher = getCellTypePredicate(type);

            // use the column or row matrix to visit the existing cell models faster
            // (test by checking whether the matcher does not accept missing cell models)
            if (!matcher(null)) {

                // the model iterator created by the cell matrix
                // TODO: use a parallel iterator in the column matrix for a small number of columns
                var modelIt = (colIntervals.size() === 1) ?
                    colMatrix.createCellIterator(colIntervals, rowIntervals, reverse) :
                    rowMatrix.createCellIterator(rowIntervals, colIntervals, reverse);

                // filter for specific cell types, and transform cell models to their addresses
                return new TransformIterator(modelIt, function (cellModel) {
                    return matcher(cellModel) ? { value: cellModel.a.clone(), model: cellModel } : null;
                });
            }

            // the outer iterator for all rows in the row intervals
            rowIntervals = IntervalArray.get(rowIntervals);
            var rowIt = rowIntervals.indexIterator({ reverse: reverse });
            // the generator callback for all columns in the column intervals
            colIntervals = IntervalArray.get(colIntervals);
            var colItGen = colIntervals.indexIterator.bind(colIntervals, { reverse: reverse });
            // create a nested iterator that visits all rows and columns, and returns a result with a cell address
            return new NestedIterator(rowIt, colItGen, function (rowResult, colResult) {
                var address = new Address(colResult.value, rowResult.value);
                return { value: address, model: self.getCellModel(address) };
            });
        }

        /**
         * Creates an iterator that generates the cell addresses for all, or
         * specific, cells contained in the passed cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.type='all']
         *      Specifies which type of cells will be covered by the iterator.
         *      See method CellCollection.createAddressIterator() for details.
         *  - {Boolean|String} [options.visible=false]
         *      Specifies how to handle hidden columns and rows. See method
         *      CellCollection.createAddressIterator() for details.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cell addresses in each range
         *      will be visited in reversed order.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {Address} value
         *      The cell address currently visited.
         *  - {CellModel|Null} model
         *      The cell model at the current address if existing; otherwise
         *      null.
         *  - {Range} orig
         *      The address of the original cell range (from the passed range
         *      array) containing the current cell address.
         *  - {Number} index
         *      The array index of the original cell range contained in the
         *      result property 'orig'.
         */
        function createAddressIterator(ranges, options) {

            // whether to visit visible columns and/or rows only
            var visible = getVisibleMode(options);
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);

            // creates the appropriate iterator for the passed cell range
            function createRangeIterator(range) {

                // the column intervals of the range to be visited (all columns, or visible columns only)
                var colIntervals = visible.cols ? colCollection.getVisibleIntervals(range.colInterval()) : new IntervalArray(range.colInterval());
                // the row intervals of the range to be visited (all rows, or visible rows only)
                var rowIntervals = visible.rows ? rowCollection.getVisibleIntervals(range.rowInterval()) : new IntervalArray(range.rowInterval());

                // create an iterator that visits the cells row-by-row through the column intervals
                return (colIntervals && rowIntervals) ? createAddresIteratorForIntervals(colIntervals, rowIntervals, options) : Iterator.EMPTY;
            }

            // combines the results of the outer and inner iterators
            function createIteratorResult(arrayResult, rangeResult) {
                return { value: rangeResult.value, model: rangeResult.model, orig: arrayResult.value, index: arrayResult.index };
            }

            // create the nested iterator, combining an outer array iterator with an inner address iterator for each range
            var iterator = RangeArray.get(ranges).iterator({ reverse: reverse });
            return new NestedIterator(iterator, createRangeIterator, createIteratorResult);
        }

        function createParallelAddressIterator(ranges, options) {

            var iterators = ranges.map(function (range) { return createAddressIterator(range, options); });

            var iterator = new ParallelIterator(iterators, function (address, result) {
                return result.orig.indexAt(address);
            });

            return new TransformIterator(iterator, function (addresses, result) {
                result.value = ranges.map(function (range) { return range.addressAt(result.offset); });
                return result;
            });
        }

        /**
         * Returns a callback function that converts a cell address to a
         * numeric sorting index that is oriented horizontally, intended to be
         * used by synchronized and parallel address iterators.
         *
         * @returns {Function}
         *  A callback function that converts a cell address to a numeric
         *  sorting index.
         */
        function createAddressIndexer() {

            // the number of columns in the sheet
            var colCount = docModel.getMaxCol() + 1;

            // convert the passed cell address to a row-oriented sort index
            return function (address) { return address[1] * colCount + address[0]; };
        }

        /**
         * Returns the new format code for a cell to be applied while changing
         * the contents of that cell.
         *
         * @param {Address} address
         *  The address of the cell to be changed.
         *
         * @param {ParsedFormat} parsedFormat
         *  The parsed format intended to be applied for the cell, e.g. as
         *  received after parsing an edit string, or after calculating the
         *  result of a formula.
         *
         * @returns {String|Null}
         *  The format code contained in the parsed format, if it needs to be
         *  applied to the specified cell; or null, if the cell's number format
         *  will be retained.
         */
        function getNewCellFormatCode(address, parsedFormat) {

            // ignore the standard number formats
            if (parsedFormat.isStandard()) { return null; }

            // ignore changes inside any of the date/time categories (e.g. do not change a date cell to a time cell)
            var oldParsedFormat = self.getParsedFormat(address);
            if (parsedFormat.hasEqualCategory(oldParsedFormat, { anyDateTime: true })) { return null; }

            // return the format code of the parsed format on success
            return parsedFormat.formatCode;
        }

        /**
         * Returns an object with various flags specifying how a cell will be
         * changed according to the passed cell content data.
         *
         * @param {Object} contents
         *  The cell content data to be examined.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Boolean} undefine
         *      Whether the cell will be undefined (physically deleted from
         *      this collection).
         *  - {Boolean} blank
         *      The cell will become blank (value and formula will be removed).
         *  - {Boolean} value
         *      The cell value will be changed, and the cell may not be blank
         *      afterwards (this flag will only be set, if 'blank' is false).
         *  - {Boolean} formula
         *      The cell formula will be changed, or the cell formula will be
         *      deleted without resulting in a blank cell (this flag will only
         *      be set, if 'blank' is false).
         *  - {Boolean} style
         *      The cell formatting attributes, or the auto-style will be
         *      changed.
         */
        function getChangeFlags(contents) {

            var undefine = contents.u === true;
            var blank = undefine || ((contents.v === null) && (contents.f === null));

            return {
                undefine: undefine,
                blank: blank,
                value: !blank && ('v' in contents),
                formula: !blank && (('f' in contents) || ('si' in contents) || ('sr' in contents)),
                style: !undefine && (('s' in contents) || ('a' in contents) || ('format' in contents) || ('table' in contents))
            };
        }

        /**
         * Divides the passed cell range addresses into column ranges, row
         * ranges, and other cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Object}
         *  A result descriptor with the following properties:
         *  - {RangeArray} [colRanges]
         *      The column ranges from the passed range array. The property
         *      will be missing, if the passed range array does not contain any
         *      column ranges.
         *  - {RangeArray} [rowRanges]
         *      The row ranges from the passed range array that are not ranges
         *      covering the entire sheet (considered to be column ranges). The
         *      property will be missing, if the passed range array does not
         *      contain any row ranges.
         *  - {RangeArray} [cellRanges]
         *      The remaining cell ranges from the passed range array that are
         *      neither column ranges nor row ranges. The property will be
         *      missing, if the passed range array does not contain any regular
         *      cell ranges.
         */
        function getRangeGroups(ranges) {
            return RangeArray.group(ranges, function (range) {
                return docModel.isColRange(range) ? 'colRanges' : docModel.isRowRange(range) ? 'rowRanges' : 'cellRanges';
            });
        }

        /**
         * Tags the passed cell range addresses with an additional 'rangeMode'
         * property specifying how to process the cell formatting of the range.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {RangeArray}
         *  Copies of all passed ranges, with additional 'rangeMode' properties
         *  as expected e.g. by the method generateCellContentOperations(). The
         *  row ranges will be split into row ranges covering rows with and
         *  without active custom auto-styles.
         */
        function getTaggedRanges(ranges) {

            // group the ranges into column ranges, row ranges, and other cell ranges
            var rangeGroups = RangeArray.group(ranges, function (range) {
                return docModel.isColRange(range) ? 'colRanges' : docModel.isRowRange(range) ? 'rowRanges' : 'cellRanges';
            });

            // the resulting ranges to be filled, with additional 'rangeMode' property
            var taggedRanges = new RangeArray();

            // Split row ranges into rows with and without custom auto-style. In rows without custom
            // auto-style, the new formatting attributes must be merged with column default formatting
            // (range mode 'row:merge'). Otherwise, only existing cells need to be processed in rows
            // that already contain an active custom auto-style (range mode 'row:def').
            if (rangeGroups.rowRanges) {
                // convert the ranges to row intervals, forward all existing fill data objects
                var rowIntervals = IntervalArray.map(rangeGroups.rowRanges, function (range) {
                    var interval = range.rowInterval();
                    interval.rangeMode = range.rangeMode;
                    interval.fillData = range.fillData;
                    return interval;
                });
                // split the rows into formatted and unformatted ranges, forward all existing fill data objects
                Iterator.forEach(rowCollection.createStyleIterator(rowIntervals), function (interval, result) {
                    var rowRange = docModel.makeRowRange(interval);
                    rowRange.rangeMode = result.orig.rangeMode || ((result.style === null) ? 'row:merge' : 'row:def');
                    rowRange.fillData = result.orig.fillData;
                    taggedRanges.push(rowRange);
                });
            }

            // Add the column ranges. New formatting attributes must be merged over active custom
            // auto-styles of the rows (range mode 'col:merge').
            if (rangeGroups.colRanges) {
                rangeGroups.colRanges.forEach(function (colRange) {
                    colRange.rangeMode = colRange.rangeMode || 'col:merge';
                });
                taggedRanges.append(rangeGroups.colRanges);
            }

            // Add the remaining ranges not covering entire columns or rows.
            if (rangeGroups.cellRanges) {
                taggedRanges.append(rangeGroups.cellRanges);
            }

            return taggedRanges;
        }

        /**
         * Returns the range addresses of the adjacent cells of all passed cell
         * ranges.
         */
        function getAdjacentRanges(ranges, columns, leading) {
            var maxIndex = leading ? 0 : docModel.getMaxIndex(columns);
            var checkRange = leading ? function (range) { return range.getStart(columns) > 0; } : function (range) { return range.getEnd(columns) < maxIndex; };
            var makeRange = leading ? function (range) { return range.lineRange(columns, -1); } : function (range) { return range.lineRange(columns, range.size(columns)); };
            return RangeArray.map(ranges, function (range) { return checkRange(range) ? makeRange(range) : null; });
        }

        /**
         * Generates the cell operations, and the undo operations, to change
         * the specified contents for multiple cells in the sheet, and
         * optionally the appropriate hyperlink operations.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Address} address
         *  The address of the top-left cell to be modified.
         *
         * @param {Array<Object>} matrix
         *  An array of row descriptors (with properties 'c' and 'r') to change
         *  multiple cells at once. See description of the public method
         *  CellCollection.generateCellContentOperations() for details.
         *  Additionally, the following internal properties are supported:
         *  - {String} [contents.rangeMode='cell:merge']
         *      Specifies which cells will be processed in the ranges covered
         *      by the contents matrix. The following values
         *      are supported:
         *      - 'cell:merge' (default):
         *          Process all cells covered by a cell range. Used for regular
         *          cell formatting that will be merged over column and row
         *          default styles.
         *      - 'row:def':
         *          Process only existing cells (do not create new cells). Used
         *          when formatting entire rows with own active default style.
         *      - 'row:merge':
         *          Merge attributes for undefined cells with column default
         *          styles. Used when formatting entire rows without own active
         *          default style.
         *      - 'col:merge':
         *          Merge attributes for undefined cells with active row
         *          default styles. Used when formatting entire columns.
         *  - {String} [contents.cacheKey]
         *      A unique key for the formatting attributes and auto-style
         *      carried in this cell contents object. If set, an internal cache
         *      will be filled with the resulting auto-style identifier, and
         *      subsequent cell entries with the same cache key and old
         *      auto-style will resolve the new auto-style from that cache,
         *      instead of trying to receive the auto-style identifier from the
         *      auto-style collection (performance optimization).
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the finalize method of
         *  the cell operations builders passed to the callback function. See
         *  CellOperationsBuilder.finalizeOperations() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the addresses of the ranges
         *  that have really been changed, after all operations have been
         *  generated.
         */
        var generateCellContentOperations = SheetUtils.profileAsyncMethod('CellCollection.generateCellContentOperations()', function (generator, address, matrix, options) {
            SheetUtils.log('address=' + address + ' matrix=', matrix);

            // the cell operation builder
            var operationsBuilder = new CellOperationsBuilder(sheetModel, generator, { startAddress: address });
            // cell range arrays per URL, will be merged together below
            var linkRangesMap = {};
            // the start address for the next row (or row band)
            var nextRowAddress = address.clone();
            // the number of cells already processed, for overflow detection
            var changedCount = 0;
            // the indexer for the synchronized address iterators
            var addressIndexer = createAddressIndexer();

            // creates an appropriate address iterator for the passed cell range and cell content properties
            function createIterators(range, contents) {

                // the change flags specifying how the cell will be potentially changed
                var changeFlags = getChangeFlags(contents);
                // special behavior when clearing values without new formatting
                var clearValues = changeFlags.blank && !changeFlags.style;

                function createIteratorForRange(iterRange, type) {
                    var repeatCols = type === 'all';
                    var colCount = repeatCols ? iterRange.cols() : 1;
                    var destRange = repeatCols ? iterRange.leadingCol() : iterRange;
                    var iterator = createAddressIterator(destRange, { type: type });
                    return new TransformIterator(iterator, function (address, result) {
                        result.contents = contents;
                        result.colCount = colCount;
                        result.countCells = !clearValues;
                        return result;
                    });
                }

                function createIteratorForIntervals(colIntervals, rowIntervals, type) {
                    var iterator = createAddresIteratorForIntervals(colIntervals, rowIntervals, { type: type });
                    return new TransformIterator(iterator, function (address, result) {
                        result.contents = contents;
                        result.colCount = 1;
                        result.countCells = !clearValues;
                        return result;
                    });
                }

                // undefine cells: visit all existing cell models, regardless of range mode (cells/columns/rows)
                if (changeFlags.undefine) { return createIteratorForRange(range, 'defined'); }

                // removing values, but not the formulas, is an illegal operation
                if ((contents.v === null) && (contents.f !== null)) {
                    Utils.error('CellCollection.generateCellContentOperations(): illegal operation: clearing values without formulas');
                    return null;
                }

                // clear values without new formatting: visit value cells only
                if (clearValues) { return createIteratorForRange(range, 'value'); }

                // when setting values or formulas, all cells must be visited (regardless of passed range mode)
                var rangeMode = contents.rangeMode || 'cell:merge';
                if (changeFlags.value || changeFlags.formula || (changeFlags.style && (rangeMode === 'cell:merge'))) {
                    return createIteratorForRange(range, 'all');
                }

                // early exit, if nothing will be changed at all (remaining change flags are 'blank' and 'style')
                if (!changeFlags.blank && !changeFlags.style) { return null; }

                // formatting of blank ranges specific to range mode
                switch (rangeMode) {

                    // create an iterator that visits existing cells only
                    case 'row:def':
                        return createIteratorForRange(range, 'defined');

                    // When changing the formatting of unformatted rows, the column intervals with default auto-style do
                    // not need to be merged with the new formatting attributes (the new row default auto-style will suffice),
                    // but column intervals with other auto-styles need to be merged with the new formatting attributes.
                    case 'row:merge':
                        var colIntervals1 = new IntervalArray();
                        var colIntervals2 = new IntervalArray();
                        Iterator.forEach(colCollection.createStyleIterator(range.colInterval()), function (colInterval, result) {
                            // undefined cells in columns with the default style do not need to be processed
                            var colIntervals = autoStyles.isDefaultStyleId(result.style) ? colIntervals1 : colIntervals2;
                            colIntervals.push(colInterval);
                        });
                        var colIterator1 = createIteratorForIntervals(colIntervals1.merge(), range.rowInterval(), 'defined');
                        var colIterator2 = createIteratorForIntervals(colIntervals2.merge(), range.rowInterval(), 'all');
                        return [colIterator1, colIterator2];

                    // When changing the formatting of columns, the row intervals without custom auto-style (row attribute
                    // 'customFormat' set to false) do not need to be merged with the new formatting attributes (the new column
                    // default auto-style will suffice), but row intervals with custom formatting need to be merged with the
                    // new formatting attributes.
                    case 'col:merge':
                        var rowIterator = rowCollection.createStyleIterator(range.rowInterval());
                        return new NestedIterator(rowIterator, function (rowInterval, result) {
                            // if the current row interval is not formatted, create a cell iterator that visits existing cells only
                            var rowRange = Range.createFromIntervals(range.colInterval(), rowInterval);
                            return createIteratorForRange(rowRange, (result.style === null) ? 'defined' : 'all');
                        });
                }

                Utils.error('CellCollection.generateCellContentOperations(): invalid range mode "' + rangeMode + '"');
                return null;
            }

            // use a time-sliced loop on the row array to prevent freezing browser
            var promise = self.iterateArraySliced(matrix, function (rowData) {

                // row repetition count
                var rowCount = rowData.r || 1;
                // the start address of the next cell contents range
                var rowAddress = nextRowAddress.clone();
                // update the address of the next row band
                nextRowAddress[1] += rowCount;

                // skip entire row (or multiple rows), if no 'c' property exists
                if (!_.isArray(rowData.c)) { return; }

                // the cell iterators for all cell data objects in this row band
                var nextCellAddress = rowAddress.clone();
                var rangeIterators = rowData.c.reduce(function (iterators, contents) {

                    // repetition count for the cell data
                    var colCount = contents.r || 1;
                    // address of the cell range covered by the cell contents object
                    var range = Range.createFromAddressAndSize(nextCellAddress.clone(), colCount, rowCount);

                    // update address of the next cell data object
                    nextCellAddress[0] += colCount;

                    // collect the URLs for hyperlinks
                    if (contents.url) {
                        var linkRanges = linkRangesMap[contents.url] || (linkRangesMap[contents.url] = new RangeArray());
                        linkRanges.push(range);
                    }

                    // create cell iterators (will be null, if the cells will not change at all, e.g. URL only)
                    var newIterators = createIterators(range, contents);
                    return newIterators ? iterators.concat(newIterators) : iterators;
                }, []);

                // create an iterator that visits the cells in order row-by-row
                var syncIt = new OrderedIterator(rangeIterators, addressIndexer);
                return self.iterateSliced(syncIt, function (address, result) {

                    // process the cells
                    operationsBuilder.createCells(address, result.contents, result.colCount);

                    // count the changed cells, early exit on overflow
                    if (result.countCells) {
                        changedCount += result.colCount;
                        if (changedCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                            return SheetUtils.makeRejected('cells:overflow');
                        }
                    }
                }, 'CellCollection.generateCellContentOperations');
            }, 'CellCollection.generateCellContentOperations');

            // generate all operations, return the addresses of all changed cell ranges
            promise = promise.then(function () {

                var hyperlinkCollection = sheetModel.getHyperlinkCollection();

                // first, create operations to remove hyperlinks
                if ('' in linkRangesMap) {
                    hyperlinkCollection.generateHyperlinkOperations(generator, linkRangesMap[''].merge(), '');
                    delete linkRangesMap[''];
                }

                // create new hyperlinks for the cells
                _.each(linkRangesMap, function (linkRanges, url) {
                    hyperlinkCollection.generateHyperlinkOperations(generator, linkRanges.merge(), url);
                });

                // generate the cell operations, return addresses of changed ranges to caller
                return operationsBuilder.finalizeOperations(options);
            });

            return promise.done(function (changedRanges) {
                SheetUtils.log('changed=' + changedRanges);
            });
        });

        /**
         * Generates the cell operations, and the undo operations, to change
         * the contents for equally formatted or filled cell ranges in the
         * sheet, and optionally the appropriate hyperlink operations.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray} taggedRanges
         *  The cell ranges to generate operations for. Each range address in
         *  this array may contain the following additional properties:
         *  - {String} [rangeMode='cell:merge']
         *      Specifies which cells will be processed in the range. Usually,
         *      the method getTaggedRanges() should be used to create this
         *      property. See method generateCellContentOperations() for more
         *      details.
         *  - {Object} [fillData]
         *      Additional formatting settings for the range. These data
         *      objects will be collected in an array, and will be passed to
         *      the callback function (see below).
         *
         * @param {Function} callback
         *  A callback function that will be invoked for each cell range of the
         *  partition of the passed cell ranges. Receives the fill data objects
         *  extracted from the original ranges covered by the current partition
         *  range as simple array in the first parameter. MUST return the cell
         *  contents object for the range to be inserted into the internal cell
         *  contents matrix.
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the finalize method of
         *  the cell operations builders passed to the callback function. See
         *  CellOperationsBuilder.finalizeOperations() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated.
         */
        var generateCellRangeOperations = (function () {

            // returns the effective (stronger) range mode of the passed range modes
            function mergeRangeMode(rangeMode1, rangeMode2) {
                // short-cut: equal range modes
                if (rangeMode1 === rangeMode2) { return rangeMode1; }
                // ignore range mode 'skip' (fall-back to other range mode)
                if (rangeMode1 === 'skip') { return rangeMode2; }
                if (rangeMode2 === 'skip') { return rangeMode1; }
                // needing to process all cells in a range always wins over every other range mode
                // (cell formatting wins over column and row default formatting)
                if ((rangeMode1 === 'cell:merge') || (rangeMode2 === 'cell:merge')) { return 'cell:merge'; }
                // row formatting wins over column formatting
                return (rangeMode1 === 'col:merge') ? rangeMode2 : rangeMode1;
            }

            // creates and returns  a new row entry in the passed contents matrix
            function addMatrixRow(matrix, count) {
                var rowData = { c: [], r: count };
                matrix.push(rowData);
                return rowData;
            }

            // the actual implementation returned from local scope
            function generateCellRangeOperations(generator, taggedRanges, callback, options) {
                SheetUtils.log('ranges=' + taggedRanges);

                // early exit for empty arrays
                if (taggedRanges.empty()) { return self.createResolvedPromise(new RangeArray()); }

                // the start address of the bounding range, used as anchor cell for the matrix
                var startAddress = taggedRanges.boundary().start;
                // the address of the next free entry in the matrix
                var currAddress = startAddress.clone();
                // do not waste time to collect the fill data of the original ranges, if the callback does not use them
                var collectFillData = callback.length > 0;
                // the contents matrix for all ranges
                var matrix = [];

                // process all row bands, use a time-sliced loop to prevent freezing browser
                var promise = self.iterateArraySliced(taggedRanges.getRowBands({ ranges: true }), function (rowBand) {

                    // add empty rows into the matrix to be skipped
                    if (currAddress[1] < rowBand.first) {
                        addMatrixRow(matrix, rowBand.first - currAddress[1]);
                    }

                    // create a new row in the matrix
                    var rowData = addMatrixRow(matrix, rowBand.size());

                    // convert the ranges covered by the row band to tagged column intervals
                    var colIntervals = IntervalArray.map(rowBand.ranges, function (range) {
                        var interval = range.colInterval();
                        interval.rangeMode = range.rangeMode;
                        interval.fillData = range.fillData;
                        return interval;
                    });

                    // iterate over a (sorted) column partition of the ranges covered by the row band
                    var promise2 = self.iterateArraySliced(colIntervals.partition(), function (colInterval) {

                        // skip the gaps between the ranges
                        if (currAddress[0] < colInterval.first) {
                            rowData.c.push({ r: colInterval.first - currAddress[0] });
                        }

                        // determine the range mode to be used for the current range
                        var rangeMode = colInterval.coveredBy.reduce(function (rangeMode, colInterval) {
                            return mergeRangeMode(rangeMode, colInterval.rangeMode || 'cell:merge');
                        }, 'col:merge');

                        // collect the fill data of all source ranges covered by the current row band and column interval
                        // (performance optimization: do not collect fill data, if callback function does not use them)
                        var fillDataArray = collectFillData ? _.pluck(colInterval.coveredBy, 'fillData').filter(_.identity) : null;

                        // create the cell contents entry for the current range
                        var contents = _.clone(callback(fillDataArray));
                        contents.r = colInterval.size();
                        contents.rangeMode = rangeMode;
                        rowData.c.push(contents);

                        // move current address behind the range processed here
                        currAddress[0] = colInterval.last + 1;

                    }, 'CellCollection.generateCellRangeOperations');

                    // back to first column for the next row band
                    return promise2.done(function () {
                        currAddress[0] = startAddress[0];
                        currAddress[1] = rowBand.last + 1;
                    });

                }, 'CellCollection.generateCellRangeOperations');

                // generate the cell operations from the matrix
                return promise.then(function () {
                    return generateCellContentOperations(generator, startAddress, matrix, options);
                });
            }

            return SheetUtils.profileAsyncMethod('CellCollection.generateCellRangeOperations()', generateCellRangeOperations);
        }());

        function generateChainedColIntervalOperations(promise, rangeGroups, callback) {
            return rangeGroups.colRanges ? promise.then(function (changedRanges) {
                var colIntervals = IntervalArray.map(rangeGroups.colRanges, function (range) {
                    var colInterval = range.colInterval();
                    colInterval.fillData = range.fillData;
                    return colInterval;
                });
                return callback(colIntervals).then(function (changedIntervals) {
                    return changedRanges.append(docModel.makeColRanges(changedIntervals));
                });
            }) : promise;
        }

        function generateChainedRowIntervalOperations(promise, rangeGroups, callback) {
            return rangeGroups.rowRanges ? promise.then(function (changedRanges) {
                var rowIntervals = IntervalArray.map(rangeGroups.rowRanges, function (range) {
                    var rowInterval = range.rowInterval();
                    rowInterval.fillData = range.fillData;
                    return rowInterval;
                });
                return callback(rowIntervals).then(function (changedIntervals) {
                    return changedRanges.append(docModel.makeRowRanges(changedIntervals));
                });
            }) : promise;
        }

        function generateChainedCellRangeOperations(promise, callback) {
            return promise.then(function (changedRanges) {
                return callback().then(function (changedRanges2) {
                    changedRanges = changedRanges.append(changedRanges2).merge();
                    SheetUtils.log('changed=' + changedRanges);
                    return changedRanges;
                });
            });
        }

        function getAutoFormulaRange(startAddress, direction, skipStart, boundRange) {

            var address1 = null;
            var address2 = null;

            // find the largest range next to the start cell containing numbers only
            var cellIt = self.createLinearAddressIterator(startAddress, direction, { type: 'value', boundRange: boundRange, skipStart: skipStart });
            Iterator.forEach(cellIt, function (address) {

                var value = self.getValue(address);

                if (typeof value !== 'number') {
                    if (address1) {
                        return Utils.BREAK;
                    } else {
                        return;
                    }
                }
                if (address2 && ((Math.abs(address2[0] - address[0]) > 1) || (Math.abs(address2[1] - address[1]) > 1))) {
                    return Utils.BREAK;
                }

                if (!address1) { address1 = address; }
                address2 = address;
            });

            if (address1 && address2) {
                var startCol = startAddress[0];
                var startRow = startAddress[1];

                if (direction === 'up') { startRow--; } else { startCol--; }

                return { address1: address1, range: Range.create(startCol, startRow, address2[0], address2[1]) };
            }
        }

        // autofill ------------------------------------------------------------

        // analyze dedicated ranges and decides which type of autoFill should grab here (increase, copy, ...)
        function getAutoFillType(obj, columns, reverse, copy) {

            var range = obj.baseRange;
            if (range.size(!columns) < 3) {
                return copy ? 'copy' : 'increase';
            }

            var iterator = range.interval(!columns).iterator({ reverse: reverse });
            var arrNr = [];
            var parts = null;
            var i = 0;
            var partIndex = (obj.valueType === 'stringNumber') ? 2 : 1;
            var regex = (obj.valueType === 'stringNumber') ? TRAILING_INT_RE : LEADING_INT_RE;
            var lastValue = null;
            var rise = null;

            Iterator.forEach(iterator, function () {
                var address     = range.addressAt(i),
                    cell        = self.getCellModel(address),
                    nr          = null;

                parts = regex.exec(cell.v);
                nr = parseInt(parts[partIndex], 10);
                arrNr.push(nr);

                if (i > 0) {
                    if (rise === null || rise === (nr - lastValue)) {
                        rise = (nr - lastValue);
                    } else {
                        rise = false;
                    }
                }

                lastValue = nr;

                i++;
            });

            var slope = MathUtils.slope(arrNr);
            if (rise !== false && rise === slope) {
                return copy ? 'copy' : 'increase';
            }

            return 'copy';
        }

        // calculates the average and slope of the given array of numbers
        function calculateAverageAndSlope(arrNumbers, reverse) {
            var slope = null,
                average = null;

            if (arrNumbers.length > 1) {
                var array = (reverse) ? arrNumbers.reverse() : arrNumbers,
                    summe = _.reduce(array, function (memo, cellValue) { return memo + parseFloat(cellValue); }, 0);

                average = (summe / arrNumbers.length);
                slope = MathUtils.slope(array);

            } else {
                average = parseFloat(arrNumbers[0]);
                slope = (reverse) ? -1 : 1;
            }
            return {
                slope: slope,
                average: average
            };
        }

        // analyze a range and create a plan for the cell-operations for autofill
        function analyzeForAutoFill(range, vertical, leading, copy) {

            // detect the type of the value
            function detectValueType(cellModel) {

                if (cellModel.isAnyFormula()) { return 'formula'; }
                if (cellModel.isError()) { return 'error'; }

                if (cellModel.isNumber()) {
                    var category = cellModel.pf.category;
                    return ((category !== 'standard') && (category !== 'custom') && (category !== 'percent')) ? category : 'number';
                }

                if (cellModel.isText()) {
                    if (listCollection.listContains(cellModel.v)) { return 'definedList'; }
                    if (TRAILING_INT_RE.test(cellModel.v)) { return 'stringNumber'; }
                    if (LEADING_INT_RE.test(cellModel.v)) { return 'numberString'; }
                    return 'string';
                }

                return cellModel.isBlank() ? '' : null;
            }

            function analyzeDateTimeInterval(arrDates, type) {
                var returnObj = false;

                if (arrDates.length > 1) {
                    var i               = 0,
                        interval        = null,
                        unit            = null,

                        intervalDays    = null,
                        intervalMonth   = null,
                        intervalYears   = null,
                        intervalMilli   = null,
                        intervalSeconds = null,
                        intervalMinutes = null,
                        intervalHours   = null,

                        lastUnit        = null,
                        lastInterval    = null,
                        lastDateObj     = null;

                    var linear = _.every(arrDates, function (number) {
                        var date        = numberFormatter.convertNumberToDate(number),
                            momentObj   = moment({ year: date.getFullYear(), month: date.getMonth(), day: date.getDate(), hours: date.getHours(), minutes: date.getMinutes(), seconds: date.getSeconds(), milliseconds: date.getMilliseconds() }).utc();

                        if (i > 0) {
                            intervalMilli   = lastDateObj.diff(momentObj, 'milliseconds', true) * -1;
                            intervalSeconds = lastDateObj.diff(momentObj, 'seconds', true) * -1;
                            intervalMinutes = lastDateObj.diff(momentObj, 'minutes', true) * -1;
                            intervalHours   = lastDateObj.diff(momentObj, 'hours', true) * -1;
                            intervalDays    = lastDateObj.diff(momentObj, 'days', true) * -1;
                            intervalMonth   = lastDateObj.diff(momentObj, 'month', true) * -1;
                            intervalYears   = lastDateObj.diff(momentObj, 'years', true) * -1;

                            if (intervalYears % 1 === 0) {
                                unit = 'years';         interval = intervalYears;
                            } else if (intervalMonth % 1 === 0) {
                                unit = 'months';        interval = intervalMonth;
                            } else if (intervalDays % 1 === 0) {
                                unit = 'days';          interval = intervalDays;
                            } else if (intervalHours % 1 === 0) {
                                unit = 'hours';         interval = intervalHours;
                            } else if (intervalMinutes % 1 === 0) {
                                unit = 'minutes';       interval = intervalMinutes;
                            } else if (intervalSeconds % 1 === 0) {
                                unit = 'seconds';       interval = intervalSeconds;
                            } else if (intervalMilli % 1 === 0) {
                                unit = 'milliseconds';  interval = intervalMilli;
                            }

                            if (lastUnit === null && lastInterval === null) {
                                lastUnit = unit;
                                lastInterval = interval;
                            } else if (lastUnit !== unit || lastInterval !== interval) {
                                return false;
                            }
                        }

                        lastDateObj = momentObj;
                        i++;
                        return true;
                    });

                    if (linear && interval !== null && unit !== null) {
                        returnObj = {
                            interval: interval,
                            unit: unit
                        };
                    }

                } else {
                    returnObj = {
                        interval: 1,
                        unit: (type === 'time') ? 'hour' : 'day'
                    };
                }

                return returnObj;
            }

            function analyzeDefinedListInterval(arr) {
                var lastList        = null,
                    collector       = [],
                    lastIndexArray  = {},
                    linear          = null,
                    interval        = null;

                if (arr.length > 1) {
                    // collect and count all list-indexes whose matching the given array-values
                    _.each(arr, function (val) {
                        var value = val.toLowerCase();
                        listCollection.each(function (list, i) {
                            lastIndexArray[i] = lastIndexArray[i] || [];

                            if (collector.length === 0 && _.contains(list, value)) {
                                collector.push(i);
                                lastIndexArray[i].push(_.indexOf(list, value));

                            } else if (_.contains(list, value) && collector.indexOf(i) !== -1) {
                                collector.push(i);
                                lastIndexArray[i].push(_.indexOf(list, value));

                            } else if (_.contains(list, value) && collector.indexOf(i) === -1) {
                                collector.push(i);
                                lastIndexArray[i].push(_.indexOf(list, value));
                            }
                        });
                    });

                    // cumulate array-values
                    var counts = {};
                    collector.forEach(function (x) { counts[x] = (counts[x] || 0) + 1; });
                    var collectorValues = _.values(counts);
                    var maxStrikes = Math.max.apply(null, collectorValues);

                    // get list-index with the most matching-strikes (in case of multiple matches, get the first one)
                    _.some(counts, function (count, i) {
                        if (count === maxStrikes) { lastList = i; return true; }
                    });

                    var lastDistance = null,
                        lastIndex    = null;

                    if (lastIndexArray[lastList].length > 1) {
                        linear = _.every(lastIndexArray[lastList], function (index) {
                            var currentDistance = null;
                            if (lastDistance === null && lastIndex === null) {
                                lastIndex = index;
                            } else if (lastDistance === null) {
                                lastDistance = (index - lastIndex);
                                if (lastDistance < 0) { lastDistance += listCollection.getList(lastList).length; }
                                lastIndex = index;
                            } else {
                                currentDistance = (index - lastIndex);
                                if (currentDistance < 0) { currentDistance += listCollection.getList(lastList).length; }
                                if (lastDistance !== currentDistance) {
                                    return false;
                                }
                                lastDistance = currentDistance;
                                lastIndex = index;
                            }
                            return true;
                        });
                        interval = (linear) ? lastDistance : null;

                    } else {
                        linear = true;
                        interval = 1;
                    }

                } else {
                    var value = arr[0].toLowerCase();
                    listCollection.some(function (list, i) {
                        if (_.contains(list, value)) {
                            lastList = i;
                            lastIndexArray[lastList] = lastIndexArray[lastList] || [];
                            lastIndexArray[lastList].push(_.indexOf(list, value));
                            return true;
                        }
                    });

                    linear = true;
                    interval = 1;
                }

                return {
                    list: parseInt(lastList, 10),
                    indexArr: lastIndexArray[lastList],
                    linear: linear,
                    interval: interval
                };
            }

            // compares values of the last and current cell to find similar types
            function compareValues(type, last, current) {
                var partsLast = null,
                    partsLastColl = null,
                    partsCurrent = null;

                if (type === 'stringNumber') {
                    partsLast = TRAILING_INT_RE.exec(last);
                    partsCurrent = TRAILING_INT_RE.exec(current);
                    return partsLast[1].toLowerCase() === partsCurrent[1].toLowerCase();

                } else if (type === 'numberString') {
                    partsLast = LEADING_INT_RE.exec(last);
                    partsCurrent = LEADING_INT_RE.exec(current);
                    return partsLast[2].toLowerCase() === partsCurrent[2].toLowerCase();

                } else if (type === 'definedList') {
                    partsCurrent = [];
                    partsLastColl = [];
                    listCollection.each(function (list, listIndex) {
                        if (_.contains(list, last.toLowerCase()))    { partsLastColl.push(listIndex); }
                        if (_.contains(list, current.toLowerCase())) { partsCurrent.push(listIndex); }
                    });

                    if (partsLastColl.length > 1 && _.intersection(partsLastColl, partsCurrent).length > 0) {
                        partsLast = _.first(_.intersection(partsLastColl, partsCurrent));
                    } else {
                        partsLast = _.first(partsLastColl);
                    }

                    if (lastKnownList === null) {
                        lastKnownList = partsLast;
                    }

                    if (partsCurrent.indexOf(partsLast) !== -1) {
                        partsCurrent = partsLast;
                    }

                    return (partsLast !== null && partsLast === partsCurrent && lastKnownList === partsLast);

                }
                return false;
            }

            var iterator            = range.interval(!vertical).iterator({ reverse: leading }),
                planingResult       = [],
                i                   = 0,
                lastKnownList       = null,
                lastKnownRange      = null,
                lastKnownValueType  = null,
                lastKnownValue      = null;

            Iterator.forEach(iterator, function () {
                var address     = range.addressAt(i),
                    cell        = self.getCellModel(address),
                    valueType   = null;

                if (cell) {
                    valueType = detectValueType(cell);

                    if (lastKnownRange !== null && lastKnownValueType !== null) {
                        if (valueType === lastKnownValueType && ((valueType !== 'stringNumber' && valueType !== 'numberString' && valueType !== 'definedList') || (lastKnownValue !== null && compareValues(valueType, lastKnownValue, cell.v)))) {
                            lastKnownRange.end.move(1, !vertical);
                        } else {
                            lastKnownList = null;
                            lastKnownRange = null;
                            lastKnownValueType = null;
                        }
                    }

                    if (lastKnownRange === null) {
                        lastKnownRange = new Range(address.clone(), address.clone());
                    }
                    if (lastKnownValueType === null) {
                        lastKnownValueType = valueType;
                    }

                    lastKnownValue = cell.v;
                } else {
                    lastKnownRange = null;
                    lastKnownValueType = null;
                    lastKnownValue = null;
                }

                planingResult.push({
                    baseCell: address,
                    baseRange: lastKnownRange,
                    valueType: valueType
                });

                i++;
            });

            // analyze the fill type (increase, copy, ...) for each found range
            _.each(planingResult, function (obj) {
                if (obj.valueType === 'stringNumber' || obj.valueType === 'numberString') {
                    obj.fillType = getAutoFillType(obj, vertical, leading, copy);

                } else if (obj.valueType === 'date' || obj.valueType === 'time' || obj.valueType === 'datetime') {
                    var interval = analyzeDateTimeInterval(_.reduce(self.getRangeContents(obj.baseRange, { blanks: true }), function (memo, cell) { memo.push(cell.value); return memo; }, []), obj.valueType);
                    if (interval !== false && !copy) {
                        obj.fillType = 'increase';
                        obj.unit = interval.unit;
                        obj.interval = interval.interval;
                    } else {
                        obj.fillType = 'copy';
                    }

                } else if (obj.valueType === 'number') {
                    if (range.singleLine(!vertical) === copy) {
                        // reversed logic for single numbers (auto-increase in copy mode) as in MSXL
                        obj.fillType = 'increase';
                    } else {
                        obj.fillType = 'copy';
                    }

                } else if ((obj.valueType === 'formula') && !copy) {
                    obj.fillType = 'relocate';

                } else if (obj.valueType === 'definedList' && !copy) {
                    var listInterval = analyzeDefinedListInterval(_.reduce(self.getRangeContents(obj.baseRange, { blanks: true }), function (memo, cell) { memo.push(cell.value); return memo; }, []));

                    if (listInterval.linear) {
                        obj.fillType     = 'increase';
                        obj.list         = listInterval.list;
                        obj.interval     = listInterval.interval;
                        obj.rangeIndexes = listInterval.indexArr;

                    } else {
                        obj.fillType = 'copy';
                    }

                } else {
                    obj.fillType = 'copy';
                }
            });

            // return the planing-result
            return planingResult;
        }

        // generates autofill-operations for cells
        function generateAutoFillOperations(generator, range, direction, count, options) {

            // calculate the result for the autofill-operations
            function calculateResult(plan, round, sourceLineCount) {

                var fromModel            = self.getCellModel(plan.baseCell),
                    resultValue          = null,
                    resultFormula        = null,
                    string               = null;

                // INCREASE (values like numbers, dates, defined lists, ...)
                if (plan.fillType === 'increase') {
                    // count up a index for each range separately
                    rangeIndex[plan.baseRange.toString()] = (rangeIndex[plan.baseRange.toString()] + 1) || 0;

                    if (plan.valueType === 'date' || plan.valueType === 'time' || plan.valueType === 'datetime') {
                        var cellDate    = numberFormatter.convertNumberToDate(self.getCellModel(plan.baseRange.start).v),
                            momentObj   = moment({ year: cellDate.getFullYear(), month: cellDate.getMonth(), day: cellDate.getDate(), hours: cellDate.getHours(), minutes: cellDate.getMinutes(), seconds: cellDate.getSeconds(), milliseconds: cellDate.getMilliseconds() }).utc(),
                            currIndex   = rangeIndex[plan.baseRange.toString()];

                        if (leading) {
                            momentObj.subtract((currIndex + 1) * plan.interval, plan.unit);
                        } else {
                            momentObj.add((plan.baseRange.size(!vertical) + currIndex) * plan.interval, plan.unit);
                        }

                        resultValue = numberFormatter.convertDateToNumber(momentObj.toDate());

                    } else if (plan.valueType === 'definedList') {
                        var list = listCollection.getList(plan.list),
                            currI = (rangeIndex[plan.baseRange.toString()] + 1),
                            newIndex = null;

                        if (leading) {
                            newIndex = plan.rangeIndexes[0] - (currI * plan.interval);
                            while (newIndex < 0) { newIndex = list.length + newIndex; }

                        } else {
                            newIndex = plan.rangeIndexes[plan.rangeIndexes.length - 1] + (currI * plan.interval);
                            if (newIndex >= list.length) { newIndex %= list.length; }
                        }

                        resultValue = list[newIndex];

                        // make the whole string uppercase, if the initial value is uppercase too
                        if (self.getCellModel(plan.baseRange.start).v.toUpperCase() === self.getCellModel(plan.baseRange.start).v) {
                            resultValue = resultValue.toUpperCase();
                        // make first letter uppercase, if the first initial letter is uppercase too
                        } else if (self.getCellModel(plan.baseRange.start).v[0].toUpperCase() === self.getCellModel(plan.baseRange.start).v[0]) {
                            resultValue = resultValue[0].toUpperCase() + resultValue.slice(1);
                        }

                    } else {
                        var numberGetter = function (memo, cell) {
                            memo.push(cell.value);
                            return memo;
                        };
                        var valueGetter = function (memo, cell) {
                            var arrValue    = regex.exec(cell.value),
                                value       = (arrValue && arrValue[nrIndex]) ? arrValue[nrIndex] : null,
                                number      = (value) ? parseInt(value, 10) : null;

                            memo.push(number);
                            return memo;
                        };

                        var startCellIndex  = ((plan.baseRange.size(!vertical) + 1) / 2),
                            multiplier      = (((plan.baseRange.size(!vertical) + 1) - startCellIndex) + rangeIndex[plan.baseRange.toString()]),

                            strIndex        = (plan.valueType === 'stringNumber') ? 1 : 2,
                            nrIndex         = (plan.valueType === 'stringNumber') ? 2 : 1,
                            regex           = (plan.valueType === 'stringNumber') ? TRAILING_INT_RE : LEADING_INT_RE,
                            getter          = (plan.valueType === 'number') ? numberGetter : valueGetter,
                            calcObj         = calculateAverageAndSlope(_.reduce(self.getRangeContents(plan.baseRange, { blanks: true }), getter, []), leading);

                        // calculate the new number
                        resultValue = (calcObj.average + (multiplier * calcObj.slope));

                        // when the source-value is a combination of string and number
                        if (plan.valueType === 'stringNumber' || plan.valueType === 'numberString') {
                            // get the string(-part) of the first cell from the baseRange
                            string          = regex.exec(self.getCellModel(plan.baseRange.start).v)[strIndex]; // need to use the first cell, because of the case-sensitifity
                            // add the string to the resultValue
                            resultValue     = (plan.valueType === 'stringNumber') ? string + Math.abs(resultValue) : Math.abs(resultValue) + string;
                        }
                    }

                // RELOCATE (for formulas and sharedFormulas)
                } else if (plan.fillType === 'relocate') {
                    // set the cell value to zero, so that the user sees the "0" shortly till the formulaengine calculates the correct result
                    resultValue = 0;
                    // calculate the target address for relocating the formula
                    var targetAddress = fromModel.a.clone().move(round * sourceLineCount * (leading ? -1 : 1), !vertical);
                    // resolve the formula expression of a shared formula
                    var sharedModel = getSharedModel(fromModel);
                    if (sharedModel) {
                        resultFormula = sharedModel.getFormula('op', targetAddress);
                    } else {
                        resultFormula = fromModel.getFormula('op', targetAddress);
                    }

                // COPY (values)
                } else if (plan.baseRange) {
                    if (plan.valueType === 'formula') {
                        resultFormula = self.getFormula(fromModel.a, 'op');
                    }

                    resultValue = fromModel ? fromModel.v : null;
                }

                // returns the result within an object
                return {
                    from: fromModel,
                    style: self.getStyleId(plan.baseCell),
                    result: {
                        value: resultValue,
                        formula: resultFormula
                    }
                };
            }

            // creates a matrix for the range-autofill-operations
            function createMatrix(sourceRange, index) {

                // adds a cell-config-object to the matrix
                function addToMatrix(cell, row_i, col_i) {
                    var i = (vertical) ? row_i : col_i;

                    if (!matrix[i]) { matrix[i] = { c: [] }; }
                    matrix[i].c.push(cell);
                }

                var // target range
                    targetRange     = SheetUtils.getAdjacentRange(sourceRange, direction, count),
                    // the index of the foreach of the targetRange
                    i               = 0,
                    // the index of the sourceRange-loop
                    round           = 0,
                    // generate plan for iterating over targetRange with help of sourceRange
                    autoFillPlaning = analyzeForAutoFill(sourceRange, vertical, leading, copy),
                    // the count of the lines from the sourceRange
                    sourceLineCount = sourceRange.size(!vertical);

                // in reverse case, switch the base plan to be able to reuse the index-count
                if (leading) { autoFillPlaning.reverse(); }

                // iterate over all single colums/rows to push new content to target-ranges
                Iterator.forEach(targetRange.iterator({ reverse: leading }), function () {
                    var sourceIndex     = (i % sourceLineCount),
                        plan            = autoFillPlaning[sourceIndex],
                        result          = null;

                    // count up the roundIndex, everytime sourceIndex is "0"
                    if (sourceIndex === 0) { round++; }

                    // calculate the actual autofill-result
                    result = calculateResult(plan, round, sourceLineCount);

                    // when there is a formula (and no new calculated value), use the old
                    // value for now
                    if (!_.isNull(result.result.formula) && result.result.value === 0) {
                        result.result.value = result.from.v;
                    }

                    // add result to the matrix
                    addToMatrix({
                        f: result.result.formula,
                        s: (!_.isNull(result.style)) ? result.style : null,
                        v: result.result.value
                    }, i, (index - startIndex));

                    i++;
                });
            }

            // creates a contents-array for the autofill-operations
            function createArray(sourceRange) {
                var contents        = [],
                    // target range
                    targetRange     = SheetUtils.getAdjacentRange(sourceRange, direction, count),
                    // the index of the foreach of the targetRange
                    i               = 0,
                    // the index of the sourceRange-loop
                    round           = 0,
                    // generate plan for iterating over targetRange with help of sourceRange
                    autoFillPlaning = analyzeForAutoFill(sourceRange, vertical, leading, copy),
                    // the count of the lines from the sourceRange
                    sourceLineCount = sourceRange.size(!vertical);

                // in reverse case, switch the base plan to be able to reuse the index-count
                if (leading) { autoFillPlaning.reverse(); }

                // iterate over all single colums/rows to push new content to target-ranges
                Iterator.forEach(targetRange.iterator({ reverse: leading }), function () {
                    var sourceIndex     = (i % sourceLineCount),
                        plan            = autoFillPlaning[sourceIndex],
                        result          = null;

                    // count up the roundIndex, everytime sourceIndex is "0"
                    if (sourceIndex === 0) { round++; }

                    // calculate the actual autofill-result
                    result = calculateResult(plan, round, sourceLineCount);

                    // the resulting cell-config-object
                    var cell = {
                        f: result.result.formula,
                        s: (!_.isNull(result.from)) ? result.from.s : null,
                        v: result.result.value
                    };

                    // push cell-config-object to contents-array
                    if (vertical) {
                        contents.push({ c: [cell] });
                    } else {
                        if (!contents[0]) { contents.push({ c: [] }); }
                        contents[0].c.push(cell);
                    }

                    i++;
                });
                return contents;
            }

            // whether to expand/delete columns or rows
            var vertical = SheetUtils.isVerticalDir(direction);
            // whether to expand/shrink the leading or trailing border
            var leading = SheetUtils.isLeadingDir(direction);

            var // should the type be inverted (copy to increase aso)
                copy            = Utils.getBooleanOption(options, 'copy', false),
                // start index of col/row (to create autofill ranges)
                startIndex      = vertical ? range.start[0] : range.start[1],
                // complete column(s) selected
                completeCol     = docModel.isColRange(range),
                // complete row(s) selected
                completeRow     = docModel.isRowRange(range),
                // object which holds the index for each separate group within the sourceRange
                rangeIndex      = {},
                // col/row interval
                interval        = range.interval(!vertical),
                parallelRanges  = [],
                matrix          = [];

            // if one or more complete col/row is selected for autofill
            if (completeRow || completeCol) {
                var arrContents = {};

                // generate ranges for the parallel iterator
                var iterator = (vertical ? rowMatrix : colMatrix).createIndexIterator(interval);
                Iterator.forEach(iterator, function (index) {
                    if (completeRow) {
                        parallelRanges.push(Range.create(0, index, range.end[0], index));
                    } else {
                        parallelRanges.push(Range.create(index, 0, index, range.end[1]));
                    }
                });

                // generate contents-array for operations (with parallel-iterator)
                Iterator.forEach(createParallelAddressIterator(parallelRanges, { type: 'defined' }), function (arrAddresses) {
                    var start    = completeRow ? new Address(arrAddresses[0][0], range.start[1]) : new Address(range.start[0], arrAddresses[0][1]),
                        end      = completeRow ? new Address(arrAddresses[0][0], range.end[1]) : new Address(range.end[0], arrAddresses[0][1]),
                        // range to work on currently
                        useRange = Range.createFromAddresses(start, end),
                        // get contents-array for range
                        contents = createArray(useRange);

                    // reverse contents-array if necessary
                    if (leading) { (vertical ? contents : contents[0].c).reverse(); }

                    var targetRange = SheetUtils.getAdjacentRange(useRange, direction, count);
                    arrContents[targetRange.start.toString()] = contents;
                });

                // generate cell operations (async)
                return self.iterateSliced(new ObjectIterator(arrContents), function (contents, options) {
                    return self.generateCellContentOperations(generator, Address.parse(options.key), contents);
                }, 'CellCollection.generateAutoFillOperations');

            // if an normal range is selected for autofill
            } else {
                // iterate over all ranges and create the matrix
                Iterator.forEach(range.interval(vertical), function (index) {
                    createMatrix(range.lineRange(vertical, (index - startIndex)), index);
                });

                // reverse new cell values if necessary
                if (leading) {
                    if (!vertical) {
                        _.each(matrix, function (data) {
                            data.c.reverse();
                        });
                    } else {
                        matrix.reverse();
                    }
                }

                // generate the cell content operations
                var targetRange = SheetUtils.getAdjacentRange(range, direction, count);
                return self.generateCellContentOperations(generator, targetRange.start, matrix);
            }
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { modelMap: modelMap };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * contents from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {CellCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the cell models of the source collection
            cloneData.modelMap.forEach(function (cellModel) {
                implInsertCellModel(cellModel.clone(sheetModel));
            });

            // update the used range, and trigger the 'change:usedrange' event
            this.updateUsedRange();
        };

        /**
         * Callback handler for the document operation 'changeCells'. Changes
         * the contents and auto-styles of the cells in this collection, and
         * triggers a 'change:cells' event for all changed cells.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'changeCells' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyChangeCellsOperation = function (context) {

            // the address of the top-left cell
            var start = context.getJSONAddress('start');
            // the cell contents matrix (array of row data objects)
            var matrix = context.getArr('contents');
            // the maximum column and row indexes in the sheet
            var maxCol = docModel.getMaxCol();
            var maxRow = docModel.getMaxRow();
            // the address of the current cell
            var currAddress = start.clone();
            // optimizations for import process
            var importing = context.importing;
            // descriptor with the addresses of all changed cells
            var changeDesc = importing ? null : new ChangeDescriptor();
            // the indexes of all shared formulas with changed anchor formula
            var sharedIndexSet = importing ? null : new IndexSet();

            // creates a cell contents object from the passed raw JSON data
            function createCellContents(jsonData, range) {

                // the contents object expected by the method updateCellModel()
                var contents = {};

                // update cell value (special handling for error codes needed)
                if ('v' in jsonData) {
                    context.ensure(!('e' in jsonData), 'unexpected cell error code');
                    var type = typeof jsonData.v;
                    context.ensure((jsonData.v === null) || (type === 'number') || (type === 'string') || (type === 'boolean'), 'invalid cell value type');
                    contents.v = jsonData.v;
                } else if ('e' in jsonData) {
                    context.ensure(typeof jsonData.e === 'string', 'invalid cell error code');
                    contents.v = formulaGrammar.getErrorCode(jsonData.e) || ErrorCode.NA;
                }

                // update cell formula
                if ('f' in jsonData) {
                    if (jsonData.f === null) {
                        contents.f = null;
                    } else {
                        context.ensure((typeof jsonData.f === 'string') && (jsonData.f.length > 0), 'invalid formula expression');
                        contents.f = jsonData.f;
                    }
                }

                // index of a shared formula
                if ('si' in jsonData) {
                    if (jsonData.si === null) {
                        contents.si = null;
                    } else {
                        context.ensure((typeof jsonData.si === 'number') && (jsonData.si >= 0), 'invalid shared formula identifier');
                        contents.si = Math.floor(jsonData.si);
                    }
                }

                // bounding range of a shared formula
                if ('sr' in jsonData) {
                    if (jsonData.sr === null) {
                        contents.sr = null;
                    } else {
                        context.ensure(typeof jsonData.sr === 'string', 'invalid shared formula range');
                        contents.sr = formulaParser.parseRange('op', jsonData.sr);
                        context.ensure(contents.sr, 'invalid shared formula range');
                    }
                }

                // bounding range of a matrix formula
                if ('mr' in jsonData) {
                    if (jsonData.mr === null) {
                        contents.mr = null;
                    } else {
                        context.ensure(typeof jsonData.mr === 'string', 'invalid matrix formula range');
                        contents.mr = formulaParser.parseRange('op', jsonData.mr);
                        context.ensure(contents.mr, 'invalid matrix formula range');
                    }
                }

                // update cell auto-style
                if ('s' in jsonData) {
                    context.ensure(typeof jsonData.s === 'string', 'invalid auto-style identifier');
                    contents.s = jsonData.s;
                }

                // fail safety: check number of changed cells (prevent freezing browser)
                if (!odf && (range.cells() > SheetUtils.MAX_FILL_CELL_COUNT)) {
                    context.error('too many cells');
                }

                // ODF import: warn but do not fail for formatted cells without contents
                // (workaround for bug 48015: use half of row size to ignore default row formatting)
                if (odf && (range.cells() > maxCol / 2) && ('s' in contents) && !('v' in contents) && !('f' in contents) && !('si' in contents) && !('sr' in contents)) {
                    Utils.warn('CellCollection.applyChangeCellsOperation(): Overflow error! Skipping formatting of ' + range.cells() + ' cells.');
                    contents = {};
                }

                return contents;
            }

            // collects changed cells by value, and by auto-style (performance: do not collect during import)
            var collectChangedAddresses = importing ? _.noop : function (address, changeFlags) {
                changeDesc.addAddress(address, changeFlags);
                if (changeFlags.formula) {
                    var cellModel = self.getCellModel(address);
                    if (cellModel && cellModel.isSharedAnchor()) {
                        sharedIndexSet.insert(cellModel.si);
                    }
                }
            };

            // process all rows in the contents array
            matrix.forEach(function (rowData, index) {

                // check that the row descriptor is an object
                context.ensure(_.isObject(rowData), 'invalid row entry in cell matrix');

                // number of repetitions for this row entry
                var rowRepeat = Utils.getIntegerOption(rowData, 'r', 1);
                context.ensure(currAddress[1] + rowRepeat - 1 <= maxRow, 'too many rows in cell matrix');

                // skip entire rows without 'c' property
                if (!('c' in rowData)) {
                    currAddress[1] += rowRepeat;
                    return;
                }

                // the 'c' property may be the array index of another row data object with 'c' property being an array
                var rowContents = rowData.c;
                if (_.isNumber(rowContents)) {
                    context.ensure((0 <= rowContents) && (rowContents < index), 'invalid array index to row data');
                    rowContents = matrix[rowContents].c;
                }

                // now, the 'c' property must be an array
                context.ensure(_.isArray(rowContents), 'cell data must be an array');

                // process all cell entries in the row array
                rowContents.forEach(function (jsonData) {
                    context.ensure(_.isObject(jsonData), 'cell entry must be an object');

                    // number of repetitions for this cell
                    var colRepeat = Utils.getIntegerOption(jsonData, 'r', 1);

                    // check validity of the covered column interval
                    context.ensure(currAddress[0] + colRepeat - 1 <= maxCol, 'row element in cell contents too long');

                    // immediately skip cells that will not be modified at all (without loop)
                    if (CellOperationsBuilder.isEmptyJSON(jsonData)) {
                        currAddress[0] += colRepeat;
                        return;
                    }

                    // build the cell range covered by the current JSON entry, update the current address
                    var range = Range.createFromAddressAndSize(currAddress, colRepeat, rowRepeat);
                    currAddress[0] += colRepeat;

                    // explicitly delete cell models, if the 'u' property is set
                    if (jsonData.u === true) {
                        // the ranges MUST be iterated in reversed order, to not disturb the cell matrix iterators while deleting the models
                        Iterator.forEach(createAddressIterator(range, { type: 'defined', reverse: true }), function (address) {
                            var changeFlags = deleteCellModel(address);
                            collectChangedAddresses(address, changeFlags);
                        });
                        return;
                    }

                    // the contents object expected by the method updateCellModel()
                    var contents = createCellContents(jsonData, range);
                    if (_.isEmpty(contents)) { return; }

                    // create/update the cell models, collect addresses of all changed cells
                    Iterator.forEach(range, function (address) {
                        var changeFlags = updateCellModel(address, contents, importing);
                        collectChangedAddresses(address, changeFlags);
                    });
                });

                // update the current address
                currAddress[0] = start[0];
                currAddress[1] += rowRepeat;
            });

            // add the addresses of all formula cells that are part of a shared formula with changed anchor
            if (changeDesc && sharedIndexSet && !sharedIndexSet.empty()) {
                var changeFlags = { formula: true };
                sharedIndexSet.forEach(function (sharedIndex) {
                    sharedModelMap.with(sharedIndex, function (sharedModel) {
                        sharedModel.addressSet.forEach(function (address) {
                            changeDesc.addAddress(address, changeFlags);
                        });
                    });
                });
            }

            // notify change event listeners
            if (changeDesc && !changeDesc.empty()) {
                var results = context.getOptStr('scope') === 'filter';
                this.trigger('change:cells', changeDesc, context.external, results);
            }

            // update the used range, and trigger the 'change:usedrange' event
            this.updateUsedRange();
        };

        /**
         * Callback handler for the document operation 'moveCells'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'moveCells' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyMoveCellsOperation = function (context) {

            // the range to be inserted or cleared
            var range = context.getJSONRange();
            // the move direction
            var direction = context.getEnum('dir', Direction);
            // whether to move the cells through columns
            var columns = !SheetUtils.isVerticalDir(direction);
            // whether to insert cells (move towards end of sheet)
            var insert = !SheetUtils.isLeadingDir(direction);
            // the move descriptor containing all needed data
            var moveDesc = new MoveDescriptor(docModel, range.interval(!columns), range.interval(columns), columns, insert);

            // first, delete all existing cell models in 'blankRange' (the cells MUST be iterated
            // in reversed order, to not disturb the cell matrix iterators while deleting the models)
            var blankRange = moveDesc.createRange(moveDesc.deleteIntervals.first());
            var blankIterator = createAddressIterator(blankRange, { type: 'defined', reverse: true });
            Iterator.forEach(blankIterator, function (address, result) {
                implRemoveCellModel(result.model);
            });

            // when deleting a cell range (move towards the beginning), there may not be anything left to move
            if (!moveDesc.moveFromIntervals.empty()) {

                // the source range containing cell models that need to be moved (may be empty)
                var moveRange = moveDesc.createRange(moveDesc.moveFromIntervals.first());
                // a temporary array used to collect moved cell models (prevents overwriting existing models)
                var tempModels = [];
                // the signed distance to modify the cell address index
                var moveDist = range.size(columns) * (insert ? 1 : -1);
                // column matrix does not need to be updated, when moving cells up/down;
                // or when moving the cells in all rows of the entire used area to the left/right
                var usedRowInterval = columns ? rowMatrix.getUsedInterval() : null;
                var skipColMatrix = !usedRowInterval || moveRange.rowInterval().contains(usedRowInterval);
                // row matrix does not need to be updated, when moving cells to left/right;
                // or when moving the cells in all columns of the entire used area up/down
                var usedColInterval = !columns ? colMatrix.getUsedInterval() : null;
                var skipRowMatrix = !usedColInterval || moveRange.colInterval().contains(usedColInterval);

                // move all cell models into the temporary map (the cells MUST be iterated in reversed
                // order, to not disturb the cell matrix iterator while removing the models)
                Iterator.forEach(createAddressIterator(moveRange, { type: 'defined', reverse: true }), function (address, result) {

                    // remove the cell from the containers, before the address will be changed
                    var cellModel = result.model;
                    implRemoveCellModel(cellModel, skipColMatrix, skipRowMatrix);

                    // adjust the cell address, and store the cell in the temporary array
                    cellModel.a.move(moveDist, columns);
                    tempModels.push(cellModel);
                });

                // insert all moved cell models back into the containers
                tempModels.forEach(function (cellModel) {
                    implInsertCellModel(cellModel, skipColMatrix, skipRowMatrix);
                });
            }

            // notify move event listeners
            if (!context.importing) {
                this.trigger('move:cells', moveDesc, context.external);
            }

            // update the used range, and trigger the 'change:usedrange' event
            this.updateUsedRange();
        };

        // protected methods --------------------------------------------------

        /**
         * Post-processing of the cell collection, after all import operations
         * have been applied successfully.
         *
         * @internal
         *  Called from the application import process. MUST NOT be called from
         *  external code.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the cell collection has been
         *  post-processed successfully; or rejected when any error has
         *  occurred.
         */
        this.postProcessCells = function () {

            // create the token arrays of all formula cells
            return this.iterateSliced(modelMap, function (cellModel) {
                cellModel._updateTokenArray(sheetModel);
            }, 'CellCollection.postProcessCells');
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the current zero-based index of the sheet containing this
         * cell collection.
         *
         * @returns {Number}
         *  The current zero-based index of the sheet containing this cell
         *  collection.
         */
        this.getSheetIndex = function () {
            return sheetModel.getIndex();
        };

        /**
         * Returns the identifier for the auto-style of the row or column to be
         * used for an undefined cell at the specified position.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Boolean} [ignoreRowStyle=false]
         *  If set to true, an existing row auto-style will be ignored, and
         *  this method effectively returns the default column auto-style of
         *  the specified cell.
         *
         * @returns {String}
         *  The identifier of the row auto-style, if the specified row contains
         *  the 'customFormat' flag (and the ignore flag is not set), otherwise
         *  the identifier of the column auto-style.
         */
        this.getDefaultStyleId = function (address, ignoreRowStyle) {
            var rowDesc = ignoreRowStyle ? null : rowCollection.getEntry(address[1]);
            return (rowDesc && rowDesc.merged.customFormat) ? rowDesc.style : colCollection.getEntry(address[0]).style;
        };

        /**
         * Returns whether the specified cell is visible (located in a visible
         * column, AND a visible row).
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the specified cell is visible.
         */
        this.isVisibleCell = function (address) {
            return colCollection.isEntryVisible(address[0]) && rowCollection.isEntryVisible(address[1]);
        };

        /**
         * Returns the internal cell model with the specified cell address from
         * the collection.
         *
         * @attention
         *  The method is provided for optimized access to internal cell data.
         *  The cell model MUST NOT BE CHANGED in order to retain the internal
         *  consistency of the cell collection!
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {CellModel|Null}
         *  The cell model; or null, if the cell does not exist in this cell
         *  collection. MUST NOT be changed!
         */
        this.getCellModel = function (address) {
            return modelMap.get(address.key(), null);
        };

        /**
         * Returns whether the specified cell is blank (no result value). A
         * blank cell may contain formatting attributes.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the specified cell is blank.
         */
        this.isBlankCell = function (address) {
            var cellModel = this.getCellModel(address);
            return !cellModel || cellModel.isBlank();
        };

        /**
         * Returns the value (or formula result) of the specified cell.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Number|String|Boolean|ErrorCode|Null}
         *  The current value (or formula result) of the specified cell.
         */
        this.getValue = function (address) {
            var cellModel = this.getCellModel(address);
            return cellModel ? cellModel.v : null;
        };

        /**
         * Returns the default display string of the specified cell.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {String|Null}
         *  The default display string of the specified cell (regardless of the
         *  column width); or null, if the cell cannot display its value (e.g.
         *  due to an invalid number format).
         */
        this.getDisplayString = function (address) {
            var cellModel = this.getCellModel(address);
            return cellModel ? cellModel.d : '';
        };

        /**
         * Returns whether the specified cell is a normal formula cell, or part
         * of a shared formula or matrix formula.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the specified cell is a formula cell.
         */
        this.isFormulaCell = function (address) {
            var cellModel = this.getCellModel(address);
            if (cellModel && cellModel.isAnyFormula()) { return true; }
            return matrixRangeSet.containsAddress(address);
        };

        /**
         * Returns whether the specified cell contains an actual formula
         * expression, either as normal formula cell, or as anchor cell of a
         * shared formula or matrix formula.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the specified cell contains an actual formula expression.
         */
        this.isAnchorCell = function (address) {
            var cellModel = this.getCellModel(address);
            return !!cellModel && cellModel.isAnyAnchor();
        };

        /**
         * Returns the token array of the specified formula cell.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.fullMatrix=false]
         *      If set to true, returns the token array of a matrix formula for
         *      every cell covered by the matrix. By default, the token array
         *      will be returned for the anchor cell of the matrix formula
         *      only.
         *  - {String} [options.grammarId]
         *      If specified, must be the identifier of a formula grammar. The
         *      descriptor returned by this method will contain an additional
         *      property 'formula' with the formula expression of the cell as
         *      string, according to the formula grammar. If omitted, the
         *      formula expression will not be generated (better performance).
         *
         * @returns {Object|Null}
         *  A result object with the token array and the reference address of
         *  the specified cell; or null, if the cell does not contain a
         *  formula. The result object will contain the following properties:
         *  - {String} type
         *      The formula type specifier. The value 'normal' represents a
         *      normal formula cell. The value 'shared' represents a cell that
         *      is part of a shared formula. The value 'matrix' represents a
         *      cell that is part of a matrix formula.
         *  - {TokenArray} tokenArray
         *      The token array with the parsed formula expression used to
         *      calculate the result value of the cell.
         *  - {Address} refAddress
         *      The reference address of the token array. For normal formula
         *      cells, this address is equal to the passed cell address. For
         *      shared formulas, this is the anchor address of the shared
         *      formula (the address of the cell containing the definition of
         *      the entire shared formula).
         *  - {Range|Null} matrixRange
         *      The bounding range of a matrix formula; or null, if the cell is
         *      not part of a matrix formula.
         *  - {String|Null} formula
         *      The formula expression for the specified cell, if the option
         *      'grammarId' has been passed; otherwise null.
         */
        this.getTokenArray = function (address, options) {

            // the cell model (may be null for a cell inside a matrix formula)
            var cellModel = this.getCellModel(address);
            // the grammar identifier to generate the formula expression for
            var grammarId = Utils.getStringOption(options, 'grammarId', null);

            // use the model of a shared formula as result object
            var sharedModel = cellModel ? getSharedModel(cellModel) : null;
            if (sharedModel && sharedModel.refAddress) {
                return {
                    tokenArray: sharedModel.tokenArray,
                    refAddress: sharedModel.refAddress,
                    matrixRange: null,
                    formula: grammarId ? sharedModel.getFormula(grammarId, address) : null
                };
            }

            // resolve matrix range for other cells in a matrix formula
            var refAddress = address;
            if ((!cellModel || !cellModel.t) && Utils.getBooleanOption(options, 'fullMatrix', false)) {
                var matrixRange = matrixRangeSet.findOneByAddress(address);
                refAddress = matrixRange ? matrixRange.start : null;
                cellModel = refAddress ? this.getCellModel(refAddress) : null;
            }

            // resolve the token array of a normal formula cell, and for a matrix anchor cell
            return (cellModel && cellModel.t) ? {
                tokenArray: cellModel.t,
                refAddress: refAddress.clone(),
                matrixRange: cellModel.mr ? cellModel.mr.clone() : null,
                formula: grammarId ? cellModel.getFormula(grammarId) : null
            } : null;
        };

        /**
         * Returns the formula expression of the cell at the specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar for the formula expression.
         *  See SpreadsheetDocument.getFormulaGrammar() for more details.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.fullMatrix=false]
         *      If set to true, returns the formula expression of a matrix
         *      formula for every cell covered by the matrix. By default, the
         *      formula expression will be returned for the anchor cell of the
         *      matrix formula only.
         *
         * @returns {String|Null}
         *  The formula expression of the specified cell, if it is a formula
         *  cell; otherwise null.
         */
        this.getFormula = function (address, grammarId, options) {
            var tokenDesc = this.getTokenArray(address, _.extend({}, options, { grammarId: grammarId }));
            return tokenDesc ? tokenDesc.formula : null;
        };

        /**
         * Returns the bounding range of the matrix formula the specified cell
         * is located in.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Range|Null}
         *  The bounding range of the matrix formula the specified cell is
         *  located in; or null, if the cell is not part of a matrix formula.
         */
        this.getMatrixRange = function (address) {
            var cellModel = this.getCellModel(address);
            if (cellModel && cellModel.mr) { return cellModel.mr.clone(); }
            var matrixRange = matrixRangeSet.findOneByAddress(address);
            return matrixRange ? matrixRange.clone() : null;
        };

        /**
         * Returns whether any cell in the passed cell range addresses is part
         * of a matrix formula.
         *
         * @param {RanmgeArray|Range} ranges
         *  An array of cell range addresses, ot the address of a single cell
         *  range.
         *
         * @param {String} [matchType]
         *  Specifies which matrix ranges from this collection will match. See
         *  method RangeSet.findIterator() for details.
         *
         * @returns {Boolean}
         *  Whether any cell in the passed cell range addresses is part of a
         *  matrix formula.
         */
        this.coversAnyMatrixRange = function (ranges, matchType) {
            return RangeArray.some(ranges, function (range) {
                return matrixRangeSet.findOne(range, matchType);
            });
        };

        /**
         * Returns the identifier of the effective auto-style used by the cell
         * at the specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {String}
         *  The identifier of the cell auto-style, if the cell is defined,
         *  otherwise the default auto-style of the row or column.
         */
        this.getStyleId = function (address) {
            var cellModel = this.getCellModel(address);
            return cellModel ? cellModel.s : this.getDefaultStyleId(address);
        };

        /**
         * Returns the effective merged attribute set of the specified cell.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Object}
         *  The merged attribute set of the cell, if it is defined, otherwise
         *  the attribute set of the default auto-style of the row or column.
         */
        this.getAttributeSet = function (address) {
            return autoStyles.getMergedAttributeSet(this.getStyleId(address));
        };

        /**
         * Returns the effective text orientation settings of the cell at the
         * specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Object}
         *  A descriptor containing various text orientation properties. See
         *  method SheetUtils.getTextOrientation() for details.
         */
        this.getTextOrientation = function (address) {
            var cellModel = this.getCellModel(address);
            var attrSet = this.getAttributeSet(address);
            return SheetUtils.getTextOrientation(cellModel ? cellModel.v : null, attrSet.cell.alignHor, cellModel ? cellModel.d : '');
        };

        /**
         * Returns the effective parsed number format of the cell at the
         * specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {ParsedFormat}
         *  The parsed number format of the cell, if it is defined, otherwise
         *  the parsed format of the default auto-style of the row or column.
         */
        this.getParsedFormat = function (address) {
            return autoStyles.getParsedFormat(this.getStyleId(address));
        };

        /**
         * Returns whether the number format category of the specified cell is
         * "text", i.e. whether no automatic parsing of input text happens for
         * that cell.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the number format category of the specified cell is "text".
         */
        this.isTextFormat = function (address) {
            return this.getParsedFormat(address).category === 'text';
        };

        /**
         * Returns the URL of a hyperlink as returned by a cell formula (via
         * the function HYPERLINK) at the specified address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink as returned by a cell formula (via the
         *  function HYPERLINK) at the specified address.
         */
        this.getFormulaURL = function (address) {
            var cellModel = this.getCellModel(address);
            return cellModel ? cellModel.url : null;
        };

        /**
         * Sets the URL of a hyperlink returned by the formula of the specified
         * cell (via the function HYPERLINK).
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {String|Null} url
         *  The resulting URL returned by a HYPERLINK function contained in the
         *  cell formula.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setFormulaURL = function (address, url) {
            var cellModel = this.getCellModel(address);
            if (cellModel) { cellModel.setURL(url); }
            return this;
        };

        /**
         * Returns the effective URL of a hyperlink at the specified address.
         * If the cell contains a regular hyperlink, and a cell formula with a
         * HYPERLINK function, the regular hyperlink will be preferred. See
         * description of the methods CellCollection.getCellURL(), and
         * CellCollection.getFormulaURL() for more details.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The effective URL of a hyperlink at the specified address.
         */
        this.getEffectiveURL = function (address) {
            var cellURL = sheetModel.getHyperlinkCollection().getCellURL(address);
            return (cellURL === null) ? this.getFormulaURL(address) : cellURL;
        };

        // cell edit mode -----------------------------------------------------

        /**
         * Returns the formula expression of the specified cell in 'ui' grammar
         * for the cell edit mode, with a leading equality sign.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.matrixBraces=false]
         *      If set to true, and the specified cell contains a matrix
         *      formula, the resulting text will be enclosed in curly braces.
         *
         * @returns {String|Null}
         *  The formula expression of the cell for cell edit mode; or null, if
         *  the cell does not contain a formula. If the cell contains the
         *  'hidden' attribute, and the sheet is locked, an empty string will
         *  be returned instead of the formula expression.
         */
        this.getEditFormula = function (address, options) {

            // get the raw formula expression, return null for non-formula cells
            var tokenDesc = this.getTokenArray(address, { grammarId: 'ui', fullMatrix: true });
            if (!tokenDesc) { return null; }

            // hide formula if the cell contains the hidden attribute, and the sheet is locked
            if (sheetModel.isLocked() && this.getAttributeSet(address).cell.hidden) { return ''; }

            // add the leading equality sign, add curly braces for matrix formulas
            var formula = '=' + tokenDesc.formula;
            if (tokenDesc.matrixRange && Utils.getBooleanOption(options, 'matrixBraces', false)) {
                formula = '{' + formula + '}';
            }

            return formula;
        };

        /**
         * Returns the text of the specified cell for the cell edit mode.
         * Returns the formula expression for formula cells (in 'ui' grammar
         * with the leading equality sign), otherwise the cell value formatted
         * with an appropriate default number format.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method CellCollection.getEditFormula().
         *
         * @returns {String}
         *  The text for the cell edit mode.
         */
        this.getEditString = function (address, options) {

            // prefer formula expression over formula result
            var formula = this.getEditFormula(address, options);
            if (formula !== null) { return formula; }

            // the result value of the cell to be formatted
            var value = this.getValue(address);
            var valueType = Scalar.getType(value);

            // strings: use plain unformatted string for editing (bug 34421: add an apostrophe if necessary)
            if ((valueType === Scalar.Type.STRING) && (value.length > 0)) {
                return ((/^['=]/).test(value) || (typeof numberFormatter.parseValue(value) !== 'string')) ? ('\'' + value) : value;
            }

            // booleans: use plain boolean literal for editing
            if (valueType === Scalar.Type.BOOLEAN) {
                return docModel.getFormulaGrammar('ui').getBooleanName(value);
            }

            // error codes: use plain error code literal for editing
            if (valueType === Scalar.Type.ERROR) {
                return docModel.getFormulaGrammar('ui').getErrorName(value);
            }

            // numbers: use appropriate number representation according to number format category
            if ((valueType === Scalar.Type.NUMBER) && isFinite(value)) {

                // the resulting formatted value
                var formatted = null;

                // process different format categories
                var category = this.getParsedFormat(address).category;
                switch (category) {

                    // percent: multiply by 100, add percent sign without whitespace
                    case 'percent':
                        value *= 100; // may become infinite
                        if (isFinite(value)) {
                            formatted = numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT) + '%';
                        }
                        break;

                    // automatically show date and/or time, according to the number
                    case 'date':
                    case 'time':
                    case 'datetime':

                        // number of milliseconds
                        var milliSecs = Math.round(value * DateUtils.MSEC_PER_DAY);
                        // number of days
                        var date = Math.floor(milliSecs / DateUtils.MSEC_PER_DAY);
                        // number of milliseconds in the day
                        var time = Math.floor(milliSecs % DateUtils.MSEC_PER_DAY);
                        // whether to add the date and time part to the result
                        var showDate = (category !== 'time') || (date !== 0);
                        var showTime = (category !== 'date') || (time !== 0);
                        // the resulting format code
                        var formatCode = (showDate ? DATE_FORMAT : '') + ((showDate && showTime) ? ' ' : '') + (showTime ? TIME_FORMAT : '');
                        // the parsed number format
                        var parsedFormat = numberFormatter.getParsedFormat(formatCode);

                        // the resulting formatted value (may be null for invalid dates)
                        formatted = numberFormatter.formatValue(parsedFormat, value);
                        break;
                }

                // use standard number format for all other format codes
                return (formatted !== null) ? formatted : numberFormatter.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT);
            }

            // empty cells
            return '';
        };

        // range iterators ----------------------------------------------------

        /**
         * Creates an iterator that generates the cell addresses for all, or
         * specific, cells contained in the passed cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, used
         *  to generate cell addresses from.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  SheetUtils.getIteratorRanges() to define a custom start position
         *  inside the passed cell ranges (i.e. the options "reverse",
         *  "startIndex", "startAddress", "skipStart", and "wrap"), and the
         *  following additional options:
         *  - {String} [options.type='all']
         *      Specifies which type of cells will be covered by the iterator.
         *      Must be one of the following values:
         *      - 'all': Generates the addresses for all defined and undefined
         *          cells in the passed ranges.
         *      - 'defined': Generates the addresses for all defined cells that
         *          really exist in this collection, regardless of their
         *          content value (including blank but formatted cells).
         *      - 'value': Generates the addresses for all non-blank cells
         *          (with any content value), regardless if the value is given
         *          literally, or is the result of a formula.
         *      - 'formula': Generates the addresses of all formula cells (i.e.
         *          all normal formula cells, all cells that are part of shared
         *          formulas, and the anchor cells of matrix formulas, but not
         *          the remaining cells of matrix formulas).
         *      - 'anchor': Generates the addresses of all formula cells with
         *          an actual formula definition (i.e. all normal formula
         *          cells, and the anchor cells of shared formulas and matrix
         *          formulas, but not the remaining cells of shared formulas
         *          and matrix formulas).
         *  - {Boolean|String} [options.visible=false]
         *      Specifies how to handle hidden columns and rows:
         *      - true (boolean): Only visible cells (cells in visible columns,
         *          AND visible rows) will be covered by the iterator.
         *      - 'columns': Cells in hidden columns will be skipped, but cells
         *          in visible columns and hidden rows will be visited.
         *      - 'rows': Cells in hidden rows will be skipped, but cells in
         *          visible rows and hidden columns will be visited.
         *      - false (boolean, default): By default, all visible and hidden
         *          cells in the passed cell ranges will be covered.
         *  - {Boolean} [options.covered=false]
         *      If set to true, visits all cells covered and hidden by a merge
         *      range. By default, covered cells in merged ranges will not be
         *      visited.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cell addresses in each range
         *      will be processed in reversed order.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {Address} value
         *      The address of the cell currently visited.
         *  - {Range} orig
         *      The address of the original cell range (from the passed range
         *      array) containing the current cell address.
         *  - {Number} index
         *      The array index of the original cell range contained in the
         *      result property 'orig'.
         */
        this.createAddressIterator = function (ranges, options) {

            // collection of merged ranges in the active sheet
            var mergeCollection = sheetModel.getMergeCollection();
            // whether to visit visible columns and/or rows only
            var visible = getVisibleMode(options);
            // the shrink method for merged ranges
            var shrinkMethod = visible.cols ? (visible.rows ? 'both' : 'columns') : (visible.rows ? 'rows' : null);
            // whether to visit cells covered (hidden) by merged ranges
            var covered = Utils.getBooleanOption(options, 'covered', false);
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);

            // returns the appropriate address iterator for the passed cell range
            function createIteratorForRange(range) {

                // create an iterator that visits the cell addresses in the range
                var iterator = createAddressIterator(range, options);
                if (covered || mergeCollection.isEmpty()) { return iterator; }

                // returns the merged ranges overlapping the current range on first invocation (for performance:
                // if the range does not contain any matching cells, the merged ranges will not be collected at all)
                var getMergedRanges = _.once(function () {

                    // the merged ranges covering the cell range currently iterated
                    var mergedRanges = mergeCollection.getMergedRanges(range);

                    // shrink merged ranges to their visible parts in visible cells mode
                    if (shrinkMethod && !mergedRanges.empty()) {
                        mergedRanges = RangeArray.map(mergedRanges, function (mergedRange) {
                            mergedRange = sheetModel.shrinkRangeToVisible(mergedRange, shrinkMethod);
                            // remove resulting single-cell ranges from the array
                            return (mergedRange && !mergedRange.single()) ? mergedRange : null;
                        });
                    }

                    return mergedRanges;
                });

                // filter cells that are covered by merged ranges
                return new FilterIterator(iterator, function (address) {
                    var mergedRange = getMergedRanges().findByAddress(address);
                    return !mergedRange || mergedRange.startsAt(address);
                });
            }

            // split and reorder the passed ranges according to the passed options (the new ranges will contain
            // the additional properties 'orig' and 'index'), and create an array iterator for these ranges
            var arrayIt = SheetUtils.getIteratorRanges(ranges, options).iterator({ reverse: reverse });

            // create the resulting iterator, combining the array iterator with an inner address iterator for each range
            return new NestedIterator(arrayIt, createIteratorForRange, function (arrayResult, rangeResult) {
                // the range from the range array iterator (contains original range and index)
                var iterRange = arrayResult.value;
                return { value: rangeResult.value, orig: iterRange.orig, index: iterRange.index };
            });
        };

        /**
         * Creates an iterator that generates the cell addresses for all, or
         * specific, cells contained in a single column or row, while moving
         * away from the start cell into the specified directions.
         *
         * @param {Address} address
         *  The address of the cell to start the iteration process from (the
         *  option 'skipStart' can be used, if iteration shall start with the
         *  nearest available neighbor of the start cell, and not with the cell
         *  itself).
         *
         * @param {String} directions
         *  Specifies how to move to the next cell addresses while iterating.
         *  Supported values are 'up', 'down', 'left', or 'right', or any space
         *  separated list of these values. Multiple directions will be
         *  processed in the same order as specified in this parameter.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.type='all']
         *      Specifies which type of cells will be covered by the iterator.
         *      See method CellCollection.createAddressIterator() for details.
         *  - {Boolean|String} [options.visible=false]
         *      Specifies how to handle hidden columns and rows. See method
         *      CellCollection.createAddressIterator() for details.
         *  - {Boolean} [options.covered=false]
         *      If set to true, visits all cells covered and hidden by a merge
         *      range. By default, covered cells in merged ranges will not be
         *      visited.
         *  - {Range} [options.boundRange]
         *      If specified, the line iterator will be restricted to that cell
         *      range. If omitted, the entire area of the sheet will be used.
         *  - {Boolean} [options.skipStart=false]
         *      If set to true, the iterator will start at the nearest neighbor
         *      of the cell specified in the 'address' parameter instead of
         *      that cell (nearest visible cell, if option 'visible' has been
         *      set).
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {Address} value
         *      The daddress of the cell currently visited.
         *  - {String} direction
         *      The position of the current cell, relative to the start cell
         *      (one of the directions passed to the parameter 'direction' of
         *      this method).
         */
        this.createLinearAddressIterator = function (address, directions, options) {

            // the bounding range to restrict the visited area
            var boundRange = Utils.getObjectOption(options, 'boundRange', docModel.getSheetRange());
            // number of cells to eb skipped before iteration starts
            var skipCount = Utils.getBooleanOption(options, 'skipStart', false) ? 1 : 0;

            // prepare the cell ranges to be visited
            var ranges = RangeArray.map(directions.split(/\s+/), function (direction) {

                // whether to iterate with variable column index
                var columns = /^(left|right)$/.test(direction);
                // whether to iterate in reversed order (towards first column/row)
                var reverse = /^(up|left)$/.test(direction);

                // check passed direction identifier
                if (!columns && !reverse && (direction !== 'down')) {
                    Utils.error('CellCollection.createLinearAddressIterator(): invalid direction "' + direction + '"');
                    return null;
                }

                // the entire column/row range containing the passed address
                var range = docModel.makeFullRange(address.get(!columns), !columns);

                // reduce range to leading or trailing part, according to start address and direction
                if (reverse) {
                    range.setEnd(address.get(columns) - skipCount, columns);
                } else {
                    range.setStart(address.get(columns) + skipCount, columns);
                }

                // skip the current direction, if the range becomes invalid (e.g. 'up' in first row);
                // reduce the range to the bounding range (this may invalidate the range too)
                range = (range.getStart(columns) <= range.getEnd(columns)) ? boundRange.intersect(range) : null;
                if (!range) { return null; }

                // put additional information into the range
                range.direction = direction;
                range.columns = columns;
                range.reverse = reverse;
                return range;
            });

            // create the nested iterator, combining an outer array iterator with an inner address iterator for each range
            return new NestedIterator(ranges.iterator(), function (range) {
                return self.createAddressIterator(range, _.extend({}, options, { reverse: range.reverse }));
            }, function (outerResult, innerResult) {
                innerResult.direction = outerResult.value.direction;
                return innerResult;
            });
        };

        /**
         * Creates an iterator that visits the equally formatted parts in the
         * passed cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, to be
         *  visited by the iterator.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  SheetUtils.getIteratorRanges() to define a custom start position
         *  inside the passed cell ranges (i.e. the options "reverse",
         *  "startIndex", "startAddress", "skipStart", and "wrap"), and the
         *  following additional options:
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible cells (cells in visible columns,
         *      AND visible rows) will be covered by the iterator. By default,
         *      all visible and hidden cells will be covered.
         *  - {Boolean} [options.covered=false]
         *      If set to true, visits all cells covered and hidden by a merge
         *      range. By default, covered cells in merged ranges will not be
         *      visited.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cell addresses in each range
         *      will be processed in reversed order.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {Range} value
         *      The current cell range address.
         *  - {String} style
         *      The auto-style identifier for all cells in the range.
         *  - {Range} orig
         *      The address of the original cell range (from the passed range
         *      array) containing the current cell range.
         *  - {Number} index
         *      The array index of the original cell range contained in the
         *      result property 'orig'.
         */
        this.createStyleIterator = function (ranges, options) {

            // collection of merged ranges in the active sheet
            var mergeCollection = sheetModel.getMergeCollection();
            // whether to visit visible cells only
            var visible = Utils.getBooleanOption(options, 'visible', false);
            // whether to visit cells covered (hidden) by merged ranges
            var covered = Utils.getBooleanOption(options, 'covered', false);
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);

            // Creates a style iterator for a single cell. Used as short-cut for single cells to improve
            // performance, to prevent to create all the column/row intervals, sub-iterators, etc.
            function createStyleIteratorForAddress(address) {
                var skipCell = !covered && mergeCollection.isHiddenCell(address);
                return skipCell ? Iterator.EMPTY : new SingleIterator({
                    value: new Range(address.clone()),
                    style: self.getStyleId(address)
                });
            }

            // Creates an iterator that tries to combine existing equally formatted cells into sub-ranges.
            // The result objects of the iterator will contain the following properties:
            // - {IntervalArray} colIntervals
            //      The column intervals of all equally formatted ranges of the iterator step; each
            //      interval in the array with the additional property 'style'.
            // - {Interval} rowInterval
            //      The row interval containing all equally formatted ranges of the iterator step.
            function createIteratorForExistingCells(range) {

                // creates a column interval for the cell in the passed result object of a cell model iterator
                function createColInterval(result) {
                    var colInterval = new Interval(result.model.a[0]);
                    colInterval.style = result.model.s;
                    return colInterval;
                }

                // creates and initializes the property 'colIntervals' in the passed result if missing
                function initializeColIntervals(result) {
                    return result.colIntervals || (result.colIntervals = new IntervalArray(createColInterval(result)));
                }

                // creates and initializes the property 'rowInterval' in the passed result if missing
                function initializeRowInterval(result) {
                    return result.rowInterval || (result.rowInterval = new Interval(result.model.a[1]));
                }

                // an iterator that visits all existing cells in the range
                var iterator = createAddressIterator(range, { type: 'defined', visible: visible, reverse: reverse });

                // filter for existing cells with an auto-style that differs from its active column/row style
                iterator = new FilterIterator(iterator, function (address, result) {
                    return !autoStyles.areEqualStyleIds(result.model.s, self.getDefaultStyleId(address));
                });

                // a combining iterator that reduces all existing formatted cells of a single row into a column interval array
                // with auto-style identifiers (inserted as new property 'colIntervals' into the iterator result)
                iterator = new ReduceIterator(iterator, function (result1, result2) {

                    // immediately return if the new cell model is located in another row (iterator will actually step to 'result1')
                    var cellModel = result2.model;
                    if (result1.model.a[1] !== cellModel.a[1]) { return; }

                    // get or create the column interval array (will be initialized with column index of result1 if missing)
                    var colIntervals = initializeColIntervals(result1);

                    // try to extend the interval array (adjacent cell with equal formatting)
                    var colInterval = reverse ? colIntervals.first() : colIntervals.last();
                    if (autoStyles.areEqualStyleIds(colInterval.style, cellModel.s)) {
                        if (!reverse && (colInterval.last + 1 === cellModel.a[0])) {
                            colInterval.last += 1;
                            return result1;
                        }
                        if (reverse && (cellModel.a[0] + 1 === colInterval.first)) {
                            colInterval.first -= 1;
                            return result1;
                        }
                    }

                    // create an interval for the new cell (push or unshift according to direction)
                    colInterval = createColInterval(result2);
                    if (reverse) { colIntervals.unshift(colInterval); } else { colIntervals.push(colInterval); }
                    return result1;
                });

                // a combining iterator that tries to collapse multiple rows with equally formatted existing cells
                iterator = new ReduceIterator(iterator, function (result1, result2) {

                    // initialize the row interval in the first iterator result if missing
                    var rowInterval = initializeRowInterval(result1);

                    // the second iterator result must represent a row that can extend the row interval
                    if (reverse ? (result2.model.a[1] + 1 !== rowInterval.first) : (rowInterval.last + 1 !== result2.model.a[1])) { return null; }

                    // create missing column intervals in both result objects for comparison
                    var colIntervals1 = initializeColIntervals(result1);
                    var colIntervals2 = initializeColIntervals(result2);

                    // compare the positions and auto-style identifiers of both column interval arrays
                    var equalIntervals = (colIntervals1.length === colIntervals2.length) && colIntervals1.every(function (colInterval1, index) {
                        var colInterval2 = colIntervals2[index];
                        return colInterval1.equals(colInterval2) && autoStyles.areEqualStyleIds(colInterval1.style, colInterval2.style);
                    });

                    // column intervals are equal: expand the row interval, and continue to search for adjacent rows
                    if (equalIntervals) {
                        if (reverse) { rowInterval.first -= 1; } else { rowInterval.last += 1; }
                        return result1;
                    }
                });

                // ensure that all iterator results contain column and row intervals (reduce iterators may skip single entries)
                return new TransformIterator(iterator, function (value, result) {
                    initializeColIntervals(result);
                    initializeRowInterval(result);
                    return result;
                });
            }

            // returns an interval array for the passed column interval (or the visible parts) to be used for rows with
            // an own active auto-style (rows with 'customFormat' flag set to true), and caches it for the iterator
            var getColIntervals = _.memoize(function (colInterval, rowStyleId) {
                var colIntervals = visible ? colCollection.getVisibleIntervals(colInterval) : new IntervalArray(colInterval.clone());
                return Utils.addProperty(colIntervals, 'style', rowStyleId);
            }, function (colInterval, rowStyleId) { return colInterval.toString() + ':' + rowStyleId; });

            // collect the default column auto-styles of the column interval to be used for unformatted rows that show
            // the default column auto-styles (rows with 'customFormat' flag set to false), and caches it for the iterator
            var getColStyleIntervals = _.memoize(function (colInterval) {
                var iterator = colCollection.createStyleIterator(colInterval, { visible: visible });
                return Iterator.reduce(new IntervalArray(), iterator, function (intervals, interval, result) {
                    interval.style = result.style;
                    intervals.push(interval);
                    return intervals;
                });
            }, function (colInterval) { return colInterval.toString(); });

            // collect the row bands of the merged ranges covering the passed cell range
            function getMergedRowBands(range) {

                // get merged ranges in the current range
                var mergedRanges = mergeCollection.getMergedRanges(range);
                if (mergedRanges.empty()) { return null; }

                // skip all cells but the top-left cell of the merged ranges
                var skipRanges = new RangeArray();
                mergedRanges.forEach(function (mergedRange) {
                    if (!mergedRange.singleCol()) { skipRanges.push(Range.create(mergedRange.start[0] + 1, mergedRange.start[1], mergedRange.end[0], mergedRange.start[1])); }
                    if (!mergedRange.singleRow()) { skipRanges.push(Range.create(mergedRange.start[0], mergedRange.start[1] + 1, mergedRange.end[0], mergedRange.end[1])); }
                });

                // reduce the ranges to the visible areas if the 'visible' flag is set, reduce to the passed cell range address
                if (visible) { skipRanges = RangeArray.map(skipRanges, sheetModel.shrinkRangeToVisible.bind(sheetModel)); }
                skipRanges = skipRanges.intersect(range);
                if (skipRanges.empty()) { return null; }

                // return the row bands containing the column intervals
                return skipRanges.getRowBands({ intervals: true });
            }

            // reduce the passed column intervals with auto-style identifiers to the parts not covered by merged ranges
            function reduceStyleColIntervals(styleColIntervals, skipColIntervals) {
                var partColIntervals = styleColIntervals.concat(skipColIntervals).partition();
                return partColIntervals.reject(function (partColInterval) {
                    return partColInterval.coveredBy.some(function (origColInterval) {
                        if (!('style' in origColInterval)) { return true; }
                        partColInterval.style = origColInterval.style;
                        delete partColInterval.coveredBy;
                    });
                });
            }

            // returns a style iterator for the passed equally formatted and completely visivle row interval,
            // and the entire column interval (with hidden columns that need to be ignored in visible mode)
            function createStyleIteratorForRowInterval(colInterval, rowInterval, rowStyleId) {

                // short-cut for single cells (do not create all the column/row intervals etc.)
                if (colInterval.single() && rowInterval.single()) {
                    return createStyleIteratorForAddress(new Address(colInterval.first, rowInterval.first));
                }

                // an iterator that provides row intervals with existing, equally-formatted cells (as column intervals)
                var range = Range.createFromIntervals(colInterval, rowInterval);
                // the column intervals with auto-style identifiers (use column auto-styles, if the row interval is not formatted)
                var colIntervals = (rowStyleId === null) ? getColStyleIntervals(colInterval) : getColIntervals(colInterval, rowStyleId);
                // an iterator that provides row intervals with existing, equally-formatted cells (as column intervals)
                var cellsIt = createIteratorForExistingCells(range);
                // the iterator provides settings for a row interval with existing formatted cells
                var cellsResult = cellsIt.next();
                // the row bands of the merged ranges covering the current range to be skipped
                var mergedRowBands = covered ? null : getMergedRowBands(range);
                // the current row index (for the gaps between row intervals of existing cells)
                var currRow = reverse ? rowInterval.last : rowInterval.first;

                // visit the row intervals with existing cells, and the gaps between these row intervals (blank rows)
                var rowIntervalIt = new GeneratorIterator(function () {

                    // the iterator is done if the current row moves outside the processed row interval
                    if (!rowInterval.containsIndex(currRow)) { return { done: true }; }

                    // the next row interval to be visited (either a gap between cells, or a row interval with cells)
                    var currRowInterval = null;
                    // the column intervals with style identifiers
                    var styleColIntervals = null;

                    // find the next row interval to be processed, and build the column intervals with auto-style identifiers
                    if (cellsResult.done) {
                        // no more formatted cells available: process the remaining rows to the end (reverse mode: to the beginning) of the entire row interval
                        currRowInterval = reverse ? new Interval(rowInterval.first, currRow) : new Interval(currRow, rowInterval.last);
                        styleColIntervals = colIntervals;
                    } else if (!cellsResult.rowInterval.containsIndex(currRow)) {
                        // 'currRow' is above the next formatted cells (reverse mode: below the cells): process the gap of blank rows
                        currRowInterval = reverse ? new Interval(cellsResult.rowInterval.last + 1, currRow) : new Interval(currRow, cellsResult.rowInterval.first - 1);
                        styleColIntervals = colIntervals;
                    } else {
                        // process a row interval that contains existing formatted cells
                        currRowInterval = reverse ? new Interval(cellsResult.rowInterval.first, currRow) : new Interval(currRow, cellsResult.rowInterval.last);
                        styleColIntervals = colCollection.mergeStyleIntervals(colIntervals, cellsResult.colIntervals);
                    }

                    // shorten the current row interval according to the next merged row band
                    var nextMergedRowBand = !mergedRowBands ? null : reverse ? mergedRowBands.last() : mergedRowBands.first();
                    if (nextMergedRowBand) {
                        if (nextMergedRowBand.containsIndex(currRow)) {
                            // current row interval starts in a row band with merged ranges
                            if (reverse) {
                                currRowInterval.first = Math.max(currRowInterval.first, nextMergedRowBand.first);
                            } else {
                                currRowInterval.last = Math.min(currRowInterval.last, nextMergedRowBand.last);
                            }
                            // reduce the column intervals to the parts that are not covered by merged ranges
                            styleColIntervals = reduceStyleColIntervals(styleColIntervals, nextMergedRowBand.intervals);
                        } else {
                            // current row interval starts before a row band with merged ranges (reverse mode: after a merged row band)
                            if (reverse) {
                                currRowInterval.first = Math.max(currRowInterval.first, nextMergedRowBand.last + 1);
                            } else {
                                currRowInterval.last = Math.min(currRowInterval.last, nextMergedRowBand.first - 1);
                            }
                        }
                    }

                    // index of the next row to be processed
                    currRow = reverse ? (currRowInterval.first - 1) : (currRowInterval.last + 1);

                    // fetch next cell result, if the row interval of the current result has been processed
                    if (!cellsResult.done && (reverse ? (currRow < cellsResult.rowInterval.first) : (cellsResult.rowInterval.last < currRow))) {
                        cellsResult = cellsIt.next();
                    }

                    // delete the row band for merged columns, if it has been processed
                    if (nextMergedRowBand && (reverse ? (currRow < nextMergedRowBand.first) : (nextMergedRowBand.last < currRow))) {
                        if (reverse) { mergedRowBands.pop(); } else { mergedRowBands.shift(); }
                    }

                    return { value: currRowInterval, colIntervals: styleColIntervals };
                });

                return new NestedIterator(rowIntervalIt, function (subRowInterval, result) {
                    var arrayIt = new ArrayIterator(result.colIntervals, { reverse: reverse });
                    return new TransformIterator(arrayIt, function (subColInterval) {
                        return { value: Range.createFromIntervals(subColInterval, subRowInterval), style: subColInterval.style };
                    });
                });
            }

            // creates an iterator that visits all equally formatted sub-ranges in the passed range
            function createStyleIteratorForRange(range) {

                // Reduce the passed range to the visible area if the 'visible' flag is set. Continue with
                // the next cell range, if no more visible cells are left (the range shrinks to null).
                if (visible && !(range = sheetModel.shrinkRangeToVisible(range))) { return null; }
                // short-cut for single cells (do not create all the column/row intervals etc.)
                if (range.single()) { return createStyleIteratorForAddress(range.start); }

                // create a nested iterator that returns the results of a style iterator for each row interval
                var rowStyleIt = rowCollection.createStyleIterator(range.rowInterval(), options);
                return new NestedIterator(rowStyleIt, function (rowInterval, result) {
                    return createStyleIteratorForRowInterval(range.colInterval(), rowInterval, result.style);
                });
            }

            // split and reorder the passed ranges according to the passed options (the new ranges will contain
            // the additional properties 'orig' and 'index'), and create an array iterator for these ranges
            var arrayIt = SheetUtils.getIteratorRanges(ranges, options).iterator({ reverse: reverse });

            // create the resulting iterator, combining the array iterator with an inner style iterator for each range
            return new NestedIterator(arrayIt, createStyleIteratorForRange, function (arrayResult, styleResult) {
                // the range from the range array iterator (contains original range and index)
                var range = arrayResult.value;
                return { value: styleResult.value, style: styleResult.style, orig: range.orig, index: range.index };
            });
        };

        /**
         * Returns an iterator that provides the subtotal results of specific
         * subranges and/or cells contained in the passed ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, whose
         *  subtotals will be returned.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the subtotal
         *  result (class SubtotalResult) as value.
         */
        this.createSubtotalIterator = function (ranges) {

            // restrict the used range to its visible part (subtotals are collected from visible cells only)
            var usedRange = this.getVisibleUsedRange();
            // restrict the passed ranges to the visible used range of this collection
            ranges = usedRange ? RangeArray.get(ranges).intersect(usedRange) : null;

            // return immediately if the ranges do not cover any existing cells
            if (!ranges || ranges.empty()) { return new SingleIterator({ value: new SubtotalResult() }); }

            // get the column intervals of the ranges covering the used range vertically
            var colIntervals = RangeArray.filter(ranges, function (range) {
                return (range.start[1] === usedRange.start[1]) && (range.end[1] === usedRange.end[1]);
            }).colIntervals().merge();

            // get the row intervals of the ranges covering the used range horizontally
            var rowIntervals = RangeArray.filter(ranges, function (range) {
                return (range.start[0] === usedRange.start[0]) && (range.end[0] === usedRange.end[0]);
            }).rowIntervals().merge();

            // decide whether to use the cached subtotals of the column or row intervals (more covered cells)
            var coveredCellsByCols = colIntervals.size() * usedRange.rows();
            var coveredCellsByRows = rowIntervals.size() * usedRange.cols();
            if (coveredCellsByRows < coveredCellsByCols) {
                rowIntervals.clear();
                ranges = ranges.difference(RangeArray.createFromIntervals(colIntervals, usedRange.rowInterval()));
            } else {
                colIntervals.clear();
                ranges = ranges.difference(RangeArray.createFromIntervals(usedRange.colInterval(), rowIntervals));
            }

            // Get the subtotal results of the column/row intervals from the matrix (cached internally), and remove
            // the column or row ranges from the passed ranges (the remaining ranges will be iterated individually).
            var matrixIt = null;
            if (!colIntervals.empty()) {
                matrixIt = colMatrix.createSubtotalIterator(colIntervals);
                ranges = ranges.difference(RangeArray.createFromIntervals(colIntervals, usedRange.rowInterval()));
            } else if (!rowIntervals.empty()) {
                matrixIt = rowMatrix.createSubtotalIterator(rowIntervals);
                ranges = ranges.difference(RangeArray.createFromIntervals(usedRange.colInterval(), rowIntervals));
            }

            // Create a cell iterator (do not collect the subtotals of a cell multiple times, always visit the cells
            // covered by merged ranges which is faster and matches the behavior of Excel where covered cells are
            // always empty, and OpenOffice where covered cells are always counted into the subtotals).
            var cellIt = null;
            if (!ranges.empty()) {
                cellIt = createAddressIterator(ranges.merge(), { type: 'value', visible: true });
                cellIt = new TransformIterator(cellIt, function (address, result) { return { value: result.model.v }; });
            }

            // create a combined iterator that collects cached subtotals, and separate cell values
            return new SerialIterator(matrixIt, cellIt);
        };

        // range data access --------------------------------------------------

        /**
         * Finds the address of the first available cell of a specific type in
         * the passed cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, to be
         *  searched.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method CellCollection.createAddressIterator().
         *
         * @returns {Address|Null}
         *  The address of the first available cell matching the specified cell
         *  type; or null, if no cell has been found.
         */
        this.findFirstCell = function (ranges, options) {
            // find the first cell in the search ranges
            var result = this.createAddressIterator(ranges, options).next();
            return result.done ? null : result.value;
        };

        /**
         * Returns the address of the next available cell of a specific type
         * (any blank cells, defined cells only, or content cells only) in the
         * specified direction.
         *
         * @param {Address} address
         *  The address of the cell whose nearest adjacent cell will be
         *  searched.
         *
         * @param {String} direction
         *  The direction to look for the cell. Must be one of the values
         *  'left', 'right', 'up', or 'down'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.type='all']
         *      Specifies the type of the next cell to be searched. See method
         *      CellCollection.createAddressIterator() for details.
         *  - {Boolean} [options.visible=false]
         *      If set to true, only a visible cell (in a visible column, AND a
         *      visible row) will be found. By default, all visible and hidden
         *      cells will be searched.
         *  - {Range} [options.boundRange]
         *      If specified, the search area will be restricted to that cell
         *      range. If omitted, the entire area of the sheet will be used.
         *
         * @returns {Address|Null}
         *  The address of the next available cell matching the specified cell
         *  type; or null, if no cell has been found.
         */
        this.findFirstCellLinear = function (address, direction, options) {
            // get the first result object of a linear iterator into the specified direction
            var result = this.createLinearAddressIterator(address, direction, _.extend({}, options, { skipStart: true })).next();
            return result.done ? null : result.value;
        };

        /**
         * Returns whether all passed ranges are blank (no values). Ignores
         * cell formatting attributes.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Boolean}
         *  Whether all cells in the passed ranges are blank.
         */
        this.areRangesBlank = function (ranges) {
            // return whether a cell iterator does not find a single value cell
            ranges = RangeArray.get(ranges).merge();
            return createAddressIterator(ranges, { type: 'value' }).next().done;
        };

        /**
         * Returns the contents of all cells in the passed cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, whose
         *  cell contents will be returned.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.blanks=false]
         *      If set to true, all blank cells will be included in the result.
         *      By default, only non-blank cells will be collected.
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible cells (cells in visible columns,
         *      AND in visible rows) will be included in the result. Otherwise,
         *      all visible and hidden cells will be returned.
         *  - {Boolean} [options.covered=false]
         *      If set to true, includes all cells covered (hidden) by a merge
         *      range into the result. By default, covered cells in merged
         *      ranges will not be included.
         *  - {Boolean} [options.attributes=false]
         *      If set to true, the result will contain the identifier of the
         *      cell auto-style, the merged formatting attributes, and the
         *      parsed number format of the cells.
         *  - {Boolean} [options.display=false]
         *      If set to true, the result will contain the formatted display
         *      strings of the cells in the property 'display'.
         *  - {Boolean} [options.compressed=false]
         *      If set to true, the result array will be optimized: Consecutive
         *      cells with equal contents and formatting will be represented by
         *      a single array element with an additional property 'count'.
         *      Especially useful, if large ranges located in the unused area
         *      of the sheet are queried.
         *  - {Number} [options.maxCount]
         *      If specified, the maximum number of cells that will be returned
         *      in the result, regardless how large the passed ranges are.
         *
         * @returns {Array<Object>}
         *  The contents of the cells in the passed ranges. The result will be
         *  an array of cell content objects. Each cell content object will
         *  contain the following properties:
         *  - {Number|String|Boolean|ErrorCode|Null} value
         *      The cell value, or formula result (null for blank cells).
         *  - {Object} [style]
         *      The identifier of the cell auto-style. Will only be set, if the
         *      option 'attributes' has been set to true (see above).
         *  - {Object} [attributes]
         *      The merged formatting attributes of the cell, resolved from the
         *      cell auto-style. Will only be set, if the option 'attributes'
         *      has been set to true (see above).
         *  - {ParsedFormat} [format]
         *      The parsed number format code of the cell. Will only be set, if
         *      the option 'attributes' has been set to true (see above).
         *  - {String} [display]
         *      The display string of the cell, according to its current value
         *      and number format. If the value cannot be formatted with the
         *      current number format, the string '#######' will be used
         *      instead. Will only be set, if the option 'display' has been set
         *      to true (see above).
         *  - {Number} [count]
         *      Always set in compressed mode (see option 'compressed' above).
         *      Contains the number of consecutive cells with equal contents
         *      and formatting represented by this array element.
         *  The cells in the array will be in order of the passed ranges. Cells
         *  in each single range will be collected row-by-row. The result array
         *  will contain the additional property 'count' that represents the
         *  total number of cells contained in the result (this value will be
         *  different to the length of the array in compressed mode).
         */
        this.getRangeContents = function (ranges, options) {

            // whether to include blank cells
            var blanks = Utils.getBooleanOption(options, 'blanks', false);
            // whether to skip cells of hidden columns/rows
            var visible = Utils.getBooleanOption(options, 'visible', false);
            // whether to cells covered by merged ranges
            var covered = Utils.getBooleanOption(options, 'covered', false);
            // compressed mode
            var compressed = Utils.getBooleanOption(options, 'compressed', false);
            // the maximum number of cells to be included in the result
            var maxCount = Utils.getIntegerOption(options, 'maxCount', null, 1);
            // the result array returned by this method
            var contents = [];
            // the number of cells already inserted into the result (differs to array length in compressed mode)
            contents.count = 0;

            // creates a new result entry with value property, and optional display property
            var makeEntry = Utils.getBooleanOption(options, 'display', false) ?
                function (value, display) { return { value: value, display: (display === null) ? '#######' : display }; } :
                function (value) { return { value: value }; };

            // inserts the passed result entry into the array (handles compressed mode)
            function pushEntry(entry, count) {

                // reduce the passed count, if the limit will be reached
                if (maxCount) { count = Math.min(count, maxCount - contents.count); }
                if (count === 0) { return; }

                // update the total count of cells in the result array
                contents.count += count;

                // the preceding result, for compressed mode
                var prevEntry = compressed ? _.last(contents) : null;

                // increase count of previous entry, if the new entry is equal
                if (prevEntry) {
                    entry.count = prevEntry.count;
                    if ((prevEntry.value === entry.value) && (prevEntry.style === entry.style) && (prevEntry.display === entry.display)) {
                        prevEntry.count += count;
                        return;
                    }
                }

                // insert the new entry into the array (prepare count for compressed mode)
                if (compressed) {
                    entry.count = count;
                    contents.push(entry);
                } else {
                    for (; count > 0; count -= 1) { contents.push(entry); }
                }
            }

            // pushes result entries for blank cells without formatting attributes
            var pushBlanks = blanks ? function (count) {
                pushEntry(makeEntry(null, ''), count);
            } : _.noop;

            // collects all cells in the passed range without formatting attributes
            function collectWithoutAttributes(range) {

                // linear index of next expected cell in the range (used to fill blank cells)
                var blankIndex = 0;

                // fill the result array with existing cells and preceding blank cells
                var addressIt = self.createAddressIterator(range, { type: 'value', visible: visible, covered: covered });
                Iterator.forEach(addressIt, function (address) {

                    // the linear index of the address in the range
                    var index = range.indexAt(address);
                    // the cell model at the specified address (exists always)
                    var cellModel = self.getCellModel(address);

                    // insert blank cell entries before the current cell model, and the cell model
                    pushBlanks(index - blankIndex);
                    pushEntry(makeEntry(cellModel.v, cellModel.d), 1);
                    blankIndex = index + 1;

                    // early exit, if the limit has been reached
                    if (maxCount === contents.count) { return Utils.BREAK; }
                });

                // fill up following blank cells
                pushBlanks(range.cells() - blankIndex);
            }

            // collects all cells in the passed range with formatting attributes
            function collectWithAttributes(range) {
                // TODO: use a cell iterator that collects equally formatted blank ranges
                Iterator.forEach(self.createAddressIterator(range, { type: blanks ? 'all' : 'value', visible: visible, covered: covered }), function (address) {
                    var entry = makeEntry(self.getValue(address), self.getDisplayString(address));
                    entry.style = self.getStyleId(address);
                    entry.attributes = autoStyles.getMergedAttributeSet(entry.style);
                    entry.format = autoStyles.getParsedFormat(entry.style);
                    pushEntry(entry, 1);
                    // early exit, if the limit has been reached
                    if (maxCount === contents.count) { return Utils.BREAK; }
                });
            }

            // collect all cell contents
            var addAttributes = Utils.getBooleanOption(options, 'attributes', false);
            RangeArray.forEach(ranges, addAttributes ? collectWithAttributes : collectWithoutAttributes);
            return contents;
        };

        /**
         * Returns the content range (the entire range containing consecutive
         * content cells) surrounding the specified cell range.
         *
         * @param {Range} range
         *  The address of the cell range whose content range will be searched.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.directions='up down left right']
         *      All directions the passed range will be expanded to, as
         *      space-separated token list with the tokens 'up', 'down',
         *      'left', and 'right'. If omitted, the range will be expanded to
         *      all four directions.
         *
         * @returns {Range}
         *  The address of the content range of the specified cell range.
         */
        this.findContentRange = SheetUtils.profileMethod('CellCollection.findContentRange()', function (range, options) {

            // collection of merged ranges in the active sheet
            var mergeCollection = sheetModel.getMergeCollection();
            // the resulting content range (start by expanding to merged ranges)
            var currRange = mergeCollection.expandRangeToMergedRanges(range);
            // the result of the last iteration, to detect exiting the loop
            var lastRange = null;
            // all directions the range will be expanded to
            var directions = Utils.getTokenListOption(options, 'directions', ['up', 'down', 'left', 'right']);

            // returns the next column/row index outside the passed cell range
            function getNextIndex(srcRange, forward, columns) {
                // index of the next column/row outside the range
                var startIndex = forward ? (srcRange.getEnd(columns) + 1) : (srcRange.getStart(columns) - 1);
                // return the index if it is valid, otherwise -1
                return (forward ? (startIndex > docModel.getMaxIndex(columns)) : (startIndex < 0)) ? -1 : startIndex;
            }

            // array iterator function, must be defined outside the do-while loop
            function expandRangeBorder(direction, index) {

                // whether to move towards the end of the sheet
                var forward = /^(down|right)$/.test(direction);
                // whether to move through columns in the same row
                var columns = /^(left|right)$/.test(direction);
                // the descriptor of the next column/row
                var nextIndex = getNextIndex(currRange, forward, columns);

                // do nothing, if the range already reached the sheet borders
                if (nextIndex < 0) {
                    directions.splice(index, 1);
                    return false;
                }

                // build a single column/row range next to the current range
                var boundRange = currRange.clone();
                boundRange.setBoth(nextIndex, columns);

                // expand sideways, unless these directions are not active anymore
                if (_.contains(directions, columns ? 'up' : 'left')) {
                    nextIndex = getNextIndex(boundRange, false, !columns);
                    if (nextIndex >= 0) { boundRange.setStart(nextIndex, !columns); }
                }
                if (_.contains(directions, columns ? 'down' : 'right')) {
                    nextIndex = getNextIndex(boundRange, true, !columns);
                    if (nextIndex >= 0) { boundRange.setEnd(nextIndex, !columns); }
                }

                // find a content cell next to the current range
                var result = createAddressIterator(boundRange, { type: 'value' }).next();
                if (result.done) { return; }

                // expand the current range to the found content cell
                var currAddress = forward ? currRange.end : currRange.start;
                currAddress.set(result.value.get(columns), columns);

                // performance: expand further into the direction as long as there are adjacent content cells
                var iterator = self.createLinearAddressIterator(result.value, direction, { skipStart: true });
                Iterator.some(iterator, function (address) {
                    if (self.isBlankCell(address)) { return true; }
                    currAddress.set(address.get(columns), columns);
                });
            }

            // expand in all four directions until the range does not change anymore
            do {

                // store result of last iteration for comparison below
                lastRange = currRange.clone();

                // expand current range into all remaining directions
                Utils.iterateArray(directions, expandRangeBorder, { reverse: true });

                // expand the range to include merged ranges
                currRange = mergeCollection.expandRangeToMergedRanges(currRange);

            } while ((directions.length > 0) && currRange.differs(lastRange));

            // return the resulting range
            return currRange;
        });

        /**
         * Returns the address of the first or last cell with the specified
         * formatting attributes in the passed cell ranges, regardless whether
         * the cell is actually defined, or the attribute values have been
         * derived from column or row attributes. Uses an optimized search
         * algorithm for the empty areas between the defined cells, regarding
         * intervals of equally formatted columns and rows, and the position of
         * merged ranges covering these column/row default formatting.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, whose
         *  cells will be searched for the attributes.
         *
         * @param {Object} attributeSet
         *  The incomplete attribute set to be matched against the formatting
         *  attributes in the passed cell ranges.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method CellCollection.createStyleIterator().
         *
         * @returns {Object|Null}
         *  A result descriptor for the first (or last, depending on direction)
         *  cell with matching formatting attributes; or null, if no such cell
         *  exists in the passed cell ranges. The cell descriptor will contain
         *  the following properties:
         *  - {Address} address
         *      The address of the first cell with the matching attributes.
         *  - {Range} range
         *      The original range from the passed range array containing the
         *      cell with the matching attributes.
         *  - {Number} index
         *      The array index of the cell range contained in the property
         *      'range'.
         */
        this.findCellWithAttributes = SheetUtils.profileMethod('CellCollection.findCellWithAttributes()', function (ranges, attributeSet, options) {

            // whether to find the last matching cell
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            // create a style iterator that visits equally formatted cell ranges in row bands
            var iterator = this.createStyleIterator(ranges, options);

            // filter for ranges with matching formatting attributes
            iterator = new FilterIterator(iterator, function (range, result) {
                var cellAttributeSet = autoStyles.getMergedAttributeSet(result.style);
                return AttributeUtils.matchesAttributesSet(cellAttributeSet, attributeSet);
            });

            // fetch the first result from the iterator
            var result = iterator.next();
            if (result.done) { return null; }

            // return a result descriptor on success
            var address = reverse ? result.value.end : result.value.start;
            return { address: address.clone(), range: result.orig, index: result.index };
        });

        /**
         * Returns whether the passed cell range contains a header row
         * according to the cell contents.
         *
         * @param {Range} range
         *  A single cell range address, whose cells will be sorted.
         *
         * @returns {Boolean}
         *  Whether the passed cell range contains a header row according to
         *  the cell contents.
         */
        this.hasHeaderRow = function (range) {

            // return the type of the cell-value
            function detectValueType(cellModel) {
                if (!cellModel) { return null; }

                if (cellModel.isAnyFormula()) {
                    return 'formula';
                }

                if (cellModel.isNumber()) {
                    return cellModel.pf.isAnyDateTime() ? 'date' : 'number';
                }

                if (cellModel.isText()) {
                    return listCollection.listContains(cellModel.v) ? 'definedList' : 'string';
                }

                return cellModel.isBlank() ? '' : null;
            }

            // return an array of types from a range
            function typeDetection(range) {
                return Iterator.map(range, function (address) {
                    return detectValueType(self.getCellModel(address));
                });
            }

            // decides whether a header-line is present, or not
            function hasHeadline(headType, contentTypes) {
                if (
                    (headType === 'definedList'         && _.uniq(contentTypes).length >= 1) ||
                    (headType === 'date'                && _.uniq(contentTypes).length >= 1) ||
                    (headType === null                  && _.uniq(contentTypes).length >= 1) ||
                    (_.uniq(contentTypes).length === 1  && contentTypes[0] !== headType)
                ) {
                    return 'yes';
                }

                if (
                    (headType === 'number'  && _.uniq(contentTypes).length >= 1) ||
                    (headType === 'string'  && _.uniq(contentTypes).length === 1 && contentTypes[0] === 'string') ||
                    (headType === 'number'  && _.uniq(contentTypes).length === 1 && contentTypes[0] === 'number') ||
                    (headType === 'date'    && _.uniq(contentTypes).length === 1 && contentTypes[0] === 'date') ||
                    (headType === 'formula' && _.uniq(contentTypes).length === 1 && contentTypes[0] === 'formula') ||
                    (headType === 'formula' && _.uniq(contentTypes).length >= 1)
                ) {
                    return 'no';
                }

                return 'maybe';
            }

            var startCol                = range.start[0],
                startRow                = range.start[1],

                tableCollection         = sheetModel.getTableCollection(),
                arrTables               = tableCollection.findTables(range, { autoFilter: true }),

                arrHeadTypes            = [],
                arrContentTypes         = [],

                states                  = { yes: 0, no: 0, maybe: 0 };

            // when the selection covers exactly one table, go on ...
            //   ... otherwise, it isn't relevant whether the tables have headerRows or not
            if (arrTables.length === 1) {
                if (arrTables[0].getRange().equals(range)) {
                    return arrTables[0].hasHeaderRow();
                }
            }

            // creates an object which counts 'yes', 'no' and 'maybe' to decide whether
            // we have a headline or not
            Iterator.forEach(range.colInterval(), function (index) {
                var currentRange = range.colRange(index - startCol);
                var contentRange = currentRange.clone().setStart(startRow + 1, false);

                var headType = detectValueType(self.getCellModel(currentRange.start));
                arrHeadTypes.push(headType);

                var contentTypes = typeDetection(contentRange);
                arrContentTypes = arrContentTypes.concat(contentTypes);

                if (hasHeadline(headType, contentTypes) === 'yes') {
                    states.yes++;

                } else if (hasHeadline(headType, contentTypes) === 'no') {
                    states.no++;

                } else {
                    states.maybe++;
                }
            });

            // if the leading line is empty, there is no header
            if (_.uniq(arrHeadTypes).length === 1 && arrHeadTypes[0] === null) {
                return false;
            }

            // when 'yes' predominates, return 'true'
            if (states.yes > states.no) {
                return true;
            }

            // when 'no' predominates, check 'yes' + 'maybe' against 'no'
            if (states.no > states.yes) {
                return states.yes + states.maybe > states.no;
            }

            // otherwise, check 'maybe'
            return states.maybe > 0;
        };

        /**
         * Returns the source range for an automatically generated subtotal
         * formula for a single cell.
         *
         * @param {Address} address
         *  The address of the target cell for the subtotal formula.
         *
         * @returns {Range|Null}
         *  The source range for a subtotal formula for the specified target
         *  cell; or null, if no such range could be found.
         */
        this.findAutoFormulaRange = function (address) {

            var top = getAutoFormulaRange(address, 'up', true);
            var left = getAutoFormulaRange(address, 'left', true);

            // prefer the range located nearer to the target cell
            if (top && left) {
                var preferLeft = Math.abs(address[0] - left.address1[0]) < Math.abs(address[1] - top.address1[1]);
                return preferLeft ? left.range : top.range;
            }

            if (top) { return top.range; }
            if (left) { return left.range; }

            return null;
        };

        // operation generators -----------------------------------------------

        /**
         * Converts the passed text to a cell contents object as expected by
         * the other operation generator methods of this class.
         *
         * @param {String} parseText
         *  The text to be parsed to a cell contents object. The text may
         *  represent a plain or formatted number, a boolean value, an error
         *  code, or a formula expression.
         *
         * @param {String} contextType
         *  The context type used to evaluate the formula result. See method
         *  FormulaInterpreter.interpretTokens() for  details.
         *
         * @param {Address} refAddress
         *  The reference address used to interpret a formula expression.
         *
         * @param {Range} [matrixRange]
         *  The bounding range of a matrix formula. Used if the passed context
         *  type is 'mat'. If omitted, a 1x1 matrix formula is assumed. MUST
         *  contain the passed reference address.
         *
         * @returns {Object}
         *  The result object, with the following properties:
         *  - {Number|String|Boolean|ErrorCode|Null} v
         *      The resulting value with correct data type, or the calculated
         *      result of a formula expression.
         *  - {String|Null} f
         *      The formula expression, translated to the native formula
         *      grammar used in document operations, without leading equality
         *      sign; or null, if the passed text is not a formula expression.
         *  - {Object|Null} result
         *      The result descriptor of the formula interpreter, as returned
         *      by the method FormulaInterpreter.interpretTokens(); or null, if
         *      the passed text is not a formula expression.
         *  - {String|Number} [format]
         *      A specific number format code (as string), or the identifier of
         *      a preset format code (as integer), associated to the parsed
         *      number, or the result of the formula. If omitted, the number
         *      format shall not be modified.
         *  - {String} [url]
         *      The trimmed URL, if the passed text represents a valid URL; or
         *      an empty string, if the cell currently contains a hyperlink,
         *      and the passed text is an empty string (causes to delete the
         *      existing hyperlink).
         */
        this.parseCellValue = function (parseText, contextType, refAddress, matrixRange) {

            // the resulting contents object
            var contents = { v: parseText, f: null, result: null };

            // do not parse the text for the number format category 'text'
            if (this.isTextFormat(refAddress)) { return contents; }

            // handle specific strings not to be recognized as formulas
            if (/^[-+=]+$/.test(parseText)) { parseText = '\'' + parseText; }

            // try to parse the text as (formatted) value (empty strings will result in value null)
            var parseResult = numberFormatter.parseFormattedValue(parseText, { blank: true });
            contents.v = parseResult.value;

            // the identifier or format code of a number format (may be changed via formula result)
            var parsedFormat = numberFormatter.getParsedFormat(parseResult.format);

            // If the parsed value is still a string (no conversion to numbers etc. possible), and the original
            // string starts with specific characters, parse and interpret a formula expression in UI grammar.
            // This check excludes simple numbers with leading sign, e.g. '-1234' (which have been converted to
            // numbers already), but includes other formula expressions starting with a sign character instead
            // of an equality sign, e.g. '-A1+A2'.
            if ((typeof contents.v === 'string') && /^[-+=]./.test(parseText)) {

                // parse the formula in UI grammar, and translate to native grammar
                var tokenArray = sheetModel.createCellTokenArray();
                tokenArray.parseFormula('ui', parseText.replace(/^=/, ''), { autoCorrect: true, refAddress: refAddress });
                contents.f = tokenArray.getFormula('op', { refAddress: refAddress, targetAddress: refAddress });

                // calculate the formula result, handle formula errors
                var result = contents.result = tokenArray.interpretFormula(contextType, { refAddress: refAddress, targetAddress: refAddress, matrixRange: matrixRange });
                contents.v = Scalar.getCellValue(result.value);

                // the formula engine may provide an arbitrary number format with the result
                if ('format' in result) { parsedFormat = result.format; }
            }

            // add a formatting attribute for the number format, if the cell's category changes, but ignore
            // the standard number formats, and ignore changes inside any of the date/time categories
            // (e.g. do not change a date cell to a time cell)
            var formatCode = parsedFormat ? getNewCellFormatCode(refAddress, parsedFormat) : null;
            if (formatCode) { contents.format = formatCode; }

            // try to parse a URL from the passed text
            if (contents.v === null) {
                // delete the URL explicitly (via an empty string), if text is cleared
                if (sheetModel.getHyperlinkCollection().getCellURL(refAddress)) { contents.url = ''; }
            } else if (!contents.f && _.isString(contents.v)) {
                var url = HyperlinkUtils.checkForHyperlink(contents.v);
                if (_.isString(url) && (url.length > 0)) { contents.url = url; }
            }

            return contents;
        };

        /**
         * Generates the content objects needed to update the bounding ranges
         * and formula expressions of the anchor cells of the shared formulas.
         *
         * @returns {Array<Object>}
         *  The addresses of all changed cells with their cell content objects.
         *  The array will be sorted by cell address. Each array element is an
         *  object with the following properties:
         *  - {Address} address
         *      The address of the cell that is part of a shared formula.
         *  - {Object} contents
         *      The cell contents object, as exprected by a cell operations
         *      builder (class CellOperationsBuilder).
         */
        this.generateUpdateSharedFormulaContents = function () {

            // the resulting cell contents descriptors
            var contentsArray = [];

            // process all shared formulas in this cell collection
            sharedModelMap.forEach(function (sharedModel) {

                // skip shared formulas that have been deleted completely
                if (sharedModel.addressSet.empty() || !sharedModel.refAddress) { return; }

                // get the current anchor cell (may be missing)
                var oldAnchor = sharedModel.anchorAddress;
                var oldAnchorModel = oldAnchor ? this.getCellModel(oldAnchor) : null;

                // get the new anchor cell and bounding range of the shared formula
                var newAnchor = sharedModel.addressSet.first();
                var newRange = sharedModel.addressSet.boundary();
                var newAnchorModel = this.getCellModel(newAnchor);
                if (!newAnchorModel) { return; }

                // cell properties to be applied for the old and new anchor cell
                var oldContents = {};
                var newContents = {};

                // remove the bounding range from the old anchor cell (e.g. after expanding the shared formula)
                if (oldAnchorModel && (oldAnchorModel !== newAnchorModel) && (oldAnchorModel.si === sharedModel.index)) {
                    if (oldAnchorModel.f) { oldContents.f = null; }
                    if (oldAnchorModel.sr) { oldContents.sr = null; }
                }

                // update the bounding range and formula expression of the shared formula in the new anchor cell
                if (!newAnchorModel.sr || newAnchorModel.sr.differs(newRange)) {
                    newContents.sr = newRange;
                    var formula = sharedModel.getFormula('op', newAnchor);
                    if (newAnchorModel.f !== formula) { newContents.f = formula; }
                }

                // create the cell contents objects, if something will be changed
                if (!_.isEmpty(oldContents)) {
                    contentsArray.push({ address: oldAnchor, contents: oldContents });
                }
                if (!_.isEmpty(newContents)) {
                    contentsArray.push({ address: newAnchor, contents: newContents });
                }

            }, this);

            // sort the contents descriptors by cell address
            contentsArray.sort(function (entry1, entry2) {
                return entry1.address.compareTo(entry2.address);
            });

            return contentsArray;
        };

        /**
         * Generates the content objects needed to fill selected ranges
         * with clipboard-contents repeatedly.
         *
         * @param {RangeArray} targetRanges
         *  The selected ranges we want to paste to.
         *
         * @param {Object} contents
         *  The single contents object generated out of the clipboard.
         *
         * @param {Object} [options]
         *  Optional parameters.
         *  - {Boolean} intern
         *      Copy/Paste inside the appsuite.
         *  - {Boolean} cut
         *      Cut or Copy event.
         *  - {Object} selection
         *      The current selection.
         *  - {Range} sourceRange
         *      The source range. Only at internal copy.
         *
         * @returns {Array<Object>}
         *  All calculated single ranges with contents object,
         *  merged ranges and some more for each pastable ranges. Furthermore it
         *  contains the conditional formats, if at least one exists.
         */
        this.getRepeatedCellContents = function (targetRanges, contents, options) {

            function getSourceContentsObject(sourceStart, targetStart) {

                var tokenArray   = sheetModel.createCellTokenArray(),
                    lastFormula  = null;

                var values = _.map(contents, function (row) {
                    var rowObject = { c: [] };
                    rowObject.c = _.map(row.c, function (cell) {
                        if (intern) {
                            cell = _.clone(cell);
                            if (cell.f) {
                                tokenArray.parseFormula('op:ooxml', cell.f, { refAddress: sourceStart });
                                if (cut) {
                                    cell.f = tokenArray.getFormula('op');
                                } else {
                                    cell.f = tokenArray.getFormula('op', { refAddress: sourceStart, targetAddress: targetStart });
                                }

                                if (cell.mr) {
                                    cell.mr.moveBoth((targetStart[0] - sourceStart[0]), true);
                                    cell.mr.moveBoth((targetStart[1] - sourceStart[1]), false);
                                }

                                lastFormula = cell.f;
                            }

                            if (cell.format === null) { delete cell.format; }
                            return cell;
                        }

                        var parsed = { value: cell.v };
                        // if the value is a string, try to parse a format
                        if (_.isString(cell.v)) {
                            // only external paste values will be parsed
                            parsed = numberFormatter.parseFormattedValue(cell.v, { blank: true });
                        }
                        var cellProperties = {
                            v: parsed.value,
                            f: null,
                            a: cell.a,
                            s: null
                        };
                        if (parsed.format) {
                            cellProperties.format = parsed.format;
                        }
                        return cellProperties;
                    });

                    return rowObject;
                });

                // calculate new result-value directly, before pasting (prevents flickering).
                if (lastFormula && values.length === 1) {
                    var firstRow = values[0].c;
                    if (firstRow.length === 1) {
                        tokenArray.parseFormula('op', lastFormula, { refAddress: targetStart });
                        var result = tokenArray.interpretFormula('val', { refAddress: targetStart, targetAddress: targetStart });
                        firstRow[0].v = result.value;
                        // TODO: handle fatal errors of interpreter, remove formula from cell data
                    }
                }

                return values;
            }

            var intern          = Utils.getBooleanOption(options, 'intern', false),
                cut             = Utils.getBooleanOption(options, 'cut', false),
                selection       = Utils.getObjectOption(options, 'selection'),
                sourceRange     = Utils.getObjectOption(options, 'sourceRange'),
                startAddress    = (!sourceRange) ? selection.address : sourceRange.start,
                newTargetRanges = [];

            // we have no sourceRange, so the copy comes from extern
            //   or
            // sourceRange does not fit in the targetRange
            if (!sourceRange || !targetRanges.every(sourceRange.sizeFitsInto, sourceRange)) {
                newTargetRanges.push({
                    address: targetRanges[0].start,
                    contents: getSourceContentsObject(startAddress, targetRanges[0].start)
                });
                return newTargetRanges;
            }

            // sourceRange does fit in the targetRange
            targetRanges.forEach(function (range) {
                var startCol        = range.start[0],
                    startRow        = range.start[1],
                    endCol          = range.end[0],
                    endRow          = range.end[1],
                    colSize         = sourceRange.cols(),
                    rowSize         = sourceRange.rows();

                var iRow = 0,
                    reachedEndRow = (startRow + iRow * rowSize) >= endRow;
                do {
                    var iCol = 0,
                        reachedEndCol = (startCol + iCol * colSize) >= endCol;
                    do {
                        var newStartCol = range.start[0] + iCol * colSize,
                            newStartRow = range.start[1] + iRow * rowSize,
                            newStart    = new Address(newStartCol, newStartRow),
                            newRange    = Range.createFromAddressAndSize(newStart, colSize, rowSize);

                        newTargetRanges.push({
                            address: newRange.start,
                            contents: getSourceContentsObject(startAddress, newRange.start)
                        });

                        iCol++;
                        reachedEndCol = (startCol + iCol * colSize) > endCol;
                    } while (!reachedEndCol);

                    iRow++;
                    reachedEndRow = (startRow + iRow * rowSize) > endRow;
                } while (!reachedEndRow);
            });

            return newTargetRanges;
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the cell formulas in this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} updateDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on its 'type' property. See
         *  method TokenArray.resolveOperation() for more details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateFormulaOperations = function (generator, updateDesc) {

            // nothing to do without any cells
            if (this.isUsedRangeEmpty()) { return this.createResolvedPromise(null); }

            // whether this sheet will be manipulated
            var ownSheet = sheetModel.getIndex() === updateDesc.sheet;
            // additional processing if cells will be moved in a sheet
            var moveCells = updateDesc.type === UpdateMode.MOVE_CELLS;
            // the cell content objects used to generate the operations for all formula cells
            var contentsMap = new ValueMap();

            // transforms the passed cell address, if the change descriptor will move cells in the own sheet
            function transformAddress(address) {
                return (ownSheet && moveCells) ? MoveDescriptor.transformAddress(address, updateDesc.moveDescs) : address;
            }

            // transforms the passed range address, if the change descriptor will move cells in the own sheet
            function transformRange(range) {
                return (ownSheet && moveCells) ? MoveDescriptor.transformRange(range, updateDesc.moveDescs) : range;
            }

            // inserts the passed address and cell contents into the contents map
            function insertCellContents(address, contents) {
                contentsMap.insert(address.key(), { address: address, contents: contents });
            }

            // generates the cell contents needed to update all shared formulas, before moving the cells
            function generateSharedFormulaContents() {

                // nothing to do for other operations than moving cells (formula expressions of shared
                // formulas will be updated in generateUpdateFormulaContents() instead, see below)
                if (!moveCells) { return self.createResolvedPromise(null); }

                // a helper token array used to generate relocated formula expressions for all shared cells
                var tokenArray = sheetModel.createCellTokenArray();
                // a set with all existing shared indexes needed to generate new unused indexes
                // (register all existing shared indexes, before generating any new shared index)
                var sharedIndexSet = IndexSet.pluck(sharedModelMap, 'index');

                // process all shared formula in this cell collection
                return self.iterateSliced(sharedModelMap, function (sharedModel) {

                    // fail-safety: skip invalid shared formulas
                    if (sharedModel.addressSet.empty() || !sharedModel.refAddress) { return; }
                    // no need to process unaffected shared formulas
                    if (!ownSheet && !sharedModel.tokenArray.hasSheet3dTokens()) { return; }

                    // In the following, the different steps will be illustrated with a specific example:
                    // A shared formula may be located in the cells B2:B5 (anchor cell is B2), and the
                    // formula expression in the anchor cell is =SUM($C$1:$C3) with relative end row
                    // reference '3'. The formula expressions in each cell will look as following:
                    //  B2: =SUM($C$1:$C3)
                    //  B3: =SUM($C$1:$C4)
                    //  B4: =SUM($C$1:$C5)
                    //  B5: =SUM($C$1:$C6)
                    //
                    // Now, a new row will be inserted between row 4 and row 5. This means that cell B5 will
                    // be moved one row down to B6, and that the formula expressions will look as following:
                    //  B2: =SUM($C$1:$C3)
                    //  B3: =SUM($C$1:$C4)
                    //  B4: =SUM($C$1:$C6) (row reference '5' transformed to row reference '6')
                    //  B6: =SUM($C$1:$C7) (row reference '6' transformed to row reference '7')
                    //

                    // the current anchor address of the shared formula
                    var oldAnchor = sharedModel.anchorAddress;
                    // group the addresses by different transformed anchor formula expressions
                    var addressGroups = new ValueMap();

                    // Calculate the resulting transformed formula expression for all cells of the shared
                    // formula, and group the cells by their resulting anchor formula expression. In the
                    // example above, the transformed formula expressions of the (resulting) cells B2, B3,
                    // B4, and B6 will be relocated back to anchor address B2, which results in the following
                    // transformed anchor formulas:
                    //  B2: =SUM($C$1:$C3) (relative to B2)
                    //  B3: =SUM($C$1:$C3) (relative to B2)
                    //  B4: =SUM($C$1:$C4) (relative to B2)
                    //  B6: =SUM($C$1:$C3) (relative to B2, original cell address was B5)
                    //
                    // Therefore, two different groups of cells will be created: The first group contains the
                    // (original) cell addresses B2, B3, and B5; and the second group contains the address B4.
                    // From these two groups, two different shared formulas will be created, the first shared
                    // formula with anchor address B2, and the second with anchor address B4.
                    //
                    var promise = self.iterateSliced(sharedModel.addressSet, function (address) {

                        // find the new position of the formula cell (ignore formula cells that will be deleted)
                        var newAddress = transformAddress(address);
                        if (!newAddress) { return; }

                        // generate and parse the formula expression for the current cell
                        var oldFormula = sharedModel.getFormula('op', address);
                        tokenArray.parseFormula('op', oldFormula, { refAddress: address });

                        // clone the address to be able to add some helper properties
                        address = address.clone();
                        address.formula = oldFormula;

                        // apply the move descriptors to the formula, put the resulting transformed formula
                        // expression as property 'formula' into the address object
                        tokenArray.resolveOperationProperty('op', updateDesc, 'formula', address);

                        // create the map key for the address groups (a formula expression relative to 'oldAnchor')
                        var groupKey = address.formula;
                        if (newAddress.differs(oldAnchor)) {
                            tokenArray.parseFormula('op', address.formula, { refAddress: newAddress });
                            groupKey = tokenArray.getFormula('op', { refAddress: newAddress, targetAddress: oldAnchor, wrapReferences: true });
                        }

                        // collect the original (!) addresses in distinct address sets according to the
                        // relocated anchor formula expressions
                        addressGroups.getOrConstruct(groupKey, AddressSet).insert(address);

                    }, 'CellCollection.generateUpdateFormulaOperations.generateSharedFormulaContents');

                    // Create the cell content objects for all cells that need to be updated. Each address group will
                    // result in a shared formula, except for groups with only one address which will result in normal
                    // formula cells. In the example above, the group with the addresses B2, B3, and B5 will result in
                    // the shared formula at anchor address B2, and the anchor formula =SUM($C$1:$C3). The second group
                    // contains address B4 only, this will result in the normal formula cell with the formula expression
                    // =SUM($C$1:$C6).
                    //
                    promise = promise.then(function () {
                        var firstPass = true;
                        return self.iterateSliced(addressGroups, function (addressSet) {

                            // the address of the new anchor cell (original cell address, the cell is not transformed yet)
                            var newAnchor = addressSet.first();

                            // break shared formula, if only a single formula cell remains
                            if (addressSet.single()) {
                                insertCellContents(newAnchor, { f: newAnchor.formula, si: null, sr: null });
                                return;
                            }

                            // find an unused index for the shared formula (reuse the own shared index in first pass)
                            var sharedIndex = firstPass ? null : sharedIndexSet.reserve();
                            firstPass = false;

                            // create the content objects for all cells in the shared formula
                            return self.iterateSliced(addressSet, function (address) {

                                // the resulting cell contents object
                                var contents = {};

                                // write a new shared index unless the old shared index will be reused in the first pass
                                if (sharedIndex !== null) { contents.si = sharedIndex; }

                                // create the additional properties for the anchor cell (transform the bounding range
                                // of the address set, this will never fail because the address set contains only
                                // addresses that will not be deleted by the cell move operation).
                                if (address.equals(newAnchor)) {
                                    contents.f = newAnchor.formula;
                                    contents.sr = transformRange(addressSet.boundary());
                                }

                                // always store the new cell contents (even if this object is empty,
                                // this prevents repeated processing in the next step, see below)
                                insertCellContents(address, contents);

                            }, 'CellCollection.generateUpdateFormulaOperations.generateSharedFormulaContents');
                        }, 'CellCollection.generateUpdateFormulaOperations.generateSharedFormulaContents');
                    });

                    return promise;
                }, 'CellCollection.generateUpdateFormulaOperations.generateSharedFormulaContents');
            }

            // generates the cell contents needed to update all matrix formulas, before moving the cells
            function generateMatrixFormulaContents() {

                // process all matrix formulas in this cell collection
                return self.iterateSliced(matrixRangeSet, function (matrixRange) {

                    // the resulting cell contents object
                    var contents = {};

                    // get the transformed range address, nothing to do for deleted matrixes
                    var newRange = transformRange(matrixRange);
                    if (!newRange) { return; }
                    if (matrixRange.differs(newRange)) { contents.mr = newRange; }

                    // the formula expression is contained in the top-left cell of the matrix formula
                    var address = matrixRange.start;
                    // the cell model (should always exist, but check for fail-safety)
                    var cellModel = self.getCellModel(address);
                    if (!cellModel || !cellModel.t) { return; }

                    // calculate the new formula expression
                    cellModel.t.resolveOperationProperty('op', updateDesc, 'f', contents);

                    // always store the new cell contents (even if this object is empty,
                    // this prevents repeated processing in the next step, see below)
                    insertCellContents(address, contents);

                }, 'CellCollection.generateUpdateFormulaOperations.generateMatrixFormulaContents');
            }

            // generates the cell contents needed to update all (remaining) formula cells
            function generateUpdateFormulaContents() {

                var iterator = createAddressIterator(self.getUsedRange(), { type: 'anchor' });
                return self.iterateSliced(iterator, function (address, result) {

                    // fail-safety: token array must exist
                    if (!result.model.t) { return; }

                    // cells contained in the contents array have been processed already (shared and matrix formulas)
                    if (contentsMap.has(address.key())) { return; }

                    // do not update cells that will be deleted with a 'moveCells' operation
                    if (moveCells && !transformAddress(address)) { return; }

                    // store the refreshed formula expression
                    result.model.t.resolveOperation('op', updateDesc, function (newFormula) {
                        insertCellContents(address, { f: newFormula });
                    });

                }, 'CellCollection.generateUpdateFormulaOperations.generateUpdateFormulaContents');
            }

            // process all asynchronous update steps one after the other
            return SheetUtils.takeAsyncTime('CellCollection.generateUpdateFormulaOperations()', function () {
                return Utils.invokeChainedAsync(
                    // generate cell contents for shared formulas
                    generateSharedFormulaContents,
                    // generate cell contents for matrix formulas
                    generateMatrixFormulaContents,
                    // visit all anchor formula cells (cells with actual formula expression)
                    generateUpdateFormulaContents,
                    // finalize the cell operation
                    function () { return self.generateContentsArrayOperations(generator, contentsMap.values(), { skipShared: true }); }
                );
            });
        };

        /**
         * Generates the operations to convert all shared formula cells in the
         * specified range to regular formula cells.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Range} range
         *  The address of the cell range where all shared formula cells will
         *  be converted to regular formula cells.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateBreakSharedFormulaOperations = function (generator, range) {

            // collect the addresses of all shared formulas in the passed range
            var sharedAddresses = sharedModelMap.reduce(new AddressArray(), function (addresses, sharedModel) {
                return addresses.append(sharedModel.addressSet.findAddresses(range));
            });

            // visit all cells attached to a shared formula from top-left to bottom-right
            sharedAddresses.sort();

            // an operations builder for all affected shared formulas
            var operationsBuilder = new CellOperationsBuilder(sheetModel, generator);

            // visit all cells attached to a shared formula from top-left to bottom-right
            var promise = this.iterateArraySliced(sharedAddresses, function (address) {
                var cellModel = self.getCellModel(address);
                sharedModelMap.with(cellModel.si, function (sharedModel) {
                    if (cellModel.isSharedAnchor()) {
                        operationsBuilder.createCells(address, { si: null, sr: null }, 1);
                    } else {
                        operationsBuilder.createCells(address, { f: sharedModel.getFormula('op', address) }, 1);
                    }
                });
            }, 'CellCollection.generateBreakSharedFormulaOperations');

            // finalize the cell operations, return the addresses of the changed range
            return promise.then(function () {
                return operationsBuilder.finalizeOperations();
            });
        };

        /**
         * Generates the cell operations, and the undo operations, to change
         * the specified contents for a single cell, or multiple cells in the
         * sheet, and optionally the appropriate hyperlink operations.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Address} address
         *  The address of the top-left cell to be modified.
         *
         * @param {Object|Array<Object>} contents
         *  An object with the new properties to be applied at a single cell,
         *  or an array of row descriptors with cell property objects to change
         *  multiple cells at once. The row descriptors are objects with the
         *  following properties:
         *  - {Array<Object>} [rowData.c]
         *      An array of cell contents objects (see below for details). Can
         *      be omitted to skip one row, or multiple rows.
         *  - {Number} [rowData.r=1]
         *      The row repetition count. If set to a value greater than one,
         *      multiple consecutive rows will be changed simultaneously with
         *      the row descriptor.
         *  A cell content object (either passed directly to this method, or as
         *  element of the "c" array property of a row descriptor) supports the
         *  following properties:
         *  - {Number|String|Boolean|ErrorCode|Null} [contents.v]
         *      The new value of the cell. The value null represents a blank
         *      cell. If omitted, the current cell value will not be changed.
         *  - {String|Null} [contents.f]
         *      The new formula expression for the cell (in the native formula
         *      grammar of the file format). If set to null, the current
         *      formula will be removed from the cell. If omitted, the current
         *      formula will not be changed. If the "v" property has been set
         *      to null (clear the cell value), this property MUST be set to
         *      null too in order to delete the formula (formulas must have a
         *      non-null result value).
         *  - {String} [contents.s]
         *      The name of a new auto-style to be applied at the cell. If this
         *      property and the properties "a", "format", and "table" have
         *      been omitted, the current auto-style of the cell will not be
         *      changed.
         *  - {Object} [contents.a]
         *      An attribute set with explicit formatting attributes to be
         *      applied at the cell. If this property, and the properties "s",
         *      "format", and "table" have been omitted, the current auto-style
         *      of the cell will not be changed.
         *  - {Number|String} [contents.format]
         *      The identifier of a number format, or an explicit format code,
         *      to be inserted into the auto-style of the cell. If this
         *      property, and the properties "s", "a", and "table" have been
         *      omitted, the current auto-style of the cell will not be
         *      changed.
         *  - {Object} [contents.table]
         *      An attribute set resolved form a table style sheet, to be
         *      merged "under" the cell auto-style, explicit cell attributes,
         *      and cell style. If this property, and the properties "s", "a",
         *      and "format" have been omitted, the current auto-style of the
         *      cell will not be changed.
         *  - {Boolean} [contents.u=false]
         *      If set to true, the cell will be deleted explicitly (removed
         *      completely from the cell collection). The previous properties
         *      for changing the cell contents or formatting MUST NOT be used
         *      in this case.
         *  - {Number} [contents.r=1]
         *      The repetition count causing to change the specified number of
         *      consecutive cells in the same row.
         *  - {String} [contents.url]
         *      If set to a non-empty string, a hyperlink for the cell will be
         *      created. If set to an empty string, an existing URL will be
         *      removed. If multiple content objects in a two-dimensional array
         *      contain URLs, they will be merged to coherent cell ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated.
         */
        this.generateCellContentOperations = function (generator, address, contents) {

            // convert a single object to a contents matrix
            var matrix = _.isArray(contents) ? contents : [{ c: [contents] }];

            // let the private method do all the work
            return generateCellContentOperations(generator, address, matrix);
        };

        /**
         * Generates the cell operations, and the undo operations, to change
         * the specified contents for a list of single cells.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<Object>} contentsArray
         *  A list with the descriptors for all cells to be changed, in no
         *  specific order. Each array element MUST be an object with the
         *  following properties:
         *  - {Address} address
         *      The address of the cell to be modified. The cell addresses in
         *      the array elements MUST be unique.
         *  - {Object} contents
         *      The cell contents descriptor for the cell. See method
         *      CellCollection.generateCellContentOperations() for details.
         *
         * @param {Object} [options]
         *  Optional parameters passed to the constructor of the class
         *  CellOperationsBuilder, and the following options:
         *  - {Boolean} [options.sorted=false]
         *      If set to true, the contents array passed to this method is
         *      guaranteed to be sorted by cell address. It will not be sorted
         *      again internally (performance optimization).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated.
         */
        this.generateContentsArrayOperations = function (generator, contentsArray, options) {

            // create a cell operations builder for the formula cells
            var operationsBuilder = new CellOperationsBuilder(sheetModel, generator, options);

            // sort the contents descriptors by cell address
            if (!Utils.getBooleanOption(options, 'sorted', false)) {
                contentsArray.sort(function (entry1, entry2) { return entry1.address.compareTo(entry2.address); });
            }

            // insert all cells into the operations builder
            var promise = this.iterateArraySliced(contentsArray, function (entry) {
                operationsBuilder.createCells(entry.address, entry.contents, 1);
            }, 'CellCollection.generateContentsArrayOperations');

            // finalize the cell operations, return the addresses of the changed range
            return promise.then(function () {
                return operationsBuilder.finalizeOperations();
            });
        };

        /**
         * Generates the cell operations, and the undo operations, to fill the
         * passed cell ranges in the sheet with some individual contents.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *  Each range object MUST contain an additional property 'fillData'
         *  with cell properties to be applied at all cells in the respective
         *  range. See method CellCollection.generateFillOperations() for more
         *  details about the properties. Overlapping ranges MUST NOT contain
         *  the same cell properties with different values (e.g.: set value 1
         *  or value 2 to the same cells in overlapping ranges), otherwise the
         *  result will be undefined. However, incomplete cell property objects
         *  of overlapping ranges will be merged together for the overlapping
         *  parts of the ranges (e.g.: setting the auto-style 'a1' to the range
         *  A1:B2, and the value 2 to the range B2:C3 will result in setting
         *  both the auto-style and the value to cell B2).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'cols:overflow': Too many entire columns in the cell ranges.
         *  - 'rows:overflow': Too many entire rows in the cell ranges.
         */
        this.generateRangeOperations = SheetUtils.profileAsyncMethod('CellCollection.generateRangeOperations()', function (generator, ranges) {
            SheetUtils.log('ranges=' + ranges);

            // divide the original ranges into column ranges, row ranges, and regular cell ranges
            var rangeGroups = getRangeGroups(ranges);
            // add the 'rangeMode' property to the cell ranges
            var taggedRanges = getTaggedRanges(ranges);

            // transport the changed ranges through the promise chain
            var promise = this.createResolvedPromise(new RangeArray());

            // let the row collection generate the operations for entire formatted rows
            promise = generateChainedRowIntervalOperations(promise, rangeGroups, function (rowIntervals) {
                return rowCollection.generateIntervalOperations(generator, rowIntervals);
            });

            // let the column collection generate the operations for entire formatted columns
            promise = generateChainedColIntervalOperations(promise, rangeGroups, function (colIntervals) {
                return colCollection.generateIntervalOperations(generator, colIntervals);
            });

            // generate the cell operations for all ranges, merge cell content objects of overlapping ranges
            return generateChainedCellRangeOperations(promise, function () {
                return generateCellRangeOperations(generator, taggedRanges, function (fillDataArray) {
                    return fillDataArray.reduce(_.extend, {});
                });
            });
        });

        /**
         * Generates the cell operations, and the undo operations, to fill the
         * same cell contents into entire cell ranges in the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} contents
         *  The new properties to be applied at all cells. See public method
         *  CellCollection.generateCellContentOperations() for details.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'cols:overflow': Too many entire columns in the cell ranges.
         *  - 'rows:overflow': Too many entire rows in the cell ranges.
         */
        this.generateFillOperations = SheetUtils.profileAsyncMethod('CellCollection.generateFillRangeOperations()', function (generator, ranges, contents) {
            SheetUtils.log('ranges=' + ranges + ' contents=', contents);

            // do not modify the cell contents object passed from caller
            contents = _.clone(contents);
            // add an arbitrary cache key to the contents object that causes internal auto-style caching
            // (all ranges will be formatted with the same new formatting attributes)
            contents.cacheKey = _.uniqueId('key');

            // divide the original ranges into column ranges, row ranges, and regular cell ranges
            var rangeGroups = getRangeGroups(ranges);
            // add the 'rangeMode' property to the cell ranges
            var taggedRanges = getTaggedRanges(ranges);
            // the change flags specifying how the cell will be potentially changed
            var changeFlags = getChangeFlags(contents);

            // if ranges will be formatted, the cell ranges not covering entire columns or rows must not exceed a specific limit
            if (changeFlags.style && rangeGroups.cellRanges && (rangeGroups.cellRanges.merge().cells() > SheetUtils.MAX_FILL_CELL_COUNT)) {
                return SheetUtils.makeRejected('cells:overflow');
            }

            // when filling with values/formulas, the entire ranges must not exceed a specific limit (but not when clearing or undefining ranges)
            if ((changeFlags.value || changeFlags.formula) && (taggedRanges.merge().cells() > SheetUtils.MAX_FILL_CELL_COUNT)) {
                return SheetUtils.makeRejected('cells:overflow');
            }

            // start a promise chain
            var promise = this.createResolvedPromise();

            // create or remove hyperlinks for the cell ranges
            if (typeof contents.url === 'string') {
                var url = contents.url;
                delete contents.url;
                promise = promise.then(function () {
                    return sheetModel.getHyperlinkCollection().generateHyperlinkOperations(generator, ranges, url);
                });
            }

            // transport the changed ranges through the promise chain
            promise = promise.then(function () { return new RangeArray(); });

            // let the row collection generate the operations for entire formatted rows
            promise = generateChainedRowIntervalOperations(promise, rangeGroups, function (rowIntervals) {
                return rowCollection.generateFillOperations(generator, rowIntervals, contents);
            });

            // let the column collection generate the operations for entire formatted columns
            promise = generateChainedColIntervalOperations(promise, rangeGroups, function (colIntervals) {
                return colCollection.generateFillOperations(generator, colIntervals, contents);
            });

            // generate the cell operations for all ranges, using the same cell contents object for all ranges
            // (the same contents object can be recycled, generateCellRangeOperations() creates a clone internally)
            return generateChainedCellRangeOperations(promise, function () {
                return generateCellRangeOperations(generator, taggedRanges, _.constant(contents));
            });
        });

        /**
         * Generates the cell operations, and the undo operations, to fill the
         * cells next to the specified range with automatically generated
         * values, formulas, and formatting.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Range} range
         *  The address of the source cell range with values, formulas, and
         *  formatting to be expanded into one direction.
         *
         * @param {Direction} direction
         *  The direction in which the specified cell range will be expanded.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the cell range into the
         *  specified direction (positive values), or to shrink the cell range
         *  (negative values).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.copy=false]
         *      If set to true, source values will be copied, instead of
         *      incrementing the existing values automatically.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'cols:overflow': Too many entire columns in the cell ranges.
         *  - 'rows:overflow': Too many entire rows in the cell ranges.
         */
        this.generateAutoFillOperations = SheetUtils.profileAsyncMethod('CellCollection.generateAutoFillOperations()', function (generator, range, direction, count, options) {
            SheetUtils.log('source=' + range + ' dir=' + direction + ' count=' + count);

            // the target range to be filled from the source range
            var targetRange = SheetUtils.getAdjacentRange(range, direction, count);
            // a private undo generator for restoring deleted contents at the end
            var undoGenerator = sheetModel.createOperationGenerator({ applyImmediately: true });

            // first step for undo: delete the cells generated by auto-fill
            var hasCells = !this.areRangesBlank(range);
            if (hasCells) {
                var clearContents = [{ c: [{ u: true, r: targetRange.cols() }], r: targetRange.rows() }];
                generator.generateCellOperation(Operations.CHANGE_CELLS, targetRange.start, { contents: clearContents }, { undo: true });
            }

            // clear all target cells (must be done first, before e.g. trying to merge cell ranges)
            var promise = this.generateFillOperations(undoGenerator, targetRange, { u: true }).done(function () {
                generator.appendOperations(undoGenerator);
            });

            // copy formatting of entire columns/rows
            var columns = !SheetUtils.isVerticalDir(direction);
            if (docModel.isFullRange(range, columns)) {
                promise = promise.then(function () {
                    var collection = columns ? colCollection : rowCollection;
                    return collection.generateAutoFillOperations(generator, range.interval(columns), direction, count);
                });
            }

            // copy all merged ranges contained in the source range
            promise = promise.then(function () {
                return sheetModel.getMergeCollection().generateAutoFillOperations(generator, range, direction, count);
            });

            // copy all hyperlinks contained in the source range
            promise = promise.then(function () {
                return sheetModel.getHyperlinkCollection().generateAutoFillOperations(generator, range, direction, count);
            });

            // copy cell contents and styles
            promise = promise.then(function () {
                if (hasCells) {
                    return generateAutoFillOperations(generator, range, direction, count, options);
                }
            });

            // append the undo operations to restore the cell contents at the end
            promise.done(function () {
                generator.appendOperations(undoGenerator, { undo: true });
            });

            return promise;
        });

        /**
         * Generates the cell operations, and the undo operations, to fill the
         * cell ranges in the sheet with outer and/or inner borders.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Object} borderAttributes
         *  The border attributes to be applied at all cells. May contain
         *  regular border attributes, as supported by the document operations
         *  ('borderTop', 'borderBottom', 'borderLeft', 'borderRight') which
         *  will be applied at the outer boundaries of the passed ranges; and
         *  the pseudo attributes 'borderInsideHor' and 'borderInsideVert',
         *  which will be applied to the inner cells of the cell ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'cols:overflow': Too many entire columns in the cell ranges.
         *  - 'rows:overflow': Too many entire rows in the cell ranges.
         */
        this.generateBorderOperations = (function () {

            // Returns the parts of the passed ranges that will receive a new border attribute.
            // Parameters 'columns'/'leading' specify the position of the cell border to be modified.
            // The resulting ranges will contain an additional property 'fillData' with border settings
            // as expected by the helper method SheetOperationGenerator.createBorderContents().
            function getChangedBorderRanges(ranges, borderAttributes, columns, leading) {

                // the resulting ranges
                var resultRanges = new RangeArray();

                // adds the passed style settings to the passed ranges, and appends them to the resulting range array
                function appendRanges(newRanges, fillData) {
                    return resultRanges.append(Utils.addProperty(newRanges, 'fillData', fillData));
                }

                // the keys of the borders to be processed
                var outerKey = SheetUtils.getOuterBorderKey(columns, leading);
                var innerKey = SheetUtils.getInnerBorderKey(columns);

                // get the border attributes, nothing to do without outer nor inner border attribute
                var outerBorder = borderAttributes[SheetUtils.getBorderName(outerKey)];
                var innerBorder = borderAttributes[SheetUtils.getBorderName(innerKey)];
                if (!outerBorder && !innerBorder) { return null; }

                // add the adjacent ranges whose opposite borders need to be deleted while setting a range border
                // (if the outer border will not be changed, the borders of the adjacent cells will not be modified neither)
                if (outerBorder) {
                    // the adjacent ranges at the specified range border whose existing borders will be deleted
                    var adjacentRanges = getAdjacentRanges(ranges, columns, leading);
                    // opposite borders will be deleted, e.g. the right border left of the original range (but not if
                    // they are equal to the outer border set at the cell range, to reduce the amount of changed cells)
                    var oppositeKey = SheetUtils.getOuterBorderKey(columns, !leading);
                    appendRanges(adjacentRanges, { key: oppositeKey, keepBorder: outerBorder });
                }

                // shortcut (outer and inner borders are equal): return entire ranges instead of splitting into outer/inner
                if (outerBorder && innerBorder && Border.isEqual(outerBorder, innerBorder)) {
                    return appendRanges(ranges.clone(true), { key: outerKey, border: outerBorder, cacheType: 'outer' });
                }

                // divide the ranges into ranges covering entire columns/rows and other ranges
                var rangeGroups = ranges.group(function (range) { return docModel.isFullRange(range, !columns) ? 'fullRanges' : 'cellRanges'; });

                // do not set outer borders at entire column/row ranges, use inner border for all cells
                if (innerBorder && rangeGroups.fullRanges) {
                    appendRanges(rangeGroups.fullRanges.clone(true), { key: outerKey, border: innerBorder, cacheType: 'outer' });
                }

                // outer border will be set to the outer cells only
                if (outerBorder && rangeGroups.cellRanges) {
                    var subRangeMethod = columns ? (leading ? 'leadingCol' : 'trailingCol') : (leading ? 'headerRow' : 'footerRow');
                    appendRanges(RangeArray.invoke(rangeGroups.cellRanges, subRangeMethod), { key: outerKey, border: outerBorder, cacheType: 'outer' });
                }

                // inner border will be set to the remaining cells
                if (innerBorder && rangeGroups.cellRanges) {
                    var offset = leading ? 1 : 0;
                    appendRanges(RangeArray.map(rangeGroups.cellRanges, function (range) {
                        return range.singleLine(columns) ? null : range.lineRange(columns, offset, range.size(columns) - 1);
                    }), { key: outerKey, border: innerBorder, cacheType: 'inner' });
                }

                return resultRanges;
            }

            // the actual implementation of the public method returned from local scope
            function generateBorderOperations(generator, ranges, borderAttributes) {
                SheetUtils.log('ranges=' + ranges + ' attributes=', borderAttributes);

                // ensure an array, remove duplicates (but do not merge the ranges)
                ranges = RangeArray.get(ranges).unify();
                // divide the original ranges into column ranges, row ranges, and regular cell ranges
                var rangeGroups = getRangeGroups(ranges);

                // prepare the range arrays with the parts of the passed ranges that will get a new single border
                var rangesT = getChangedBorderRanges(ranges, borderAttributes, false, true);
                var rangesB = getChangedBorderRanges(ranges, borderAttributes, false, false);
                var rangesL = getChangedBorderRanges(ranges, borderAttributes, true,  true);
                var rangesR = getChangedBorderRanges(ranges, borderAttributes, true,  false);
                // add the 'rangeMode' property to the cell ranges
                var taggedRanges = getTaggedRanges(new RangeArray(rangesT, rangesB, rangesL, rangesR));

                // root cache key for using the same auto-style cache keys across all generators
                var cacheKey = _.uniqueId('key');
                // transport the changed ranges through the promise chain
                var promise = this.createResolvedPromise(new RangeArray());

                // let the row collection generate the operations for entire formatted rows
                promise = generateChainedRowIntervalOperations(promise, rangeGroups, function (rowIntervals) {
                    return rowCollection.generateBorderOperations(generator, rowIntervals, borderAttributes, cacheKey);
                });

                // let the column collection generate the operations for entire formatted columns
                promise = generateChainedColIntervalOperations(promise, rangeGroups, function (colIntervals) {
                    return colCollection.generateBorderOperations(generator, colIntervals, borderAttributes, cacheKey);
                });

                // generate the cell operations for all ranges, use the border attributes resolved above
                return generateChainedCellRangeOperations(promise, function () {
                    return generateCellRangeOperations(generator, taggedRanges, function (fillDataArray) {
                        return generator.createBorderContents(fillDataArray, cacheKey);
                    });
                });
            }

            return SheetUtils.profileAsyncMethod('CellCollection.generateBorderOperations()', generateBorderOperations);
        }());

        /**
         * Generates the cell operations, and the undo operations, to change
         * the formatting of existing border lines of the passed cell ranges in
         * the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Border} border
         *  A border attribute value, which may be incomplete. All properties
         *  contained in this object (color, line style, line width) will be
         *  set for all visible borders in the cell range. Omitted border
         *  properties will not be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         *  - 'cells:overflow': The cell ranges contain too many filled cells.
         *  - 'cols:overflow': Too many entire columns in the cell ranges.
         *  - 'rows:overflow': Too many entire rows in the cell ranges.
         */
        this.generateVisibleBorderOperations = (function () {

            // Returns the adjacent ranges of the passed ranges whose border will be changed additionally to the
            // covered ranges. The resulting ranges will contain an additional property 'fillData' with border
            // settings as expected by the helper method SheetOperationGenerator.createVisibleBorderContents().
            function getAdjacentBorderRanges(ranges, columns, leading) {
                var adjacentRanges = getAdjacentRanges(ranges, columns, leading);
                return Utils.addProperty(adjacentRanges, 'fillData', { keys: SheetUtils.getOuterBorderKey(columns, !leading) });
            }

            // the actual implementation of the public method returned from local scope
            function generateVisibleBorderOperations(generator, ranges, border) {
                SheetUtils.log('ranges=' + ranges + ' border=' + JSON.stringify(border));

                // ensure an array, remove duplicates (but do not merge the ranges)
                ranges = RangeArray.get(ranges).unify();
                // divide the original ranges into column ranges, row ranges, and regular cell ranges
                var rangeGroups = getRangeGroups(ranges);

                // all borders need to be changed inside the passed ranges
                var fillRanges = Utils.addProperty(ranges.clone(true), 'fillData', { keys: 'tblrdu' });
                // the adjacent cells outside the ranges (the adjoining borders will be changed too)
                var rangesT = getAdjacentBorderRanges(ranges, false, true);
                var rangesB = getAdjacentBorderRanges(ranges, false, false);
                var rangesL = getAdjacentBorderRanges(ranges, true,  true);
                var rangesR = getAdjacentBorderRanges(ranges, true,  false);
                // add the 'rangeMode' property to the cell ranges
                var taggedRanges = getTaggedRanges(new RangeArray(fillRanges, rangesT, rangesB, rangesL, rangesR));

                // root cache key for using the same auto-style cache keys across all generators
                var cacheKey = _.uniqueId('key');
                // transport the changed ranges through the promise chain
                var promise = this.createResolvedPromise(new RangeArray());

                // let the row collection generate the operations for entire formatted rows
                promise = generateChainedRowIntervalOperations(promise, rangeGroups, function (rowIntervals) {
                    return rowCollection.generateVisibleBorderOperations(generator, rowIntervals, border, cacheKey);
                });

                // let the column collection generate the operations for entire formatted columns
                promise = generateChainedColIntervalOperations(promise, rangeGroups, function (colIntervals) {
                    return colCollection.generateVisibleBorderOperations(generator, colIntervals, border, cacheKey);
                });

                // generate the cell operations for all ranges, use the border attributes resolved above
                return generateChainedCellRangeOperations(promise, function () {
                    return generateCellRangeOperations(generator, taggedRanges, function (fillDataArray) {
                        return generator.createVisibleBorderContents(fillDataArray, border, cacheKey);
                    });
                });
            }

            return SheetUtils.profileAsyncMethod('CellCollection.generateVisibleBorderOperations()', generateVisibleBorderOperations);
        }());

        /**
         * Generates the cell operations, and the undo operations, to merge the
         * specified cell ranges. The first value cell of each cell range will
         * be moved to the top-left corner of the cell range, and all other
         * cells covered by the ranges will be deleted. The creation of column
         * or row formatting for merged ranges covering entire columns or rows
         * will be handled automatically.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *  Multiple ranges MUST NOT overlap each other!
         *
         * @param {MergeMode} mergeMode
         *  The merge mode for the specified ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated.
         */
        this.generateMergeCellsOperations = (function () {

            function getRangeBorders(range) {

                // the resulting range borders
                var resultBorderMap = {};

                function updateBorder(styleBorderMap, borderKey) {
                    var styleBorder = styleBorderMap[borderKey];
                    if (!(borderKey in resultBorderMap)) {
                        resultBorderMap[borderKey] = styleBorder;
                    } else if ((resultBorderMap[borderKey] !== null) && !Border.isEqual(resultBorderMap[borderKey], styleBorder)) {
                        resultBorderMap[borderKey] = null;
                    }
                }

                // collect all outer borders and diagonal borders of the range
                var iterator = self.createStyleIterator(range, { covered: true });
                Iterator.forEach(iterator, function (styleRange, result) {

                    // the visible border attributes of the auto-style, mapped by border key
                    var styleBorderMap = autoStyles.getBorderMap(result.style);

                    // update all border attributes according to the position of the current style range
                    if (range.start[0] === styleRange.start[0]) { updateBorder(styleBorderMap, 'l'); }
                    if (range.start[1] === styleRange.start[1]) { updateBorder(styleBorderMap, 't'); }
                    if (range.end[0] === styleRange.end[0]) { updateBorder(styleBorderMap, 'r'); }
                    if (range.end[1] === styleRange.end[1]) { updateBorder(styleBorderMap, 'b'); }
                    updateBorder(styleBorderMap, 'd');
                    updateBorder(styleBorderMap, 'u');
                });

                return resultBorderMap;
            }

            function generateMergeCellsOperations(generator, ranges, mergeMode) {
                SheetUtils.log('ranges=' + ranges + ' mode=' + mergeMode.toString());

                // nothing to do when unmerging cells
                if (mergeMode === MergeMode.UNMERGE) {
                    return this.createResolvedPromise(new RangeArray());
                }

                // whether to merge single columns in the passed ranges
                var vertical = mergeMode === MergeMode.VERTICAL;
                // whether to merge single rows in the passed ranges
                var horizontal = mergeMode === MergeMode.HORIZONTAL;
                // the ranges to be processed, with additional information in 'fillData' property
                var fillRanges = new RangeArray();
                // the border ranges to be processed, with additional information in 'borderAttrs' property
                var borderRanges = new RangeArray();
                // a private generator for restoring all range borders
                var borderGenerator = sheetModel.createOperationGenerator({ applyImmediately: true });

                // horizontal/vertical merging does not have any effect for single-column/row ranges
                ranges = RangeArray.reject(ranges, function (range) {
                    return horizontal ? range.singleCol() : vertical ? range.singleRow() : range.single();
                });

                // process each range for its own (ranges do not overlap)
                var promise = this.iterateArraySliced(ranges, function (range) {

                    // number of subranges to be processed according to merge type
                    var count = vertical ? range.cols() : horizontal ? range.rows() : 1;

                    // process all columns/rows separately, or the entire range, according to merge type
                    return this.repeatSliced(function (index) {

                        // the subrange to be merged (search for a value cell, same formatting)
                        var partRange = vertical ? range.colRange(index) : horizontal ? range.rowRange(index) : range.clone();
                        fillRanges.push(partRange);

                        // find the first cell model with a value in the current range
                        var result = createAddressIterator(partRange, { type: 'value' }).next();
                        var cellModel = result.done ? null : result.model;

                        // fill the entire range with the same formatting (either from first value cell, or from origin)
                        var styleId = cellModel ? cellModel.s : this.getStyleId(partRange.start);
                        partRange.fillData = { s: styleId };

                        // collect the current outer borders and diagonal borders of the merged range
                        var rangeBorderMap = getRangeBorders(partRange);

                        // delete the inner borders in a multi-column or multi-row range
                        var styleBorderMap = autoStyles.getBorderMap(styleId);
                        var styleBorderAttrs = {};
                        if (!partRange.singleCol() && (styleBorderMap.l || styleBorderMap.r)) {
                            styleBorderAttrs.borderLeft = styleBorderAttrs.borderRight = Border.NONE;
                        }
                        if (!partRange.singleRow() && (styleBorderMap.t || styleBorderMap.b)) {
                            styleBorderAttrs.borderTop = styleBorderAttrs.borderBottom = Border.NONE;
                        }
                        ['d', 'u'].forEach(function (borderKey) {
                            if ((rangeBorderMap[borderKey] === null) && (styleBorderMap[borderKey])) {
                                styleBorderAttrs[SheetUtils.getBorderName(borderKey)] = Border.NONE;
                            }
                        });
                        if (!_.isEmpty(styleBorderAttrs)) {
                            partRange.fillData.a = { cell: styleBorderAttrs };
                        }

                        // process the outer borders of the merged range separately
                        var outerBorderAttrs = {};
                        _.each(rangeBorderMap, function (rangeBorder, borderKey) {
                            if ((borderKey === 'd') || (borderKey === 'u')) { return; }
                            var borderName = SheetUtils.getBorderName(borderKey);
                            if (rangeBorder === null) {
                                // hide outer border line, if it is ambiguous
                                outerBorderAttrs[borderName] = Border.NONE;
                            } else if (Border.isVisibleBorder(rangeBorder)) {
                                // set a visible border line as outer border
                                outerBorderAttrs[borderName] = rangeBorder;
                            } else if (styleBorderMap[borderKey]) {
                                // hide a visible border of the auto-style
                                outerBorderAttrs[borderName] = Border.NONE;
                            }
                        });
                        if (!_.isEmpty(outerBorderAttrs)) {
                            var borderRange = partRange.clone();
                            borderRange.borderAttrs = outerBorderAttrs;
                            borderRanges.push(borderRange);
                        }

                        // move contents of the first value cell to the origin of the merged range
                        if (cellModel) {

                            // move value and formula (but not the formatting, see below) to top-left corner
                            if (!partRange.startsAt(cellModel.a)) {
                                var startRange = new Range(partRange.start);
                                startRange.rangeMode = 'skip';
                                startRange.fillData = { v: cellModel.v, f: cellModel.f };
                                fillRanges.push(startRange);
                            }

                            // delete values and formulas from all other cells of the range
                            new RangeArray(partRange).difference(new Range(partRange.start)).forEach(function (remainRange) {
                                remainRange.rangeMode = 'skip';
                                remainRange.fillData = { v: null, f: null };
                                fillRanges.push(remainRange);
                            });
                        }
                    }, 'CellCollection.generateMergeCellsOperations', { cycles: count });
                }, 'CellCollection.generateMergeCellsOperations');

                // generate the resulting fill operations
                promise = promise.then(function () {
                    return self.generateRangeOperations(generator, fillRanges);
                });

                // homogeneous border for each single edge
                promise = promise.then(function () {
                    return self.iterateArraySliced(borderRanges, function (borderRange) {
                        return self.generateBorderOperations(borderGenerator, borderRange, borderRange.borderAttrs);
                    }, 'CellCollection.generateMergeCellsOperations');
                });

                return promise.done(function () {
                    generator.appendOperations(borderGenerator);
                    generator.prependOperations(borderGenerator, { undo: true });
                });
            }

            return SheetUtils.profileAsyncMethod('CellCollection.generateMergeCellsOperations()', generateMergeCellsOperations);
        }());

        /**
         * Generates the operations, and the undo operations, to insert blank
         * cells at the specified cell ranges, or to delete the specified cell
         * ranges, by moving the remaining existing cells to the left, right,
         * top, or bottom.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Direction} direction
         *  Where to move the existing cells to.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the resulting merged index
         *  intervals) when all operations have been generated, or that will be
         *  rejected with an object with 'cause' property set to one of the
         *  following error codes:
         *  - 'cells:pushoff': Inserting new cells is not possible because this
         *      would push existing cells off the end of the sheet.
         *  - 'formula:matrix:insert': Inserting new cells is not possible
         *      because this would split a matrix formula.
         *  - 'formula:matrix:delete': Deleting cells is not possible because
         *      this would shorten a matrix formula.
         *  - 'table:move': Deleting rows would invalidate a table range.
         */
        this.generateMoveCellsOperations = (function () {

            // generates the operations needed to insert auto-styles into the new inserted empty cells
            function generateInsertStyleOperations(generator, moveDescs, columns) {
                return self.iterateArraySliced(moveDescs, function (moveDesc) {
                    return self.iterateArraySliced(moveDesc.targetIntervals, function (blankInterval) {

                        // nothing to do when inserting at the beginning of the sheet
                        if (blankInterval.first === 0) { return; }

                        // Create a parallel iterator that visits all columns/rows with at least one defined cell before or after the insertion line.
                        // Example: When inserting columns C:D, these columns are now empty, and the old existing cells in column B need to be extended
                        // into the new blank columns, with special border processing of the (formerly adjacent) cells in column B and column E.
                        var prevLine = moveDesc.createLineRange(blankInterval.first - 1);
                        var nextLine = moveDesc.createLineRange(blankInterval.last + 1);
                        var iterator = createParallelAddressIterator([prevLine, nextLine], { type: 'defined' });

                        // collect the range addresses for inserting defined cells into the new blank ranges
                        var fillRanges = new RangeArray();
                        var promise = self.iterateSliced(iterator, function (addresses) {

                            // get the identifiers of the auto-styles before and after the new blank range
                            var styleId1 = self.getStyleId(addresses[0]);
                            var styleId2 = self.getStyleId(addresses[1]);

                            // calculate the new auto-style to be inserted into the blank cells
                            var newStyleId = generator.generateInsertedAutoStyle(styleId1, styleId2, columns);

                            // try to extend an existing fill range, otherwise create a new range
                            var lastFillRange = fillRanges.last();
                            if (lastFillRange && (lastFillRange.getEnd(!columns) + 1 === addresses[0].get(!columns)) && autoStyles.areEqualStyleIds(lastFillRange.fillData.s, newStyleId)) {
                                lastFillRange.end.move(1, !columns);
                            } else {
                                var fillRange = new Range(addresses[0].clone().move(1, columns), addresses[1].clone().move(-1, columns));
                                fillRange.fillData = { s: newStyleId };
                                fillRanges.push(fillRange);
                            }
                        }, 'CellCollection.generateInsertStyleOperations');

                        // create the fill operations, but ignore the undo operations
                        return promise.then(function () {
                            return generateCellRangeOperations(generator, fillRanges, function (fillDataArray) {
                                return fillDataArray[0];
                            }, { createUndo: false });
                        });
                    }, 'CellCollection.generateInsertStyleOperations');
                }, 'CellCollection.generateInsertStyleOperations');
            }

            // implementation helper that handles the operation order for preparation/postprocessing tasks;
            // used by interval generators (entire columns/rows), as well as cell range generators
            function implGenerateMoveOperations(generator, moveDescs, columns, insert, callback) {

                // the cell ranges containing the cells that need to be restored on undo
                var deleteRanges = RangeArray.map(moveDescs, function (moveDesc) {
                    return moveDesc.createRanges(moveDesc.deleteIntervals);
                });

                // check that the move operation does not push cells outside the sheet
                if (insert && self.findFirstCell(deleteRanges, { type: 'value', covered: true })) {
                    return SheetUtils.makeRejected('cells:pushoff');
                }

                // check that matrixes will not be split when moving the ranges
                if (matrixRangeSet.some(function (matrixRange) {
                    var newRange = MoveDescriptor.transformRange(matrixRange, moveDescs);
                    return newRange && !matrixRange.equalSize(newRange);
                })) {
                    return SheetUtils.makeRejected(insert ? 'formula:matrix:insert' : 'formula:matrix:delete');
                }

                // check whether table ranges can be moved
                var tableCollection = sheetModel.getTableCollection();
                if (!tableCollection.canMoveCells(moveDescs)) {
                    return SheetUtils.makeRejected('table:move');
                }

                // the formula update descriptor to be passed to the generator methods of the collections
                var updateDesc = { type: UpdateMode.MOVE_CELLS, sheet: sheetModel.getIndex(), moveDescs: moveDescs };

                // a private undo generator for restoring existing cells that will be deleted
                var restoreCellsGenerator = sheetModel.createOperationGenerator();
                // a private operations generator for additional operations from other sheet collections
                var beforeMoveGenerator = sheetModel.createOperationGenerator({ applyImmediately: true });
                // a private operations generator for additional operations from other sheet collections
                var afterMoveGenerator = sheetModel.createOperationGenerator({ applyImmediately: true });
                // a private operations generator for additional operations for drawing objects (anchor handling)
                var drawingGenerator = sheetModel.createOperationGenerator({ applyImmediately: true });

                // create undo operations to restore the existing cells in the deleted ranges
                // (must be done before generating and immediately applying the 'moveCells' operations)
                var promise = self.generateFillOperations(restoreCellsGenerator, deleteRanges, { u: true });

                // generate undo operations for implicitly deleted contents in the sheet (merged ranges, table ranges,
                // drawing objects) before moving the cells (which will destroy these contents)
                promise = promise.then(function () {
                    return sheetModel.generateBeforeMoveCellsOperations(beforeMoveGenerator, moveDescs);
                });

                // update the formula expressions for all sheets in the document before moving the cells
                // bug 58155: but not drawing objects (needs updated columns/rows to get correct anchor positions)
                promise = promise.then(function () {
                    return docModel.generateUpdateFormulaOperations(beforeMoveGenerator, updateDesc, { drawingMode: 'skip' });
                });

                // append the (already applied) document operations to the root operation generator
                promise = promise.then(function () {
                    generator.appendOperations(beforeMoveGenerator);
                });

                // invoke the callback function to generate the actual move operations (will use the root operation generator)
                promise = promise.then(callback);

                // insertion mode: create the operations to copy the cell auto-styles into the new blank ranges
                if (insert) {
                    promise = promise.then(function () {
                        return generateInsertStyleOperations(afterMoveGenerator, updateDesc.moveDescs, columns);
                    });
                }

                // bug 39869: create the missing header labels for table ranges, after inserting columns (only in the own sheet)
                if (columns && insert) {
                    promise = promise.then(function () {
                        return tableCollection.generateMissingHeaderLabelOperations(afterMoveGenerator);
                    });
                }

                // bug 58155: update drawing objects after move (needs updated columns/rows to get correct anchor positions)
                // undo operations must be applied after moving the cells too in order to assign them to correct cells (esp. ODF)
                promise = promise.then(function () {
                    return docModel.generateUpdateFormulaOperations(drawingGenerator, updateDesc, { drawingMode: 'only' });
                });

                // append the (already applied) document operations to the root operation generator
                promise = promise.then(function () {
                    generator.appendOperations(afterMoveGenerator).appendOperations(drawingGenerator);
                });

                // insert the undo operations in correct order into the generator
                promise = promise.then(function () {
                    // put the undo operations to be applied before the move in front of the root generator
                    generator.prependOperations(afterMoveGenerator, { undo: true });
                    // bug 58155: drawing undo operations must be applied after moving the cells too in order to assign
                    // them to correct cells (esp. ODF), but before restoring deleted drawings (beforeMoveGenerator)
                    generator.appendOperations(drawingGenerator, { undo: true });
                    // put the undo operations to be applied after the move to the end of the root generator
                    generator.appendOperations(beforeMoveGenerator, { undo: true });
                    // bug 51455: restore cell contents as very last step
                    generator.appendOperations(restoreCellsGenerator, { undo: true });
                });

                return promise;
            }

            // generates the operations to insert/delete entire columns or rows; the returned
            // promise will be resolved with an array containing the move descriptor as element
            function generateMoveIntervalOperations(generator, intervals, columns, insert) {

                // the band interval (full interval in opposite direction)
                var bandInterval = new Interval(0, docModel.getMaxIndex(!columns));
                // create the move descriptor containing all needed interval information
                var moveDesc = new MoveDescriptor(docModel, bandInterval, intervals, columns, insert);

                // create the insert/delete operations for all intervals
                return implGenerateMoveOperations(generator, [moveDesc], columns, insert, function () {
                    var collection = columns ? colCollection : rowCollection;
                    return collection.generateMoveOperations(generator, moveDesc);
                });
            }

            // generates the 'moveCells' operations to move the specified cell ranges;
            // the returned promise will be resolved with an array of move descriptors
            function generateMoveRangeOperations(generator, ranges, columns, insert) {

                // create the column/row bands of the ranges, containing the merged interval arrays per band
                var bandIntervals = ranges[columns ? 'getRowBands' : 'getColBands']({ intervals: true });
                // the move descriptors for all column/row bands to resolve the final promise with
                var moveDescs = bandIntervals.map(function (bandInterval) {
                    return new MoveDescriptor(docModel, bandInterval, bandInterval.intervals, columns, insert);
                });

                // generate and apply the 'moveCells' operations for all intervals in all bands
                return implGenerateMoveOperations(generator, moveDescs, columns, insert, function () {

                    // the move direction for document operations, and for undo operations
                    var moveDir = SheetUtils.getDirection(!columns, !insert);
                    var undoDir = SheetUtils.getDirection(!columns, insert);

                    // process all target intervals of all move descriptors
                    return self.iterateArraySliced(moveDescs, function (moveDesc) {
                        return self.iterateArraySliced(moveDesc.targetIntervals, function (interval) {
                            var targetRange = moveDesc.createRange(interval);
                            generator.generateMoveCellsOperation(targetRange, moveDir);
                            generator.generateMoveCellsOperation(targetRange, undoDir, { undo: true, prepend: true });
                        }, 'CellCollection.generateMoveOperations.generateMoveRangeOperations', { reverse: !insert }); // delete mode: in reversed order
                    }, 'CellCollection.generateMoveOperations.generateMoveRangeOperations');
                });
            }

            // the actual implementation of the public method returned from local scope
            function generateMoveCellsOperations(generator, ranges, direction) {
                SheetUtils.log('ranges=' + ranges + ' dir=' + direction.toString());

                // divide the ranges into column, row, and cell ranges (mixed range types not supported)
                var rangeGroups = getRangeGroups(ranges);
                if (_.isEmpty(rangeGroups)) { return self.createResolvedPromise(new IntervalArray()); }
                if (!Utils.hasSingleProperty(rangeGroups)) { return self.createRejectedPromise(); }

                // whether to change the column index of the cells (exit on wrong type of ranges)
                var columns = !SheetUtils.isVerticalDir(direction);
                // whether to insert or delete the columns/rows
                var insert = !SheetUtils.isLeadingDir(direction);

                // fail if inserting/deleting vertically in a column range, or horizontally in a row range
                if (columns ? rangeGroups.rowRanges : rangeGroups.colRanges) { return $.Deferred().reject(); }

                // create insert/delete operations for entire columns or rows, or 'moveCells' operations for simple cell ranges,
                // together with all necessary undo operations to restore all deleted cells, and to move back all shifted cells
                if (rangeGroups.colRanges) { return generateMoveIntervalOperations(generator, rangeGroups.colRanges.colIntervals(), true, insert); }
                if (rangeGroups.rowRanges) { return generateMoveIntervalOperations(generator, rangeGroups.rowRanges.rowIntervals(), false, insert); }
                return generateMoveRangeOperations(generator, rangeGroups.cellRanges, columns, insert);
            }

            return SheetUtils.profileAsyncMethod('CellCollection.generateMoveCellsOperations()', generateMoveCellsOperations);
        }());

        /**
         * Inserts one or more formulas calculating subtotal results into or
         * next to the specified cell range.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} funcKey
         *  The resource key of the subtotal function to be inserted into the
         *  formulas.
         *
         * @param {Range} range
         *  The source range used to generate the formulas.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the address of the target
         *  range inserted into the generated formulas when all operations have
         *  been generated.
         */
        this.generateAutoFormulaOperations = function (generator, funcKey, range) {

            function createCellContents(sourceRange, refAddress) {

                // generate the formula expression, and create a token array
                var formula = formulaGrammar.generateAutoFormula(docModel, funcKey, sourceRange, refAddress);
                var tokenArray = sheetModel.createCellTokenArray();
                tokenArray.parseFormula('op', formula, { refAddress: refAddress });

                // calculate the formula result, this will also return the resulting number format
                var result = tokenArray.interpretFormula('val', { refAddress: refAddress, targetAddress: refAddress });

                // create the cell contents object, and add the number format (except standard formats)
                var contents = { f: formula, v: result.value };
                var formatCode = result.format ? getNewCellFormatCode(refAddress, result.format) : null;
                if (formatCode) { contents.format = formatCode; }

                return contents;
            }

            function findNextEmptyAddress(address, direction) {
                var cellIt = self.createLinearAddressIterator(address, direction);
                var emptyAddress = null;
                Iterator.forEach(cellIt, function (address) {
                    if (self.isBlankCell(address)) {
                        emptyAddress = address;
                        return Utils.BREAK;
                    }
                });
                return emptyAddress;
            }

            var singleRow = range.singleRow();
            var contents = [];
            var emptyRow = -1;
            var x = -1;
            var targetRange = null;
            var startAddress = null;

            if (singleRow) {
                var firstAddress = self.findFirstCell(range, { type: 'value' });
                if (!firstAddress) {
                    targetRange = range;
                    singleRow = false;
                    var firstCol = getAutoFormulaRange(range.start.clone(), 'up', true).range;
                    if (!firstCol) { return self.createResolvedPromise(range); }
                    range = Range.create(firstCol.start[0], firstCol.start[1], range.end[0], range.end[1]);
                }
            }
            if (singleRow) {
                startAddress = findNextEmptyAddress(new Address(range.end[0], Math.max(range.start[1], range.end[1])), 'right');

                var contentRange = Range.create(range.start[0], range.start[1], Math.min(range.end[0], startAddress[0] - 1), range.end[1]);

                contents.push({ c: [createCellContents(contentRange, startAddress)] });
                var targetX = startAddress[0] <= range.end[0] ? range.end[0] : range.end[0] + 1;
                targetRange = Range.create(range.start[0], range.start[1], targetX, range.end[1]);

            } else {

                for (x = range.start[0]; x <= range.end[0]; x++) {
                    var next = findNextEmptyAddress(new Address(x, range.start[1]), 'down');
                    emptyRow = Math.max(emptyRow, next[1]);
                }

                var targetY = emptyRow <= range.end[1] ? range.end[1] : range.end[1] + 1;
                startAddress = new Address(range.start[0], Math.max(emptyRow, range.end[1]));
                var sourceRange = Range.create(range.start[0], range.start[1], range.start[0], targetY - 1);
                var contentLine = [];
                contents.push({ c: contentLine });

                for (x = range.start[0]; x <= range.end[0]; x++) {
                    sourceRange.setBoth(x, true);
                    contentLine.push(createCellContents(sourceRange, new Address(x, Math.max(emptyRow, range.end[1]))));
                }
                if (!targetRange) {
                    targetRange = Range.create(range.start[0], range.start[1], range.end[0], targetY);
                }
            }
            // generate and apply the operations
            return self.generateCellContentOperations(generator, startAddress, contents).then(_.constant(targetRange));
        };

        /**
         * Generates the cell operations, and the undo operations, to sort the
         * specified cell range according to one or more columns/rows.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Range} sortRange
         *  The address of the cell range to be sorted, without header cells.
         *
         * @param {SortDirection} sortDir
         *  The direction in which to sort the cell range.
         *
         * @param {Array<Object>} sortRules
         *  The sorting rules to be applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the addresses of the ranges
         *  that have really been changed) when all operations have been
         *  generated, or that will be rejected with an object with 'cause'
         *  property set to one of the following error codes:
         */
        this.generateSortOperations = SheetUtils.profileAsyncMethod('CellCollection.generateSortOperations()', function (generator, sortRange, sortDir, sortRules) {
            SheetUtils.log('range=' + sortRange);

            // immediately exit if the range contains any merged cell ranges
            if (sheetModel.getMergeCollection().coversAnyMergedRange(sortRange)) {
                return SheetUtils.makeRejected('sort:merge:overlap');
            }

            // prepare the data array used for sorting (cell values in order of sort rules)
            var vertical = sortDir === SortDirection.VERTICAL;
            var collection = vertical ? rowCollection : colCollection;
            var intervals = collection.getVisibleIntervals(sortRange.interval(!vertical));
            var promise = this.reduceSliced([], intervals.indexIterator(), function (sortLines, index) {
                var address = new Address(index, index);
                var values = sortRules.map(function (sortRule) {
                    address.set(sortRule.index, vertical);
                    return self.getValue(address);
                });
                sortLines.push({ fromIndex: index, values: values });
                return sortLines;
            }, 'CellCollection.generateSortOperations.prepareLinesArray');

            // sort the lines array by cell values
            promise = promise.then(function (sortLines) {

                // prepare scalar comparator options (empty cells always to the bottom of the range)
                var compareOptions = sortRules.map(function (sortRule) {
                    return { nullMode: sortRule.descending ? 'less' : 'greater' };
                });

                // sort the lines array by cell values
                sortLines.sort(function (sortLine1, sortLine2) {
                    return Utils.compareArrays(sortLine1.values, sortLine2.values, function (v1, v2, ai) {
                        var result = Scalar.compare(v1, v2, compareOptions[ai]);
                        return sortRules[ai].descending ? -result : result;
                    });
                });

                // filter lines that will not change their position
                var indexIterator = intervals.indexIterator();
                return sortLines.filter(function (sortLine) {
                    sortLine.toIndex = indexIterator.next().value;
                    return sortLine.fromIndex !== sortLine.toIndex;
                });
            });

            // generate the operations needed to sort the range
            promise = promise.then(function (sortLines) {

                // nothing to do if the sort order does not change at all
                if (sortLines.length === 0) { return; }

                // all cell contents to be moved
                var contentsArray = [];

                // hyperlink ranges to be sorted in the sort range
                var hyperlinkCollection = sheetModel.getHyperlinkCollection();
                var deletedLinkRanges = new RangeArray();
                var movedLinkRanges = new ValueMap();

                // collect all single-line hyperlink ranges inside the sorting range
                var allLinkRanges = hyperlinkCollection.getLinkRanges(sortRange).filter(function (linkRange) {
                    return sortRange.contains(linkRange) && linkRange.singleLine(!vertical);
                });

                // group hyperlink ranges by line index (according to sorting direction)
                var linkRangeGroups = allLinkRanges.group(function (linkRange) {
                    return linkRange.start.get(!vertical);
                });

                // cell comments to be sorted in the sort range
                var commentCollection = sheetModel.getCommentCollection();
                var commentsFrom = new AddressArray();
                var commentsTo = new AddressArray();

                // a private undo generator for restoring deleted contents at the end
                var undoGenerator = sheetModel.createOperationGenerator({ applyImmediately: true });

                // convert all shared formulas in the sorted range to regular formula cells
                var promise2 = self.generateBreakSharedFormulaOperations(undoGenerator, sortRange);

                // process the sort range, collect all contents to be sorted
                promise2 = promise2.then(function () {

                    // visit all cells (also in hidden columns/rows) in the sort line
                    var cellInterval = sortRange.interval(vertical);
                    return self.iterateArraySliced(sortLines, function (sortLine) {

                        // cell addresses for source and target sort line
                        var fromAddress = new Address(sortLine.fromIndex, sortLine.fromIndex);
                        var toAddress = new Address(sortLine.toIndex, sortLine.toIndex);

                        // process cell contents and comments for each cell in the sort line
                        Iterator.forEach(cellInterval, function (index) {

                            // create and initialize copies of source/dest addresses
                            fromAddress = fromAddress.clone().set(index, vertical);
                            toAddress = toAddress.clone().set(index, vertical);

                            // create and collect all content objects for moved cells
                            var cellModel = self.getCellModel(fromAddress);
                            var contents = {};
                            if (!cellModel) {
                                contents.u = true;
                            } else {
                                contents.v = cellModel.v;
                                contents.f = cellModel.f;
                                contents.s = cellModel.s;
                            }
                            contentsArray.push({ address: toAddress, contents: contents });

                            // collect start and end addresses of all moved cell comments
                            var commentModel = commentCollection.getByAddress(fromAddress);
                            if (commentModel) {
                                commentsFrom.push(fromAddress);
                                commentsTo.push(toAddress);
                            }
                        });

                        // process all hyperlinks in the sort line
                        var fromLinkRanges = linkRangeGroups[sortLine.fromIndex];
                        if (fromLinkRanges) {
                            deletedLinkRanges.append(fromLinkRanges);
                            fromLinkRanges.forEach(function (fromLinkRange) {
                                var toLinkRange = fromLinkRange.clone().setBoth(sortLine.toIndex, !vertical);
                                movedLinkRanges.getOrConstruct(fromLinkRange.url, RangeArray).push(toLinkRange);
                            });
                        }
                    }, 'CellCollection.generateSortOperations.prepareContents');
                });

                // delete sorted hyperlinks from old cell positions (via undoGenerator, to restore them at the end)
                promise2 = promise2.then(function () {
                    if (!deletedLinkRanges.empty()) {
                        return hyperlinkCollection.generateHyperlinkOperations(undoGenerator, deletedLinkRanges, '');
                    }
                });

                // create the cell operation that sorts the specified range
                promise2 = promise2.then(function () {
                    generator.appendOperations(undoGenerator);
                    return self.generateContentsArrayOperations(generator, contentsArray, { skipShared: true });
                });

                // create the hyperlink operations for single-cell hyperlinks
                promise2 = promise2.then(function () {
                    return self.iterateSliced(movedLinkRanges, function (ranges, result) {
                        return hyperlinkCollection.generateHyperlinkOperations(generator, ranges, result.key);
                    }, 'CellCollection.generateSortOperations.generateHyperlinks');
                });

                // create the drawing operations that relocate the cell comments
                promise2 = promise2.then(function () {
                    if (commentsFrom.length > 0) {
                        return sheetModel.getCommentCollection().generateMoveAnchorOperations(generator, commentsFrom, commentsTo);
                    }
                });

                // append the operations needed to restore deleted contents at the end of all other undo operations
                return promise2.done(function () {
                    generator.appendOperations(undoGenerator, { undo: true });
                });
            });

            return promise;
        });

        /**
         * Generates the operations to update all formula expressions in the
         * document, after cells that have been cut from another sheet, and
         * pasted within this sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} sourceSheet
         *  The index of the sheet containing the cells that have been cut.
         *
         * @param {Range} sourceRange
         *  The address of the cell range that has been cut.
         *
         * @param {Range} targetRange
         *  The address of the cell range the cells have been pasted into.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateCutPasteOperations = function (generator, sourceSheet, sourceRange, targetRange) {

            // a private operations generator needed to be able to prepend the undo operations
            var generator2 = sheetModel.createOperationGenerator({ applyImmediately: true });
            // the change descriptor for the cut/paste operation
            var updateDesc = { type: UpdateMode.CUT_PASTE, sourceSheet: sourceSheet, sourceRange: sourceRange, targetRange: targetRange };

            // update the formula expressions for all sheets in the document
            var promise = docModel.generateUpdateFormulaOperations(generator2, updateDesc);

            // prepend the undo operations, and append the change operations
            return promise.done(function () {
                generator.appendOperations(generator2);
                generator.prependOperations(generator2, { undo: true });
            });
        };

        // initialization -----------------------------------------------------

        // additional processing and event handling after the document has been imported
        sheetModel.waitForImportSuccess(function () {

            // update the sheet indexes after the sheet collection has been manipulated
            this.listenTo(docModel, 'transform:sheet', function (event, fromSheet, toSheet) {
                modelMap.forEach(function (cellModel) {
                    if (cellModel.t) { cellModel.t.transformSheet(fromSheet, toSheet); }
                });
            });

        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            colMatrix.destroy();
            rowMatrix.destroy();
            self = docModel = numberFormatter = formulaGrammar = formulaParser = autoStyles = null;
            sheetModel = colCollection = rowCollection = listCollection = null;
            modelMap = colMatrix = rowMatrix = sharedModelMap = matrixRangeSet = null;
        });

    } }); // class CellCollection

    // exports ================================================================

    return CellCollection;

});
