/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/commentmixin', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    // mix-in class CommentMixin ==============================================

    /**
     * Implementations of all controller items for manipulating cell comments
     * in the active sheet, intended to be mixed into a document controller
     * instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the current cell
     *  selection.
     */
    function CommentMixin(docView) {

        // the document model
        var docModel = docView.getDocModel();

        // the model and collections of the active sheet
        var sheetModel = null;
        var commentCollection = null;

        // the model of a cell comment made visible temporarily by "search" action
        var tempShowCommentModel = null;

        // private methods ----------------------------------------------------

        /**
         * Leaves text edit mode before modifying comments, and checks that the
         * comments in the active sheet are not locked.
         *
         * @param {String} [errorCode]
         *  An error code to be returned if the comments are locked.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the comments can be modified, or
         *  rejected, if the comments are locked.
         */
        function ensureUnlockedComments(errorCode) {
            return docView.leaveTextEditMode().then(function () {
                return docView.ensureUnlockedComments(errorCode ? { errorCode: errorCode } : null);
            });
        }

        function searchComment(reverse) {

            // the remaining ranges in the sheet to be visited, starting at the active cell
            var ranges = SheetUtils.getIteratorRanges(docModel.getSheetRange(), {
                reverse: reverse,
                startAddress: docView.getActiveCell(),
                skipStart: true
            });

            // create an iterator visiting the available cell comment models
            var iterator = commentCollection.createModelIterator({
                ranges: ranges,
                direction: 'horizontal',
                reverse: reverse,
                visibleCells: true
            });

            // resolve the next available comment model
            var iterResult = iterator.next();

            // show a message if there are no more comments available
            if (iterResult.done) {
                docView.yellMessage('comment:search:finished', true);
                return;
            }

            // select and scroll to the anchor cell of the comment
            var commentModel = iterResult.value;
            var address = commentModel.getAnchor();
            docView.selectCell(address);
            docView.scrollToCell(address);

            // make the comment visible temporarily
            docView.tempShowDrawingFrame(commentModel);
            tempShowCommentModel = commentModel;
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            'comment/model': {
                parent: 'sheet/operation/cell',
                enable: function () { return this.getValue() !== null; },
                get: function () { return commentCollection.getByAddress(docView.getActiveCell()); }
            },

            'comment/delete': {
                parent: 'comment/model',
                set: function () {

                    // the active cell comment model
                    var commentModel = this.getParentValue();
                    // check that the comments in the active sheet are not locked
                    var promise = ensureUnlockedComments('comment:delete:locked');

                    // generate and apply the operations
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {
                            return commentModel.generateDeleteOperations(generator);
                        }, { storeSelection: true });
                    });

                    // show warning alerts if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            'comment/toggle': {
                parent: 'comment/model',
                set: function () {

                    // the active cell comment model
                    var commentModel = this.getParentValue();
                    // leave edit mode (bug 56937: comments CAN be toggled in locked sheets)
                    var promise = docView.leaveTextEditMode();

                    // generate and apply the operations
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {
                            var drawingAttrs = { hidden: commentModel.isVisible() };
                            return commentModel.generateChangeOperations(generator, { drawing: drawingAttrs });
                        }, { storeSelection: true });
                    });

                    // show warning alerts if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            // parent iten that is enabled if the active sheet contains any comments
            'comment/available/enabled': {
                enable: function () { return commentCollection.hasComments(); }
            },

            // parent iten that is enabled if the active sheet contains comments in visible columns/rows
            'comment/available/visible/enabled': {
                enable: function () { return commentCollection.hasComments({ visibleCells: true }); }
            },

            'comment/delete/all': {
                parent: ['sheet/operation/cell', 'comment/available/enabled'],
                set: function () {

                    // check that the comments in the active sheet are not locked
                    var promise = ensureUnlockedComments('comment:delete:locked');

                    // generate and apply the operations
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {
                            return commentCollection.generateDeleteAllCommentsOperations(generator);
                        });
                    });

                    // show warning alerts if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            'comment/toggle/all': {
                parent: ['sheet/operation/cell', 'comment/available/visible/enabled'],
                set: function () {

                    // leave edit mode (bug 56937: comments CAN be toggled in locked sheets)
                    var promise = docView.leaveTextEditMode();

                    // generate and apply the operations
                    promise = promise.then(function () {
                        return sheetModel.createAndApplyOperations(function (generator) {
                            return commentCollection.generateToggleAllCommentsOperations(generator);
                        });
                    });

                    // show warning alerts if necessary
                    return docView.yellOnFailure(promise);
                }
            },

            'comment/show/next': {
                parent: 'comment/available/visible/enabled',
                set: function () { searchComment(false); }
            },

            'comment/show/prev': {
                parent: 'comment/available/visible/enabled',
                set: function () { searchComment(true); }
            }
        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function () {
            sheetModel = docView.getSheetModel();
            commentCollection = sheetModel.getCommentCollection();
        });

        // forget cached comment model made visible temporarily
        this.listenTo(docView, 'comment:delete', function (event, commentModel) {
            if (commentModel === tempShowCommentModel) {
                tempShowCommentModel = null;
            }
        });

        // hide comments that have been made visible temporarily
        this.listenTo(docView, 'change:selection:prepare', function () {
            if (tempShowCommentModel) {
                docView.tempHideDrawingFrame(tempShowCommentModel);
                tempShowCommentModel = null;
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = docView = sheetModel = commentCollection = null;
            tempShowCommentModel = null;
        });

    } // class CommentMixin

    // exports ================================================================

    return CommentMixin;

});
