/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Jonas Regier <jonas.regier@open-xchange.com>
 */

define('io.ox/office/presentation/view/popup/slidepanecontextmenu', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/popup/contextmenu',
    'io.ox/office/presentation/view/controls',
    'gettext!io.ox/office/presentation/main'
], function (Utils, ContextMenu, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;

    // class SlidePaneContextMenu =============================================

    /**
     * A context menu for sheet tab buttons. Provides menu actions to
     * manipulate the active sheet, or the sheet collection of the document.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {PresentationView} docView
     *  The presentation view containing this instance.
     */
    function SlidePaneContextMenu(docView, rootNode) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        // TODO enable key like in speadsheet??
        ContextMenu.call(this, docView, rootNode, {
            delay: 200
        });

        // private methods ----------------------------------------------------

        /**
         * ... before the context menu will be shown.

        function contextMenuPrepareHandler(event) {
        }
        */

        function initMenu() {

            // remove all existing controls from context menu
            self.destroyAllGroups();

            self.addGroup('slide/insertslide', new Button(docView, { label: gt('Insert slide') }), { focusTarget: rootNode }) // TODO focus target documentieren
                .addGroup('slide/insertlayoutslide', new Button(docView, { label: gt('Insert layout slide') }), { focusTarget: rootNode })
                .addGroup('slide/duplicateslides', new Button(docView, { label: gt('Duplicate slide') }), { focusTarget: rootNode })
                .addGroup('slide/deleteslide', new Button(docView, { label: gt('Delete slide') }), { focusTarget: rootNode })

                .addGroup('slide/hideslide', new Button(docView, { label: gt('Hide slide') }), { focusTarget: rootNode })
                .addGroup('slide/unhideslide', new Button(docView, { label: gt('Unhide slide') }), { focusTarget: rootNode })

                .addSeparator()
                .addGroup('layoutslidepicker/changelayout', new Controls.ChangeLayoutSlidePicker(docView, {

                    // overwrites
                    tooltip: null,
                    label: gt('Change layout'),
                    icon: 'fa-chevron-right',

                    // specialized additional parameters
                    iconPos: 'trailing',
                    caret: false,
                    anchorBorder: 'right bottom top left',

                    // speicial handling for odf layout slides
                    isOdfLayout: docView.getApp().isODF()
                }));

            if (docView.getApp().isODF()) {
                self.addSeparator()
                    .addGroup('layoutslidepicker/changemaster', new Controls.ChangeLayoutSlidePicker(docView, {

                        // overwrites
                        tooltip: null,
                        label: gt('Change master'),
                        icon: 'fa-chevron-right',

                        // specialized additional parameters
                        iconPos: 'trailing',
                        caret: false,
                        anchorBorder: 'right bottom top left',

                        // speicial handling for odf layout slides (but this is change of master slide)
                        isOdfLayout: false
                    }));
            }

            self.updateAllGroups(); //Bug 47352
        }

        // public methods -----------------------------------------------------

        this.getXY = function (event) {
            // TODO doc stick to slide
            if (!event.stickToSlide) {
                return {
                    pageX: event.sourceEvent.pageX,
                    pageY: event.sourceEvent.pageY
                };
            } else {
                var t = Utils.getNodePositionInPage(rootNode.children('.selected').find('.slidePaneThumbnail'));
                // calculate the position right to the slide with 5px offset
                return {
                    pageX: t.left + t.width + 5,
                    pageY: t.top
                };
            }
        };

        // initialization -----------------------------------------------------

        // preprocessing before the context menu will be shown
        //this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // to keep the focus after pressing esc to close the popup
        this.on('popup:hide', function () {
            Utils.setFocus(rootNode);
        });

        this.on('popup:beforeshow', initMenu); //Bug 47352

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = rootNode = null;
        });

    } // class SlidePaneContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: SlidePaneContextMenu });

});
