/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/presentation/view/dialog/insertfooterdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/textframework/view/controls',
    'gettext!io.ox/office/presentation/main',
    'less!io.ox/office/presentation/view/dialog/presentationdialogs'
], function (Utils, DateUtils, Forms, LocaleData, BaseDialog, Controls, gt) {

    'use strict';

    // Global constants and variables
    var dialogWidth = Utils.SMALL_DEVICE ? 320 : 420;

    // class InsertFooterDialog ===============================================

    /**
     * The insert date field modal dialog.
     * Provides localized date formats available for inserting into Presentation document.
     *
     *
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {PresentationView} docView
     *  The view instance containing this dialog.
     *
     * @param {Array} [dialogOptions]
     *  Options for filling the dialog
     *  - {Boolean} [dialogOptions.ftr]
     *      Footer placeholder exists on slide
     *  - {String} [dialogOptions.ftrText]
     *      Text inside footer placeholder
     *  - {Boolean} [dialogOptions.sldNum]
     *      Slide number placeholder exists on slide
     *  - {Boolean} [dialogOptions.dt]
     *      Datetime placeholder exists on slide
     *  - {String} [dialogOptions.dtType]
     *      Datetime placeholder contains date field and has this field type.
     *  - {String} [dialogOptions.dtText]
     *      Datetime placeholder contains fixed date as text.
     */
    var InsertFooterDialog = BaseDialog.extend(function (docView, dialogOptions) {

        var
            docModel = docView.getDocModel(),
            numberFormatter = docModel.getNumberFormatter(),
            // fetch locale date formats
            categoryCodesDate = numberFormatter.getCategoryCodes('date'),
            // whether this is an ODF application
            isODF = docModel.getApp().isODF(),
            // default fall back date option
            defaultDateOption = 'datetime1',
            insertFieldTitle = /*#. Insert footer fields from dialog menu*/ gt('Footer'),
            // container node for first column (category drop-down, fields type list)
            $controlArea = null,
            dateCheckbox = null,
            dateContainer = null,
            footerTxtContainer = null,
            slideNumCheckbox = null,
            footerTxtCheckbox = null,
            autoRadioBtn = null,
            fixedRadioBtn = null,
            applyallBtn = null;

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, docView, {
            title: insertFieldTitle,
            width: dialogWidth,
            okLabel: gt('Apply')
        });

        // private methods ----------------------------------------------------

        /**
         * Create current date(time) value string with passed format type.
         *
         * @param {String} formatCode
         * @return {String}
         */
        function getDateTimeRepresentation(formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        }

        /**
         * Helper function for enabling or disabling the specified DOM elements.
         *
         * @param {jQuery} nodes
         *  The DOM elements to be manipulated. If this object is a jQuery
         *  collection, uses all nodes it contains.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the DOM elements.
         */
        function enableNodes(nodes, state) {
            nodes.toggleClass(Forms.DISABLED_CLASS, !state);
            if (state) {
                nodes.removeAttr('aria-disabled');
                nodes.prop('tabindex', 0);
            } else {
                nodes.attr('aria-disabled', true);
                nodes.prop('tabindex', -1);
            }
        }

        /**
         * Generate field values for selected value (date), and insert them into list.
         *
         * returns {String} generated html string for select box
         */
        function populateFieldFormats() {
            var localDateFormatList = [];

            if (categoryCodesDate.length) {
                _.each(categoryCodesDate, function (formatCode, index) {
                    localDateFormatList.push({ value: (isODF ? formatCode.value : 'datetime' + (index + 1)), label: getDateTimeRepresentation(formatCode.value) });
                    if (isODF && index === 0) { defaultDateOption = formatCode.value; }
                });
            } else {
                localDateFormatList.push({ value: LocaleData.SHORT_DATE, label: getDateTimeRepresentation(LocaleData.SHORT_DATE) },
                    { value: (isODF ? LocaleData.LONG_DATE : 'datetime1'), label: getDateTimeRepresentation(LocaleData.LONG_DATE) });
                if (isODF) { defaultDateOption = LocaleData.SHORT_DATE; }
            }

            return Forms.createSelectBoxMarkup(localDateFormatList);
        }

        /**
         * Utility function for extracting object with given type from array of objects.
         *
         * @param {Array} optionsArray
         *  Array of objects.
         * @param {String} type
         *  Type of object which we are searching.
         * @returns {Object|undefined}
         */
        function extractByType(optionsArray, type) {
            return _.findWhere(optionsArray, { type: type });
        }

        /**
         * Initialize controls of the dialog.
         */
        function initControls() {
            var dateObject = extractByType(dialogOptions, 'dt');
            var isDate = !!dateObject;
            var isAutoDate = (dateObject && dateObject.automatic) || false;
            var slideNumObject = extractByType(dialogOptions, 'sldNum');
            var isSlideNum = !!slideNumObject;
            var footerObject = extractByType(dialogOptions, 'ftr');
            var isFooter = !!footerObject;
            var isAutoRadioEnabled = isDate ? isAutoDate : true;
            var isFixedRadioEnabled = isDate ? !isAutoDate : false;

            var dateTimeBtn = Forms.createButtonMarkup({ attributes: { class: 'date-time-btn', role: 'checkbox' }, label: gt('Date and time') });
            var slideNumberBtn = Forms.createButtonMarkup({ attributes: { class: 'slide-number-btn', role: 'checkbox' }, label: gt('Slide number') });
            var slideNumContainer = $('<div class="slidenum-container">').append(slideNumberBtn);
            var footerBtn = Forms.createButtonMarkup({ attributes: { class: 'footer-btn', role: 'checkbox' }, label: gt('Footer') });
            var fixedDateTextArea = Forms.createInputMarkup({ attributes: { disabled: true } });
            var autoRadio = Forms.createButtonMarkup({ attributes: { class: 'automatic-radio dradio', role: 'radio' }, label: gt('Update automatically') });
            var fixedRadio = Forms.createButtonMarkup({ attributes: { class: 'fixed-radio dradio', role: 'radio' }, label: gt('Fixed date') });

            dateContainer = $('<div class="date-container">').append(autoRadio, populateFieldFormats(), fixedRadio, fixedDateTextArea);
            // depending from auto or fixed date, disable select and input fields
            dateContainer.find('select').prop('disabled', !isDate || !isAutoDate).toggleClass('disabled', !isDate || !isAutoDate).val((dateObject && dateObject.dtType) || defaultDateOption);
            dateContainer.find('input').prop('disabled', !isDate || isAutoDate).toggleClass('disabled', !isDate || isAutoDate);
            if (isDate && !isAutoDate) {
                dateContainer.find('input').val(dateObject && dateObject.dtText);
            } else {
                dateContainer.find('input').val(dateContainer.find('option:selected').text());
            }
            // if date chexbox is checked, enable auto and fixed radios
            enableNodes(dateContainer.find('.dradio'), isDate);

            var footerTxtTextArea = Forms.createInputMarkup({ attributes: !isFooter ? { disabled: true } : {} });
            footerTxtContainer = $('<div class="footertxt-container">').append(footerBtn, footerTxtTextArea);
            footerTxtContainer.find('input').val(footerObject && footerObject.ftrText);

            // append all containers to control area
            $controlArea.append(dateTimeBtn, dateContainer, footerTxtContainer, slideNumContainer);

            // date checkbox
            dateCheckbox = $controlArea.find('.date-time-btn');
            Forms.checkButtonNodes(dateCheckbox, isDate, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(dateCheckbox);
            // auto date radio
            autoRadioBtn = $controlArea.find('.automatic-radio');
            Forms.checkButtonNodes(autoRadioBtn, isAutoRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(autoRadioBtn);
            // fixed date radio
            fixedRadioBtn = $controlArea.find('.fixed-radio');
            Forms.checkButtonNodes(fixedRadioBtn, isFixedRadioEnabled, { design: 'radio', ambiguous: false });
            Forms.setButtonKeyHandler(fixedRadioBtn);

            // footer text checkbox
            footerTxtCheckbox = $controlArea.find('.footer-btn');
            Forms.checkButtonNodes(footerTxtCheckbox, isFooter, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(footerTxtCheckbox);
            // slide number checkbox
            slideNumCheckbox = $controlArea.find('.slide-number-btn');
            Forms.checkButtonNodes(slideNumCheckbox, isSlideNum, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(slideNumCheckbox);
        }

        function actionHandler(action) {
            var selectedOption;
            var formatIndex;
            var obj = null;
            var checkedOptions = [];
            var diffContent;
            var diffState;
            var odfRepresentation = null;

            if (Forms.isCheckedButtonNode(dateCheckbox)) {
                if (Forms.isCheckedButtonNode(autoRadioBtn)) {
                    selectedOption = $controlArea.find('option:selected');
                    formatIndex = selectedOption.length ? (selectedOption.index() + 1) : null;
                    if (isODF) { odfRepresentation = categoryCodesDate[selectedOption.index()]; }
                    obj = { type: 'dt', automatic: true, formatIndex: formatIndex, representation: isODF ? (odfRepresentation ? odfRepresentation.value : '') : selectedOption[0].text };
                } else {
                    obj = { type: 'dt', automatic: false, representation: dateContainer.find('input').val() };
                }
                checkedOptions.push(obj);
            }
            if (Forms.isCheckedButtonNode(slideNumCheckbox)) {
                checkedOptions.push({ type: 'sldNum' });
            }
            if (Forms.isCheckedButtonNode(footerTxtCheckbox)) {
                checkedOptions.push({ type: 'ftr', ftrText: footerTxtContainer.find('input').val() });
            }

            if (isODF) {
                return docModel.getFieldManager().insertFooterODF(checkedOptions, (action === 'applyall'));
            }

            switch (action) {
                case BaseDialog.OK_ACTION:
                    diffContent = docModel.getFieldManager().diffInputContent(dialogOptions, checkedOptions);
                    diffState = docModel.getFieldManager().diffStateCheckboxes(dialogOptions, checkedOptions);
                    return docModel.getFieldManager().insertFooter(diffContent, diffState);
                case 'applyall':
                    return docModel.getFieldManager().insertAllFooters(checkedOptions);
            }
        }

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        $controlArea = $('<div class="control-area">');

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-presentation-dialog-insert-footer')
            .toggleClass('mobile', !!Utils.SMALL_DEVICE)
            .append($controlArea);

        // create the layout of the dialog
        initControls();

        // block keyboard input during applying of operations
        this.on('show', function () {
            docModel.setBlockKeyboardEvent(true);
        });

        // handler for the OK button
        this.setOkHandler(actionHandler);

        // add additional button
        applyallBtn = this.createButton('applyall', gt('Apply to all'), { buttonStyle: 'primary', actionHandler: actionHandler });

        // ensure proper button order on small devices, because base dialog makes reorder
        if (Utils.SMALL_DEVICE) {
            this.on('beforeshow', function () {
                applyallBtn.parent().detach();
                this.getFooter().children('.row').append(applyallBtn.parent());
            });
        }

        // listen on checkbox to change icon
        dateCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
            // enable/disable date subgroup
            enableNodes(dateContainer.find('.dradio'), Forms.isCheckedButtonNode(this));
            dateContainer.find('select, input').prop('disabled', !Forms.isCheckedButtonNode(this)).toggleClass('disabled', !Forms.isCheckedButtonNode(this));
            if (Forms.isCheckedButtonNode(autoRadioBtn)) {
                dateContainer.find('select').prop('disabled', !Forms.isCheckedButtonNode(this)).toggleClass('disabled', !Forms.isCheckedButtonNode(this));
                dateContainer.find('input').prop('disabled', true).addClass('disabled');
            } else {
                dateContainer.find('input').prop('disabled', !Forms.isCheckedButtonNode(this)).toggleClass('disabled', !Forms.isCheckedButtonNode(this));
                dateContainer.find('select').prop('disabled', true).addClass('disabled');
            }
        });
        slideNumCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
        });
        footerTxtCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
            // enable/disable footer textarea
            footerTxtContainer.find('input').prop('disabled', !Forms.isCheckedButtonNode(this));

        });

        autoRadioBtn.on('click', function () {
            Forms.checkButtonNodes(fixedRadioBtn, false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            dateContainer.find('select').prop('disabled', false).removeClass('disabled');
            dateContainer.find('input').prop('disabled', true).addClass('disabled');
        });

        fixedRadioBtn.on('click', function () {
            Forms.checkButtonNodes(autoRadioBtn, false, { design: 'radio', ambiguous: false });
            Forms.checkButtonNodes(this, true, { design: 'radio', ambiguous: false });
            dateContainer.find('input').prop('disabled', false).removeClass('disabled');
            dateContainer.find('select').prop('disabled', true).addClass('disabled');
        });

        this.registerDestructor(function () {
            docModel.setBlockKeyboardEvent(false);
            docModel = numberFormatter = categoryCodesDate = dateContainer = applyallBtn = null;
            $controlArea = dateCheckbox = footerTxtContainer = null;
            slideNumCheckbox = footerTxtCheckbox = autoRadioBtn = fixedRadioBtn = null;
        });

    }); // class InsertFooterDialog

    // exports ================================================================

    return InsertFooterDialog;

});
