/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/presentation/utils/presentationutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/tk/render/rectangle',
    'gettext!io.ox/office/presentation/main'
], function (Utils, DrawingUtils, DrawingFrame, AttributeUtils, DOM, Position, Rectangle, gt) {

    'use strict';

    // static class PresentationUtils =========================================

    var PresentationUtils = {};

    PresentationUtils.PLACEHOLDER_DRAWING_IMPORTANT_ATTRS = ['left', 'top', 'width', 'height', 'rotation', 'flipH', 'flipV'];

    /**
     * The default type for place holders in presentation attributes.
     *
     * @constant
     */
    PresentationUtils.DEFAULT_PLACEHOLDER_TYPE = 'body';

    /**
     * The table place holder type.
     *
     * @constant
     */
    PresentationUtils.TABLE_PLACEHOLDER_TYPE = 'tbl';

    /**
     * The image place holder type.
     *
     * @constant
     */
    PresentationUtils.IMAGE_PLACEHOLDER_TYPE = 'pic';

    /**
     * A name used to differentiate between the two place holder types 'body'.
     * There is a content body, if 'body' is not set a explicit attribute and
     * a text body, if the phType is explicitely set to 'body'. This marker
     * can be used to mark content body place holders.
     *
     * @constant
     */
    PresentationUtils.CONTENTBODY = 'contentbody';

    /**
     * The title types for place holders title drawings.
     */
    PresentationUtils.TITLE_PLACEHOLDER_TYPES = {
        title: 1,
        ctrTitle: 1
    };

    PresentationUtils.TITLE_OR_SUBTITLE_PLACEHOLDER_TYPES = {
        title: 1,
        ctrTitle: 1,
        subTitle: 1
    };

    /**
     * The place holder types, whose content cannot be modified or that cannot be
     * deleted on master slides in ODF.
     */
    PresentationUtils.ODF_MASTER_READONLY_TYPES = {
        title: 1,
        body: 1
    };

    /**
     * The place holder drawing types, in that no text content can be inserted.
     */
    PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE = {
        media: 1,
        clipArt: 1,
        chart: 1,
        dgm: 1,
        tbl: 1,
        pic: 1
    };

    /**
     * Defining the name of that attributes family that must be defined as explicit
     * attribute to a place holder drawing that cannot contain text.
     * If a place holder drawing of type 'tbl' contains the 'table' family in its
     * explicit attributes, it cannot be empty (in this case the template images for
     * inserting content must not be inserted). The same is true for a place holder
     * drawing of type 'pic', if the explicit attributes contain the image family.
     */
    PresentationUtils.NO_TEXT_PLACEHOLDER_CONTENT_PROPERTY = {
        media: null,
        clipArt: 'image',
        chart: null,
        dgm: null,
        tbl: 'table',
        pic: 'image'
    };

    /**
     * The default texts that are inserted, if the place holder drawing is empty.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS = {
        body: gt('Click to add text'),
        title: gt('Click to add title'),
        ctrTitle: gt('Click to add title'),
        subTitle: gt('Click to add subtitle'),
        media: gt('Media clip'),
        clipArt: gt('Online image'),
        chart: gt('Chart'),
        dgm: gt('Smart Art'),
        tbl: gt('Click to add table'),
        pic: gt('Click to add image'),
        sldNum: gt('Slide Number'),
        ftr: gt('Footer'),
        dt: gt('Date')
    };

    /**
     * The default texts that are inserted, if the place holder drawing is empty (ODF format).
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_ODF = {
        body: gt('Click to add text'),
        title: gt('Click to add title'),
        ctrTitle: gt('Click to add title'),
        subTitle: gt('Click to add text'),
        media: gt('Media clip'),
        clipArt: gt('Online image'),
        chart: gt('Chart'),
        dgm: gt('Smart Art'),
        tbl: gt('Click to add table'),
        pic: gt('Click to add image'),
        sldNum: gt('Slide Number'),
        ftr: gt('Footer'),
        dt: gt('Date')
    };

    /**
     * The default texts that are inserted, if the place holder drawing with click button is empty.
     * Info: These strings need to be used by the filter.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_CLICK = {
        pic: gt('Click icon to add image'),
        tbl: gt('Click icon to add table'),
        chart: gt('Click icon to add chart'),
        dgm: gt('Click icon to add smart art graphic'),
        clipArt: gt('Click icon to add clip art'),
        media: gt('Click icon to add media')
    };

    /**
     * The supported buttons inside a place holder drawing.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS = {
        contentBody: ['picture', 'table'],
        body: null,
        title: null,
        ctrTitle: null,
        subTitle: null,
        media: ['media'],
        clipArt: ['clipart'],
        chart: ['chart'],
        dgm: ['smartart'],
        tbl: ['table'],
        pic: ['picture'],
        sldNum: null,
        ftr: null,
        dt: null
    };

    /**
     * The supported buttons inside a place holder drawing in ODF format.
     */
    PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS_ODF = {
        contentBody: ['picture', 'table'],
        body: ['picture', 'table'],
        title: null,
        ctrTitle: null,
        subTitle: null,
        media: ['media'],
        clipArt: ['clipart'],
        chart: ['chart'],
        dgm: ['smartart'],
        tbl: ['table'],
        pic: ['picture'],
        sldNum: null,
        ftr: null,
        dt: null
    };

    /**
     * Supported names for inserting custom drawing placeholders.
     */
    PresentationUtils.PLACEHOLDER_DRAWING_NAMES = {
        content:    gt('Content placeholder'),
        contentv:   gt('Vertical content placeholder'),
        text:       gt('Text placeholder'),
        textv:      gt('Vertical text placeholder'),
        picture:    gt('Image placeholder'),
        table:      gt('Table placeholder')
    };

    /**
     * The template text inserted into supported placeholder drawings.
     */
    PresentationUtils.PLACEHOLDER_TEXT_CONTENT = [
        gt('Edit Master text styles'),
        gt('Second level'),
        gt('Third level'),
        gt('Fourth level'),
        gt('Fifth level')
    ];

    /**
     * The template text inserted into title placeholder drawing.
     */
    PresentationUtils.PLACEHOLDER_TITLE_CONTENT = gt('Click to edit Master title style');

    /**
     * The default index for place holders in presentation attributes.
     *
     * @constant
     */
    PresentationUtils.DEFAULT_PLACEHOLDER_INDEX = 0;

    // methods ----------------------------------------------------------------

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes. This means, that the phType is set, or that the phIndex is specified. It is possible,
     * that only the phIndex is specified. In this case the phType is defaulted to 'body'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes.
     */
    PresentationUtils.isPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && (attrs.presentation.phType || _.isNumber(attrs.presentation.phIndex)));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a title place holder drawing. This means, that the phType is set to 'title' or
     * 'ctrTitle'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'title' drawing.
     */
    PresentationUtils.isTitlePlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.TITLE_PLACEHOLDER_TYPES[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a title or subtitle place holder drawing. This means, that the phType is set to
     * 'title', 'ctrTitle' or 'subTitle'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'title' or 'subtitle' drawing.
     */
    PresentationUtils.isTitleOrSubtitlePlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.TITLE_OR_SUBTITLE_PLACEHOLDER_TYPES[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing. This means, that the phType is set to 'body' or
     * that it is not defined.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing.
     */
    PresentationUtils.isBodyPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && _.isNumber(attrs.presentation.phIndex) && (attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE || !attrs.presentation.phType));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing for inserting text only. This means, that the
     * phType is set explicitely to 'body'.
     * Info: In ODF file format this is quite different, because there is no place holder of type body, that
     *       accepts only text input.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing
     *  that allows only text insertion.
     */
    PresentationUtils.isTextBodyPlaceHolderAttributeSet = function (attrs, isODF) {
        var odf = isODF || false;
        return !!(!odf && attrs && attrs.presentation && _.isNumber(attrs.presentation.phIndex) && (attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing that allows any content (text, table, picture, ...).
     * This means, that the phType is explicitely undefined and the phIndex is explicitely defined.
     * Info: In ODF file format this is quite different. The phType is set to 'body' and it is not necessary,
     *       that a phIndex is specified.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing
     *  that allows input of any content (text, table, picture, ...).
     */
    PresentationUtils.isContentBodyPlaceHolderAttributeSet = function (attrs, isODF) {
        var odf = isODF || false;
        return !!(attrs && attrs.presentation && (_.isNumber(attrs.presentation.phIndex) || odf) && (!attrs.presentation.phType || (odf && attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE)));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) describes a place holder drawing
     * in which no text can be inserted.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of a place holder drawing, in
     *  that no text can be inserted.
     */
    PresentationUtils.isPlaceHolderWithoutTextAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set describes a place holder drawing with user defined template text.
     *
     * @param {Object} attrs
     *  The attribute object. This might be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of a place holder drawing with
     *  user defined template text.
     */
    PresentationUtils.isCustomPromptPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.customPrompt);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a place holder drawing in that in ODF in the master view the content cannot be
     * modified.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of a drawing that cannot
     *  be deleted or whose content cannot be modified in ODF masterview.
     */
    PresentationUtils.isODFReadOnlyPlaceHolderAttributeSet = function (attrs) {
        return !!(attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.ODF_MASTER_READONLY_TYPES[attrs.presentation.phType]);
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a body place holder drawing without specified index. This means, that the phType
     * is set to 'body' and the phIndex is not specified (used for ODF).
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'body' drawing without
     *  specified index.
     */
    PresentationUtils.isBodyPlaceHolderAttributeSetWithoutIndex = function (attrs) {
        return !!(attrs && attrs.presentation && !_.isNumber(attrs.presentation.phIndex) && (attrs.presentation.phType === PresentationUtils.DEFAULT_PLACEHOLDER_TYPE));
    };

    /**
     * Checking, whether the specified attribute set (explicit attributes from a drawing) specifies
     * a place holder drawing whose size and/or position was modified by the user.
     * This check is important for inheritance from master slide in ODP documents.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set specifies a place holder drawing whose size
     *  and/or position was modified by the user..
     */
    PresentationUtils.isUserModifiedPlaceHolderAttributeSet = function (attrs) {
        return !!((attrs && attrs.presentation && attrs.presentation.userTransformed) || false);
    };

    /**
     * Check, whether the specified drawing is a drawing that contains place holder attributes. This means, that
     * the phType is set, or that the phIndex is specified. It is possible, that only the phIndex is specified. In
     * this case the phType is defaulted to 'body'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes.
     */
    PresentationUtils.isPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Helper function to get the place holder type from a specified drawing. If it cannot be determined,
     * null is returned.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @param {Object} [drawingAttrs]
     *  The explicit drawing attributes. They can be specified for performance reasons.
     *
     * @returns {String|Null}
     *  The place holder type or null if it cannot be determined.
     */
    PresentationUtils.getPlaceHolderDrawingType = function (drawing, drawingAttrs) {

        var // the explicit drawing attributes
            attrs = drawingAttrs || AttributeUtils.getExplicitAttributes(drawing);

        return (attrs && attrs.presentation && attrs.presentation.phType) ? attrs.presentation.phType : null;
    };

    /**
     * Helper function to get the place holder index from a specified drawing. If it cannot be determined,
     * null is returned.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @param {Object} [drawingAttrs]
     *  The explicit drawing attributes. They can be specified for performance reasons.
     *
     * @returns {Number|Null}
     *  The place holder index or null if it cannot be determined.
     */
    PresentationUtils.getPlaceHolderDrawingIndex = function (drawing, drawingAttrs) {

        var // the explicit drawing attributes
            attrs = drawingAttrs || AttributeUtils.getExplicitAttributes(drawing);

        return (attrs && attrs.presentation && _.isNumber(attrs.presentation.phIndex)) ? attrs.presentation.phIndex : null;
    };

    /**
     * Check, whether the specified drawing is a title drawing that contains place holder attributes. This means, that
     * the phType is set to 'title' or 'ctrTitle'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'title'.
     */
    PresentationUtils.isTitlePlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTitlePlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a title or a subtitle drawing that contains place holder attributes.
     * This means, that the phType is set to 'title', 'ctrTitle' or 'subTitle'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'title' or 'subTitle'.
     */
    PresentationUtils.isTitleOrSubtitlePlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTitleOrSubtitlePlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a body drawing that contains place holder attributes. This means, that
     * the phType is set to 'body or is not specified and the the phIndex is specified.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'body'.
     */
    PresentationUtils.isBodyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isBodyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a body drawing that contains place holder attributes. This means, that
     * the phType is set explicitely to 'body'. In this case, only text content can be inserted into the
     * place holder drawing.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'body' that allows
     *  only text insertion.
     */
    PresentationUtils.isTextBodyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTextBodyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a body drawing that contains place holder attributes. This means, that
     * the phType is explicitely undefined. In this case, any content can be inserted into the place holder drawing
     * (text, table, picture, ...).
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'body' that allow input
     *  of any content into the 'body'.
     */
    PresentationUtils.isContentBodyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isContentBodyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a place holder drawing, in that no text can be inserted. In this
     * case it contains no paragraphs (in master/layout view a paragraph is inserted, but not in the document
     * view). Instead a button is available that can be used to insert a table or an image.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing, that does not support text input.
     */
    PresentationUtils.isPlaceHolderWithoutTextDrawing = function (drawing) {
        return PresentationUtils.isPlaceHolderWithoutTextAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is an empty place holder drawing. For all types of place
     * holders it is checked, if the place holder contains the possible content.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is an empty place holder drawing.
     */
    PresentationUtils.isEmptyPlaceHolderDrawing = function (drawing) {

        // the explicit attributes of the drawing
        var attrs = AttributeUtils.getExplicitAttributes(drawing);
        // whether this is a place holder that does not contain text and that contains no content
        var isEmpty = false;
        // the selector to find a node that only exists, if the place holder is not empty
        var attributesProperty = null;

        // First check: Has the place holder a valid type
        if (PresentationUtils.isPlaceHolderWithoutTextAttributeSet(attrs)) {

            if ($(drawing).attr('data-type') !== 'undefined') {
                // type: 'undefined' : handling place holder drawings with unknown content -> they are never empty
                // -> otherwise checking the explicit attributes to find out, if the drawing contains content like a table or an image
                attributesProperty = PresentationUtils.NO_TEXT_PLACEHOLDER_CONTENT_PROPERTY[attrs.presentation.phType];

                // ... and is the place holder empty (or contains only the template text
                if (!attrs || !attributesProperty || !attrs[attributesProperty]) {
                    isEmpty = true;
                }

            }
        } else if (PresentationUtils.isPlaceHolderAttributeSet(attrs)) {
            // handling of those place holders, that allow text input (text body place holders and content body place holders)
            // but also title, ctrTitle, ...
            isEmpty = DOM.isEmptyTextframe(drawing, { ignoreTemplateText: true });
        }

        return isEmpty;
    };

    /**
     * Check, whether the specified drawing is a place holder drawing that contains a user
     * specified template text.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing with user defined template text.
     */
    PresentationUtils.isCustomPromptPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isCustomPromptPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a place holder drawing that cannot be deleted or whose content
     * cannot be modified in master view in ODF files.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing that cannot be deleted or whose content
     *  cannot be modified in master view in ODF files.
     */
    PresentationUtils.isODFReadOnlyPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isODFReadOnlyPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a place holder drawing with explicitely set phType to 'body' and
     * undefined 'phIndex' (used for ODF).
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing with explicitely set phType to 'body' and
     *  undefined 'phIndex'.
     */
    PresentationUtils.isBodyPlaceHolderWithoutIndexDrawing = function (drawing) {
        return PresentationUtils.isBodyPlaceHolderAttributeSetWithoutIndex(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Checking, whether the specified drawing is a place holder drawing whose size
     * and/or position was modified by the user.
     * This check is important for inheritance from master slide in ODP documents.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing is a place holder drawing whose size
     *  and/or position was modified by the user.
     */
    PresentationUtils.isUserModifiedPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isUserModifiedPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Checking, whether the specified drawing requires the property 'userTransformed'.
     * This is the case of place holder drawings in ODF, that are moved or resized
     * by the user.
     * This check is important for inheritance from master slide in ODP documents.
     * This check can be used after a drawing was moved or resized by the user, so
     * that a valid operation can be generated.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing requires the property 'userTransformed'. This is
     *  the case, if it is a place holder drawing whose size and/or position was modified
     *  by the user.
     */
    PresentationUtils.requiresUserTransformedProperty = function (drawing) {
        var attrs = AttributeUtils.getExplicitAttributes(drawing);
        return PresentationUtils.isPlaceHolderAttributeSet(attrs) && !PresentationUtils.isUserModifiedPlaceHolderAttributeSet(attrs);
    };

    /**
     * Getting an object containing the keys 'target', 'type' and 'index' for a specified drawing. This
     * object can be used to get attributes and list styles from the model for the drawing
     *
     * Info: Place holder drawings cannot be grouped, they are always direct children of a slide.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @argument {Object} options
     *  Optional parameters:
     *  @param {Boolean} [useDefaultValues=true]
     *      If set to true, the default values for drawing place holder type and index are used.
     *      Otherwise the explicit attributes for type and index are inserted into the return object,
     *      but this might not be defined.
     *
     * @returns {Object}
     *  An object describing 'target' (the ID of the slide), 'type' (the type of the drawing) and
     *  'index' of a specified drawing.
     */
    PresentationUtils.getDrawingModelDescriptor = function (drawing, options) {

        var // the explicit attributes set at the drawing
            drawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
            // whether the default values for type and index shall be returned
            useDefaultValues = Utils.getBooleanOption(options, 'useDefaultValues', true),
            // the return object containing keys 'target', 'type' and 'index'
            drawingObj = null;

        if (PresentationUtils.isPlaceHolderAttributeSet(drawingAttrs)) {

            drawingObj = {};

            drawingObj.target = DOM.getTargetContainerId($(drawing).parent()); // place holder drawings are direct children of slide
            drawingObj.type = drawingAttrs.presentation.phType;
            drawingObj.index = drawingAttrs.presentation.phIndex;

            if (useDefaultValues) {
                if (drawingObj.type && !_.isNumber(drawingObj.index)) { drawingObj.index = PresentationUtils.DEFAULT_PLACEHOLDER_INDEX; }
                if (_.isNumber(drawingObj.index) && !drawingObj.type) { drawingObj.type = PresentationUtils.DEFAULT_PLACEHOLDER_TYPE; }
            }
        }

        return drawingObj;
    };

    /**
     * Receiving a valid set for the properties 'type' and 'index', so that the model can be accessed.
     * Typically the specified attributes parameter is the set of explicit drawing attributes.
     *
     * @param {Object} attrs
     *  The attributes object. This are typically the explicit attributes set at a drawing.
     *
     * @returns {Object|Null}
     *  An object containing the properties 'type' and 'index'. 'type' is a string, that describes
     *  the place holder type, 'index' is a number describing the index specified for the given
     *  place holder type.
     *  If the specified attribute does not contain a 'presentation' property or inside this
     *  'presentation' property neither 'phType' nor 'phIndex' are defined, null is returned.
     */
    PresentationUtils.getValidTypeIndexSet = function (attrs) {

        // it is necessary that at least one of phType or phIndex is defined in the attributes
        if (!attrs || !attrs.presentation || !('phType' in attrs.presentation || 'phIndex' in attrs.presentation)) {
            return null;
        }

        return {
            type: Utils.getStringOption(attrs.presentation, 'phType', PresentationUtils.DEFAULT_PLACEHOLDER_TYPE),
            index: Utils.getIntegerOption(attrs.presentation, 'phIndex', PresentationUtils.DEFAULT_PLACEHOLDER_INDEX)
        };
    };

    /**
     * Getting the default text for a place holder drawing. This text is dependent from the place holder type.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {String|Null}
     *  The default text for a place holder drawing. Or null, if it cannot be determined.
     */
    PresentationUtils.getPlaceHolderTemplateText = function (drawing, isODF) {

        var // the place holder type
            type = PresentationUtils.getPlaceHolderDrawingType(drawing),
            // whether this is an ODF application.
            odf = isODF || false;

        if (!type && _.isNumber(PresentationUtils.getPlaceHolderDrawingIndex(drawing))) {
            type = PresentationUtils.DEFAULT_PLACEHOLDER_TYPE;
        }

        return type ? (odf ? PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_ODF[type] : PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS[type]) : null;
    };

    /**
     * Getting a list of strings, that describe the required buttons for a specified place holder type.
     * If the type is the empty string, it is defaulted to the place holder content body.
     *
     * @param {type} String
     *  The string describing the place holder type. The content body type must use the empty string.
     *
     * @param {Boolean} [isODF=false]
     *  Whether this is an ODF application. If not specified, the value is set to 'false'.
     *
     * @returns {Array|Null}
     *  An array containing strings for the required buttons for the specified place holder type.
     */
    PresentationUtils.getPlaceHolderTemplateButtonsList = function (type, isODF) {
        var buttonMap = isODF ? PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS_ODF : PresentationUtils.PLACEHOLDER_TEMPLATE_BUTTONS;
        return buttonMap[type || 'contentBody']; // using special name for content body place holders
    };

    /**
     * Getting the default representation string for fields in footer place holder types in ODF.
     *
     * @param {type} String
     *  The string describing the place holder type.
     *
     * @returns {String}
     *  The default representation string for fields in footer place holder types in ODF. Or an
     *  empty string, if it cannot be determined.
     */
    PresentationUtils.getODFFooterRepresentation = function (type) {
        var representationContent = PresentationUtils.PLACEHOLDER_TEMPLATE_TEXTS_ODF[type];
        return representationContent ? ('<' + representationContent + '>') : '';
    };

    /**
     * This function converts the properties of a rectangle object ('top', 'left',
     * 'width' and 'height'). Specified is a rectangle that has values in pixel
     * relative to the app-content-root node. This is for example created by the
     * selectionBox. All properties are converted into 1/100 mm relative to the
     * slide. Therefore the returned values can directly be used for the creation
     * of operations. The zoom level is also taken into account.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {Object} box
     *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
     *  The values are given in pixel relative to the app-content-root node.
     *
     * @param {Boolean} [notScaleBoxSize=false]
     *  Optional flag to prevent the scaling from the box size. The default is false,
     *  so when it's not provided the box is not scaled. This is useful when a box is
     *  not based on the px size from the selection box (created with the mouse), but
     *  with fixed values. These fixed values are not depended on the scaling and
     *  therefore must not be scaled.
     *
     *
     * @returns {Object}
     *  The rectangle with the properties 'top', 'left', 'width' and 'height'.
     *  The values are given in 1/100 mm relative to the slide.
     */
    PresentationUtils.convertAppContentBoxToOperationBox = function (app, box, notScaleBoxSize) {

        var // the current zoom factor
            zoomFactor = app.getView().getZoomFactor(),
            // the active slide node
            activeSlide = app.getModel().getSlideById(app.getModel().getActiveSlideId()),
            // the offset of the active slide inside the content root node
            slideOffset = activeSlide.offset(), // the slide is positioned absolutely
            // the content root node, the base for the position calculation
            contentRootNode = app.getView().getContentRootNode(),
            // the position of the content root node
            pos = contentRootNode.offset(),
            // the horizontal scroll shift
            scrollLeft = contentRootNode.scrollLeft(),
            // the vertical scroll shift
            scrollTop = contentRootNode.scrollTop(),
            // the new box object
            operationBox = {},
            // flag if the width/height from the box should be scaled by the zoomFactor or not
            notScaleBoxSizeFlag = _.isBoolean(notScaleBoxSize) ? notScaleBoxSize : false;

        // the left position of the user defined box relative to the slide (can be negative)
        operationBox.left = (box.left - (slideOffset.left - pos.left) - scrollLeft) / zoomFactor;
        // the top position of the user defined box relative to the slide (can be negative)
        operationBox.top = (box.top - (slideOffset.top - pos.top) - scrollTop) / zoomFactor;
        // the width of the box
        operationBox.width = notScaleBoxSizeFlag ? box.width : Utils.round(box.width / zoomFactor, 1);
        // the height of the box
        operationBox.height = notScaleBoxSizeFlag ? box.height : Utils.round(box.height / zoomFactor, 1);

        // converting to 1/100 mm
        operationBox.left = Utils.convertLengthToHmm(operationBox.left, 'px');
        operationBox.top = Utils.convertLengthToHmm(operationBox.top, 'px');
        operationBox.width = Utils.convertLengthToHmm(operationBox.width, 'px');
        operationBox.height = Utils.convertLengthToHmm(operationBox.height, 'px');

        //workaround for Bug 52977, TODO: opBox is broken when zoomed out!!!
        if (operationBox.left < 0) { operationBox.left = 0; }

        return operationBox;
    };

    /**
     * Helper function, for extending operation with the drawing-attributes listed
     * in PresentationUtils.PLACEHOLDER_DRAWING_IMPORTANT_ATTRS.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     *
     * @param {Object} operationProperties
     *  the operation properties to generate operations. 'attrs.drawing' must be set
     */
    PresentationUtils.extendPlaceholderProp = function (app, drawingNode, operationProperties) {
        if (operationProperties && operationProperties.attrs && operationProperties.attrs.drawing) {
            var expAttrs = AttributeUtils.getExplicitAttributes(drawingNode);
            if (PresentationUtils.isPlaceHolderAttributeSet(expAttrs)) {
                var mergedDrawing = app.getModel().getDrawingStyles().getElementAttributes(drawingNode).drawing;
                _.each(mergedDrawing, function (mAttr, mKey) {

                    if (!(mKey in operationProperties.attrs.drawing) &&
                        _.contains(PresentationUtils.PLACEHOLDER_DRAWING_IMPORTANT_ATTRS, mKey) &&
                        !(mKey in expAttrs.drawing)) {
                        operationProperties.attrs.drawing[mKey] = mAttr;
                    }

                });
            }
        }
    };

    /**
     * Helper function that checks, if the specified drawing can handle the specified event.
     * This is the case, if the click happens for example into a text frame or into a range
     * that is filled by a canvas. But this is not the case, if the click happened next to
     * the filled range inside a canvas, for example next to a line.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {Event|jQuery.Event} event
     *  A browser event, expected to contain the numeric properties 'pageX'
     *  and 'pageY'.
     *
     * @param {jQuery} drawing
     *  The drawing node that is checked, if it has to handle the event.
     *
     * @param {Object} [drawingAttributes]
     *  The attributes of the family 'drawing' of the specified drawing node. For performance
     *  reasons this attributes can set as parameter, if they are already available in the
     *  calling function.
     *
     * @returns {Boolean}
     *  Whether the specified drawing has to handle the specified event.
     */
    PresentationUtils.checkEventPosition = function (app, event, drawing, drawingAttributes) {

        // whether the specified drawing can handle this event
        var handleEvent = true;
        // the tolerance, for example the distance of the click event to a line
        var radius = 6;
        // the model object
        var model = app.getModel();
        // the width of the range that is checked in the canvas node
        var rangeX = null;
        // the height of the range that is checked in the canvas node
        var rangeY = null;

        // several scenarios for fast exit
        // TODO: All nodes inside a text frame, not only text spans and paragraphs
        if (DOM.isTextSpan(event.target) || DOM.isParagraphNode(event.target)) { return handleEvent; }

        // allow moving of drawings without filling and without border, for example text frames
        if (event.target && $(event.target.parentNode).is('.borders')) { return handleEvent; }

        // place holders drawings always handle the event
        if (PresentationUtils.isPlaceHolderDrawing(drawing)) { return handleEvent; }

        // TODO: handle grouped drawings
        // if (DrawingFrame.isGroupedDrawingFrame(drawing)) { return handleEvent; }

        if (DrawingFrame.isGroupDrawingFrame(drawing)) { return !handleEvent; }

        // check only shapes, connectors and groups. All other drawings must be handled (this includes 'groups' (<- TODO))
        if (!DrawingFrame.isShapeDrawingFrame(drawing)) { return handleEvent; }

        // the canvas node inside the drawing
        var canvasNode = DrawingFrame.getCanvasNode(drawing);
        if (!canvasNode || canvasNode.length === 0) { return handleEvent; } // fast exit, drawings without canvas are always handled

        // getting explicit or merged attributes (TODO)
        var drawingAttrs = drawingAttributes || AttributeUtils.getExplicitAttributes(drawing, { family: 'drawing' });
        // the offset of the slide relative to the window (offset() can be used, because slide is never rotated)
        var slideOffset = model.getActiveSlide().offset();
        // horizontal event position relative to slide
        var eventX = 0;
        // vertical event position relative to slide
        var eventY = 0;
        // the left offset of the drawing (in px)
        var drawingLeft = 0;
        // the top offset of the drawing (in px)
        var drawingTop = 0;
        // the width of the canvas node
        var drawingWidth = 0;
        // the height of the canvas node
        var drawingHeight = 0;
        // whether the drawing is flipped horizontally
        var flipH = drawingAttrs.flipH;
        // whether the drawing is flipped vertically
        var flipV = drawingAttrs.flipV;
        // generating drawing and event position normalizing grouping and rotation
        var drawingObj = PresentationUtils.normalizeRotationAndGrouping(app, drawing, event, slideOffset);
        // a rectangle object required for rotation calculation (containing position and size of the canvas node)
        var rect = drawingObj.rect;

        // TODO: On small devices the point cannot be determined sometimes -> drawing must handle the event
        if (!drawingObj.point) { return handleEvent; }

        // the calculated drawing values including grouping and rotation
        drawingLeft = rect.left;
        drawingTop = rect.top;
        drawingWidth = rect.width;
        drawingHeight = rect.height;

        // the calculated event position including rotation
        eventX = drawingObj.point.left;
        eventY = drawingObj.point.top;

        // event position relative to drawing
        eventX -= drawingLeft;
        eventY -= drawingTop;

        if (flipH) { eventX = drawingWidth - eventX; }
        if (flipV) { eventY = drawingHeight - eventY; }

        rangeX = Math.max(eventX - radius, 0);
        rangeY = Math.max(eventY - radius, 0);

        // the context of the canvas node
        var context = canvasNode[0].getContext('2d');
        // the image data inside the specified range
        var imageData = context.getImageData(rangeX, rangeY, 2 * radius, 2 * radius);

        // do not handle the event, if all values inside imageData are 0
        handleEvent = !_.every(imageData.data, function (num) { return num === 0; });

        // Debug code to visualize the handled range
        //        if (event.type === 'mousedown') {
        //            var length = imageData.data.length;
        //            var j = 0;
        //            for (j = 0; j < length; j += 1) { imageData.data[j] = 128; } // show range
        //            context.putImageData(imageData, rangeX, rangeY);
        //        }

        return handleEvent;
    };

    /**
     * Helper function that tries to find a drawing node that can handle the specified event
     * and that is located behind a specified drawing node. The specified drawing node is for
     * example a line drawing node, that catched the event, but cannot handle it, because the
     * click was next to the line. Therefore all underlying drawings are investigated, if they
     * can handle the event. This is the case, if the event position is inside a text frame or
     * inside a canvas node with filled content.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {Event|jQuery.Event} event
     *  A browser event, expected to contain the numeric properties 'pageX'
     *  and 'pageY'.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node for that is checked, if any of its underlying drawing nodes have  to
     *  handle the event.
     *
     * @returns {Object|Null}
     *  An object containing the keys 'drawing' and 'textFrameNode'. If a drawing is found, that
     *  handles the event, the drawing node is available at the key 'drawing'. If a textframe
     *  inside this drawing can handle this event, the text frame node is saved with the key
     *  'textFrameNode'. If no drawing is found, null is returned.
     */
    PresentationUtils.findValidDrawingNodeBehindDrawing = function (app, event, drawingNode) {

        // the counter for the drawings
        var counter = 0;
        // whether a drawing was found, that can handle the event
        var foundDrawing = false;
        // whether the drawing node is a grouped drawing
        var isGroupedDrawingNode = DrawingFrame.isGroupedDrawingFrame(drawingNode);
        // the drawing node, whose parent is the slide
        var topLevelDrawing = isGroupedDrawingNode ? DrawingFrame.getGroupNode(drawingNode, { farthest: true, rootNode: app.getModel().getCurrentRootNode() }) : drawingNode;
        // all underlying drawings are the previous drawings in the DOM
        var allDrawingsBehind = topLevelDrawing.prevAll('.drawing');
        // the number of underlying drawings
        var maxCounter = allDrawingsBehind.length;
        // the drawing object as returned from 'checkEventPositionInDrawing'
        var drawingObj = null;

        // iterating over all underlying drawings
        while (counter < maxCounter && !foundDrawing) {

            // one drawing behind the specified drawing
            var oneDrawing = $(allDrawingsBehind[counter]);
            counter++; // increasing the counter for next drawing

            drawingObj = PresentationUtils.checkEventPositionInDrawing(app, event.pageX, event.pageY, oneDrawing);

            if (drawingObj) { foundDrawing = true; }
        }

        return drawingObj;
    };

    /**
     * Helper function that checks, if an event, that happened at position eventX and eventY (in px relative
     * to the slide) must be handled by the specified drawing.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {Number} eventPageX
     *  The horizontal event position in pixel like event.pageX.
     *
     * @param {Number} eventPageY
     *  The vertical event position in pixel like event.pageY.
     *
     * @param {jQuery} oneDrawing
     *  The drawing node for that is checked, if it has to handle the event.
     *
     * @returns {Object|Null}
     *  An object containing the keys 'drawing' and 'textFrameNode' and 'info'. If the specified drawing (or one
     *  of its child drawings) handles this event, the drawing node is available at the key 'drawing'. If a textframe
     *  inside this drawing can handle this event, the text frame node is saved with the key 'textFrameNode'. If
     *  'textFrameNode' is specified, an additional 'info' object is specified, that contains the calculated data
     *  for following more precise calculations (for example to calculate a precise text position).
     *  If no drawing is found, null is returned.
     */
    PresentationUtils.checkEventPositionInDrawing = function (app, eventPageX, eventPageY, oneDrawing) {

        // the model object
        var model = app.getModel();
        // the slide offset relative to the browser window (offset() can be used, because slide is not rotated)
        var slideOffset = model.getActiveSlide().offset();
        // getting explicit or merged attributes
        var drawingAttrs = AttributeUtils.getExplicitAttributes(oneDrawing, { family: 'drawing' });
        // getting the rectangle for the drawing
        var drawingObj = PresentationUtils.normalizeRotationAndGrouping(app, oneDrawing, { pageX: eventPageX, pageY: eventPageY }, slideOffset);
        // a rectangle object of the drawing (or the expanded canvas)
        var rect = drawingObj.rect;
        // a rectangle object of the drawing
        var drawingRect = drawingObj.drawingRect;
        // the horizontal event position relative to slide
        var eventX = drawingObj.point.left;
        // the vertical event position relative to slide
        var eventY = drawingObj.point.top;
        // the current zoom factor
        var zoomFactor = app.getView().getZoomFactor();
        // a text frame node inside the drawing
        var textFrameNode = null;
        // the drawing node, that can handle the specified event
        var validDrawing = null;
        // whether a drawing was found, that can handle the event
        var foundDrawing = false;
        // an info object containing position of cell or text frame and event position
        var info = null;

        // helper function to get a rectangle for the specified text frame
        function getTextFrameRectangle(textFrameNode, drawingRect, drawingAttrs) {

            // the expansion of the canvas to the left out of the drawing in px
            var textFrameLeft = Math.round(Utils.convertCssLength(textFrameNode.css('left'), 'px'));
            // the expansion of the canvas to the top out of the drawing in px
            var textFrameTop = Math.round(Utils.convertCssLength(textFrameNode.css('top'), 'px'));
            // the width of the text frame node
            var textFrameWidth = textFrameNode.width();
            // the height of the text frame node
            var textFrameHeight = textFrameNode.height();
            // whether the drawing is flipped horizontally
            var flipH = drawingAttrs.flipH;
            // whether the drawing is flipped vertically
            var flipV = drawingAttrs.flipV;

            if (flipH) { textFrameLeft = (drawingRect.width - textFrameWidth - textFrameLeft); }
            if (flipV) { textFrameTop = (drawingRect.height - textFrameHeight - textFrameTop); }

            textFrameLeft += drawingRect.left;
            textFrameTop += drawingRect.top;

            return new Rectangle(textFrameLeft, textFrameTop, textFrameWidth, textFrameHeight);
        }

        // Helper function to handle drawings inside groups
        function handleOneDrawingInGroup(oneGroupedDrawing) {

            // skipping over all internal group  drawings
            if (DrawingFrame.isGroupDrawingFrame(oneGroupedDrawing)) { return; }

            oneGroupedDrawing = $(oneGroupedDrawing);

            // a rectangle object required for rotation calculation
            var drawingObj = PresentationUtils.normalizeRotationAndGrouping(app, oneGroupedDrawing, { pageX: eventPageX, pageY: eventPageY }, slideOffset);
            // the position and size of the drawing node (including the expanded canvas)
            var canvasRect = drawingObj.rect;
            // the position and size of the drawing node (not the expanded canvas)
            var drawingRect = drawingObj.drawingRect;

            // the calculated event position including rotation
            eventX = drawingObj.point.left;
            eventY = drawingObj.point.top;

            // check, if the event happened inside the drawing node
            // getting explicit or merged attributes
            var groupedDrawingAttrs = AttributeUtils.getExplicitAttributes(oneGroupedDrawing, { family: 'drawing' });

            // check, if the event position is located inside the drawing rectangle
            if (DrawingUtils.pointInsideRect({ x: eventX, y: eventY }, canvasRect, 6)) {
                checkEventPositionInsideDrawing(oneGroupedDrawing, groupedDrawingAttrs, drawingRect);
            }
        }

        // Helper function for tables -> no grouping, no rotation
        // But the valid table cell must be determined
        function checkEventPositionInsideTable(oneDrawing) {

            // a rectangle object with the data for the table cell
            var cellRect = null;
            // the cell node, that contains the event position
            var cell = _.find(oneDrawing.find('div.cell'), function (oneCell) {

                oneCell = $(oneCell);

                var cellOffset = oneCell.offset();
                var cellLeft = (cellOffset.left - slideOffset.left) / zoomFactor;
                var cellTop = (cellOffset.top - slideOffset.top) / zoomFactor;
                cellRect = new Rectangle(cellLeft, cellTop, oneCell.width(), oneCell.height());

                return DrawingUtils.pointInsideRect({ x: eventX, y: eventY }, cellRect, 1);
            });

            if (cell) {
                validDrawing = oneDrawing;
                textFrameNode = cell; // using cell as textFrameNode -> will be used for logical position and cursor type
                foundDrawing = true;
                info = { rect: cellRect, posX: eventX, posY: eventY }; // can be used for more precise calculations later
            }

        }

        // Helper function to check, if a click (or another event) happened inside a specified drawing.
        // The event might be inside a text frame or inside the canvas content. But if the event
        // position is outside of the canvas content (and inside the drawing), the value for
        // foundDrawing will be false.
        function checkEventPositionInsideDrawing(oneDrawing, drawingAttrs, oneRect) {

            // check, if the click happened inside a text frame in the drawing
            // -> for this the function 'checkEventPosition' does not need to
            //    return true, because the drawing might have no filling
            // This test is needed for two reasons:
            // 1. 'checkEventPosition' returns false, because the drawing has no
            //    filling. But the click might have happened into an existing
            //    textFrame, so that this drawing needs to handle this event.
            // 2. 'checkEventPosition' returns true. Then it is the question, if
            //    the drawing shall be selected or the text inside the drawing.
            // In both cases the drawing must handle the event.
            if (DrawingFrame.isTextFrameShapeDrawingFrame(oneDrawing)) {

                // does the drawing contain a text frame node -> was the click inside this text frame node?
                textFrameNode = DrawingFrame.getTextFrameNode(oneDrawing);

                if (textFrameNode) {

                    // getting a rectangle object for the text frame
                    // -> using the smaller drawingRect (not rect!), so that drawing position is used and not position of expanded canvas
                    var textFrameRect = getTextFrameRectangle(textFrameNode, oneRect, drawingAttrs);

                    if (DrawingUtils.pointInsideRect({ x: eventX, y: eventY }, textFrameRect, 1)) {
                        validDrawing = oneDrawing;
                        foundDrawing = true;
                        info = { rect: textFrameRect, posX: eventX, posY: eventY }; // can be used for more precise calculations later
                    } else {
                        textFrameNode = null;
                    }
                }
            }

            // if the click did not happen inside a text frame node, the canvas content need to be checked
            if (!foundDrawing) {
                // this might be a valid drawing (but maybe this is also a line and the click was next to the shape).
                if (PresentationUtils.checkEventPosition(app, { pageX: eventPageX, pageY: eventPageY, target: null }, oneDrawing, drawingAttrs)) {
                    validDrawing = oneDrawing;
                    foundDrawing = true;
                }
            }

        }

        // check, if the event position is located inside the drawing rectangle (no further rotation required)
        if (DrawingUtils.pointInsideRect({ x: eventX, y: eventY }, rect, 6)) {
            // handling groups
            if (DrawingFrame.isGroupDrawingFrame(oneDrawing)) {
                // check all children of the group, if they can handle the event
                _.each(oneDrawing.find('.drawing'), handleOneDrawingInGroup); // collecting all drawings in all levels
            } else if (DrawingFrame.isTableDrawingFrame(oneDrawing)) {
                checkEventPositionInsideTable(oneDrawing, drawingAttrs, rect);
            } else {
                checkEventPositionInsideDrawing(oneDrawing, drawingAttrs, drawingRect);
            }
        }

        return foundDrawing ? { drawing: validDrawing, textFrameNode: textFrameNode, info: info } : null;
    };

    /**
     * A helper function that can be used to calculate the event position and the position of a
     * drawing normalized corresponding to grouping and rotation. This is typically used to
     * determine, whether the position of a specified event is located inside the specified
     * drawing.
     *
     * Info: This function should work independent from the application (tested yet only for
     *       Presentation app).
     * Info: Using jQuery offset() does not work for rotated drawings. It returns the left and
     *       top position of an unrotated rectangle generated around the rotated drawing.
     *
     * Using the properties of the returning object of this function, it is easy to check,
     * whether the point specified by 'pageX' and 'pageY' in the event object is located
     * inside the specified drawing. The function DrawingUtils.pointInsideRect() can be used
     * without handling any further rotation, grouping or expanded canvas node.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {jQuery} drawing
     *  The drawing node whose position and size is calculated and for that the event position
     *  is modified caused by rotation.
     *
     * @param {Event|jQuery.Event} [event]
     *  Optinally a browser event or another object that is expected to contain the numeric
     *  properties 'pageX' and 'pageY'. The coordinates of this point are adapted corresponding
     *  to the rotation of the drawing and all its parent group drawins.
     *
     * @param {Object} [offset]
     *  An optional offset for the positions of drawing and event. This offset object has to
     *  contain the properties 'left' and 'top' that contain the offset value in pixel. This
     *  can be used for example to calculate the positions relative to the slide in the
     *  presentation app.
     *
     * @returns {Object}
     *  An object containing the properties 'rect', 'drawingRect' and 'point'.
     *  The value of 'rect' is a rectangle object, that describes the position and the size of
     *  the drawing (for example relative to the slide), or, if it exists, the position and size
     *  of the expanded canvas node inside the drawing.
     *  The value of 'drawingRect' is a rectangle object, that describes the position and the size
     *  of the drawing (for example relative to the slide) and not of an optionally existing
     *  expanded canvas node inside the drawing. Therefore the rectangle described by
     *  'drawingRect' might be smaller than the rectangle described by 'rect'.
     *  The value of 'point' is an object containing the properties 'left' and 'top' that
     *  contain the normalized position of the event (in pixel) relative to the specified
     *  offset (for example the slide). If no event is specified, the values for 'left' and
     *  'top' are set to 0. If the point cannot be determined, null is returned for this property.
     */
    PresentationUtils.normalizeRotationAndGrouping = function (app, drawing, event, offset) {

        // the zoom factor
        var zoomFactor = app.getView().getZoomFactor();
        // whether the drawing frame is inside a group
        var isGroupedDrawing = DrawingFrame.isGroupedDrawingFrame(drawing);
        // whether a event object is specified
        var handleEventPosition = !!event;
        // the offset that is used for the event position, for example the slide offset
        var localOffset = offset ? offset : { left: 0, top: 0 };
        // the vertical event position
        var eventX = 0;
        // the horizontal event position
        var eventY = 0;
        // the left position of the group(s) in pixel
        var groupLeft = 0;
        // the top position of the group(s) in pixel
        var groupTop = 0;
        // a helper point for the event after rotation is assigned
        var point = 0;
        // whether the event contains all required information
        var isEventProblem = false;
        // getting explicit or merged attributes (TODO)
        var drawingAttrs = AttributeUtils.getExplicitAttributes(drawing, { family: 'drawing' });
        // an optional rotation angle of the drawing
        var rotationAngle = drawingAttrs.rotation;
        // whether the drawing has a canvas expansion node
        var hasCanvasExpansion = drawing.children('.canvasexpansion').length > 0;
        // a rectangle object for the drawing node
        var drawingRect = null;
        // a rectangle object for the drawing node, or, if available the expanded canvas node
        var rect = null;

        // helper function to get a rectangle for a specified drawing
        function getDrawingRectangle(oneDrawing, drawingAttrs) {

            // the left offset of the drawing (in px)
            var drawingLeft = 0;
            // the top offset of the drawing (in px)
            var drawingTop = 0;
            // whether this is a grouped drawing
            var isGroupedDrawing = DrawingFrame.isGroupedDrawingFrame(oneDrawing);

            if (!_.isNumber(drawingAttrs.left) || !_.isNumber(drawingAttrs.top)) {  // TODO: better specification
                drawingAttrs = app.getModel().getDrawingStyles().getElementAttributes(oneDrawing).drawing;
            }

            if (isGroupedDrawing) {
                // the values in the attributes cannot be used for grouped drawings
                drawingLeft = Math.round(Utils.convertCssLength(oneDrawing.css('left'), 'px'));
                drawingTop = Math.round(Utils.convertCssLength(oneDrawing.css('top'), 'px'));
            } else {
                drawingLeft = Utils.convertHmmToLength(drawingAttrs.left, 'px', 1);
                drawingTop = Utils.convertHmmToLength(drawingAttrs.top, 'px', 1);
            }

            drawingLeft += groupLeft;
            drawingTop += groupTop;

            // the width of the drawing, respectively the expanded canvas
            var drawingWidth = oneDrawing.width();
            // the height of the drawing, respectively the expanded canvas
            var drawingHeight = oneDrawing.height();

            return new Rectangle(drawingLeft, drawingTop, drawingWidth, drawingHeight);
        }

        // helper function for handling an expanded canvas inside the drawing.
        function handleExpandedCanvas(rect, drawingAttrs, canvasNode) {

            // the width of the canvas node in the drawing
            var canvasWidth = canvasNode.width();
            // the height of the canvas node in the drawing
            var canvasHeight = canvasNode.height();
            // whether the drawing is flipped horizontally
            var flipH = drawingAttrs.flipH;
            // whether the drawing is flipped vertically
            var flipV = drawingAttrs.flipV;
            // the expansion of the canvas to the left in the drawing
            var leftExpansion = 0;
            // the expansion of the canvas to the top in the drawing
            var topExpansion = 0;

            if (flipH) {
                leftExpansion = -(canvasWidth - rect.width + Math.round(Utils.convertCssLength(canvasNode.css('left'), 'px'))); // calculating right expansion
            } else {
                leftExpansion = Math.round(Utils.convertCssLength(canvasNode.css('left'), 'px')); // simply using left expansion
            }

            if (flipV) {
                topExpansion = -(canvasHeight - rect.height + Math.round(Utils.convertCssLength(canvasNode.css('top'), 'px'))); // calculating bottom expansion
            } else {
                topExpansion = Math.round(Utils.convertCssLength(canvasNode.css('top'), 'px')); // simply using top expansion
            }

            // saving position and size of the drawing (will be returned by this function)
            drawingRect = _.copy(rect);

            rect.left += leftExpansion; // handling the expansion to the left
            rect.top += topExpansion; // handling the expansion to the top
            rect.width = canvasWidth; // using canvas size instead of drawing size
            rect.height = canvasHeight; // using canvas size instead of drawing size
        }

        // helper function to resolve all groups and rotations in groups
        function handleGroupingAndRotation(drawing) {

            // a collector for all group nodes
            var groupNodeCollector = [];

            groupNodeCollector.push(DrawingFrame.getGroupNode(drawing));

            // collecting all group drawings
            while (DrawingFrame.isGroupedDrawingFrame(_.last(groupNodeCollector))) {
                groupNodeCollector.push(DrawingFrame.getGroupNode(_.last(groupNodeCollector)));
            }

            groupNodeCollector = groupNodeCollector.reverse(); // reverting the order, starting with the top level group node

            // reverting rotation from outer drawing to inner drawing
            _.each(groupNodeCollector, function (groupNode) {

                var groupAttrs = AttributeUtils.getExplicitAttributes(groupNode, { family: 'drawing' });
                var groupRect = null;

                if (handleEventPosition && _.isNumber(groupAttrs.rotation)) {
                    groupRect = getDrawingRectangle(groupNode, groupAttrs);
                    point = DrawingUtils.rotatePointWithAngle(groupRect.centerX(), groupRect.centerY(), eventX, eventY, groupAttrs.rotation); // point relative to unrotated group
                    eventX = point.left; // modifying the event position with every group
                    eventY = point.top; // modifying the event position with every group
                }

                // adding all top and left values of groups (for grouped drawings the attributes cannot be used)
                groupLeft += Math.round(Utils.convertCssLength(groupNode.css('left'), 'px'));
                groupTop += Math.round(Utils.convertCssLength(groupNode.css('top'), 'px'));
            });
        }

        // calculating the event position relative to the specified offset
        if (handleEventPosition) {
            if (!_.isNumber(event.pageX) || !_.isNumber(event.pageY)) {
                isEventProblem = true; // TODO: Happens on small devices
            } else {
                eventX = (event.pageX - localOffset.left) / zoomFactor;
                eventY = (event.pageY - localOffset.top) / zoomFactor;
            }
        }

        // handling grouped drawings
        if (isGroupedDrawing) { handleGroupingAndRotation(drawing); }

        rect = getDrawingRectangle(drawing, drawingAttrs);

        // handling an optional rotation
        if (handleEventPosition && _.isNumber(rotationAngle) && rotationAngle !== 0) {
            // using the drawing node as rectangle to calculate rotation, not the canvas node, even if canvas expansion exists
            point = DrawingUtils.rotatePointWithAngle(rect.centerX(), rect.centerY(), eventX, eventY, rotationAngle);
            eventX = point.left;
            eventY = point.top;
        }

        // using the data of the canvas expansion to calculate position and size, if drawing is not large enough
        if (hasCanvasExpansion) { handleExpandedCanvas(rect, drawingAttrs, DrawingFrame.getCanvasNode(drawing)); }

        return { rect: rect, drawingRect: drawingRect ? drawingRect : rect, point: (isEventProblem ? null : { left: eventX, top: eventY }) };
    };

    /**
     * A helper function that tries to calculate the best logical text position for an event
     * position, that is inside a specified table cell or text frame.
     *
     * @param {Application} app
     *  The application containing this instance.
     *
     * @param {jQuery} drawing
     *  The drawing node that contains the specified text frame or table cell node.
     *
     * @param {HtmlElement|jQuery} drawing
     *  The text frame node or table cell, inside which the text position shall be calculated.
     *
     * @param {Object} [infoObject]
     *  The info object as returned by the function 'PresentationUtils.checkEventPositionInDrawing'.
     *  It contains the rectangle data for the cell or the text frame node and the calculated
     *  event position.
     *  info = { rect: textFrameRect, posX: eventX, posY: eventY };
     *
     * @returns {Number[]|Null}
     *  The calculated logical position or null, if it could not be determined.
     */
    PresentationUtils.getBackgroundTextPosition = function (app, drawing, textFrameNode, infoObject) {

        // the calculated logical text position
        var pos = null;
        // the current root node
        var activeRootNode = app.getModel().getCurrentRootNode();

        textFrameNode = $(textFrameNode);

        if ((textFrameNode).parent().is('.emptytextframe')) {
            pos = Position.getLastTextPositionInTextFrame(activeRootNode, drawing);
        } else {
            var allParagraphs = textFrameNode.find('div.p');
            var nodeOffset = textFrameNode.offset();

            _.each(allParagraphs, function (oneParagraph) {
                if (pos) { return; }
                var paraOffset = $(oneParagraph).offset();
                var paraX = (paraOffset.left - nodeOffset.left) / app.getView().getZoomFactor();
                var paraY = (paraOffset.top - nodeOffset.top) / app.getView().getZoomFactor();
                var posX = infoObject.posX - infoObject.rect.left - paraX;
                var posY = infoObject.posY - infoObject.rect.top - paraY;
                var point = Position.getPositionInsideParagraph(activeRootNode, oneParagraph, posX, posY, app.getView().getZoomFactor() * 100);
                if (point && point.pos) { pos = point.pos; }
            });
        }

        // default is last position in text frame or table cell
        pos = pos || (DrawingFrame.isTableDrawingFrame(drawing) ? Position.getLastTextPositionInTableCell(activeRootNode, textFrameNode) : Position.getLastTextPositionInTextFrame(activeRootNode, drawing));

        return pos;
    };

    // exports ================================================================

    return PresentationUtils;

});
