/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/drawingstyles', [
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/model/drawingstylecollection',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'gettext!io.ox/office/presentation/main'
], function (PresentationUtils, Forms, Utils, AttributeUtils, DrawingStyleCollection, DrawingFrame, DOM, gt) {

    'use strict';

    var parseInt = window.parseInt;

    // definitions for presentation-specific drawing attributes
    var DEFINITIONS = {

        /**
         * Whether the drawing can be grouped or not.
         */
        noGroup: { def: false },

        /**
         * Drawing rotation
         */
        rotation: { def: 0 }
    };

    // class PresentationDrawingStyles ========================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends DrawingStyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    var PresentationDrawingStyles = DrawingStyleCollection.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // collector for the place holder attributes in layout or master slides
        var placeHolderCollector = {};

        // special behavior for ODF documents
        var odf = docModel.getApp().isODF();

        // base constructor ---------------------------------------------------

        DrawingStyleCollection.call(this, docModel, {
            families: 'presentation table changes',
            baseAttributesResolver: resolveLayoutAttributes
        });

        // private methods ----------------------------------------------------

        /**
         * Setting the width of the drawing.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingWidth(drawing, attributes) {

            if (!DrawingFrame.isGroupedDrawingFrame(drawing)) {
                var
                    width = attributes.drawing.width;

                // bugfix :: BUG#45141 :: https://bugs.open-xchange.com/show_bug.cgi?id=45141 :: Vertical drawing object is shown diagonally
                if (_.isString(width)) {
                    var
                        value = parseInt(width, 10),
                        tuple = width.split(value),
                        unit  = (tuple.length === 2) ? $.trim(tuple[1]) : '';

                    if (unit === '%') {
                        drawing.width([value, unit].join(''));
                    }/* else if (unit !== '') {
                     drawing.width(Utils.convertLength(value, unit, 'px', 1));
                     }*/
                } else {
                    drawing.width(Utils.convertHmmToLength(width, 'px', 1));
                }
            }
        }

        /**
         * Setting the height of the drawing. This needs to be done, because it might be
         * dependent from other attribute values.
         *
         * Info: Height must be set to drawing with autoResizeHeight set to true, if the
         *       text is vertically oriented inside the drawing (55562)
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} attributes
         *  A map with the merged drawing attributes, as name/value pairs.
         */
        function setDrawingHeight(drawing, attributes) {

            var drawingHeight = 0;

            if (!(DrawingFrame.isGroupedDrawingFrame(drawing) || DrawingFrame.isTableDrawingFrame(drawing) || DrawingFrame.isAutoResizeHeightDrawingFrameWithHorizontalText(drawing, attributes.shape))) {
                drawingHeight = Math.max(Utils.convertHmmToLength(attributes.drawing.height, 'px', 1), 1);  // at least 1px, even for lines and connectors
                drawing.height(drawingHeight);
            } else if (DrawingFrame.isTableDrawingFrame(drawing) || DrawingFrame.isAutoResizeHeightDrawingFrameWithHorizontalText(drawing, attributes.shape)) {
                // -> no setting of height for grouped drawings, tables and for shapes with property 'autoResizeHeight'
                drawing.height(''); // if 'autoResizeHeight' is set, an explicit height must be removed
            }
        }

        /**
         * Helper function to update attributes of children of drawings. This is only used for text frames yet.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node as jQuery object, whose children need an update of element formatting.
         */
        function handleChildAttributes(drawing) {

            var // the explicit drawing attributes
                explicitDrawingAttrs = AttributeUtils.getExplicitAttributes(drawing),
                // the paragraph style sheet container
                paragraphStyles = null,
                // the container for the paragraphs and tables inside the drawing
                textFrameNode = null;

            // check, if the empty text frame needs an artificial border and a default text
            self.handleEmptyPlaceHolderDrawing(drawing, explicitDrawingAttrs);

            // checking the fill color (task 36385)
            if (explicitDrawingAttrs && (explicitDrawingAttrs.fill || explicitDrawingAttrs.character || explicitDrawingAttrs.paragraph)) {
                textFrameNode = DrawingFrame.getTextFrameNode(drawing);

                if (textFrameNode.length > 0) {
                    paragraphStyles = docModel.getStyleCollection('paragraph');
                    Utils.iterateSelectedDescendantNodes(textFrameNode, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                        paragraphStyles.updateElementFormatting(paragraph);
                    }, undefined, { children: true });
                }
            }
        }

        /**
         * Update the formatting of the table node inside the specified drawing.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node as jQuery object, whose table needs an update of formatting.
         */
        function handleTableAttributes(drawing) {

            var tableNode = $(drawing).children('.content:not(.copy)').find('table');

            if (tableNode.length > 0) {
                docModel.getTableStyles().updateElementFormatting(tableNode);
            }
        }

        /**
         * Checking the existence of place holder drawings on a specified slide.
         *
         * @param {String} slideId
         *  The ID of the slide, on that the drawing nodes will be searched.
         *
         * @param {Object} placeHolderSet
         *  An object containing the saved place holder attributes for the specified slide.
         *  This object might be modified within this function.
         */
        function removeNonExistentPlaceHolderDrawings(slideId, placeHolderSet) {

            var ignoreList = ['dt', 'ftr', 'sldNum']; // drawing types to be ignored

            _.each(_.keys(placeHolderSet), function (type) {
                if (!_.contains(ignoreList, type)) {
                    _.each(_.keys(placeHolderSet[type]), function (index) {
                        var drawings = self.getAllPlaceHolderDrawingsOnSlideBySelector(slideId, type, index);
                        if (drawings.length === 0) {
                            delete placeHolderSet[type][index];
                            if (_.keys(placeHolderSet[type]).length === 0) { delete placeHolderSet[type]; }
                        }
                    });
                }
            });
        }

        /**
         * If the layout of a slide is changed, it is allowed, that a content place holder drawing (that
         * allows input of text, table, picture, ...) that already contains text, is replaced by a text
         * body place holder (52805).
         *
         * @param {String} slideId
         *  The ID of the slide, on that the drawing nodes will be searched.
         *
         * @param {Object} set1
         *  An object containing the saved place holder attributes for the specified slide.
         *  This object might be modified within this function.
         *
         * @param {Object} set2
         *  An object containing the place holder attributes for the new assigned slide layout.
         */
        function checkContentBodyPlaceHolderDrawings(slideId, set1, set2) {

            // a container with all keys of set1.contentbody
            var contentbodyKeys = null;
            // the length of the contentbodyKeys object
            var numberOfContentBodies = 0;
            // the number of 'body' place holder drawings in set2
            var numberOfBodies = 0;
            // the maximum possible number of conversions in this function
            var conversionMax = 0;
            // the number of already converted drawings
            var convertedDrawings = 0;
            // the number of already checked drawings
            var checkedDrawings = 0;

            // modifying the 'set1' from 'contentbody' to 'body' if possible
            function checkBodyDrawing(index) {

                // converting only content bodies, that are filled with text
                var oneDrawing = self.getAllPlaceHolderDrawingsOnSlideBySelector(slideId, 'body', index);

                if (oneDrawing.length === 1) {
                    // check, if the drawing contains text and is not empty and is no table
                    if (DrawingFrame.isTextFrameShapeDrawingFrame(oneDrawing) && !DOM.isEmptyTextframe(oneDrawing, { ignoreTemplateText: true }) && !DrawingFrame.isTableDrawingFrame(oneDrawing)) {
                        set1.body = set1.body || {};
                        set1.body[index] = set1.contentbody[index];
                        delete set1.contentbody[index];
                        if (_.isEmpty(set1.contentbody)) { delete set1.contentbody; }
                        convertedDrawings++;
                    }
                }
            }

            if (set1.contentbody && !set1.body && !set2.contentbody && set2.body) {
                // 'contentbody' can be handled like (text) 'body', if it already contains text content
                contentbodyKeys = _.keys(set1.contentbody);
                numberOfContentBodies = contentbodyKeys.length;
                numberOfBodies = _.keys(set2.body).length;
                conversionMax = Math.min(numberOfContentBodies, numberOfBodies);

                while ((checkedDrawings < numberOfContentBodies) && (convertedDrawings < conversionMax)) {
                    _.each(contentbodyKeys, checkBodyDrawing);
                    checkedDrawings++;
                }
            }

        }

        /**
         * Marks drawing objects that are placeholders with specific element
         * attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node, as jQuery object.
         */
        function assignPlaceholderAttributes(drawing) {

            // the explicit attributes at the specified element
            // for place holder type and place holder index only the explicit attributes
            // must be evaluated. Otherwise every drawing will become a place holder drawing,
            // because the merge attributes contain the default values.
            // -> marking presentation placeholder drawings
            var elemAttrs = AttributeUtils.getExplicitAttributes(drawing, { direct: true });

            var keySet = PresentationUtils.isPlaceHolderAttributeSet(elemAttrs) ? PresentationUtils.getValidTypeIndexSet(elemAttrs) : null;

            drawing.attr({
                placeholdertype: keySet ? keySet.type : null,
                placeholderindex: keySet ? keySet.index : null
            });
        }

        /**
         * Handling drawing that are absolute positioned. This are all drawings that are not in-line
         * with text. This is especially important, if the drawing anchor was modified. In this case
         * some clean up at the node is required.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map with the merged drawing attributes, as name/value pairs.
         *
         * @param {Number} borderWidth
         *  The width of the drawing of the border in 1/100 mm.
         *
         * @param {Number} halfBorderWidth
         *  The half width of the drawings border in 1/100 mm.
         *
         * @param {Number} zoomFactor
         *  The current zoom factor in percent.
         */
        function assignDrawingAttributes(drawing, mergedAttributes) {

            // the drawing attributes of the passed attribute map
            var drawingAttrs = mergedAttributes.drawing;

            // drawing is absolutely positioned
            drawing.addClass('absolute');

            // for place holder type and place holder index only the explicit attributes
            // must be evaluated. Otherwise every drawing will become a place holder drawing,
            // because the merge attributes contain the default values.
            // -> marking presentation placeholder drawings
            assignPlaceholderAttributes(drawing);

            // apply CSS formatting to drawing node
            drawing.css({
                left: Utils.convertHmmToLength(drawingAttrs.left, 'px', 1),
                top: Utils.convertHmmToLength(drawingAttrs.top, 'px', 1)
            });

            // special behavior for special hidden drawings in PP (55595, 55741)
            // -> not toggling 'display' directly, because 'display: block' at the drawing node might overwrite layout and master slide visibility
            if (!odf) { drawing.toggleClass('forcehiddenfile', drawingAttrs.left === 0 && drawingAttrs.top === 0 && drawingAttrs.width === 0 && drawingAttrs.height === 0); }
        }

        /**
         * Adding attributes of master and layout slide into the merge attributes. This has to happen
         * before the explicit attributes are set into the specified attribute set.
         *
         * @param {jQuery} drawing
         *  The drawing node as jQuery object.
         */
        function resolveLayoutAttributes(drawing) {

            var // the attribute set of the place holder elements in layout and/or master slide
                placeHolderAttributes = null,
                // the explicit attributes at the specified element
                elemAttrs = AttributeUtils.getExplicitAttributes(drawing),
                // the IDs of document slide, layout slide and master slide
                id = null, layoutId = null, masterId = null,
                // an object containing the place holder type and index as property
                keySet = null;

            if (PresentationUtils.isPlaceHolderAttributeSet(elemAttrs)) {

                id = DOM.getTargetContainerId(drawing.parent()); // checking only direct children of slide
                keySet = PresentationUtils.getValidTypeIndexSet(elemAttrs);

                if (keySet) {
                    if (docModel.isStandardSlideId(id)) {
                        layoutId = docModel.getLayoutSlideId(id);
                        masterId = docModel.getMasterSlideId(layoutId);
                        placeHolderAttributes = getMergedPlaceHolderAttributes(layoutId, masterId, keySet.type, keySet.index);

                    } else if (docModel.isLayoutSlideId(id)) {
                        masterId = docModel.getMasterSlideId(id);
                        placeHolderAttributes = getMergedPlaceHolderAttributes(null, masterId, keySet.type, keySet.index);
                    }
                }
            }

            return placeHolderAttributes;
        }

        /**
         * Will be called for every drawing node whose attributes have been changed. Repositions
         * and reformats the drawing according to the passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family, containing the
         *  effective attribute values merged from style sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            // No formatting is done during loading the document. But in Presentation, invisible slides might be
            // formatted much later (or not at all). But some minimum formatting might be necessary, to work with
            // unformatted slides. This is for example the 'type' and the 'index' of place holder drawings. Only
            // if this is set correctly, the selectors that are used to find drawings on slides, work reliable.
            if (!self.isImportFinished) {
                assignPlaceholderAttributes(drawing);
                return;
            }

            // the attributes of the generic 'drawing' family
            var drawingAttrs = mergedAttributes.drawing;

            // update CSS class for drawings that are not groupable
            drawing.toggleClass(DrawingFrame.NO_GROUP_CLASS, drawingAttrs.noGroup);

            // rotation
            var degrees = drawingAttrs.rotation || 0;
            DrawingFrame.updateCssTransform(drawing, degrees);
            drawing.toggleClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME, degrees !== 0);

            // nothing to do, if paragraph is a group content object
            // -> in this case the group is responsible for the correct attributes
            if (DrawingFrame.isGroupContentNode(drawing.parent())) {
                drawing.addClass('absolute grouped');  // absolute positioned drawing inside its group!
                // do NOT set generic formatting to the drawing frame -> this is handled by the drawing group node!
                // -> avoid to call: DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes); (Performance of groups!)
                // The slideFormatManager takes care of order of grouped drawings (-> the group node is formatted as last node)
                // So if this formatting is not triggered by slideFormatManager, it is necessary that the group is formatted after the grouped node (55849)
                // -> but this should happen only once for all grouped drawings inside a group -> debouncing the call
                if (!drawing.hasClass('slideformat')) { docModel.updateDrawingsDebounced(DrawingFrame.getGroupNode(drawing)); }

                // Grouped images have to be formatted completely, at least once after loading document (with operations)
                // or after pasting a group that contains image drawings or duplicating a slide (56742).
                // In drawingFrame.js 'updateImageAttributesInGroup' is called instead of 'updateImageFormatting' for
                // grouped drawings and this 'short' version does not handle all image properties.
                // The formatting of all other drawings has to be started by their group node and not here for performance
                // reasons.
                // Grouped tables (ODP only) also need to be formatted once completely (58537).
                if (!drawing.hasClass('formatrequired')) {
                    return; // Performance: This should be the default (only grouped images must not return here)
                }
            }

            // setting the drawing height
            setDrawingHeight(drawing, mergedAttributes);

            // setting the drawing width
            setDrawingWidth(drawing, mergedAttributes);

            // setting the drawing attributes except width and height (those are already set)
            assignDrawingAttributes(drawing, mergedAttributes);

            // set generic formatting to the drawing frame
            DrawingFrame.updateFormatting(docModel.getApp(), drawing, mergedAttributes);

            // removed marker class for required formatting after formatting is done
            drawing.removeClass('formatrequired');

            // update formatting of all paragraphs inside the text frame content node (36385)
            if (DrawingFrame.isTableDrawingFrame(drawing)) {
                handleTableAttributes(drawing);
            } else if (DrawingFrame.isTextFrameShapeDrawingFrame(drawing) || (PresentationUtils.isPlaceHolderWithoutTextDrawing(drawing) && PresentationUtils.isEmptyPlaceHolderDrawing(drawing))) {
                // update formatting of all paragraphs inside the text frame content node (36385)
                handleChildAttributes(drawing);
            } else if (odf && DrawingFrame.isShapeDrawingFrame(drawing) && !PresentationUtils.isPlaceHolderWithoutTextAttributeSet(mergedAttributes)) {
                handleChildAttributes(drawing);
            }

        }

        /**
         * Saving the specified drawing attributes in the model container.
         *
         * This function can be called directly from insertDrawing operation. In that
         * case the attributes already contain the full set of properties from the
         * families 'drawing' and 'presentation'.
         *
         * @param {String} target
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Number} index
         *  The place holder index.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target, type or index are not defined, null is returned.
         */
        function savePlaceHolderAttributes(target, type, index, attrs, options) {

            var // whether list style handling is required
                listStyle = attrs && attrs.listStyle,
                // the level specified in the list styles
                level = null,
                // a helper object for merging list style attributes
                drawingAttrs = null,
                // the old place holder attributes
                oldAttrs = null,
                // whether an update of dependent slide shall be triggered immediately
                saveOldAttrs = Utils.getBooleanOption(options, 'saveOldAttrs', false);

            if (!(target && type && _.isNumber(index))) { return null; }

            placeHolderCollector[target] = placeHolderCollector[target] || {};
            placeHolderCollector[target][type] = placeHolderCollector[target][type] || {};
            placeHolderCollector[target][type][index] = placeHolderCollector[target][type][index] || {};

            // optionally saving a deep copy of the old value in the return object for later usage
            if (saveOldAttrs) { oldAttrs = _.copy(placeHolderCollector[target][type][index], true); }

            // merging the attributes, that are not list styles ('autoclear' is required for undo, so that attributes can be removed)
            self.extendAttributeSet(placeHolderCollector[target][type][index], attrs, { autoClear: true });

            // special handling for list style attribute at drawings
            if (listStyle) {
                for (level in listStyle) {
                    drawingAttrs = placeHolderCollector[target][type][index];
                    drawingAttrs.listStyle = drawingAttrs.listStyle || {};
                    drawingAttrs.listStyle[level] = drawingAttrs.listStyle[level] || {};
                    self.extendAttributeSet(drawingAttrs.listStyle[level], listStyle[level], { autoClear: true });
                }
            }

            // returning an object with the complete registration information
            return { target: target, placeHolderType: type, placeHolderIndex: index, attrs: attrs, oldAttrs: oldAttrs };
        }

        /**
         * Getting the merged place holder attributes from master slide and layout slide.
         *
         * @param {String} [layoutId]
         *  The id of the layout slide.
         *
         * @param {String} [masterId]
         *  The id of the master slide.
         *
         * @param {String} placeHoderType
         *  The drawing place holder type.
         *
         * @param {Number} placeHoderIndex
         *  The drawing place holder index.
         *
         * @returns {Object|Null}
         *  An object containing the attributes from the layout slide and the master slide. If attributes are defined
         *  at both slide, the layout slide overwrites the values from the master slide.
         */
        function getMergedPlaceHolderAttributes(layoutId, masterId, placeHolderType, placeHolderIndex) {

            var // the layout slide attributes
                layoutAttrs = layoutId ? self.getPlaceHolderAttributes(layoutId, placeHolderType, placeHolderIndex) : null,
                // the master slide attributes
                masterAttrs = masterId ? self.getPlaceHolderAttributes(masterId, placeHolderType, placeHolderIndex) : null,
                // the attributes of layout and master slide
                attributes = null;

            // merging the collected attributes from master and layout slides and drawings (never overwrite attribute set)
            if (masterAttrs) { attributes = self.extendAttributeSet({}, masterAttrs); }
            if (layoutAttrs) { attributes = self.extendAttributeSet(attributes || {}, layoutAttrs); }

            return attributes;
        }

        /**
         * Handler function for the event 'removed:slide'. This event is triggered from the model
         * after a slide is removed. Within this function the place holder information specified
         * for the master or layout slides will be removed, if a master or layout slide was
         * removed.
         *
         * @param {jQuery.Event} event
         *  The unused jQuery event object.
         *
         * @param {Object} eventOptions
         *  The information sent, if a 'removed:slide' event is triggered by the model. This object
         *  contains the ID of the removed slide, whether the master view is active and whether the
         *  removed slide is a master or layout slide.
         */
        function removeFromPlaceholderCollector(evt, eventOptions) {

            var // the ID of a master or layout slide (or null, if a document slide was removed)
                target = eventOptions.isMasterOrLayoutSlide ? eventOptions.id : null;

            if (target && placeHolderCollector[target]) {
                delete placeHolderCollector[target]; // removing all place holder info from model
            }
        }

        /**
         * Handling the visualization of empty text frames without explicit border. In the
         * standard view it is necessary to assign a dotted line to these empty text frames.
         * In the layout or master view, such a dotted line is always assigned to the place
         * holder drawings.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Boolean} isEmptyPlaceHolderWithoutText
         *  Whether the drawing is an empty place holder in which no text can be inserted (table, image, ...)
         *
         * @param {Object} [explicitDrawingAttrs]
         *  The (explicit) attributes of the drawing node. If they are not specified,
         *  the explicit attributes at the specified drawing node are evaluated.
         */
        function handleBorderForEmptyTextFrames(drawing, isEmptyPlaceHolderWithoutText, explicitDrawingAttrs) {
            var setClass = isEmptyPlaceHolderWithoutText || DOM.isEmptyTextframeWithoutBorder(drawing, explicitDrawingAttrs, { ignoreTemplateText: true });
            // set class DrawingFrame.EMPTYTEXTFRAME_CLASS at empty text frames without border
            $(drawing).children('.content').toggleClass(DrawingFrame.EMPTYTEXTFRAME_CLASS, setClass);
            // setting 'ignoreTemplateText' to true, so that an already inserted template text does not 'fill' the drawing.
            // -> it must still be empty, if the drawing is formatted twice.
        }

        /**
         * Handling a default text that is inserted into empty place holder drawings.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Object} [options]
         *  Some additional options that are supported by the model function 'getDefaultTextForTextFrame'.
         *  Additionally supported:
         *  @param {Boolean} [options.ignoreTemplateText=false]
         *      This option is handled within DOM.isEmptyTextFrame to define, that text frames
         *      that contain template text are also empty.
         *
         * @returns {Boolean}
         *  Whether the default template text was inserted into the drawing.
         */
        function handleTemplateText(drawing, options) {
            return docModel.addTemplateTextIntoTextSpan(drawing, options);
        }

        /**
         * ODP only.
         * Adding a paragraph into empty place holder drawings. After loading they do not contain
         * a paragraph, if they contain no content.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node.
         */
        function handleMissingParagraphInPlaceHolderDrawing(drawing) {

            var // the implicit paragraph node
                implicitParagraph = null,
                // the textframe node
                textFrameNode = null,
                // whether an update of drawing is required
                drawingUpdateRequired = false;

            if ($(drawing).find('div.p').length === 0) {

                textFrameNode = DrawingFrame.getTextFrameNode(drawing); // during loading the text frame already exists (and was already formatted, so it must be reused)

                if (textFrameNode.length === 0) { // create new text frame node required -> updateDrawingFormatting is also required (should happen only when loading document with operations)
                    textFrameNode = DrawingFrame.prepareDrawingFrameForTextInsertion(drawing);
                    drawingUpdateRequired = true;
                }

                if (textFrameNode && textFrameNode.length > 0) {
                    implicitParagraph = docModel.getValidImplicitParagraphNode();
                    // no bullets allowed for implicit paragraphs title place holder drawings
                    if (odf && PresentationUtils.isTitleOrSubtitlePlaceHolderDrawing(drawing)) {
                        // adding paragraph attributes without operation (because it is an implicit paragraph) -> no inherited bullets in titles
                        docModel.getParagraphStyles().setElementAttributes(implicitParagraph, { paragraph: { bullet: { type: 'none' }, indentLeft: 0, indentFirstLine: 0 } });
                    }
                    // inserting an implicit paragraph into the drawing
                    textFrameNode.append(implicitParagraph);
                    docModel.getParagraphStyles().updateElementFormatting(implicitParagraph);
                    // taking care of implicit paragraphs
                    if (odf) { docModel.handleTriggeringListUpdate(implicitParagraph); }
                    // starting drawing formatting if required (should only be necessary after load with operations)
                    if (drawingUpdateRequired) { docModel.getDrawingStyles().updateElementFormatting(drawing); }
                }
            }
        }

        /**
         * Helper function, that adds the place holder text from layout drawings for place holder
         * types, that cannot contain text. This are tables, images, ... .
         * There is no paragraph in the document slide, that can contain the template text. Therefore
         * the complete content from the drawing node in the corresponding layout slide is cloned.
         * During loading the document it happens, that the document slide is formatted before its
         * layout slide (which directly follows the document slide). Therefore it might be necessary
         * to try to get the template text later again.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node.
         *
         * @param {String} type
         *  The type of the place holder drawing.
         *
         * @param {Number|Null} index
         *  The index of the place holder drawing.
         *
         * @param {String} slideId
         *  The ID of the slide that contains the drawing.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.update=false]
         *      Whether an existing old text shall be removed. If this option is set to true and
         *      there is no old content, this function returns without any modification. This
         *      guarantees, that only empty place holder drawings get the template text.
         */
        function addTemplateReadOnlyText(drawing, type, index, slideId, options) {

            // the container for the read-only inserted text span for tables, images, ...
            var templateReadOnlyText = null;
            // the id of the layout slide
            var layoutId = docModel.getLayoutSlideId(slideId);
            // the corresponding drawing on the layout slide
            var layoutDrawing = self.getAllPlaceHolderDrawingsOnSlideBySelector(layoutId, type, index);
            // whether an existing text shall be replaced
            var update = Utils.getBooleanOption(options, 'update', false);
            // an already existing template text
            var oldTemplateText = update ? $(drawing).children(PresentationDrawingStyles.TEMPLATE_READONLY_TEXT_CONTAINER_SELECTOR) : null;

            function registerLaterCall() {
                docModel.executeDelayed(function () { // try later again, because the layout slide is formatted now
                    addTemplateReadOnlyText(drawing, type, index, slideId);
                }, 'DrawingStyles.addTemplateReadOnlyText', 200);
            }

            if (update && oldTemplateText.length === 0) { return; } // do nothing, if update is true, but there is no old text

            // getting the text from the layout slide
            if (layoutDrawing.length > 0) {
                templateReadOnlyText = $('<div>', { contenteditable: false }).addClass(PresentationDrawingStyles.TEMPLATE_READONLY_TEXT_CONTAINER_CLASS);
                templateReadOnlyText.append(layoutDrawing.find('div.p').clone(true));
                if (update) { oldTemplateText.remove(); } // remove existing old text
                $(drawing).append(templateReadOnlyText);
            } else if (docModel.getSlideFormatManager().isUnformattedSlide(layoutId)) {  // the layout slide formatting has not finished yet
                registerLaterCall();
            }
        }

        /**
         * Inserting the container for the template images into a specified drawing.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node.
         *
         * @param {String} type
         *  The type of the place holder drawing (from the explicit drawing attributes).
         *
         * @param {Number|Null} index
         *  The index of the place holder drawing (from the explicit drawing attributes).
         *
         * @param {String} slideId
         *  The ID of the slide that contains the drawing.
         */
        function insertImageTemplateContainer(drawing, type, index, slideId) {

            // the container node for the template images
            var imageContainerNode = null;
            // the list with the required buttons for the specified type
            var buttonList = PresentationUtils.getPlaceHolderTemplateButtonsList(type, odf);
            // whether some text also needs to be added. This is the case for place holder drawings
            // without text input. In this case the text from the layout slide needs to be used.
            var addText = (type in PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE);

            if (buttonList) {

                imageContainerNode = $('<div>').addClass(PresentationDrawingStyles.TEMPLATE_DRAWING_CONTAINER_CLASS);

                _.each(buttonList, function (buttonString) {

                    var // the button node
                       buttonNode = null;

                    // calculate the CSS fill attributes
                    switch (buttonString) {
                        case 'table':
                            buttonNode = $('<a tabindex="0">' + Forms.createIconMarkup('fa-table') + '</a>');
                            buttonNode.addClass(DOM.TABLE_TEMPLATE_BUTTON_CLASS);
                            Forms.setToolTip(buttonNode, gt('Insert table'));
                            break;
                        case 'picture':
                            buttonNode = $('<a tabindex="0">' + Forms.createIconMarkup('fa-image') + '</a>');
                            buttonNode.addClass(DOM.IMAGE_TEMPLATE_BUTTON_CLASS);
                            Forms.setToolTip(buttonNode, gt('Insert image'));
                            break;
                        case 'media':
                            buttonNode = $('<a tabindex="0">' + Forms.createIconMarkup('fa-film') + '</a>');
                            Forms.setToolTip(buttonNode, gt('Insert media clip'));
                            break;
                        case 'chart':
                            buttonNode = $('<a tabindex="0">' + Forms.createIconMarkup('fa-bar-chart') + '</a>');
                            Forms.setToolTip(buttonNode, gt('Insert chart'));
                            break;
                        case 'clipart':
                            buttonNode = $('<a tabindex="0">' + Forms.createIconMarkup('fa-globe') + '</a>');
                            Forms.setToolTip(buttonNode, gt('Insert online image'));
                            break;
                        case 'smartart':
                            buttonNode = $('<a tabindex="0">' + Forms.createIconMarkup('fa-list-alt') + '</a>');
                            Forms.setToolTip(buttonNode, gt('Insert smart art graphic'));
                            break;
                        default:
                            Utils.warn('DrawingStyles.insertImageTemplateContainer(): unknown button type: ' + buttonString);
                    }

                    if (buttonNode) {
                        buttonNode.addClass(DOM.PLACEHOLDER_TEMPLATE_BUTTON_CLASS);
                        imageContainerNode.append(buttonNode);
                    }
                });

                $(drawing).append(imageContainerNode);

                if (addText && !docModel.isLayoutOrMasterId(slideId)) { addTemplateReadOnlyText(drawing, type, index, slideId); }

                // centering the image container node in the drawing
                imageContainerNode.css('left', (0.5 * ($(drawing).width() - imageContainerNode.width())) + 'px');
                imageContainerNode.css('top', (0.5 * ($(drawing).height() - imageContainerNode.height())) + 'px');
            }
        }

        /**
         * This function modifies the return object from the function 'self.getAllPlaceHolderAttributesForId(firstId)'
         * in that way, that it can be differentiated between body place holders that are text body place holders (type
         * 'body' is explicitely set) and content body place holders ('type' not set).
         *
         * @param {Object} set
         *  The object from function 'self.getAllPlaceHolderAttributesForId(firstId)', in which the key 'body' collects
         *  all place holders that can contain text. This are text place holders and content place holders.
         *
         * @returns {Object}
         *  An object, in which text content place holders and content body place holders are separated. Content body
         *  place holder are stored under the key 'contentbody' (PresentationUtils.CONTENTBODY).
         */
        function handleDifferentBodySet(set) {

            var newSet = _.isObject(set) ? {} : null;
            var allIndizes = null;

            _.each(_.keys(set), function (type) {
                if (type === 'body') {
                    allIndizes = _.keys(set[type]);
                    _.each(allIndizes, function (index) {
                        var attrs = set[type][index];
                        if (PresentationUtils.isContentBodyPlaceHolderAttributeSet(attrs)) {
                            newSet[PresentationUtils.CONTENTBODY] = newSet[PresentationUtils.CONTENTBODY] || {};
                            newSet[PresentationUtils.CONTENTBODY][index] = attrs;
                        } else {
                            newSet.body = newSet.body || {};
                            newSet.body[index] = attrs;
                        }
                    });
                } else {
                    newSet[type] = set[type];
                }
            });

            return newSet;
        }

        /**
         * Updating the position of an already existing image container node. This might
         * be necessary after a change of layout.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node.
         *
         * @param {jQuery} imageContainerNode
         *  The image container node.
         */
        function updateImageContainerNode(drawing, imageContainerNode) {
            // centering the image container node in the drawing
            imageContainerNode.css('left', (0.5 * ($(drawing).width() - imageContainerNode.width())) + 'px');
            imageContainerNode.css('top', (0.5 * ($(drawing).height() - imageContainerNode.height())) + 'px');
        }

        /**
         * Removing the container for the template images from a specified drawing.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node.
         */
        function removeImageTemplateContainer(drawing) {
            // Removing the container node(s) for the template images and the (read-only) template text
            $(drawing).children(PresentationDrawingStyles.TEMPLATE_DRAWING_CONTAINER_SELECTOR + ', ' + PresentationDrawingStyles.TEMPLATE_READONLY_TEXT_CONTAINER_SELECTOR).remove();
        }

        /**
         * Emptying the complete place holder model. This is necessary, if a new snapshot is applied.
         */
        function emptyPlaceHolderModel() {
            placeHolderCollector = {};
        }

        // public methods -----------------------------------------------------

        /**
         * Saving the drawing attributes in a container, if the drawing is a place holder
         * element.
         *
         * @param {String} target
         *  The id of the layout or master slide, that defines the attributes for
         *  the presentation place holder type.
         *
         * @param {Object} attrs
         *  The attributes object, that contains the attributes for the specified
         *  target and place holder type. This are the explicit attributes set at
         *  the drawing.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node, that gets the specified attributes assigned.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.saveOldAttrs=false]
         *      Whether the old attributes shall be saved and returned in the return object.
         *
         * @returns {Object|Null}
         *  If the attributes are registered, an object with the complete registration
         *  information is returned. The object's keys are: 'target', 'placeHolderType',
         *  'placeHolderIndex', 'attrs' and 'oldAttrs'. If the attributes were not registered,
         *  because target is not set (no master or layout slide) or the attributes
         *  contain no place holder information (no place holder drawing), null is
         *  returned.
         */
        this.savePlaceHolderAttributes = function (target, attrs, drawing, options) {

            if (!target || !PresentationUtils.isPlaceHolderDrawing(drawing)) { return null; }

            var keySet = PresentationUtils.getValidTypeIndexSet(AttributeUtils.getExplicitAttributes(drawing, { direct: true }));

            // target must be defined and the attributes must have phType or phIndex specified
            // -> this is a placeholder drawing in master or layout slide -> the attributes are saved in the model.
            return savePlaceHolderAttributes(target, keySet.type, keySet.index, attrs, options);
        };

        /**
         * If a place holder drawing is deleted on a master or layout slide, it is necessary
         * to remove its place holder attributes from the model.
         *
         * @param {jQuery|Node} drawing
         *  The drawing node, that gets the specified attributes assigned.
         */
        this.deletePlaceHolderAttributes = function (drawing) {

            var // the target of the slide that contains the drawing
                target = '',
                // the explicit drawing attributes
                drawingAttrs = null,
                // the type of the place holder drawing
                type = '',
                // the index of the place holder drawing
                index = 0;

            if (!PresentationUtils.isPlaceHolderDrawing(drawing)) { return null; }

            // checking, if the drawing is located on a master or layout slide
            target = DOM.getTargetContainerId($(drawing).parent()); // place holder drawings are direct children of slide

            if (docModel.isLayoutOrMasterId(target)) {

                // getting type and index from the drawing
                drawingAttrs = AttributeUtils.getExplicitAttributes(drawing);

                if (placeHolderCollector[target]) {
                    type = PresentationUtils.getPlaceHolderDrawingType(drawing, drawingAttrs) || PresentationUtils.DEFAULT_PLACEHOLDER_TYPE;
                    if (placeHolderCollector[target][type]) {
                        index = PresentationUtils.getPlaceHolderDrawingIndex(drawing, drawingAttrs) || PresentationUtils.DEFAULT_PLACEHOLDER_INDEX;
                        if (placeHolderCollector[target][type][index]) {
                            delete placeHolderCollector[target][type][index];
                            if (_.isEmpty(placeHolderCollector[target][type])) {
                                delete placeHolderCollector[target][type];
                                if (_.isEmpty(placeHolderCollector[target])) {
                                    delete placeHolderCollector[target];
                                }
                            }
                        }
                    }
                }
            }

        };

        /**
         * Receiving the attributes for a drawing with place holder attribute. In this case the attributes from the
         * parent layers can be used. If the drawing is inside a 'standard' slide, the attributes might be defined
         * in the layout or the master slide. If the drawing is inside a layout slide, the missing attributes might
         * be defined in the master slide.
         *
         * Info: For master slides the index needs to be ignored. This can easily be seen for the place holder types
         *       'ftr', 'dt' and 'sldNum'. In the default master slide they have not the same indices as used in the
         *       layout slides.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Number} index
         *  The place holder index.
         *
         * @returns {Object|Null}
         *  The attributes defined for the specified target and place holder type.
         */
        this.getPlaceHolderAttributes = function (target, type, index) {

            var // the attributes assigned to target and type
                attributes = null,
                // the available indices for this type of place holder
                indices = null;

            // switching type, if not supported on master slide (or on layout slide in ODF)
            if ((docModel.isMasterSlideId(target) || (odf && docModel.isLayoutSlideId(target))) && PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES[type]) { type = PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES[type]; }

            if (placeHolderCollector && placeHolderCollector[target] && placeHolderCollector[target][type]) {
                attributes = placeHolderCollector[target][type];

                if (attributes[index]) {
                    attributes = attributes[index];
                } else {
                    if (docModel.isMasterSlideId(target)) {
                        // if this is a master slide, the index can be ignored
                        indices = _.keys(attributes);
                        if (indices && indices.length > 0) {
                            attributes = attributes[indices[0]]; // using the first key
                        }
                    }
                }
            }

            return attributes ? _.copy(attributes, true) : null;
        };

        /**
         * Receiving all place holder attributes that are stored for a specific master or layout node.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @returns {Object|Null}
         *  An object containing all place holder attributes defined for the specified layout/master
         *  slide. Or null, if this slide contains no place holder attributes.
         */
        this.getAllPlaceHolderAttributesForId = function (target) {
            return placeHolderCollector[target] || null;
        };

        /**
         * Return highest placeholder index used for a specific master or layout node.
         *
         * @param {String} target
         *  ID of target container node.
         *
         * @returns {Number}
         *  Highest placeholder index used on given slide, or 0 if there are no placeholder indexes used.
         */
        this.getHighestPlaceholderIndexForId = function (target) {
            var placeholderAttrs =  placeHolderCollector[target];
            var placeholderKeys = [];
            _.each(placeholderAttrs, function (plHolder) {
                _.each(_.keys(plHolder), function (oneKey) {
                    if (oneKey && _.isString(oneKey)) {
                        oneKey = parseInt(oneKey, 10);
                        placeholderKeys.push(oneKey);
                    }
                });
            });
            placeholderKeys.sort(function (a, b) { return a - b; });

            return _.last(placeholderKeys) || 0;
        };

        /**
         * Receiving a place holder index that is not used for a specified slide ID
         *
         * @param {String} id
         *  ID of a layout or master slide. If this is a document slide, the id of the corresponding
         *  layout slide is used.
         *
         * @returns {Number}
         *  A free index for a place holder drawing
         */
        this.getFreePlaceHolderIndex = function (id) {

            var // the index number
                index = 0,
                // all used indices
                allUsedIndices = [],
                // the id of the layout or master slide
                layoutId = docModel.isLayoutOrMasterId(id) ? id : docModel.getLayoutSlideId(id),
                // all layout drawings attribute sets
                allAttributes = self.getAllPlaceHolderAttributesForId(layoutId),
                // the maximum of the used values
                max = 0;

            // iterating over all types
            _.each(allAttributes, function (type) {
                allUsedIndices = allUsedIndices.concat(_.keys(allAttributes[type]));
            });

            max = Math.max.apply(Math, allUsedIndices);

            if (_.isNumber(max)) { index = max + 1; }

            return index;
        };

        /**
         * Receiving the complete list style object. This is necessary, if a snap shot is created, because these
         * list styles are not saved in the data object of the drawing. The other attributes are all saved in the
         * data-object, so that they can be restored later.
         *
         * Info: This function shall only be used for snap shot creation.
         *
         * @returns {Object}
         *  The collector object for the list style attributes assigned to place holder drawings.
         */
        this.getAllDrawingListStyleAttributes = function () {

            var // the object containing the drawing list styles
                allDrawingListStyles = null;

            // iterating over target, type and index
            _.each(_.keys(placeHolderCollector), function (target) {
                _.each(_.keys(placeHolderCollector[target]), function (type) {
                    _.each(_.keys(placeHolderCollector[target][type]), function (index) {

                        var drawingAttrs = placeHolderCollector[target][type][index];

                        if (drawingAttrs.listStyle) {
                            allDrawingListStyles = allDrawingListStyles || {};
                            allDrawingListStyles[target] = allDrawingListStyles[target] || {};
                            allDrawingListStyles[target][type] = allDrawingListStyles[target][type] || {};
                            allDrawingListStyles[target][type][index] = allDrawingListStyles[target][type][index] || {};

                            allDrawingListStyles[target][type][index].listStyle = _.copy(drawingAttrs.listStyle, true);
                        }

                    });
                });
            });

            return allDrawingListStyles;
        };

        /**
         * Setting the full object for the drawing list styles. This is necessary after applying a snapshot.
         *
         * Info: This function shall only be used for applying snapshot.
         *
         * @param {Object} listStyles
         *  The object containing the full set of drawing list styles. This was typically saved before using
         *  the function 'getAllDrawingListStyleAttributes'.
         */
        this.setAllDrawingListStyleAttributes = function (listStyles) {

            // iterating over target, type and index
            _.each(_.keys(listStyles), function (target) {
                _.each(_.keys(listStyles[target]), function (type) {
                    _.each(_.keys(listStyles[target][type]), function (index) {

                        var drawingAttrs = listStyles[target][type][index];

                        if (drawingAttrs.listStyle) {
                            placeHolderCollector = placeHolderCollector || {};
                            placeHolderCollector[target] = placeHolderCollector[target] || {};
                            placeHolderCollector[target][type] = placeHolderCollector[target][type] || {};
                            placeHolderCollector[target][type][index] = placeHolderCollector[target][type][index] || {};

                            placeHolderCollector[target][type][index].listStyle = _.copy(drawingAttrs.listStyle, true);
                        }
                    });
                });
            });
        };

        /**
         * Emptying the complete place holder model. This is necessary, if a new snapshot is applied.
         */
        this.emptyPlaceHolderModel = function () {
            emptyPlaceHolderModel();
        };

        /**
         * Getting the list styles for a specified drawing that are saved in the model. Or null, if the
         * specified drawing is no place holder drawing or if no list styles exist for this drawing.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node to be checked.
         *
         * @returns {Object|Null}
         *  The list styles for the specified drawing that are saved in the model. Or null, if the
         *  specified drawing is no place holder drawing or if no list styles exist for this drawing.
         */
        this.getAllListStylesAtPlaceHolderDrawing = function (drawing) {

            var // an object containing target, type and index for the specified drawing
                descriptor = null,
                // the drawing list style object
                drawingListStyles = null;

            // getting target, type and index
            descriptor = PresentationUtils.getDrawingModelDescriptor(drawing);

            if (!descriptor) { return drawingListStyles; }  // this might not be a place holder drawing

            // the descriptor object must contain the keys 'target', 'type' and 'index' for the specified drawing

            if (placeHolderCollector[descriptor.target] &&
                placeHolderCollector[descriptor.target][descriptor.type] &&
                placeHolderCollector[descriptor.target][descriptor.type][descriptor.index] &&
                placeHolderCollector[descriptor.target][descriptor.type][descriptor.index].listStyle) {
                drawingListStyles = { listStyle: _.copy(placeHolderCollector[descriptor.target][descriptor.type][descriptor.index].listStyle, true) };
            }

            return drawingListStyles;
        };

        /**
         * Getting the list styles for a specified drawing that are saved in the model (for place
         * holder drawings) or directly at the data attribute of the drawing (for non-place holder
         * drawings). If no list styles exist for this drawing, null is returned.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node to be checked.
         *
         * @returns {Object|Null}
         *  The list styles for the specified drawing that are saved in the model or in the data
         *  object of the drawing. If no list styles exist for this drawing, null is returned.
         */
        this.getAllListStylesAtDrawing = function (drawing) {

            var // the drawing list style object
                drawingListStyles = self.getAllListStylesAtPlaceHolderDrawing(drawing);

            if (!drawingListStyles) {
                // there might also be list styles attributes set at the drawing data object
                // for non placeholder drawings -> see function 'setDrawingListStyleAttributes'
                if ($(drawing).data('listStyle')) {
                    drawingListStyles = { listStyle: _.copy($(drawing).data('listStyle'), true) };
                }
            }

            return drawingListStyles;
        };

        /**
         * Checking, whether a specified place holder type is allowed on a slide with a specified ID.
         * This function is especially used to find, if place holders of type 'dt', 'ftr' or 'sldNum'
         * are to be inserted on a slide or not. For all other place holder types, true is returned.
         *
         * @param {String} id
         *  The slide ID.
         *
         * @param {String} type
         *  The place holder type.
         *
         * @param {Object} [attrs]
         *  The attributes of family 'slide'. If these attributes are not specified, the specified id
         *  is used to determine this attributes. For performance reasons, this attributes can be given
         *  into this function, if they are already available.
         *
         * @returns {Boolean}
         *  Whether a specified place holder type is allowed on a slide with a specified ID. This value
         *  can only be false for place holders of type 'dt', 'ftr' or 'sldNum'.
         */
        this.isSupportedPlaceHolderForSlideId = function (id, phType, attrs) {

            var // whether the slide with the specified ID allows inserting of place holder drawings of the specified type
                isSupported = true,
                // the merged slide attributes, to determine, which place holder drawings will be inserted
                mergedSlideAttributes = attrs || docModel.getMergedSlideAttributesByFamily(id, 'slide');

            // The place holder drawings of type 'dt', 'ftr' and 'sldNum' are not always created
            // on the new slide. This is specified at the layout and/or master slide.
            if ((phType === 'dt' && !mergedSlideAttributes.isDate) ||
                (phType === 'ftr' && !mergedSlideAttributes.isFooter) ||
                (phType === 'sldNum' && !mergedSlideAttributes.isSlideNum)) {
                isSupported = false;
            }

            return isSupported;
        };

        /**
         * Helper function to find different and identical place holder drawings in two
         * specified layout slides.
         *
         * The return object contains three properties, that have an array as value. The
         * property names are 'onlyFirst', 'onlySecond' and 'both'.
         *
         * The values are arrays, that contain objects, that describe a place holder drawing
         * by its type and its index. Therefore these objects have the properties 'type'
         * and 'index'.
         *
         * So the properties 'onlyFirst', 'onlySecond' and 'both' describe the place holder
         * drawings that exist only on the first slide, only on the second slide or in both
         * slides.
         *
         * @param {String} slideId
         *  The ID of the slide, that gets a new layout assigned.
         *
         * @param {String} firstId
         *  The ID of the first layout slide.
         *
         * @param {String} secondId
         *  The ID of the second layout slide.
         *
         * @param {Object} firstSlideAttrs
         *  The attributes of 'slide' family for the first slide
         *
         * @param {Object} secondSlideAttrs
         *  The attributes of 'slide' family for the second slide
         *
         * @returns {Object}
         *  A comparison object with the properties 'onlyFirst', 'onlySecond' and 'both' that
         *  describe the place holder drawings that exist only on the first slide, only on the
         *  second slide or in both slides.
         */
        this.comparePlaceHolderSet = function (slideId, firstId, secondId, firstSlideAttrs, secondSlideAttrs) {

            var // the comparison object with the properties 'onlyFirst', 'onlySecond' and 'both'
                comparison = { onlyFirst: [], onlySecond: [], both: [] },
                // the saved place holder attributes for the first specified layout ID
                first = _.copy(self.getAllPlaceHolderAttributesForId(firstId), true),
                // the saved place holder attributes for the second specified layout ID
                second = _.copy(self.getAllPlaceHolderAttributesForId(secondId), true);

            // handling differentitation between text body and content body place holders
            first = handleDifferentBodySet(first);
            second = handleDifferentBodySet(second);

            // Checking, if the place holder drawings still do exist
            if (first) { removeNonExistentPlaceHolderDrawings(slideId, first); }

            // Content body place holders can be converted to text body placeholders, if they already contain text (52805)
            if (first && second) { checkContentBodyPlaceHolderDrawings(slideId, first, second); }

            _.each(_.keys(first), function (type) {
                if (self.isSupportedPlaceHolderForSlideId(firstId, type, firstSlideAttrs)) {
                    _.each(_.keys(first[type]), function (index) {
                        if (second && second[type] && second[type][index] && self.isSupportedPlaceHolderForSlideId(secondId, type, secondSlideAttrs)) {
                            comparison.both.push({ type: type, index: parseInt(index, 10) });
                        } else {
                            comparison.onlyFirst.push({ type: type, index: parseInt(index, 10) });
                        }
                    });
                }
            });

            _.each(_.keys(second), function (type) {
                if (self.isSupportedPlaceHolderForSlideId(secondId, type, secondSlideAttrs)) {
                    _.each(_.keys(second[type]), function (index) {
                        if (!first || !first[type] || !first[type][index]) {
                            comparison.onlySecond.push({ type: type, index: parseInt(index, 10) });
                        }
                    });
                }
            });

            // some magic conversion:
            // title <-> ctrTitle
            // body <-> subTitle

            return comparison;
        };

        /**
         * After a change of the slide layout it is possible that the new
         * layout contains some other types for the place holder drawings
         * than the old layout. Some types are converted into each other.
         * For example a place holder drawing of type 'title' can be
         * interpreted as 'ctrTitle' after changing the layout slide.
         *
         * The allowed conversions are listed in the constant:
         * PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES.
         *
         * The parameter 'comparePlaceHolder' is an object created by the
         * function 'PresentationDrawingStyles.comparePlaceHolderSet'.
         * It has to contain the properties 'onlyFirst', 'onlySecond'
         * and 'both'.
         *
         * This parameter object is modified within this function.
         *
         * @param {Object} comparePlaceHolder
         *  The object that describe the differences in the place holder
         *  drawings of two layout slides. This object is generated by the
         *  function 'PresentationDrawingStyles.comparePlaceHolderSet' and
         *  contains the properties 'onlyFirst', 'onlySecond' and 'both'.
         *  This object is modified within this function.
         *
         * @returns {Object[]}
         *  An array of objects. The objects inside this array describe
         *  the allowed switches of drawing place holder. Every object
         *  contains the properties 'from' and 'to'. The values are
         *  the place holder drawing info objects, that contain the
         *  type and index for a place holder drawing.
         */
        this.analyzeComparePlaceHolder = function (comparePlaceHolder) {

            var // an array of supported switches for the place holder drawings
                switches = [];

            // helper function to find the specified type in the collector
            function foundType(type, allPlaceHolders) {
                return _.find(allPlaceHolders, function (placeHolder) {
                    return placeHolder.type === type;
                });
            }

            if (comparePlaceHolder && comparePlaceHolder.onlyFirst.length > 0 && comparePlaceHolder.onlySecond.length > 0) {

                _.each(PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES, function (value, key) {

                    if (!odf) {
                        if (value === 'body') { value = PresentationUtils.CONTENTBODY; }
                        if (key === 'body') { key = PresentationUtils.CONTENTBODY; }
                    }

                    var first = foundType(key, comparePlaceHolder.onlyFirst),
                        second = foundType(value, comparePlaceHolder.onlySecond);

                    // searching the key in onlyFirst and the value in onlySecond
                    if (first && second) {

                        // adding to switches object
                        switches.push({ from: first, to: second });

                        // removing items from collectors
                        comparePlaceHolder.onlyFirst = _.without(comparePlaceHolder.onlyFirst, first);
                        comparePlaceHolder.onlySecond = _.without(comparePlaceHolder.onlySecond, second);
                    }
                });
            }

            return switches;
        };

        /**
         * Collecting all place holder drawings on the specified slide, that have no definition in the
         * layout slide. This can happen after switching the layout slide. It is possible, that the new
         * layout slide has less place holder drawing definitions.
         *
         * @param {String} id
         *  The ID of the specified slide.
         *
         * @returns {jQuery}
         *  A jQuery collection of all place holder drawings, that have no definition in the layout slide.
         *  This can be drawings of any type, not only of type 'body'.
         */
        this.getAllUnassignedPhDrawings = function (id) {
            return self.getAllPlaceHolderDrawingsOnSlideByIndex(id, self.getMsDefaultPlaceHolderIndex());
        };

        /**
         * Helper function for updating all place holder drawings on a
         * specified slide.
         *
         * @param {Node|jQuery} slide
         *  The (jQueryfied) slide node.
         */
        this.updatePlaceHolderDrawings = function (slide) {

            // simply update all drawings that are place holder types
            _.each($(slide).children('div[placeholdertype]'), function (drawing) {

                // updating the drawing itself
                self.updateElementFormatting($(drawing));

                // ... and updating the paragraphs
                _.each($(drawing).find(DOM.PARAGRAPH_NODE_SELECTOR), function (para) {
                    // updating all paragraphs, that have a modified attribute set
                    docModel.getParagraphStyles().updateElementFormatting($(para));
                    docModel.updateListsDebounced($(para));
                });

            });

        };

        /**
         * Helper function for updating all drawings on a specified slide.
         *
         * @param {Node|jQuery} slide
         *  The (jQueryfied) slide node.
         */
        this.updateAllDrawingsOnSlide = function (slide) {

            // simply update all drawings that are place holder types
            _.each($(slide).children('div.drawing'), function (drawing) {

                // updating the drawing itself
                self.updateElementFormatting($(drawing));

                // ... and updating the paragraphs
                _.each($(drawing).find(DOM.PARAGRAPH_NODE_SELECTOR), function (para) {
                    // updating all paragraphs, that have a modified attribute set
                    docModel.getParagraphStyles().updateElementFormatting($(para));
                    docModel.updateListsDebounced($(para));
                });

            });

        };

        /**
         * Updating the template text in place holder drawing that cannot contain text content. This
         * are the place holder drawings of type 'tbl', 'pic', ... . The text of these place holder
         * drawings cannot be modified in a document slide.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node.
         *
         * @param {String} type
         *  The type of the place holder drawing.
         *
         * @param {Number|Null} index
         *  The index of the place holder drawing.
         *
         * @param {String} slideId
         *  The ID of the slide that contains the drawing.
         */
        this.updateTemplateReadOnlyText = function (drawing, type, index, slideId) {
            addTemplateReadOnlyText(drawing, type, index, slideId, { update: true });
        };

        /**
         * Handling the visualization of template buttons for inserting image or table.
         * This template buttons are visible in empty place holder drawings of type
         * 'body'.
         *
         * @param {Node|jQuery} drawing
         *  The drawing node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Boolean} addImages
         *  Whether the template images shall be inserted or removed.
         *
         * @param {String} [slideId]
         *  The ID of the slide containing the drawing. If not specified, it is
         *  determined within this function.
         */
        this.handleImageTemplateContainer = function (drawing, addImages, slideId) {

            // the container node for the template images
            var imageContainerNode = null;
            // the explicit attributes of the drawing
            var expDrawingAttrs = AttributeUtils.getExplicitAttributes(drawing);
            // the place holder type
            var placeHolderType = null;
            // the place holder index
            var placeHolderIndex = null;

            // helper function to get the image container node in a drawing
            function setImageContainerNode() {
                imageContainerNode = $(drawing).children(PresentationDrawingStyles.TEMPLATE_DRAWING_CONTAINER_SELECTOR);
            }

            // handling place holder drawings
            if (PresentationUtils.isPlaceHolderAttributeSet(expDrawingAttrs)) {

                placeHolderType = expDrawingAttrs.presentation.phType || '';

                if (!slideId) { slideId = docModel.getSlideId($(drawing).parent()); }

                if (docModel.isLayoutOrMasterId(slideId)) {

                    if (docModel.isLayoutSlideId(slideId)) { // not inserting for master slides

                        setImageContainerNode(); // in the master view the icons are always visible (and have no effect)

                        if (imageContainerNode.length === 0) {
                            insertImageTemplateContainer(drawing, placeHolderType, placeHolderIndex, slideId);
                        } else if (imageContainerNode.length === 1) {
                            // maybe a positioning update is required (after changing the layout)
                            updateImageContainerNode(drawing, imageContainerNode);
                        }
                    }

                } else {
                    if (addImages) {

                        setImageContainerNode();

                        if (imageContainerNode.length === 0) {
                            placeHolderIndex = _.isNumber(expDrawingAttrs.presentation.phIndex) ? expDrawingAttrs.presentation.phIndex : null;
                            insertImageTemplateContainer(drawing, placeHolderType, placeHolderIndex, slideId);
                        } else if (imageContainerNode.length === 1) {
                            // maybe a positioning update is required (after changing the layout)
                            updateImageContainerNode(drawing, imageContainerNode);
                        }
                    } else {
                        removeImageTemplateContainer(drawing);
                    }
                }
            }
        };

        /**
         * Handling the visualization of empty text frames for a complete slide.
         *
         * @param {Node|jQuery} slide
         *  The slide node for that the visualization is done for all empty
         *  text frames.
         *
         * @param {Object} [options]
         *  Some additional options:
         *  @param {Boolean} [options.ignoreTemplateText=false]
         *      This option is handled within DOM.isEmptyTextFrame to define, that text frames
         *      that contain template text are also empty.
         */
        this.handleAllEmptyPlaceHolderDrawingsOnSlide = function (slide, options) {
            _.each($(slide).children('div.drawing'), function (drawing) {
                self.handleEmptyPlaceHolderDrawing(drawing, null, options);
            });
        };

        /**
         * Handling the visualization of empty text frames. In the standard view it is
         * necessary to assign a dotted line to these empty text frames and to insert
         * a template text.
         * In the layout or master view, such a dotted line is always assigned to the place
         * holder drawings.
         * In PowerPoint only place holder drawings get the dotted border (even if
         * they have a background color). Empty text frames do not get a border.
         *
         * @param {Node|jQuery} drawing
         *  The DOM node to be checked. If this object is a jQuery collection, uses
         *  the first DOM node it contains.
         *
         * @param {Object} [attributes]
         *  The (explicit) attributes of the drawing node. If they are not specified,
         *  the explicit attributes at the specified drawing node are evaluated.
         *
         * @param {Object} [options]
         *  Some additional options:
         *  @param {Boolean} [options.ignoreTemplateText=false]
         *      This option is handled within DOM.isEmptyTextFrame to define, that text frames
         *      that contain template text are also empty.
         *
         * @returns {Boolean}
         *  Whether text was inserted into the specified drawing.
         */
        this.handleEmptyPlaceHolderDrawing = function (drawing, attributes, options) {

            var // whether the default template text was inserted into the drawing
                addedText = false,
                // the id of the slide containing the specified drawing
                slideId = null,
                // whether the slide is a master or layout slide
                isLayoutOrMasterSlide = false,
                // whether the drawing is an empty place holder in which no text can be inserted (table, image, ...)
                isEmptyNoTextPlaceHolder = false,
                // the selection object
                selection = null;

            if (PresentationUtils.isPlaceHolderDrawing(drawing)) {
                if (odf) { handleMissingParagraphInPlaceHolderDrawing(drawing); }
                slideId = docModel.getSlideId($(drawing).parent());
                isLayoutOrMasterSlide = docModel.isLayoutOrMasterId(slideId);
                isEmptyNoTextPlaceHolder = !isLayoutOrMasterSlide && PresentationUtils.isPlaceHolderWithoutTextDrawing(drawing) && PresentationUtils.isEmptyPlaceHolderDrawing(drawing);
                handleBorderForEmptyTextFrames(drawing, isEmptyNoTextPlaceHolder, attributes || AttributeUtils.getExplicitAttributes(drawing));
                if (!isEmptyNoTextPlaceHolder) { addedText = handleTemplateText(drawing, options ? _.extend(options, { id: slideId }) : { id: slideId }); }
                self.handleImageTemplateContainer(drawing, (isLayoutOrMasterSlide ? true : (isEmptyNoTextPlaceHolder || DOM.isEmptyTextframe(drawing, { ignoreTemplateText: true }))), slideId); // handling the template buttons for inserting image or table

                if (addedText && isLayoutOrMasterSlide) {
                    if (slideId && !docModel.getSlideFormatManager().isUnformattedSlide(slideId)) {
                        docModel.forceTriggerOfSlideStateUpdateEvent(slideId);
                    }
                    if (PresentationUtils.isPlaceHolderWithoutTextDrawing(drawing)) { // trigger an event, if text was added inside drawings that do not contain text (picture, chart, ...)
                        docModel.forceTriggerOfTemplateTextUpdatedEvent(slideId, drawing);
                    }
                }

                // setting selection again, if it is currently a text selection inside the empty place holder drawing
                if (addedText) {
                    selection = docModel.getSelection();
                    if (selection.isAdditionalTextframeSelection() && Utils.getDomNode(drawing) === Utils.getDomNode(selection.getSelectedTextFrameDrawing())) {
                        selection.setTextSelection(selection.getStartPosition(), selection.getEndPosition());
                    }
                }

            }

            return addedText;
        };

        /**
         * Collecting from a specified slide all place holder drawings with the specified
         * type and index. If the index is not specified, all drawings with the specified
         * type will be returned.
         *
         * @param {String} id
         *  The ID of the slide.
         *
         * @param {String|String[]} type
         *  The type(s) of the drawing. It is possible to determine several types. This is a
         *  convenience function, so that subtitle can be changed, too, if a body was modified.
         *  In this case the parameter 'index' will be ignored.
         *
         * @param {String|Number} [index]
         *  The index of the drawing. If not specified, all drawings with the specified type
         *  with any index will be collected.
         *
         * @returns {jQuery}
         *  A collection of all place holder drawings, that are direct children of
         *  the slide.
         */
        this.getAllPlaceHolderDrawingsOnSlideBySelector = function (id, type, index) {

            var // the slide node
                slide = docModel.getSlideById(id),
                // the selector specific to the place holder type and index. This is used to find the drawing
                // that need an update of formatting (the model is already updated).
                selector = '',
                // whether the index will be ignored
                ignoreIndexValue = false;

            if (!slide) { return $(); }

            if (type === PresentationUtils.CONTENTBODY) { type = 'body'; }

            if (_.isArray(type)) {
                _.each(type, function (oneType) {
                    if (oneType === PresentationUtils.CONTENTBODY) { oneType = 'body'; }
                    if (selector) { selector += ', '; }
                    selector += 'div[placeholdertype=' + oneType + ']';
                    if (oneType === 'body') { selector += '[placeholderindex], div[placeholderindex]:not([placeholdertype])'; }
                });
            } else {

                ignoreIndexValue = (!_.isNumber(index) && !_.isString(index));  // check if the index is specified
                selector = 'div[placeholdertype=' + type + ']';

                // if type is body, the 'placeholderindex' must be correct, but 'placeholdertype' might not be set
                if (type === 'body') {
                    if (ignoreIndexValue) {
                        selector += '[placeholderindex], div[placeholderindex]:not([placeholdertype])';
                    } else {
                        selector += '[placeholderindex=' + index + '], div[placeholderindex=' + index + ']:not([placeholdertype])';
                    }
                } else if (!ignoreIndexValue) {
                    selector += '[placeholderindex=' + index + ']';
                }
            }

            return slide.children(selector);
        };

        /**
         * Collecting from a specified slide all place holder drawing of specified
         * type and index.
         *
         * @param {String} id
         *  The ID of the slide.
         *
         * @param {String|Number} index
         *  The index of the drawing.
         *
         * @returns {jQuery}
         *  A collection of the place holder drawings with the specified index. All those
         *  drawings are direct children of the slide. This function can be used to find
         *  all unassigned place holder drawings with index
         *  'PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX'.
         *  This drawings can be drawings of any type, not only of type 'body'.
         */
        this.getAllPlaceHolderDrawingsOnSlideByIndex = function (id, index) {

            var // the slide node
                slide = docModel.getSlideById(id),
                // the selector specific to the place holder index.
                selector = 'div[placeholderindex=' + index + ']';

            return slide.children(selector);
        };

        /**
         * Collecting from a specified slide all place holder drawings.
         *
         * @param {String} id
         *  The ID of the slide.
         *
         * @returns {jQuery}
         *  A collection of the place holder drawings.
         */
        this.getAllPlaceHolderDrawingsOnSlide = function (id) {

            var // the slide node
                slide = docModel.getSlideById(id),
                // the selector specific to the place holder index.
                selector = 'div[placeholderindex]';

            return slide.children(selector);
        };

        /**
         * Getter for the list of allowed drawing type switches during change of layout.
         *
         * @returns {Object}
         *  The object that contains all allowed drawing type switches during a change
         *  of the layout of a slide.
         */
        this.getListOfAllowedDrwawingTypeSwitches = function () {
            return PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES;
        };

        /**
         * Getter for the constant value of 'PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX'.
         *
         * @returns {Number}
         *  The default place holder index that is used in PP for place holder drawings
         *  that cannot be assigned to place holder drawing of current layout slide.
         */
        this.getMsDefaultPlaceHolderIndex = function () {
            return PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX;
        };

        /**
         * Getter for the master slide type converter.
         *
         * @returns {Object}
         *  The object describing the master slide type conversions.
         */
        this.getMasterSlideTypeConverter = function () {
            return _.copy(PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES);
        };

        /**
         * Setting list style attributes at the drawing node for non-place holder drawings (45279).
         * For place holder drawings there is a model, but it is also possible, that the
         * list styles are directly defined at a non-placeholder drawing node. In this case,
         * it is necessary to save this data directly at the node.
         *
         * Info: Using this data object must be an exception. It is only valid for non
         *       place holder drawings with directly set list styles. For all other drawings
         *       the model in the object 'placeHolderCollector' must be used.
         *
         * @param {jQuery|Node} drawingNode
         *  The drawing node.
         *
         * @param {Object} attrs
         *  The attributes object.
         */
        this.setDrawingListStyleAttributes = function (drawingNode, attrs) {
            if (attrs && attrs.listStyle && !PresentationUtils.isPlaceHolderDrawing(drawingNode)) {
                $(drawingNode).data('listStyle', attrs.listStyle);
            }
        };

        /**
         * Getting the list style attributes directly from the drawing node for non-place holder
         * drawings (45279). For place holder drawings there is a model, but it is also possible,
         * that the list styles are directly defined at a non placeholder drawing node. In this case,
         * it is necessary to read this data directly from the node.
         *
         * @param {jQuery|Node} drawingNode
         *  The drawing node.
         *
         * @param {Object} attrs
         *  The attributes object.
         */
        this.getDrawingListStyleAttributes = function (drawingNode) {
            return $(drawingNode).data('listStyle');
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributes('drawing', DEFINITIONS);

        // register the formatting handler for DOM elements (will be called during document import too)
        this.registerFormatHandler(updateDrawingFormatting, { duringImport: false });

        // register handler for removing content for place holder collector
        this.listenTo(docModel, 'removed:slide', removeFromPlaceholderCollector);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = null;
        });

    } }); // class PresentationDrawingStyles

    // constants --------------------------------------------------------------

    /**
     * Helper object with allowed switches of drawing types during
     * a change of a layout slide.
     */
    PresentationDrawingStyles.ALLOWED_DRAWING_TYPE_SWITCHES = {
        title: 'ctrTitle',
        ctrTitle: 'title',
        body: 'subTitle',
        subTitle: 'body'
    };

    /**
     * Helper object with forced switches for receiving the place holder
     * attributes from a master slide. Example: A master never has a place
     * holder of type 'subTitle'. Instead the 'body' must be used.
     *
     * Info: Please also check 'SlideAttributesMixin.LIST_TYPE_CONVERTER'.
     */
    PresentationDrawingStyles.MASTER_PLACEHOLDER_TYPE_SWITCHES = {
        subTitle: 'body',
        ctrTitle: 'title'
    };

    /**
     * The CSS class used to mark the template image container nodes inside
     * a drawing node.
     *
     * @constant
     */
    PresentationDrawingStyles.TEMPLATE_DRAWING_CONTAINER_CLASS = 'templatedrawingcontainer';

    /**
     * A jQuery selector that matches nodes representing a text frame node that
     * are 'classical' text frames in odf format.
     *
     * @constant
     */
    PresentationDrawingStyles.TEMPLATE_DRAWING_CONTAINER_SELECTOR = '.' + PresentationDrawingStyles.TEMPLATE_DRAWING_CONTAINER_CLASS;

    /**
     * The CSS class used to mark the template text in empty place holder nodes that cannot contain text content.
     * This are the place holder templates defined in 'PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE'.
     *
     * @constant
     */
    PresentationDrawingStyles.TEMPLATE_READONLY_TEXT_CONTAINER_CLASS = 'templatereadonlytext';

    /**
     * A jQuery selector that matches nodes representing read-only text information inside empty place holder
     * nodes that cannot contain text content. This are the place holder templates defined in
     * 'PresentationUtils.NO_TEXT_PLACEHOLDER_TYPE'.
     *
     * @constant
     */
    PresentationDrawingStyles.TEMPLATE_READONLY_TEXT_CONTAINER_SELECTOR = '.' + PresentationDrawingStyles.TEMPLATE_READONLY_TEXT_CONTAINER_CLASS;

    /**
     * The default place holder index for drawings of type 'body'.
     */
    PresentationDrawingStyles.MS_DEFAULT_PLACEHOLDER_INDEX = 4294967295;

    // exports ================================================================

    return PresentationDrawingStyles;

});
