/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/dialog/hyperlinkdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialog/basedialog',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/editframework/view/editlabels',
    'gettext!io.ox/office/editframework/main'
], function (Utils, BaseDialog, HyperlinkUtils, Labels, gt) {

    'use strict';

    // a unique DOM identifier for the label of the display text input field
    var DISPLAY_LABEL_ID = _.uniqueId('display');

    // a unique DOM identifier for the display text input field
    var DISPLAY_INPUT_ID = _.uniqueId('display');

    // a unique DOM identifier for the label of the URL input field
    var URL_LABEL_ID = _.uniqueId('url');

    // a unique DOM identifier for the URL input field
    var URL_INPUT_ID = _.uniqueId('url');

    // class HyperlinkDialog ==================================================

    /**
     * A modal dialog with two text fields allowing to edit the URL and the
     * text representation of a hyperlink.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {EditView} docView
     *  The document view instance that has created this dialog.
     *
     * @param {String} [url]
     *  The initial URL to be shown in the dialog. If omitted, the text field
     *  for the URL will appear empty.
     *
     * @param {String} [display]
     *  The initial display string for the URL to be shown in the dialog. If
     *  omitted, the text field for the display string will appear empty.
     */
    var HyperlinkDialog = BaseDialog.extend(function (docView, url, display) {

        // whether to keep URL and display text in sync
        var syncDisplay = !display || (url === display);

        // the input field for the URL
        var urlInputNode = $('<input>', {
            value: url || '',
            tabindex: 0,
            id: URL_INPUT_ID,
            role: 'textbox',
            'aria-labelledby': URL_LABEL_ID,
            'aria-required': true
        }).addClass('form-control');

        // the input field for the display text of the URL
        var displayInputNode = $('<input>', {
            value: display || url,
            tabindex: 0,
            id: DISPLAY_INPUT_ID,
            role: 'textbox',
            'aria-labelledby': DISPLAY_LABEL_ID,
            'aria-required': true
        }).addClass('form-control');

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, docView, {
            width: 400,
            title: url ? Labels.EDIT_HYPERLINK_LABEL : Labels.INSERT_HYPERLINK_LABEL,
            okLabel: gt('Insert')
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the trimmed URL from the input field.
         *
         * @returns {String}
         *  The trimmed URL from the input field.
         */
        function getUrl() {
            // bug 46225: remove all embedded whitespace from the URL
            return urlInputNode.val().replace(/[\x00-\x1f\x80-\x9f\s]+/g, '');
        }

        function updateUrlField() {
            var currUrl = getUrl();
            urlInputNode.attr('aria-invalid', currUrl.length === 0);
            if (syncDisplay) { displayInputNode.val(currUrl); }
        }

        function updateDisplayField() {
            syncDisplay = false;
        }

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // Bug 43868: ios auto correction should be of for input fields
        // TODO: use Forms.createInputMarkup() to create inputs here, it already has this fix
        if (Utils.IOS) {
            urlInputNode.attr({ autocorrect: 'off', autocapitalize: 'none' });
            displayInputNode.attr({ autocorrect: 'off', autocapitalize: 'none' });
        }

        // add the input controls to the dialog body
        this.append(
            $('<div class="form-group">').append(
                $('<label id="' +  URL_LABEL_ID + '" for="' + URL_INPUT_ID + '">')
                    .text(/*#. "Insert URL" dialog: The URL itself */ gt('URL')),
                urlInputNode
            ),
            $('<div class="form-group">').append(
                $('<label id="' + DISPLAY_LABEL_ID + '" for="' + DISPLAY_INPUT_ID + '">')
                    .text(/*#. "Insert URL" dialog: The display text of the URL */ gt('Text')),
                displayInputNode
            )
        );

        // register a change handler to toggle the insert button
        this.addOkValidator(function () { return getUrl().length > 0; });
        this.validateOn(urlInputNode, 'input');

        // update input fields when typing
        urlInputNode.on('input', updateUrlField);
        displayInputNode.on('input', updateDisplayField);

        // set initial focus
        this.setFocusNode(urlInputNode);

        // Bugfix 40844: when the user focus the displayInputNode via tab or mouse click,
        // the text in the displayInputNode should be selected
        displayInputNode.on('focus', function () {
            // a workaround is needed for the browsers below
            // otherwise on mouseup the selection is removed again
            if (_.browser.Chrome || _.browser.Safari || _.browser.Edge) {
                // the workaround: it's also mentioned here https://code.google.com/p/chromium/issues/detail?id=4505
                _.defer(function () { displayInputNode.select(); });
            // this is how it's supposed to work normally
            } else {
                displayInputNode.select();
            }
        });

        // handler for the OK button (keep dialog open, if the current URL is invalid)
        this.setOkHandler(function () {
            var currUrl = getUrl();
            var display = displayInputNode.val();

            // add http: as default if protocol is missing
            var newUrl = (currUrl.indexOf(':') < 0) ? ('http://' + currUrl) : currUrl;

            // show warning and keep the dialog open, if the URL is invalid
            if (!HyperlinkUtils.hasSupportedProtocol(newUrl) || !HyperlinkUtils.isValidURL(newUrl)) {
                docView.yell({ type: 'warning', message: Labels.INVALID_HYPERLINK_LABEL });
                return new $.Deferred().reject();
            }

            // return result object for a valid URL
            return { url: newUrl, text: display };
        }, { keepOpen: 'fail' });

        // add a Remove button, if a URL has been passed
        if (url) {
            this.createButton('remove', gt('Remove'), {
                alignLeft: true,
                buttonStyle: 'warning',
                actionHandler: _.constant({ url: null, text: null })
            });
        }

    }); // class HyperlinkDialog

    // exports ================================================================

    return HyperlinkDialog;

});
