/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/userbadge', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/baseframework/view/control/userpicture',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/editframework/view/editlabels',
    'io.ox/office/tk/utils/driveutils',
    'gettext!io.ox/office/editframework/main'
], function (Utils, Forms, UserPicture, Config, Labels, DriveUtils, gt) {

    'use strict';

    // class UserBadge ========================================================

    /**
     * A badge displaying properties of an arbitrary user, consisting the
     * user's profile picture, display name, and active status.
     *
     * @constructor
     *
     * @extends UserPicture
     *
     * @param {EditView} docView
     *  The document view containing this control instance.
     *
     * @param {Object|Null} clientData
     *  The initial user information. Must contain the string properties
     *  'userId', 'userName', 'colorIndex', 'remote', and 'editor'. See method
     *  EditApplication.getActiveClients() for more details.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that will be passed to base class UserPicture. The
     *  following additional options are supported:
     *  - {Boolean} [initOptions.showOwnColor=false]
     *      If set to true, the current user will be shown like a remote user
     *      with a colored status box.
     */
    function UserBadge(docView, clientData, initOptions) {

        // the link containing the user name
        var userNameNode = (clientData.guest || DriveUtils.isGuest() || !Config.CONTACTS_AVAILABLE) ? $('<div class="user-name disabled">') : $('<a class="user-name halo-link">');

        // the status node at the right border
        var userStatusNode = $('<div class="user-status">');

        // the color box in the status node
        var userColorNode = $('<div class="user-color">').appendTo(userStatusNode);

        // whether to show the current user with a colored status box too
        var showOwnColor = Utils.getBooleanOption(initOptions, 'showOwnColor', false);

        // base constructor ---------------------------------------------------

        UserPicture.call(this, docView, clientData, Utils.extendOptions(initOptions, { size: 32 }));

        // private methods ----------------------------------------------------

        function updateHandler(currentData) {

            // the user name to be inserted into the elements
            var userName = null;
            // tooltip for edit rights
            var tooltip = null;

            if (!currentData.userName) {
                userName = gt('Guest');
            } else {
                userName = _.noI18n(currentData.userName);
            }

            // add current user name and user identifier to link
            userNameNode.text(userName).data('internal_userid', currentData.userId);

            // add color for remote users
            userStatusNode.toggleClass('user-' + currentData.colorIndex, Config.SHOW_REMOTE_SELECTIONS && (showOwnColor || currentData.remote));

            // mark user with editing rights with an appropriate icon (bug 36482: not during offline mode)
            userColorNode.empty();
            // delete tooltip
            Forms.setToolTip(userColorNode, '');
            if (currentData.editor && !docView.getApp().isOffline()) {
                userColorNode.append(Forms.createIconMarkup(Labels.EDIT_ICON));

                if (currentData.remote) {
                    //#. shown as tooltip in a list of all users working on the same document
                    tooltip = gt('This user has edit rights for the document.');
                } else {
                    //#. shown as tooltip in a list of all users working on the same document
                    tooltip = gt('You have edit rights for the document.');
                }
                Forms.setToolTip(userColorNode, tooltip);
            }
        }

        // initialization -----------------------------------------------------

        // initialization of the root node of this badge
        this.getNode().addClass('user-badge').attr('role', 'dialog');

        // insert the additional nodes into the group
        this.addChildNodes($('<div class="user-name">').append(userNameNode), userStatusNode);

        // update handler for user name etc.
        this.registerUpdateHandler(updateHandler);

        // destroy all class members
        this.registerDestructor(function () {
            docView = clientData = initOptions = userNameNode = userStatusNode = userColorNode = null;
        });

    } // class UserBadge

    // exports ================================================================

    // derive this class from class UserPicture
    return UserPicture.extend({ constructor: UserBadge });

});
