/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/dialog/insertimagedrivedialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/drawinglayer/utils/imageutils',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, DriveUtils, ImageUtils, gt) {

    'use strict';

    // constants ==============================================================

    var RECENT_IMAGE_FOLDER_KEY = 'recentImageFolder';

    // class InsertImageDriveDialog ===========================================

    /**
     * A dialog that provides a file picker from Drive for a new image to be
     * inserted into the document.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {EditView} docView
     *  The document view that is creating the dialog.
     */
    function InsertImageDriveDialog(docView) {

        var app = docView.getApp();

        // private methods ----------------------------------------------------

        /**
         * Inserts the image file of the passed infostore descriptor into the
         * document.
         *
         * @param {Object} fileDesc
         *  The infostore descriptor of the file being inserted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the image has been uploaded to
         *  the server; or rejected, in case of an error.
         */
        function insertDriveImage(fileDesc) {

            if (!fileDesc) { return $.Deferred().reject(); }

            var fileName = fileDesc.filename;

            var params = {
                add_ext: fileName.substring(fileName.lastIndexOf('.') + 1),
                add_fileid: fileDesc.id,
                add_folderid: fileDesc.folder_id,
                alt_filename: fileName
            };

            return app.sendActionRequest('addfile', params).then(function (data) {
                return {
                    url: Utils.getStringOption(data, 'added_filename', ''),
                    name: ImageUtils.getFileNameWithoutExt(fileName)
                };
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Shows the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         * the default action; or rejected, if the dialog has been canceled.
         */
        this.show = function () {

            return require([
                'io.ox/files/filepicker',
                'io.ox/files/mobile-navbar-extensions' // prefetch
            ]).then(function (FilePicker) {
                var deferred = app.createDeferred('InsertImageDriveDialog.show');
                var recentImageFolder = app.getUserSettingsValue(RECENT_IMAGE_FOLDER_KEY) || DriveUtils.getStandardPicturesFolderId();
                var done = false;

                var dialogPromise = new FilePicker({
                    // filter files of disabled applications (capabilities)
                    filter: function (file) {
                        return ImageUtils.hasFileImageExtension(file.filename);
                    },
                    sorter: function (file) {
                        return (file.filename || file.title).toLowerCase();
                    },
                    primaryButtonText: gt('OK'),
                    header: gt('Insert Image'),
                    folder: recentImageFolder,
                    uploadButton: true,
                    multiselect: false,
                    //uploadFolder: DriveUtils.getStandardDocumentsFolderId(),
                    hideTrashfolder: true,
                    acceptLocalFileType: 'image/*',
                    createFolderButton: false,
                    cancel: deferred.reject,
                    close: function () {
                        if (!done) {
                            deferred.reject();
                        }
                    },
                    initialize: function (dialog) {
                        docView.closeDialogOnReadOnlyMode(dialog);
                    }
                });

                dialogPromise.done(function (selectedFiles) {
                    if (app.isEditable()) {
                        var selectedFile = selectedFiles[0];
                        if (selectedFile) {
                            done = true;
                            insertDriveImage(selectedFile).done(function (imageFragment) {
                                deferred.resolve(imageFragment);
                            });

                            app.setUserSettingsValue(RECENT_IMAGE_FOLDER_KEY, selectedFile.folder_id);
                        } else {
                            deferred.reject();
                        }
                    } else {
                        deferred.reject();
                    }
                });

                dialogPromise.fail(function () {
                    deferred.reject();
                });

                return deferred.promise();
            });
        };

    } // class InsertImageDriveDialog

    // exports ================================================================

    return InsertImageDriveDialog;

});
