/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/control/arrowstylepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/path',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/tk/control/radiolist',
    'gettext!io.ox/office/drawinglayer/main'
], function (Utils, Forms, ValueMap, Path, Canvas, RadioList, gt) {

    'use strict';

    // the canvas element used to generate the bitmaps
    var canvas = new Canvas(Canvas.SINGLETON).initialize({ width: 100, height: 18 });

    // maps unique bitmap keys to the data URLs
    var bitmapUrlMap = new ValueMap();

    // private global functions ===============================================

    function createArrowPath(type, tail, width, height, x, y) {

        if (type === 'triangle' || type === 'arrow') {
            x += tail ? width : -width;
        }

        var path = new Path();
        switch (type) {
            case 'diamond':
                path.pushDiamond(x, y, width, height, 0);
                break;
            case 'triangle':
                path.pushTriangle(x, y, width, height, 0, tail);
                break;
            case 'arrow':
                path.pushArrow(x, y, width, height, 0, tail);
                break;
            case 'oval':
                path.pushEllipse(x, y, width, height, 0);
                break;
        }
        return path;
    }

    /**
     * Creates a bitmap for the specified border line style, and returns its
     * data URL.
     *
     * @param {String} style
     *  The arrow style identifier (two arrow type tokens, separated by a
     *  colon).
     *
     * @param {String} color
     *  The effective CSS line color.
     *
     * @returns {String}
     *  The data URL of the generated bitmap.
     */
    function getArrowStyleBitmapUrl(style, color) {

        // return data URL of a bitmap already created, or create a new bitmap
        return bitmapUrlMap.getOrCreate(style + ',' + color, function () {

            canvas.clear().render(function (context, width, height) {

                var arrows = style.split(':');
                var head = arrows[0];
                var tail = arrows[1];

                var y = Math.floor(height / 2) - 0.5;
                var arrowWidth = height / 2;
                var arrowHeight = height / 2;

                context.setLineStyle({ style: color, width: 1 });
                context.setFillStyle(color);

                var lineStartX  = arrowWidth;
                var lineEndX = width - arrowWidth;

                var headPath = createArrowPath(head, false, arrowWidth - 1, arrowHeight - 1, lineStartX, y);
                var tailPath = createArrowPath(tail, true, arrowWidth - 1, arrowHeight - 1, lineEndX, y);

                if (head === 'none' || head === 'arrow') { lineStartX = 1; }
                if (tail === 'none' || tail === 'arrow') { lineEndX = width - 1; }

                context.drawLine(lineStartX, y, lineEndX, y);
                context.drawPath(headPath, (head === 'arrow') ? 'stroke' : 'fill');
                context.drawPath(tailPath, (tail === 'arrow') ? 'stroke' : 'fill');
            });

            // convert the bitmap to a data URL
            return canvas.getDataURL();
        });
    }

    // class ArrowStylePicker ================================================

    /**
     * A generic drop-down list control for border styles.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Array<Object>} listEntries
     *  An array of descriptors for the entries of the drop-down list. Each
     *  descriptor is an object with the following properties:
     *  - {String} value
     *      The value associated to the list item.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  RadioList.
     */
    var ArrowStylePicker = RadioList.extend({ constructor: function (docView, listEntries, initOptions) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions({
            icon: 'fa-exchange',
            //#. arrow styles at the end of line and connector objects
            label: gt.pgettext('arrows', 'Arrow style'),
            //#. arrow styles at the end of line and connector objects
            tooltip: gt.pgettext('arrows', 'Change line endings'),
            updateCaptionMode: 'none',
            smallerVersion: { hideLabel: true },
            dropDownVersion: { label: Utils.getStringOption(initOptions, 'tooltip') }
        }, initOptions));

        // initialization -----------------------------------------------------

        // register a handler for menu item creation for the dropdown list.
        this.listenOnceTo(this.getMenu(), 'popup:beforeshow', function () {
            listEntries.forEach(function (entry) {
                var buttonNode = self.getMenu().createItemNode(entry.value);
                var bitmapUrl1 = getArrowStyleBitmapUrl(entry.value, Forms.LessVar.TEXT_COLOR);
                var bitmapUrl2 = getArrowStyleBitmapUrl(entry.value, Forms.LessVar.LIST_SELECTED_TEXT_COLOR);
                Forms.setCaptionBitmapIcon(buttonNode, bitmapUrl1, bitmapUrl2);
            });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = listEntries = initOptions = null;
        });

    } }); // class ArrowStylePicker

    // exports ================================================================

    return ArrowStylePicker;

});
