/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/indexedcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/drawinglayer/model/drawingcollection'
], function (Utils, Iterator, DrawingCollection) {

    'use strict';

    // convenience shortcuts
    var ArrayIterator = Iterator.ArrayIterator;

    // class IndexedCollection ================================================

    /**
     * Represents a drawing collection that stores and addresses its drawing
     * models by a simple integral index.
     *
     * Additionally to the events that will be triggered by the base class
     * DrawingCollection, an instance of this class triggers the following
     * events:
     * - 'move:drawing'
     *      After a drawing model in this collection has been moved to a new
     *      index. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The JQuery event object.
     *      (2) {DrawingModel} drawingModel
     *          The drawing model that has been moved.
     *      (3) {Number} fromIndex
     *          The old sorting index of the drawing model before it has been
     *          moved to its new position.
     *
     * @constructor
     *
     * @extends DrawingCollection
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {Object} initOptions
     *  Optional parameters:
     *  - {DrawingModel} [initOptions.parentModel]
     *      The parent drawing model owning this instance as embedded drawing
     *      collection. If omitted, this instance will be a root collection
     *      without parent drawing model.
     */
    var IndexedCollection = DrawingCollection.extend({ constructor: function (docModel, modelFactory, initOptions) {

        // maps drawing model indexes (used as positions) to drawing models
        var drawingModels = [];

        // base constructor ---------------------------------------------------

        DrawingCollection.call(this, docModel, modelFactory, Utils.extendOptions(initOptions, {
            clientResolveModelHandler: resolveDrawingModel,
            clientResolvePositionHandler: resolveDrawingPosition,
            clientResolveIndexHandler: resolveDrawingIndex,
            clientRegisterModelHandler: registerDrawingModel,
            clientUnregisterModelHandler: unregisterDrawingModel
        }));

        // private methods ----------------------------------------------------

        /**
         * Tries to return a drawing model contained in this sheet at the
         * specified position.
         *
         * @param {Array<Number>} position
         *  An arbitrary document position.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified position, if available;
         *  otherwise null.
         */
        function resolveDrawingModel(position) {
            return ((position.length === 1) && drawingModels[position[0]]) || null;
        }

        /**
         * Tries to return the position of the specified drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  An arbitrary drawing model.
         *
         * @param {Array<Number>|Null}
         *  The position of the passed drawing model, if it is part of this
         *  drawing collection, otherwise null.
         */
        function resolveDrawingPosition(drawingModel) {
            var index = drawingModels.indexOf(drawingModel);
            return (index >= 0) ? [index] : null;
        }

        /**
         * Tries to return the Z order index of the specified drawing model.
         *
         * @param {DrawingModel} drawingModel
         *  An arbitrary drawing model.
         *
         * @param {Number|Null}
         *  The Z order index of the passed drawing model, if it is part of
         *  this drawing collection, otherwise null.
         */
        function resolveDrawingIndex(drawingModel) {
            var index = drawingModels.indexOf(drawingModel);
            return (index >= 0) ? index : null;
        }

        /**
         * Registers a new drawing model that has just been inserted into the
         * drawing collection of this sheet.
         *
         * @param {DrawingModel} drawingModel
         *  The new drawing model.
         *
         * @param {Array<Number>} position
         *  The position of the new drawing object.
         *
         * @returns {Boolean}
         *  Whether the passed position is valid, and the drawing model has
         *  been registered successfully.
         */
        function registerDrawingModel(drawingModel, position) {
            var index = (position.length === 1) ? position[0] : -1;
            if ((index >= 0) && (index <= drawingModels.length)) {
                drawingModels.splice(index, 0, drawingModel);
                return true;
            }
            return false;
        }

        /**
         * Unregisters a drawing model that has been deleted from the drawing
         * collection of this sheet.
         *
         * @param {DrawingModel} drawingModel
         *  The deleted drawing model.
         *
         * @param {Array<Number>} position
         *  The position of the deleted drawing object.
         *
         * @returns {Boolean}
         *  Whether the passed position is valid, and the drawing model has
         *  been unregistered successfully.
         */
        function unregisterDrawingModel(drawingModel, position) {
            var index = (position.length === 1) ? position[0] : -1;
            if (drawingModel === drawingModels[index]) {
                drawingModels.splice(index, 1);
                return true;
            }
            return false;
        }

        // public methods -----------------------------------------------------

        /**
         * Creates an iterator that visits all drawing models in this drawing
         * collection. The drawing models will be visited in order of their
         * document positions (Z order).
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options of a generic array
         *  iterator (see class Iterator.ArrayIterator for details); and the
         *  following options:
         *  - {Boolean} [options.deep=false]
         *      If set to true, all drawing objects embedded in other drawing
         *      objects will be visited too.
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible drawing objects will be visited.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {DrawingModel} value
         *      The drawing model currently visited.
         *  - {Number} index
         *      The array index of the drawing model in this collection.
         */
        this.createModelIterator = function (options) {
            // create an array iterator for the drawing models
            var iterator = new ArrayIterator(drawingModels, options);
            // filter the drawing models according to the passed options
            return this.implCreateModelIterator(iterator, options);
        };

        /**
         * Moves the model of a drawing object located at the passed document
         * position to a new sorting index.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object (if the array without the last
         *  element points to an existing drawing object).
         *
         * @param {Number} index
         *  The new sorting index for the drawing model inside its parent
         *  collection.
         *
         * @returns {Boolean}
         *  Whether the drawing model has been moved successfully.
         */
        this.moveModel = function (position, index) {

            // find the drawing object in this collection
            var model = (position.length > 0) ? drawingModels[position[0]] : null;
            if (!model) { return false; }

            // remaining elements: move a drawing model in an embedded collection
            if (position.length > 1) {
                var childCollection = model.getChildCollection();
                return !!childCollection && childCollection.moveModel(position.slice(1), index);
            }

            // validate the target index
            if ((index < 0) || (index >= drawingModels.length)) { return false; }

            // nothing to do if the index does not change
            var oldIndex = position[0];
            if (oldIndex === index) { return true; }

            // remove drawing mode from old position, and insert it at the new position
            drawingModels.splice(oldIndex, 1);
            drawingModels.splice(index, 0, model);

            // notify all listeners
            this.trigger('move:drawing', model, oldIndex);
            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members
        this.registerDestructor(function () {
            // drawing models are owned by the base class DrawingCollection, do not destroy them here!
            docModel = drawingModels = null;
        });

    } }); // class IndexedCollection

    // exports ================================================================

    return IndexedCollection;

});
