/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/app/applicationlauncher', [
    'io.ox/core/extensions',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/config',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/baseframework/view/baselabels',
    'less!io.ox/office/baseframework/view/basestyle'
], function (ext, Utils, Config, Forms, Scheduler, DriveUtils, Labels) {

    'use strict';

    // private global functions ===============================================

    /**
     * Creates and initializes an application window for an application with
     * the specified module name.
     *
     * @param {ox.ui.App} app
     *  The application to create the window for.
     *
     * @param {jQuery} [rootNode]
     *  The root application node to embed the window into.
     *
     * @returns {ox.ui.Window}
     *  The application window.
     */
    function createAppWindow(app, rootNode) {

        // create the core application window
        var appWindow = ox.ui.createWindow({ name: app.getName(), search: false, chromeless: true, page: rootNode });
        app.setWindow(appWindow);

        // bug 53506: track whether the busy mode is currently active
        var isBusy = false;

        // add the new method isBusy()
        appWindow.isBusy = function () {
            return isBusy;
        };

        // override the method busy() to add specific behavior
        appWindow.busy = _.wrap(appWindow.busy, function (busyMethod, pct, sub, callback, options) {

            // invoke core method
            return busyMethod.call(this, pct, sub, function () {

                // the window blocker element (bound to 'this')
                var blockerNode = this;
                // the header/footer container nodes
                var headerNode = blockerNode.find('.header');
                var footerNode = blockerNode.find('.footer');

                // initialization when busy mode starts
                if (!isBusy) {
                    isBusy = true;

                    // clean header/footer container nodes
                    headerNode.empty();
                    footerNode.empty();

                    // add special marker class for custom CSS formatting
                    blockerNode.addClass('io-ox-office-blocker').css('opacity', '');

                    // add top label to header area
                    var topLabel = Utils.getStringOption(options, 'topLabel', null);
                    if (topLabel) {
                        headerNode.append($('<div class="clear-title">').text(topLabel));
                    }

                    // bug 59041: add a custom quit button (no quit button anymore on top-bar application tabs)
                    var quitButton = $(Forms.createButtonMarkup(Labels.QUIT_BUTTON_OPTIONS));
                    quitButton.addClass('quit-button').on('click', function () {
                        app.trigger('docs:busy:quit'); // 53941 -> quit button calls cancel handler
                        app.quit();
                    });
                    headerNode.append(quitButton);

                    // on IE, remove stripes (they are displayed too nervous and flickering)
                    if (_.browser.IE) {
                        blockerNode.find('.progress-striped').removeClass('progress-striped');
                    }
                }

                // always invoke the user-defined callback function
                if (_.isFunction(callback)) {
                    return callback.call(appWindow, blockerNode, headerNode, footerNode);
                }
            });
        });

        // override the method idle() to add specific behavior
        appWindow.idle = _.wrap(appWindow.idle, function (idleMethod) {
            if (isBusy) {
                idleMethod.call(this);
                isBusy = false;
            }
            return this;
        });

        return appWindow;
    }

    // static class ApplicationLauncher =======================================

    var ApplicationLauncher = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns the document type from the passed application module name.
     * The document type equals the last directory name of the module name.
     *
     * @param {String} moduleName
     *  The application module name.
     *
     * @returns {String}
     *  The document type of the passed module name.
     */
    ApplicationLauncher.getDocumentType = function (moduleName) {
        return moduleName.substr(moduleName.lastIndexOf('/') + 1);
    };

    /**
     * Returns the stand-alone mode from the server-side configuration flag
     * 'standalone', or the application's stand-alone URL hash fragment. This
     * function also checks if the application is loaded in an <iframe> and
     * gets the correct hash fragment from the it's source URL.
     *
     * @returns {Boolean}
     *  The state of the stand-alone mode.
     */
    ApplicationLauncher.getStandaloneMode = function () {
        if (Config.getFlag('module/standalone')) { return true; }
        var standalone = (window.self !== window.top) ? _.deserialize(window.self.location.hash.substr(1)).standalone : _.url.hash('standalone');
        return standalone ? (standalone.toLowerCase() === 'true' || standalone === '1') : false;
    };

    /**
     * Returns a running application instance for the specified file descriptor
     * if existing.
     *
     * @param {String} moduleName
     *  The application type identifier.
     *
     * @param {Object} fileDesc
     *  A file descriptor containing the file identifier, file name, and other
     *  important settings for a file.
     *
     * @returns {BaseApplication|Null}
     *  A running application instance for the specified file descriptor if
     *  existing; otherwise null.
     */
    ApplicationLauncher.getRunningApp = function (moduleName, fileDesc) {

        // find running application for the specified module type
        var runningApps = ox.ui.App.get(moduleName).filter(function (app) {
            return app.isEqualFileDescriptor && app.isEqualFileDescriptor(fileDesc);
        });

        // return running application if existing
        if (runningApps.length > 1) {
            Utils.warn('LaunchApplication.getRunningApp(): found multiple applications for the same file.');
        }
        return runningApps[0] || null;
    };

    /**
     * Creates a static launcher that has to be returned by the main module of
     * each application type. The launcher tries to find a running application
     * which is working on a file described in the launch options passed to the
     * launcher. If no such application exists, it creates and returns a new
     * application object.
     *
     * @param {String} moduleName
     *  The application type identifier.
     *
     * @param {Object} [appOptions]
     *  Optional parameters controlling the creation of the ox.ui.App base
     *  class instance:
     *  @param {String} [appOptions.icon]
     *      If specified, the CSS class name of a Bootstrap icon that will be
     *      shown in the top level launcher tab next to the application title.
     *  @param {String} [appOptions.trackingId = assigned moduleName]
     *      trackingId is a unique String as moduleName but better readable
     *      used by CoreMetrics' tracking
     *
     * @returns {Object}
     *  The launcher object expected by the method ox.launch().
     */
    ApplicationLauncher.createLauncher = function (moduleName, appOptions) {

        // hide core AppSuite topbar in standalone mode
        if (ApplicationLauncher.getStandaloneMode()) {
            ext.point('io.ox/core/topbar').disable('default');
        }

        // callback method invoked from ox.launch()
        function getApp(launchOptions) {

            // set application title
            function setAppTitle(fileDesc) {
                if (!app || !app.getWindow() || isViewerMode) { return; }

                // the name of the edited file, with and without extension
                var fullFileName = DriveUtils.getSanitizedFileName(fileDesc, '\xa0');
                var shortFileName = DriveUtils.getFileBaseName(fullFileName);

                app.setTitle(shortFileName);
                app.getWindow().setTitle(shortFileName);
            }

            // returns the portal app instance
            function getPortalApp() {
                return ox.ui.apps.find(function (app) {
                    return app.getWindow() && (app.getName() === 'io.ox/portal');
                });
            }

            // handles 'beforeshow' events of the window
            // special handling for smatphones, make sure the Portal window is closed when a Documents app is launched - bug #57537
            function windowBeforeShowHandler() {
                var portal = getPortalApp();
                if (portal && !isViewerMode) {
                    portal.getWindow().hide();
                }
            }

            // Load from URL: different launch options are given, check if we have 'simple' file descriptor (file id, folder id)
            // in the URL hash params and build new simple launch options from it
            if (!launchOptions || launchOptions.id) {
                if (Config.getUrlFlag('convert')) {
                    launchOptions = {
                        action: 'convert',
                        file: null,
                        template: true,
                        keepFile: true,
                        target_filename: _.url.hash('destfilename'),
                        target_folder_id: _.url.hash('destfolderid'),
                        templateFile: { folder_id: launchOptions.folder, id: launchOptions.id }
                    };
                    _.url.hash('convert', null);
                    _.url.hash('destfilename', null);
                    _.url.hash('destfolderid', null);
                } else {
                    launchOptions = { action: 'load', file: { folder_id: launchOptions.folder, id: launchOptions.id, source: 'drive' } };
                }
                launchOptions.actionSource = 'url';
            }

            // get file descriptor from options
            var fileDesc = Utils.getObjectOption(launchOptions, 'file', null);
            // workaround for missing source info, if missing it must be drive
            if (fileDesc && !fileDesc.source) { fileDesc.source = 'drive'; }

            // whether the application was launched by the OX Viewer
            var isViewerMode = (Utils.getStringOption(launchOptions, 'mode', null) === 'viewer-mode');

            // find running application for the specified document
            if (!isViewerMode) {
                var runningApp = ApplicationLauncher.getRunningApp(moduleName, fileDesc);
                if (runningApp && (runningApp.getLaunchOption('mode') !== 'viewer-mode')) { return runningApp; }
            }

            // the icon shown in the top bar launcher
            var icon = Utils.getStringOption(appOptions, 'icon');
            icon = _.isString(icon) ? Forms.getIconClasses(icon) : '';

            // no running application: create and initialize a new application object
            var app = ox.ui.createApp({
                name: moduleName,
                closable: true,
                userContent: icon.length > 0,
                userContentIcon: icon,
                userContentClass: ApplicationLauncher.getDocumentType(moduleName) + '-launcher',
                trackingId: Utils.getStringOption(appOptions, 'trackingId', moduleName),
                help: Utils.getOption(appOptions, 'help'),
                hideTaskbarEntry: isViewerMode  // the app plugged into the Viewer should not have a taskbar entry
            });

            // create and initialize the application window
            var appWindow = createAppWindow(app, Utils.getOption(launchOptions, 'page'));

            // special handling for smatphones, make sure the Portal window is closed when a Documents app is launched - bug #57537
            if (_.device('smartphone')) {
                appWindow.on('beforeshow', windowBeforeShowHandler);
            }

            // the name of the edited file, with and without extension
            var fullFileName = DriveUtils.getSanitizedFileName(fileDesc, '\xa0');

            // bug 38117: a title must be set to always get the launcher button in the top bar
            setAppTitle(fileDesc);

            // bug 46405: override the launch method of the application instance to be able to fetch
            // the application mix-in class on demand, and to defer invocation of the original launch
            // method until the application has been extended with that mix-in class
            app.setLauncher(function () {

                // check that some launch options have been passed (action must always be part)
                if (!_.isObject(launchOptions) || !_.isString(launchOptions.action)) {
                    var windowId = appWindow.nodes.outer.attr('id');
                    return Scheduler.createDeferred(windowId, 'ApplicationLauncher.setLauncher');
                }

                // take start time of application launch for performance logging
                launchOptions.appLaunchTime = _.now();

                // promise chain for authorization and launching
                var promise = $.when();

                // show a busy blocker screen to give the user a quick feedback that the app is starting
                if (launchOptions.hidden !== true) {
                    appWindow.show();
                    appWindow.busy(null, null, null, { topLabel: fullFileName });
                }

                // Bug 52041: After a browser refresh, the file name does not exist in the launch
                // options object. To check whether the file is encrypted, the file descriptor
                // needs to be fetched from the server.
                if (fileDesc && !fileDesc.filename && (fileDesc.source === 'drive')) {
                    Utils.warn('ApplicationLauncher: missing file name, requesting from server');
                    promise = promise.then(function () {
                        return DriveUtils.getFile(fileDesc).done(function (file) {
                            fullFileName = DriveUtils.getSanitizedFileName(file, '\xa0');
                            setAppTitle(file);
                            if (appWindow.isBusy()) {
                                appWindow.idle().busy(null, null, null, { topLabel: fullFileName });
                            }
                        });
                    });
                }

                // add authorization code for encrypted files to launch options
                promise = promise.then(function () {
                    // the 'if' MUST be located inside the promise.then(), otherwise 'fullFileName'
                    // may not have been initialized yet (see above)
                    if (DriveUtils.hasGuardExt(fullFileName) && !launchOptions.auth_code) {
                        return DriveUtils.authorizeGuard({ data: fileDesc }).done(function (auth_code) {
                            launchOptions.auth_code = auth_code;
                        });
                    }
                });

                // load the application mix-in constructor for the current application type
                promise = promise.then(function () {
                    return require([app.getName() + '/app/application']).done(function (ApplicationClass) {

                        // the application may have been closed very quickly before RequireJS
                        // provides the application module
                        if (!app.cid) { return; }

                        // construct with mix-in application class (with the disadvantage that
                        // the application is not an instance of ApplicationClass)
                        ApplicationClass.call(app, launchOptions);
                    });
                });

                // close the application immediately if anything fails during launching
                promise.fail(function () {
                    app.quit();
                });

                // bug 46405: do not return a pending promise (otherwise it is possible to open
                // multiple application instances for the same document)
                return $.when();
            });

            return app;
        }

        // ox.launch() expects an object with the method getApp()
        return { getApp: getApp };
    };

    // exports ================================================================

    return ApplicationLauncher;
});
