/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/components/drawing/imagecropframe',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils'
], function (AppHelper, ImageCropFrame, DrawingFrame, AttributeUtils) {

    'use strict';

    // class ImageCropFrame ================================================

    describe('Textframework class ImageCropFrame', function () {

        // private helpers ----------------------------------------------------

        var //  the operations to be applied by the document model
            OPERATIONS = [
                { name: 'setDocumentAttributes', attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } } },
                { name: 'insertMasterSlide', id: 'masterId_1' },
                { name: 'insertLayoutSlide', id: 'layoutId_1', target: 'masterId_1' },
                { name: 'insertSlide', start: [0], target: 'layoutId_1' },
                { name: 'insertDrawing', start: [0, 0], type: 'image', attrs: { drawing: { id: '4000010', name: 'image1', width: 6000, height: 4000, top: 0, left: 0, aspectLocked: true }, image: { imageUrl: '' }, line: { type: 'none' } } }
            ],

            model = null,
            selection = null;

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            selection = model.getSelection();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ImageCropFrame).to.be.an('Object');
        });

        // public methods -----------------------------------------------------

        describe('method "ImageCropFrame.drawCropFrame"', function () {

            it('should exist', function () {
                expect(ImageCropFrame).to.respondTo('drawCropFrame');
            });

            it('should enter correct crop mode through models handleDrawingCrop, using drawCropFrame', function () {
                selection.setTextSelection([0, 0], [0, 1]);
                var drawing = selection.getSelectedDrawing();
                expect(drawing.hasClass(DrawingFrame.ACTIVE_CROPPING_CLASS)).to.be.equal(false);
                model.handleDrawingCrop(true);
                expect(drawing.hasClass(DrawingFrame.ACTIVE_CROPPING_CLASS)).to.be.equal(true);
                model.handleDrawingCrop(false);
                expect(drawing.hasClass(DrawingFrame.ACTIVE_CROPPING_CLASS)).to.be.equal(false);

            });
        });

        describe('method "ImageCropFrame.getDrawingMoveCropOp"', function () {

            it('should exist', function () {
                expect(ImageCropFrame).to.respondTo('getDrawingMoveCropOp');
            });

            it('should calculate correct crop values on drawing move event', function () {
                selection.setTextSelection([0, 0], [0, 1]);
                var drawing = selection.getSelectedDrawing();
                // simulating image node, as no url is provided with operation, and no url would work
                // 453px => 12000 hmm
                // 302px => 8000 hmm
                drawing.find('.placeholder').replaceWith($('<div class="content" style=""><div class="cropping-frame"><img src="" style="left: 0px; top: 0px; width: 453px; height: 302px; position: absolute;"></div></div>'));
                model.handleDrawingCrop(true);
                var oldAttrs = AttributeUtils.getExplicitAttributes(drawing).drawing;
                var newAttrs = { left: 0, top: 0 };
                var cropProps = ImageCropFrame.getDrawingMoveCropOp(drawing, oldAttrs, newAttrs);
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(50); // as the image is 50% bigger from the frame, width 12000 / 6000
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(50); // as the image is 50% bigger from the frame, height 8000 / 4000

                newAttrs = { left: 1000, top: 1000 }; // simulate move 1000hmm to right and bottom
                cropProps = ImageCropFrame.getDrawingMoveCropOp(drawing, oldAttrs, newAttrs);
                expect(cropProps.cropLeft).to.be.equal(8); // 1000/6000 = 16% * 50% = 8%
                expect(cropProps.cropRight).to.be.equal(42); // 50% - 8%
                expect(cropProps.cropTop).to.be.equal(13); // 1000/4000 = 25%* 50% = ~13%
                expect(cropProps.cropBottom).to.be.equal(37); // 50% - 13%

                newAttrs = { left: 2000, top: 2000 }; // simulate move 1000hmm to right and bottom
                cropProps = ImageCropFrame.getDrawingMoveCropOp(drawing, oldAttrs, newAttrs);
                expect(cropProps.cropLeft).to.be.equal(17); // 2000/6000 = 33% * 50% = ~17%
                expect(cropProps.cropRight).to.be.equal(33); // 50% - 17%
                expect(cropProps.cropTop).to.be.equal(25); // 2000/4000 = 50%* 50% = ~25%
                expect(cropProps.cropBottom).to.be.equal(25); // 50% - 25%
            });
        });

        describe('method "ImageCropFrame.getDrawingResizeCropOp"', function () {

            it('should exist', function () {
                expect(ImageCropFrame).to.respondTo('getDrawingResizeCropOp');
            });

            it('should calculate correct crop values on drawing resize event', function () {
                selection.setTextSelection([0, 0], [0, 1]);
                var drawing = selection.getSelectedDrawing();
                // simulating image node, as no url is provided with operation, and no url would work
                drawing.find('.placeholder').replaceWith($('<div class="content" style=""><div class="cropping-frame"><img src="" style="left: 0px; top: 0px; width: 453px; height: 302px; position: absolute;"></div></div>'));
                model.handleDrawingCrop(true);

                var oldAttrs = AttributeUtils.getExplicitAttributes(drawing).drawing;
                var newAttrs = { left: 0, top: 0 };
                var useLeft = false, useTop = false;
                var cropProps = ImageCropFrame.getDrawingResizeCropOp(drawing, oldAttrs, newAttrs, useLeft, useTop);
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(50); // as the image is 50% bigger from the frame, width 12000 / 6000
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(50); // as the image is 50% bigger from the frame, height 8000 / 4000

                newAttrs = { width: 9000 };
                cropProps = ImageCropFrame.getDrawingResizeCropOp(drawing, oldAttrs, newAttrs, useLeft, useTop);
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(25);
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(50); // as the image is 50% bigger from the frame, height 8000 / 4000

                newAttrs = { width: 12000, height: 6000 };
                cropProps = ImageCropFrame.getDrawingResizeCropOp(drawing, oldAttrs, newAttrs, useLeft, useTop);
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(0);
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(25);

                newAttrs = { width: 12000, height: 8000 };
                cropProps = ImageCropFrame.getDrawingResizeCropOp(drawing, oldAttrs, newAttrs, useLeft, useTop);
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(0);
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(0);
            });
        });

        describe('method "ImageCropFrame.calculateCropOperation"', function () {

            it('should exist', function () {
                expect(ImageCropFrame).to.respondTo('calculateCropOperation');
            });

            it('should calculate correct crop values basted on dialog fields', function () {
                // ImageCropFrame.calculateCropOperation(drawingWidth, drawingHeight, frameWidth, frameHeight, offsetX, offsetY);
                var cropProps = ImageCropFrame.calculateCropOperation(6000, 4000, 6000, 4000, 0, 0);

                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(0);
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(0);

                cropProps = ImageCropFrame.calculateCropOperation(6000, 4000, 12000, 8000, 0, 0);
                // dialog values are centered and with offset.
                // In this case img is 2x bigger from the frame and centered => 25% in all directions
                expect(cropProps.cropLeft).to.be.equal(25);
                expect(cropProps.cropRight).to.be.equal(25);
                expect(cropProps.cropTop).to.be.equal(25);
                expect(cropProps.cropBottom).to.be.equal(25);

                cropProps = ImageCropFrame.calculateCropOperation(6000, 4000, 12000, 8000, 3000, 0);
                // dialog values are centered and with offset.
                // In this case img is 2x bigger from the frame and aligned to left => left 0, right 50%
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(50);
                expect(cropProps.cropTop).to.be.equal(25);
                expect(cropProps.cropBottom).to.be.equal(25);

                cropProps = ImageCropFrame.calculateCropOperation(6000, 4000, 12000, 8000, -3000, 0);
                // dialog values are centered and with offset.
                // In this case img is 2x bigger from the frame and aligned to right => left 50%, right 0%
                expect(cropProps.cropLeft).to.be.equal(50);
                expect(cropProps.cropRight).to.be.equal(0);
                expect(cropProps.cropTop).to.be.equal(25);
                expect(cropProps.cropBottom).to.be.equal(25);

                cropProps = ImageCropFrame.calculateCropOperation(6000, 4000, 12000, 8000, 3000, 2000);
                // dialog values are centered and with offset.
                // In this case img is 2x bigger from the frame
                // and aligned to left and top => left 0, right 50%, top: 0%, bottom: 50%
                expect(cropProps.cropLeft).to.be.equal(0);
                expect(cropProps.cropRight).to.be.equal(50);
                expect(cropProps.cropTop).to.be.equal(0);
                expect(cropProps.cropBottom).to.be.equal(50);

                cropProps = ImageCropFrame.calculateCropOperation(6000, 4000, 12000, 8000, -1500, -1000);

                expect(cropProps.cropLeft).to.be.equal(37.5);
                expect(cropProps.cropRight).to.be.equal(12.5);
                expect(cropProps.cropTop).to.be.equal(37.5);
                expect(cropProps.cropBottom).to.be.equal(12.5);
            });
        });

    });
});
