/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/text/model/listhandlermixin',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/utils'
], function (AppHelper, AttributeUtils, ListHandlerMixin, DOM, Utils) {

    'use strict';

    // OX Text mix-in class ListHandlerMixin ==========================================

    describe('OX Text mix-in class ListHandlerMixin', function () {

        // private helpers ----------------------------------------------------

        var model = null;
        var pageNode = null;
        var pageContentNode = null;
        var allParagraphs = null;
        var para0 = null;
        var para1 = null;
        var para2 = null;
        var para3 = null;
        var para4 = null;
        var para5 = null;
        var para6 = null;
        var para7 = null;
        var para8 = null;
        var para9 = null;
        var predefinedNumberingListStyleBrackets = 'L30003';
        var predefinedNumberingListStyleSmallLetter = 'L30011';
        var increaseListStartValue = 10;

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes',
                attrs: {
                    document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                    page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                    character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                    paragraph: { lineHeight: { type: 'percent', value: 119 }, marginBottom: 352 }
                }
            },
            { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading 1',
                attrs: {
                    character: { bold: true, fontName: 'Times New Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                    paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                },
                parent: 'Normal',
                uiPriority: 9
            },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', text: 'ABC_0', start: [0, 0] },
            { name: 'insertParagraph', start: [1] },
            { name: 'insertText', start: [1, 0], text: 'ABC_1' },
            { name: 'insertParagraph', start: [2] },
            { name: 'insertText', start: [2, 0], text: 'ABC_2' },
            { name: 'insertDrawing', attrs: { fill: { type: 'solid', color: { type: 'scheme', value: 'accent1' } }, geometry: { presetShape: 'rect' }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 50000 }] } }, shape: { anchor: 'centered', anchorCentered: false, paddingLeft: 667, paddingRight: 667, paddingTop: 468, paddingBottom: 468, wordWrap: true, horzOverflow: 'overflow', vertOverflow: 'overflow' }, drawing: { name: 'rect', width: 6667, height: 4683, anchorHorBase: 'column', anchorHorAlign: 'offset', anchorVertBase: 'paragraph', anchorVertAlign: 'offset', inline: false, textWrapMode: 'none', anchorHorOffset: 1217, anchorVertOffset: -25268 } }, start: [2, 5], type: 'shape' },
            { name: 'insertParagraph', start: [2, 5, 0], attrs: { character: { color: { type: 'scheme', value: 'light1' } }, paragraph: { alignment: 'center' } } },
            { name: 'insertText', text: 'ABC in Drawing', start: [2, 5, 0, 0] },
            { name: 'insertParagraph', start: [3] },
            { name: 'insertText', start: [3, 0], text: 'ABC_3' },
            { name: 'insertParagraph', start: [4] },
            { name: 'insertText', start: [4, 0], text: 'ABC_4' },
            { name: 'insertParagraph', start: [5] },
            { name: 'insertText', start: [5, 0], text: 'ABC_5' },
            { name: 'insertParagraph', start: [6] },
            { name: 'insertText', start: [6, 0], text: 'ABC_6' },
            { name: 'insertParagraph', start: [7] },
            { name: 'insertText', start: [7, 0], text: 'ABC_7' },
            { name: 'insertParagraph', start: [8] },
            { name: 'insertText', start: [8, 0], text: 'ABC_8' },
            { name: 'insertParagraph', start: [9] },
            { name: 'insertText', start: [9, 0], text: 'ABC_9' }
        ];

        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(ListHandlerMixin).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "getNode"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('getNode');
            });

            it('should exist also as getCurrentRootNode', function () {
                expect(model).to.respondTo('getCurrentRootNode');
            });

            it('should return the same node as getCurrentRootNode', function () {
                expect(Utils.getDomNode(model.getNode()) === Utils.getDomNode(model.getCurrentRootNode())).to.equal(true);
            });

            it('should have a correct prepared document with 10 paragraphs', function () {

                pageNode = model.getCurrentRootNode();
                pageContentNode = DOM.getPageContentNode(pageNode);
                allParagraphs = pageContentNode.children(DOM.CONTENT_NODE_SELECTOR);

                expect(allParagraphs.length).to.equal(10);

                para0 = $(allParagraphs[0]);
                para1 = $(allParagraphs[1]);
                para2 = $(allParagraphs[2]);
                para3 = $(allParagraphs[3]);
                para4 = $(allParagraphs[4]);
                para5 = $(allParagraphs[5]);
                para6 = $(allParagraphs[6]);
                para7 = $(allParagraphs[7]);
                para8 = $(allParagraphs[8]);
                para9 = $(allParagraphs[9]);

                expect(DOM.isParagraphNode(para0)).to.equal(true);
                expect(DOM.isParagraphNode(para1)).to.equal(true);
                expect(DOM.isParagraphNode(para2)).to.equal(true);
                expect(DOM.isParagraphNode(para3)).to.equal(true);
                expect(DOM.isParagraphNode(para4)).to.equal(true);
                expect(DOM.isParagraphNode(para5)).to.equal(true);
                expect(DOM.isParagraphNode(para6)).to.equal(true);
                expect(DOM.isParagraphNode(para7)).to.equal(true);
                expect(DOM.isParagraphNode(para8)).to.equal(true);
                expect(DOM.isParagraphNode(para9)).to.equal(true);
            });
        });

        describe('method "DOM.isListParagraphNode"', function () {
            it('should exist', function () {
                expect(DOM).to.respondTo('isListParagraphNode');
            });
            it('should contain no lists in the document', function () {
                expect(DOM.isListParagraphNode(para0)).to.equal(false);
                expect(DOM.isListParagraphNode(para1)).to.equal(false);
                expect(DOM.isListParagraphNode(para2)).to.equal(false);
                expect(DOM.isListParagraphNode(para3)).to.equal(false);
                expect(DOM.isListParagraphNode(para4)).to.equal(false);
                expect(DOM.isListParagraphNode(para5)).to.equal(false);
                expect(DOM.isListParagraphNode(para6)).to.equal(false);
                expect(DOM.isListParagraphNode(para7)).to.equal(false);
                expect(DOM.isListParagraphNode(para8)).to.equal(false);
                expect(DOM.isListParagraphNode(para9)).to.equal(false);
            });
        });

        describe('method "createDefaultList"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('createDefaultList');
            });

            it('should set the default numbering list style to the selected paragraph', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([1, 1], [5, 1]);

                model.createDefaultList('numbering');

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(true);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');

                    done();
                });
            });

            it('should generate a new list, if new selection is behind non-list paragraph', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([7, 1], [8, 1]);

                model.createDefaultList('numbering');

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(true);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(true); // !
                    expect(DOM.isListParagraphNode(para8)).to.equal(true); // !
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('4.');
                    expect(para5.children().first().text()).to.equal('5.');
                    expect(para7.children().first().text()).to.equal('1.'); // !
                    expect(para8.children().first().text()).to.equal('2.'); // !

                    done();
                });
            });

        });

        describe('method "setListStartValue"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('setListStartValue');
            });

            it('should set a new start value to the selected paragraph in the list', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([4, 1], [4, 2]);

                model.setListStartValue({ continue: false });

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(true);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(true);
                    expect(DOM.isListParagraphNode(para8)).to.equal(true);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('1.'); // !
                    expect(para5.children().first().text()).to.equal('2.'); // !
                    expect(para7.children().first().text()).to.equal('1.');
                    expect(para8.children().first().text()).to.equal('2.');

                    done();
                });
            });

            it('should continue the list at the selected paragraph', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([7, 1], [7, 2]);

                model.setListStartValue({ continue: true });

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(true);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(true);
                    expect(DOM.isListParagraphNode(para8)).to.equal(true);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('1.');
                    expect(para5.children().first().text()).to.equal('2.');
                    expect(para7.children().first().text()).to.equal('3.'); // !
                    expect(para8.children().first().text()).to.equal('4.'); // !

                    done();
                });
            });

            it('should do nothing, if first paragraph of list is selected and continue the list', function () {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([1, 1], [1, 2]);

                var operationCount = model.setListStartValue({ continue: true });

                expect(operationCount).to.equal(0); // no operation generated
            });

        });

        describe('method "createDefaultList"', function () {

            it('should expand the numbering list with the paragraph directly following the selected numbering list paragraph', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([6, 1], [6, 2]);

                model.createDefaultList('numbering');

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(true);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true); // !
                    expect(DOM.isListParagraphNode(para7)).to.equal(true);
                    expect(DOM.isListParagraphNode(para8)).to.equal(true);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para4.children().first().text()).to.equal('1.');
                    expect(para5.children().first().text()).to.equal('2.');
                    expect(para6.children().first().text()).to.equal('3.'); // !
                    expect(para7.children().first().text()).to.equal('4.'); // !
                    expect(para8.children().first().text()).to.equal('5.'); // !

                    done();
                });
            });
        });

        describe('method "removeListAttributes"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('removeListAttributes');
            });

            it('should remove the list attributes from selected paragraphs', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([4, 1], [8, 1]);

                model.removeListAttributes();

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(false);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');

                    done();
                });
            });
        });

        describe('method "createList"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('createList');
            });

            it('should set a list style to the selected paragraphs with increased start value', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([5, 0]);

                model.createList('numbering', { left: '', right: '.', listStartValue: increaseListStartValue, numberFormat: 'decimal', startPosition: [5, 0] });

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true); // !
                    expect(DOM.isListParagraphNode(para6)).to.equal(true); // !
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para5.children().first().text()).to.equal(increaseListStartValue + '.'); // !
                    expect(para6.children().first().text()).to.equal((increaseListStartValue + 1) + '.');  // !

                    done();
                });
            });
        });

        describe('method "setListStartValue"', function () {

            it('should set the start value of the selecected list with increased start number to the first value', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([5, 1], [5, 2]);

                model.setListStartValue({ continue: false }); // using 'continue: false' -> setting first start value

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para5.children().first().text()).to.equal('1.');
                    expect(para6.children().first().text()).to.equal('2.');

                    done();
                });
            });

            it('should set the previous increased start value via undo', function (done) {

                var undoPromise = model.getUndoManager().undo();
                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                // model.on('listformatting:done', $.noop); // avoid resolving 'listFormattingDef' too early

                undoPromise.always(function () {

                    expect(undoPromise.state()).to.equal('resolved');

                    model.on('listformatting:done', function () {
                        listFormattingDef.resolve();
                    });

                    listFormattingDef.always(function () {

                        expect(DOM.isListParagraphNode(para0)).to.equal(false);
                        expect(DOM.isListParagraphNode(para1)).to.equal(true);
                        expect(DOM.isListParagraphNode(para2)).to.equal(true);
                        expect(DOM.isListParagraphNode(para3)).to.equal(true);
                        expect(DOM.isListParagraphNode(para4)).to.equal(false);
                        expect(DOM.isListParagraphNode(para5)).to.equal(true);
                        expect(DOM.isListParagraphNode(para6)).to.equal(true);
                        expect(DOM.isListParagraphNode(para7)).to.equal(false);
                        expect(DOM.isListParagraphNode(para8)).to.equal(false);
                        expect(DOM.isListParagraphNode(para9)).to.equal(false);

                        // checking state after formatting of list
                        expect(para1.children().first().text()).to.equal('1.');
                        expect(para2.children().first().text()).to.equal('2.');
                        expect(para3.children().first().text()).to.equal('3.');
                        expect(para5.children().first().text()).to.equal(increaseListStartValue + '.');
                        expect(para6.children().first().text()).to.equal((increaseListStartValue + 1) + '.');

                        done();
                    });
                });
            });

            it('should set the start value of the selecected list with increased start number to the first value', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([5, 1], [5, 2]);

                model.setListStartValue({ continue: true }); // using 'continue: true' -> continuing previous list

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('1.');
                    expect(para2.children().first().text()).to.equal('2.');
                    expect(para3.children().first().text()).to.equal('3.');
                    expect(para5.children().first().text()).to.equal('4.');
                    expect(para6.children().first().text()).to.equal('5.');

                    done();
                });
            });
        });

        describe('method "createSelectedListStyle"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('createSelectedListStyle');
            });

            it('should modify the list style, even if the list is splitted in two parts', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([1, 1]); // no range selected!

                model.createSelectedListStyle(predefinedNumberingListStyleBrackets, 0);

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('(2)');
                    expect(para3.children().first().text()).to.equal('(3)');
                    expect(para5.children().first().text()).to.equal('(4)');
                    expect(para6.children().first().text()).to.equal('(5)');

                    done();
                });
            });

        });

        describe('method "setAttribute"', function () {

            it('should exist', function () {
                expect(model).to.respondTo('setAttribute');
            });

            it('should increase the list level of the selected paragraphs', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([2, 1], [3, 1]); // selecting two paragraphs

                model.setAttribute('paragraph', 'listLevel', 1); // increasing the list level of selected paragraphs

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('a.');  // !
                    expect(para3.children().first().text()).to.equal('b.');  // !
                    expect(para5.children().first().text()).to.equal('(2)'); // !
                    expect(para6.children().first().text()).to.equal('(3)'); // !

                    done();
                });
            });

        });

        describe('method "setListStartValue"', function () {

            it('should set the start value of a paragraph with increase list level without changing paragraphs with lower list levels', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([3, 1]);

                model.setListStartValue({ continue: false }); // using 'continue: false' -> setting first start value

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('a.');
                    expect(para3.children().first().text()).to.equal('a.');  // !
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para6.children().first().text()).to.equal('(3)');

                    done();
                });
            });

            it('should continue a numbering list of a paragraph with increase list level without changing paragraphs with lower list levels', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([3, 1]);

                model.setListStartValue({ continue: true }); // using 'continue: true' -> continue previous list

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('a.');
                    expect(para3.children().first().text()).to.equal('b.');  // !
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para6.children().first().text()).to.equal('(3)');

                    done();
                });
            });

        });

        describe('method "setAttribute"', function () {

            it('should increase the list level of the selected paragraphs and modify the numbering of the affected list', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([3, 1]);

                model.setAttribute('paragraph', 'listLevel', 2); // increasing the list level of selected paragraph

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('a.');
                    expect(para3.children().first().text()).to.equal('A.');  // !
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para6.children().first().text()).to.equal('(3)');

                    done();
                });
            });

            it('should decrease the list level of the selected paragraphs and modify the numbering of the affected list with list level 1', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([3, 1]);

                model.setAttribute('paragraph', 'listLevel', 1); // decreasing the list level of selected paragraph

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('a.');
                    expect(para3.children().first().text()).to.equal('b.');  // !
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para6.children().first().text()).to.equal('(3)');

                    done();
                });
            });

            it('should decrease the list level of the selected paragraphs and modify the numbering of the affected list with list level 0', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([3, 1]);

                model.setAttribute('paragraph', 'listLevel', 0); // decreasing the list level of selected paragraph

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(true);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para1.children().first().text()).to.equal('(1)');
                    expect(para2.children().first().text()).to.equal('a.');
                    expect(para3.children().first().text()).to.equal('(2)'); // !
                    expect(para5.children().first().text()).to.equal('(3)'); // !
                    expect(para6.children().first().text()).to.equal('(4)'); // !

                    done();
                });
            });

        });

        describe('method "removeListAttributes"', function () {

            it('should remove all list attributes indenpendent from list level or paragraph list style', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([0, 1], [9, 1]);

                model.removeListAttributes();

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(false);
                    expect(DOM.isListParagraphNode(para2)).to.equal(false);
                    expect(DOM.isListParagraphNode(para3)).to.equal(false);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(false);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    done();
                });
            });
        });

        describe('method "setAttributes"', function () {

            it('should set paragraph style "Heading" to selected paragraphs', function () {

                var explicitAttributes = null;
                var paraStyle = 'Heading1';

                model.getSelection().setTextSelection([0, 1]);
                model.setAttributes('paragraph', { styleId: paraStyle }, { clear: true });

                model.getSelection().setTextSelection([5, 1]);
                model.setAttributes('paragraph', { styleId: paraStyle }, { clear: true });

                explicitAttributes = AttributeUtils.getExplicitAttributes(para0);
                expect(explicitAttributes.styleId).to.equal(paraStyle);

                explicitAttributes = AttributeUtils.getExplicitAttributes(para5);
                expect(explicitAttributes.styleId).to.equal(paraStyle);
            });
        });

        describe('method "createSelectedListStyle"', function () {

            it('should set a specified list style to first heading paragraph and count correctly', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([0, 1]);

                model.createSelectedListStyle(predefinedNumberingListStyleBrackets, 0);

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(false);
                    expect(DOM.isListParagraphNode(para2)).to.equal(false);
                    expect(DOM.isListParagraphNode(para3)).to.equal(false);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(false);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');

                    done();
                });
            });

            it('should set a specified list style to second heading paragraph and count correctly', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([5, 1]);

                model.createSelectedListStyle(predefinedNumberingListStyleBrackets, 0);

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(false);
                    expect(DOM.isListParagraphNode(para2)).to.equal(false);
                    expect(DOM.isListParagraphNode(para3)).to.equal(false);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');
                    expect(para5.children().first().text()).to.equal('(2)');

                    done();
                });
            });

            it('should set a specified list style to non-headings paragraphs and count correctly, even if the same list style is used', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([1, 1], [3, 1]);

                model.createSelectedListStyle(predefinedNumberingListStyleBrackets, 0);

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');
                    expect(para1.children().first().text()).to.equal('(1)'); // ! same list style, but not continuing the list in heading
                    expect(para2.children().first().text()).to.equal('(2)');
                    expect(para3.children().first().text()).to.equal('(3)');
                    expect(para5.children().first().text()).to.equal('(2)');

                    done();
                });
            });

            it('should set a specified list style to non-headings paragraphs and count correctly', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([1, 1], [3, 1]);

                model.createSelectedListStyle(predefinedNumberingListStyleSmallLetter, 0);

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');
                    expect(para1.children().first().text()).to.equal('a.');
                    expect(para2.children().first().text()).to.equal('b.');
                    expect(para3.children().first().text()).to.equal('c.');
                    expect(para5.children().first().text()).to.equal('(2)');

                    done();
                });
            });

            it('should set a specified list style to second block of non-headings paragraphs and count correctly', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([7, 1], [9, 1]);

                model.createSelectedListStyle(predefinedNumberingListStyleSmallLetter, 0);

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(true);
                    expect(DOM.isListParagraphNode(para8)).to.equal(true);
                    expect(DOM.isListParagraphNode(para9)).to.equal(true);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');
                    expect(para1.children().first().text()).to.equal('a.');
                    expect(para2.children().first().text()).to.equal('b.');
                    expect(para3.children().first().text()).to.equal('c.');
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para7.children().first().text()).to.equal('a.');
                    expect(para8.children().first().text()).to.equal('b.');
                    expect(para9.children().first().text()).to.equal('c.');

                    done();
                });
            });

            it('should resume an existing list independent from heading paragraphs', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([7, 1]);

                model.setListStartValue({ continue: true });

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(true);
                    expect(DOM.isListParagraphNode(para8)).to.equal(true);
                    expect(DOM.isListParagraphNode(para9)).to.equal(true);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');
                    expect(para1.children().first().text()).to.equal('a.');
                    expect(para2.children().first().text()).to.equal('b.');
                    expect(para3.children().first().text()).to.equal('c.');
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para7.children().first().text()).to.equal('d.');
                    expect(para8.children().first().text()).to.equal('e.');
                    expect(para9.children().first().text()).to.equal('f.');

                    done();
                });
            });

            it('should reset start value of an existing list independent from heading paragraphs', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([2, 1]);

                model.setListStartValue({ continue: false });

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(true);
                    expect(DOM.isListParagraphNode(para1)).to.equal(true);
                    expect(DOM.isListParagraphNode(para2)).to.equal(true);
                    expect(DOM.isListParagraphNode(para3)).to.equal(true);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(true);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(true);
                    expect(DOM.isListParagraphNode(para8)).to.equal(true);
                    expect(DOM.isListParagraphNode(para9)).to.equal(true);

                    // checking state after formatting of list
                    expect(para0.children().first().text()).to.equal('(1)');
                    expect(para1.children().first().text()).to.equal('a.');
                    expect(para2.children().first().text()).to.equal('a.');
                    expect(para3.children().first().text()).to.equal('b.');
                    expect(para5.children().first().text()).to.equal('(2)');
                    expect(para7.children().first().text()).to.equal('c.');
                    expect(para8.children().first().text()).to.equal('d.');
                    expect(para9.children().first().text()).to.equal('e.');

                    done();
                });
            });
        });

        describe('method "removeListAttributes"', function () {

            it('should remove all list attributes indenpendent from the paragraph styles', function (done) {

                var listFormattingDef = $.Deferred(); // promise that will be resolved after list formatting is completed

                model.on('listformatting:done', function () {
                    listFormattingDef.resolve();
                });

                model.getSelection().setTextSelection([0, 1], [9, 1]);

                model.removeListAttributes();

                listFormattingDef.always(function () {

                    expect(listFormattingDef.state()).to.equal('resolved');

                    expect(DOM.isListParagraphNode(para0)).to.equal(false);
                    expect(DOM.isListParagraphNode(para1)).to.equal(false);
                    expect(DOM.isListParagraphNode(para2)).to.equal(false);
                    expect(DOM.isListParagraphNode(para3)).to.equal(false);
                    expect(DOM.isListParagraphNode(para4)).to.equal(false);
                    expect(DOM.isListParagraphNode(para5)).to.equal(false);
                    expect(DOM.isListParagraphNode(para6)).to.equal(false);
                    expect(DOM.isListParagraphNode(para7)).to.equal(false);
                    expect(DOM.isListParagraphNode(para8)).to.equal(false);
                    expect(DOM.isListParagraphNode(para9)).to.equal(false);

                    done();
                });
            });
        });

    });
});
