/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/editframework/utils/operationerror',
    'io.ox/office/editframework/model/operationcontext',
    'io.ox/office/spreadsheet/model/operationcontext'
], function (AppHelper, OperationError, OperationContext, SheetOperationContext) {

    'use strict';

    // class SheetOperationContext ============================================

    describe('Spreadsheet class SheetOperationContext', function () {

        it('should exist', function () {
            expect(SheetOperationContext).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // base properties for the test operation that are not convertible to addresses, ranges, etc.
        var operation = { name: 'test', num: 1.5, obj: { a: 1 }, arr: [1] };

        // initialize test document
        var docModel = null, addrContext = null, addrListContext;
        var jsonIntContext = null, jsonAddrContext = null, jsonRangeContext = null, jsonRangeListContext = null;
        AppHelper.createSpreadsheetApp('ooxml').done(function (app) {
            docModel = app.getModel();

            // test context for cell addresses
            addrContext = new SheetOperationContext(docModel, _.extend({
                address1: 'A1',
                address2: 'BB99',
                address3: 'ZZZZ1',
                address4: 'A999999999'
            }, operation), false);
            addrContext.bindGet = function (name) { return this.getAddress.bind(this, name); };
            addrContext.bindOpt = function (name) { return this.getOptAddress.bind(this, name); };

            // test context for cell address lists
            addrListContext = new SheetOperationContext(docModel, _.extend({
                addresses1: 'A1',
                addresses2: 'A1 B2 \t BB99',
                addresses3: 'A1 ZZZZ1',
                addresses4: 'A1 42'
            }, operation), false);
            addrListContext.bindGet = function (name) { return this.getAddressList.bind(this, name); };
            addrListContext.bindOpt = function (name) { return this.getOptAddressList.bind(this, name); };

            // test context for column/row intervals
            jsonIntContext = new SheetOperationContext(docModel, _.extend({
                interval1: { start: 1, end: 999999 },
                interval2: { start: 1 },
                interval3: { start: 'a' },
                interval4: { start: Number.NaN },
                start: 2
            }, operation), false);
            jsonIntContext.bindGet = function (name, columns) { return this.getJSONInterval.bind(this, name, columns); };
            jsonIntContext.bindOpt = function (name, columns) { return this.getOptJSONInterval.bind(this, name, columns); };

            // test context for cell addresses
            jsonAddrContext = new SheetOperationContext(docModel, _.extend({
                address1: [1, 2],
                address2: [999999, 2],
                address3: [null, 2],
                start: [2, 3]
            }, operation), false);
            jsonAddrContext.bindGet = function (name) { return this.getJSONAddress.bind(this, name); };
            jsonAddrContext.bindOpt = function (name) { return this.getOptJSONAddress.bind(this, name); };

            // test context for cell range addresses
            jsonRangeContext = new SheetOperationContext(docModel, _.extend({
                range1: { start: [1, 2], end: [3, 4] },
                range2: { start: [1, 2] },
                range3: { start: [1, 2], end: [3, null] },
                start: [2, 3]
            }, operation), false);
            jsonRangeContext.bindGet = function (name) { return this.getJSONRange.bind(this, name); };
            jsonRangeContext.bindOpt = function (name) { return this.getOptJSONRange.bind(this, name); };

            // test context for cell range address arrays
            jsonRangeListContext = new SheetOperationContext(docModel, _.extend({
                ranges: [{ start: [1, 2] }],
                ranges2: [{ start: [1, 2] }, { start: [2, 3], end: [3, 4] }],
                ranges3: [{ start: [2, 3], end: [3, null] }],
                ranges4: []
            }, operation), false);
            jsonRangeListContext.bindGet = function (name) { return this.getJSONRangeList.bind(this, name); };
            jsonRangeListContext.bindOpt = function (name) { return this.getOptJSONRangeList.bind(this, name); };
        });

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a context', function () {
                var context = new SheetOperationContext(docModel, operation, false);
                expect(context).to.be.an.instanceof(SheetOperationContext);
                expect(context).to.be.an.instanceof(OperationContext);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getAddress');
                expect(OperationContext).to.not.respondTo('getAddress');
            });
            it('should return existing properties', function () {
                expect(addrContext.getAddress('address1')).to.stringifyTo('A1');
                expect(addrContext.getAddress('address2')).to.stringifyTo('BB99');
            });
            it('should throw on missing properties', function () {
                expect(addrContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(addrContext.bindGet('num')).to.throw(OperationError);
                expect(addrContext.bindGet('obj')).to.throw(OperationError);
                expect(addrContext.bindGet('arr')).to.throw(OperationError);
                expect(addrContext.bindGet('address3')).to.throw(OperationError);
                expect(addrContext.bindGet('address4')).to.throw(OperationError);
            });
        });

        describe('method "getOptAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptAddress');
                expect(OperationContext).to.not.respondTo('getOptAddress');
            });
            it('should return existing properties', function () {
                expect(addrContext.getOptAddress('address1')).to.stringifyTo('A1');
                expect(addrContext.getOptAddress('address2')).to.stringifyTo('BB99');
            });
            it('should return null for missing properties', function () {
                expect(addrContext.getOptAddress('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(addrContext.bindOpt('num')).to.throw(OperationError);
                expect(addrContext.bindOpt('obj')).to.throw(OperationError);
                expect(addrContext.bindOpt('arr')).to.throw(OperationError);
                expect(addrContext.bindOpt('address3')).to.throw(OperationError);
                expect(addrContext.bindOpt('address4')).to.throw(OperationError);
            });
        });

        describe('method "getAddressList"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getAddressList');
                expect(OperationContext).to.not.respondTo('getAddressList');
            });
            it('should return existing properties', function () {
                expect(addrListContext.getAddressList('addresses1')).to.stringifyTo('A1');
                expect(addrListContext.getAddressList('addresses2')).to.stringifyTo('A1,B2,BB99');
            });
            it('should throw on missing properties', function () {
                expect(addrListContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(addrListContext.bindGet('num')).to.throw(OperationError);
                expect(addrListContext.bindGet('obj')).to.throw(OperationError);
                expect(addrListContext.bindGet('arr')).to.throw(OperationError);
                expect(addrListContext.bindGet('addresses3')).to.throw(OperationError);
                expect(addrListContext.bindGet('addresses4')).to.throw(OperationError);
            });
        });

        describe('method "getOptAddressList"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptAddressList');
                expect(OperationContext).to.not.respondTo('getOptAddressList');
            });
            it('should return existing properties', function () {
                expect(addrListContext.getOptAddressList('addresses1')).to.stringifyTo('A1');
                expect(addrListContext.getOptAddressList('addresses2')).to.stringifyTo('A1,B2,BB99');
            });
            it('should return null for missing properties', function () {
                expect(addrListContext.getOptAddressList('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(addrListContext.bindOpt('num')).to.throw(OperationError);
                expect(addrListContext.bindOpt('obj')).to.throw(OperationError);
                expect(addrListContext.bindOpt('arr')).to.throw(OperationError);
                expect(addrListContext.bindOpt('addresses3')).to.throw(OperationError);
                expect(addrListContext.bindOpt('addresses4')).to.throw(OperationError);
            });
        });
        describe('method "getJSONInterval"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getJSONInterval');
                expect(OperationContext).to.not.respondTo('getJSONInterval');
            });
            it('should return existing properties', function () {
                expect(jsonIntContext.getJSONInterval('interval1', false)).to.stringifyTo('2:1000000');
                expect(jsonIntContext.getJSONInterval('interval2', false)).to.stringifyTo('2:2');
                expect(jsonIntContext.getJSONInterval('interval2', true)).to.stringifyTo('2:2');
                expect(jsonIntContext.getJSONInterval(true)).to.stringifyTo('3:3');
            });
            it('should throw on missing properties', function () {
                expect(jsonIntContext.bindGet('wrong', true)).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonIntContext.bindGet('num', true)).to.throw(OperationError);
                expect(jsonIntContext.bindGet('obj', true)).to.throw(OperationError);
                expect(jsonIntContext.bindGet('arr', true)).to.throw(OperationError);
                expect(jsonIntContext.bindGet('interval1', true)).to.throw(OperationError);
                expect(jsonIntContext.bindGet('interval3', true)).to.throw(OperationError);
                expect(jsonIntContext.bindGet('interval4', true)).to.throw(OperationError);
            });
        });

        describe('method "getOptJSONInterval"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptJSONInterval');
                expect(OperationContext).to.not.respondTo('getOptJSONInterval');
            });
            it('should return existing properties', function () {
                expect(jsonIntContext.getOptJSONInterval('interval1', false)).to.stringifyTo('2:1000000');
                expect(jsonIntContext.getOptJSONInterval('interval2', false)).to.stringifyTo('2:2');
                expect(jsonIntContext.getOptJSONInterval('interval2', true)).to.stringifyTo('2:2');
                expect(jsonIntContext.getOptJSONInterval(true)).to.stringifyTo('3:3');
            });
            it('should return null for missing properties', function () {
                expect(jsonIntContext.getOptJSONInterval('wrong', true)).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonIntContext.bindOpt('num', true)).to.throw(OperationError);
                expect(jsonIntContext.bindOpt('obj', true)).to.throw(OperationError);
                expect(jsonIntContext.bindOpt('arr', true)).to.throw(OperationError);
                expect(jsonIntContext.bindOpt('interval1', true)).to.throw(OperationError);
                expect(jsonIntContext.bindOpt('interval3', true)).to.throw(OperationError);
                expect(jsonIntContext.bindOpt('interval4', true)).to.throw(OperationError);
            });
        });

        describe('method "getJSONAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getJSONAddress');
                expect(OperationContext).to.not.respondTo('getJSONAddress');
            });
            it('should return existing properties', function () {
                expect(jsonAddrContext.getJSONAddress('address1')).to.stringifyTo('B3');
                expect(jsonAddrContext.getJSONAddress('start')).to.stringifyTo('C4');
                expect(jsonAddrContext.getJSONAddress()).to.stringifyTo('C4');
            });
            it('should throw on missing properties', function () {
                expect(jsonAddrContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonAddrContext.bindGet('num')).to.throw(OperationError);
                expect(jsonAddrContext.bindGet('obj')).to.throw(OperationError);
                expect(jsonAddrContext.bindGet('arr')).to.throw(OperationError);
                expect(jsonAddrContext.bindGet('address2')).to.throw(OperationError);
                expect(jsonAddrContext.bindGet('address3')).to.throw(OperationError);
            });
        });

        describe('method "getOptJSONAddress"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptJSONAddress');
                expect(OperationContext).to.not.respondTo('getOptJSONAddress');
            });
            it('should return existing properties', function () {
                expect(jsonAddrContext.getOptJSONAddress('address1')).to.stringifyTo('B3');
                expect(jsonAddrContext.getOptJSONAddress('start')).to.stringifyTo('C4');
                expect(jsonAddrContext.getOptJSONAddress()).to.stringifyTo('C4');
            });
            it('should return null for missing properties', function () {
                expect(jsonAddrContext.getOptJSONAddress('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonAddrContext.bindOpt('num')).to.throw(OperationError);
                expect(jsonAddrContext.bindOpt('obj')).to.throw(OperationError);
                expect(jsonAddrContext.bindOpt('arr')).to.throw(OperationError);
                expect(jsonAddrContext.bindOpt('address2')).to.throw(OperationError);
                expect(jsonAddrContext.bindOpt('address3')).to.throw(OperationError);
            });
        });

        describe('method "getJSONRange"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getJSONRange');
                expect(OperationContext).to.not.respondTo('getJSONRange');
            });
            it('should return existing properties', function () {
                expect(jsonRangeContext.getJSONRange('range1')).to.stringifyTo('B3:D5');
                expect(jsonRangeContext.getJSONRange('range2')).to.stringifyTo('B3:B3');
                expect(jsonRangeContext.getJSONRange()).to.stringifyTo('C4:C4');
            });
            it('should throw on missing properties', function () {
                expect(jsonRangeContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonRangeContext.bindGet('num')).to.throw(OperationError);
                expect(jsonRangeContext.bindGet('obj')).to.throw(OperationError);
                expect(jsonRangeContext.bindGet('arr')).to.throw(OperationError);
                expect(jsonRangeContext.bindGet('range3')).to.throw(OperationError);
            });
        });

        describe('method "getOptJSONRange"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptJSONRange');
                expect(OperationContext).to.not.respondTo('getOptJSONRange');
            });
            it('should return existing properties', function () {
                expect(jsonRangeContext.getOptJSONRange('range1')).to.stringifyTo('B3:D5');
                expect(jsonRangeContext.getOptJSONRange('range2')).to.stringifyTo('B3:B3');
                expect(jsonRangeContext.getOptJSONRange()).to.stringifyTo('C4:C4');
            });
            it('should return null for missing properties', function () {
                expect(jsonRangeContext.getOptJSONRange('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonRangeContext.bindOpt('num')).to.throw(OperationError);
                expect(jsonRangeContext.bindOpt('obj')).to.throw(OperationError);
                expect(jsonRangeContext.bindOpt('arr')).to.throw(OperationError);
                expect(jsonRangeContext.bindOpt('range3')).to.throw(OperationError);
            });
        });

        describe('method "getJSONRangeList"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getJSONRangeList');
                expect(OperationContext).to.not.respondTo('getJSONRangeList');
            });
            it('should return existing properties', function () {
                expect(jsonRangeListContext.getJSONRangeList('ranges')).to.stringifyTo('B3:B3');
                expect(jsonRangeListContext.getJSONRangeList('ranges2')).to.stringifyTo('B3:B3,C4:D5');
                expect(jsonRangeListContext.getJSONRangeList()).to.stringifyTo('B3:B3');
            });
            it('should throw on missing properties', function () {
                expect(jsonRangeListContext.bindGet('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonRangeListContext.bindGet('num')).to.throw(OperationError);
                expect(jsonRangeListContext.bindGet('obj')).to.throw(OperationError);
                expect(jsonRangeListContext.bindGet('arr')).to.throw(OperationError);
                expect(jsonRangeListContext.bindGet('ranges3')).to.throw(OperationError);
                expect(jsonRangeListContext.bindGet('ranges4')).to.throw(OperationError);
            });
        });

        describe('method "getOptJSONRangeList"', function () {
            it('should exist', function () {
                expect(SheetOperationContext).to.respondTo('getOptJSONRangeList');
                expect(OperationContext).to.not.respondTo('getOptJSONRangeList');
            });
            it('should return existing properties', function () {
                expect(jsonRangeListContext.getOptJSONRangeList('ranges')).to.stringifyTo('B3:B3');
                expect(jsonRangeListContext.getOptJSONRangeList('ranges2')).to.stringifyTo('B3:B3,C4:D5');
                expect(jsonRangeListContext.getOptJSONRangeList()).to.stringifyTo('B3:B3');
            });
            it('should return null for missing properties', function () {
                expect(jsonRangeListContext.getOptJSONRangeList('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(jsonRangeListContext.bindOpt('num')).to.throw(OperationError);
                expect(jsonRangeListContext.bindOpt('obj')).to.throw(OperationError);
                expect(jsonRangeListContext.bindOpt('arr')).to.throw(OperationError);
                expect(jsonRangeListContext.bindOpt('ranges3')).to.throw(OperationError);
                expect(jsonRangeListContext.bindOpt('ranges4')).to.throw(OperationError);
            });
        });
    });

    // ========================================================================
});
