/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/movedescriptor',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/parser/tokens'
], function (AppHelper, SheetHelper, MoveDescriptor, FormulaUtils, Tokens) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var i = SheetHelper.i;
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var Matrix = FormulaUtils.Matrix;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;

    // create a CellRef instance on-the-fly from $1.$1 notation (column/row offsets)
    function cell(text) {
        var m = /^(\$)?(-?\d+)\.(\$)?(-?\d+)$/i.exec(text);
        return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
    }

    // create a SheetRef instance on-the-fly
    function sh(sheet, abs) {
        return new SheetRef(sheet, !_.isBoolean(abs) || abs);
    }

    // module Tokens ==========================================================

    describe('Spreadsheet module Tokens', function () {

        it('should exist', function () {
            expect(Tokens).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet 4' }, // Sheet name with white-space
            { name: 'insertName', label: 'name', formula: 'Sheet1!$A$1' }, // global name
            { name: 'insertName', sheet: 1, label: 'name', formula: 'Sheet1!$A$1' }, // sheet-local name
            { name: 'insertTable', sheet: 0, table: 'Table1', start: [1, 1], end: [3, 4], attrs: { table: { headerRow: true, footerRow: true } } },
            { name: 'changeCells', sheet: 0, start: 'B2', contents: [['Col1', 'Col2', 'Col3']] }
        ];

        // initialize test document
        var docModel = null, opGrammar = null, uiGrammar = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            opGrammar = docModel.getFormulaGrammar('op');
            uiGrammar = docModel.getFormulaGrammar('ui');
        });

        // class FixedToken ===================================================

        describe('class FixedToken', function () {

            var FixedToken = Tokens.FixedToken;
            it('should exist', function () {
                expect(FixedToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new FixedToken('type', 'text');

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('type');
                });
            });

            describe('method "isType"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('isType');
                });
                it('should test the token type', function () {
                    expect(token.isType('type')).to.equal(true);
                    expect(token.isType('other')).to.equal(false);
                });
            });

            describe('method "matchesType"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('matchesType');
                });
                it('should match the token type', function () {
                    expect(token.matchesType(/^t/)).to.equal(true);
                    expect(token.matchesType(/^o/)).to.equal(false);
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal('text');
                    expect(token.getText(uiGrammar)).to.equal('text');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('type[text]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('getValue');
                });
                it('should return the value', function () {
                    expect(token.getValue()).to.equal('text');
                });
            });

            describe('method "setValue"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('setValue');
                });
                it('should return whether the value changes', function () {
                    var token1 = new FixedToken('type', 'text');
                    expect(token1.setValue('text')).to.equal(false);
                    expect(token1.getValue()).to.equal('text');
                    expect(token1.setValue('other')).to.equal(true);
                    expect(token1.getValue()).to.equal('other');
                    expect(token1.getText(opGrammar)).to.equal('other');
                    expect(token1.getText(uiGrammar)).to.equal('other');
                });
            });

            describe('method "appendValue"', function () {
                it('should exist', function () {
                    expect(FixedToken).to.respondTo('appendValue');
                });
                it('should return whether the value changes', function () {
                    var token1 = new FixedToken('type', 'other');
                    expect(token1.appendValue('')).to.equal(false);
                    expect(token1.getValue()).to.equal('other');
                    expect(token1.appendValue('text')).to.equal(true);
                    expect(token1.getValue()).to.equal('othertext');
                    expect(token1.getText(opGrammar)).to.equal('othertext');
                    expect(token1.getText(uiGrammar)).to.equal('othertext');
                });
            });
        });

        // class OperatorToken ================================================

        describe('class OperatorToken', function () {

            var OperatorToken = Tokens.OperatorToken;
            it('should exist', function () {
                expect(OperatorToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new OperatorToken('list');

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(OperatorToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('op');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(OperatorToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal(',');
                    expect(token.getText(uiGrammar)).to.equal(';');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(OperatorToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('op[list]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(OperatorToken).to.respondTo('getValue');
                });
                it('should return the value', function () {
                    expect(token.getValue()).to.equal('list');
                });
            });
        });

        // class SeparatorToken ===============================================

        describe('class SeparatorToken', function () {

            var SeparatorToken = Tokens.SeparatorToken;
            it('should exist', function () {
                expect(SeparatorToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new SeparatorToken();

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(SeparatorToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('sep');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(SeparatorToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal(',');
                    expect(token.getText(uiGrammar)).to.equal(';');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(SeparatorToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('sep');
                });
            });
        });

        // class ParenthesisToken =============================================

        describe('class ParenthesisToken', function () {

            var ParenthesisToken = Tokens.ParenthesisToken;
            it('should exist', function () {
                expect(ParenthesisToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var openToken = new ParenthesisToken(true);
            var closeToken = new ParenthesisToken(false);

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(ParenthesisToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(openToken.getType()).to.equal('open');
                    expect(closeToken.getType()).to.equal('close');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(ParenthesisToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(openToken.getText(opGrammar)).to.equal('(');
                    expect(openToken.getText(uiGrammar)).to.equal('(');
                    expect(closeToken.getText(opGrammar)).to.equal(')');
                    expect(closeToken.getText(uiGrammar)).to.equal(')');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(ParenthesisToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(openToken.toString()).to.equal('open');
                    expect(closeToken.toString()).to.equal('close');
                });
            });
        });

        // class LiteralToken =================================================

        describe('class LiteralToken', function () {

            var LiteralToken = Tokens.LiteralToken;
            it('should exist', function () {
                expect(LiteralToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var numToken = new LiteralToken(12.5);
            var strToken = new LiteralToken('a"b"c');
            var falseToken = new LiteralToken(false);
            var trueToken = new LiteralToken(true);
            var errToken = new LiteralToken(ErrorCode.REF);
            var nullToken = new LiteralToken(null);

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(LiteralToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(numToken.getType()).to.equal('lit');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(LiteralToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(numToken.getText(opGrammar)).to.equal('12.5');
                    expect(numToken.getText(uiGrammar)).to.equal('12,5');
                    expect(strToken.getText(opGrammar)).to.equal('"a""b""c"');
                    expect(strToken.getText(uiGrammar)).to.equal('"a""b""c"');
                    expect(falseToken.getText(opGrammar)).to.equal('FALSE');
                    expect(falseToken.getText(uiGrammar)).to.equal('FALSCH');
                    expect(trueToken.getText(opGrammar)).to.equal('TRUE');
                    expect(trueToken.getText(uiGrammar)).to.equal('WAHR');
                    expect(errToken.getText(opGrammar)).to.equal('#REF!');
                    expect(errToken.getText(uiGrammar)).to.equal('#BEZUG!');
                    expect(nullToken.getText(opGrammar)).to.equal('');
                    expect(nullToken.getText(uiGrammar)).to.equal('');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(LiteralToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(numToken.toString()).to.equal('lit[12.5]');
                    expect(strToken.toString()).to.equal('lit["a""b""c"]');
                    expect(falseToken.toString()).to.equal('lit[FALSE]');
                    expect(trueToken.toString()).to.equal('lit[TRUE]');
                    expect(errToken.toString()).to.equal('lit[#REF]');
                    expect(nullToken.toString()).to.equal('lit[null]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(LiteralToken).to.respondTo('getValue');
                });
                it('should return the value', function () {
                    expect(numToken.getValue()).to.equal(12.5);
                    expect(strToken.getValue()).to.equal('a"b"c');
                    expect(falseToken.getValue()).to.equal(false);
                    expect(trueToken.getValue()).to.equal(true);
                    expect(errToken.getValue()).to.equal(ErrorCode.REF);
                    expect(nullToken.getValue()).to.equal(null);
                });
            });

            describe('method "setValue"', function () {
                it('should exist', function () {
                    expect(LiteralToken).to.respondTo('setValue');
                });
                it('should return whether the value changes', function () {
                    var token = new LiteralToken(12.5);
                    expect(token.setValue(12.5)).to.equal(false);
                    expect(token.getValue()).to.equal(12.5);
                    expect(token.getText(uiGrammar)).to.equal('12,5');
                    expect(token.setValue(1)).to.equal(true);
                    expect(token.getValue()).to.equal(1);
                    expect(token.getText(uiGrammar)).to.equal('1');
                    expect(token.setValue('a"b"c')).to.equal(true);
                    expect(token.getValue()).to.equal('a"b"c');
                    expect(token.getText(uiGrammar)).to.equal('"a""b""c"');
                    expect(token.setValue(true)).to.equal(true);
                    expect(token.getValue()).to.equal(true);
                    expect(token.getText(uiGrammar)).to.equal('WAHR');
                    expect(token.setValue(ErrorCode.REF)).to.equal(true);
                    expect(token.getValue()).to.equal(ErrorCode.REF);
                    expect(token.getText(uiGrammar)).to.equal('#BEZUG!');
                    expect(token.setValue(null)).to.equal(true);
                    expect(token.getValue()).to.equal(null);
                    expect(token.getText(uiGrammar)).to.equal('');
                });
            });
        });

        // class MatrixToken ==================================================

        describe('class MatrixToken', function () {

            var MatrixToken = Tokens.MatrixToken;
            it('should exist', function () {
                expect(MatrixToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var matrix = new Matrix([[12.5, 'a"b"c'], [true, ErrorCode.REF]]);
            var token = new MatrixToken(matrix);

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(MatrixToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('mat');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(MatrixToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal('{12.5,"a""b""c";TRUE,#REF!}');
                    expect(token.getText(uiGrammar)).to.equal('{12,5;"a""b""c"|WAHR;#BEZUG!}');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(MatrixToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('mat[{12.5;"a""b""c"|TRUE;#REF}]');
                });
            });

            describe('method "getMatrix"', function () {
                it('should exist', function () {
                    expect(MatrixToken).to.respondTo('getMatrix');
                });
                it('should return the matrix', function () {
                    expect(token.getMatrix()).to.equal(matrix);
                });
            });
        });

        // class FunctionToken ================================================

        describe('class FunctionToken', function () {

            var FunctionToken = Tokens.FunctionToken;
            it('should exist', function () {
                expect(FunctionToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var token = new FunctionToken('SUM');

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(FunctionToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(token.getType()).to.equal('func');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(FunctionToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(token.getText(opGrammar)).to.equal('SUM');
                    expect(token.getText(uiGrammar)).to.equal('SUMME');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(FunctionToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(token.toString()).to.equal('func[SUM]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(FunctionToken).to.respondTo('getValue');
                });
                it('should return the value of the token', function () {
                    expect(token.getValue()).to.equal('SUM');
                });
            });
        });

        // class ReferenceToken ===============================================

        describe('class ReferenceToken', function () {

            var ReferenceToken = Tokens.ReferenceToken;
            it('should exist', function () {
                expect(ReferenceToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var cellToken, rangeToken, errToken, sheet1Token, sheet2Token, sheetErrToken;
            before(function () {
                cellToken = new ReferenceToken(docModel, cell('1.$2'), null, null, null);
                rangeToken = new ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
                errToken = new ReferenceToken(docModel, null, null, null, null);
                sheet1Token = new ReferenceToken(docModel, cell('1.$2'), null, sh(1), null);
                sheet2Token = new ReferenceToken(docModel, cell('1.$2'), null, sh(1), sh(3));
                sheetErrToken = new ReferenceToken(docModel, cell('1.$2'), null, sh(-1), null);
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(cellToken.getType()).to.equal('ref');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(cellToken.getText(opGrammar)).to.equal('B$3');
                    expect(cellToken.getText(uiGrammar)).to.equal('B$3');
                    expect(rangeToken.getText(opGrammar)).to.equal('B$3:$D5');
                    expect(rangeToken.getText(uiGrammar)).to.equal('B$3:$D5');
                    expect(errToken.getText(opGrammar)).to.equal('#REF!');
                    expect(errToken.getText(uiGrammar)).to.equal('#BEZUG!');
                    expect(sheet1Token.getText(opGrammar)).to.equal('Sheet2!B$3');
                    expect(sheet1Token.getText(uiGrammar)).to.equal('Sheet2!B$3');
                    expect(sheet2Token.getText(opGrammar)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                    expect(sheet2Token.getText(uiGrammar)).to.equal('\'Sheet2:Sheet 4\'!B$3');
                    expect(sheetErrToken.getText(opGrammar)).to.equal('#REF!');
                    expect(sheetErrToken.getText(uiGrammar)).to.equal('#BEZUG!');
                });
                it('should handle simple/complex sheet names correctly', function () {
                    var token1 = new ReferenceToken(docModel, cell('0.0'), null, sh('Sheet1'), null);
                    expect(token1.getText(opGrammar)).to.equal('Sheet1!A1');
                    expect(token1.getText(uiGrammar)).to.equal('Sheet1!A1');
                    var token2 = new ReferenceToken(docModel, cell('0.0'), null, sh('Sheet 2'), null);
                    expect(token2.getText(opGrammar)).to.equal('\'Sheet 2\'!A1');
                    expect(token2.getText(uiGrammar)).to.equal('\'Sheet 2\'!A1');
                    var token3 = new ReferenceToken(docModel, cell('0.0'), null, sh('1Sheet'), null);
                    expect(token3.getText(opGrammar)).to.equal('\'1Sheet\'!A1');
                    expect(token3.getText(uiGrammar)).to.equal('\'1Sheet\'!A1');
                    var token4 = new ReferenceToken(docModel, cell('0.0'), null, sh('Sheet+1'), null);
                    expect(token4.getText(opGrammar)).to.equal('\'Sheet+1\'!A1');
                    expect(token4.getText(uiGrammar)).to.equal('\'Sheet+1\'!A1');
                    var token5 = new ReferenceToken(docModel, cell('0.0'), null, sh('Sheet1'), sh('Sheet2'));
                    expect(token5.getText(opGrammar)).to.equal('Sheet1:Sheet2!A1');
                    expect(token5.getText(uiGrammar)).to.equal('Sheet1:Sheet2!A1');
                    var token6 = new ReferenceToken(docModel, cell('0.0'), null, sh('Sheet1'), sh('Sheet 2'));
                    expect(token6.getText(opGrammar)).to.equal('\'Sheet1:Sheet 2\'!A1');
                    expect(token6.getText(uiGrammar)).to.equal('\'Sheet1:Sheet 2\'!A1');
                    var token7 = new ReferenceToken(docModel, cell('0.0'), null, sh('Sheet 1'), sh('Sheet2'));
                    expect(token7.getText(opGrammar)).to.equal('\'Sheet 1:Sheet2\'!A1');
                    expect(token7.getText(uiGrammar)).to.equal('\'Sheet 1:Sheet2\'!A1');
                });
                it('should treat booleans as complex sheet names', function () {
                    var token1 = new ReferenceToken(docModel, cell('0.0'), null, sh('TRUE'));
                    expect(token1.getText(opGrammar)).to.equal('\'TRUE\'!A1');
                    expect(token1.getText(uiGrammar)).to.equal('TRUE!A1');
                    var token2 = new ReferenceToken(docModel, cell('0.0'), null, sh('WAHR'));
                    expect(token2.getText(opGrammar)).to.equal('WAHR!A1');
                    expect(token2.getText(uiGrammar)).to.equal('\'WAHR\'!A1');
                });
                it('should treat cell addresses as complex sheet names', function () {
                    var token1 = new ReferenceToken(docModel, cell('0.0'), null, sh('A1'));
                    expect(token1.getText(opGrammar)).to.equal('\'A1\'!A1');
                    expect(token1.getText(uiGrammar)).to.equal('\'A1\'!A1');
                    var token2 = new ReferenceToken(docModel, cell('0.0'), null, sh('ZZZZ1'));
                    expect(token2.getText(opGrammar)).to.equal('ZZZZ1!A1');
                    expect(token2.getText(uiGrammar)).to.equal('ZZZZ1!A1');
                    var token3 = new ReferenceToken(docModel, cell('0.0'), null, sh('R1C1'));
                    expect(token3.getText(opGrammar)).to.equal('\'R1C1\'!A1');
                    expect(token3.getText(uiGrammar)).to.equal('\'R1C1\'!A1');
                    var token4 = new ReferenceToken(docModel, cell('0.0'), null, sh('Z1S1'));
                    expect(token4.getText(opGrammar)).to.equal('Z1S1!A1');
                    expect(token4.getText(uiGrammar)).to.equal('\'Z1S1\'!A1');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(cellToken.toString()).to.equal('ref[B$3]');
                    expect(rangeToken.toString()).to.equal('ref[B$3:$D5]');
                    expect(errToken.toString()).to.equal('ref[#REF]');
                    expect(sheet1Token.toString()).to.equal('ref[$1!B$3]');
                    expect(sheet2Token.toString()).to.equal('ref[$1:$3!B$3]');
                    expect(sheetErrToken.toString()).to.equal('ref[#REF!B$3]');
                });
            });

            describe('method "hasSheetRef"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('hasSheetRef');
                });
                it('should return the correct result', function () {
                    expect(cellToken.hasSheetRef()).to.equal(false);
                    expect(rangeToken.hasSheetRef()).to.equal(false);
                    expect(errToken.hasSheetRef()).to.equal(false);
                    expect(sheet1Token.hasSheetRef()).to.equal(true);
                    expect(sheet2Token.hasSheetRef()).to.equal(true);
                    expect(sheetErrToken.hasSheetRef()).to.equal(true);
                });
            });

            describe('method "hasSheetError"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('hasSheetError');
                });
                it('should return the correct result', function () {
                    expect(cellToken.hasSheetError()).to.equal(false);
                    expect(rangeToken.hasSheetError()).to.equal(false);
                    expect(errToken.hasSheetError()).to.equal(false);
                    expect(sheet1Token.hasSheetError()).to.equal(false);
                    expect(sheet2Token.hasSheetError()).to.equal(false);
                    expect(sheetErrToken.hasSheetError()).to.equal(true);
                });
            });

            describe('method "getRange3D"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('getRange3D');
                });
                it('should return the correct range address', function () {
                    expect(cellToken.getRange3D({ refSheet: 0 })).to.stringifyTo('0:0!B3:B3');
                    expect(cellToken.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:B3');
                    expect(rangeToken.getRange3D({ refSheet: 0 })).to.stringifyTo('0:0!B3:D5');
                    expect(rangeToken.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:D5');
                    expect(errToken.getRange3D({ refSheet: 0 })).to.equal(null);
                    expect(errToken.getRange3D({ refSheet: 1 })).to.equal(null);
                    expect(sheet1Token.getRange3D({ refSheet: 0 })).to.stringifyTo('1:1!B3:B3');
                    expect(sheet1Token.getRange3D({ refSheet: 1 })).to.stringifyTo('1:1!B3:B3');
                    expect(sheet2Token.getRange3D()).to.stringifyTo('1:3!B3:B3');
                    expect(sheet2Token.getRange3D({ refSheet: 0 })).to.stringifyTo('1:3!B3:B3');
                    expect(sheet2Token.getRange3D({ refSheet: 1 })).to.stringifyTo('1:3!B3:B3');
                    expect(sheetErrToken.getRange3D({ refSheet: 0 })).to.equal(null);
                    expect(sheetErrToken.getRange3D({ refSheet: 1 })).to.equal(null);
                });
                it('should return the correct range address without reference sheet', function () {
                    expect(cellToken.getRange3D()).to.equal(null);
                    expect(rangeToken.getRange3D()).to.equal(null);
                    expect(errToken.getRange3D()).to.equal(null);
                    expect(sheet1Token.getRange3D()).to.stringifyTo('1:1!B3:B3');
                    expect(sheetErrToken.getRange3D()).to.equal(null);
                });
                it('should relocate cell range without wrapping', function () {
                    expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2') })).to.stringifyTo('0:0!A3:A3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('C2') })).to.equal(null);
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9') })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2') })).to.equal(null);
                    expect(rangeToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.stringifyTo('0:0!D3:F10');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B5') })).to.stringifyTo('0:0!A1:D3');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('C5') })).to.equal(null);
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B6') })).to.equal(null);
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9') })).to.stringifyTo('0:0!D3:F12');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2') })).to.equal(null);
                    expect(errToken.getRange3D({ refSheet: 0, targetAddress: a('E6') })).to.equal(null);
                });
                it('should relocate cell range with wrapping', function () {
                    expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, targetAddress: a('XFD1'), wrapReferences: true })).to.stringifyTo('0:0!A3:A3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!A3:A3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('C2'), wrapReferences: true })).to.stringifyTo('0:0!XFD3:XFD3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9'), wrapReferences: true })).to.stringifyTo('0:0!F3:F3');
                    expect(cellToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!XFB3:XFB3');
                    expect(rangeToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.stringifyTo('0:0!D3:F10');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B5'), wrapReferences: true })).to.stringifyTo('0:0!A1:D3');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('C5'), wrapReferences: true })).to.stringifyTo('0:0!D1:XFD3');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B6'), wrapReferences: true })).to.stringifyTo('0:0!A3:D1048576');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('B2'), targetAddress: a('F9'), wrapReferences: true })).to.stringifyTo('0:0!D3:F12');
                    expect(rangeToken.getRange3D({ refSheet: 0, refAddress: a('F9'), targetAddress: a('B2'), wrapReferences: true })).to.stringifyTo('0:0!D3:XFB1048574');
                    expect(errToken.getRange3D({ refSheet: 0, targetAddress: a('E6'), wrapReferences: true })).to.equal(null);
                });
            });

            describe('method "hasRelCol"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('hasRelCol');
                });
                it('should return true for references with relative column', function () {
                    expect(cellToken.hasRelCol()).to.equal(true);
                    expect(rangeToken.hasRelCol()).to.equal(true);
                    expect(errToken.hasRelCol()).to.equal(false);
                    var relToken = new ReferenceToken(docModel, cell('1.2'), cell('3.4'), null, null);
                    expect(relToken.hasRelCol()).to.equal(true);
                    var absToken = new ReferenceToken(docModel, cell('$1.$2'), cell('$3.$4'), null, null);
                    expect(absToken.hasRelCol()).to.equal(false);
                });
            });

            describe('method "hasRelRow"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('hasRelRow');
                });
                it('should return true for references with relative row', function () {
                    expect(cellToken.hasRelRow()).to.equal(false);
                    expect(rangeToken.hasRelRow()).to.equal(true);
                    expect(errToken.hasRelRow()).to.equal(false);
                    var relToken = new ReferenceToken(docModel, cell('1.2'), cell('3.4'), null, null);
                    expect(relToken.hasRelRow()).to.equal(true);
                    var absToken = new ReferenceToken(docModel, cell('$1.$2'), cell('$3.$4'), null, null);
                    expect(absToken.hasRelRow()).to.equal(false);
                });
            });

            describe('method "setRange"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('setRange');
                });
                it('should change range address', function () {
                    var token1 = new ReferenceToken(docModel, cell('1.$2'), null, null, null);
                    expect(token1).to.stringifyTo('ref[B$3]');
                    expect(token1.setRange(r('B3:B3'))).to.equal(false);
                    expect(token1).to.stringifyTo('ref[B$3]');
                    expect(token1.setRange(r('D5:D5'))).to.equal(true);
                    expect(token1).to.stringifyTo('ref[D$5]');
                    expect(token1.setRange(r('B3:D5'))).to.equal(true);
                    expect(token1).to.stringifyTo('ref[B$3:D$5]');
                    var token2 = new ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), null, null);
                    expect(token2.setRange(r('B3:D5'))).to.equal(false);
                    expect(token2).to.stringifyTo('ref[B$3:$D5]');
                    expect(token2.setRange(r('C4:D5'))).to.equal(true);
                    expect(token2).to.stringifyTo('ref[C$4:$D5]');
                    expect(token2.setRange(r('C4:E6'))).to.equal(true);
                    expect(token2).to.stringifyTo('ref[C$4:$E6]');
                    expect(token2.setRange(r('C4:C4'))).to.equal(true);
                    expect(token2).to.stringifyTo('ref[C$4:$C4]');
                });
                it('should convert reference error to valid reference', function () {
                    var token = new ReferenceToken(docModel, null, null, null, null);
                    expect(token).to.stringifyTo('ref[#REF]');
                    expect(token.setRange(r('B3:D5'))).to.equal(true);
                    expect(token).to.stringifyTo('ref[$B$3:$D$5]');
                });
            });

            describe('method "relocateRange"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('relocateRange');
                });
                it('should relocate cell range without wrapping', function () {
                    var token = new ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), sh(1), null);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('B2'), a('B2'), false)).to.equal(false);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('A2'), a('B4'), false)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!C$3:$D7]');
                    expect(token.relocateRange(a('B4'), a('A2'), false)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('C1'), a('A1'), false)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!#REF]');
                });
                it('should relocate cell range with wrapping', function () {
                    var token = new ReferenceToken(docModel, cell('1.$2'), cell('$3.4'), sh(1), null);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('B2'), a('B2'), true)).to.equal(false);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('A2'), a('B4'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!C$3:$D7]');
                    expect(token.relocateRange(a('B4'), a('A2'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!B$3:$D5]');
                    expect(token.relocateRange(a('C1'), a('A1'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!XFD$3:$D5]');
                    expect(token.relocateRange(a('A6'), a('A1'), true)).to.equal(true);
                    expect(token).to.stringifyTo('ref[$1!XFD$3:$D1048576]');
                });
            });

            describe('method "transformSheet"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('transformSheet');
                });
                it('should do nothing for tokens without sheet', function () {
                    expect(cellToken.transformSheet(null, 0)).to.equal(false);
                    expect(cellToken).to.stringifyTo('ref[B$3]');
                    expect(rangeToken.transformSheet(null, 0)).to.equal(false);
                    expect(rangeToken).to.stringifyTo('ref[B$3:$D5]');
                });
                it('should transform sheet index', function () {
                    var token1 = new ReferenceToken(docModel, cell('1.$2'), null, sh(1), null);
                    expect(token1.transformSheet(null, 1)).to.equal(true);
                    expect(token1).to.stringifyTo('ref[$2!B$3]');
                    expect(token1.transformSheet(null, 3)).to.equal(false);
                    expect(token1).to.stringifyTo('ref[$2!B$3]');
                    expect(token1.transformSheet(1, null)).to.equal(true);
                    expect(token1).to.stringifyTo('ref[$1!B$3]');
                    expect(token1.transformSheet(2, null)).to.equal(false);
                    expect(token1).to.stringifyTo('ref[$1!B$3]');
                    var token2 = new ReferenceToken(docModel, cell('1.$2'), null, sh(1), sh(3));
                    expect(token2.transformSheet(null, 1)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                    expect(token2.transformSheet(null, 4)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$2:$5!B$3]');
                    expect(token2.transformSheet(null, 6)).to.equal(false);
                    expect(token2).to.stringifyTo('ref[$2:$5!B$3]');
                    expect(token2.transformSheet(4, null)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$2:$4!B$3]');
                    expect(token2.transformSheet(1, null)).to.equal(true);
                    expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                    expect(token2.transformSheet(4, null)).to.equal(false);
                    expect(token2).to.stringifyTo('ref[$1:$3!B$3]');
                });
                it('should do nothing for token with sheet error', function () {
                    expect(sheetErrToken.transformSheet(null, 0)).to.equal(false);
                    expect(sheetErrToken).to.stringifyTo('ref[#REF!B$3]');
                });
            });

            describe('method "resolveMovedCells"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('resolveMovedCells');
                });

                function makeMoveDescs(interval, columns, insert) {
                    return [new MoveDescriptor(docModel, columns ? i('1:1048576') : i('A:XFD'), i(interval), columns, insert)];
                }

                it('should transform single cell address', function () {
                    var token = new ReferenceToken(docModel, cell('2.$2'), null, null, null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true), { refSheet: 1 })).to.equal('D$3');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('B:C', true, true), { refSheet: 1 })).to.equal('E$3');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:D', true, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true), { refSheet: 1 })).to.equal('C$4');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('2:3', false, true), { refSheet: 1 })).to.equal('C$5');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:4', false, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false), { refSheet: 1 })).to.equal('B$3');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('B:C', true, false), { refSheet: 1 })).to.equal('#REF!');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:D', true, false), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false), { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false))).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false), { refSheet: 1 })).to.equal('C$2');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('2:3', false, false), { refSheet: 1 })).to.equal('#REF!');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:4', false, false), { refSheet: 1 })).to.equal(null);
                });
                it('should transform range address', function () {
                    var token = new ReferenceToken(docModel, cell('2.$2'), cell('$4.4'), null, null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, true), { refSheet: 1 })).to.equal('D$3:$F5');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:F', true, true), { refSheet: 1 })).to.equal('C$3:$H5');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('F:F', true, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, true), { refSheet: 1 })).to.equal('C$4:$E6');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:6', false, true), { refSheet: 1 })).to.equal('C$3:$E8');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('6:6', false, true), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('A:A', true, false), { refSheet: 1 })).to.equal('B$3:$D5');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('D:F', true, false), { refSheet: 1 })).to.equal('C$3:$C5');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('C:E', true, false), { refSheet: 1 })).to.equal('#REF!');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('F:F', true, false), { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('1:1', false, false), { refSheet: 1 })).to.equal('C$2:$E4');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('4:6', false, false), { refSheet: 1 })).to.equal('C$3:$E3');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('3:5', false, false), { refSheet: 1 })).to.equal('#REF!');
                    expect(token.resolveMovedCells(opGrammar, 1, makeMoveDescs('6:6', false, false), { refSheet: 1 })).to.equal(null);
                });
                it('should transform references with sheet correctly', function () {
                    var token = new ReferenceToken(docModel, cell('2.$2'), null, sh(1), null);
                    var moveDescs = makeMoveDescs('A:A', true, true);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs)).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 0 })).to.equal('Sheet2!D$3');
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 1 })).to.equal('Sheet2!D$3');
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs)).to.equal('Sheet2!D$3');
                });
                it('should not transform references with sheet range', function () {
                    var token = new ReferenceToken(docModel, cell('2.$2'), null, sh(1), sh(3));
                    var moveDescs = makeMoveDescs('A:A', true, true);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 0, moveDescs)).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 1, moveDescs)).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 2, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 2, moveDescs, { refSheet: 1 })).to.equal(null);
                    expect(token.resolveMovedCells(opGrammar, 2, moveDescs)).to.equal(null);
                });
                it('should not transform error references', function () {
                    var moveDescs = makeMoveDescs('A:A', true, true);
                    expect(errToken.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                    expect(sheetErrToken.resolveMovedCells(opGrammar, 0, moveDescs, { refSheet: 0 })).to.equal(null);
                });
            });

            describe('method "resolveCutPasteCells"', function () {
                it('should exist', function () {
                    expect(ReferenceToken).to.respondTo('resolveCutPasteCells');
                });

                it('should transform single cell address', function () {
                    var token1 = new ReferenceToken(docModel, cell('1.1'), null, null, null); // B2
                    expect(token1.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 0, sheet: 0 })).to.equal('C3');
                    expect(token1.resolveCutPasteCells(opGrammar, r('D4:F6'), r('G4:I6'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    var token2 = new ReferenceToken(docModel, cell('$1.$1'), null, null, null); // $B$2
                    expect(token2.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 0, sheet: 0 })).to.equal('$B$2');
                    expect(token2.resolveCutPasteCells(opGrammar, r('D4:F6'), r('G4:I6'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    var token3 = new ReferenceToken(docModel, cell('$1.1'), null, null, null); // $B2
                    expect(token3.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 0, sheet: 0 })).to.equal('$B3');
                    expect(token3.resolveCutPasteCells(opGrammar, r('D4:F6'), r('G4:I6'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    var token4 = new ReferenceToken(docModel, cell('1.$1'), null, null, null); // B$2
                    expect(token4.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 0, sheet: 0 })).to.equal('C$2');
                    expect(token4.resolveCutPasteCells(opGrammar, r('D4:F6'), r('G4:I6'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    var token5 = new ReferenceToken(docModel, cell('1.1'), null, sh(0), null); // Sheet1!B2
                    expect(token5.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 0, sheet: 0 })).to.equal('Sheet1!C3');
                    expect(token5.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 1, sheet: 0 })).to.equal('Sheet1!C3');
                    expect(token5.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 0, sheet: 1 })).to.equal(null);
                    expect(token5.resolveCutPasteCells(opGrammar, r('A1:C3'), r('B2:D4'), { refSheet: 1, sheet: 1 })).to.equal(null);
                    expect(token5.resolveCutPasteCells(opGrammar, r('D4:F6'), r('G4:I6'), { refSheet: 0, sheet: 0 })).to.equal(null);
                });

                it('should transform cell range (one single col; up/down/other)', function () {
                    var token = new ReferenceToken(docModel, cell('2.9'), cell('2.14'), null, null); // C10:C15

                    // complete
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C15'), r('D20:D25'), { refSheet: 0, sheet: 0 })).to.equal('D20:D25');

                    // inside
                    expect(token.resolveCutPasteCells(opGrammar, r('C11:C14'), r('D20:D25'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // start to top (expand range)
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C11'), r('C8:C9'), { refSheet: 0, sheet: 0 })).to.equal('C8:C15');
                    expect(token.resolveCutPasteCells(opGrammar, r('C9:C10'), r('C1:C2'), { refSheet: 0, sheet: 0 })).to.equal('C2:C15');

                    // start to bottom (shrink range)
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C11'), r('C12:C13'), { refSheet: 0, sheet: 0 })).to.equal('C12:C15');
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C11'), r('C14:C15'), { refSheet: 0, sheet: 0 })).to.equal('C12:C15');
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C11'), r('C15:C16'), { refSheet: 0, sheet: 0 })).to.equal('C12:C16');
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C11'), r('C25:C26'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // start to ? (do nothing)
                    expect(token.resolveCutPasteCells(opGrammar, r('C10:C11'), r('D25:D26'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // end to top (shrink range)
                    expect(token.resolveCutPasteCells(opGrammar, r('C14:C15'), r('C11:C12'), { refSheet: 0, sheet: 0 })).to.equal('C10:C13');
                    expect(token.resolveCutPasteCells(opGrammar, r('C14:C15'), r('C10:C11'), { refSheet: 0, sheet: 0 })).to.equal('C10:C13');
                    expect(token.resolveCutPasteCells(opGrammar, r('C14:C15'), r('C9:C10'), { refSheet: 0, sheet: 0 })).to.equal('C9:C13');         // the calculated Range must be bigger than 1, if the original Range was bigger than 1 (don't ask ;)
                    expect(token.resolveCutPasteCells(opGrammar, r('C14:C15'), r('C1:C2'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // end to bottom (expand range)
                    expect(token.resolveCutPasteCells(opGrammar, r('C14:C15'), r('C17:C18'), { refSheet: 0, sheet: 0 })).to.equal('C10:C18');
                    expect(token.resolveCutPasteCells(opGrammar, r('C15:C16'), r('C17:C18'), { refSheet: 0, sheet: 0 })).to.equal('C10:C17');

                    // end to ? (do nothing)
                    expect(token.resolveCutPasteCells(opGrammar, r('C14:C15'), r('A1:A2'), { refSheet: 0, sheet: 0 })).to.equal(null);
                });

                it('should transform cell range (one single row; left/right/other)', function () {
                    var token = new ReferenceToken(docModel, cell('4.9'), cell('9.9'), null, null); // E10:J10

                    // complete
                    expect(token.resolveCutPasteCells(opGrammar, r('D10:K10'), r('H20:O20'), { refSheet: 0, sheet: 0 })).to.equal('I20:N20');

                    // inside
                    expect(token.resolveCutPasteCells(opGrammar, r('F10:I10'), r('H20:K20'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // start to right (shrink range)
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:F10'), r('G10:H10'), { refSheet: 0, sheet: 0 })).to.equal('G10:J10');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:F10'), r('I10:J10'), { refSheet: 0, sheet: 0 })).to.equal('G10:J10');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:F10'), r('J10:K10'), { refSheet: 0, sheet: 0 })).to.equal('G10:K10');        // the calculated Range must be bigger than 1, if the original Range was bigger than 1 (don't ask ;)
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:F10'), r('M10:N10'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // start to left (expand range)
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:F10'), r('A10:B10'), { refSheet: 0, sheet: 0 })).to.equal('A10:J10');
                    expect(token.resolveCutPasteCells(opGrammar, r('D10:E10'), r('B10:C10'), { refSheet: 0, sheet: 0 })).to.equal('C10:J10');

                    // start to ? (do nothing)
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:F10'), r('C20:D20'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // end to left (shrink range)
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:J10'), r('F10:G10'), { refSheet: 0, sheet: 0 })).to.equal('E10:H10');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:J10'), r('E10:F10'), { refSheet: 0, sheet: 0 })).to.equal('E10:H10');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:J10'), r('D10:E10'), { refSheet: 0, sheet: 0 })).to.equal('D10:H10');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:J10'), r('A10:B10'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // end to right (expand range)
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:J10'), r('M10:N10'), { refSheet: 0, sheet: 0 })).to.equal('E10:N10');
                    expect(token.resolveCutPasteCells(opGrammar, r('J10:K10'), r('M10:N10'), { refSheet: 0, sheet: 0 })).to.equal('E10:M10');

                    // end to ? (do nothing)
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:J10'), r('M11:N11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                });

                it('should transform cell range (top/right/bottom/left/other)', function () {
                    var token = new ReferenceToken(docModel, cell('4.9'), cell('8.13'), null, null); // E10:I14

                    // complete
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:J15'), r('G25:L30'), { refSheet: 0, sheet: 0 })).to.equal('G25:K29');

                    // inside
                    expect(token.resolveCutPasteCells(opGrammar, r('G11:H12'), r('K18:L19'), { refSheet: 0, sheet: 0 })).to.equal(null);

                    // top/left-edge ------------------------------------------------------------------------------------------------------------
                    // top/left -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('E7:G8'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('E9:G10'),  { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/left -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('A10:C11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('D10:F11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/left -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('E12:G13'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('E22:G23'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/left -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('G10:I11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('L10:N11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/left -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:G11'), r('A1:C2'),   { refSheet: 0, sheet: 0 })).to.equal(null);

                    // top/right-edge ------------------------------------------------------------------------------------------------------------
                    // top/right -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('G7:I8'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('G9:I10'),  { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/right -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('E10:G11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('A10:C11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/right -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('G12:I13'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('G22:I23'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/right -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('L10:N11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('H10:J11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top/right -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('G10:I11'), r('A1:C2'),   { refSheet: 0, sheet: 0 })).to.equal(null);

                    // bottom/left-edge ------------------------------------------------------------------------------------------------------------
                    // bottom/left -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('E11:G12'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('E3:G4'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/left -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('D13:F14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('A13:C14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/left -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('E14:G15'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('E22:G23'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/left -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('G13:I14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('L13:N14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/left -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:G14'), r('A1:C2'),   { refSheet: 0, sheet: 0 })).to.equal(null);

                    // bottom/right-edge ------------------------------------------------------------------------------------------------------------
                    // bottom/right -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('G11:I12'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('G3:I4'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/right -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('D13:F14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('A13:C14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/right -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('G14:I15'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('G22:I23'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/right -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('I13:K14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('L13:N14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom/right -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('G13:I14'), r('A1:C2'),   { refSheet: 0, sheet: 0 })).to.equal(null);

                    // top-edge ------------------------------------------------------------------------------------------------------------
                    // top -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('E8:I8'),   { refSheet: 0, sheet: 0 })).to.equal('E8:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('D9:J11'),  r('D4:J6'),   { refSheet: 0, sheet: 0 })).to.equal('E5:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('F10:H10'), r('F9:H9'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('D10:H10'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('D9:J11'),  r('A9:G11'),  { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('F10:H10'), r('C10:E10'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('E11:I11'), { refSheet: 0, sheet: 0 })).to.equal('E11:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('E13:I13'), { refSheet: 0, sheet: 0 })).to.equal('E11:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('D9:J10'), r('D12:J13'), { refSheet: 0, sheet: 0 })).to.equal('E13:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E9:I10'), r('E12:I13'), { refSheet: 0, sheet: 0 })).to.equal('E13:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E9:I10'), r('E13:I14'), { refSheet: 0, sheet: 0 })).to.equal('E11:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E9:I10'), r('E14:I15'), { refSheet: 0, sheet: 0 })).to.equal('E11:I15');
                    expect(token.resolveCutPasteCells(opGrammar, r('E9:I10'), r('E13:I14'), { refSheet: 0, sheet: 0 })).to.equal('E11:I14');

                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('E14:I14'), { refSheet: 0, sheet: 0 })).to.equal('E11:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('E17:I17'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('G10:K10'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('L10:P10'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // top -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:I10'), r('A1:E5'),   { refSheet: 0, sheet: 0 })).to.equal(null);

                    // right-edge ------------------------------------------------------------------------------------------------------------
                    // right -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('I7:I11'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('I2:I7'),  { refSheet: 0, sheet: 0 })).to.equal(null);
                    // right -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('H10:H14'), { refSheet: 0, sheet: 0 })).to.equal('E10:H14');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('F10:F14'), { refSheet: 0, sheet: 0 })).to.equal('E10:H14');
                    expect(token.resolveCutPasteCells(opGrammar, r('I9:J15'),  r('G9:H15'),  { refSheet: 0, sheet: 0 })).to.equal('E10:G14');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('E10:E14'), { refSheet: 0, sheet: 0 })).to.equal('E10:H14');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('B10:B14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // right -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('I12:I16'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('I22:I26'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // right -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('J10:J14'), { refSheet: 0, sheet: 0 })).to.equal('E10:J14');
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'), r('L10:L14'), { refSheet: 0, sheet: 0 })).to.equal('E10:L14');
                    expect(token.resolveCutPasteCells(opGrammar, r('I9:J15'),  r('K9:L15'),  { refSheet: 0, sheet: 0 })).to.equal('E10:K14');
                    // right -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('I10:I14'),  r('A1:A5'),  { refSheet: 0, sheet: 0 })).to.equal(null);

                    // bottom-edge ------------------------------------------------------------------------------------------------------------
                    // bottom -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('F14:H14'), r('F12:H12'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E13:I13'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E11:I11'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E10:I10'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E2:I2'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('D14:J15'), r('D13:J14'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('D14:J15'), r('D11:J12'), { refSheet: 0, sheet: 0 })).to.equal('E10:I11');

                    expect(token.resolveCutPasteCells(opGrammar, r('D14:J15'), r('D10:J11'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');

                    // bottom -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('C14:G14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('A14:E14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('F14:H14'), r('F15:H15'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E15:I15'), { refSheet: 0, sheet: 0 })).to.equal('E10:I15');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E18:I18'), { refSheet: 0, sheet: 0 })).to.equal('E10:I18');
                    expect(token.resolveCutPasteCells(opGrammar, r('D14:J15'), r('D15:J16'), { refSheet: 0, sheet: 0 })).to.equal('E10:I15');

                    expect(token.resolveCutPasteCells(opGrammar, r('E12:I13'), r('E11:I12'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E13:I14'), r('E11:I12'), { refSheet: 0, sheet: 0 })).to.equal('E10:I12');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('E11:I11'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I15'), r('E11:I12'), { refSheet: 0, sheet: 0 })).to.equal('E10:I11');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I15'), r('E10:I11'), { refSheet: 0, sheet: 0 })).to.equal('E10:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I15'), r('E9:I10'), { refSheet: 0, sheet: 0 })).to.equal('E9:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I15'), r('E8:I9'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I15'), r('E9:I10'), { refSheet: 0, sheet: 0 })).to.equal('E9:I13');
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I15'), r('E8:I9'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('D12:J14'), r('D9:J11'), { refSheet: 0, sheet: 0 })).to.equal('E9:I11');
                    expect(token.resolveCutPasteCells(opGrammar, r('E12:I14'), r('E9:I11'), { refSheet: 0, sheet: 0 })).to.equal('E9:I11');

                    // bottom -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('G14:K14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('K14:O14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // bottom -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('E14:I14'), r('A1:E1'),   { refSheet: 0, sheet: 0 })).to.equal(null);

                    // left-edge ------------------------------------------------------------------------------------------------------------
                    // left -> up
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('E8:E12'),  { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('E2:E7'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('D9:E15'),  r('D2:E8'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                    // left -> left
                    expect(token.resolveCutPasteCells(opGrammar, r('E11:E13'), r('D11:D13'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('D10:D14'), { refSheet: 0, sheet: 0 })).to.equal('D10:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('A10:A14'), { refSheet: 0, sheet: 0 })).to.equal('A10:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('D9:E15'),  r('A9:B15'),  { refSheet: 0, sheet: 0 })).to.equal('B10:I14');
                    // left -> down
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('E11:E15'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('E21:E25'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    // left -> right
                    expect(token.resolveCutPasteCells(opGrammar, r('E11:E13'), r('F9:F13'),  { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('F10:F14'), { refSheet: 0, sheet: 0 })).to.equal('F10:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('H10:H14'), { refSheet: 0, sheet: 0 })).to.equal('F10:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('I10:I14'), { refSheet: 0, sheet: 0 })).to.equal('F10:I14');
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('M10:M14'), { refSheet: 0, sheet: 0 })).to.equal(null);
                    expect(token.resolveCutPasteCells(opGrammar, r('D9:E15'),  r('G9:H14'),  { refSheet: 0, sheet: 0 })).to.equal('H10:I14');
                    // left -> ???
                    expect(token.resolveCutPasteCells(opGrammar, r('E10:E14'), r('A1:A5'),   { refSheet: 0, sheet: 0 })).to.equal(null);
                });
            });
        });

        // class NameToken ====================================================

        describe('class NameToken', function () {

            var NameToken = Tokens.NameToken;
            it('should exist', function () {
                expect(NameToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var nameToken, sheet2NameToken, extDoc0NameToken, extDoc1NameToken, errorNameToken, tableToken, sheet1TableToken, sheet2TableToken, errorTableToken;
            before(function () {
                nameToken = new NameToken(docModel, 'name', null, null);
                sheet2NameToken = new NameToken(docModel, 'Name', sh(1), null);
                extDoc0NameToken = new NameToken(docModel, 'name', null, 0);
                extDoc1NameToken = new NameToken(docModel, 'name', null, 1);
                errorNameToken = new NameToken(docModel, 'NAME', sh(-1), null);
                tableToken = new NameToken(docModel, 'table1', null, null);
                sheet1TableToken = new NameToken(docModel, 'Table1', sh(0), null);
                sheet2TableToken = new NameToken(docModel, 'TABLE1', sh(1), null);
                errorTableToken = new NameToken(docModel, 'Table1', sh(-1), null);
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(nameToken.getType()).to.equal('name');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(nameToken.getText(opGrammar)).to.equal('name');
                    expect(nameToken.getText(uiGrammar)).to.equal('name');
                    expect(sheet2NameToken.getText(opGrammar)).to.equal('Sheet2!Name');
                    expect(sheet2NameToken.getText(uiGrammar)).to.equal('Sheet2!Name');
                    expect(extDoc0NameToken.getText(opGrammar)).to.equal('[0]!name');
                    expect(extDoc0NameToken.getText(uiGrammar)).to.equal('[0]!name');
                    expect(extDoc1NameToken.getText(opGrammar)).to.equal('[1]!name');
                    expect(extDoc1NameToken.getText(uiGrammar)).to.equal('[1]!name');
                    expect(errorNameToken.getText(opGrammar)).to.equal('[0]!NAME');
                    expect(errorNameToken.getText(uiGrammar)).to.equal('[0]!NAME');
                    expect(tableToken.getText(opGrammar)).to.equal('table1');
                    expect(tableToken.getText(uiGrammar)).to.equal('table1');
                    expect(sheet1TableToken.getText(opGrammar)).to.equal('Sheet1!Table1');
                    expect(sheet1TableToken.getText(uiGrammar)).to.equal('Sheet1!Table1');
                    expect(sheet2TableToken.getText(opGrammar)).to.equal('Sheet2!TABLE1');
                    expect(sheet2TableToken.getText(uiGrammar)).to.equal('Sheet2!TABLE1');
                    expect(errorTableToken.getText(opGrammar)).to.equal('[0]!Table1');
                    expect(errorTableToken.getText(uiGrammar)).to.equal('[0]!Table1');
                });
                it('should use a custom label', function () {
                    expect(nameToken.getText(opGrammar, { label: 'new' })).to.equal('new');
                    expect(nameToken.getText(uiGrammar, { label: 'new' })).to.equal('new');
                    expect(sheet2NameToken.getText(opGrammar, { label: 'new' })).to.equal('Sheet2!new');
                    expect(sheet2NameToken.getText(uiGrammar, { label: 'new' })).to.equal('Sheet2!new');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(nameToken.toString()).to.equal('name[name]');
                    expect(sheet2NameToken.toString()).to.equal('name[$1!Name]');
                    expect(errorNameToken.toString()).to.equal('name[#REF!NAME]');
                    expect(tableToken.toString()).to.equal('name[table1]');
                    expect(sheet1TableToken.toString()).to.equal('name[$0!Table1]');
                    expect(sheet2TableToken.toString()).to.equal('name[$1!TABLE1]');
                    expect(errorTableToken.toString()).to.equal('name[#REF!Table1]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('getValue');
                });
                it('should return the value of the token', function () {
                    expect(nameToken.getValue()).to.equal('name');
                    expect(sheet2NameToken.getValue()).to.equal('Name');
                    expect(errorNameToken.getValue()).to.equal('NAME');
                    expect(tableToken.getValue()).to.equal('table1');
                    expect(sheet1TableToken.getValue()).to.equal('Table1');
                    expect(sheet2TableToken.getValue()).to.equal('TABLE1');
                    expect(errorTableToken.getValue()).to.equal('Table1');
                });
            });

            describe('method "hasSheetRef"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('hasSheetRef');
                });
                it('should return the correct result', function () {
                    expect(nameToken.hasSheetRef()).to.equal(false);
                    expect(sheet2NameToken.hasSheetRef()).to.equal(true);
                    expect(errorNameToken.hasSheetRef()).to.equal(true);
                    expect(tableToken.hasSheetRef()).to.equal(false);
                    expect(sheet1TableToken.hasSheetRef()).to.equal(true);
                    expect(sheet2TableToken.hasSheetRef()).to.equal(true);
                    expect(errorTableToken.hasSheetRef()).to.equal(true);
                });
            });

            describe('method "hasSheetError"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('hasSheetError');
                });
                it('should return the correct result', function () {
                    expect(nameToken.hasSheetError()).to.equal(false);
                    expect(sheet2NameToken.hasSheetError()).to.equal(false);
                    expect(errorNameToken.hasSheetError()).to.equal(true);
                    expect(tableToken.hasSheetError()).to.equal(false);
                    expect(sheet1TableToken.hasSheetError()).to.equal(false);
                    expect(sheet2TableToken.hasSheetError()).to.equal(false);
                    expect(errorTableToken.hasSheetError()).to.equal(true);
                });
            });

            describe('method "resolveNameModel"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('resolveNameModel');
                });
                it('should return globally defined name', function () {
                    var globalName = docModel.getNameCollection().getName('name');
                    expect(nameToken.resolveNameModel(null)).to.equal(globalName);
                    expect(nameToken.resolveNameModel(0)).to.equal(globalName);
                    expect(extDoc0NameToken.resolveNameModel(null)).to.equal(globalName);
                    expect(extDoc0NameToken.resolveNameModel(0)).to.equal(globalName);
                });
                it('should return sheet-local defined name', function () {
                    var localName = docModel.getSheetModel(1).getNameCollection().getName('name');
                    expect(nameToken.resolveNameModel(1)).to.equal(localName);
                    expect(sheet2NameToken.resolveNameModel(null)).to.equal(localName);
                    expect(sheet2NameToken.resolveNameModel(0)).to.equal(localName);
                    expect(sheet2NameToken.resolveNameModel(1)).to.equal(localName);
                });
                it('should return null for unknown names', function () {
                    expect(new NameToken(docModel, 'invalid', null, null).resolveNameModel(0)).to.equal(null);
                    expect(new NameToken(docModel, 'invalid', sh(1), null).resolveNameModel(0)).to.equal(null);
                    expect(new NameToken(docModel, 'name', sh(0), null).resolveNameModel(null)).to.equal(null);
                    expect(new NameToken(docModel, 'name', sh(0), null).resolveNameModel(0)).to.equal(null);
                    expect(new NameToken(docModel, 'name', sh(0), null).resolveNameModel(1)).to.equal(null);
                });
                it('should return null for token with sheet error', function () {
                    expect(errorNameToken.resolveNameModel(null)).to.equal(null);
                    expect(errorNameToken.resolveNameModel(0)).to.equal(null);
                });
                it('should return null for token with external name', function () {
                    expect(extDoc1NameToken.resolveNameModel(null)).to.equal(null);
                    expect(extDoc1NameToken.resolveNameModel(0)).to.equal(null);
                });
                it('should return null for table tokens', function () {
                    expect(tableToken.resolveNameModel(null)).to.equal(null);
                    expect(sheet1TableToken.resolveNameModel(null)).to.equal(null);
                    expect(sheet2TableToken.resolveNameModel(null)).to.equal(null);
                    expect(errorTableToken.resolveNameModel(null)).to.equal(null);
                });
            });

            describe('method "resolveTableModel"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('resolveTableModel');
                });
                it('should return the table model', function () {
                    var tableModel = docModel.getTable('Table1');
                    expect(tableToken.resolveTableModel()).to.equal(tableModel);
                    expect(sheet1TableToken.resolveTableModel()).to.equal(tableModel);
                    expect(sheet2TableToken.resolveTableModel()).to.equal(null);
                    expect(errorTableToken.resolveTableModel()).to.equal(null);
                });
                it('should return null for name tokens', function () {
                    expect(nameToken.resolveTableModel()).to.equal(null);
                    expect(sheet2NameToken.resolveTableModel()).to.equal(null);
                    expect(errorNameToken.resolveTableModel()).to.equal(null);
                });
            });

            describe('method "transformSheet"', function () {
                it('should exist', function () {
                    expect(NameToken).to.respondTo('transformSheet');
                });
                it('should do nothing for tokens without sheet', function () {
                    expect(nameToken.transformSheet(null, 0)).to.equal(false);
                    expect(nameToken).to.stringifyTo('name[name]');
                });
                it('should transform sheet index', function () {
                    var token = new NameToken(docModel, 'name', sh(1), null);
                    expect(token).to.stringifyTo('name[$1!name]');
                    expect(token.transformSheet(null, 1)).to.equal(true);
                    expect(token).to.stringifyTo('name[$2!name]');
                    expect(token.transformSheet(null, 3)).to.equal(false);
                    expect(token).to.stringifyTo('name[$2!name]');
                    expect(token.transformSheet(1, null)).to.equal(true);
                    expect(token).to.stringifyTo('name[$1!name]');
                    expect(token.transformSheet(2, null)).to.equal(false);
                    expect(token).to.stringifyTo('name[$1!name]');
                    expect(token.transformSheet(1, null)).to.equal(true);
                    expect(token).to.stringifyTo('name[#REF!name]');
                });
                it('should do nothing for token with sheet error', function () {
                    expect(errorNameToken.transformSheet(null, 0)).to.equal(false);
                    expect(errorNameToken).to.stringifyTo('name[#REF!NAME]');
                });
            });
        });

        // class MacroToken ===================================================

        describe('class MacroToken', function () {

            var MacroToken = Tokens.MacroToken;
            it('should exist', function () {
                expect(MacroToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var macroToken, sheetToken, errorToken;
            before(function () {
                macroToken = new MacroToken(docModel, 'Sum', null);
                sheetToken = new MacroToken(docModel, 'Macro', sh(3));
                errorToken = new MacroToken(docModel, 'Macro', sh(-1));
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(macroToken.getType()).to.equal('macro');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('getText');
                });
                it('should return the string representation', function () {
                    expect(macroToken.getText(opGrammar)).to.equal('_xludf.Sum');
                    expect(macroToken.getText(uiGrammar)).to.equal('Sum');
                    expect(sheetToken.getText(opGrammar)).to.equal('\'Sheet 4\'!Macro');
                    expect(sheetToken.getText(uiGrammar)).to.equal('\'Sheet 4\'!Macro');
                    expect(errorToken.getText(opGrammar)).to.equal('[0]!Macro');
                    expect(errorToken.getText(uiGrammar)).to.equal('[0]!Macro');
                    var udfToken = new MacroToken(docModel, '_xludf.SUM', null);
                    expect(udfToken.getText(opGrammar)).to.equal('_xludf.SUM');
                    expect(udfToken.getText(uiGrammar)).to.equal('SUM');
                });
                it('should use a custom label', function () {
                    expect(macroToken.getText(opGrammar, { label: 'new' })).to.equal('new');
                    expect(macroToken.getText(uiGrammar, { label: 'new' })).to.equal('new');
                    expect(sheetToken.getText(opGrammar, { label: 'new' })).to.equal('\'Sheet 4\'!new');
                    expect(sheetToken.getText(uiGrammar, { label: 'new' })).to.equal('\'Sheet 4\'!new');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(macroToken.toString()).to.equal('macro[Sum]');
                    expect(sheetToken.toString()).to.equal('macro[$3!Macro]');
                    expect(errorToken.toString()).to.equal('macro[#REF!Macro]');
                });
            });

            describe('method "getValue"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('getValue');
                });
                it('should return the value of the token', function () {
                    expect(macroToken.getValue()).to.equal('Sum');
                    expect(sheetToken.getValue()).to.equal('Macro');
                    expect(errorToken.getValue()).to.equal('Macro');
                });
            });

            describe('method "hasSheetRef"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('hasSheetRef');
                });
                it('should return the correct result', function () {
                    expect(macroToken.hasSheetRef()).to.equal(false);
                    expect(sheetToken.hasSheetRef()).to.equal(true);
                    expect(errorToken.hasSheetRef()).to.equal(true);
                });
            });

            describe('method "hasSheetError"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('hasSheetError');
                });
                it('should return the correct result', function () {
                    expect(macroToken.hasSheetError()).to.equal(false);
                    expect(sheetToken.hasSheetError()).to.equal(false);
                    expect(errorToken.hasSheetError()).to.equal(true);
                });
            });

            describe('method "transformSheet"', function () {
                it('should exist', function () {
                    expect(MacroToken).to.respondTo('transformSheet');
                });
                it('should do nothing for tokens without sheet', function () {
                    expect(macroToken.transformSheet(null, 0)).to.equal(false);
                    expect(macroToken).to.stringifyTo('macro[Sum]');
                });
                it('should transform sheet index', function () {
                    var token = new MacroToken(docModel, 'Sum', sh(1));
                    expect(token).to.stringifyTo('macro[$1!Sum]');
                    expect(token.transformSheet(null, 1)).to.equal(true);
                    expect(token).to.stringifyTo('macro[$2!Sum]');
                    expect(token.transformSheet(null, 3)).to.equal(false);
                    expect(token).to.stringifyTo('macro[$2!Sum]');
                    expect(token.transformSheet(1, null)).to.equal(true);
                    expect(token).to.stringifyTo('macro[$1!Sum]');
                    expect(token.transformSheet(2, null)).to.equal(false);
                    expect(token).to.stringifyTo('macro[$1!Sum]');
                    expect(token.transformSheet(1, null)).to.equal(true);
                    expect(token).to.stringifyTo('macro[#REF!Sum]');
                });
                it('should do nothing for token with sheet error', function () {
                    expect(errorToken.transformSheet(null, 0)).to.equal(false);
                    expect(errorToken).to.stringifyTo('macro[#REF!Macro]');
                });
            });
        });

        // class TableToken ===================================================

        describe('class TableToken', function () {

            var TableToken = Tokens.TableToken;
            it('should exist', function () {
                expect(TableToken).to.be.a('function');
            });

            // private helpers ------------------------------------------------

            var tableToken;
            before(function () {
                tableToken = new TableToken(docModel, 'Table1');
            });

            // public methods -------------------------------------------------

            describe('method "getType"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('getType');
                });
                it('should return the token type', function () {
                    expect(tableToken.getType()).to.equal('table');
                });
            });

            describe('method "getText"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('getText');
                });
                function testToken(tableName, tableOptions, textOptions, expOp, expUi) {
                    var token = new TableToken(docModel, tableName, tableOptions);
                    expect(token.getText(opGrammar, textOptions)).to.equal(expOp);
                    expect(token.getText(uiGrammar, textOptions)).to.equal(expUi);
                }
                it('should return the string representation', function () {
                    testToken('Table1', null, null, 'Table1[]', 'Table1');
                    testToken('Table1', { regionKey: 'ALL' }, null, 'Table1[#All]', 'Table1[#Alle]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA' }, null, 'Table1[[#Headers],[#Data]]', 'Table1[[#Kopfzeilen];[#Daten]]');
                    testToken('Table1', { regionKey: 'DATA,TOTALS' }, null, 'Table1[[#Data],[#Totals]]', 'Table1[[#Daten];[#Ergebnisse]]');
                    testToken('Table1', { col1Name: 'Col 1' }, null, 'Table1[Col 1]', 'Table1[Col 1]');
                    testToken('Table1', { col1Name: ' Col ' }, null, 'Table1[[ Col ]]', 'Table1[[ Col ]]');
                    testToken('Table1', { col1Name: ' ' }, null, 'Table1[[ ]]', 'Table1[[ ]]');
                    testToken('Table1', { col1Name: 'Col1', col2Name: 'Col2' }, null, 'Table1[[Col1]:[Col2]]', 'Table1[[Col1]:[Col2]]');
                    testToken('Table1', { regionKey: 'ALL', col1Name: 'Col1' }, null, 'Table1[[#All],[Col1]]', 'Table1[[#Alle];[Col1]]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', col2Name: 'Col2' }, null, 'Table1[[#Headers],[#Data],[Col1]:[Col2]]', 'Table1[[#Kopfzeilen];[#Daten];[Col1]:[Col2]]');
                    testToken('Table1', { regionKey: 'ROW' }, null, 'Table1[#This Row]', 'Table1[@]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1' }, null, 'Table1[[#This Row],[Col1]]', 'Table1[@Col1]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col 1' }, null, 'Table1[[#This Row],[Col 1]]', 'Table1[@[Col 1]]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col:1' }, null, 'Table1[[#This Row],[Col:1]]', 'Table1[@[Col:1]]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1', col2Name: 'Col2' }, null, 'Table1[[#This Row],[Col1]:[Col2]]', 'Table1[@[Col1]:[Col2]]');
                });
                it('should add arbitrary whitespace characters', function () {
                    testToken('Table1', { openWs: true }, null, 'Table1[ ]', 'Table1');
                    testToken('Table1', { regionKey: 'ALL', openWs: true, closeWs: true }, null, 'Table1[ [#All] ]', 'Table1[ [#Alle] ]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', openWs: true }, null, 'Table1[ [#Headers],[#Data],[Col1]]', 'Table1[ [#Kopfzeilen];[#Daten];[Col1]]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', closeWs: true }, null, 'Table1[[#Headers],[#Data],[Col1] ]', 'Table1[[#Kopfzeilen];[#Daten];[Col1] ]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', sepWs: true }, null, 'Table1[[#Headers], [#Data], [Col1]]', 'Table1[[#Kopfzeilen]; [#Daten]; [Col1]]');
                    testToken('Table1', { regionKey: 'HEADERS,DATA', col1Name: 'Col1', openWs: true, closeWs: true, sepWs: true }, null, 'Table1[ [#Headers], [#Data], [Col1] ]', 'Table1[ [#Kopfzeilen]; [#Daten]; [Col1] ]');
                    testToken('Table1', { col1Name: 'Col 1', openWs: true, closeWs: true }, null, 'Table1[ [Col 1] ]', 'Table1[ [Col 1] ]');
                    testToken('Table1', { col1Name: ' Col ', openWs: true, closeWs: true }, null, 'Table1[ [ Col ] ]', 'Table1[ [ Col ] ]');
                    testToken('Table1', { col1Name: ' ', openWs: true, closeWs: true }, null, 'Table1[ [ ] ]', 'Table1[ [ ] ]');
                    testToken('Table1', { col1Name: 'Col1', col2Name: 'Col2', openWs: true, closeWs: true }, null, 'Table1[ [Col1]:[Col2] ]', 'Table1[ [Col1]:[Col2] ]');
                    testToken('Table1', { regionKey: 'ROW', openWs: true, closeWs: true }, null, 'Table1[ [#This Row] ]', 'Table1[ @ ]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1', openWs: true, closeWs: true, sepWs: true }, null, 'Table1[ [#This Row], [Col1] ]', 'Table1[ @Col1 ]');
                });
                it('should omit table name in specific situations', function () {
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 0 }, 'Table1[Col1]', 'Table1[Col1]');
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 0, unqualifiedTables: true }, 'Table1[Col1]', '[Col1]');
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('A1'), targetAddress: a('A1'), refSheet: 0, unqualifiedTables: true }, 'Table1[Col1]', 'Table1[Col1]');
                    testToken('Table1', { col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 1, unqualifiedTables: true }, 'Table1[Col1]', 'Table1[Col1]');
                    testToken('Table1', { regionKey: 'ROW', col1Name: 'Col1' }, { refAddress: a('B3'), targetAddress: a('B3'), refSheet: 0, unqualifiedTables: true }, 'Table1[[#This Row],[Col1]]', '[@Col1]');
                });
                it('should use a custom table name', function () {
                    testToken('Table1', { regionKey: 'ALL' }, { tableName: 'New' }, 'New[#All]', 'New[#Alle]');
                });
            });

            describe('method "toString"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('toString');
                });
                it('should return the string representation', function () {
                    expect(tableToken.toString()).to.equal('table[Table1]');
                });
            });

            describe('method "getTableName"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('getTableName');
                });
                it('should return the table name', function () {
                    expect(tableToken.getTableName()).to.equal('Table1');
                });
            });

            describe('method "getRegionKey"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('getRegionKey');
                });
                it('should return the region key', function () {
                    expect(tableToken.getRegionKey()).to.equal('DATA'); // default for missing region key
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ALL' }).getRegionKey()).to.equal('ALL');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'DATA,TOTALS' }).getRegionKey()).to.equal('DATA,TOTALS');
                    expect(new TableToken(docModel, 'Table1', { col1Name: 'Col1' }).getRegionKey()).to.equal('DATA');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ALL', col1Name: 'Col1' }).getRegionKey()).to.equal('ALL');
                });
            });

            describe('method "hasColumnRange"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('hasColumnRange');
                });
                it('should return whether the token refers to table columns', function () {
                    expect(tableToken.hasColumnRange()).to.equal(false);
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'TOTALS' }).hasColumnRange()).to.equal(false);
                    expect(new TableToken(docModel, 'Table1', { col1Name: 'Col1' }).hasColumnRange()).to.equal(true);
                    expect(new TableToken(docModel, 'Table1', { col1Name: 'Col1', col2Name: 'Col2' }).hasColumnRange()).to.equal(true);
                });
            });

            describe('method "resolveTableModel"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('resolveTableModel');
                });
                it('should return the table model', function () {
                    var tableModel = docModel.getTable('Table1');
                    expect(tableToken.resolveTableModel()).to.equal(tableModel);
                });
                it('should return null for invalid table name', function () {
                    expect(new TableToken(docModel, '__invalid__').resolveTableModel()).to.equal(null);
                });
            });

            describe('method "getRange3D"', function () {
                it('should exist', function () {
                    expect(TableToken).to.respondTo('getRange3D');
                });
                it('should return the range', function () {
                    expect(tableToken.getRange3D(a('A1'))).to.stringifyTo('0:0!B3:D4');
                    expect(tableToken.getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D4');
                });
                it('should return the correct table region', function () {
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ALL' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:D5');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'HEADERS' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:D2');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'DATA' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D4');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'TOTALS' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B5:D5');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'HEADERS,DATA' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:D4');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'DATA,TOTALS' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D5');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B1'))).to.equal(null);
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B2'))).to.equal(null);
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B3'))).to.stringifyTo('0:0!B3:D3');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B4'))).to.stringifyTo('0:0!B4:D4');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B5'))).to.equal(null);
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW' }).getRange3D(a('B6'))).to.equal(null);
                });
                it('should return the correct table columns', function () {
                    expect(new TableToken(docModel, 'Table1', { col1Name: 'Col1' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:B4');
                    expect(new TableToken(docModel, 'Table1', { col1Name: 'Col2', col2Name: 'Col3' }).getRange3D(a('B2'))).to.stringifyTo('0:0!C3:D4');
                    expect(new TableToken(docModel, 'Table1', { col1Name: 'Col3', col2Name: 'Col1' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B3:D4');
                });
                it('should return the correct combined ranges', function () {
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ALL', col1Name: 'Col1' }).getRange3D(a('B2'))).to.stringifyTo('0:0!B2:B5');
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW', col1Name: 'Col1', col2Name: 'Col2' }).getRange3D(a('B2'))).to.equal(null);
                    expect(new TableToken(docModel, 'Table1', { regionKey: 'ROW', col1Name: 'Col1', col2Name: 'Col2' }).getRange3D(a('B4'))).to.stringifyTo('0:0!B4:C4');
                });
                it('should return null for invalid table name', function () {
                    expect(new TableToken(docModel, '__invalid__').getRange3D(a('B2'))).to.equal(null);
                });
            });
        });
    });

    // ========================================================================
});
