/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/subtotalresult',
    'io.ox/office/spreadsheet/model/cellcollection'
], function (AppHelper, SheetHelper, Iterator, SheetUtils, SubtotalResult, CellCollection) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var ra = SheetHelper.ra;
    var rangesMatcher = SheetHelper.mergedRangesMatcher;
    var ErrorCode = SheetUtils.ErrorCode;

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        it('should exist', function () {
            expect(CellCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', styleId: 'a0', attrs: {}, default: true },

            // empty test sheet for operation generator methods
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },

            // test data for CellCollection.hasHeaderRow()
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [
                ['headline', 'headline', 'headline', 9],
                [2, 2, 2, 'text1'],
                [3, 2, 5, 12],
                [4, 2, 4, 0],
                [5, 2, 1, 'text2']
            ] },
            { name: 'changeCells', sheet: 1, start: 'A11', contents: [
                ['headline', 2, 2, 4, 5],
                ['headline', 2, 5, 12, 4],
                ['headline', 2, 4, 0, 4],
                [9, 'text1', 1, 5, 'text2']
            ] },

            // test data for CellCollection.createSubtotalIterator()
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeCells', sheet: 2, start: 'A1', contents: [
                [1,    2,    null, 'a',  4],
                [8,    16,   null, 'b',  32],
                [null, null, null, 'c',  null],
                ['d',  'e',  'f',  'g',  null],
                [64,   128,  null, null, 256]
            ] },

            // test data for CellCollection.getUsedRange()
            { name: 'insertSheet', sheet: 3, sheetName: 'Sheet4' },
            { name: 'changeCells', sheet: 3, start: 'C9', contents: [[1]] },
            { name: 'changeCells', sheet: 3, start: 'H2', contents: [[1]] }
        ];

        // initialize test document
        var docModel = null, sheetModel0 = null, cellCollection0 = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel0 = docModel.getSheetModel(0);
            cellCollection0 = sheetModel0.getCellCollection();
        });

        // color attribute values
        var AUTO = { type: 'auto' };
        var RED = { type: 'rgb', value: 'FF0000' };

        // border attribute values
        var NONE = { style: 'none' };
        var THIN = { style: 'solid', width: 26, color: AUTO };
        var DBL = { style: 'double', width: 79, color: AUTO };
        var THIN_RED = { style: 'solid', width: 26, color: RED };
        var DBL_RED = { style: 'double', width: 79, color: RED };

        // build border attributes by character shortcuts
        function borders() {
            var attrs = {};
            function add(keys, border) {
                _.each(keys, function (key) {
                    attrs[SheetUtils.getBorderName(key)] = border;
                });
            }
            for (var i = 0; i < arguments.length; i += 2) {
                add(arguments[i], arguments[i + 1]);
            }
            return attrs;
        }

        // creates a matcher function for Chai's satisfy() assertion that maches attribute values
        function expectAttributes(attrs, expected) {
            expect(attrs).to.be.an('object');
            _.each(expected, function (value, name) {
                expect(attrs).to.have.a.property(name).that.deep.equals(value);
            });
        }

        function getCellCollection(sheet) {
            return docModel.getSheetModel(sheet).getCellCollection();
        }

        function expectCellAttributes(sheet, range, expected) {
            var cellCollection = getCellCollection(sheet);
            range = r(range);
            for (var address = range.start.clone(); address[1] <= range.end[1]; address[1] += 1) {
                for (address[0] = range.start[0]; address[0] <= range.end[0]; address[0] += 1) {
                    var cellAttrs = cellCollection.getAttributeSet(address).cell;
                    //console.log(address + ' attrs=' + JSON.stringify(cellAttrs));
                    expectAttributes(cellAttrs, expected);
                }
            }
        }

        // changes the value of a single cell
        function changeCell(sheet, address, value) {
            docModel.invokeOperationHandler({ name: 'changeCells', sheet: sheet, start: a(address).toJSON(), contents: [{ c: [{ v: value }] }] });
        }

        // public methods -----------------------------------------------------

        describe('method "getUsedRange"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('getUsedRange');
            });
            it('should return the used area', function () {
                expect(cellCollection0.getUsedRange()).to.equal(null);
                var cellCollection = getCellCollection(3);
                expect(cellCollection.getUsedRange()).to.stringifyTo('C2:H9');
            });
        });

        describe('method "hasHeaderRow"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('hasHeaderRow');
            });
            it('should detect headlines', function () {
                var cellCollection = getCellCollection(1);
                expect(cellCollection.hasHeaderRow(r('A1:D5'))).to.equal(true);
                expect(cellCollection.hasHeaderRow(r('A2:D5'))).to.equal(false);
                expect(cellCollection.hasHeaderRow(r('A11:D15'))).to.equal(false);
                expect(cellCollection.hasHeaderRow(r('B11:D15'))).to.equal(false);
            });
        });

        // range iterators ----------------------------------------------------

        describe('method "createAddressIterator"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('createAddressIterator');
            });
        });

        describe('method "createLinearAddressIterator"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('createLinearAddressIterator');
            });
        });

        describe('method "createStyleIterator"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('createStyleIterator');
            });
        });

        describe('method "createSubtotalIterator"', function () {

            var cellCollection = null;
            before(function () { cellCollection = getCellCollection(2); });

            function expectSubtotals(ranges, expSum, expCells) {
                var iterator = cellCollection.createSubtotalIterator(ranges);
                var result = Iterator.reduce(new SubtotalResult(), iterator, function (prev, next) { return prev.add(next); });
                expect(result.sum).to.equal(expSum);
                expect(result.cells).to.equal(expCells);
            }

            it('should exist', function () {
                expect(cellCollection).to.respondTo('createSubtotalIterator');
            });
            it('should return the correct subtotal results', function () {
                expectSubtotals(r('A1:A1'), 1, 1);
                expectSubtotals(r('A1:B2'), 27, 4);
                expectSubtotals(r('A1:C3'), 27, 4);
                expectSubtotals(r('A1:D4'), 27, 11);
                expectSubtotals(r('A1:E5'), 511, 16);
                expectSubtotals(r('B2:E5'), 432, 9);
                expectSubtotals(r('C3:E5'), 256, 4);
                expectSubtotals(r('D4:E5'), 256, 2);
                expectSubtotals(r('E5:E5'), 256, 1);
                expectSubtotals(ra('A1:D4 B2:E5'), 443, 14);
                expectSubtotals(ra('A1:C1048576 C3:E5'), 475, 12);
                expectSubtotals(ra('A1:XFD3 C3:E5'), 319, 12);
                expectSubtotals(ra('A1:C1048576 A1:XFD3'), 255, 14);
            });
            it('should return the correct subtotal results with hidden columns/rows', function () {
                docModel.invokeOperationHandler({ name: 'changeRows', sheet: 2, start: 1, attrs: { row: { visible: false } } });
                docModel.invokeOperationHandler({ name: 'changeColumns', sheet: 2, start: 1, attrs: { column: { visible: false } } });
                expectSubtotals(r('A1:A1'), 1, 1);
                expectSubtotals(r('A1:B2'), 1, 1);
                expectSubtotals(r('A1:C3'), 1, 1);
                expectSubtotals(r('A1:D4'), 1, 6);
                expectSubtotals(r('A1:E5'), 325, 9);
                expectSubtotals(r('B2:E5'), 256, 4);
                expectSubtotals(r('C3:E5'), 256, 4);
                expectSubtotals(r('D4:E5'), 256, 2);
                expectSubtotals(r('E5:E5'), 256, 1);
                expectSubtotals(ra('A1:D4 B2:E5'), 257, 7);
                expectSubtotals(ra('A1:C1048576 C3:E5'), 321, 7);
                expectSubtotals(ra('A1:XFD3 C3:E5'), 261, 7);
                expectSubtotals(ra('A1:C1048576 A1:XFD3'), 69, 7);
            });
            it('should return the correct subtotal results after changing a cell', function () {
                changeCell(2, 'E4', 0.5);
                expectSubtotals(r('A1:A1'), 1, 1);
                expectSubtotals(r('A1:B2'), 1, 1);
                expectSubtotals(r('A1:C3'), 1, 1);
                expectSubtotals(r('A1:D4'), 1, 6);
                expectSubtotals(r('A1:E5'), 325.5, 10);
                expectSubtotals(r('B2:E5'), 256.5, 5);
                expectSubtotals(r('C3:E5'), 256.5, 5);
                expectSubtotals(r('D4:E5'), 256.5, 3);
                expectSubtotals(r('E5:E5'), 256, 1);
                expectSubtotals(ra('A1:D4 B2:E5'), 257.5, 8);
                expectSubtotals(ra('A1:C1048576 C3:E5'), 321.5, 8);
                expectSubtotals(ra('A1:XFD3 C3:E5'), 261.5, 8);
                expectSubtotals(ra('A1:C1048576 A1:XFD3'), 69, 7);
            });
        });

        // operation generators -----------------------------------------------

        describe('method "parseCellValue"', function () {

            function expectParseResult(parseText, exp) {
                var parseResult = cellCollection0.parseCellValue(parseText, 'val', a('A1'));
                expect(parseResult).to.have.a.property('f', exp.f || null);
                if (exp.type) {
                    expect(parseResult).to.have.a.property('v', (exp.v === null) ? 0 : exp.v);
                    expect(parseResult).to.have.a.property('result').that.is.an('object');
                    expect(parseResult.result).to.have.a.property('type', exp.type);
                    expect(parseResult.result).to.have.a.property('value', exp.v);
                    if (exp.code) {
                        expect(parseResult.result).to.have.a.property('code', exp.code);
                    } else {
                        expect(parseResult.result).to.not.have.a.property('code');
                    }
                    if (exp.format) {
                        expect(parseResult.result).to.have.a.property('format').that.is.an('object');
                        expect(parseResult.result.format).to.have.a.property('formatCode', exp.format);
                    } else {
                        expect(parseResult.result).to.not.have.a.property('format');
                    }
                } else {
                    expect(parseResult).to.have.a.property('v', exp.v);
                    expect(parseResult).to.have.a.property('result', null);
                }
                if (exp.format && (exp.format !== 'General')) {
                    expect(parseResult).to.have.a.property('format', exp.format);
                } else {
                    expect(parseResult).to.not.have.a.property('format');
                }
            }

            it('should exist', function () {
                expect(cellCollection0).to.respondTo('parseCellValue');
            });
            it('should parse literal values', function () {
                expectParseResult('', { v: null });
                expectParseResult('0', { v: 0 });
                expectParseResult('42,5', { v: 42.5 });
                expectParseResult('-42', { v: -42 });
                expectParseResult('wahr', { v: true });
                expectParseResult('Falsch', { v: false });
                expectParseResult('#Wert!', { v: ErrorCode.VALUE });
                expectParseResult('abc', { v: 'abc' });
                expectParseResult('0a', { v: '0a' });
                expectParseResult(' wahr', { v: ' wahr' });
                expectParseResult('true', { v: 'true' });
                expectParseResult('#value!', { v: '#value!' });
            });
            it('should parse formatted numbers', function () {
                expectParseResult('1e3', { v: 1000, format: '0.00E+00' });
                expectParseResult('-100 %', { v: -1, format: '0%' });
                expectParseResult('24.4.1977', { v: 28239, format: 'DD.MM.YYYY' });
            });
            it('should parse formula expressions', function () {
                expectParseResult('=1+1', { v: 2, f: '1+1', type: 'valid' });
                expectParseResult('=SUMME(1;2;3)', { v: 6, f: 'SUM(1,2,3)', type: 'valid' });
                expectParseResult('=DATUM(2015;1;1)', { v: 42005, f: 'DATE(2015,1,1)', type: 'valid', format: 'DD.MM.YYYY' });
                expectParseResult('+2*3', { v: 6, f: '+2*3', type: 'valid' });
                expectParseResult('-2*3', { v: -6, f: '-2*3', type: 'valid' });
                expectParseResult('=1+', { v: ErrorCode.NA, f: '1+', type: 'error', code: 'missing' });
                expectParseResult('=ZZ9999', { v: null, f: 'ZZ9999', type: 'valid' });
            });
            it('should handle leading apostrophes', function () {
                expectParseResult('\'0', { v: '0' });
                expectParseResult('\'WAHR', { v: 'WAHR' });
                expectParseResult('\'#WERT!', { v: '#WERT!' });
                expectParseResult('\'=1+1', { v: '=1+1' });
            });
            it('should not parse special repeated characters as formulas', function () {
                expectParseResult('=', { v: '=' });
                expectParseResult('+', { v: '+' });
                expectParseResult('-', { v: '-' });
                expectParseResult('===', { v: '===' });
                expectParseResult('+++', { v: '+++' });
                expectParseResult('---', { v: '---' });
            });
        });

        describe('method "generateCellContentOperations"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('generateCellContentOperations');
            });
            it('should generate correct cell operations', function () {
                expect(cellCollection0.getValue(a('A1'))).to.equal(null);
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateCellContentOperations(generator, a('A1'), { v: 42 });
                return promise.then(function (changed) {
                    expect(changed).to.deep.equal(ra('A1:A1'));
                    expect(cellCollection0.getValue(a('A1'))).to.equal(42);
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expect(cellCollection0.getValue(a('A1'))).to.equal(null);
                });
            });
        });

        describe('method "generateFillOperations"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('generateFillOperations');
            });
            // it('should generate correct cell operations', function () {
            // });
        });

        describe('method "generateBorderOperations"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('generateBorderOperations');
            });
            it('should create a border around a range of cells', function () {
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateBorderOperations(generator, r('B2:E5'), borders('tblr', THIN));
                return promise.then(function (changed) {
                    expect(changed).to.satisfy(rangesMatcher('B2:E2 B3:B4 E3:E4 B5:E5'));
                    expectCellAttributes(0, 'B2:B2', borders('tl', THIN, 'br', NONE));
                    expectCellAttributes(0, 'C2:D2', borders('t', THIN, 'blr', NONE));
                    expectCellAttributes(0, 'E2:E2', borders('tr', THIN, 'bl', NONE));
                    expectCellAttributes(0, 'B3:B4', borders('l', THIN, 'tbr', NONE));
                    expectCellAttributes(0, 'E3:E4', borders('r', THIN, 'tbl', NONE));
                    expectCellAttributes(0, 'B5:B5', borders('bl', THIN, 'tr', NONE));
                    expectCellAttributes(0, 'C5:D5', borders('b', THIN, 'tlr', NONE));
                    expectCellAttributes(0, 'E5:E5', borders('br', THIN, 'tl', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes(0, 'B2:E5', borders('tblr', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                });
            });
            it('should merge a new border over an existing border', function () {
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateBorderOperations(generator, ra('E4:F6 C2:C2 C6:C6'), borders('tblr', DBL));
                return promise.then(function (changed) {
                    expect(changed).to.satisfy(rangesMatcher('E4:F6 C2:C2 C5:C6'));
                    expectCellAttributes(0, 'E4:E4', borders('tl', DBL, 'r', THIN, 'b', NONE));
                    expectCellAttributes(0, 'F4:F4', borders('tr', DBL, 'bl', NONE));
                    expectCellAttributes(0, 'E5:E5', borders('l', DBL, 'br', THIN, 't', NONE));
                    expectCellAttributes(0, 'F5:F5', borders('r', DBL, 'tbl', NONE));
                    expectCellAttributes(0, 'E6:E6', borders('bl', DBL, 'tr', NONE));
                    expectCellAttributes(0, 'F6:F6', borders('br', DBL, 'tl', NONE));
                    expectCellAttributes(0, 'C1:C1', borders('tblr', NONE));
                    expectCellAttributes(0, 'C2:C2', borders('tblr', DBL));
                    expectCellAttributes(0, 'C5:C5', borders('tblr', NONE));
                    expectCellAttributes(0, 'C6:C6', borders('tblr', DBL));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes(0, 'E4:E4', borders('r', THIN, 'tbl', NONE));
                    expectCellAttributes(0, 'E5:E5', borders('br', THIN, 'tl', NONE));
                    expectCellAttributes(0, 'E6:E6', borders('tblr', NONE));
                    expectCellAttributes(0, 'F4:F6', borders('tblr', NONE));
                    expectCellAttributes(0, 'C2:C2', borders('t', THIN, 'blr', NONE));
                    expectCellAttributes(0, 'C5:C5', borders('b', THIN, 'tlr', NONE));
                    expectCellAttributes(0, 'C6:C6', borders('tblr', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                });
            });
            it('should fill a range with inner borders', function () {
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateBorderOperations(generator, r('A10:E12'), borders('tblrhv', THIN));
                return promise.then(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('A10:E12'));
                    expectCellAttributes(0, 'A10:E12', borders('tblr', THIN));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes(0, 'A10:E12', borders('tblr', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                });
            });
            it('should remove adjacent borders when deleting borders', function () {
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateBorderOperations(generator, r('B11:B11'), borders('tblr', NONE));
                return promise.then(function (changed) {
                    expect(changed).to.satisfy(rangesMatcher('B10:B10 A11:C11 B12:B12'));
                    expectCellAttributes(0, 'B10:B10', borders('tlr', THIN, 'b', NONE));
                    expectCellAttributes(0, 'A11:A11', borders('tbl', THIN, 'r', NONE));
                    expectCellAttributes(0, 'B11:B11', borders('tblr', NONE));
                    expectCellAttributes(0, 'C11:C11', borders('tbr', THIN, 'l', NONE));
                    expectCellAttributes(0, 'B12:B12', borders('blr', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes(0, 'A10:C12', borders('tblr', THIN));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                });
            });
            it('should remove adjacent borders when setting different borders', function () {
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateBorderOperations(generator, r('D11:D11'), borders('tblr', DBL));
                return promise.then(function (changed) {
                    expect(changed).to.satisfy(rangesMatcher('D10:D10 C11:E11 D12:D12'));
                    expectCellAttributes(0, 'D10:D10', borders('tlr', THIN, 'b', NONE));
                    expectCellAttributes(0, 'C11:C11', borders('tb', THIN, 'lr', NONE));
                    expectCellAttributes(0, 'D11:D11', borders('tblr', DBL));
                    expectCellAttributes(0, 'E11:E11', borders('tbr', THIN, 'l', NONE));
                    expectCellAttributes(0, 'D12:D12', borders('blr', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes(0, 'D10:E12', borders('tblr', THIN));
                    expectCellAttributes(0, 'C11:C11', borders('tbr', THIN, 'l', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                });
            });
        });

        describe('method "generateVisibleBorderOperations"', function () {
            it('should exist', function () {
                expect(cellCollection0).to.respondTo('generateVisibleBorderOperations');
            });
            it('should change border color of existing borders', function () {
                var generator = sheetModel0.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection0.generateVisibleBorderOperations(generator, r('B1:D5'), { color: RED });
                return promise.then(function (changed) {
                    expect(promise.state()).to.equal('resolved');
                    expect(changed).to.satisfy(rangesMatcher('B2:D2 B3:B5 D5:D5 C6:C6 E4:E5'));
                    expectCellAttributes(0, 'B2:B2', borders('tl', THIN_RED, 'br', NONE));
                    expectCellAttributes(0, 'C2:C2', borders('tblr', DBL_RED));
                    expectCellAttributes(0, 'D2:D2', borders('t', THIN_RED, 'blr', NONE));
                    expectCellAttributes(0, 'B3:B4', borders('l', THIN_RED, 'tbr', NONE));
                    expectCellAttributes(0, 'B5:B5', borders('bl', THIN_RED, 'tr', NONE));
                    expectCellAttributes(0, 'D5:D5', borders('b', THIN_RED, 'tlr', NONE));
                    expectCellAttributes(0, 'C6:C6', borders('t', DBL_RED, 'blr', DBL));
                    expectCellAttributes(0, 'E4:E4', borders('l', DBL_RED, 't', DBL, 'r', THIN, 'b', NONE));
                    expectCellAttributes(0, 'E5:E5', borders('l', DBL_RED, 'br', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectCellAttributes(0, 'B2:B2', borders('tl', THIN, 'br', NONE));
                    expectCellAttributes(0, 'C2:C2', borders('tblr', DBL));
                    expectCellAttributes(0, 'D2:D2', borders('t', THIN, 'blr', NONE));
                    expectCellAttributes(0, 'B3:B4', borders('l', THIN, 'tbr', NONE));
                    expectCellAttributes(0, 'B5:B5', borders('bl', THIN, 'tr', NONE));
                    expectCellAttributes(0, 'D5:D5', borders('b', THIN, 'tlr', NONE));
                    expectCellAttributes(0, 'C6:C6', borders('tblr', DBL));
                    expectCellAttributes(0, 'E4:E4', borders('tl', DBL, 'r', THIN, 'b', NONE));
                    expectCellAttributes(0, 'E5:E5', borders('l', DBL, 'br', THIN, 't', NONE));
                    expect(docModel.applyOperations(generator)).to.equal(true);
                });
            });
        });
    });

    // ========================================================================
});
