/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (AppHelper, SheetHelper, Iterator, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var r = SheetHelper.r;
    var NULL = SheetHelper.ErrorCode.NULL;
    var DIV0 = SheetHelper.ErrorCode.DIV0;
    var SortDirection = SheetUtils.SortDirection;

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        // private helpers ----------------------------------------------------

        // the values to be sorted
        var TEST_VALUES = [
            [null, null, true],
            [3,    'c',  null],
            [null, null, null],
            [null, 'c',  '1'],
            [1,    null, NULL],
            [5,    'a',  DIV0],
            [3,    'b',  1]
        ];

        // the values around the sorted values that MUST NOT be sorted
        var FRAME_VALUES = [
            ['A1', 'B1', 'C1', 'D1', 'E1'],
            ['A2', null, null, null, 'E2'],
            ['A3', null, null, null, 'E3'],
            ['A4', null, null, null, 'E4'],
            ['A5', null, null, null, 'E5'],
            ['A6', null, null, null, 'E6'],
            ['A7', null, null, null, 'E7'],
            ['A8', null, null, null, 'E8'],
            ['A9', 'B9', 'C9', 'D9', 'E9']
        ];

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A1', contents: FRAME_VALUES },
            { name: 'changeCells', sheet: 0, start: 'B2', contents: TEST_VALUES }
        ];

        // initialize test document
        var docModel = null;
        var sheetModel = null, cellCollection = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel = docModel.getSheetModel(0);
            cellCollection = sheetModel.getCellCollection();
        });

        function expectRangeValues(expected) {
            Iterator.forEach(r('A1:E9'), function (address) {
                var exp = FRAME_VALUES[address[1]][address[0]];
                if (exp) {
                    var value = cellCollection.getValue(address);
                    expect(value).to.equal(exp);
                }
            });
            Iterator.forEach(r('B2:D8'), function (address) {
                var exp = expected[address[1] - 1][address[0] - 1];
                var value = cellCollection.getValue(address);
                expect(value).to.equal(exp);
            });
        }

        function describeSortTest(title, indexes, horizontal, expected) {
            describe('generate sort operations (' + title + ')', function () {

                var generator = null;
                var sortRules = indexes.map(function (index) {
                    return { index: Math.abs(index), descending: index < 0 };
                });

                it('should sort the cells', function () {
                    generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                    var sortDir = horizontal ? SortDirection.HORIZONTAL : SortDirection.VERTICAL;
                    var promise = cellCollection.generateSortOperations(generator, r('B2:D8'), sortDir, sortRules);
                    return promise.then(function () { expectRangeValues(expected); });
                });

                it('should restore cells with Undo', function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    expectRangeValues(TEST_VALUES);
                });
            });
        }

        // public methods -----------------------------------------------------

        describe('method "generateSortOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateSortOperations');
            });

            describeSortTest('test V1: column B ascending', [1], false, [
                [1,    null, NULL],
                [3,    'c',  null],
                [3,    'b',  1],
                [5,    'a',  DIV0],
                [null, null, true],
                [null, null, null],
                [null, 'c',  '1']
            ]);

            describeSortTest('test V2: column B descending', [-1], false, [
                [5,    'a',  DIV0],
                [3,    'c',  null],
                [3,    'b',  1],
                [1,    null, NULL],
                [null, null, true],
                [null, null, null],
                [null, 'c',  '1']
            ]);

            describeSortTest('test V3: column C ascending', [2], false, [
                [5,    'a',  DIV0],
                [3,    'b',  1],
                [3,    'c',  null],
                [null, 'c',  '1'],
                [null, null, true],
                [null, null, null],
                [1,    null, NULL]
            ]);

            describeSortTest('test V4: column C descending', [-2], false, [
                [3,    'c',  null],
                [null, 'c',  '1'],
                [3,    'b',  1],
                [5,    'a',  DIV0],
                [null, null, true],
                [null, null, null],
                [1,    null, NULL]
            ]);

            describeSortTest('test V5: column D ascending', [3], false, [
                [3,    'b',  1],
                [null, 'c',  '1'],
                [null, null, true],
                [1,    null, NULL],
                [5,    'a',  DIV0],
                [3,    'c',  null],
                [null, null, null]
            ]);

            describeSortTest('test V6: column D descending', [-3], false, [
                [5,    'a',  DIV0],
                [1,    null, NULL],
                [null, null, true],
                [null, 'c',  '1'],
                [3,    'b',  1],
                [3,    'c',  null],
                [null, null, null]
            ]);

            describeSortTest('test V7: column B asc, column C asc', [1, 2], false, [
                [1,    null, NULL],
                [3,    'b',  1],
                [3,    'c',  null],
                [5,    'a',  DIV0],
                [null, 'c',  '1'],
                [null, null, true],
                [null, null, null]
            ]);

            describeSortTest('test V8: column B asc, column C desc', [1, -2], false, [
                [1,    null, NULL],
                [3,    'c',  null],
                [3,    'b',  1],
                [5,    'a',  DIV0],
                [null, 'c',  '1'],
                [null, null, true],
                [null, null, null]
            ]);

            describeSortTest('test V9: column B desc, column C asc', [-1, 2], false, [
                [5,    'a',  DIV0],
                [3,    'b',  1],
                [3,    'c',  null],
                [1,    null, NULL],
                [null, 'c',  '1'],
                [null, null, true],
                [null, null, null]
            ]);

            describeSortTest('test V10: column B desc, column C desc', [-1, -2], false, [
                [5,    'a',  DIV0],
                [3,    'c',  null],
                [3,    'b',  1],
                [1,    null, NULL],
                [null, 'c',  '1'],
                [null, null, true],
                [null, null, null]
            ]);

            describeSortTest('test H1: row 4 asc', [4], true, _.zip(
                [true, null, null, '1',  NULL, DIV0, 1],
                [null, 'c',  null, 'c',  null, 'a', 'b'],
                [null, 3,    null, null, 1,    5,   3]
            ));

            describeSortTest('test H2: row 4 desc', [-4], true, _.zip(
                [null, 'c',  null, 'c',  null, 'a', 'b'],
                [true, null, null, '1',  NULL, DIV0, 1],
                [null, 3,    null, null, 1,    5,   3]
            ));
        });
    });

    // ========================================================================
});
