/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (AppHelper, SheetHelper, Iterator, SheetUtils) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var ra = SheetHelper.ra;

    var LEFT = SheetUtils.Direction.LEFT;
    var RIGHT = SheetUtils.Direction.RIGHT;
    var UP = SheetUtils.Direction.UP;
    var DOWN = SheetUtils.Direction.DOWN;

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', styleId: 'a0', attrs: {}, default: true },
            { name: 'insertName', label: 'global', formula: 'Sheet1!E2+Sheet1!$E$2+Sheet2!E2+Sheet2!$E$2' },
            { name: 'insertName', label: 'global2', formula: 'global' },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'B2', contents: [
                [{ v: 1, f: 'E2+1' }, { v: 2, f: 'F2+2' }, null, { v: 5, f: 'Sheet2!E2+5' }, { v: 6, f: 'Sheet2!F2+6' }],
                [{ v: 3, f: 'E3+3' }, { v: 4, f: 'F3+4' }, null, { v: 7, f: 'Sheet2!E3+7' }, { v: 8, f: 'Sheet2!F3+8' }]
            ] },
            { name: 'changeCells', sheet: 0, start: 'B5', contents: [
                [{ v: 1, f: 'E2', si: 0, sr: 'B5:C6' }, { v: 2, si: 0 }, null, { v: 5, f: 'Sheet2!E2', si: 1, sr: 'E5:F6' }, { v: 6, si: 1 }],
                [{ v: 3, si: 0 },                       { v: 4, si: 0 }, null, { v: 7, si: 1 },                              { v: 8, si: 1 }]
            ] },
            { name: 'changeCells', sheet: 0, start: 'B8', contents: [
                [{ v: 1, f: 'E2:F3', mr: 'B8:C9' }, 2, null, { v: 5, f: 'Sheet2!E2:F3', mr: 'E8:F9' }, 6],
                [3,                                 4, null, 7,                                        8]
            ] },
            { name: 'changeCells', sheet: 0, start: 'B11', contents: [['Col1', 'Col2'], [1, 2]] }, // table range
            { name: 'changeCells', sheet: 0, start: 'A13', contents: [[{ v: 12, f: 'SUM(Table1[],Table2[],Table1[Col1],Table1[Col2],Table2[Col1],Table2[Col2],Table2[[Col1]:[Col2]])' }]] },
            { name: 'changeCells', sheet: 0, start: 'A14', contents: [[{ v: 32, f: 'SUM(global,global2)' }]] },
            { name: 'insertValidation', sheet: 0, index: 0, ranges: 'B8:C9', type: 'between', value1: 'B2+Sheet2!B2', value2: 'B11+Sheet2!B11' },
            { name: 'insertCFRule', sheet: 0, id: '0', ranges: 'B8:C9', type: 'between', value1: 'B2+Sheet2!B2', value2: 'B11+Sheet2!B11', priority: 1, attrs: {} },
            { name: 'insertTable', sheet: 0, table: 'Table1', start: 'B11', end: 'C12', attrs: { table: { headerRow: true } } },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: 'E2', contents: [[11, 12], [13, 14]] },
            { name: 'changeCells', sheet: 1, start: 'B11', contents: [['Col1', 'Col2'], [1, 2]] }, // table range
            { name: 'insertValidation', sheet: 1, index: 0, ranges: 'B8:C9', type: 'between', value1: 'B2+Sheet1!B2', value2: 'B11+Sheet1!B11' },
            { name: 'insertValidation', sheet: 1, index: 1, ranges: 'D8:E10', type: 'between', value1: 'B2+Sheet1!B2', value2: 'B11+Sheet1!B11' },
            { name: 'insertValidation', sheet: 1, index: 2, ranges: 'F10:G10', type: 'between', value1: 'B2+Sheet1!B2', value2: 'B11+Sheet1!B11' },
            { name: 'insertValidation', sheet: 1, index: 3, ranges: 'H9:H9 I10:I10', type: 'list' },
            { name: 'insertValidation', sheet: 1, index: 4, ranges: 'H10:H10 I9:I9', type: 'list' },
            { name: 'insertCFRule', sheet: 1, id: '0', ranges: 'B8:C9', type: 'between', value1: 'B2+Sheet1!B2', value2: 'B11+Sheet1!B11', priority: 1, attrs: {} },
            { name: 'insertCFRule', sheet: 1, id: '1', ranges: 'D8:E10', type: 'between', value1: 'B2+Sheet1!B2', value2: 'B11+Sheet1!B11', priority: 2, attrs: {} },
            { name: 'insertCFRule', sheet: 1, id: '2', ranges: 'F10:G10', type: 'between', value1: 'B2+Sheet1!B2', value2: 'B11+Sheet1!B11', priority: 3, attrs: {} },
            { name: 'insertTable', sheet: 1, table: 'Table2', start: 'B11', end: 'C12', attrs: { table: { headerRow: true } } },

            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'insertDrawing', start: [2, 0], type: 'chart', attrs: { drawing: { anchorType: 'twoCell', startCol: 0, startRow: 0, endCol: 1, endRow: 1 } } },
            { name: 'insertChartDataSeries', start: [2, 0], series: 0, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet1!$J$1', names: 'Sheet1!$I$2:$I$3', values: 'Sheet1!$J$2:$J$3' } } },
            { name: 'insertChartDataSeries', start: [2, 0], series: 1, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet1!$K$1', names: 'Sheet1!$I$2:$I$3', values: 'Sheet1!$K$2:$K$3' } } },
            { name: 'insertChartDataSeries', start: [2, 0], series: 2, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet2!$K$1', names: 'Sheet1!$I$2:$I$3', values: 'Sheet2!$K$2:$K$3' } } },
            { name: 'setChartAxisAttributes', start: [2, 0], axis: 0, axPos: 'b', crossAx: 1, attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartAxisAttributes', start: [2, 0], axis: 1, axPos: 'l', crossAx: 0, attrs: { line: { type: 'none', width: 18 }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartTitleAttributes', start: [2, 0], attrs: { text: { link: 'Sheet1!$J$10:$J$11' } } },
            { name: 'setChartTitleAttributes', start: [2, 0], axis: 0, attrs: { text: { link: 'Sheet1!$K$10:$K$11' } } },
            { name: 'setChartTitleAttributes', start: [2, 0], axis: 1, attrs: { text: { link: 'Sheet2!$K$10:$K$11' } } }
        ];

        // initialize test document
        var docModel = null;
        var sheetModel1 = null, cellCollection1 = null;
        var sheetModel2 = null, cellCollection2 = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel1 = docModel.getSheetModel(0);
            cellCollection1 = sheetModel1.getCellCollection();
            sheetModel2 = docModel.getSheetModel(1);
            cellCollection2 = sheetModel2.getCellCollection();
        });

        function expectApplyUndo(generator) {
            expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
        }

        function expectTokenDesc(cellCollection, address, options, expRef, expMatRange, expFormula) {
            var tokenDesc = cellCollection.getTokenArray(a(address), options);
            if (!expRef) {
                expect(tokenDesc).to.be.equal(null);
            } else {
                expect(tokenDesc).to.be.an('object');
                expect(tokenDesc).to.have.a.property('tokenArray').that.is.an('object');
                expect(tokenDesc).to.have.a.property('refAddress').that.stringifiesTo(expRef);
                expect(tokenDesc).to.have.a.property('matrixRange').that.stringifiesTo(expMatRange || 'null');
                expect(tokenDesc).to.have.a.property('formula', expFormula);
            }
        }

        function getCellModel(sheet, address) {
            return docModel.getSheetModel(sheet).getCellCollection().getCellModel(a(address));
        }

        function expectUndefinedCell(sheet, address) {
            expect(getCellModel(sheet, address)).to.equal(null);
        }

        function expectFormulaCell(sheet, address, f, si, sr, mr) {
            var cellModel = getCellModel(sheet, address);
            expect(cellModel).to.be.an('object');
            expect(cellModel).to.have.a.property('f', f);
            if (si === 'any') {
                expect(cellModel).to.have.a.property('si').that.is.a('number');
            } else {
                expect(cellModel).to.have.a.property('si', si);
            }
            expect(cellModel).to.have.a.property('sr').that.stringifiesTo(sr || 'null');
            expect(cellModel).to.have.a.property('mr').that.stringifiesTo(mr || 'null');
            return cellModel;
        }

        function getValModel(sheet, index) {
            var cloneData = docModel.getSheetModel(sheet).getValidationCollection()._getCloneData();
            return cloneData.valModels[index] || null;
        }

        function expectValidation(sheet, index, expRanges, expValue1, expValue2) {
            var valModel = getValModel(sheet, index);
            expect(valModel).to.be.an('object');
            expect(valModel.getTargetRanges()).to.stringifyTo(expRanges);
            var attrs = valModel.getMergedAttributeSet(true).validation;
            expect(attrs.value1).to.equal(expValue1);
            expect(attrs.value2).to.equal(expValue2);
            return valModel;
        }

        function expectDeletedVal(sheet, index) {
            expect(getValModel(sheet, index)).to.equal(null);
        }

        function getRuleModel(sheet, id) {
            return docModel.getSheetModel(sheet).getCondFormatCollection().getRuleModel(id);
        }

        function expectRule(sheet, id, expRanges, expValue1, expValue2) {
            var ruleModel = getRuleModel(sheet, id);
            expect(ruleModel).to.be.an('object');
            expect(ruleModel.getTargetRanges()).to.stringifyTo(expRanges);
            Iterator.forEach(ruleModel.createTokenArrayIterator(), function (tokenArray, result) {
                expect(result.key).to.match(/^value[12]$/);
                var formula = tokenArray.getFormula('op');
                switch (result.key) {
                    case 'value1': expect(formula).to.equal(expValue1); break;
                    case 'value2': expect(formula).to.equal(expValue2); break;
                }
            });
            return ruleModel;
        }

        function expectDeletedRule(sheet, id) {
            expect(getRuleModel(sheet, id)).to.equal(null);
        }

        function getNameModel(sheet, label) {
            var parentModel = (sheet === null) ? docModel : docModel.getSheetModel(sheet);
            return parentModel.getNameCollection().getName(label);
        }

        function expectName(sheet, label, expFormula) {
            var nameModel = getNameModel(sheet, label);
            expect(nameModel).to.be.an('object');
            expect(nameModel.getFormula('op')).to.equal(expFormula);
            return nameModel;
        }

        function getChartModel(sheet, pos) {
            return docModel.getSheetModel(sheet).getDrawingCollection().getModel([pos]);
        }

        function getChartSeriesModel(sheet, pos, series) {
            return getChartModel(sheet, pos).getSeriesModel(series);
        }

        function expectChartSeriesLinks(sheet, pos, series, expTitle, expNames, expValues) {
            var seriesModel = getChartSeriesModel(sheet, pos, series);
            expect(seriesModel).to.be.an('object');
            var attrSet = seriesModel.getMergedAttributeSet(true);
            expect(attrSet.series.title).to.equal(expTitle);
            expect(attrSet.series.names).to.equal(expNames);
            expect(attrSet.series.values).to.equal(expValues);
        }

        function getChartTitleModel(sheet, pos, axis) {
            return getChartModel(sheet, pos).getTitleModel(axis);
        }

        function expectChartTitleLink(sheet, pos, axis, expFormula) {
            var titleModel = getChartTitleModel(sheet, pos, axis);
            expect(titleModel).to.be.an('object');
            var attrSet = titleModel.getMergedAttributeSet(true);
            expect(attrSet.text.link).to.equal(expFormula);
        }

        // public methods -----------------------------------------------------

        describe('method "applyChangeCellsOperation"', function () {
            it('should import normal formulas', function () {
                expectFormulaCell(0, 'B2', 'E2+1', null, null, null);
                expectFormulaCell(0, 'F3', 'Sheet2!F3+8', null, null, null);
            });
            it('should import shared formulas', function () {
                expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C6', null);
                expectFormulaCell(0, 'C6', null, 0, null, null);
                expectFormulaCell(0, 'E5', 'Sheet2!E2', 1, 'E5:F6', null);
                expectFormulaCell(0, 'F6', null, 1, null, null);
            });
            it('should import matrix formulas', function () {
                expectFormulaCell(0, 'B8', 'E2:F3', null, null, 'B8:C9');
                expectFormulaCell(0, 'C9', null, null, null, null);
                expectFormulaCell(0, 'E8', 'Sheet2!E2:F3', null, null, 'E8:F9');
                expectFormulaCell(0, 'F9', null, null, null, null);
            });
        });

        describe('method "isFormulaCell"', function () {
            it('should exist', function () {
                expect(cellCollection1).to.respondTo('isFormulaCell');
            });
            it('should return false for blank and value cells', function () {
                expect(cellCollection2.isFormulaCell(a('A1'))).to.equal(false);
                expect(cellCollection2.isFormulaCell(a('E2'))).to.equal(false);
            });
            it('should return true for normal formula cells', function () {
                expect(cellCollection1.isFormulaCell(a('B2'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('C3'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('E2'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('F3'))).to.equal(true);
            });
            it('should return true for cells in a shared formula', function () {
                expect(cellCollection1.isFormulaCell(a('B5'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('C6'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('E5'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('F6'))).to.equal(true);
            });
            it('should return true for cells in a matrix formula', function () {
                expect(cellCollection1.isFormulaCell(a('B8'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('C9'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('E8'))).to.equal(true);
                expect(cellCollection1.isFormulaCell(a('F9'))).to.equal(true);
            });
        });

        describe('method "isAnchorCell"', function () {
            it('should exist', function () {
                expect(cellCollection1).to.respondTo('isAnchorCell');
            });
            it('should return false for blank and value cells', function () {
                expect(cellCollection2.isAnchorCell(a('A1'))).to.equal(false);
                expect(cellCollection2.isAnchorCell(a('E2'))).to.equal(false);
            });
            it('should return true for normal formula cells', function () {
                expect(cellCollection1.isAnchorCell(a('B2'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('C3'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('E2'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('F3'))).to.equal(true);
            });
            it('should return true for anchor cells of a shared formula', function () {
                expect(cellCollection1.isAnchorCell(a('B5'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('C6'))).to.equal(false);
                expect(cellCollection1.isAnchorCell(a('E5'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('F6'))).to.equal(false);
            });
            it('should return true for anchor cells of a matrix formula', function () {
                expect(cellCollection1.isAnchorCell(a('B8'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('C9'))).to.equal(false);
                expect(cellCollection1.isAnchorCell(a('E8'))).to.equal(true);
                expect(cellCollection1.isAnchorCell(a('F9'))).to.equal(false);
            });
        });

        describe('method "getTokenArray"', function () {
            it('should exist', function () {
                expect(cellCollection1).to.respondTo('getTokenArray');
            });
            it('should return null for blank and value cells', function () {
                expectTokenDesc(cellCollection2, 'A1', null, null);
                expectTokenDesc(cellCollection2, 'E2', null, null);
            });
            it('should return the data for normal formula cells', function () {
                expectTokenDesc(cellCollection1, 'B2', null, 'B2', null, null);
                expectTokenDesc(cellCollection1, 'B2', { grammarId: 'op' }, 'B2', null, 'E2+1');
                expectTokenDesc(cellCollection1, 'F3', { grammarId: 'op' }, 'F3', null, 'Sheet2!F3+8');
            });
            it('should return the data for cells in a shared formula', function () {
                expectTokenDesc(cellCollection1, 'B5', { grammarId: 'op' }, 'B5', null, 'E2');
                expectTokenDesc(cellCollection1, 'F6', { grammarId: 'op' }, 'E5', null, 'Sheet2!F3');
            });
            it('should return the data for cells in a matrix formula', function () {
                expectTokenDesc(cellCollection1, 'B8', { grammarId: 'op' }, 'B8', 'B8:C9', 'E2:F3');
                expectTokenDesc(cellCollection1, 'F9', { grammarId: 'op' }, null);
                expectTokenDesc(cellCollection1, 'F9', { fullMatrix: true }, 'E8', 'E8:F9', null);
                expectTokenDesc(cellCollection1, 'F9', { grammarId: 'op', fullMatrix: true }, 'E8', 'E8:F9', 'Sheet2!E2:F3');
            });
        });

        describe('method "getFormula"', function () {
            it('should exist', function () {
                expect(cellCollection1).to.respondTo('getFormula');
            });
            it('should return null for blank and value cells', function () {
                expect(cellCollection2.getFormula(a('A1'), 'op')).to.equal(null);
                expect(cellCollection2.getFormula(a('E2'), 'op')).to.equal(null);
            });
            it('should return the data for normal formula cells', function () {
                expect(cellCollection1.getFormula(a('B2'), 'op')).to.equal('E2+1');
                expect(cellCollection1.getFormula(a('F3'), 'op')).to.equal('Sheet2!F3+8');
            });
            it('should return the data for cells in a shared formula', function () {
                expect(cellCollection1.getFormula(a('B5'), 'op')).to.equal('E2');
                expect(cellCollection1.getFormula(a('F6'), 'op')).to.equal('Sheet2!F3');
            });
            it('should return the data for cells in a matrix formula', function () {
                expect(cellCollection1.getFormula(a('B8'), 'op')).to.equal('E2:F3');
                expect(cellCollection1.getFormula(a('F9'), 'op')).to.equal(null);
                expect(cellCollection1.getFormula(a('F9'), 'op', { fullMatrix: true })).to.equal('Sheet2!E2:F3');
            });
        });

        describe('method "getMatrixRange"', function () {
            it('should exist', function () {
                expect(cellCollection1).to.respondTo('getMatrixRange');
            });
            it('should return the matrix range', function () {
                expect(cellCollection1.getMatrixRange(a('A1'))).to.equal(null);
                expect(cellCollection1.getMatrixRange(a('B8'))).to.stringifyTo('B8:C9');
                expect(cellCollection1.getMatrixRange(a('F9'))).to.stringifyTo('E8:F9');
            });
        });

        describe('method "coversAnyMatrixRange"', function () {
            it('should exist', function () {
                expect(cellCollection1).to.respondTo('coversAnyMatrixRange');
            });
            it('should return whether the range overlaps with a matrix', function () {
                expect(cellCollection1.coversAnyMatrixRange(r('A1:B4'))).to.equal(false);
                expect(cellCollection1.coversAnyMatrixRange(r('B8:B9'))).to.equal(true);
                expect(cellCollection1.coversAnyMatrixRange(r('C9:D10'))).to.equal(true);
            });
        });

        describe('method "generateCellContentOperations"', function () {
            it('should update shared formula when shortening the bounding range', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateCellContentOperations(generator, a('B6'), [{ c: [{ f: '1+1' }, { f: null }] }]);
                return promise.then(function () {
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C5', null);
                    expectFormulaCell(0, 'B6', '1+1', null, null, null);
                    expectFormulaCell(0, 'C6', null, null, null, null);
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C6', null);
                    expectFormulaCell(0, 'B6', null, 0, null, null);
                    expectFormulaCell(0, 'C6', null, 0, null, null);
                });
            });
            it('should update shared formula when expanding the bounding range', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateCellContentOperations(generator, a('B7'), { si: 0 });
                return promise.then(function () {
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C7', null);
                    expectFormulaCell(0, 'B7', null, 0, null, null);
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C6', null);
                    expectUndefinedCell(0, 'B7');
                });
            });
            it('should update shared formula when deleting the anchor cell', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateCellContentOperations(generator, a('E5'), { f: null });
                return promise.then(function () {
                    expectFormulaCell(0, 'E5', null, null, null, null);
                    expectFormulaCell(0, 'F5', 'Sheet2!F2', 1, 'E5:F6', null);
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'E5', 'Sheet2!E2', 1, 'E5:F6', null);
                    expectFormulaCell(0, 'F5', null, 1, null, null);
                });
            });
        });

        describe('method "generateMoveCellsOperations"', function () {
            it('should fail to split matrix formulas', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                return cellCollection1.generateMoveCellsOperations(generator, r('C1:C1048576'), RIGHT).then(function () {
                    throw new Error('operation is expected to fail');
                }, function (result) {
                    expect(result).to.deep.equal({ cause: 'formula:matrix:insert' });
                });
            });
            it('should fail to shorten matrix formulas', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                return cellCollection1.generateMoveCellsOperations(generator, r('C1:C1048576'), LEFT).then(function () {
                    throw new Error('operation is expected to fail');
                }, function (result) {
                    expect(result).to.deep.equal({ cause: 'formula:matrix:delete' });
                });
            });
            it('should move all formula cells and defined names when inserting columns', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateMoveCellsOperations(generator, r('A1:B1048576'), RIGHT);
                return promise.then(function () {
                    expectFormulaCell(0, 'D2', 'G2+1', null, null, null);
                    expectFormulaCell(0, 'E3', 'H3+4', null, null, null);
                    expectFormulaCell(0, 'G2', 'Sheet2!E2+5', null, null, null);
                    expectFormulaCell(0, 'H3', 'Sheet2!F3+8', null, null, null);
                    expectFormulaCell(0, 'D5', 'G2', 0, 'D5:E6', null);
                    expectFormulaCell(0, 'E6', null, 0, null, null);
                    expectFormulaCell(0, 'G5', 'Sheet2!E2', 1, 'G5:H6', null);
                    expectFormulaCell(0, 'H6', null, 1, null, null);
                    expectFormulaCell(0, 'D8', 'G2:H3', null, null, 'D8:E9');
                    expectFormulaCell(0, 'G8', 'Sheet2!E2:F3', null, null, 'G8:H9');
                    expectName(null, 'global', 'Sheet1!E2+Sheet1!$G$2+Sheet2!E2+Sheet2!$E$2');
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'B2', 'E2+1', null, null, null);
                    expectFormulaCell(0, 'C3', 'F3+4', null, null, null);
                    expectFormulaCell(0, 'E2', 'Sheet2!E2+5', null, null, null);
                    expectFormulaCell(0, 'F3', 'Sheet2!F3+8', null, null, null);
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C6', null);
                    expectFormulaCell(0, 'C5', null, 0, null, null);
                    expectFormulaCell(0, 'B6', null, 0, null, null);
                    expectFormulaCell(0, 'C6', null, 0, null, null);
                    expectFormulaCell(0, 'E5', 'Sheet2!E2', 1, 'E5:F6', null);
                    expectFormulaCell(0, 'F5', null, 1, null, null);
                    expectFormulaCell(0, 'E6', null, 1, null, null);
                    expectFormulaCell(0, 'F6', null, 1, null, null);
                    expectFormulaCell(0, 'B8', 'E2:F3', null, null, 'B8:C9');
                    expectFormulaCell(0, 'E8', 'Sheet2!E2:F3', null, null, 'E8:F9');
                    expectName(null, 'global', 'Sheet1!E2+Sheet1!$E$2+Sheet2!E2+Sheet2!$E$2');
                });
            });
            it('should move all formula cells and defined names when deleting rows', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateMoveCellsOperations(generator, ra('A2:XFD2 A5:XFD5'), UP);
                return promise.then(function () {
                    expectFormulaCell(0, 'B2', 'E2+3', null, null, null);
                    expectFormulaCell(0, 'E2', 'Sheet2!E3+7', null, null, null);
                    expectFormulaCell(0, 'B4', 'E2', 0, 'B4:C4', null);
                    expectFormulaCell(0, 'E4', 'Sheet2!E3', 1, 'E4:F4', null);
                    expectFormulaCell(0, 'B6', 'E2:F2', null, null, 'B6:C7');
                    expectFormulaCell(0, 'E6', 'Sheet2!E2:F3', null, null, 'E6:F7');
                    expectName(null, 'global', 'Sheet1!E2+Sheet1!#REF!+Sheet2!E2+Sheet2!$E$2');
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'B2', 'E2+1', null, null, null);
                    expectFormulaCell(0, 'C3', 'F3+4', null, null, null);
                    expectFormulaCell(0, 'E2', 'Sheet2!E2+5', null, null, null);
                    expectFormulaCell(0, 'F3', 'Sheet2!F3+8', null, null, null);
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C6', null);
                    expectFormulaCell(0, 'C5', null, 0, null, null);
                    expectFormulaCell(0, 'B6', null, 0, null, null);
                    expectFormulaCell(0, 'C6', null, 0, null, null);
                    expectFormulaCell(0, 'E5', 'Sheet2!E2', 1, 'E5:F6', null);
                    expectFormulaCell(0, 'F5', null, 1, null, null);
                    expectFormulaCell(0, 'E6', null, 1, null, null);
                    expectFormulaCell(0, 'F6', null, 1, null, null);
                    expectFormulaCell(0, 'B8', 'E2:F3', null, null, 'B8:C9');
                    expectFormulaCell(0, 'E8', 'Sheet2!E2:F3', null, null, 'E8:F9');
                    expectName(null, 'global', 'Sheet1!E2+Sheet1!$E$2+Sheet2!E2+Sheet2!$E$2');
                });
            });
            it('should split shared formulas when inserting cells inbetween', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateMoveCellsOperations(generator, r('A6:XFD6'), DOWN);
                return promise.then(function () {
                    var si1 = expectFormulaCell(0, 'B5', 'E2', 'any', 'B5:C5', null).si;
                    expectFormulaCell(0, 'C5', null, si1, null, null);
                    var si2 = expectFormulaCell(0, 'B7', 'E3', 'any', 'B7:C7', null).si;
                    expectFormulaCell(0, 'C7', null, si2, null, null);
                    expectUndefinedCell(0, 'B6');
                    expectUndefinedCell(0, 'C6');
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'B5', 'E2', 0, 'B5:C6', null);
                    expectFormulaCell(0, 'C5', null, 0, null, null);
                    expectFormulaCell(0, 'B6', null, 0, null, null);
                    expectFormulaCell(0, 'C6', null, 0, null, null);
                    expectFormulaCell(0, 'E5', 'Sheet2!E2', 1, 'E5:F6', null);
                    expectFormulaCell(0, 'F6', null, 1, null, null);
                });
            });
            it('should update data validation and formatting rules when inserting rows', function () {
                var generator = sheetModel2.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection2.generateMoveCellsOperations(generator, r('A10:XFD10'), DOWN);
                return promise.then(function () {
                    expectValidation(0, 0, 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B12');
                    expectValidation(1, 0, 'B8:C10', 'B2+Sheet1!B2', 'B12+Sheet1!B11');
                    expectValidation(1, 1, 'D8:E11', 'B2+Sheet1!B2', 'B12+Sheet1!B11');
                    expectValidation(1, 2, 'F11:G11', 'B2+Sheet1!B2', 'B12+Sheet1!B11');
                    expectValidation(1, 3, 'H9:H10,I11:I11', '', '');
                    expectValidation(1, 4, 'I9:I10,H11:H11', '', '');
                    expectRule(0, '0', 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B12');
                    expectRule(1, '0', 'B8:C10', 'B2+Sheet1!B2', 'B12+Sheet1!B11');
                    expectRule(1, '1', 'D8:E11', 'B2+Sheet1!B2', 'B12+Sheet1!B11');
                    expectRule(1, '2', 'F11:G11', 'B2+Sheet1!B2', 'B12+Sheet1!B11');
                    expectApplyUndo(generator);
                    expectValidation(0, 0, 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B11');
                    expectValidation(1, 0, 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectValidation(1, 1, 'D8:E10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectValidation(1, 2, 'F10:G10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectValidation(1, 3, 'H9:H9,I10:I10', '', '');
                    expectValidation(1, 4, 'I9:I9,H10:H10', '', '');
                    expectRule(0, '0', 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B11');
                    expectRule(1, '0', 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectRule(1, '1', 'D8:E10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectRule(1, '2', 'F10:G10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                });
            });
            it('should update data validation and formatting rules when deleting rows', function () {
                var generator = sheetModel2.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection2.generateMoveCellsOperations(generator, r('A10:XFD10'), UP);
                return promise.then(function () {
                    expectValidation(0, 0, 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B10');
                    expectValidation(1, 0, 'B8:C9', 'B2+Sheet1!B2', 'B10+Sheet1!B11');
                    expectValidation(1, 1, 'D8:E9', 'B2+Sheet1!B2', 'B10+Sheet1!B11');
                    expectValidation(1, 2, 'H9:H9', '', '');
                    expectValidation(1, 3, 'I9:I9', '', '');
                    expectDeletedVal(1, 4);
                    expectRule(0, '0', 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B10');
                    expectRule(1, '0', 'B8:C9', 'B2+Sheet1!B2', 'B10+Sheet1!B11');
                    expectRule(1, '1', 'D8:E9', 'B2+Sheet1!B2', 'B10+Sheet1!B11');
                    expectDeletedRule(1, '2');
                    expectApplyUndo(generator);
                    expectValidation(0, 0, 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B11');
                    expectValidation(1, 0, 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectValidation(1, 1, 'D8:E10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectValidation(1, 2, 'F10:G10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectValidation(1, 3, 'H9:H9,I10:I10', '', '');
                    expectValidation(1, 4, 'I9:I9,H10:H10', '', '');
                    expectRule(0, '0', 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B11');
                    expectRule(1, '0', 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectRule(1, '1', 'D8:E10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectRule(1, '2', 'F10:G10', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                });
            });
            it('should update table references when deleting table', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateMoveCellsOperations(generator, r('A11:XFD12'), UP);
                return promise.then(function () {
                    expectFormulaCell(0, 'A11', 'SUM(#REF!,Table2[],#REF!,#REF!,Table2[Col1],Table2[Col2],Table2[[Col1]:[Col2]])', null, null, null);
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'A13', 'SUM(Table1[],Table2[],Table1[Col1],Table1[Col2],Table2[Col1],Table2[Col2],Table2[[Col1]:[Col2]])', null, null, null);
                });
            });
            it('should update table references when deleting columns', function () {
                var generator = sheetModel2.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection2.generateMoveCellsOperations(generator, r('C1:C1048576'), LEFT);
                return promise.then(function () {
                    expectFormulaCell(0, 'A13', 'SUM(Table1[],Table2[],Table1[Col1],Table1[Col2],Table2[Col1],#REF!,#REF!)', null, null, null);
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'A13', 'SUM(Table1[],Table2[],Table1[Col1],Table1[Col2],Table2[Col1],Table2[Col2],Table2[[Col1]:[Col2]])', null, null, null);
                });
            });
            it('should update chart source links when inserting rows', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateMoveCellsOperations(generator, ra('A3:XFD3 A11:XFD11'), DOWN);
                return promise.then(function () {
                    expectChartSeriesLinks(2, 0, 0, 'Sheet1!$J$1', 'Sheet1!$I$2:$I$4', 'Sheet1!$J$2:$J$4');
                    expectChartSeriesLinks(2, 0, 1, 'Sheet1!$K$1', 'Sheet1!$I$2:$I$4', 'Sheet1!$K$2:$K$4');
                    expectChartSeriesLinks(2, 0, 2, 'Sheet2!$K$1', 'Sheet1!$I$2:$I$4', 'Sheet2!$K$2:$K$3');
                    expectChartTitleLink(2, 0, null, 'Sheet1!$J$11:$J$13');
                    expectChartTitleLink(2, 0, 0, 'Sheet1!$K$11:$K$13');
                    expectChartTitleLink(2, 0, 1, 'Sheet2!$K$10:$K$11');
                    expectApplyUndo(generator);
                    expectChartSeriesLinks(2, 0, 0, 'Sheet1!$J$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$J$2:$J$3');
                    expectChartSeriesLinks(2, 0, 1, 'Sheet1!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$K$2:$K$3');
                    expectChartSeriesLinks(2, 0, 2, 'Sheet2!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet2!$K$2:$K$3');
                    expectChartTitleLink(2, 0, null, 'Sheet1!$J$10:$J$11');
                    expectChartTitleLink(2, 0, 0, 'Sheet1!$K$10:$K$11');
                    expectChartTitleLink(2, 0, 1, 'Sheet2!$K$10:$K$11');
                });
            });
            it('should update chart source links when deleting columns', function () {
                var generator = sheetModel1.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection1.generateMoveCellsOperations(generator, r('J1:J1048576'), LEFT);
                return promise.then(function () {
                    expectChartSeriesLinks(2, 0, 0, 'Sheet1!#REF!', 'Sheet1!$I$2:$I$3', 'Sheet1!#REF!');
                    expectChartSeriesLinks(2, 0, 1, 'Sheet1!$J$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$J$2:$J$3');
                    expectChartTitleLink(2, 0, null, 'Sheet1!#REF!');
                    expectChartTitleLink(2, 0, 0, 'Sheet1!$J$10:$J$11');
                    expectChartTitleLink(2, 0, 1, 'Sheet2!$K$10:$K$11');
                    expectApplyUndo(generator);
                    expectChartSeriesLinks(2, 0, 0, 'Sheet1!$J$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$J$2:$J$3');
                    expectChartSeriesLinks(2, 0, 1, 'Sheet1!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$K$2:$K$3');
                    expectChartSeriesLinks(2, 0, 2, 'Sheet2!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet2!$K$2:$K$3');
                    expectChartTitleLink(2, 0, null, 'Sheet1!$J$10:$J$11');
                    expectChartTitleLink(2, 0, 0, 'Sheet1!$K$10:$K$11');
                    expectChartTitleLink(2, 0, 1, 'Sheet2!$K$10:$K$11');
                });
            });
        });

        describe('method "generateUpdateFormulaOperations"', function () {
            it('should update formulas after renaming a sheet', function () {
                var generator = docModel.createSheetOperationGenerator({ applyImmediately: true });
                var promise = docModel.generateRenameSheetOperations(generator, 1, 'Sheet99');
                return promise.then(function () {
                    expectFormulaCell(0, 'E2', 'Sheet99!E2+5', null, null, null);
                    expectFormulaCell(0, 'F3', 'Sheet99!F3+8', null, null, null);
                    expectFormulaCell(0, 'E5', 'Sheet99!E2', 1, 'E5:F6', null);
                    expectFormulaCell(0, 'E8', 'Sheet99!E2:F3', null, null, 'E8:F9');
                    expectValidation(0, 0, 'B8:C9', 'B2+Sheet99!B2', 'B11+Sheet99!B11');
                    expectValidation(1, 0, 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectRule(0, '0', 'B8:C9', 'B2+Sheet99!B2', 'B11+Sheet99!B11');
                    expectRule(1, '0', 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectName(null, 'global', 'Sheet1!E2+Sheet1!$E$2+Sheet99!E2+Sheet99!$E$2');
                    expectChartSeriesLinks(2, 0, 0, 'Sheet1!$J$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$J$2:$J$3');
                    expectChartSeriesLinks(2, 0, 1, 'Sheet1!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$K$2:$K$3');
                    expectChartSeriesLinks(2, 0, 2, 'Sheet99!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet99!$K$2:$K$3');
                    expectChartTitleLink(2, 0, null, 'Sheet1!$J$10:$J$11');
                    expectChartTitleLink(2, 0, 0, 'Sheet1!$K$10:$K$11');
                    expectChartTitleLink(2, 0, 1, 'Sheet99!$K$10:$K$11');
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'E2', 'Sheet2!E2+5', null, null, null);
                    expectFormulaCell(0, 'F3', 'Sheet2!F3+8', null, null, null);
                    expectFormulaCell(0, 'E5', 'Sheet2!E2', 1, 'E5:F6', null);
                    expectFormulaCell(0, 'E8', 'Sheet2!E2:F3', null, null, 'E8:F9');
                    expectValidation(0, 0, 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B11');
                    expectValidation(1, 0, 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectRule(0, '0', 'B8:C9', 'B2+Sheet2!B2', 'B11+Sheet2!B11');
                    expectRule(1, '0', 'B8:C9', 'B2+Sheet1!B2', 'B11+Sheet1!B11');
                    expectName(null, 'global', 'Sheet1!E2+Sheet1!$E$2+Sheet2!E2+Sheet2!$E$2');
                    expectChartSeriesLinks(2, 0, 0, 'Sheet1!$J$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$J$2:$J$3');
                    expectChartSeriesLinks(2, 0, 1, 'Sheet1!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet1!$K$2:$K$3');
                    expectChartSeriesLinks(2, 0, 2, 'Sheet2!$K$1', 'Sheet1!$I$2:$I$3', 'Sheet2!$K$2:$K$3');
                    expectChartTitleLink(2, 0, null, 'Sheet1!$J$10:$J$11');
                    expectChartTitleLink(2, 0, 0, 'Sheet1!$K$10:$K$11');
                    expectChartTitleLink(2, 0, 1, 'Sheet2!$K$10:$K$11');
                });
            });
            it('should update formulas after relabeling a defined name', function () {
                var generator = docModel.createSheetOperationGenerator({ applyImmediately: true });
                var promise = docModel.getNameCollection().generateChangeNameOperations(generator, 'global', 'changed', null);
                return promise.then(function () {
                    expectFormulaCell(0, 'A14', 'SUM(changed,global2)', null, null, null);
                    expectName(null, 'global2', 'changed');
                    expectApplyUndo(generator);
                    expectFormulaCell(0, 'A14', 'SUM(global,global2)', null, null, null);
                    expectName(null, 'global2', 'global');
                });
            });
        });
    });

    // ========================================================================
});
