/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/editframework/utils/attributeutils'
], function (AttributeUtils) {

    'use strict';

    // static class AttributeUtils ============================================

    describe('EditFramework module AttributeUtils', function () {

        it('should exist', function () {
            expect(AttributeUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var attributeSet = {
            paragraph: {
                testattribute: 'exists'
            }
        };
        var correctResult = {
            paragraph: {
                testattribute: 'exists',
                inserted: 'yes'
            }
        };
        var node  = $('<div>').data('attributes', _.extend(_.clone(attributeSet), { styleId: 'MyStyle' }));
        var node2 = $('<div>').data('attributes', _.extend(_.clone(attributeSet), { styleId: 'MyStyle' }));
        var node3 = $('<div>').data('attributes', _.clone(attributeSet));

        // static methods -----------------------------------------------------

        describe('method "hasAttribute"', function () {
            it('should exist', function () {
                expect(AttributeUtils).itself.to.respondTo('hasAttribute');
            });

            it('should return true for check if paragraph has the attribute "testattribute"', function () {
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'testattribute')).to.equal(true);
            });
            it('should return false for check if paragraph has the attribute "zonk"', function () {
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'zonk')).to.equal(false);
            });
        });

        describe('method "insertAttribute"', function () {
            it('should exist', function () {
                expect(AttributeUtils).itself.to.respondTo('insertAttribute');
            });

            it('XXX', function () {
                var attr = _.clone(attributeSet);
                expect(AttributeUtils.hasAttribute(attr, 'newFamily', 'newattribute')).to.equal(false);
                AttributeUtils.insertAttribute(attr, 'newFamily', 'newattribute', 'added');
                expect(AttributeUtils.hasAttribute(attr, 'newFamily', 'newattribute')).to.equal(true);
            });

            it('should insert "newattribute". "hasAttribute" should return "true" so.', function () {
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'newattribute')).to.equal(false);
                AttributeUtils.insertAttribute(attributeSet, 'paragraph', 'newattribute', 'added');
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'newattribute')).to.equal(true);
            });
        });

        describe('method "deleteAttribute"', function () {
            it('should exist', function () {
                expect(AttributeUtils).itself.to.respondTo('deleteAttribute');
            });

            it('should delete "testattribute". "hasAttribute" should return "false" so.', function () {
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'newattribute')).to.equal(true);
                AttributeUtils.deleteAttribute(attributeSet, 'paragraph', 'newattribute');
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'newattribute')).to.equal(false);
                AttributeUtils.deleteAttribute(attributeSet, 'table', 'notExisting');
            });
        });

        describe('method "getExplicitAttributes"', function () {
            it('should exist', function () {
                expect(AttributeUtils).itself.to.respondTo('getExplicitAttributes');
            });

            it('should equal to the defined "attributeDefinitions.paragraph"', function () {
                expect(AttributeUtils.getExplicitAttributes(node, { family: 'paragraph' })).to.deep.equal(attributeSet.paragraph);
                expect(AttributeUtils.getExplicitAttributes(node, { family: 'table' })).to.be.an('object');
            });
        });

        describe('method "getElementStyleId"', function () {
            it('should exist', function () {
                expect(AttributeUtils).itself.to.respondTo('getElementStyleId');
            });
            it('should equal to the defined StyleId ("MyStyle")', function () {
                expect(AttributeUtils.getElementStyleId(node)).to.be.equal('MyStyle');
                expect(AttributeUtils.getElementStyleId(node)).not.to.equal(null);
                expect(AttributeUtils.getElementStyleId(node3)).to.equal(null);
            });
        });

        describe('method "hasEqualElementAttributes"', function () {
            it('should exist', function () {
                expect(AttributeUtils).itself.to.respondTo('hasEqualElementAttributes');
            });
            it('should be true at comparing "node"/"node1" and false at comparing "node"/"node3"', function () {
                expect(AttributeUtils.hasEqualElementAttributes(node, node2)).to.equal(true);
                expect(AttributeUtils.hasEqualElementAttributes(node, node3)).to.equal(false);
            });
        });

        describe('method "insertAttribute" with "missing" option', function () {
            it('should insert (only missing) attributes. "hasAttribute" should return "true" so.', function () {
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'inserted')).to.equal(false);
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'testattribute')).to.equal(true);
                AttributeUtils.insertAttribute(attributeSet, 'paragraph', 'inserted', 'yes', true);
                AttributeUtils.insertAttribute(attributeSet, 'paragraph', 'testattribute', 'new', true);
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'inserted')).to.equal(true);
                expect(AttributeUtils.hasAttribute(attributeSet, 'paragraph', 'testattribute')).to.equal(true);
                expect(attributeSet).to.deep.equal(correctResult);
            });
        });
    });

    // ========================================================================
});
